/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/gdif/texture.c#1 $
*/

#include <stdlib.h>

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/dll_in.h>
#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/gd.h>

#define ERR_RETURN(A) { \
    ERRerror("GDtexture_get_data", 0, ERR_ORIG, A); return(0); \
}

void GDtexture_ref_inc(GDtexture *texture)
{
  if (texture)
    texture->ref_count++;
}

void GDtexture_ref_dec(GDtexture *texture)
{
  if (texture && --texture->ref_count < 1)
    GDtexture_free(texture);
}


/* This function is called in a "delayed" fashion via the
   GDget_local routine. When we try to get the local texture
   structure pointer - if it doesn't exist the create routine
   is called. This will happen in two cases: either thru the
   texture update function that is called at instance time or
   via the object attach texture function that is called when the
   texture is attached to the object.
*/
int GDtexture_create(OMobj_id texture_id)
{
   GDtexture *texture;

   /* allocate a properties structure and initialize it */
   ALLOCN(texture, GDtexture, 1, "can't allocate texture");

   /* reference texture */
   GDtexture_ref_inc(texture);

   /* save OM element id in local data structures */
   texture->texture_id= texture_id;
   /* Save the id of the update function. This will be used to
      to add dependencies - we add dependencies between update functions.
   */
   texture->upd_func = OMfind_subobj(texture_id, OMstr_to_name("upd_func"), OM_OBJ_RW);

   GDset_local(texture_id, (char *)texture);
   return(1);
}

void GDtexture_free_data(GDtexture *texture)
{
    if (texture->data)
	ARRfree(texture->data);
    texture->data= NULL;
}

void GDtexture_free_colormap(GDtexture *texture)
{
    if (texture->colormap)
	ARRfree(texture->colormap);
    texture->colormap= NULL;
}

void GDtexture_free(GDtexture *texture)
{
  /* free renderer specific texture data if there's a func for it */
  if (texture->txtr_free_func)
    (*texture->txtr_free_func)(texture);

  GDtexture_free_data(texture);
  GDtexture_free_colormap(texture);
  free(texture);
}


/* This routine is invoked when the database element is being deleted */
int GDtexture_delete(OMobj_id texture_id)
{
   GDtexture *texture = NULL;

   if (OMis_null_obj(texture_id))
      return(1);
   texture = (GDtexture *)GDget_local(texture_id, (OMpfi)GDtexture_create);

   if (texture) {
     GDtexture_ref_dec(texture);
   }
   GDclear_local(texture_id);
   return(1);
}

void GDtexture_reset(OMobj_id texture_id)
{
   GDtexture *texture;
   unsigned char *data, *colormap;
   int tile, alpha, filter, type, stage;

   if (OMis_null_obj(texture_id))
      return;

   /* init arrays to perform initialization */
   data = NULL;
   colormap = NULL;
   tile = GD_TILE_CLAMP;
   alpha = GD_ALPHA_REPLACE;
   filter = GD_FILTER_POINT;
   type = GD_TYPE_UNFILTERED;
   stage = GD_STAGE_PRESPECULAR;

   /* initialize data base copy of properties */
   GDtexture_set_tile(texture_id, tile);
   GDtexture_set_alpha(texture_id, alpha);
   GDtexture_set_filter(texture_id, filter);
   GDtexture_set_type(texture_id, type);
   GDtexture_set_stage(texture_id, stage);

   /* initialize local copy of data */
   texture = (GDtexture *)GDget_local(texture_id, (OMpfi)GDtexture_create);

   texture->tile= tile;
   texture->alpha= alpha;
   texture->filter= filter;
   texture->type= type;
   texture->stage= stage;

   texture->changed = 0;
   texture->data_changed = 0;
   texture->cmap_changed = 0;
}

int GDtexture_update(OMobj_id texture_id)
{
   GDtexture *texture;
   int upd_seq, tmp_int;

   /* get pointer to local storage, return if NULL */
   texture = (GDtexture *)GDget_local(texture_id, (OMpfi)GDtexture_create);
   if (!texture)
      return(0);

   /* Get the sequence number of the update function. We can use
      this to determine if any sub-elements in props have changed.
      A mode of 0 means of the element itself.
   */
   upd_seq = OMget_obj_seq(texture->upd_func, OMnull_obj, 0);

   /* update data */
   GDtexture_update_data(texture);

   /* update colormap - nothing for now */
   GDtexture_update_colormap(texture);

   /* update tile */
   if (GDget_updint_val(texture_id, "tile", upd_seq, &tmp_int)) {
      texture->tile = tmp_int;
      if (texture->data)
         texture->changed= 1;
   }
   /* update alpha */
   if (GDget_updint_val(texture_id, "alpha", upd_seq, &tmp_int)) {
      texture->alpha = tmp_int;
      /* Treat this case as if the data has changed since we
	 will potentially create a different type of texture map
	 from the input data.
      */
      if (texture->data) {
         texture->data_changed = 1;
         texture->changed= 1;
      }
   }
   /* update filter */
   if (GDget_updint_val(texture_id, "filter", upd_seq, &tmp_int)) {
      texture->filter = tmp_int;
      if (texture->data)
         texture->changed= 1;
   }
   /* update type */
   if (GDget_updint_val(texture_id, "type", upd_seq, &tmp_int)) {
      texture->type = tmp_int;
      /* Treat this case as if the data has changed since we
	 will have to create a different type of texture map
	 from the input data.
      */
      if (texture->data) {
         texture->data_changed = 1;
         texture->changed= 1;
      }
   }
   /* update stage */
   if (GDget_updint_val(texture_id, "stage", upd_seq, &tmp_int)) {
      texture->stage = tmp_int;
      if (texture->data)
         texture->changed= 1;
   }
   return(1);
}

/* This routine gets the data from the field.
 */
/* 64-bit porting. Only Modified Internally */
int GDtexture_get_data(GDtexture *texture)
{
    int type, ndim, dims_size;
    xp_long *dims, node_size;

    /* first free the old data if any */
    GDtexture_free_data(texture);

    if (FLDget_ndim(texture->data_id, &ndim) != 1)
	return(0);

    if (ndim != 2 && ndim != 3)
	return(0);

    texture->ndims = ndim;

    if (FLDget_dims(texture->data_id, &dims, &dims_size) != 1)
	return(0);

    texture->width= (int)dims[0];
    texture->height= (int)dims[1];

    if (ndim == 3)
	texture->depth= dims[2];
    else
	texture->depth= 0;

    ARRfree(dims);

    /* the veclen of the field is actually the ncomps of the texture! */
    if (FLDget_node_data_veclen(texture->data_id, 0, &texture->ncomps) != 1)
	return(0);

    /* ALWAYS SPECIFIES COMP 0 */
    if (FLDget_node_data(texture->data_id, 0, &type, (char **)&texture->data,
			 &node_size, OM_GET_ARRAY_RD) != 1)
	return(0);

    return(1);
}

/* This routine gets the data from the texture colormap.
 */
/* 64-bit porting. Only Modified Internally */
int GDtexture_get_colormap(GDtexture *texture)
{
    int type;
    xp_long node_size;
    int veclen;
    xp_long num_entries;

    /* first free the old colormap if any */
    GDtexture_free_colormap(texture);

    if (FLDget_nnodes(texture->colormap_id, &num_entries) != 1)
	return(0);

    if (num_entries != 256)
    {
	ERR_RETURN("[GDtexture_get_colormap]: Texture colormap should have 256 entries.\n");
    }

    if (FLDget_node_data_veclen(texture->colormap_id, 0, &veclen) != 1)
	return(0);

    if (veclen != 4)
    {
	ERR_RETURN("[GDtexture_get_colormap]: Texture colormap should be a 4-vector.\n");
    }

    if (FLDget_node_data(texture->colormap_id, 0, &type, (char **)&texture->colormap,
			 &node_size, OM_GET_ARRAY_RD) != 1)
	return(0);

    return(1);
}

void GDtexture_update_data(GDtexture *texture)
{
   OMobj_id data_id, seq_id;
   int seq;

   /* See if anything is attached already. */
   if (!GDget_refer_db(texture->texture_id, "data", &data_id)) {
      /* If no input data for texture, but we have renderable
         data currently attached, detach the field by
         setting field id to null and sequence number to 0
      */
      if (!OMis_null_obj(texture->data_id)) {
         texture->data_id = OMnull_obj;
         texture->data_seq = 0;
	 texture->data_changed = 1;
         texture->changed= 1;
	 GDtexture_free_data(texture);
     }
   }
   else {
      /* Input data for texture, check to see if we have 
         any data currently
      */
      if (OMis_null_obj(texture->data_id)) {
         /* We are attaching a new field to the texture */
         texture->data_id = data_id;
         texture->data_seq = OMget_obj_seq(texture->data_id, OMnull_obj, OM_SEQ_VAL);
	 texture->data_changed = 1;
         texture->changed= 1;
	 GDtexture_get_data(texture);
      }
      else {
         /* We already have a field attached. Check for the possibility 
            that it is a different field - unlikely I hope
         */
         if (!OMequal_objs(texture->data_id, data_id)) {
            /* We have a different field, so we need to invalidate
               the existing cache and set the new field id and
               sequence number.
            */
            texture->data_id = data_id;
            texture->data_seq = OMget_obj_seq(texture->data_id, OMnull_obj, OM_SEQ_VAL);
	    texture->data_changed = 1;
            texture->changed= 1;
	    GDtexture_get_data(texture);
         }
         else {
            /* We have the same field, check to see if has actually changed
               by reading the sequence number. Use the template that specifies
               NOT to check if xform changes since it may be in the field and
               this should not cause the cache to become invalid.
            */
            seq_id = OMfind_subobj(OMtempl_obj, OMstr_to_name("GDdata_seq_templ"),
                OM_OBJ_RD);
            if (OMis_null_obj(seq_id))
	       ERRerror("GDtexture_update_data", 0, ERR_ORIG, "Can't find sequence template");
            seq = OMget_obj_seq(texture->data_id, seq_id, OM_SEQ_VAL);
            if (seq != texture->data_seq) {
               texture->data_seq = seq;
	       texture->data_changed = 1;
               texture->changed= 1;
	       GDtexture_get_data(texture);
            }
         }
      }
   }
}

void GDtexture_update_colormap(GDtexture *texture)
{
   OMobj_id colormap_id, seq_id;
   int seq;

   /* See if anything is attached already. */
   if (!GDget_refer_db(texture->texture_id, "colormap", &colormap_id)) {
      /* If no input colormap for texture, but we have renderable
         colormap currently attached, detach the field by
         setting field id to null and sequence number to 0
      */
      if (!OMis_null_obj(texture->colormap_id)) {
         texture->colormap_id = OMnull_obj;
         texture->colormap_seq = 0;
	 texture->cmap_changed = 1;
	 texture->changed = 1;
	 GDtexture_free_colormap(texture);
     }
   }
   else {
      /* Input colormap for texture, check to see if we have 
         any colormap currently
      */
      if (OMis_null_obj(texture->colormap_id)) {
         /* We are attaching a new field to the texture */
         texture->colormap_id = colormap_id;
         texture->colormap_seq = OMget_obj_seq(texture->colormap_id, OMnull_obj, OM_SEQ_VAL);
	 texture->cmap_changed = 1;
	 texture->changed = 1;
	 GDtexture_get_colormap(texture);
      }
      else {
         /* We already have a field attached. Check for the possibility 
            that it is a different field - unlikely I hope
         */
         if (!OMequal_objs(texture->colormap_id, colormap_id)) {
            /* We have a different field, so we need to invalidate
               the existing cache and set the new field id and
               sequence number.
            */
            texture->colormap_id = colormap_id;
            texture->colormap_seq = OMget_obj_seq(texture->colormap_id, OMnull_obj, OM_SEQ_VAL);
	    texture->cmap_changed = 1;
	    texture->changed = 1;
	    GDtexture_get_colormap(texture);
         }
         else {
            /* We have the same field, check to see if has actually changed
               by reading the sequence number. Use the template that specifies
               NOT to check if xform changes since it may be in the field and
               this should not cause the cache to become invalid.
            */
            seq_id = OMfind_subobj(OMtempl_obj, OMstr_to_name("GDdata_seq_templ"),
                OM_OBJ_RD);
            if (OMis_null_obj(seq_id))
	       ERRerror("GDtexture_update_colormap", 0, ERR_ORIG, "Can't find sequence template");
            seq = OMget_obj_seq(texture->colormap_id, seq_id, OM_SEQ_VAL);
            if (seq != texture->colormap_seq) {
               texture->colormap_seq = seq;
	       texture->cmap_changed = 1;
	       texture->changed = 1;
	       GDtexture_get_colormap(texture);
            }
         }
      }
   }
}

void GDtexture_set_tile(OMobj_id texture_id, int tile)
{
   GDset_int_val(texture_id, "tile", tile);
}

void GDtexture_get_tile(OMobj_id texture_id, int *tile)
{
   if (GDget_int_val(texture_id, "tile", tile) != 1)
      *tile = GD_TILE_CLAMP;
}

void GDtexture_set_alpha(OMobj_id texture_id, int alpha)
{
   GDset_int_val(texture_id, "alpha", alpha);
}

void GDtexture_get_alpha(OMobj_id texture_id, int *alpha)
{
   if (GDget_int_val(texture_id, "alpha", alpha) != 1)
      *alpha = GD_ALPHA_REPLACE;
}

void GDtexture_set_filter(OMobj_id texture_id, int filter)
{
   GDset_int_val(texture_id, "filter", filter);
}

void GDtexture_get_filter(OMobj_id texture_id, int *filter)
{
   if (GDget_int_val(texture_id, "filter", filter) != 1)
      *filter = GD_FILTER_POINT;
}

void GDtexture_set_type(OMobj_id texture_id, int type)
{
   GDset_int_val(texture_id, "type", type);
}

void GDtexture_get_type(OMobj_id texture_id, int *type)
{
   if (GDget_int_val(texture_id, "type", type) != 1)
      *type = GD_TYPE_UNFILTERED;
}

void GDtexture_set_stage(OMobj_id texture_id, int stage)
{
   GDset_int_val(texture_id, "stage", stage);
}

void GDtexture_get_stage(OMobj_id texture_id, int *stage)
{
   if (GDget_int_val(texture_id, "stage", stage) != 1)
      *stage = GD_STAGE_PRESPECULAR;
}
