/*
        This file is under Perforce control
        $Id: //depot/express/fcs70/hdf5/examp/write_rect.c#1 $

Example Unix command line to compile (if HDF5 is installed in /usr/local):
cc write_rect.c -I/usr/local/include /usr/local/lib/libhdf5.a -lm -lz
*/

#include <stdio.h>

#ifdef WIN32
/* #define _HDF5USEDLL_ */ /* Define this if using HDF5 DLL */
#endif
#include <hdf5.h>

static void write_int_attr( hid_t parent_id, const char *name, int value )
{
    hid_t h5_dspace_id, h5_attr_id;

    h5_dspace_id = H5Screate( H5S_SCALAR );
    if( h5_dspace_id > 0 ) {
        h5_attr_id = H5Acreate( parent_id, name, H5T_NATIVE_INT_g,
                                h5_dspace_id, H5P_DEFAULT );
        if( h5_attr_id > 0 ) {
            H5Awrite( h5_attr_id, H5T_NATIVE_INT_g, &value );
            H5Aclose( h5_attr_id );	/* close attribute */
        }
        H5Sclose( h5_dspace_id );	/* close dataspace */
    }
    return;
}

static void write_string_attr( hid_t parent_id, const char *name,
                               const char *value )
{
    hid_t h5_dspace_id, h5_dtype_id, h5_attr_id;

    h5_dspace_id = H5Screate( H5S_SCALAR );
    if (h5_dspace_id > 0) {
        h5_dtype_id   = H5Tcopy( H5T_C_S1_g );
        if( h5_dtype_id > 0 ) {
            H5Tset_size( h5_dtype_id, strlen(value)+1 );
            h5_attr_id = H5Acreate ( parent_id, name,
                                     h5_dtype_id, h5_dspace_id, H5P_DEFAULT );
            if( h5_attr_id > 0 ) {
                H5Awrite( h5_attr_id, h5_dtype_id, (char *)value );
                H5Aclose( h5_attr_id );	/* close attribute */
            }
            H5Tclose( h5_dtype_id );	/* close datatype */
        }
        H5Sclose( h5_dspace_id );	/* close dataspace */
    }
    return;
}

int main( int argc, char *argv[] )
{
    hid_t h5_file_id, h5_root_id, h5_nd_id;
    hid_t h5_dspace_id, h5_attr_id, h5_dset_id;
    hsize_t h5_dims[2];

    char *filename;

    int fld_nspace = 2;
    int fld_ndim   = 2;
    int fld_dims[2] = { 4, 3 };

    int fld_npoints = 7;	/* Sum of the dimensions for Rectilinear. */
    /* fld_points[fld_npoints][fld_nspace] */
    float fld_points[7][2] = { { 0.0, 0.0 }, 
                               { 1.0, 0.0 },
                               { 2.0, 0.0 },
                               { 4.0, 0.0 },
                               { 0.0, 0.0 },
                               { 0.0, 1.0 },
                               { 0.0, 7.0 } };
    /* fld_node_data[fld_dims[1]][fld_dims[0]][veclen] */
    /* omit veclen if veclen == 1 */
    int fld_node_data[3][4] = { {0,  1,  2,  3},
                                {4,  5,  6,  7},
                                {8,  9, 10, 11} };

    /* Initialize HDF5 library.  OK to call multiple times. */
    if( H5open() < 0 ) {
        fprintf( stderr, "Could not initialize HDF5 library." );
        return -1;
    }

    if( argc > 1 )
        filename = argv[1];
    else
        return -1;

    /* Create a new HDF5 file using H5_ACC_TRUNC access, default
     * file creation properties, and default file access properties.
     */
    h5_file_id = H5Fcreate( filename,
                            H5F_ACC_TRUNC,  /* overwrite existing files */
                            H5P_DEFAULT,    /* default file creation    */
                            H5P_DEFAULT     /* default file access      */
                            );
    if( h5_file_id < 0 ) return -1;

    /* Grab handle to the HDF5 root group, which exists by default
     * and does not have to be created.
     */
    h5_root_id = H5Gopen( h5_file_id, "/" );
    if( h5_root_id < 0 ) return -1;

    /* The next three attributes are optional, but at least one
     * of them should appear in the HDF5 file so that the reader
     * can determine that this is a Rectilinear field.
     */
    write_string_attr( h5_root_id, "XP_CLASS", "Mesh_Rect+Node_Data" );
    write_int_attr( h5_root_id, "grid_type", 2 );
    write_int_attr( h5_root_id, "npoints", fld_npoints );

    /* 2D field */
    write_int_attr( h5_root_id, "nspace", fld_nspace );
    write_int_attr( h5_root_id, "ndim", fld_ndim );

    /*
     * dimensions
     */

    h5_dims[0] = fld_ndim;
    /* Create a simple 1D dataspace */
    h5_dspace_id = H5Screate_simple( 1, h5_dims, NULL );
    if( h5_dspace_id > 0 ) {
        /* Create an attribute */
        h5_attr_id = H5Acreate( h5_root_id, "dims",
                                H5T_NATIVE_INT_g, h5_dspace_id,
                                H5P_DEFAULT );
        if (h5_attr_id > 0) {
            H5Awrite( h5_attr_id, H5T_NATIVE_INT_g, fld_dims );
            H5Aclose( h5_attr_id );	/* close attribute */
        }
        H5Sclose( h5_dspace_id );	/* close dataspace */
    }

    /*
     * points.  This is optional because the reader, like the .fld file
     * reader, will fill in default of 0, dims-1 for the points.
     */

    h5_dims[0] = fld_npoints;  /* npoints */
    h5_dims[1] = fld_nspace; /* nspace */
    /* Create a simple 2D dataspace */
    h5_dspace_id = H5Screate_simple( 2, h5_dims, NULL );
    if( h5_dspace_id > 0 ) {
        /* Create the 2D dataset.
         */
        h5_dset_id = H5Dcreate( h5_root_id, "points",
                                H5T_NATIVE_FLOAT_g,
                                h5_dspace_id,  H5P_DEFAULT );
        if( h5_dset_id > 0 ) {
            H5Dwrite( h5_dset_id, H5T_NATIVE_FLOAT,
                      h5_dspace_id, h5_dspace_id, H5P_DEFAULT,
                      fld_points );
            H5Dclose( h5_dset_id );	/* close dataset */
        }
        H5Sclose( h5_dspace_id );	/* close dataspace */    
    }

    /*
     * node data
     */

    /* one node data component */
    write_int_attr( h5_root_id, "nnode_data", 1 );

    /* Create a group to hold the node data component.
     * Note the naming convention used to represent the notion
     * of arrays of groups.  A second node data component would
     * be a new group named "node_data[1]".
     */
    h5_nd_id = H5Gcreate(h5_root_id, "node_data[0]", 0);
    if( h5_nd_id < 0 ) {
        H5Gclose( h5_root_id );
        H5Fclose( h5_file_id );
        return -1;
    }
    else {
        /* Optional, but recommended.  The reader will default the
         * veclen to 1 if it can't find it in the HDF5 file.
         */
        write_int_attr( h5_nd_id, "veclen", 1 );

        /* Optional, but recommended since the labels
         * are used in the UI.
         */
        write_string_attr( h5_nd_id, "labels", "velocity" );

        /* Optional */
        write_string_attr( h5_nd_id, "units", "furlongs per fortnight" );

        /*
         *  node data values
         */

        /* Create a simple 2D dataspace. */
        /* NOTE ORDER OF THE DIMS. fld_dims is in column-major
         * order and HDF5 uses row-major order, so we need to
         * reverse the order of the dims.
         */
        h5_dims[0] = fld_dims[1];
        h5_dims[1] = fld_dims[0];
        /* If veclen > 1, then veclen would be added here to the
         * hdf5 dimensions for a 3D dataspace.
         */
        h5_dspace_id = H5Screate_simple( 2, h5_dims, NULL );
        if( h5_dspace_id > 0 ) {
            /* Create the dataset for node_data[0].values, remember
             * its a child of the node data group, not the root group.
             */
            h5_dset_id = H5Dcreate( h5_nd_id, "values",
                                    H5T_NATIVE_INT_g,
                                    h5_dspace_id,  H5P_DEFAULT );
            if( h5_dset_id > 0 ) {
                H5Dwrite( h5_dset_id, H5T_NATIVE_INT,
                          h5_dspace_id, h5_dspace_id, H5P_DEFAULT,
                          fld_node_data );
                H5Dclose( h5_dset_id );	/* close dataset */
            }
            H5Sclose( h5_dspace_id );	/* close dataspace */    
        }

        H5Gclose( h5_nd_id );   /* close node data group */
    }

    H5Gclose( h5_root_id ); /* close root group */

    H5Fclose( h5_file_id ); /* close file */

    return 0;
}
