/*
			Copyright (c) 2002 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/hdf5/xp_mods/h5_array.c#1 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>  /* time_t */

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/f_utils.h>
#include <avs/err.h>
#include <avs/om.h>

#include "h5_xp.h"

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

int HDF5write_array_update( OMobj_id, OMevent_mask, int );
int HDF5read_array_update( OMobj_id, OMevent_mask, int );

static int HDF5write_array( hid_t h5_root_id, const char *name,
                            OMobj_id arr_id );

static int HDF5read_array( hid_t h5_root_id, const char *var_name,
                           OMobj_id arr_id );

#undef ERR_RETURN
#define ERR_RETURN(A) {ERRerror("HDF5 write array", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

/* 64-bit proting. Only Modified Internally */
int
HDF5write_array_update(OMobj_id mod_id, OMevent_mask mask, int seq_num)
{
    OMobj_id file_id, var_id, arr_id;

    hid_t h5_file_id, h5_root_id;
    herr_t h5_stat;

    char *temp_str = NULL, *filename = NULL;
    char file_buf[AVS_PATH_MAX];
    char /* *root_str = NULL, */ *var_str = NULL;
    int newfile;

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );
    var_id  = OMfind_subobj( mod_id, OMstr_to_name("variable"), OM_OBJ_RD );
    arr_id  = OMfind_subobj( mod_id, OMstr_to_name("inArr"), OM_OBJ_RD );

    if( OMget_str_val( file_id, &temp_str, 0 ) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if (temp_str) { free (temp_str); temp_str = NULL; }

    if (!filename) {
        ERR_RETURN( "Bad filename" );
    }

    /* Initialize HDF5 library.  OK to call multiple times. */
    if( H5open() < 0 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }
    /* Detect a problem with global variables and DLL settings. */
    if( H5T_NATIVE_INT_g == 0 || H5T_NATIVE_INT_g == -1 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }

    if( OMget_name_int_val( mod_id, OMstr_to_name("newfile"), &newfile )
        != OM_STAT_SUCCESS )
      newfile = 1;

    if( newfile == 0 ) {
        time_t mod_time = 0;
        if( !FILEexists(filename, FILE_READ|FILE_WRITE, &mod_time) )
          newfile = 1;
    }

    if( newfile ) {
        /*
         * Create a new HDF5 file using H5_ACC_TRUNC access, default
         * file creation properties, and default file access properties.
         */
        h5_file_id = H5Fcreate( filename,
                                H5F_ACC_TRUNC,  /* overwrite existing files */
                                H5P_DEFAULT,    /* default file creation    */
                                H5P_DEFAULT );  /* default file access      */
        if( h5_file_id < 0 ) {
            ERR_RETURN( "Couldn't create output file" );
        }
    }
    else {
        /*
         * Open a existing HDF5 file.
         */
        h5_file_id = H5Fopen( filename,
                              H5F_ACC_RDWR, /* read-write access to file */
                              H5P_DEFAULT );  /* default file access      */
        if( h5_file_id < 0 ) {
            ERR_RETURN( "Couldn't open output file" );
        }
    }


    /* Grab root group */
    h5_root_id = H5Gopen(h5_file_id, "/");

    if( OMget_name_str_val( mod_id, OMstr_to_name("variable"), &var_str, 0 )
        != OM_STAT_SUCCESS ) {
        var_str = strdup( "array" );
    }

    if( HDF5write_array( h5_root_id, var_str, arr_id ) != METHOD_SUCCESS ) {
        ERR_RETURN( "Error while writing file" );
    }

    free( var_str );

    H5Gclose( h5_root_id );

    h5_stat = H5Fclose( h5_file_id );

    if( h5_stat < 0 ) {
        ERRerror( "HDF5 write_array", 1, ERR_ORIG,
                  "Error %d while closing HDF5 file", h5_stat );
      return METHOD_FAILURE;
    }

    return METHOD_SUCCESS;
}


#undef ERR_RETURN
#define ERR_RETURN(A) {ERRerror("HDF5 read array", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

int
HDF5read_array_update(OMobj_id mod_id, OMevent_mask mask, int seq_num)
{
    OMobj_id file_id, var_id, arr_id;

    char *temp_str = NULL, *filename = NULL;
    char file_buf[AVS_PATH_MAX];

    char *p, *var_name = NULL, *var_path = NULL;

    hid_t h5_file_id, h5_root_id;
    herr_t h5_stat;

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );
    var_id  = OMfind_subobj( mod_id, OMstr_to_name("variable"), OM_OBJ_RD );
    /* out */
    arr_id  = OMfind_subobj( mod_id, OMstr_to_name("outArr"), OM_OBJ_RW );

    if( OMget_str_val(file_id, &temp_str, 0) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if (temp_str) { free (temp_str); temp_str = NULL; }

    if( OMget_str_val(var_id, &var_path, 0) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get variable path." );

    /* Initialize HDF5 library.  OK to call multiple times. */
    if( H5open() < 0 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }
    /* Detect a problem with global variables and DLL settings. */
    if( H5T_NATIVE_INT_g == 0 || H5T_NATIVE_INT_g == -1 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }

    /*
     * Open a existing HDF5 file.
     */
    h5_file_id = H5Fopen( filename,
                          H5F_ACC_RDONLY,  /* read-only access to file */
                          H5P_DEFAULT     /* default file access      */
                          );

    if( h5_file_id < 0 ) {
        ERR_RETURN( "Couldn't open input file" );
    }

    p = strrchr( var_path, '/' );
    if( p == NULL ) {
        /* No directory, just a variable (dataset or attribute) name */
        /* Grab H5 root group */
        h5_root_id = H5Gopen(h5_file_id, "/");
        var_name = strdup( var_path );
    }
    else {
        /* Get the variable name */
        var_name = strdup( p+1 );
        /* Isolate the directory portion of the name */
        *p = 0;
        /* Grab H5 group */
        h5_root_id = H5Gopen(h5_file_id, var_path );
    }
    if (var_path) { free (var_path); var_path = NULL; }

    HDF5read_array( h5_root_id, var_name, arr_id );

    if (var_name) { free (var_name); var_name = NULL; }

    H5Gclose( h5_root_id );

    h5_stat = H5Fclose( h5_file_id );

    if( h5_stat < 0 ) {
        ERRerror( "HDF5 read_field", 1, ERR_ORIG,
                  "Error %d while closing HDF5 file", h5_stat );
      return METHOD_FAILURE;
    }

    return METHOD_SUCCESS;
}

#undef ERR_RETURN
#define ERR_RETURN(A) {ERRerror("HDF5 write array", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

/* 64-bit porting. Only Modified Internally */
static int
HDF5write_array( hid_t h5_parent_id, const char *name, OMobj_id arr_id )
{
    int ndim;
    xp_long dims[OM_ARRAY_MAXDIM];
    int i, dtype;
    char *array = NULL;

    dtype = OM_TYPE_UNSET;
    ndim = 0;

    if( OMget_array( arr_id, &dtype, &array, &ndim, dims,
                             OM_GET_ARRAY_RD ) == OM_STAT_SUCCESS ) {

        hid_t h5_dtype_file, h5_dtype_app;
        hsize_t h5_dims[OM_ARRAY_MAXDIM];
        hid_t h5_dspace_id, h5_attr_id;

        HDF5map_dtypes_wr( dtype, &h5_dtype_file, &h5_dtype_app );

        /* h5_dims */
        for( i = 0; i < ndim; ++i ) {
            /* See comments in read_array */
            h5_dims[i] = dims[ndim-(i+1)];
        }

        h5_dspace_id = H5Screate_simple( ndim, h5_dims, NULL );
        if( h5_dspace_id > 0 ) {
            h5_attr_id   = H5Acreate( h5_parent_id, name,
                                      h5_dtype_file, h5_dspace_id,
                                      H5P_DEFAULT );
            if( h5_attr_id > 0 ) {
                H5Awrite( h5_attr_id, h5_dtype_app, array );
                H5Aclose( h5_attr_id );	/* close attribute */
            }
            H5Sclose( h5_dspace_id );	/* close dataspace */
        }

        ARRfree( array );
    }

    return METHOD_SUCCESS;
}

#undef ERR_RETURN
#define ERR_RETURN(A) {ERRerror("HDF5 read array", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

/* 64-bit porting. Only Modified Internally */
static int
HDF5read_array( hid_t h5_root_id, const char *var_name, OMobj_id arr_id )
{
    hid_t h5_dset_id, h5_attr_id, h5_dspace_id, h5_dtype_id;
    hid_t h5_dtype_app;

    hsize_t h5_dims[H5S_MAX_RANK];
    int h5_ndims;

    int ndim;
    xp_long dims[OM_ARRAY_MAXDIM];
    int i, dtype;
    char *array = NULL;

    if( (h5_dset_id = probe_dataset( h5_root_id, var_name, 0 )) > 0 ) {
        h5_dtype_id  = H5Dget_type( h5_dset_id );
        h5_dspace_id = H5Dget_space( h5_dset_id );

        HDF5map_dtypes_rd( h5_dtype_id, &h5_dtype_app, &dtype );

        h5_ndims = H5Sget_simple_extent_ndims( h5_dspace_id );
        if( h5_ndims > 0 ) {
            H5Sget_simple_extent_dims( h5_dspace_id, h5_dims, NULL );
            /*
             * Switch from row-major HDF5 dimensions to column-major
             * OM dimensions.  Even without the flipping, the copy step
             * is necessary because sizeof(h5_dims[0]) might not be the same
             * as sizeof(dims[0]).
             */
            ndim = h5_ndims;
            for( i = 0; i < ndim; ++i ) {
                dims[ndim-(i+1)] = h5_dims[i];
            }
            if( OMget_array( arr_id, &dtype, &array, &ndim, dims,
                             OM_GET_ARRAY_WR ) == OM_STAT_SUCCESS ) {
                H5Dread( h5_dset_id, h5_dtype_app, H5S_ALL, H5S_ALL,
                         H5P_DEFAULT, array );
                ARRfree( array );
            }
        }

        H5Sclose( h5_dspace_id );
        H5Tclose( h5_dtype_id );
        H5Dclose( h5_dset_id );
    }
    else if( (h5_attr_id = probe_attribute( h5_root_id, var_name, 0 )) > 0 ) {
        h5_dtype_id  = H5Aget_type( h5_attr_id );
        h5_dspace_id = H5Aget_space( h5_attr_id );

        HDF5map_dtypes_rd( h5_dtype_id, &h5_dtype_app, &dtype );

        h5_ndims = H5Sget_simple_extent_ndims( h5_dspace_id );
        if( h5_ndims > 0 ) {
            H5Sget_simple_extent_dims( h5_dspace_id, h5_dims, NULL );
            /*
             * Switch from row-major HDF5 dimensions to column-major
             * OM dimensions.
             */
            ndim = h5_ndims;
            for( i = 0; i < ndim; ++i ) {
                dims[ndim-(i+1)] = h5_dims[i];
            }
            if( OMget_array( arr_id, &dtype, &array, &ndim, dims,
                             OM_GET_ARRAY_WR ) == OM_STAT_SUCCESS ) {
                H5Aread( h5_attr_id, h5_dtype_app, array );
                ARRfree( array );
            }
        }

        H5Sclose( h5_dspace_id );
        H5Tclose( h5_dtype_id );
        H5Aclose( h5_attr_id );
    }
    else
        return METHOD_FAILURE;

    return METHOD_SUCCESS;
}
