/*
			Copyright (c) 2001 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/netcdf/xp_mods/nc_wr_fld.c#1 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>  /* time_t */

#define	XP_WIDE_API	/* Use Wide APIs */
#include <avs/f_utils.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
/* Someday rename to unidata/netcdf.h */
#include <avs/netcdf.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

typedef struct _ncvar {                  /* variable */
  char name[NC_MAX_NAME];
  int id;
  nc_type type;
  int ndims;
  int dim_ids[NC_MAX_VAR_DIMS];
} ncvar_t;


typedef struct _ncdim {                  /* dimension */
  char name[NC_MAX_NAME];
  size_t len;
  int coord_var;       /* coordinate variable flag */
  int var_id;
  float *points;
} ncdim_t;


#define ERR_RETURN(A) {ERRerror("netcdf write field", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

typedef enum { FLD_Unspecified = 0, FLD_UNIF, FLD_RECT, FLD_STRUCT} FLD_Type;

int NCwrite_field_update( OMobj_id, OMevent_mask, int );

static int NCwrite_field( const char *, OMobj_id );

int NCwrite_field_update(OMobj_id mod_id, OMevent_mask mask, int seq_num)
{
    OMobj_id file_id, trigger_id, tmp_in_id, in_id;

    char *temp_str = NULL, *filename = NULL;
    char file_buf[AVS_PATH_MAX];

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );
    tmp_in_id  = OMfind_subobj( mod_id, OMstr_to_name("inFld"), OM_OBJ_RD );

    trigger_id  = OMfind_subobj( mod_id, OMstr_to_name("trigger"),
                                 OM_OBJ_RD );

    /* OK if missing entirely, means we are running in non-UI mode. */
    if( !OMis_null_obj( trigger_id ) ) {
        if( !(OM_EVENT_INST & mask) && !OMchanged( trigger_id, seq_num ) )
            return METHOD_SUCCESS;
    }

    if( OMget_str_val( file_id, &temp_str, 0 ) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if (temp_str) { free (temp_str); temp_str = NULL; }

    if (!filename) {
        ERR_RETURN( "Bad filename" );
    }

    if( OMget_obj_val( tmp_in_id, &in_id ) != OM_STAT_SUCCESS ) {
       ERR_RETURN( "Can't access input field" );
    }
    if( NCwrite_field( filename, in_id ) != METHOD_SUCCESS ) {
        ERR_RETURN( "Error while writing file" );
    }

    return METHOD_SUCCESS;
}

static void
NCtext_attr_xp2nc( int nc_id, int nc_var_id, const char *nc_att_name,
                   OMobj_id parent_id, const char *xp_name )
{
  OMobj_id sub_id;
  char buffer[256];

  sub_id = OMfind_subobj( parent_id, OMstr_to_name(xp_name), OM_OBJ_RD );

  if( !OMis_null_obj( sub_id ) ) {
      char *p = buffer;
      buffer[0] = 0;
      if( (OMget_str_val( sub_id, &p, sizeof(buffer) ) == OM_STAT_SUCCESS)
          && (buffer[0] != 0) )
          nc_put_att_text( nc_id, nc_var_id, nc_att_name,
                           strlen(buffer)+1, buffer );
  }
}

/* 64-bit porting. Only Modified Internally */
static int NCwrite_field( const char *filename, OMobj_id fld_id )
{
    int i, j;
    xp_long k;
    char buffer[256];

    int nc_stat;
    int nc_id;
    int nc_dim_ids[NC_MAX_DIMS];

    int nc_cv_var_ids[NC_MAX_VAR_DIMS];   /* "coordinate variables" */
    int nc_nd_var_ids[NC_MAX_VARS];       /* node data components */
    int nc_coord_var_id;

    FLD_Type fld_type;
    int fld_ndims, fld_nspace, fld_ncomps, dims_size;
    xp_long *fld_dims;

    /*
     * First things first.  What kind of input field do we have?
     */

    if( FLDget_ndim( fld_id, &fld_ndims ) != OM_STAT_SUCCESS ) {
        ERR_RETURN( "Could not find field ndims.  Field type must be Unif, Rect, or Struct" );
    }

    {
        static OMobj_id grd_unif_tmpl, grd_rect_tmpl, grd_struct_tmpl;
        static int init = 0;

        if( !init ) {
            grd_unif_tmpl =   OMfind_subobj( OMtempl_obj, OMstr_to_name("Grid_Unif"), OM_OBJ_RD );
            grd_rect_tmpl =   OMfind_subobj( OMtempl_obj, OMstr_to_name("Grid_Rect"), OM_OBJ_RD );
            grd_struct_tmpl = OMfind_subobj( OMtempl_obj, OMstr_to_name("Grid_Struct"), OM_OBJ_RD );

            init = 1;
        }

#define match_flags (OM_MATCH_EXACT | OM_MATCH_NO_ERROR)

        if( OMmatch_obj( grd_unif_tmpl, fld_id, match_flags ) == OM_STAT_SUCCESS )
            /* test: npoints == 2 and the points array
               has a valid value */
            fld_type = FLD_UNIF;
        else if( OMmatch_obj( grd_rect_tmpl, fld_id, match_flags ) == OM_STAT_SUCCESS )
            /* test: npoints == sum(dims), and the points array
               has a valid value */
            fld_type = FLD_RECT;
        else if( OMmatch_obj( grd_struct_tmpl, fld_id, match_flags ) == OM_STAT_SUCCESS )
            /* test: often npoints is unset, but not always, the
               points array will not have a valid value, and coordinates.values
               will have a value other than get_coords_rect or get_coords_unif
            */
            fld_type = FLD_STRUCT;
        else {
            ERR_RETURN( "Field type must be Unif, Rect, or Struct" );
        }

#undef match_flags

    }

    ncopts = NC_VERBOSE;
    nc_stat = nc_create( filename, NC_CLOBBER, &nc_id );
    if( nc_stat != NC_NOERR ) {
        ERR_RETURN( "Couldn't create output file" );
    }

    /* nC global attributes */

    {
        const char *xp_class_name = NULL;
        switch( fld_type ) {
        case FLD_UNIF:   xp_class_name = "Mesh_Unif+Node_Data"; break;
        case FLD_RECT:   xp_class_name = "Mesh_Rect+Node_Data"; break;
        case FLD_STRUCT: xp_class_name = "Mesh_Struct+Node_Data";
        }
        nc_put_att_text( nc_id, NC_GLOBAL, "XP_CLASS",
                         strlen(xp_class_name)+1, xp_class_name );
#if 0
        fprintf( stderr, "XP_CLASS: %s\n", xp_class_name );
#endif
    }

    NCtext_attr_xp2nc( nc_id, NC_GLOBAL, "title", fld_id, "NC_title" );

    strcpy( buffer, "Created by AVS/Express" );
    nc_put_att_text( nc_id, NC_GLOBAL, "history",
                     strlen(buffer)+1, buffer );

    /* nC dimensions */

    {
        int i;
        OMobj_id dim_names_id;

        /* "point" (axis) information */
        float *fld_points = NULL;
        xp_long pts_size = 0;
        if( fld_type == FLD_UNIF || fld_type == FLD_RECT ) {
            if (FLDget_points(fld_id, &fld_points, &pts_size,
                              OM_GET_ARRAY_RD) != OM_STAT_SUCCESS) {
                fld_points = NULL;
                pts_size = 0;
            }
        }

        FLDget_dims( fld_id, &fld_dims, &dims_size );

        dim_names_id = OMfind_subobj( fld_id, OMstr_to_name("NC_dim_names"),
                                      OM_OBJ_RD );

        /* Define the dimensions.
         * Flip the order of the dimensions, see read code for explaination.
         */
        for( i = 0; i < fld_ndims; ++i ) {
            size_t nc_dim_len;
            j = fld_ndims - (i + 1);
            sprintf( buffer, "dim%d", i );
            if( !OMis_null_obj( dim_names_id ) ) {
                char *p = buffer;
                OMget_str_array_val( dim_names_id, j, &p, sizeof(buffer) );
            }
            nc_dim_len = fld_dims[j]; /* or NC_UNLIMITED */
            nc_def_dim( nc_id, buffer, nc_dim_len, &nc_dim_ids[i] );

            /* Unif and Rect fields: decide whether to define a
             * nC coordinate variable for the dimension.
             */
            if( fld_points &&
                (fld_type == FLD_UNIF || fld_type == FLD_RECT) ) {
                int cv_needed_flag, nc_var_id;
                cv_needed_flag = 0;
                if( fld_type == FLD_UNIF ) {
                    if( (fld_points[j] == 0) &&
                        (fld_points[j+fld_ndims] == (fld_dims[j] - 1) ) )
                        /* No need for a coordinate variable */
                        cv_needed_flag = 0;
                    else cv_needed_flag = 1;
                }
                else if( fld_type == FLD_RECT )
                    /* Its possible its not needed for all dimensions,
                       but it would be a pain to figure it all out.
                    */
                    cv_needed_flag = 1;

                if( cv_needed_flag )
                    nc_stat = nc_def_var( nc_id, buffer, NC_FLOAT,
                                          1, &nc_dim_ids[i], &nc_var_id );
                else
                    nc_var_id = -1;

                nc_cv_var_ids[i] = nc_var_id;
            }
        }
        if( fld_points ) ARRfree( fld_points );
    }

    /* Struct fields: Define the xp coordinates var */
    if( fld_type == FLD_STRUCT ) {
        int nc_cv_dim_ids[2]; /* these get dropped on the floor */
        xp_long fld_npoints;

        /* nnodes should be the same as the product of the dimensions */
        FLDget_nnodes( fld_id, &fld_npoints );
        FLDget_nspace( fld_id, &fld_nspace );

        nc_stat = nc_def_dim( nc_id, "xp_npoints", fld_npoints,
                              &nc_cv_dim_ids[0] );
        nc_stat = nc_def_dim( nc_id, "xp_nspace", fld_nspace,
                              &nc_cv_dim_ids[1] );
        nc_stat = nc_def_var( nc_id, "XP_COORDINATES", NC_FLOAT,
                              2, nc_cv_dim_ids, &nc_coord_var_id );
    }

    /* nC variables */

    fld_ncomps = 0;
    if( FLDget_node_data_ncomp( fld_id, &fld_ncomps ) == OM_STAT_SUCCESS ) {
        OMobj_id node_data_id, comp_id;

        node_data_id = OMfind_subobj( fld_id, OMstr_to_name("node_data"),
                                      OM_OBJ_RW );
        if( OMis_null_obj(node_data_id) )
            return METHOD_FAILURE;  /* Don't expect this error */

        for( j=0; j<fld_ncomps; ++j ) {
            int fld_data_type;
            int node_null_flag;
            double node_null_value;

            nc_type nc_data_type;
            int nc_var_id;

            if( OMget_array_val(node_data_id, j, &comp_id, OM_OBJ_RW) !=
                OM_STAT_SUCCESS )
                return METHOD_FAILURE;  /* Don't expect this error */

            FLDget_node_data_type( fld_id, j, &fld_data_type );
            switch( fld_data_type ) {
            case DTYPE_BYTE:   nc_data_type = NC_BYTE;   break;
            case DTYPE_SHORT:  nc_data_type = NC_SHORT;  break;
            case DTYPE_INT:    nc_data_type = NC_INT;    break;
            case DTYPE_FLOAT:  nc_data_type = NC_FLOAT;  break;
            case DTYPE_DOUBLE: nc_data_type = NC_DOUBLE; break;
            default:
                return METHOD_FAILURE;
            }

            /* node data labels, use as nC variable names */
            buffer[0] = 0;
            if( (FLDget_node_data_label(fld_id, j, buffer,
                                        sizeof(buffer) ) != OM_STAT_SUCCESS )
                || (buffer[0] == 0) )
                sprintf( buffer, "var%d", j );

            nc_stat = nc_def_var (nc_id, buffer, nc_data_type,
                                  fld_ndims, nc_dim_ids, &nc_var_id);
            nc_nd_var_ids[j] = nc_var_id;

            if( fld_data_type == DTYPE_BYTE ) {
                int v_min = 0;
                int v_max = 255;
                /* Let netCDF know that these are 0-255 bytes */
                nc_put_att_int( nc_id, nc_var_id,  "valid_min",
                                NC_INT, 1, &v_min );
                nc_put_att_int( nc_id, nc_var_id,  "valid_max",
                                NC_INT, 1, &v_max );
            }

            /* variable attributes */

            /* _FillValue */
            node_null_flag = 0;
            if (FLDget_node_null_data(fld_id, j, &node_null_flag,
                                      (char *)&node_null_value) == OM_STAT_SUCCESS && node_null_flag ) {
                switch( fld_data_type ) {
                case DTYPE_BYTE: {
                    unsigned char nv_byte = *(unsigned char *)&node_null_value;
                    nc_put_att_uchar( nc_id, nc_var_id, "_FillValue",
                               NC_BYTE, 1, &nv_byte );
                }
                break;
                case DTYPE_SHORT: {
                    short nv_short = *(short *)&node_null_value;
                    nc_put_att_short( nc_id, nc_var_id, "_FillValue", 
                                      NC_SHORT, 1, &nv_short );
                }
                break;
                case DTYPE_INT: {
                    int nv_int = *(int *)&node_null_value;
                    nc_put_att_int( nc_id, nc_var_id, "_FillValue", 
                                    NC_INT, 1, &nv_int );
                }
                break;
                case DTYPE_FLOAT: {
                    float nv_float = *(float *)&node_null_value;
                    nc_put_att_float( nc_id, nc_var_id, "_FillValue",
                                      NC_FLOAT, 1, &nv_float );
                }
                break;
                case DTYPE_DOUBLE: {
                    double nv_double = *(double *)&node_null_value;
                    nc_put_att_double( nc_id, nc_var_id, "_FillValue",
                                       NC_DOUBLE, 1, &nv_double );
                }
                } /* switch */
            } /* if null data */

            /* units */

            NCtext_attr_xp2nc( nc_id, nc_var_id, "units",
                               comp_id, "units" );

            /* NC_long_name.  Not part of a standard Express field. */

            NCtext_attr_xp2nc( nc_id, nc_var_id, "long_name",
                               comp_id, "NC_long_name" );

        } /* loop over components */
    }  /* if node data */

    /* Exit netCDF define mode */

    nc_stat = nc_enddef( nc_id );
    if( nc_stat != NC_NOERR ) {
        ERRerror( "netcdf write_field", 1, ERR_ORIG,
                  "Error %d while exiting netCDF define mode", nc_stat );
        nc_stat = nc_close( nc_id );
        if( nc_stat != NC_NOERR )
            ERRerror( "netcdf write_field", 1, ERR_ORIG,
                      "Error %d while closing netCDF file", nc_stat );
        return METHOD_FAILURE;
    }

    /* nC data */

    /* Unif and Rect fields: Put the coordinate variables */
    if( fld_type == FLD_UNIF || fld_type == FLD_RECT ) {
        int nc_var_id;
        xp_long dim_size;
        /* "point" (axis) information */
        float *fld_points = NULL;
        float *nc_points = NULL;
        xp_long pts_size = 0;

        if (FLDget_points(fld_id, &fld_points, &pts_size,
                          OM_GET_ARRAY_RD) != OM_STAT_SUCCESS) {
            fld_points = NULL;
            pts_size = 0;
        }

        for( i = 0; i < fld_ndims; ++i ) {
            j = fld_ndims - (i + 1); /* flipped dimensions */
            dim_size = fld_dims[j];
            nc_var_id = nc_cv_var_ids[i];
            if( nc_var_id == -1 ) continue;
            if( fld_type == FLD_UNIF ) {
                double start = fld_points[j];
                double end   = fld_points[j+fld_ndims];
#if 0
                fprintf( stderr, 
                     "i:%d, j:%d, start:%f, end:%f, ndims:%d, dims_size:%d\n",
                         i, j, start, end, fld_ndims, dim_size );
#endif
                nc_points = malloc( dim_size * sizeof(float) );
                for( k = 0; k < dim_size; ++k ) {
                    if( k == 0 )
                        nc_points[k] = start;
                    else if( k == (dim_size-1) )
                        nc_points[k] = end;
                    else 
                        nc_points[k] = 
                            ((start*(dim_size -(k+1))) + (end*k))/((double)(dim_size-1));
#if 0
                    fprintf( stderr, "%f ", nc_points[k] );
#endif
                }
#if 0
                fprintf( stderr, "\n" );
#endif
                nc_stat = nc_put_var_float( nc_id, nc_var_id, nc_points );
            }
            else if( fld_type == FLD_RECT ) {
                xp_long dim_sum = 0;
                /* The outer loop is running in the order of the 
                 * nC dimensions, which is the reverse of the
                 * order of the XP dimensions.
                 */
                for( k = j-1; k >= 0; --k )
                    dim_sum += fld_dims[k];
#if 0
                fprintf( stderr, 
                         "i:%d, j:%d, ndims:%d, dims_size:%d, dim_sum:%ld\n",
                         i, j, fld_ndims, dim_size, dim_sum );
#endif
                nc_points = malloc( dim_size * sizeof(float) );
                for( k = 0; k < dim_size; ++k ) {
                    nc_points[k] = fld_points[(dim_sum+k)*fld_ndims + j];
#if 0
                    fprintf( stderr, "%f ", nc_points[k] );
#endif
                }
#if 0
                fprintf( stderr, "\n" );
#endif
                nc_stat = nc_put_var_float( nc_id, nc_var_id, nc_points );
                free( nc_points );
            }
        }
        if( fld_points ) ARRfree( fld_points );
    }

    /* Struct fields: Put the coordinate information */
    if( fld_type == FLD_STRUCT ) {
        /* Coordinates information */
        xp_long coord_size;
        float *coord_array = NULL;
        if (FLDget_coord(fld_id, &coord_array, &coord_size,
                         OM_GET_ARRAY_RD) == OM_STAT_SUCCESS) {
            nc_stat = nc_put_var_float( nc_id, nc_coord_var_id, coord_array );
            ARRfree( coord_array );
        }
        else {
            ERR_RETURN( "Could not get input field coords" );
        }
    }

    if( fld_dims ) {
        ARRfree( fld_dims );
        fld_dims = NULL;
    }

    /* Write out the Node_Data */

    if( fld_ncomps > 0 ) {
        for( j=0; j<fld_ncomps; ++j ) {
            int fld_data_type;
            xp_long node_data_size;
            char *node_data = NULL;
            int nc_var_id = nc_nd_var_ids[j];

            if (FLDget_node_data(fld_id, j, &fld_data_type, &node_data, 
                                 &node_data_size, OM_GET_ARRAY_RD) == 1) {
                switch( fld_data_type ) {
                case DTYPE_BYTE:
                    nc_stat = nc_put_var_uchar( nc_id, nc_var_id,
                                                (unsigned char *)node_data );
                    break;
                case DTYPE_SHORT:
                    nc_stat = nc_put_var_short( nc_id, nc_var_id,
                                                (short *)node_data );
                    break;
                case DTYPE_INT:  
                    nc_stat = nc_put_var_int( nc_id, nc_var_id,
                                              (int *)node_data );
                    break;
                case DTYPE_FLOAT:
                    nc_stat = nc_put_var_float( nc_id, nc_var_id,
                                                (float *)node_data );
                    break;
                case DTYPE_DOUBLE:
                    nc_stat = nc_put_var_double( nc_id, nc_var_id,
                                                 (double *)node_data );
                    break;
                }
                ARRfree( node_data );
            }
        }
    }

    nc_stat = nc_close( nc_id );
    if( nc_stat != NC_NOERR ) {
        ERRerror( "netcdf write_field", 1, ERR_ORIG,
                  "Error %d while closing netCDF file", nc_stat );
      return METHOD_FAILURE;
    }

    return METHOD_SUCCESS;
}
