//			Copyright (c) 1993 by
//			Advanced Visual Systems Inc.
//			All Rights Reserved
//
//	This software comprises unpublished confidential information of
//	Advanced Visual Systems Inc. and may not be used, copied or made
//	available to anyone, except in accordance with the license
//	under which it is furnished.
//
//	This file is under Perforce control
//	$Id: //depot/express/fcs70/omx/omx.cxx#1 $
//

#include <string.h>

#define XP_WIDE_API  /* Use Wide  APIs */
#include <avs/dll_in.h>
#include <avs/omx.hxx>
#include <avs/util.h>

OMXobj::OMXobj (OMobj_id obj_id,
		const char *name,
		int mode,
		OMXobj *parent_ptr,
		const char *class_path)
{
   created = 0;
   is_ref = 0;
   is_value = 1;
   need_sub_refs = 0;

   /*
    * XXX Work around hack for HP C++ compiler bug!
    */
   if ((xp_ulong)parent_ptr < 0x1234 && parent_ptr != NULL) {
      fprintf(stderr,"bogus parent ptr encountered %p %s\n",
              parent_ptr, name);
      parent = NULL;
   }
   else
      parent = parent_ptr;

   if (mode == OMX_EXISTING_OBJ) {
      v.obj_id = obj_id;
   }
   /*
    * We are creating this guy with an id but the id is volatile (i.e.
    * it could be reset at any time) so we must tell our sub-objects
    * not to cache their ids.  Instead, they must look to us to
    * get our id each time.
    */
   else if (mode == OMX_TRANSIENT_OBJ) {
      v.obj_id = obj_id;
      need_sub_refs = 1;
   }
   else if (mode == OMX_CREATE_SUB_OBJ) {
      if (name == NULL) v.obj_id = OMnull_obj;
      else {
	 /*
	  * We always initialize ourselves in "reference" mode.  The
	  * first time that someone attempts to get our value RW, we
	  * will both get our ID RW and then also turn us out of "ref" mode.
	  */
	 if (parent != NULL) {
	    is_ref = 1;        // must lookup our id each time...
     	    // propagate this flag down to the children
	    need_sub_refs = parent->need_sub_refs;
	    v.sub_name = name;
	 }
	 else {
	    v.obj_id = OMfind_str_subobj(obj_id, name, OM_OBJ_RW);
	    if (OMis_null_obj(v.obj_id)) {
               char buf1[256];			
	       ERRerror("OMX", 2, ERR_ORIG,
		        "can't find sub object: %s in parent: %s",
		        name, OMret_obj_path(obj_id,buf1,sizeof(buf1)));
	    }
	 }
      }
   }
   else {
      char buf1[128];	   
      if (class_path == NULL)
         class_path = class_name();

      if (class_path == NULL) {
	 ERRerror("OMX", 0, ERR_ORIG, "can't create object with null class");
	 v.obj_id = OMnull_obj;
	 return;
      }
      if (name == NULL) {
	 if ((name = strrchr(class_path,'.')) != NULL) name++;
	 else name = class_path;
      }
      else name = OMret_valid_obj_name(name,buf1,sizeof(buf1));

      v.obj_id = OMcreate_obj_from_path(class_path,name,obj_id);
      created = 1;
   }
}

/* 64-bit changes. Code moved here from omx.hxx due to compatability reasons */
/* 64-bit porting. Directly Modified */
int OMXobj::set_array(int type, void *array, xp_long len, int mode)
{
   return(OMset_array(obj_id(), type, array, len, mode));
}

/* 64-bit porting. Directly Modified */
int OMXobj::set_array_size(xp_long size)
{
   return(OMset_array_size(obj_id(OM_OBJ_RW), size));
}

/* 64-bit porting. Modified with Parallel APIs */
int OMXobj::get_array(int *type, void **array, int *ndim, xp_long dims[], int mode)
{

   return(OMget_array(obj_id(), type, (char **)array, ndim, dims, mode));
}

#ifdef 	WORDLENGTH_64
int OMXobj::get_array_n(int *type, void **array, int *ndim, int dims[], int mode)
{
   return(OMget_array_n(obj_id(), type, (char **)array, ndim, dims, mode));
}
#endif

/* 64-bit porting. Modified with Parallel APIs */
void *OMXobj::ret_array_ptr(int mode, xp_long *size, int *type)
{

   return(OMret_array_ptr(obj_id(), mode, size, type));
}

#ifdef	WORDLENGTH_64
void *OMXobj::ret_array_ptr_n(int mode, int *size, int *type)
{
   return(OMret_array_ptr_n(obj_id(), mode, size, type));
}
#endif

/* 64-bit porting. Modified with Parallel APIs */
void *OMXobj::ret_typed_array_ptr(int mode, int type, xp_long *size)
{
    return(OMret_typed_array_ptr(obj_id(), mode, type, size));
}

#ifdef WORDLENGTH_64
void *OMXobj::ret_typed_array_ptr_n(int mode, int type, int *size)
{
   return(OMret_typed_array_ptr_n(obj_id(), mode, type, size));
}
#endif

/* 64-bit porting. Directly Modified */
char *OMXobj::ret_str_array_val(xp_long index, char *buffer, xp_long buffer_size)
{
   return(OMret_str_array_val(obj_id(), index, buffer, buffer_size));
}

/* 64-bit porting. Directly Modified */
int OMXobj::set_str_array_val(xp_long index, const char *value)
{
   return(OMset_str_array_val(obj_id(), index, value));
}

/* 64-bit porting. Modified with Parallel APIs */
int OMXobj::get_array_size(xp_long *size) const
{
   return(OMget_array_size(obj_id(OM_OBJ_RD), size));
}

#ifdef WORDLENGTH_64
int OMXobj::get_array_size_n(int *size) const
{
   return(OMget_array_size_n(obj_id(OM_OBJ_RD), size));
}
#endif

/* 64-bit porting. Directly Modified */
xp_long OMXobj::ret_array_size() const
{
   xp_long size;
   if (get_array_size(&size) != 1)
      size = 0;
   return(size);
}

/* 64-bit porting. Directly Modified */
xp_long OMXobj::ret_num_subobjs() const
{
   xp_long n = 0;
   OMget_num_subobjs( obj_id(), &n );
   return(n);
}

/* 64-bit porting. Directly Modified */
OMobj_id OMXobj::ret_subobj_id( xp_long i, int mode )
{
   OMobj_id id = OMnull_obj;

   OMget_array_subobj( obj_id(), i, &id, mode );
   return( id );
}
/* 64-bit changes. END */

const char *
OMXobj::name() const
{
   return(OMret_obj_name(obj_id(OM_OBJ_RD)));
}

OMXgroup::OMXgroup (OMobj_id obj_id_arg,
		    const char *name,
		    int mode,
		    OMXobj *parent_ptr,
		    const char *class_path) :
   OMXobj(obj_id_arg, name, mode, parent_ptr, class_path)
{
   OMobj_id pval_id;
   /*
    * If we are not a "value" reference mode, we need our sub-objects to
    * always re-get their id.
    *
    * Note: in the array case, this obj_id can be the array's value, not
    * the group itself.  In this case, we definitely want to make sure
    * that ref mode is set.
    *
    * We use the pval case because this will return an error for groups
    * that aren't references but will succeed for links and other junk.
    */
   if (OMget_obj_pval(obj_id(OM_OBJ_RD), &pval_id, 0) != -1) {
      need_sub_refs = 1;
      is_value = 0;
   }
   else
      is_value = 1;

   class_ptr = NULL;
}

OMXappl::OMXappl (OMobj_id obj_id_arg,
		    const char *name,
		    int mode,
		    OMXobj *parent_ptr,
		    const char *class_path) :
   OMXgroup(obj_id_arg, name, mode, parent_ptr, class_path) {}

/*
 * There is no class_ptr associated with this guy
 */
void *
OMXgroup::ret_class_ptr(const char *name, void *) const
{
   /*
    * Give the user a hook to get whatever we've got here..
    *
    * Otherwise, we need the failure case so that we can make
    * sure that the class does not exist down another branch in
    * the class hierarchy.
    */
   if (name == NULL) return(class_ptr);
   return(NULL);
}

/*
 * There is no class_ptr associated with this guy
 */
void *
OMXgroup::ret_omx_ptr(const char *name)
{
   if (name == NULL) return(this);

   if (!strcmp(name,"OMXgroup"))
      return(this);

   return(NULL);
}

/*
 * If we get here, this object does not have a user's class associated
 * with it to create
 */
void
OMXgroup::init_class_ptr()
{
   class_ptr = NULL;
}

/*
 * If we get here, this object does not have a user's class associated
 * with it to create
 */
void
OMXgroup::free_class_ptr()
{
   if (class_ptr != NULL) {
      ERRerror("OMX::free_class_ptr", 0, ERR_ORIG,
		"free_class_ptr should have null class_ptr arg");
   }
}


//
// OMXobj::obj_id(int mode)
//
//   returns the object id of the specified object.
//
OMobj_id OMXobj::obj_id (int mode) const
{
   OMobj_id obj_id;

   /*
    * If we are a sub-object of a reference (or for some other reason
    * we are in this mode...) we must go back to our parent and
    * re-get the id of our object each time.
    */
   if (is_ref) {
      obj_id = OMfind_str_subobj( (*parent).obj_id(mode), v.sub_name, mode);
      /*
       * If this is RW mode and we don't require ourselves to get it
       * out in "ref_mode" each time, we convert it to storing the id
       * directly.
       */
      if (OM_OBJ_IS_RW(mode) && !need_sub_refs) {
	 // Cast away constness here; the *intent* is still const.
	 ((OMXobj *)this)->v.obj_id = obj_id;
	 ((OMXobj *)this)->is_ref = 0;
      }
   }
   else {
      obj_id = v.obj_id;
   }
   return(obj_id);
}

//
// set the object id that we are using for a particular object
//
void
OMXobj::set_obj_id (OMobj_id obj_id)
{
   /*
    * Delete our reference to this object...
    */
   if (created && !OMis_null_obj(v.obj_id))
      OMuser_destroy_obj(v.obj_id);

   v.obj_id = obj_id;
   is_ref = 0;
   created = 0;
}

OMXobj::~OMXobj ()
{
   if (created)
      OMuser_destroy_obj(obj_id());
}

OMXobj::operator OMobj_id () const
{
   return obj_id();
}

int
OMXobj::add_obj_ref(OMobj_id conn_id, int mode)
{
   return(OMadd_obj_ref(obj_id(), conn_id, mode));
}

/* 64-bit porting. Directly Modified */
int
OMXobj::insert_obj_ref(xp_long ind, OMobj_id conn_id, int mode)
{
   return(OMinsert_obj_ref(obj_id(), ind, conn_id, mode));
}

int
OMXobj::del_obj_ref(OMobj_id conn_id, int mode)
{
   return(OMdel_obj_ref(obj_id(), conn_id, mode));
}

int
OMXobj::set_obj_ref(OMobj_id conn_id, int mode)
{
   return(OMset_obj_ref(obj_id(), conn_id, mode));
}

int
OMXobj::set_obj_val(OMobj_id conn_id, int mode)
{
   return(OMset_obj_val(obj_id(), conn_id, mode));
}

const char *
OMXobj::class_name()
{
   return(NULL);
}

//
// Definitions for the OMXbprim class
//
OMXbprim::OMXbprim(OMobj_id obj_id, const char *name, int mode, OMXobj *parent_ptr, const char *path) :
	       OMXobj (obj_id, name, mode, parent_ptr, path) {}

int OMXbprim::get_int_val (int &value, int dflt) const
{
   int stat;

   if ((stat = OMget_int_val(obj_id(OM_OBJ_RD),&value)) != 1) {
      value = dflt;
      if (stat != 0) {
         char buf1[128];
         ERRerror("OMXbprim", 1, ERR_ORIG,
		  "integer value is not defined for object: %s",
		  OMret_obj_path(obj_id(OM_OBJ_RD), buf1, sizeof(buf1)));
      }
   }
   return(stat);
}

/* 64-bit porting. Newly Introduced */
int OMXbprim::get_long_val (xp_long &value, xp_long dflt) const
{
   int stat;

   if ((stat = OMget_long_val(obj_id(OM_OBJ_RD),&value)) != 1) {
      value = dflt;
      if (stat != 0) {
         char buf1[128];		  
         ERRerror("OMXbprim", 1, ERR_ORIG,
		  "long value is not defined for object: %s",
		  OMret_obj_path(obj_id(OM_OBJ_RD), buf1, sizeof(buf1)));
      }
   }
   return(stat);
}

int OMXbprim::get_real_val (double &value, double dflt) const
{
   int stat;
   
   if ((stat = OMget_real_val(obj_id(OM_OBJ_RD),&value)) != 1) {
      value = dflt;
      if (stat != 0) {
         char buf1[128];		  
         ERRerror("OMXbprim", 1, ERR_ORIG,
		  "real value is not defined for object: %s",
		  OMret_obj_path(obj_id(OM_OBJ_RD), buf1, sizeof(buf1)));
      }
   }
   return(stat);
}

int OMXbprim::get_data_type ()
{
   int type;
   if (OMget_data_type(obj_id(), &type) != OM_STAT_SUCCESS)
      type = OM_TYPE_UNSET;
   return type;
}


//
// Definitions for the OMXprim class
//
OMXprim::OMXprim(OMobj_id obj_id, const char *name, int mode, OMXobj *parent_ptr, const char *path) :
	       OMXbprim (obj_id, name, mode, parent_ptr, path) {}

const char *
OMXprim::class_name()
{
   return("STD.prim");
}


//
// Definitions for the OMXint class
//

OMXint::OMXint(OMobj_id obj_id, const char *name, int mode, OMXobj *parent_ptr,
		const char *class_path) :
	       OMXbprim (obj_id, name, mode, parent_ptr, class_path) {}

const char *
OMXint::class_name()
{
   return("int");
}

//
// Definitions for the OMXlong class
//
/* 64-bit porting. Newly Introduced */
OMXlong::OMXlong(OMobj_id obj_id, const char *name, int mode, OMXobj *parent_ptr,
		const char *class_path) :
	       OMXbprim (obj_id, name, mode, parent_ptr, class_path) {}

const char *
OMXlong::class_name()
{
   return("long");
}

//
// Definitions for the OMXreal class
//
OMXreal::OMXreal(OMobj_id obj_id, const char *name, int mode, OMXobj *parent_ptr,
		 const char *class_path) :
	         OMXbprim (obj_id, name, mode, parent_ptr, class_path) {}

const char *
OMXreal::class_name()
{
   return("double");
}


//
// Method definitions for OMXstr class
//
OMXstr::OMXstr(OMobj_id obj_id, const char *name, int mode, OMXobj *parent_ptr,
		const char *class_path) :
	       OMXbprim (obj_id, name, mode, parent_ptr, class_path)
{
   _save_str = NULL;
}

OMXstr::~OMXstr()
{
   if (_save_str != NULL) {
      FREE(_save_str);
      _save_str = NULL;
   }
}

const char *
OMXstr::class_name()
{
   return("string");
}

int OMXstr::get_str_val (char **value, char *dflt)
{
   int status;
   if (_save_str != NULL) {
      FREE(_save_str);
      _save_str = NULL;
   }

   // If the value isn't set, use the dflt. If there is a value
   // but the value is the same as the default, just return the
   // default.

   if ((status = OMget_str_val (obj_id (OM_OBJ_RD), &_save_str, 0)) != 1) {
      _save_str = NULL;
      *value = dflt;
   }
   else {
      *value = _save_str;
   }
   return status;
}

int OMXstr::set_str_val (const char *str)
{
    // the OM will free the old string.
    return OMset_str_val (obj_id (), str);
}

int OMXstr::operator== (const char *rhs)
{
    char *_v = NULL;

    get_str_val( &_v );
    if (!_v) return(0);
    return( strcmp( _v, rhs ) == 0 );
}

int OMXstr::operator== (OMXstr &rhs)
{
    char *_v = NULL;

    get_str_val( &_v );
    if (!_v) return(0);
    return( strcmp( _v, (const char *)rhs ) == 0 );
}

/* 64-bit porting. Directly Modified */
int OMXstr::operator== (xp_long rhs)
{
    char *_v = NULL;

    get_str_val( &_v );
    if (!_v) return(0);
    return( (xp_long)_v == rhs );
}

int OMXstr::operator!= (const char *rhs)
{
    char *_v = NULL;

    get_str_val( &_v );
    if (!_v) return(0);
    return( strcmp( _v, rhs ) != 0 );
}

int OMXstr::operator!= (OMXstr &rhs)
{
    char *_v = NULL;

    get_str_val( &_v );
    if (!_v) return(0);
    return( strcmp( _v, (const char *)rhs ) != 0 );
}

/* 64-bit porting. Directly Modified */
int OMXstr::operator!= (xp_long rhs)
{
    char *_v = NULL;

    get_str_val( &_v );
    if (!_v) return(0);
    return( (xp_long)_v != rhs );
}

/* 64-bit porting. Only Modified Internally */
OMXstr & OMXstr::operator+= (const char *rhs)
{
    xp_long _len;
    char *_u = NULL, *_w;

    get_str_val( &_u );
    if (_u) {
       _len = strlen( _u ) + strlen( rhs ) + 1;
       _w = (char *) malloc ( _len );
       if (_w) {
          strcpy( _w, _u );
          strcat( _w, rhs );
          set_str_val( _w );
          free( _w );
       }
    }
    return *this;
}

/* 64-bit porting. Only Modified Internally */
OMXstr & OMXstr::operator+= (OMXstr &rhs)
{
    xp_long  _len;
    char *_u = NULL, *_v = NULL, *_w;

    get_str_val( &_u );
    rhs.get_str_val( &_v ) ;
    if (_u && _v) {
       _len = strlen( _v ) + strlen( _u ) + 1;
       _w = (char *) malloc ( _len );
       if (_w) {
          strcpy( _w, _u );
          strcat( _w, _v );
          set_str_val ( _w );
          free( _w );
       }
    }
    return *this;
}

/* 64-bit porting. Directly Modified */
OMXstr & OMXstr::operator+= (const xp_long rhs)
{
    xp_long  _len;
    char  *_u = NULL, *_w;
    char  _v[20];

    get_str_val( &_u );
    if (_u) {
       sprintf( _v, "%ld", rhs );
       _len = strlen( _u ) + strlen( _v ) + 1;
       _w = (char *) malloc ( _len );
       if (_w) {
          strcpy( _w, _u );
          strcat( _w, _v );
          set_str_val ( _w );
          free( _w );
       }
    }
    return *this;
}

/* 64-bit porting. Directly Modified */
OMXstr & OMXstr::operator+= (const xp_ulong rhs)
{
    xp_long   _len;
    char  *_u = NULL, *_w;
    char  _v[11];

    get_str_val( &_u );
    if (_u) {
       sprintf( _v, "0x%lx", rhs );
       _len = strlen( _u ) + strlen( _v ) + 1;
       _w = (char *) malloc ( _len );
       if (_w) {
          strcpy( _w, _u );
          strcat( _w, _v );
          set_str_val ( _w );
          free( _w );
       }
    }
    return *this;
}

//
// Method definitions for OMXenum class
//

OMXenum::OMXenum(OMobj_id obj_id, const char *name, int mode, OMXobj *parent_ptr,
		const char *class_path) :
	       OMXstr (obj_id, name, mode, parent_ptr, class_path) {}

const char *
OMXenum::class_name()
{
   return("enum");
}


// ------------ OMXgroup_array ------------

OMXgroup *
OMXgroup_array::ret_array_member(OMobj_id val_id)
{
   return(new OMXgroup(val_id,NULL,OMX_TRANSIENT_OBJ,NULL));
}

/* 64-bit porting. Only Modified Internally */
void OMXgroup_array::reshape()
{
   xp_long size, i;
   OMobj_id val_id;
   OMXgroup **new_values;

   if (OMget_array_size(obj_id(),&size) != 1)
      size = 0;

   if (size != _arr_size) {
      /*
       * Delete any old objects that we might need to get rid of.
       */
      for (i = size; i < _arr_size; i++) {
	 _arr_values[i]->free_class_ptr();
 	 delete _arr_values[i];
      }

      if (size) {
         new_values = new OMXgroup*[size];
      }
      if (_arr_size) {
	 xp_long copy_size = _arr_size > size ? size : _arr_size;
	 if (copy_size)
	    memcpy(new_values,_arr_values,sizeof(OMXgroup *) * copy_size);
	 delete [] _arr_values;
      }
      for (i = _arr_size; i < size; i++) {
         if (OMget_array_val(obj_id(), i, &val_id, OM_OBJ_RW) != 1)
 	    val_id = OMnull_obj;

         /*
          * Make sure we have the group object (actual value) not just
          * a reference to it.  GNAT 7993 / CFS PR 9604.
          */
         OMget_obj_val(val_id, &val_id);

         new_values[i] = ret_array_member(val_id);
	 if (is_value) {
	    /*
	     * Create a C++ class for this object
	     */
	    new_values[i]->init_class_ptr();
	    /*
	     * Associate this C++ class with the OM object so that
	     * other C++ objects can get at it with OMret_omx_ptr
	     */
	    OMset_omx_ptr(val_id, new_values[i], 0);
	 }
      }
      _arr_values = new_values;
      _arr_size = size;
   }
}

//----------------------------------------------------------------------
// array index operator
//----------------------------------------------------------------------
/* 64-bit porting. Directly Modified */
OMXgroup&
OMXgroup_array::operator[](const xp_long i)
{
   OMobj_id val_id;

   /*
    * Make sure that our array is the right size...
    */
   reshape();

   if (i < 0) {
      char buf1[128];	   
      ERRerror("OMXgroup_array", 2, ERR_ORIG,
		"improper dimension: %d is less than 0 for object: %s",
		i, OMret_obj_path(obj_id(),buf1,sizeof(buf1)));
      return(*_arr_values[0]);
   }

   if (i >= _arr_size) {
      char buf1[128];	   
      ERRerror("OMXgroup_array", 3, ERR_ORIG,
		"improper dimension: %d > array size: %d, for object: %s",
		i, _arr_size, OMret_obj_path(obj_id(),buf1,sizeof(buf1)));
      return(*_arr_values[0]);
   }

   /*
    * Since the array could have been de-allocated and re-allocated
    * between calls to this guy, we need to validate that our id
    * id correct.
    */
   if (OMget_array_val(obj_id(), i, &val_id, OM_OBJ_RW) != 1)
      val_id = OMnull_obj;

   if (!OMequal_objs(_arr_values[i]->obj_id(),val_id))
      _arr_values[i]->set_obj_id(val_id);

   return(*_arr_values[i]);
}

OMXgroup_array::OMXgroup_array(OMobj_id id, const char *name, int mode,
				     OMXobj *parent_ptr) :
             		OMXobj (id, name, mode, parent_ptr)
{
   int ref_mode;

   _arr_size = 0;
   if (OMget_obj_ref_mode(obj_id(OM_OBJ_RD),&ref_mode) == 1 &&
			  ref_mode == OM_OBJ_VAL)
      is_value = 1;
   else
      is_value = 0;
}

/* 64-bit porting. Only Modified Internally */
OMXgroup_array::~OMXgroup_array()
{
   xp_long i;
   for (i = 0; i < _arr_size; i++) {
      _arr_values[i]->free_class_ptr();
      delete _arr_values[i];
   }

   if (_arr_size) {
      delete [] _arr_values;
   }
}
