/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/v/dmap.v#1 $
*/
flibrary+global DMAP <
   build_dir="dmap",
   build_cmd="$(MAKE)",
   indexed=1,
   need_cxx=1,
   link_files="-ldmap -ldbxx"
> {

/* Colors are represented in RGB Alpha space where values are 
   normalized from 0.0 to 1.0.
*/
group+OPort DmapColor<cxx_name="CXX_DmapColor"> {
  double  v1;   /* Alpha component of Color */
  double  v2;   /* 1st component of Color */
  double  v3;   /* 2nd component of Color */
  double  v4;   /* 3rd component of Color */
};

/* A DatamapValue is used to relate a specific value in a data range
   to a specific color.
*/
DmapColor DatamapValue<export_all=1> {
  double  value;     /* Value of an entry in a Datamap */
};

/* A ColorModel provides a means to map from the native RGB Alpha
   color model to another one. It also provides support for editors
   that use a ColorModel to edit a color.
*/
group+OPort ColorModel<export_all=1> {
  string model;            /* RGB, CMY, HSV &c */
  string v1Label;          /* Label for v1 */
  string v2Label;          /* Label for v2 */
  string v3Label;          /* Label for v3 */
  string v4Label;          /* Label for v4 */
  func convertToRGB;
  func convertFromRGB;
};

ColorModel RGBcolorModel {
  model = "RGB";
  v1Label = "Alpha";
  v2Label = "Red";
  v3Label = "Green";
  v4Label = "Blue";
  func convertToRGB = "convertRGBtoRGB";
  func convertfromRGB = "convertRGBtoRGB";
};

ColorModel HSVcolorModel {
  model = "HSV";
  v1Label = "Alpha";
  v2Label = "Hue";
  v3Label = "Saturation";
  v4Label = "Value";
  func convertToRGB = "convertHSVtoRGB";
  func convertfromRGB = "convertRGBtoHSV";
};

group ColorModels {
  int numColorModels => array_size(models);
  ColorModel+Port2 &models[];
};

/* A DataRangeModel determines how a curve for a data 
   range will be calculated.
*/
group+OPort DataRangeModel<export_all=1> {
  string model;
  func interpolate;
};

/* The colormap generated will have values that change linearly over
 * the range.
 */
DataRangeModel LinearRange {
  model = "LinearRange";
  func interpolate = "interpolateLinear";
};

/* The colormap generated will have values that echo the values
 * in the control points array.
 */
DataRangeModel StepRange {
  model = "StepRange";
  func interpolate = "interpolateStep";
};

group DataValues {
  int numValues => array_size(dataVals);
  float+Port2 &dataVals[];
};

group RangeModels {
  int numRangeModels => array_size(models);
  DataRangeModel+Port2 &models[];
};

/* A DataRange defines a color table lookup for a region of data. Lookups 
   into the table are made by linearly scaling a data value into the size of
   of the table.
   
   Each range is defined by its bounds, the number of elements in its
   table of colors and its table of colors.
*/
group+OPort DataRangeTempl {
  double+IPort2+req minimum;
  double+IPort2+req maximum;
  int+IPort2 size;
  DataRangeModel+IPort2 &v1Model;
  DataRangeModel+IPort2 &v2Model;
  DataRangeModel+IPort2 &v3Model;
  DataRangeModel+IPort2 &v4Model;
  int numControlPoints;
  DatamapValue+IPort2 &controlPoints[];
};

DataRangeTempl DataRange {
   numControlPoints => array_size(controlPoints);

   /* Subobjects to control the min and max of the
      range from the user interface.
   */
   float DataMinValue;
   float UIMinValue;
   float DataMaxValue;
   float UIMaxValue;
   int selectValues;
   DataValues MinValues {
     dataVals => { DataMinValue, UIMinValue };
   };
   DataValues MaxValues {
     dataVals => { DataMaxValue, UIMaxValue };
   };

   /* Subobjects to control if various UI widgets
      are active for editing.
   */
   int minActive;
   int maxActive;
   int sizeActive;

   /* Subobjects to control the type of range of
      each component from the user interface.
   */
   LinearRange LinearRange;
   StepRange StepRange;
   int selectAlphaRange;
   int selectColorRange;
   RangeModels RangeModels {
      models => { LinearRange, StepRange };
   };
};

group+Port DatamapTempl<NEcolor0=0xffff00> {
  /* The following two variables are used so dependencies
   * work properly.
   */
  float+Port2+read dataMin;
  float+Port2+read dataMax;

  long numRanges;
  DataRangeTempl+IPort2+read &ranges[];
  ColorModel+IPort &colorModel;
  ptr+nosave+nonotify localPtr<NEvisible=0>;

  func+nosave getColors<NEvisible=0>;
  func+nosave getARGBs<NEvisible=0>;
  func+nosave getMinMax<NEvisible=0>;
};

DatamapTempl Datamap {
  numRanges => array_size(ranges);

  /* Subobjects to control the color model
     from the user interface.
  */
  HSVcolorModel HSVcolorModel<export_all=1>;
  RGBcolorModel RGBcolorModel<export_all=1>;

  int currentColorModel = 0;
  ColorModels ColorModels {
     models => { HSVcolorModel, RGBcolorModel };
  };

  /* Subobject that controls if the datamap is
     editable - ranges can be added or deleted.
  */
  int editable;

  /* Initial values for min/max and color model. */
  dataMin = 0.0;
  dataMax = 255.0;
  colorModel => ColorModels.models[currentColorModel];

  localPtr+write;
  method+notify_inst init<NEvisible=0> = "DmapInstance";
  method+notify_val update<NEvisible=0> = "DmapUpdate";
  method+notify_deinst delete<NEvisible=0> = "DmapDelete";
};

module Dmap2Image {
   // The datamap to be visualized by turning it into an image
   DatamapTempl+IPort2 &dmap;

   // Dimensions of the output image;
   int+IPort2 width  = 100;
   int+IPort2 height =  20;

   // 0 means ordinary dmap,
   // 1 means checkerboard underlay to show transparency
   int+IPort2 Mode = 0;

   omethod+req+notify_inst update (
      width+read+notify+req,
      height+read+notify+req,
      Mode+read+notify+req,
      dmap+read+notify+req,
      out+write
   ) = "Dmap2Image_update";

   FLD.Image_ARGB+OPort2 out;
};

module Dmap2Colors {
   // Datamap to sample
   DatamapTempl+IPort2  &dmap;
   // sample datamap at these values 
   float+IPort2         values[];

   omethod+req+notify_inst update (
      dmap+read+notify+req,
      values+read+notify+req,
      colors+write
    ) = "Dmap2Colors_update";

    // Array of RGB (0-1.0) colors.  Size will be
    // 3 times the length of the 'values' array.
    float+OPort2        colors[];
};


/* Build a group that will restrict what subelements
   of the datamap actually cause a notification of
   the editor.
*/
group DatamapNotifyTempl {
   group ranges[] {
      double minimum;
      double maximum;
   };
};

/* Templates for the Datamap Editor module. */
group DatamapEditTempl  {
   DatamapNotifyTempl+IPort2 &dmap_in;
   group *dmapLibrary;
   int curDmapName;
   int controlPointModel;
   float+IPort2 &controlPointData[][];
   int addRange;
   int deleteRange;
   omethod+notify_inst init_func = "DMAPeditor_init";
   omethod+notify_val upd_func = "DMAPeditor";
};

DatamapEditTempl DatamapEdit {
   string dmapNames[];
   long curRange;
};

macro DefaultLinearRange {
  DatamapValue minvalue {
    v1 = 0.0;
    v2 = 0.66;
    v3 = 1.0;
    v4 = 1.0;
  };
  DatamapValue maxvalue {
    v1 = 1.0;
    v2 = 0.0;
    v3 = 1.0;
    v4 = 1.0;
  };

  DataRange+OPort2 DataRange {
    minimum => MinValues.dataVals[selectValues];
    maximum => MaxValues.dataVals[selectValues];
    size = 256;
    v1Model => LinearRange;
    v2Model => LinearRange;
    v3Model => LinearRange;
    v4Model => LinearRange;
    controlPoints => { minvalue, maxvalue };

    DataMinValue = 0;
    UIMinValue = 0;
    DataMaxValue = 255.0;
    UIMaxValue = 255.0;
    minActive = 1;
    maxActive = 1;
    sizeActive = 1;
    selectValues = 0;
    selectAlphaRange = 0;
    selectColorRange = 0;
  };
};

macro DefaultStepRange {
  DatamapValue values[5] {
    !values[0] {	/* red */
      v1 = 1.0;
      v2 = 0.0;
      v3 = 1.0;
      v4 = 1.0;
    };
    !values[1] { 	/* yellow */
      v1 = 1.0;
      v2 = 0.16;
      v3 = 1.0;
      v4 = 1.0;
    };
    !values[2] { 	/* green */
      v1 = 1.0;
      v2 = 0.33;
      v3 = 1.0;
      v4 = 1.0;
    };
    !values[3] { 	/* blue */
      v1 = 1.0;
      v2 = 0.66;
      v3 = 1.0;
      v4 = 1.0;
    };
    !values[4] { 	/* magenta */
      v1 = 1.0;
      v2 = 0.83;
      v3 = 1.0;
      v4 = 1.0;
    };
  };

  DataRange+OPort2 DataRange {
    minimum => MinValues.dataVals[selectValues];
    maximum => MaxValues.dataVals[selectValues];
    size = 4;
    v1Model => StepRange;
    v2Model => StepRange;
    v3Model => StepRange;
    v4Model => StepRange;
    controlPoints => values;

    DataMinValue = 0;
    UIMinValue = 0;
    DataMaxValue = 255.0;
    UIMaxValue = 255.0;
    minActive = 1;
    maxActive = 1;
    sizeActive = 1;
    selectValues = 0;
    selectAlphaRange = 0;
    selectColorRange = 0;
  };
};

DataRange DefaultDataRange {
  minimum => MinValues.dataVals[selectValues];
  maximum => MaxValues.dataVals[selectValues];
  size = 256;
  v1Model => RangeModels.models[selectAlphaRange];
  v2Model => RangeModels.models[selectColorRange];
  v3Model => RangeModels.models[selectColorRange];
  v4Model => RangeModels.models[selectColorRange];

  DataMinValue = 0;
  UIMinValue = 0;
  DataMaxValue = 255.0;
  UIMaxValue = 255.0;
  minActive = 1;
  maxActive = 1;
  sizeActive = 1;
  selectValues = 0;
  selectAlphaRange = 0;
  selectColorRange = 0;
};

macro+Datamap DefaultDatamap<NEvisible=0> {
  DatamapValue minvalue {
    v1 = 0.0;
    v2 = 0.66;
    v3 = 1.0;
    v4 = 1.0;
  };
  DatamapValue maxvalue {
    v1 = 1.0;
    v2 = 0.0;
    v3 = 1.0;
    v4 = 1.0;
  };

  DataRange DataRange {
    minimum => MinValues.dataVals[selectValues];
    maximum => MaxValues.dataVals[selectValues];
    size = 256;
    v1Model => RangeModels.models[selectAlphaRange];
    v2Model => RangeModels.models[selectColorRange];
    v3Model => RangeModels.models[selectColorRange];
    v4Model => RangeModels.models[selectColorRange];
    controlPoints => { <-.minvalue, <-.maxvalue};

    // See the comments in DMAP.DefaultLinear for an
    // explaination about the need for the "0 +"
    DataMinValue => 0 + <-.dataMin;
    UIMinValue = 0;
    DataMaxValue => 0 + <-.dataMax;
    UIMaxValue = 255.0;
    minActive = 1;
    maxActive = 1;
    sizeActive = 1;
    selectValues = 0;
    selectAlphaRange = 0;
    selectColorRange = 0;
  };
  editable = 0;
  currentColorModel = 0;
  dataMin = 0;
  dataMax = 255.0;
  colorModel => ColorModels.models[currentColorModel];
  ranges+IPort => { DataRange };
};

macro+Datamap GreyScaleDatamap<NEvisible=0> {
  DatamapValue minvalue {
    v1 = 0.0;
    v2 = 0.0;
    v3 = 0.0;
    v4 = 0.0;
  };
  DatamapValue maxvalue {
    v1 = 1.0;
    v2 = 0.0;
    v3 = 0.0;
    v4 = 1.0;
  };

  DataRange DataRange {
    minimum => MinValues.dataVals[selectValues];
    maximum => MaxValues.dataVals[selectValues];
    size = 256;
    v1Model => RangeModels.models[selectAlphaRange];
    v2Model => RangeModels.models[selectColorRange];
    v3Model => RangeModels.models[selectColorRange];
    v4Model => RangeModels.models[selectColorRange];
    controlPoints => { <-.minvalue, <-.maxvalue};

    DataMinValue => 0 + <-.dataMin;
    UIMinValue = 0;
    DataMaxValue => 0 + <-.dataMax;
    UIMaxValue = 255.0;
    minActive = 1;
    maxActive = 1;
    sizeActive = 1;
    selectValues = 0;
    selectAlphaRange = 0;
    selectColorRange = 0;
  };

  editable = 0;
  currentColorModel = 0;
  dataMin = 0;
  dataMax = 255.0;
  colorModel => ColorModels.models[currentColorModel];
  ranges+IPort => { DataRange };
};

macro+Datamap RedGreenDatamap<NEvisible=0> {
  DatamapValue minvalue {
    v1 = 0.0;
    v2 = 0.0;
    v3 = 1.0;
    v4 = 1.0;
  };
  DatamapValue maxvalue {
    v1 = 1.0;
    v2 = 0.33;
    v3 = 1.0;
    v4 = 1.0;
  };

  DataRange DataRange {
    minimum => MinValues.dataVals[selectValues];
    maximum => MaxValues.dataVals[selectValues];
    size = 256;
    v1Model => RangeModels.models[selectAlphaRange];
    v2Model => RangeModels.models[selectColorRange];
    v3Model => RangeModels.models[selectColorRange];
    v4Model => RangeModels.models[selectColorRange];
    controlPoints => { <-.minvalue, <-.maxvalue};

    DataMinValue => 0 + <-.dataMin;
    UIMinValue = 0;
    DataMaxValue => 0 + <-.dataMax;
    UIMaxValue = 255.0;
    minActive = 1;
    maxActive = 1;
    sizeActive = 1;
    selectValues = 0;
    selectAlphaRange = 0;
    selectColorRange = 0;
  };

  editable = 0;
  currentColorModel = 0;
  dataMin = 0;
  dataMax = 255.0;
  colorModel => ColorModels.models[currentColorModel];
  ranges+IPort => { DataRange };
};

macro+Datamap GreenBlueDatamap<NEvisible=0> {
  DatamapValue minvalue {
    v1 = 0.0;
    v2 = 0.33;
    v3 = 1.0;
    v4 = 1.0;
  };
  DatamapValue maxvalue {
    v1 = 1.0;
    v2 = 0.66;
    v3 = 1.0;
    v4 = 1.0;
  };

  DataRange DataRange {
    minimum => MinValues.dataVals[selectValues];
    maximum => MaxValues.dataVals[selectValues];
    size = 256;
    v1Model => RangeModels.models[selectAlphaRange];
    v2Model => RangeModels.models[selectColorRange];
    v3Model => RangeModels.models[selectColorRange];
    v4Model => RangeModels.models[selectColorRange];
    controlPoints => { <-.minvalue, <-.maxvalue};

    DataMinValue => 0 + <-.dataMin;
    UIMinValue = 0;
    DataMaxValue => 0 + <-.dataMax;
    UIMaxValue = 255.0;
    minActive = 1;
    maxActive = 1;
    sizeActive = 1;
    selectValues = 0;
    selectAlphaRange = 0;
    selectColorRange = 0;
  };

  editable = 0;
  currentColorModel = 0;
  dataMin = 0;
  dataMax = 255.0;
  colorModel => ColorModels.models[currentColorModel];
  ranges+IPort => { DataRange };
};

macro+Datamap BlueRedDatamap<NEvisible=0> {
  DatamapValue minvalue {
    v1 = 0.0;
    v2 = 0.66;
    v3 = 1.0;
    v4 = 1.0;
  };
  DatamapValue maxvalue {
    v1 = 1.0;
    v2 = 1.0;
    v3 = 1.0;
    v4 = 1.0;
  };

  DataRange DataRange {
    minimum => MinValues.dataVals[selectValues];
    maximum => MaxValues.dataVals[selectValues];
    size = 256;
    v1Model => RangeModels.models[selectAlphaRange];
    v2Model => RangeModels.models[selectColorRange];
    v3Model => RangeModels.models[selectColorRange];
    v4Model => RangeModels.models[selectColorRange];
    controlPoints => { <-.minvalue, <-.maxvalue};

    DataMinValue => 0 + <-.dataMin;
    UIMinValue = 0;
    DataMaxValue => 0 + <-.dataMax;
    UIMaxValue = 255.0;
    minActive = 1;
    maxActive = 1;
    sizeActive = 1;
    selectValues = 0;
    selectAlphaRange = 0;
    selectColorRange = 0;
  };

  editable = 0;
  currentColorModel = 0;
  dataMin = 0;
  dataMax = 255.0;
  colorModel => ColorModels.models[currentColorModel];
  ranges+IPort => { DataRange };
};

macro+Datamap VolRenderDatamap<NEvisible=0> {

  /* Three controls points for the two ranges. The
     middle control point will be shared between the
     two ranges.

     We want to always keep a linear color ramp from
     minvalue to maxvalue but allow the midvalue to
     move around. This is specifically to allow a
     reasonable manipulation of alpha without
     changing the colors. Of course all this would
     be a lot easier if color and alpha were kept
     separately.
  */
  float ratios[2] => { 0.5, LowerRange.UIMaxValue / dataMax };
  DatamapValue minvalue {
    v1 = 0.0;
    v2 = 0.66;
    v3 = 1.0;
    v4 = 1.0;
  };
  DatamapValue midvalue {
    v1 = 0.5;
    v2 => <-.minvalue.v2 + ((<-.maxvalue.v2 - <-.minvalue.v2) * 
	  ratios[LowerRange.selectValues]);
    v3 => <-.minvalue.v3 + ((<-.maxvalue.v3 - <-.minvalue.v3) * 
	  ratios[LowerRange.selectValues]);
    v4 => <-.minvalue.v4 + ((<-.maxvalue.v4 - <-.minvalue.v4) * 
	  ratios[LowerRange.selectValues]);
  };
  DatamapValue maxvalue {
    v1 = 1.0;
    v2 = 0.0;
    v3 = 1.0;
    v4 = 1.0;
  };

  /* Two ranges: LowerRange and UpperRange. This is done so we
     can set the alpha component to Step (instead of Linear) and
     be able to slide the data value for the middle control point.
  */
  DataRange LowerRange {
    minimum => MinValues.dataVals[selectValues];
    maximum => MaxValues.dataVals[selectValues];
    size => 256 * ratios[selectValues];
    v1Model => RangeModels.models[selectAlphaRange];
    v2Model => RangeModels.models[selectColorRange];
    v3Model => RangeModels.models[selectColorRange];
    v4Model => RangeModels.models[selectColorRange];
    controlPoints => { <-.minvalue, <-.midvalue};

    DataMinValue => 0 + <-.dataMin;
    DataMaxValue => <-.dataMin + ((<-.dataMax - <-.dataMin) *
		    ratios[selectValues]);
    UIMinValue = 0;
    UIMaxValue = 127.5;
    minActive = 1;
    maxActive = 0;
    sizeActive = 0;
    selectValues = 1;
    selectAlphaRange = 1;
    selectColorRange = 0;
  };

  DataRange UpperRange {
    minimum => MinValues.dataVals[selectValues];
    maximum => MaxValues.dataVals[selectValues];
    size => 256 - LowerRange.size;
    v1Model => RangeModels.models[selectAlphaRange];
    v2Model => RangeModels.models[selectColorRange];
    v3Model => RangeModels.models[selectColorRange];
    v4Model => RangeModels.models[selectColorRange];
    controlPoints => { <-.midvalue, <-.maxvalue};

    DataMinValue => LowerRange.DataMaxValue;
    DataMaxValue => 0 + <-.dataMax;
    UIMinValue =>  <-.LowerRange.UIMaxValue;
    UIMaxValue =  255;
    minActive = 0;
    maxActive = 1;
    sizeActive = 0;
    selectValues => <-.LowerRange.selectValues;
    selectAlphaRange = 0;
    selectColorRange = 0;
  };

  editable = 0;
  currentColorModel = 0;
  dataMin = 0;
  dataMax = 255.0;
  colorModel => ColorModels.models[currentColorModel];
  ranges+IPort => { LowerRange, UpperRange };
};

macro+Datamap HotMetalDatamap<NEvisible=0> {

  /* Four controls points for the three ranges. The
     middle control points will be shared between the
     ranges.
  */
  float LMratios[2] => { 0.5, LowerRange.UIMaxValue / dataMax };
  float MUratios[2] => { 0.75, MidRange.UIMaxValue / dataMax };
  DatamapValue value#1 {
    v1 = 0.0;
    v2 = 0.0;
    v3 = 0.0;
    v4 = 0.0;
  };
  DatamapValue value#2 {
    v1 = 0.0;
    v2 = 1.0;
    v3 = 0.0;
    v4 = 0.0;
  };
  DatamapValue value#3 {
    v1 = 0.0;
    v2 = 1.0;
    v3 = 0.5;
    v4 = 0.0;
  };
  DatamapValue value#4 {
    v1 = 0.0;
    v2 = 1.0;
    v3 = 1.0;
    v4 = 1.0;
  };

  /* Three ranges: LowerRange, MidRange and UpperRange.
  */
  DataRange LowerRange {
    minimum => MinValues.dataVals[selectValues];
    maximum => MaxValues.dataVals[selectValues];
    size => 256 * LMratios[selectValues];
    v1Model => RangeModels.models[selectAlphaRange];
    v2Model => RangeModels.models[selectColorRange];
    v3Model => RangeModels.models[selectColorRange];
    v4Model => RangeModels.models[selectColorRange];
    controlPoints => { <-.value#1, <-.value#2 };

    DataMinValue => 0 + <-.dataMin;
    DataMaxValue => <-.dataMin + ((<-.dataMax - <-.dataMin) *
		    LMratios[selectValues]);
    UIMinValue = 0;
    UIMaxValue = 127.5;
    minActive = 1;
    maxActive = 1;
    sizeActive = 0;
    selectValues = 0;
    selectAlphaRange = 0;
    selectColorRange = 0;
  };
  DataRange MidRange {
    minimum => MinValues.dataVals[selectValues];
    maximum => MaxValues.dataVals[selectValues];
    size => 256 - (256 * MUratios[selectValues]);
    v1Model => RangeModels.models[selectAlphaRange];
    v2Model => RangeModels.models[selectColorRange];
    v3Model => RangeModels.models[selectColorRange];
    v4Model => RangeModels.models[selectColorRange];
    controlPoints => { <-.value#2, <-.value#3 };

    DataMinValue => LowerRange.DataMaxValue;
    DataMaxValue => <-.dataMin + ((<-.dataMax - <-.dataMin) *
		    MUratios[selectValues]);
    UIMinValue =>  <-.LowerRange.UIMaxValue;
    UIMaxValue = 191.25;
    minActive = 1;
    maxActive = 1;
    sizeActive = 0;
    selectValues => <-.LowerRange.selectValues;
    selectAlphaRange = 0;
    selectColorRange = 0;
  };
  DataRange UpperRange {
    minimum => MinValues.dataVals[selectValues];
    maximum => MaxValues.dataVals[selectValues];
    size => 256 - (256 * MUratios[selectValues]);
    v1Model => RangeModels.models[selectAlphaRange];
    v2Model => RangeModels.models[selectColorRange];
    v3Model => RangeModels.models[selectColorRange];
    v4Model => RangeModels.models[selectColorRange];
    controlPoints => { <-.value#3, <-.value#4 };

    DataMinValue => MidRange.DataMaxValue;
    DataMaxValue => 0 + <-.dataMax;
    UIMinValue =>  <-.MidRange.UIMaxValue;
    UIMaxValue = 255.0;
    minActive = 1;
    maxActive = 1;
    sizeActive = 0;
    selectValues => <-.LowerRange.selectValues;
    selectAlphaRange = 0;
    selectColorRange = 0;
  };

  editable = 0;
  currentColorModel = 1;
  dataMin = 0;
  dataMax = 255.0;
  colorModel => ColorModels.models[currentColorModel];
  ranges+IPort => { LowerRange, MidRange, UpperRange };
};

macro+Datamap CyanYellowRedDatamap<NEvisible=0> {

  /* Three controls points for the two ranges. The
     middle control point will be shared between the
     ranges.
  */
  float ratios[2] => { 0.5, LowerRange.UIMaxValue / dataMax };
  DatamapValue value#1 {
    v1 = 0.0;
    v2 = 0.0;
    v3 = 1.0;
    v4 = 1.0;
  };
  DatamapValue value#2 {
    v1 = 0.0;
    v2 = 1.0;
    v3 = 1.0;
    v4 = 0.0;
  };
  DatamapValue value#3 {
    v1 = 0.0;
    v2 = 1.0;
    v3 = 0.0;
    v4 = 0.0;
  };

  /* Two ranges: LowerRange and UpperRange.
  */
  DataRange LowerRange {
    minimum => MinValues.dataVals[selectValues];
    maximum => MaxValues.dataVals[selectValues];
    size => 256 * ratios[selectValues];
    v1Model => RangeModels.models[selectAlphaRange];
    v2Model => RangeModels.models[selectColorRange];
    v3Model => RangeModels.models[selectColorRange];
    v4Model => RangeModels.models[selectColorRange];
    controlPoints => { <-.value#1, <-.value#2 };

    DataMinValue => 0 + <-.dataMin;
    DataMaxValue => <-.dataMin + ((<-.dataMax - <-.dataMin) *
		    ratios[selectValues]);
    UIMinValue = 0;
    UIMaxValue = 127.5;
    minActive = 1;
    maxActive = 1;
    sizeActive = 0;
    selectValues = 0;
    selectAlphaRange = 0;
    selectColorRange = 0;
  };
  DataRange UpperRange {
    minimum => MinValues.dataVals[selectValues];
    maximum => MaxValues.dataVals[selectValues];
    size => 256 - <-.LowerRange.size;
    v1Model => RangeModels.models[selectAlphaRange];
    v2Model => RangeModels.models[selectColorRange];
    v3Model => RangeModels.models[selectColorRange];
    v4Model => RangeModels.models[selectColorRange];
    controlPoints => { <-.value#2, <-.value#3 };

    DataMinValue => LowerRange.DataMaxValue;
    DataMaxValue => 0 + <-.dataMax;
    UIMinValue =>  <-.LowerRange.UIMaxValue;
    UIMaxValue = 255.0;
    minActive = 1;
    maxActive = 1;
    sizeActive = 0;
    selectValues => <-.LowerRange.selectValues;
    selectAlphaRange = 0;
    selectColorRange = 0;
  };

  editable = 0;
  currentColorModel = 1;
  dataMin = 0;
  dataMax = 255.0;
  colorModel => ColorModels.models[currentColorModel];
  ranges+IPort => { LowerRange, UpperRange };
};


macro+Datamap DefaultStepDatamap<NEvisible=0> {
  DefaultStepRange DefaultStepRange;

  editable = 0;
  currentColorModel = 0;
  dataMin = 0;
  dataMax = 255.0;
  colorModel => ColorModels.models[currentColorModel];
  ranges+IPort => { DefaultStepRange.DataRange };
};

};
