/* mpu_api.h
 * ----------------------------------------------------------------------
 * C++ Class for MPU - API for Modules.
 * ----------------------------------------------------------------------
 * Notes:
 *   Modules require access to MPUsession_info struct, to gain access to
 *   the MPUmodule_api class. Modules should never access the MPKConfig
 *   or the MPUshared classes directly.
 *
 * Contains:
 *   Tokens		 - written to token stream passed to API
 *   Macros		 - used to write tokens and data to token stream
 *   class MPUmodule_api - Manage communication between modules and MPU
 * ----------------------------------------------------------------------
 */

// Must use __MPU_MODS_H__ to prevent inclusion in gd/mpu/mpu_mods.h
// indirectly from fmpc_gen.hxx
#ifndef __MPU_MODS_H__
#define __MPU_MODS_H__

#ifndef MSDOS
#include <semaphore.h>
#else
#include "gd/mpu/w32_pthread_macros.h"
#include "gd/mpu/w32_pthread_types.h"
#endif

#include "avs/gd.h"

/* event manager settings file header with MPE version and EVM version */
#define MPU_EVM_HEADER_LEN 31
#define MPU_EVM_FILE_HEADER    "MPE/EVM MPEv6.0 EVMv1.0       $"

/* additional 30 character datablock for future use	 */
#define MPU_EVM_DATABLOCK_LEN 31
#define MPU_EVM_FILE_DATABLOCK "..............................$"

class MPUshared;

typedef enum {
  MPU_EV_TYPE_INT=1,
  MPU_EV_TYPE_REAL,
  MPU_EV_TYPE_STRING
} MPUev_handler_param_types;

typedef enum {
  // Token stream markers
  MPU_EV_LABEL_NONE=0,
  MPU_EV_LABEL_END,

  // Data
  MPU_EV_LABEL_NUM_BUTTONS,
  MPU_EV_LABEL_NUM_VALUATORS,
  
  MPU_EV_LABEL_NUM_HANDLERS,
  MPU_EV_LABEL_NUM_MAPPINGS,
  MPU_EV_LABEL_NUM_CONFIGS,
  MPU_EV_LABEL_ACTIVE_CONFIG,

  MPU_EV_LABEL_HANDLER_ID,
  MPU_EV_LABEL_HANDLER_NAME,
  MPU_EV_LABEL_HANDLER_NUM_MODES,
  MPU_EV_LABEL_HANDLER_NUM_PARAMS,
  
  MPU_EV_LABEL_MODE_ID,
  MPU_EV_LABEL_MODE_NAME,
  
  MPU_EV_LABEL_PARAM_ID,
  MPU_EV_LABEL_PARAM_NAME,
  MPU_EV_LABEL_PARAM_TYPE,
  MPU_EV_LABEL_PARAM_INT_VALUE,
  MPU_EV_LABEL_PARAM_REAL_VALUE,

  MPU_EV_LABEL_MAPPING_ID,
  MPU_EV_LABEL_MAPPING_NAME,
  MPU_EV_LABEL_MAPPING_ACTIVE,
  MPU_EV_LABEL_MAPPING_HDL_ID,
  MPU_EV_LABEL_MAPPING_HDL_MODE,
  MPU_EV_LABEL_MAPPING_REQUIRED,
  MPU_EV_LABEL_MAPPING_INVALID,
  MPU_EV_LABEL_MAPPING_NUM_PARAMS,
  MPU_EV_LABEL_MAPPING_PARAM_ID,
  MPU_EV_LABEL_MAPPING_PARAM_NAME,
  MPU_EV_LABEL_MAPPING_PARAM_TYPE,
  MPU_EV_LABEL_MAPPING_PARAM_INT_VALUE,
  MPU_EV_LABEL_MAPPING_PARAM_REAL_VALUE,
  MPU_EV_LABEL_MAPPING_DETAIL,
  
  MPU_EV_LABEL_CONFIG_ID,
  MPU_EV_LABEL_CONFIG_NAME,
  MPU_EV_LABEL_CONFIG_NUM_MAPPINGS,
  MPU_EV_LABEL_CONFIG_MAPPING_ID,
  MPU_EV_LABEL_CONFIG_ACTIVE,	// Boolean - not same as MPU_EV_LABEL_ACTIVE_CONFIG
				// above (which is an id.)

  // Actions
  MPU_EV_LABEL_SUBMIT_MAPPING,
  MPU_EV_LABEL_SET_MAPPING_ACTIVE,
  MPU_EV_LABEL_SET_MAPPING_PARAMETER,
  MPU_EV_LABEL_SUBMIT_CONFIG,
  MPU_EV_LABEL_SELECT_CONFIG,
  MPU_EV_LABEL_SELECT_CONFIG_ID,

  MPU_EV_LABEL_EDIT_MAPPING,

  MPU_EV_LABEL_REMOVE_MAPPING,
  MPU_EV_LABEL_DELETE_MAPPING,

  MPU_EV_LABEL_DELETE_ACTIVE_CONFIG,
  MPU_EV_LABEL_DELETE_CONFIG,
  
  MPU_EV_LABEL_LOAD_SETTINGS,
  MPU_EV_LABEL_SAVE_SETTINGS,

  // KGT Extras
  MPU_EV_MISC_LABEL_VALUATOR_FILTER,

  // Extensions
  MPU_EV_EXT_LABEL_XFORM

} MPUev_info_labels;


#define MPU_EV_SIZE_LABEL() { *recc += sizeof(unsigned int); }
#define MPU_EV_SIZE_LABEL_INT()  { *recc += (sizeof(unsigned int) + sizeof(int)); }
#define MPU_EV_SIZE_LABEL_UINT() { *recc += 2*sizeof(unsigned int); }
#define MPU_EV_SIZE_LABEL_FLOAT() { *recc += (sizeof(unsigned int)+sizeof(float)); }
#define MPU_EV_SIZE_INT() { *recc += sizeof(int); }
#define MPU_EV_SIZE_UINT() { *recc += sizeof(unsigned int); }
#define MPU_EV_SIZE_FLOAT() { *recc += sizeof(float); }
#define MPU_EV_SIZE_FLOAT_MATRIX() { *recc += 16*sizeof(float); }
#define MPU_EV_SIZE_STRING( str ) {  *recc += (sizeof(unsigned int) + ((strlen(str)+1)*sizeof(char))); }
#define MPU_EV_SIZE_LABEL_STRING( str ) { \
  *recc += (2*sizeof(unsigned int) + ((strlen(str)+1)*sizeof(char))); \
}

#define MPU_EV_STORE_LABEL( label ) { \
  unsigned int recv_label = label; \
  if ( byteswap ) \
     recv_label = SWAP_UINT(recv_label); \
  memcpy( (*recv)+recv_index, &recv_label, sizeof(unsigned int) ); \
  recv_index += sizeof(unsigned int); \
}

#define MPU_EV_STORE_UINT( uivalue ) { \
  unsigned int recv_label = uivalue; \
  if ( byteswap ) \
     recv_label = SWAP_UINT(recv_label); \
  memcpy( (*recv)+recv_index, &recv_label, sizeof(unsigned int) ); \
  recv_index += sizeof(unsigned int); \
}

#define MPU_EV_STORE_FLOAT( fvalue ) { \
  float recv_label = fvalue; \
  if ( byteswap ) \
     recv_label = SWAP_FLOAT(recv_label); \
  memcpy( (*recv)+recv_index, &recv_label, sizeof(float) ); \
  recv_index += sizeof(float); \
}

#define MPU_EV_STORE_FLOAT_MATRIX( fmatrix ) { \
  float *_fmat = (float *)fmatrix; \
  if ( byteswap ) { \
    float swap_mat[4][4]; \
    float recv_entry; \
    int _r_, _c_; \
    for ( _r_ = 0; _r_ < 4; _r_++ ) { \
      for ( _c_ = 0; _c_ < 4; _c_++ ) { \
         recv_entry = fmatrix[_r_][_c_]; \
         swap_mat[_r_][_c_] = SWAP_FLOAT(recv_entry); \
      } \
    } \
    _fmat = (float *)swap_mat; \
  } \
  memcpy( (*recv)+recv_index, (void *)_fmat, 16*sizeof(float) ); \
  recv_index += 16*sizeof(float); \
}

#define MPU_EV_STORE_STRING( str_value ) { \
  unsigned int len, lenswap; \
  len = strlen(str_value)+1; \
  if ( byteswap ) \
     lenswap = SWAP_UINT(len); \
  else \
     lenswap = len; \
  memcpy( (*recv)+recv_index, &lenswap, sizeof(unsigned int) ); \
  recv_index += sizeof(unsigned int); \
  \
  memcpy( (*recv)+recv_index, str_value, sizeof(char) * len ); \
  recv_index += (sizeof(char) * len); \
}

#define MPU_EV_STORE_LABEL_INT( label, ivalue ) { \
  unsigned int recv_label = label; \
  if ( byteswap ) \
     recv_label = SWAP_INT(recv_label); \
  memcpy( (*recv)+recv_index, &recv_label, sizeof(unsigned int) ); \
  recv_index += sizeof(unsigned int); \
  int val = ivalue; \
  if ( byteswap ) \
     val = SWAP_INT(val); \
  memcpy( (*recv)+recv_index, &val, sizeof(int) ); \
  recv_index += sizeof(int); \
}

#define MPU_EV_STORE_LABEL_UINT( label, uivalue ) { \
  unsigned int recv_label = label; \
  if ( byteswap ) \
     recv_label = SWAP_UINT(recv_label); \
  memcpy( (*recv)+recv_index, &recv_label, sizeof(unsigned int) ); \
  recv_index += sizeof(unsigned int); \
  unsigned int val = uivalue; \
  if ( byteswap ) \
     val = SWAP_UINT(val); \
  memcpy( (*recv)+recv_index, &val, sizeof(unsigned int) ); \
  recv_index += sizeof(unsigned int); \
}

#define MPU_EV_STORE_LABEL_FLOAT( label, fvalue ) { \
  unsigned int recv_label = label; \
  if ( byteswap ) \
     recv_label = SWAP_UINT(recv_label); \
  memcpy( (*recv)+recv_index, &recv_label, sizeof(unsigned int) ); \
  recv_index += sizeof(unsigned int); \
  float val = fvalue; \
  if ( byteswap ) \
     val = SWAP_FLOAT(val); \
  memcpy( (*recv)+recv_index, &val, sizeof(float) ); \
  recv_index += sizeof(float); \
}

#define MPU_EV_STORE_LABEL_STRING( label, str_value ) { \
  unsigned int recv_label = label; \
  if ( byteswap ) \
     recv_label = SWAP_UINT(recv_label); \
  memcpy( (*recv)+recv_index, &recv_label, sizeof(unsigned int) ); \
  recv_index += sizeof(unsigned int); \
  \
  unsigned int len, lenswap; \
  len = strlen(str_value)+1; \
  if ( byteswap ) \
    lenswap = SWAP_UINT(len); \
  else \
    lenswap = len; \
  memcpy( (*recv)+recv_index, &lenswap, sizeof(unsigned int) ); \
  recv_index += sizeof(unsigned int); \
  \
  memcpy( (*recv)+recv_index, str_value, sizeof(char) * len ); \
  recv_index += (sizeof(char) * len); \
}

#define MPU_EV_GET_LABEL( p ) { \
  memcpy( &p, recv+recv_index, sizeof(unsigned int) ); \
  if ( byteswap ) \
     p = SWAP_UINT(p); \
  recv_index += sizeof(unsigned int); \
};  
    
#define MPU_EV_GET_INT_VALUE( p ) { \
  memcpy( &p, recv+recv_index, sizeof(int) ); \
  if ( byteswap ) \
     p = SWAP_INT(p); \
  recv_index += sizeof(int); \
};
  
#define MPU_EV_GET_UINT_VALUE( p ) { \
  memcpy( &p, recv+recv_index, sizeof(unsigned int) ); \
  if ( byteswap ) \
     p = SWAP_UINT(p); \
  recv_index += sizeof(unsigned int); \
};

#define MPU_EV_GET_FLOAT_VALUE( p ) { \
  memcpy( &p, recv+recv_index, sizeof(float) ); \
  if ( byteswap ) \
     p = SWAP_FLOAT(p); \
  recv_index += sizeof(float); \
};
  
#define MPU_EV_GET_FLOAT_MATRIX( fmatrix ) { \
  memcpy( (void *)fmatrix, (void *)(recv+recv_index), 16*sizeof(float) ); \
  if ( byteswap ) { \
    float recv_entry; \
    int _r_, _c_; \
    for ( _r_ = 0; _r_ < 4; _r_++ ) { \
      for ( _c_ = 0; _c_ < 4; _c_++ ) { \
         recv_entry = fmatrix[_r_][_c_]; \
         fmatrix[_r_][_c_] = SWAP_FLOAT(recv_entry); \
      } \
    } \
  } \
  recv_index += 16*sizeof(float); \
}

#define MPU_EV_GET_STRING( p, l ) { \
  memcpy( &l, recv+recv_index, sizeof(unsigned int) ); \
  recv_index += sizeof(unsigned int); \
  if ( byteswap ) \
     l = SWAP_UINT(l); \
  if( p ) free( (void *)p ); \
  p = (char *)malloc( sizeof(char) * l ); \
  memcpy( p, recv+recv_index, sizeof(char) * l ); \
  recv_index += sizeof(char) * l; \
};


// These next two classes (MPUmodule_api and MPUsession_info) and hash defines
// must match the definitions in the mpu source. See mpu_mods.h and mpu_store.h.

#define MPU_HDL_XP_FLAGS_EXT_REQXFORM 16

class MPUmodule_api {

  struct _MPUsession_info *_session_info; // for private access to MPU API

  // Synchronization mechanisms
  struct {
    sem_t rw_lock;
    sem_t data_ready;
    sem_t data_processed;
    pthread_mutex_t cond_mutex;
    pthread_cond_t  cond_var;
    int initialized;
  } sync;
  
  int recc;
  byte *recv;
  
  float z_translate;
  class MPUserver *socket_server; // Extra 'class' keyword in this file.
  int socket_port;

  OMobj_id fmpmod_id;		// FMP Comm module to update
  void *fmpmod_ptr;		// FMP Comm module ptr

  int byteswap;			// Should we byte swap the token stream
 
public:
  MPUmodule_api( struct _MPUsession_info *si );
  ~MPUmodule_api();

  // Module-MPU API

  // Caller will wait until release() signals they can proceed. The
  // return value can be tested to see why we were released.
  int  wait( void );
  void release( int flag );

  // Keep a copy of the ID of the module that is communicating with
  // this API class.
  void module_id( OMobj_id id );
  OMobj_id module_id( void );

  // Keep a copy of the actual module pointer that is communicating with
  // the API class. Need the gen.hxx file including to use.
  void module_ptr( void *ptr );
  void *module_ptr( void );

  void	test( void );
  int	data_ready( void );
  int	data_processed( void );

  // Used by a module to make data to EMP/FMP classes that have this module_api class.
  // Blocks until that class indicates it has finished with the bytestream array.
  void ev_send_cmd( int nrecc, byte *nrecv );

  // Used by EMP/FMP classes that have this module_api class to read data from module.
  // Releases the blocked module when done.
  void process_data( void );
  
  // Get information from EV Manager class
  int  ev_enquire_manager_init( void );
  void ev_enquire_manager_trackd( int *num_buttons, int *num_valuats );
  void ev_enquire_manager( int *recc, byte **recv );

  // Free token stream alloc'd by above enquire method.
  void ev_manager_dealloc_stream( int *recc, byte **recv );

  // Ask EV manager class to create default config and mappings
  void ev_manager_defaults( void );

  // Unused
  void set_z( float new_z );
  float get_z( void );

  // Set up the socket server to receive wand data
  int create_socket_server( GDview *view );
  int init_socket_server( void (*func)( void * ), void *userdata );
  int get_socket_port( void );

  // State of byteswap flag
  void set_byteswap(int flag);
  int  get_byteswap(void);
  
};

typedef struct _MPUsession_info {
  void		*config;
  MPUshared     *shared;
  MPUmodule_api *modapi;
} MPUsession_info;

#endif
