/* mpu_wand.hxx
 * ----------------------------------------------------------------------
 * Class to manage trackdAPI and state for the wand module -gwl
 * ----------------------------------------------------------------------
 * First release: 
 *   MPE 5.1.1 
 *   MPE 6.0	Updated to use new MPE 6.0 env vars and licensing
 *   MPE 6.2    Updated to use trackdAPI 2.1 for irix/win32/linux
 *
 * PC CLUSTER (win32 only): *****
 *   There are linking problems on Win32 which prevent us linking
 *   libtrackapi. Hence these modules don't do very much - the 
 *   trackd API calls have been #defined out. Users should be using
 *   the new EVM modules with the MPU renderer anyway. If we do
 *   link against libtrackapi then express fails to start - it says
 *   the application has not been initialized correctly.
 *
 * PC CLUSTER (win32 and linux) 
 *   VRCO do not reccommend accessing trackd data using trackdAPI when
 *   using CAVELib anyway. We should modify the class to get data from
 *   CAVELib. We'll need to make the CAVELib config structure availble
 *   to the class somehow. TODO. For the time being these modules won't
 *   do very much! I.e., we disabled trackdAPI support on win32 and linux.
 *
 *   See the class implementation in mpu_wand.[h|c]xx
 * ----------------------------------------------------------------------
 */



// Prevent trackdAPI being used
#if defined(MSDOS) || defined(__linux__)
#define MPU_NO_TRACKDAPI
#else
//#include "trackdAPI.h"
#include "trackdAPI_CC.h" // trackdAPI 2.1
#endif

/* ----------------------------------------------------------------------
 * Constants
 * ----------------------------------------------------------------------
 */


// These constants are OR'd together to indicate which output ports
// should be updated, based on the control algorithm being used. If the
// user has turned on this feature then it prevents the module writing
// zeros to output ports that are not being updated by the control alg.
// For example, a translation-only algorithm does not update the rotation
// output ports. If the user wants to rotate using the trackball, we
// should not write zeros to the rotation output ports.
#define NONE 0
#define TRANS 1
#define ROT 2
#define SCALE 4
#define FLAG1 8

// Set to 1 to retrieve tracker matrix
#define USE_MATRIX 0

// Maximum number of valuators and buttons the modules can cope with.
// This is determined by how many outputs there are on the module.
#define MAX_VALUATORS 3
#define MAX_BUTTONS 4


class MPUWandMod {
protected:
  // trackdAPI class pointers.
#ifndef MPU_NO_TRACKDAPI
  TrackerReader    *tracker;
  ControllerReader *controller;
#endif

  // Setup information
  int	trackerdaemonkey;
  int	controllerdaemonkey;
  int	sensornum;
  int	useenvvars;
  int	controlstyle;
  int	is_initialized;
  int	vrolok;
  int	printsetup;
  int	printdata;
  int	limitoutput;
  int	usetracker;
  int	usecontroller;
  int	usesensor;
  unsigned short outputs_to_update;

  int	reorder_rot_all;	// Should HPR array be reaaranged?
  int	reorder_rot_wand;	// Should HPR array be reaaranged?
  char  *rot_string_all;	// String setting of hpr env var
  char  *rot_string_wand;	// String setting of hpr env var
  float rot_temp[3];		// Temp array to use when swapping rot values.
  short	rot_order_wand[3];	// Indices giving required order of HPR.
  short	rot_order_all[3];	// Indices giving required order of HPR.

  float	trackd_adjusts_wand[12]; // Offsets and scales to apply to XYZ,HPR
  float direction[3];		// Useful for movement algorithms

  int	check_key( int key );	// Simple check that the daemonkey is valid
  int	process_reorder_string( char *str, short *inds );
  void	copy_reorder_string( char **result, char *str );
  int	read_comma_values( char *str, float res[3] );

public:
  MPUWandMod();
  ~MPUWandMod();

  // Physical properties of wand device read from trackd
  int		numSensors;	// We will only read one sensor though!
  int		numButtons;
  int		numValuators;

  // Current values of wand read from trackd at this time step
  float		pos[3];		// Current position of wand
  float		rot[3];		// Current orientation of wand
  float		matrix[4][4];	// Trackd matrix giving position and orientation
				// NB: No 'old_matrix'
  int		buttons[MAX_BUTTONS];		// Status of wand buttons
  float		valuators[MAX_VALUATORS];	// Values of wand valuators

  // Previous values of wand given by trackd saved during previous time step
  float		old_pos[3];	// Previous position of wand
  float		old_rot[3];	// Previous orientation of wand
  int		old_buttons[MAX_BUTTONS];	// Previous values of wand buttons
  float		old_valuators[MAX_VALUATORS];	// Previous values of wand valuators

  // Result of movement algorithms which will be used as module outputs for
  // the current time step.
  float		out_pos[3];	// New position of selected object
  float		out_rot[3];	// New orientation of selected object
  float		out_scale;	// New scale of selected object
  int		out_reset;	// New value of 'reset' flag

  // Zero the arrays
  void reset( void );

  // Read setup information from XP_MPE environment variables.
  void read_env_vars( void );

  // 'Set' Accessors
  void tracker_key( int t_key ) { trackerdaemonkey = t_key; };
  void controller_key( int c_key ) { controllerdaemonkey = c_key; };
  void sensor( int s_num ) { sensornum = s_num; };
  void envflag( int e_flag ) { useenvvars = e_flag; };
  void style( int s_flag ) { controlstyle = s_flag; };
  void reorderrotwand( char *o_str ) { copy_reorder_string(&rot_string_wand, o_str); };
  void print_setup( int p_flag ) { printsetup = p_flag; };
  void print_data( int p_flag ) { printdata = p_flag; };
  void initialized( int i_flag ) { is_initialized = i_flag; };
  void limit_output( int l_flag ) { limitoutput = l_flag; };
  void update_flags( unsigned short u_flags ) { outputs_to_update = u_flags; };
  void use_tracker( int t_flag ) { usetracker = t_flag; };
  void use_sensor( int s_flag ) { usesensor = s_flag; };
  void use_controller( int c_flag ) { usecontroller = c_flag; };

  // 'Get' Accessors
  int tracker_key( void ) { return trackerdaemonkey; };
  int controller_key( void ) { return controllerdaemonkey; };
  int sensor( void ) { return sensornum; };
  int envflag( void ) { return useenvvars; };
  int style( void ) { return controlstyle; };
  char *reorderrotwand( void ) { return rot_string_wand; };
  int print_setup( void ) { return printsetup; };
  int print_data( void ) { return printdata; };
  int initialized( void ) { return is_initialized; };
  int limit_output( void ) { return limitoutput; };
  unsigned short update_flags( void ) { return outputs_to_update; };
  int use_tracker( void ) { return usetracker; };
  int use_sensor( void ) { return usesensor; };
  int use_controller( void ) { return usecontroller; };

  // No 'Set' accessor for license checking
  int siwmil( void ) { return vrolok; };

  // Create trackdAPI classes based on the keys above. Will dealloc any
  // previously allocated classes. Also gets physical device info from
  // trackdAPI (e.g., number of sensors.)
  int setup_tracker( void );
  int setup_controller( void );

  // Is the sensor number valid?
  int sensornum_valid( void ) { return (sensornum>=0 && sensornum<numSensors); };

  // Update XYZ, HPR, button and valutor values from sensor 'sensornum'
  void update_xyzrot( void );
  void update_buttons( void );
  void update_valuators( void );
  void update_matrix( void );

  // Update the output arrays and scale value (i.e., load them with given values.)
  // The movement algorithms may rely on this.
  void update_out_xyzrots( float x, float y, float z,
			   float xr, float yr, float zr,
			   float s );

  // Set the offsets and scales to apply to trackd values - if not
  // read from env vars.
  void update_xyz_scales( float x, float y, float z );
  void update_xyz_offsets( float x, float y, float z );
  void update_rot_scales( float x, float y, float z );
  void update_rot_offsets( float x, float y, float z );

  // Duplicate the xyz, hpr, button and valuator arrays for movement algorithms.
  void save_xyzrot( void );
  void save_valuators( void );
  void save_buttons( void );

  // Setup HPR reordering if required
  void setup_rot_reorder( void );

  // Modify values retrieved by the above functions
  void apply_offsets( void );
  void rot_reorder( void );

  // Print info about current setup
  void print_setup_info( void );
  void print_data_info( void );

  // Movement algorithms. Calculate pos and rot values based on the
  // current and previous configuration of the wand. Several algorithms
  // are available - one similar to the KGT/Morgan State modules, one
  // similar to CAVElib apps and others taht I think might be useful.

  void move_tran_but( float tran_sens, float tran_inc, float ztran_inc );
  void move_rot_but( float rot_sens, float rot_inc );
  void move_tran_val( float tran_sens, float tran_inc, float ztran_inc );
  void move_rot_val( float rot_sens, float rot_inc );
  void move_tran_abs_but( void );
  void move_rot_abs_but( void );
  void move_tran_actual_but( float tran_sens );
  void move_rot_actual_but( float rot_sens );
  void move_kgt( float tran_sens, float tran_inc, float ztran_inc,
		 float rot_sens, float rot_inc );
  void move_tran_bnum( float tran_sens, float tran_inc, float ztran_inc, int but );
  void move_rot_bnum( float rot_sens, float rot_inc, int but );
  void move_tran_drive_rot_bnum( float tran_sens, float rot_inc, int but );
  void move_kgt2( float tran_sens, float tran_inc, 
		  float rot_sens, float rot_inc,
		  float scale_sens, float scale_incr );
};

  
