/****************************************************************************
 * APPLICATION.c                                                            *
 * Each function writes out the VRML syntax for that particular object???   *
 ****************************************************************************/
#include <stdio.h>
#include "msdl_types.h"
#include "msdl_protos.h"

#define MAXINDENTSIZE 30
#define INDENTTABSIZE 4

/* Function prototypes */
void print_obj_info( Tobject * );
void create_props( Tobject * );
void create_trans( Tmatrix );
void create_general( Tobject *, Tmatrix );
void calculate_indent ( int );
void indent_more ( char * );
void indent_less ( void );

/* Global variables */
int noofindents = 0;
char indent[MAXINDENTSIZE] = "";

/* ************************************************************ */

void calculate_indent( int noofi )
{
  int i;

  strcpy( indent, "" );
  for ( i=0;
        ((i<(noofi*INDENTTABSIZE)) && (i<MAXINDENTSIZE));
        i++ )
    strcat( indent, " " );
}

void indent_more( char *grouping )
{
  printf( "%s%s {\n", indent, grouping );
  noofindents++;
  calculate_indent( noofindents );
}

void indent_less( void )
{
  printf( "%s}\n", indent );
  noofindents--;
  calculate_indent( noofindents );
}

/* ************************************************************* */


/* Error handler */

void create_problem(char *probtext, int line_number)
{
  fprintf(stderr,"MSDL ERROR : %s at line %d\n",
	  probtext, line_number);
  exit(-1);
}

void create_head( void )
{
  printf( "#VRML V1.0 ascii\n" );
  /*printf( "#A VRML file created using the MSDL to VRML converter\n" );*/
  indent_more( "Separator" );
}

void create_foot( void )
{
  indent_less();
}

void create_verbatim( Tverb* verbatim )
{
#ifdef VERBOSE
  printf( "Creating a verbatim\n" );
  printf( "Text:%s\n", verbatim->text );
#endif
}

/*****************************************************************************
      ****    ******    ********  ********    ****    ********    *****
    **    **  **    **     **     **	    **    **     **     **     *
    **    **  ******       **     *****	    **	         **       ****
    **    **  **    **  *  **     **	    **    **     **     *     **
      ****    ******     ***      ********    ****	 **      *****
 *****************************************************************************/

/*===========================================================================*/
void create_comp( Tobject* obj, Tmatrix acc_trans )
{
#ifdef VERBOSE
  printf("Creating a composite\n");
  printf("It's name is %s\n",obj->name );
  if ( obj->verb != NULL )
    printf( "It's verbatim is :%s\n", obj->verb );
  if ( obj->handedness == LEFTHANDED )
    printf( "Lefthanded\n" );
  else if ( obj->handedness == RIGHTHANDED )
    printf( "Righthanded\n" );
  else if ( obj->handedness == UNDEFINED )
    printf( "Undefined\n" );
  else
    printf( "None of these\n" );
  if ( obj->ordering == COUNTERCLOCKWISE )
    printf( "Counterclockwise\n" );
  else
    printf( "Clockwise\n" );
#endif

  /* Do with this object as you will but please be gentle!! */
 
  indent_more( "Separator" );

#ifdef CLEANUP
  free( obj );
#endif
}
/*---------------------------------------------------------------------------*/


/*===========================================================================*/
void end_comp( void )
{
#ifdef VERBOSE
  printf("Ending composite\n");
#endif

  /* if you are creating nested objects this is where you must go up a step */
  
  indent_less();

}
/*---------------------------------------------------------------------------*/
 
void create_props( Tobject* obj )
{
  indent_more( "Material" );
  if ( obj->sprops.flags & ARCmask )
    printf( "%sambientColor   "PCF" "PCF" "PCF "\n", indent,
	   obj->sprops.arc.r, obj->sprops.arc.g, obj->sprops.arc.b );
  if ( obj->sprops.flags & DRCmask )
    printf( "%sdiffuseColor   "PCF" "PCF" "PCF "\n", indent,
	   obj->sprops.drc.r, obj->sprops.drc.g, obj->sprops.drc.b );
  if ( obj->sprops.flags & SRCmask )
    printf( "%sspecularColor   "PCF" "PCF" "PCF "\n", indent,
	   obj->sprops.src.r, obj->sprops.src.g, obj->sprops.src.b );
if ( obj->sprops.flags & EMmask )
    printf( "%semissiveColor   "PCF" "PCF" "PCF "\n", indent,
	   obj->sprops.emission.r, obj->sprops.emission.g, obj->sprops.emission.b );
  if ( obj->sprops.flags & TRCmask )
    printf( "%stransparency "PCF"\n", indent,
	   (obj->sprops.trc.r + obj->sprops.trc.g + obj->sprops.trc.b)/3);
  if (obj->sprops.trc.r != obj->sprops.trc.g ||
      obj->sprops.trc.r != obj->sprops.trc.b)
    fprintf( stderr, "WARNING: averaged transparency values\n" );
  if ( obj->sprops.flags & INmask )
    printf( "%s#index\n", indent /*define as a new material - extendMaterial!*/);
  if ( obj->sprops.flags & PHmask )
    printf( "%s#ph_exp\n", indent /*this is shinyness!!!!!*/ );
  if ( obj->sprops.flags & TEXmask )
	printf( "%s#texture map\n", indent );
  if ( obj->sprops.flags & EMTmask )
	printf( "%s#emittance map\n", indent );
  indent_less();
}

void create_trans( Tmatrix m )
{
  indent_more( "MatrixTransform" );

  printf( "%smatrix "PCF" "PCF" "PCF" "PCF"\n"
	 , indent, m[0][0], m[1][0], m[2][0], m[3][0] );
  printf( "%s       "PCF" "PCF" "PCF" "PCF"\n" 
	 , indent, m[0][1], m[1][1], m[2][1], m[3][1] );
  printf( "%s       "PCF" "PCF" "PCF" "PCF"\n"
	 , indent, m[0][2], m[1][2], m[2][2], m[3][2] );
  printf( "%s       "PCF" "PCF" "PCF" "PCF"\n"
	 , indent, m[0][3], m[1][3], m[2][3], m[3][3] );

  indent_less();
}

void create_general( Tobject* obj, Tmatrix acc_trans )
{
  if ( obj->sprops.flags != 0 )
    create_props( obj );
   create_trans( acc_trans );
}
 

/*===========================================================================*/
void create_triangle( Tobject* obj, Tmatrix acc_trans )
{
  int i;

#ifdef VERBOSE
  printf("Creating a triangle\n");
  print_obj_info( obj );
	printf(" It has %4d vertices \n",obj->spec.tri.vvnfn.no_verts );
	printf(" It has %4d vertex normals \n",obj->spec.tri.vvnfn.no_vnorms );
	printf(" It has %4d facet  normals \n",obj->spec.tri.vvnfn.no_fnorms );
        printf(" It has %4d vertex colours \n",obj->spec.tri.vvnfn.no_vcolours );
#endif

  /* Do with this object as you will but please be gentle!! */
 
  indent_more( "Separator" );

   /* call a general function to handle the object's properties,
      transformations and other general attributes */
  create_general( obj, acc_trans );
 
  indent_more( "Coordinate3" );
  printf( "%spoint [\n", indent );
  for ( i=0; i<3; i++ )
    printf( "%s       "PCF" "PCF" "PCF",\n", indent, 
	   obj->spec.tri.vvnfn.data[i].x,
	   obj->spec.tri.vvnfn.data[i].y,
	   obj->spec.tri.vvnfn.data[i].z );
  printf( "%s      ]\n", indent );
  indent_less();

  indent_more( "IndexedFaceSet" );
  printf("%scoordIndex [ 0, 1, 2, -1 ]    #A triangle\n", indent );
  indent_less();

  if ( obj->spec.tri.vvnfn.no_vnorms != 0 )
    fprintf( stderr, "WARNING: vertex normals not converted" );
  if ( obj->spec.tri.vvnfn.no_fnorms != 0 )
    fprintf( stderr, "WARNING: face normals not converted" );
  if ( obj->spec.tri.vvnfn.no_vcolours != 0 )
    fprintf( stderr, "WARNING: vertex colours not converted" );

  indent_less();

#ifdef CLEANUP
  free( obj->spec.tri.vvnfn.data );
  free( obj );
#endif
}
/*---------------------------------------------------------------------------*/

/*===========================================================================*/
void create_polygon( Tobject* obj, Tmatrix acc_trans )
{
  int i;

#ifdef VERBOSE
  printf("Creating a polygon\n");
  print_obj_info( obj );
	printf(" It has %4d vertices \n",obj->spec.polg.vvnfn.no_verts );
	printf(" It has %4d vertex normals \n",obj->spec.polg.vvnfn.no_vnorms );
	printf(" It has %4d facet  normals \n",obj->spec.polg.vvnfn.no_fnorms );
        printf(" It has %4d vertex colours \n",obj->spec.polg.vvnfn.no_vcolours );
#endif

  /* Do with this object as you will but please be gentle!! */

  indent_more( "Separator" );

   /* call a general function to handle the object's properties,
      transformations and other general attributes */
  create_general( obj, acc_trans );

  indent_more( "Coordinate3" );
  printf( "%spoint [\n", indent );
  for ( i=0; i<obj->spec.polg.vvnfn.no_verts; i++ )
    printf( "%s       "PCF" "PCF" "PCF",\n", indent, 
	   obj->spec.tri.vvnfn.data[i].x,
	   obj->spec.tri.vvnfn.data[i].y,
	   obj->spec.tri.vvnfn.data[i].z );
  printf( "%s      ]\n", indent );
  indent_less();

  indent_more( "IndexedFaceSet" );
  printf("%scoordIndex [ ", indent );
  for ( i=0; i<obj->spec.polg.vvnfn.no_verts; i++ )
    printf( "%d, ", i );
  printf( "-1 ]    #A %d-sided polygon\n", obj->spec.polg.vvnfn.no_verts );
  indent_less();

  if ( obj->spec.polg.vvnfn.no_vnorms != 0 )
    fprintf( stderr, "WARNING: vertex normals not converted" );
  if ( obj->spec.polg.vvnfn.no_fnorms != 0 )
    fprintf( stderr, "WARNING: face normals not converted" );
  if ( obj->spec.polg.vvnfn.no_vcolours != 0 )
    fprintf( stderr, "WARNING: vertex colours not converted" );

  indent_less();

#ifdef CLEANUP
  free( obj->spec.polg.vvnfn.data );
  free( obj );
#endif
  

}
/*---------------------------------------------------------------------------*/

/*===========================================================================*/
void create_sphere( Tobject* obj, Tmatrix acc_trans )
{
  Tmatrix  temp_trans;

#ifdef VERBOSE
  printf("Creating a sphere\n");	
  print_obj_info( obj );
	printf(" Centre is "PCF" "PCF" "PCF" \n",
						obj->spec.sph.centre.x,
						obj->spec.sph.centre.y,
						obj->spec.sph.centre.z );
	printf(" It's radius is "PCF"\n", obj->spec.sph.radius );
#endif

  /* Do with this object as you will but please be gentle!! */

  /* Adjust the transformation matrix so that the centre of the
     sphere is at the origin */
  make_sh_mat( &(obj->spec.sph.centre), temp_trans );
  multiply_matrix( temp_trans, acc_trans, acc_trans );
 
  indent_more( "Separator" );

  /* call a general function to handle the object's properties,
     transformations and other general attributes */
  create_general( obj, acc_trans );
 
  /* create the sphere */
  indent_more( "Sphere" );
  printf( "%sradius "PCF"\n", indent, obj->spec.sph.radius );
  indent_less();

  indent_less();

#ifdef CLEANUP
  free( obj );
#endif
}
/*---------------------------------------------------------------------------*/

/*===========================================================================*/
void create_cylinder( Tobject* obj, Tmatrix acc_trans )
{
  Tfloat   width, height, depth, cylinder_height;
  Tpt3     centre, yaxis, cyl, cross, invert;
  Tmatrix  sh_mat, scale_mat, x_rot_mat, y_rot_mat, z_rot_mat, new_acc_trans;
  Tpt3		newbottom, newtop;
  Tfloat	xangle, zangle;

#ifdef VERBOSE
  printf("creating a cylinder\n");	
  print_obj_info( obj );
	printf(" Radius is "PCF"\n", obj->spec.cyl.radius );
	printf(" Capped? Top %d Bottom %d \n ",
					obj->spec.cyl.tcapped,
					obj->spec.cyl.bcapped );
#endif

  /* Do with this object as you will but please be gentle!! */ 

  unit_matrix( new_acc_trans );

 /* Adjust the transformation matrix so that the bottom left coordinate
     of the box is at the origin */
/*newbottom = transPt3( &(obj->spec.cyl.bottom), acc_trans );
newtop = transPt3( &(obj->spec.cyl.top), acc_trans );*/

newbottom.x = obj->spec.cyl.bottom.x;
newbottom.y = obj->spec.cyl.bottom.y;
newbottom.z = obj->spec.cyl.bottom.z;
newtop.x = obj->spec.cyl.top.x;
newtop.y = obj->spec.cyl.top.y;
newtop.z = obj->spec.cyl.top.z;

centre.x = newbottom.x + (newtop.x - newbottom.x)/2.0;
centre.y = newbottom.y + (newtop.y - newbottom.y)/2.0;
centre.z = newbottom.z + (newtop.z - newbottom.z)/2.0;

  make_sh_mat( &centre, sh_mat );

/* Make rotation matrices so that the cylinder is on the y-axis */

  yaxis.x = 0.0; yaxis.y = 1.0; yaxis.z = 0.0;
  cyl = ptk_subv3(&centre, &newtop);

  cross = ptk_crossv3(&cyl, &yaxis);

  cross.x = -cross.x;
  cross.y = -cross.y;
  cross.z = -cross.z;

  make_rot_mat( 'x', cross.x, x_rot_mat );
  make_rot_mat( 'y', cross.y, y_rot_mat );
  make_rot_mat( 'z', cross.z, z_rot_mat );
  multiply_matrix( x_rot_mat, new_acc_trans, new_acc_trans );
  multiply_matrix( y_rot_mat, new_acc_trans, new_acc_trans );
  multiply_matrix( z_rot_mat, new_acc_trans, new_acc_trans );

/*if ((newtop.y - newbottom.y) != 0) {
	xangle = atan( ((newtop.z - newbottom.z)/(newtop.y - newbottom.y)) );
	zangle = -atan( ((newtop.x - newbottom.x)/(newtop.y - newbottom.y)) );
	fprintf( stderr, "xangle = "PCF"\nzangle = "PCF"\n", xangle, acos(1.0));

	make_rot_mat ( 'x', (xangle /PI)*180.0, x_rot_mat );
	make_rot_mat ( 'z', (zangle /PI)*180.0, z_rot_mat );
 	multiply_matrix( x_rot_mat, new_acc_trans, new_acc_trans );
  	multiply_matrix( z_rot_mat, new_acc_trans, new_acc_trans );
}*/

  multiply_matrix( sh_mat, new_acc_trans, new_acc_trans );

  multiply_matrix( acc_trans, new_acc_trans, new_acc_trans );

/* Calculate cylinder height */
width = obj->spec.cyl.top.x - obj->spec.cyl.bottom.x;
height = obj->spec.cyl.top.y - obj->spec.cyl.bottom.y;
depth = obj->spec.cyl.top.z - obj->spec.cyl.bottom.z;
cylinder_height = sqrt( pow(width,2) + pow(height,2) + pow(depth,2));

  indent_more( "Separator" );

   /* call a general function to handle the object's properties,
      transformations and other general attributes */
  create_general( obj, new_acc_trans );

  /* create the cylinder */
  indent_more( "Cylinder" );
  printf( "%sradius "PCF"\n", indent, obj->spec.cyl.radius );
  printf( "%sheight "PCF"\n", indent, cylinder_height );
  indent_less();

  indent_less();
#if 0
  indent_more( "Separator" );

  width = obj->spec.cyl.top.x - obj->spec.cyl.bottom.x;
  height = obj->spec.cyl.top.y - obj->spec.cyl.bottom.y;
  depth = obj->spec.cyl.top.z - obj->spec.cyl.bottom.z;
  centre.x = obj->spec.cyl.bottom.x + width/2.0;
  centre.y = obj->spec.cyl.bottom.y + height/2.0;
  centre.z = obj->spec.cyl.bottom.z + depth/2.0;

  unit_matrix( rot_mat );
  if ( (obj->spec.cyl.bottom.x == obj->spec.cyl.top.x)
	   && (obj->spec.cyl.bottom.z == obj->spec.cyl.top.z) ) {
	     /* cylinder along y-axis */
	     /* don't rotate */
	     cylinder_height = height;
  } else if ( (obj->spec.cyl.bottom.y == obj->spec.cyl.top.y)
      && (obj->spec.cyl.bottom.z == obj->spec.cyl.top.z) ) {
    /* cylinder along x-axis */
    make_rot_mat ( 'z', 90.0, rot_mat );
    cylinder_height = width;
  } else if ( (obj->spec.cyl.bottom.x == obj->spec.cyl.top.x)
	   && (obj->spec.cyl.bottom.y == obj->spec.cyl.top.y) ) {
	     /* cylinder along z-axis */
    make_rot_mat ( 'x', 90.0, rot_mat );
	     cylinder_height = depth;
  } else {
    printf( "%s#WARNING - cylinder not converted correctly\n", indent );
    cylinder_height = height;  /* NO! this is completely wrong!!! */
  }

  /* Adjust the transformation matrix so that the centre of the
     cyclinder is at the origin */
  make_sh_mat( &(obj->spec.sph.centre), sh_mat );
  multiply_matrix( sh_mat, acc_trans, acc_trans );
	   
  /* Now rotate so that cylinder is along the y-axis */
  multiply_matrix( rot_mat, acc_trans, acc_trans );

  /* call a general function to handle the object's properties,
     transformations and other general attributes */
  create_general( obj, acc_trans );
 
  /* create the cylinder */
  indent_more( "Cylinder" );
  printf( "%sradius "PCF"\n", indent, obj->spec.cyl.radius );
  printf( "%sheight "PCF"\n", indent, cylinder_height );
  indent_less();

  indent_less();
#endif

#ifdef CLEANUP
  free( obj );
#endif
}
/*---------------------------------------------------------------------------*/

/*===========================================================================*/
void create_cone( Tobject* obj, Tmatrix acc_trans )
{
#ifdef VERBOSE
  printf("Creating a cone\n");	
  print_obj_info( obj );
	printf(" Radii	Top: "PCF" Bottom: "PCF"\n",
					obj->spec.cone.tradius,
					obj->spec.cone.bradius );
	printf(" Capped? Top %d Bottom %d \n ",
					obj->spec.cone.tcapped,
					obj->spec.cone.bcapped );
#endif

  /* Do with this object as you will but please be gentle!! */
 
  indent_more( "Separator" );

   /* call a general function to handle the object's properties,
      transformations and other general attributes */
  create_general( obj, acc_trans );
 
  printf( "%s#Cone\n", indent );

  indent_less();

#ifdef CLEANUP
  free( obj );
#endif
}
/*---------------------------------------------------------------------------*/

/*===========================================================================*/
void create_disc( Tobject* obj, Tmatrix acc_trans )
{

  Tmatrix  sh_mat, x_rot_mat, z_rot_mat, new_acc_trans;
  Tpt3		newcentre, newnormal;
  Tfloat	xangle, zangle;

#ifdef VERBOSE
  printf("Creating a disc\n");	
  print_obj_info( obj );
	printf(" Centre is "PCF" "PCF" "PCF" \n",
						obj->spec.disc.centre.x,
						obj->spec.disc.centre.y,
						obj->spec.disc.centre.z );
	printf(" It's radius is "PCF"\n", obj->spec.disc.radius );
#endif

  /* Do with this object as you will but please be gentle!! */
 
   /* call a general function to handle the object's properties,
      transformations and other general attributes */
  create_general( obj, acc_trans );
 
  unit_matrix( new_acc_trans );

 /* Adjust the transformation matrix so that the bottom left coordinate
     of the box is at the origin */
  newcentre = transPt3( &(obj->spec.disc.centre), acc_trans );
  newnormal = transPt3( &(obj->spec.disc.normal), acc_trans );

  make_sh_mat( &newcentre, sh_mat );

/* Make rotation matrices so that the cylinder is on the y-axis */

if (newnormal.y != 0) {
	xangle = atan( ((newnormal.z)/(newnormal.y)) );
	zangle = -atan( ((newnormal.x)/(newnormal.y)) );
} else {
	/* normal is a 90degrees to y axis */

	/*rotate around y axis onto x axis */
	/*yangle = atan( (newnormal.z)/(newnormal.x) );*/
	/* rotate around z axis onto y axis */
	/*zangle = atan( (*/
	xangle = PI/2;
	zangle = PI/2;
}

/*	make_rot_mat ( 'x', (xangle /PI)*180.0, x_rot_mat );
	make_rot_mat ( 'z', (zangle /PI)*180.0, z_rot_mat );
 	multiply_matrix( x_rot_mat, new_acc_trans, new_acc_trans );
  	multiply_matrix( z_rot_mat, new_acc_trans, new_acc_trans );*/


  multiply_matrix( sh_mat, new_acc_trans, new_acc_trans );

  multiply_matrix( acc_trans, new_acc_trans, new_acc_trans );


  indent_more( "Separator" );

   /* call a general function to handle the object's properties,
      transformations and other general attributes */
  create_general( obj, new_acc_trans );

  /* create the disc */
  indent_more( "Cylinder" );
  printf( "%sradius "PCF"\n", indent, obj->spec.disc.radius );
  printf( "%sheight 0    #A disc\n", indent );
  indent_less();

  indent_less();

#ifdef CLEANUP
  free( obj );
#endif
}
/*---------------------------------------------------------------------------*/

/*===========================================================================*/
void create_box( Tobject* obj, Tmatrix acc_trans )
{
  Tfloat   width, height, depth;
  Tpt3		bl, tr, centre;
  Tmatrix	sh_mat, new_acc_trans;	

#ifdef VERBOSE
  printf("Creating a box\n");	
  print_obj_info( obj );
	printf(" Top right is  "PCF" "PCF" "PCF" \n",
						obj->spec.box.tr.x,
						obj->spec.box.tr.y,
						obj->spec.box.tr.z );
	printf(" Bottom left is "PCF" "PCF" "PCF" \n",
						obj->spec.box.bl.x,
						obj->spec.box.bl.y,
						obj->spec.box.bl.z );
#endif

  /* Do with this object as you will but please be gentle!! */

  unit_matrix( new_acc_trans );

 /* Adjust the transformation matrix so that the bottom left coordinate
     of the box is at the origin */

bl.x = obj->spec.box.bl.x;
bl.y = obj->spec.box.bl.y;
bl.z = obj->spec.box.bl.z;
tr.x = obj->spec.box.tr.x;
tr.y = obj->spec.box.tr.y;
tr.z = obj->spec.box.tr.z;


centre.x = bl.x + (tr.x - bl.x)/2.0;
centre.y = bl.y + (tr.y - bl.y)/2.0;
centre.z = bl.z + (tr.z - bl.z)/2.0;

  make_sh_mat( &centre, sh_mat );
  multiply_matrix( sh_mat, new_acc_trans, new_acc_trans );

  multiply_matrix( acc_trans, new_acc_trans, new_acc_trans );

/* Calculate width, height and depth */
width = tr.x - bl.x;
height = tr.y - bl.y;
depth = tr.z - bl.z;

  indent_more( "Separator" );

   /* call a general function to handle the object's properties,
      transformations and other general attributes */
  create_general( obj, new_acc_trans );
 
  indent_more( "Cube" );
  printf( "%swidth   "PCF"\n", indent, width );
  printf( "%sheight  "PCF"\n", indent, height );
  printf( "%sdepth   "PCF"\n", indent, depth );
  indent_less();

  indent_less();

#ifdef CLEANUP
  free( obj );
#endif
}
/*---------------------------------------------------------------------------*/

/*===========================================================================*/
void create_nurb( Tobject* obj, Tmatrix acc_trans )
{
#ifdef VERBOSE
  printf("Creating a nurb\n");	
  print_obj_info( obj );
#endif

  /* Do with this object as you will but please be gentle!! */
 
  indent_more( "Separator" );

   /* call a general function to handle the object's properties,
      transformations and other general attributes */
  create_general( obj, acc_trans );
 
  printf( "%s#Nurbs\n", indent );

  indent_less();

#ifdef CLEANUP
  free( obj->spec.nurb.knots_u );
  free( obj->spec.nurb.knots_v );
  free( obj->spec.nurb.cpts    );
  free( obj );
#endif
}
/*---------------------------------------------------------------------------*/

/*===========================================================================*/
void create_polyhedron( Tobject* obj, Tmatrix acc_trans )
{
  int 		i, f, v;

#ifdef VERBOSE
  printf("Creating a polyhedron\n");
  print_obj_info( obj );
	printf(" It has %4d vertices \n",obj->spec.polh.vvnfn.no_verts );
	printf(" It has %4d vertex normals \n",obj->spec.polh.vvnfn.no_vnorms );
	printf(" It has %4d facet  normals \n",obj->spec.polh.vvnfn.no_fnorms );
        printf(" It has %4d vertex colours \n",obj->spec.polh.vvnfn.no_vcolours );
#endif

  /* Do with this object as you will but please be gentle!! */
 
  indent_more( "Separator" );

   /* call a general function to handle the object's properties,
      transformations and other general attributes */
  create_general( obj, acc_trans );
 
  printf( "%s#Polyhedron\n", indent );
  indent_more( "Coordinate3" );
  printf( "%spoint [\n", indent );
  for ( i=0; i<obj->spec.polg.vvnfn.no_verts; i++ )
    printf( "%s       "PCF" "PCF" "PCF",\n", indent, 
	   obj->spec.tri.vvnfn.data[i].x,
	   obj->spec.tri.vvnfn.data[i].y,
	   obj->spec.tri.vvnfn.data[i].z );
  printf( "%s      ]\n", indent );
  indent_less();

  indent_more( "IndexedFaceSet" );
  printf("%scoordIndex [ ", indent );
  
  for ( f=0; f<obj->spec.polh.flist.no_faces; f++ ) {
    /* write a face */
    /*face = obj->spec.polh.flist->face;*/
    for ( v=0; v<obj->spec.polh.flist.face[f]->no_verts; v++ )
      printf( "%d, ", obj->spec.polh.flist.face[f]->vert_index[v] );
    if ( f<(obj->spec.polh.flist.no_faces - 1) )
      printf( " -1,\n%s             ", indent );
    /*printf( "%s    face %d\n", indent, f );*/
  }
 
  printf( "-1 ]    #A %d-faced polyhedron\n", obj->spec.polh.flist.no_faces );
  indent_less();

  if ( obj->spec.polh.vvnfn.no_vnorms != 0 )
    fprintf( stderr, "WARNING: vertex normals not converted" );
  if ( obj->spec.polh.vvnfn.no_fnorms != 0 )
    fprintf( stderr, "WARNING: face normals not converted" );
  if ( obj->spec.polh.vvnfn.no_vcolours != 0 )
    fprintf( stderr, "WARNING: vertex colours not converted" );

  indent_less();

#ifdef CLEANUP
  free( obj->spec.polh.vvnfn.data );
  for( i=0 ; i<obj->spec.polh.flist.no_faces ; i++ )
    free( obj->spec.polh.flist.face[i]->vert_index );
  free( obj->spec.polh.flist.face );
  free( obj );
#endif
}
/*---------------------------------------------------------------------------*/

void print_obj_info( Tobject* obj )
{ 
  /* this prints out info about an object a bit like the sdl command walk 
     but it executed in the application writers code... that's why it's here!! */

  printf ( " Obj Name: %s\n",  obj->name );
  printf ( "  Obj Type: %i\n",  obj->otype );
  PRINTP(obj->sprops);
  /*if ( obj->sprops.flags & EMTmask )*/
  /*if ( obj->sprops.flags > 1023 )*/
  printf("!!!PROPS FLAG SET TO: %u!!!\n", obj->sprops.flags);
  printf ( "  "PCF" "PCF" "PCF"\n", 
	  obj->box.min.x, obj->box.min.y, obj->box.min.z );
  printf ( "  "PCF" "PCF" "PCF"\n", 
	  obj->box.max.x, obj->box.max.y, obj->box.max.z );
  if ( obj->verb )
    printf ( "Verbatim:%s\n", obj->verb );
  if ( obj->handedness == LEFTHANDED )
    printf ( "Lefthanded\n" );
  else
    printf ( "Righthanded\n" );
  if ( obj->ordering == COUNTERCLOCKWISE )
    printf ( "Counterclockwise\n" );
  else
    printf ( "Clockwise\n" );
}
/*****************************************************************************
	**	  **    ****    **    **  ********    *****
	**	  **  **    **  **    **     **     **     *
	**	  **  **  ****  ********     **       ****
	**	  **  **    **  **    **     **     *     **
	********  **    ****    **    **     **      *****
 *****************************************************************************/

void create_point_lt( Tlights* lite )
{
#ifdef VERBOSE
  printf( "Creating point light of type %d\n",lite->ltype );
  printf( "at "PCF" "PCF" "PCF"\n",	lite->spec.point.locn.x,
	 				lite->spec.point.locn.y,
	 				lite->spec.point.locn.z );
  PRINTrgb( "Intensity = ", lite->spec.point.inten );
#endif

  /* Do with this light as you will but please be gentle!! */

  indent_more( "PointLight" );
  printf( "%scolor        "PCF" "PCF" "PCF"\n",
	 indent,
	 lite->spec.point.inten.r,
	 lite->spec.point.inten.g,
	 lite->spec.point.inten.b );
  printf( "%slocation     "PCF" "PCF" "PCF"\n",
	 indent,
	 lite->spec.point.locn.x,
	 lite->spec.point.locn.y,
	 lite->spec.point.locn.z );
  indent_less();
  
#ifdef CLEANUP
  free( lite );
#endif
}

void create_directional_lt( Tlights* lite )
{
#ifdef VERBOSE
  printf( "Creating directional light of type %d\n",lite->ltype );
  printf( "towards "PCF" "PCF" "PCF"\n",	lite->spec.direc.along.x,
	 				lite->spec.direc.along.y,
	 				lite->spec.direc.along.z );
  PRINTrgb( "Intensity = ", lite->spec.direc.inten );
#endif

  /* Do with this light as you will but please be gentle!! */
 indent_more( "DirectionalLight" );
  printf( "%scolor        "PCF" "PCF" "PCF"\n",
	 indent,
	 lite->spec.direc.inten.r,
	 lite->spec.direc.inten.g,
	 lite->spec.direc.inten.b );
  printf( "%sdirection    "PCF" "PCF" "PCF"\n",
	 indent,
	 lite->spec.direc.along.x,
	 lite->spec.direc.along.y,
	 lite->spec.direc.along.z );
  indent_less();

#ifdef CLEANUP
  free( lite );
#endif
}  

void create_spot_lt( Tlights* lite )
{
#ifdef VERBOSE
  printf( "Creating point light of type %d\n",lite->ltype );
  printf( "at "PCF" "PCF" "PCF"\n",	lite->spec.spot.locn.x,
	 				lite->spec.spot.locn.y,
	 				lite->spec.spot.locn.z );
  printf( "towards "PCF" "PCF" "PCF"\n",lite->spec.spot.at.x,
	 				lite->spec.spot.at.y,
	 				lite->spec.spot.at.z );
  PRINTrgb( "Intensity =" , lite->spec.spot.inten );
#endif

  /* Do with this light as you will but please be gentle!! */

  indent_more( "SpotLight" );
  printf( "%scolor        "PCF" "PCF" "PCF"\n",
	 indent,
	 lite->spec.spot.inten.r,
	 lite->spec.spot.inten.g,
	 lite->spec.spot.inten.b );
  printf( "%slocation     "PCF" "PCF" "PCF"\n",
	 indent,
	 lite->spec.spot.locn.x,
	 lite->spec.spot.locn.y,
	 lite->spec.spot.locn.z );
  printf( "%sdirection    "PCF" "PCF" "PCF"\n",
	 indent,
	 lite->spec.spot.at.x - lite->spec.spot.locn.x, /*should this be a*/
	 lite->spec.spot.at.y - lite->spec.spot.locn.y, /*unit vector???*/
	 lite->spec.spot.at.z - lite->spec.spot.locn.z );
  printf( "%sdropOffRate  "PCF"\n",
	 indent,
	 0/*lite->spec.spot.exp*/ ); /*should be cos^exp???*/
  printf( "%scutOffAngle  "PCF"\n",
	 indent,
	 ((Tfloat) (lite->spec.spot.cutoff/180)) * PI );
  indent_less();

#ifdef CLEANUP
  free( lite );
#endif
}

/*---------------------------------------------------------------------------*/

/****************************************************************************
  *****  *    *  ******          *    *     *    ******  *    *
    *    *    *  *               *    *     *    *       *    *
    *    ******  *****           *    *     *    *****   * ** *
    *    *    *  *                *  *      *    *       **  **
    *    *    *  ******            **       *    ******  *    * 
 ***************************************************************************/
/*===========================================================================*/
void create_view( Tview* view )
{
#ifdef VERBOSE
  printf( "Creating a view\n" );
  printf( " from:    "PCF" "PCF" "PCF"\n",	view->from.x,
						view->from.y,
						view->from.z );
  printf( " towards: "PCF" "PCF" "PCF"\n",	view->towards.x,
						view->towards.y,
						view->towards.z );
  printf( " vpnorm:  "PCF" "PCF" "PCF"\n",	view->vpn.x,
						view->vpn.y,
						view->vpn.z );
  printf( " up:      "PCF" "PCF" "PCF"\n",	view->up.x,
						view->up.y,
						view->up.z );
  printf( " window:  "PCF" "PCF" "PCF" "PCF"\n",view->window.bl.u,
						view->window.bl.v,
						view->window.tr.u,
						view->window.tr.v );
  printf( " front:   "PCF"\n",			view->fpd );
  printf( " back:    "PCF"\n",			view->bpd );
  printf( " persp:   %d\n",			view->persp );
#endif

  /* Do with this view as you will but please be gentle!! */

  if ( view->persp )
    indent_more( "PerspectiveCamera" );
  else
  indent_more( "OrthographicCamera" );
  printf( "%sposition       "PCF" "PCF" "PCF"\n", indent,
	 view->from.x, view->from.y, view->from.z );
  printf( "%sorientation    "PCF" "PCF" "PCF"\n", indent,
	 view->towards.x - view->from.x,
	 view->towards.y - view->from.y,
	 view->towards.z - view->from.z );
  fprintf( stderr, "WARNING: view only partially defined\n" );
  
  indent_less();

#ifdef CLEANUP
  free( view );
#endif
}
/*---------------------------------------------------------------------------*/


/****************************************************************************
  *****  *    *  ******            *****    ****   ******  *   *  ******
    *    *    *  *               **     *  *    *  *       **  *  *
    *    ******  *****             ****    *       *****   * ***  *****
    *    *    *  *               *     **  *    *  *       *  **  *
    *    *    *  ******           *****     ****   ******  *   *  ******
 ***************************************************************************/
/*===========================================================================*/
void create_scene( Tscene* scene )
{
#ifdef VERBOSE
  printf( "Creating a scene\n" );
  printf( " background colour:  "PCF" "PCF" "PCF"\n", scene->background.r,
	                                              scene->background.g,
						      scene->background.b );
#endif

  /* Do with this scene as you will but please be gentle!! */

  fprintf( stderr, "WARNING: background colour not converted\n");

#ifdef CLEANUP
  free( scene );
#endif
}
/*---------------------------------------------------------------------------*/




