#include <stdio.h>

#include <avs/avs.h>
#include <avs/field.h>

#include "vinclude.h"
#include "vipl/lvstats.h"
#include "info.h"
#include "proto.h"

#include "avskhoros.h"

#define         ALL_OPTS        18

/* *****************************************/
/*  Module Descriptions                    */
/* *****************************************/

int dpeakpick_desc ()
{
   int in_port, out_port, param;
   extern int dpeakpick_compute();

   AVSset_module_name("dpeakpick",MODULE_FILTER);

   AVScreate_input_port("Input File          ","field 2D",REQUIRED);
   AVScreate_output_port("VIFF Output File","field 2D");
   ADD_FILE;
   AVSadd_parameter("which","choice","troughs","troughs,peaks",",");
   AVSadd_parameter("find all","boolean",0,0,1);
   ISLIDE(AVSadd_parameter("num peaks","integer",1,1,1000000));
   AVSadd_parameter("max/min","choice","first n","first n,max/min",",");
   AVSadd_parameter("endpoints","boolean",0,0,1);
   ISLIDE(AVSadd_parameter("window height steps","integer",1,1,1000000));
   SLIDE(AVSadd_float_parameter("min height",1.0,1.0,1000000.0));
   AVSadd_parameter("direction","choice","down vectors","down vectors,across bands",",");

   AVSset_compute_proc(dpeakpick_compute);
   return(1);
}

int dstats_desc ()
{
   int in_port, out_port, param;
   extern int dstats_compute();

   AVSset_module_name("dstats",MODULE_FILTER);

   AVScreate_input_port("Input VIFF File     ","field 2D",REQUIRED);
   AVScreate_input_port("Masking File     ","field 2D",OPTIONAL);
   AVScreate_output_port("VIFF Output File ","field 2D");
   ADD_FILE;
   AVSadd_parameter("Save All","boolean",0,0,1);
   AVSadd_parameter("mean","boolean",0,0,1);
   AVSadd_parameter("variance","boolean",0,0,1);
   AVSadd_parameter("standard deviation","boolean",0,0,1);
   AVSadd_parameter("rms","boolean",0,0,1);
   AVSadd_parameter("max value","boolean",0,0,1);
   AVSadd_parameter("x max","boolean",0,0,1);
   AVSadd_parameter("min value","boolean",0,0,1);
   AVSadd_parameter("x min","boolean",0,0,1);
   AVSadd_parameter("integral","boolean",0,0,1);
   AVSadd_parameter("integral pos","boolean",0,0,1);
   AVSadd_parameter("integral neg","boolean",0,0,1);
   AVSadd_parameter("num contrib","boolean",0,0,1);
   AVSadd_parameter("num positive","boolean",0,0,1);
   AVSadd_parameter("num negative","boolean",0,0,1);
   AVSadd_parameter("skewness","boolean",0,0,1);
   AVSadd_parameter("kurtosis","boolean",0,0,1);
   AVSadd_parameter("entropy","boolean",0,0,1);
   AVSadd_parameter("contrast","boolean",0,0,1);
   AVSadd_parameter("direction","choice","down vectors","down vectors,across bands",",");

   AVSset_compute_proc(dstats_compute);
   return(1);
}

int varviff_desc ()
{
   int in_port, out_port, param;
   extern int varviff_compute();

   AVSset_module_name("varviff",MODULE_FILTER);

   AVScreate_input_port("Input VIFF File","field 2D",REQUIRED);
   AVSadd_parameter("x","integer",0,0,512);
   AVSadd_parameter("y","integer",0,0,512);
   AVSadd_parameter("band","integer",0,0,256);

   AVSset_compute_proc(varviff_compute);
   return(1);
}

int vfileinfo_desc ()
{
   int in_port, out_port, param;
   extern int vfileinfo_compute();

   AVSset_module_name("vfileinfo",MODULE_FILTER);

   AVScreate_input_port("Input Image ","field 2D",REQUIRED);
   ADD_FILE;

   AVSset_compute_proc(vfileinfo_compute);
   return(1);
}

int vprdata_desc ()
{
   int in_port, out_port, param;
   extern int vprdata_compute();

   AVSset_module_name("vprdata",MODULE_FILTER);

   AVScreate_input_port("Input Image ","field 2D",REQUIRED);
   ADD_FILE;
   AVSadd_parameter("form","choice","matrix","column,matrix",",");

   AVSset_compute_proc(vprdata_compute);
   return(1);
}

int vstats_desc ()
{
   int in_port, out_port, param;
   extern int vstats_compute();

   AVSset_module_name("vstats",MODULE_FILTER);

   AVScreate_input_port("Input Image     ","field 2D",REQUIRED);
   AVScreate_input_port("Masking Image","field 2D",OPTIONAL);
   AVScreate_output_port("VIFF Output  ","field 2D");
   ADD_FILE;
   AVSadd_parameter("Save All","boolean",0,0,1);
   AVSadd_parameter("mean","boolean",0,0,1);
   AVSadd_parameter("variance","boolean",0,0,1);
   AVSadd_parameter("standard deviation","boolean",0,0,1);
   AVSadd_parameter("rms","boolean",0,0,1);
   AVSadd_parameter("max value","boolean",0,0,1);
   AVSadd_parameter("x max","boolean",0,0,1);
   AVSadd_parameter("y max","boolean",0,0,1);
   AVSadd_parameter("min value","boolean",0,0,1);
   AVSadd_parameter("x min","boolean",0,0,1);
   AVSadd_parameter("y min","boolean",0,0,1);
   AVSadd_parameter("integral","boolean",0,0,1);
   AVSadd_parameter("integral pos","boolean",0,0,1);
   AVSadd_parameter("integral neg","boolean",0,0,1);
   AVSadd_parameter("num contrib","boolean",0,0,1);
   AVSadd_parameter("num positive","boolean",0,0,1);
   AVSadd_parameter("num negative","boolean",0,0,1);
   AVSadd_parameter("skewness","boolean",0,0,1);
   AVSadd_parameter("kurtosis","boolean",0,0,1);
   AVSadd_parameter("entropy","boolean",0,0,1);
   AVSadd_parameter("contrast","boolean",0,0,1);
   AVSadd_parameter("compute separate","boolean",0,0,1);

   AVSset_compute_proc(vstats_compute);
   return(1);
}

/* *****************************************/
/* Module Compute Routines                 */
/* *****************************************/

int dpeakpick_compute (	AVSfield *i1,AVSfield **o1,
			char *fname,char *which,int find_all,
			int num_peaks,char *maxmin,int endpoints,
			int steps,float *minheight,char *direction)
{
   int value;
   struct xvimage  *image;
   struct p_info   **peak_info;
   struct v_info   *vect_info;
   FILE            *printdev;
   int             sig_length,
                   num_signals,
                   w,i,j;
   double          hwr;
   char            *PTS_str,
                   *pts_str,
                   *ptp_str,
                   *BV_str;


   if (i1)
      if ((image=(struct xvimage *)field_to_viff(i1))==NULL) return(0);
  
  /*
   *  Setup ascii output file if selected.
   */
  if( fname ) 
    if (! vwrite_ascii(fname, &printdev)) 
    {
      (void) fprintf(stderr, "%s:  Can't open ascii output file: \\n","dpeakpick");
      return(0);
    }

  /*
   *  Calculate number of vectors or bands.
   */

  switch(AVSchoice_number("direction",direction))
  {
     case DSP_VECTOR:
          num_signals = image->row_size * image->col_size;
          sig_length = image->num_data_bands;
          BV_str = "Vector";
          break;
     case DSP_BAND:
          num_signals = image->num_data_bands;
          sig_length = image->row_size * image->col_size; 
          BV_str = "Band";
          break;
     default:
          (void)fprintf(stderr,"%s: unknown process direction (-d): %d\n",
                       "dpeakpick", direction);
          return(0);
  }

 /*
  * Check for window option and, if selected, set window to specified
  * value.  Otherwise set window value to 1.  Then make sure that it is
  * valid.
  */
  w = steps;
  if( w > ((int)((sig_length-1)/2)) || w < 1 ) 
  {
     (void)fprintf(stderr,"%s: window size must not be greater","dpeakpick");
     (void)fprintf(stderr," than\n(number of points-1)/2,");
     (void)fprintf(stderr," or less than one.");
     (void)fprintf(stderr,"    window size = %d\n",w);
     (void)fprintf(stderr,"    number of points = %d\n",sig_length);
  }

 /*
  * Check for height to width ratio option and, if selected, set hwr to
  * specified value.  Otherwise set hwr to 0.
  */
  hwr = (double)*minheight;

  if(!ldpeakpick(image, AVSchoice_number("which",which), find_all,
                 num_peaks, AVSchoice_number("max/min",maxmin), endpoints,
                 w, hwr, AVSchoice_number("direction",direction), 
		&peak_info, &vect_info) ) 
  {
     (void)fprintf(stderr,"%s: ldpeakpick failed!\n","dpeakpick");
     return(0);
  }

  /* write to ascii file (if selected) */
  if( AVSchoice_number("which",which)) 
  {
     PTS_str = "Peaks";
     pts_str = "peaks";
     ptp_str = " peak  to  peak ";
  }
  else 
  {
     PTS_str = "Troughs";
     pts_str = "troughs";
     ptp_str = "trough to trough";
  }   

  (void) fprintf(printdev,"\n%s found in Image\n",PTS_str);
  (void) fprintf(printdev,"\nWindow size\t\t= %d\n", w);
  (void) fprintf(printdev,"Height to Width ratio\t= %f\n", hwr);
  (void) fprintf(printdev,"Number of %s specified to be found = %d\n",
                             pts_str, num_peaks);

  switch(image->data_storage_type) 
  {
     case VFF_TYP_FLOAT:
        for(i=0; i<num_signals; i++) 
        {
           (void)fprintf(printdev,"\n*******  %s #%d  ****** \n\n",
           	BV_str, i);
           (void)fprintf(printdev,"Number %s found\t\t\t%d\n",
                pts_str, vect_info[i].p_count);
           (void)fprintf(printdev,"Mean of %s distance\t%g\n",
                ptp_str, vect_info[i].p2p_dist_mean);
           (void)fprintf(printdev,"Variance of %s distance\t%g\n\n",
                      ptp_str, vect_info[i].p2p_dist_var);
           (void)fprintf(printdev,"POSITION\tVALUE\n");

           for(j=0; j<vect_info[i].p_count; j++) 
           {
              (void)fprintf(printdev,"%d\t\t%g\n", 
			peak_info[i][j].p_position,
                        peak_info[i][j].p_value);
           }
        }
        break;

     case VFF_TYP_COMPLEX:
        for(i=0; i<num_signals*2; i++) 
        {
           (void)fprintf(printdev,"\n*******  %s #%d  ****** \n",
                        BV_str, i/2);
           (void)fprintf(printdev,"\t\t\t\t\tREAL\t\tIMAGINARY\n");
           (void)fprintf(printdev,"Number %s found\t\t\t%d\t\t%d\n",
                      pts_str, vect_info[i].p_count, vect_info[i+1].p_count);
           (void)fprintf(printdev,"Mean of %s distance\t%g\t\t%g\n",
                      ptp_str, vect_info[i].p2p_dist_mean,
                      vect_info[i+1].p2p_dist_mean);
           (void)fprintf(printdev,"Variance of %s distance\t%g\t\t%g\n\n",
                      ptp_str, vect_info[i].p2p_dist_var,
                      vect_info[i+1].p2p_dist_var);

           (void)fprintf(printdev,"%s Found in Real Component of %s %d\n",
                                PTS_str, BV_str, i/2);
           (void)fprintf(printdev,"POSITION\tVALUE\n");

           for(j=0; j<vect_info[i].p_count; j++) 
           {
              (void)fprintf(printdev,"%d\t\t%g\n", 
			peak_info[i][j].p_position,
                        peak_info[i][j].p_value);
           }

           (void)fprintf(printdev,
			"\n%s Found in Imaginary Component of %s #%d\n",
                        PTS_str, BV_str, i/2);
           (void)fprintf(printdev,"POSITION\tVALUE\n");
           i++;
           for(j=0; j<vect_info[i].p_count; j++) 
           {
              (void)fprintf(printdev,"%d\t\t%g\n", 
			peak_info[i][j].p_position,
                        peak_info[i][j].p_value);
           }
        }
        break;

     default:
        (void)fprintf(stderr,"%s: unknown data storage type\n", "dpeakpick");
        return(0);
        break;
   }

   if (((*o1)=(AVSfield *)viff_to_field(image))==NULL) return (0);
 
   return (value);
}
 
int dstats_compute (	AVSfield *i1,AVSfield *i2,AVSfield **o1,
			char *fname,
			int save_all,int mean,int variance,
			int std_dev,int rms,int max_value,
			int x_max,int min_value,int x_min,int integral,
			int integral_pos,int integral_neg,
			int num_contrib, int num_positive,
			int num_negative,int skewness, int kurtosis,
			int entropy,int contrast,char *direction )
{
   int value;
   struct xvimage  *ki1,*ki2,*img3=NULL,*createimage(),*readimage();
   struct lvstat   **stats, *new_stats;
   FILE            *printdev=NULL;
   int             i, p,
   num_bands,              	/* number of bands in input image */
   opt_index;              	/* bands index for output image */

   int             num_opts;    /* num opts selected for VIFF output */
   float           *viff_out;   /* pointer to viff output */
   char            comment[512];
   char            *vbasename();
   char *program="dstats";

   ki1 = NULL;
   ki2 = NULL;

   if (i1)
      if ((ki1=(struct xvimage *)field_to_viff(i1))==NULL) return(0);
   if (i2)
      if ((ki2=(struct xvimage *)field_to_viff(i2))==NULL) return(0);

   if (fname)
   {
      if (! vwrite_ascii(fname, &printdev))
      {
         (void) fprintf(stderr, "%s:  Can't open ascii output file: \\n"
			,program);
         return(0);
      }
   }
   else 
   {
      printdev = stdout;
   }

/*
 * calculate number of stat vectors that we'll have
 */
   switch(AVSchoice_number("direction",direction))
   {
      case DSP_VECTOR:
         num_bands = ki1->row_size * ki1->col_size;
         break;
      case DSP_BAND:
         num_bands = ki1->num_data_bands;
         break;
      case 2:       /* single set of stats for entire file */
         num_bands = 1;
         break;
      default:
        (void)fprintf(stderr,"%s: unknown process direction (-d): %s\n",
                     program, direction);
        return(0);
   }

   stats = (struct lvstat **) malloc(num_bands * sizeof(struct lvstat *));
   if (stats == NULL)
   {
      (void)fprintf(stderr,"%s: No memory available for statistics structure\n",
                program);
      return(0);
   }

   for (i = 0 ; i < num_bands; i++)
   {
      stats[i] = (struct lvstat *) malloc(sizeof(struct lvstat));
      if (stats[i] == NULL)
      {
         (void)fprintf(stderr,
		"%s: No memory available for statistics structure\n",
                  program);
         return(0);
      }
   }

   if (! lvstats(ki1,ki2,MAP(ki2),stats,AVSchoice_number("direction",direction)))
   {
      (void) fprintf(stderr,"%s: lvstats Failed\n",program);
      return(0);
   }

   if (printdev!=NULL)
   {
      (void) fprintf(printdev,"\nStatistics for Input Image\n");

      if (MAP(ki2))
      {
         (void) fprintf(printdev,"Masking Image\n");
       }
       (void) fprintf(printdev,"\n");

       for (p=0 ; p < num_bands; p++)
       {
          new_stats = stats[p];
          if (AVSchoice_number("direction",direction)== DSP_VECTOR)
             (void)fprintf(printdev,
		"\n*******  Statistics for Vector #%d  ****** \n\n",p);
          if (AVSchoice_number("direction",direction)== DSP_BAND)
             (void)fprintf(printdev,
		"\n*******  Statistics for Band #%d  ****** \n\n",p);

         (void) fprintf(printdev,"  Mean: %g\n",new_stats->mean);
         (void) fprintf(printdev,"  Variance: %g\n",new_stats->var);
         (void) fprintf(printdev,"  Std. Dev: %g\n",new_stats->sdev);
         (void) fprintf(printdev,"  RMS:  %g\n",new_stats->rms);
         (void) fprintf(printdev,"  Peaks:\n");

         if (ki1->startx >= 0)  
/* print location in both subsignal and parent */
         {
            (void) fprintf(printdev,
		"    High: %g at (X) subsignal location: (%d):",
		new_stats->fmax,new_stats->maxp);
            (void) fprintf(printdev,
		" signal location: (%d)\n",
               	(new_stats->maxp + ki1->startx));
            (void) fprintf(printdev,
		"    Low : %g at (X) subsignal location: (%d):",
                new_stats->fmin,new_stats->minp);
            (void) fprintf(printdev,
		" signal location: (%d)\n",
                (new_stats->minp + ki1->startx));
         }
         else        /* signal is not a subsignal */
         {
            (void) fprintf(printdev,
	 	      	"    High: %g at (X) location: (%d)\n",
                      	new_stats->fmax,new_stats->maxp);
      	    (void) fprintf(printdev,
			"    Low : %g at (X) location: (%d)\n",
            new_stats->fmin,new_stats->minp);
     	 }
    	(void) fprintf(printdev,
		"  Total integral under the image: %g\n",
                new_stats->integ);
    	(void) fprintf(printdev,
		"  Positive part of integral under the image: %g\n",
                new_stats->pinteg);
    	(void) fprintf(printdev,
		"  Negative part of integral under the image: %g\n",
                new_stats->ninteg);
    	(void) fprintf(printdev,
		"  Contributing points: %d\n",new_stats->cpts);
    	(void) fprintf(printdev,
		"  Number of positive points in image: %d\n",
                   new_stats->pos);
    	(void) fprintf(printdev,
		"  Number of negative points in image: %d\n",
                   new_stats->neg);
    	if(new_stats->var == 0)
	{
           (void) fprintf(printdev,
		"  Skewness:  Not calculated (variance = 0)\n");
           (void) fprintf(printdev,
		"  Kurtosis:  Not calculated (variance = 0)\n");
        }
        else 
	{
           (void) fprintf(printdev,"  Skewness:  %g\n",new_stats->skewness);
           (void) fprintf(printdev,"  Kurtosis:  %g\n",new_stats->kurtosis);
        }

        if ( ki1->data_storage_type == VFF_TYP_1_BYTE)
        {
           (void) fprintf(printdev,"  Entropy:  %g\n",new_stats->entropy);
           (void) fprintf(printdev,"  Contrast:  %g\n",new_stats->contrast);
        }
      }
   }


  img3 = createimage((unsigned long) 1,                 /* col_size */
                     (unsigned long) num_bands,         /* row_size */
                     (unsigned long) VFF_TYP_FLOAT,     /* data_storage_type */
                     (unsigned long) 1,                 /* num_of_images */
                     (unsigned long) ALL_OPTS,          /* num_data_bands */
                     comment,                           /* comment */
                     (unsigned long) 0,                 /* map_row_size */
                     (unsigned long) 0,                 /* map_col_size */
                     (unsigned long) VFF_MS_NONE,       /* map_scheme */
                     (unsigned long) VFF_MAPTYP_NONE,   /* map_storage_type */
                     (unsigned long) VFF_LOC_IMPLICIT,  /* location_type */
                     (unsigned long) 0                  /* location_dim */
                    );

  if (img3 == NULL)
  {
    (void) fprintf(stderr,"%s: output file allocation failed\n",program);
    return(0);
  }

  (void)sprintf(comment,"Statistics for Image Input 1\n");
  append_comment(img3->comment, comment);

  if(MAP(i2))
  {
    (void)sprintf(comment,"Masking Image Input 2\n");
    append_comment(img3->comment, comment);
  }

  append_comment(img3->comment, "\n");

  num_opts = 0;

  if(save_all || mean)
  {
    (void)sprintf(comment,"Band %2d: Mean\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || variance)
  {
    (void)sprintf(comment,"Band %2d: Variance\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || std_dev)
  {
    (void)sprintf(comment,"Band %2d: Std Dev\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || rms)
  {
    (void)sprintf(comment,"Band %2d: RMS\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || max_value)
  {
    (void)sprintf(comment,"Band %2d: Max Val\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || x_max)
  {
    (void)sprintf(comment,"Band %2d: X Pos of Max Val\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || min_value)
  {
    (void)sprintf(comment,"Band %2d: Min Val\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || x_min)
  {
    (void)sprintf(comment,"Band %2d: X Pos of Min Val\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || integral)
  {
    (void)sprintf(comment,"Band %2d: Total Integral\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || integral_pos)
  {
    (void)sprintf(comment,"Band %2d: Pos Part of Integ\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || integral_neg)
  {
    (void)sprintf(comment,"Band %2d: Neg Part of Integ\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || num_contrib)
  {
    (void)sprintf(comment,"Band %2d: Total Contrib Pts\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || num_positive)
  {
    (void)sprintf(comment,"Band %2d: Pos Contrib Pts\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || num_negative)
  {
    (void)sprintf(comment,"Band %2d: Neg Contrib Pts\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || skewness )
  {
    (void)sprintf(comment,"Band %2d: Skewness\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || kurtosis)
  {
    (void)sprintf(comment,"Band %2d: Kurtosis\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || entropy)
  {
    (void)sprintf(comment,"Band %2d: Entropy\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || contrast)
  {
    (void)sprintf(comment,"Band %2d: Contrast\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  img3->num_data_bands = num_opts;

 /*
  * allocate for float array to contain stat data
  */

  viff_out = (float *)malloc((unsigned int)(num_opts * num_bands
              * sizeof(float)));
  if (viff_out == NULL)
  {
    (void)fprintf(stderr,"%s: cannot allocate for output info\n");
  }

  for (i=0; i<num_bands; i++)
  {
    opt_index = 0;
    new_stats = stats[i];

    if (save_all || mean)
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->mean;
      opt_index ++;
    }

    if (save_all || variance)
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->var;
      opt_index ++;
    }

    if (save_all || std_dev)
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->sdev;
      opt_index ++;
    }

    if (save_all || rms )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->rms;
      opt_index ++;
    }

    if (save_all || max_value)
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->fmax;
      opt_index ++;
    }

    if (save_all || x_max )
    {
      viff_out[i + (opt_index*num_bands)] = (float) new_stats->maxp;
      opt_index ++;
    }

    if (save_all || min_value)
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->fmin;
      opt_index ++;
    }

    if (save_all || x_min )
    {
      viff_out[i + (opt_index*num_bands)] = (float) new_stats->minp;
      opt_index ++;
    }

    if (save_all || integral )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->integ;
      opt_index ++;
    }

    if (save_all || integral_pos )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->pinteg;
      opt_index ++;
    }

    if (save_all || integral_neg )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->ninteg;
      opt_index ++;
    }

    if (save_all || num_contrib)
    {
      viff_out[i + (opt_index*num_bands)] = (float) new_stats->cpts;
      opt_index ++;
    }

    if (save_all || num_positive )
    {
      viff_out[i + (opt_index*num_bands)] = (float) new_stats->pos;
      opt_index ++;
    }

    if (save_all || num_negative )
    {
      viff_out[i + (opt_index*num_bands)] = (float) new_stats->neg;
      opt_index ++;
    }

    if (save_all || skewness )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->skewness;
      opt_index ++;
    }

    if (save_all || kurtosis )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->kurtosis;
      opt_index ++;
    }

    if (save_all || entropy )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->entropy;
      opt_index ++;
    }

    if (save_all || contrast )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->contrast;
      /*opt_index ++;*/
    }

  }  /* end of for loop throught image bands */


  img3->imagedata = (char *)viff_out;

   if (((*o1)=(AVSfield *)viff_to_field(img3))==NULL) return (0);
   
   for (i=0; i<num_bands; i++)
   {
      free((struct lvstat *)stats[i]);
   }
   free((struct lvstat **)stats);

   return (1);
}
 
int varviff_compute (	AVSfield *i1,
			int x, int y, int band )
{
   int value;
   struct xvimage *ki1=NULL;
   double var_out;
 
   if (i1)
      if ((ki1=(struct xvimage *)field_to_viff(i1))==NULL) return(0);

   value=(int)lvarviff(ki1,x,y,band,&var_out);
 
   fprintf(stderr,"Varviff Variable Output %f \n",var_out);
 
   return (value);
}
 
int vfileinfo_compute (AVSfield *i1,char *fname)
{
   int value;
   struct xvimage *ki1=NULL;
 
   FILE *fp;		

   /* fp and fname refer to the file being written to */
   FILE_OPEN(fp,fname);

   if (i1)
      if ((ki1=(struct xvimage *)field_to_viff(i1))==NULL) return(0);

   /***********************************************************************/
   /* NOTE: fname is not the correct variable to pass into lvfileinfo()   */ 
   /*	    because it is the name of the file being written to and *not* */ 
   /*	    the name of the image file.  However xvimage does not contain */
   /*	    the image name.						  */ 
   /***********************************************************************/

   value=(int)lvfileinfo(ki1,fname,fp);
 
   FILE_CLOSE(fp,fname);

   return (value);
}
 
int vprdata_compute (AVSfield *i1,char *fname,char *form)
{
   int value;
   struct xvimage *ki1=NULL;
   FILE *fp;
   int choice;

   FILE_OPEN(fp,fname);
 
   if (i1)
      if ((ki1=(struct xvimage *)field_to_viff(i1))==NULL) return(0);

   /***********************************************************************/
   /* NOTE: fname is not the correct variable to pass into lvprdata()     */ 
   /*	    because it is the name of the file being written to and *not* */ 
   /*	    the name of the image file.  However xvimage does not contain */
   /*	    the image name.						  */ 
   /***********************************************************************/

   /* lvprdata expects 0-based choices but AVSchoice_number returns       */
   /* 1-based choices, so subtract 1 from the choice                      */

   choice = AVSchoice_number("form",form);
   choice--;

   value=(int)lvprdata(ki1,choice,fname,fp);
 
   FILE_CLOSE(fp,fname);

   return (value);
}
 
int vstats_compute (	AVSfield *i1,AVSfield *i2,AVSfield **o1,
			char *fname,
                        int save_all,int mean,int variance,
                        int std_dev,int rms,int max_value,
                        int x_max,int y_max,int min_value,
			int x_min,int y_min,int integral,
                        int integral_pos,int integral_neg,
                        int num_contrib, int num_positive,
                        int num_negative,int skewness, int kurtosis,
                        int entropy,int contrast,int separate )
{
   struct xvimage *ki1=NULL;
   struct xvimage *ki2=NULL;
   struct xvimage *ko1=NULL;
struct xvimage  *img1=NULL,*img2=NULL,*img3=NULL,*createimage(),*readimage();
struct lvstat   **stats, *new_stats;
FILE            *printdev;
int             i, p,
                num_bands,              /* number of bands in input image */
                opt_index,              /* bands index for output image */
                process_dir;            /* 1=bands, 2=combine bands */

int             num_opts;               /* num opts selected for VIFF output */
float           *viff_out;              /* pointer to viff output */
char            *program = "vstats";
char            comment[512];

   if (i1)
      if ((img1=ki1=(struct xvimage *)field_to_viff(i1))==NULL) return(0);
   if (i2)
      if ((img2=ki2=(struct xvimage *)field_to_viff(i2))==NULL) return(0);

/*
 * Check to see that at least one stat is specified for the VIFF output file 
 */
if ( !save_all   && !mean &&
    !variance && !std_dev    && !rms &&
    !min_value && !x_min  && !y_min  &&
    !max_value && !x_max  && !y_max  &&
    !integral   && !integral_pos   && !integral_pos  &&
    !num_contrib  && !num_positive  && !num_negative &&
    !skewness   && !kurtosis   && !entropy   &&
    !contrast )
{
  (void)fprintf(stderr,"%s: If VIFF output file is selected, at least",program);
  (void)fprintf(stderr,"one statistic should be selected\n");
   return(0);
}

/*
 * If all flag is TRUE and any other stat flag is TRUE, set all flag to FALSE
 */
if (save_all   && (	mean   || variance  || std_dev || rms   || min_value || x_min  || y_min  || max_value || x_max  || y_max     || integral || integral_pos || integral_neg || num_contrib  || num_positive || num_negative || skewness   || kurtosis   || entropy   || contrast ) )
{
  save_all = FALSE;
}


if (fname) 		 
{
  if (! vwrite_ascii(fname, &printdev))
  {
    (void) fprintf(stderr, "%s:  Can't open ascii output file: \n",program);
    return(0);
  }
}
else 
   return(0);

if (separate)  /* calculate stats of each band separately */
{
  process_dir = 1;      /* DSP_BAND */
  num_bands = img1->num_data_bands;
}
else if (!separate)  /* single set of stats for entire file */
{
  process_dir = 2;
  num_bands = 1;
}

stats = (struct lvstat **) malloc(num_bands * sizeof(struct lvstat *));
if (stats == NULL)
{
  (void)fprintf(stderr,"%s: No memory available for statistics structure\n",
                program);
  return(0);
}

for (i = 0 ; i < num_bands; i++)
{
  stats[i] = (struct lvstat *) malloc(sizeof(struct lvstat));
  if (stats[i] == NULL)
  {
    (void)fprintf(stderr,"%s: No memory available for statistics structure\n",
                  program);
   return(0);
  }
}

if (! lvstats(ki1,ki2,MAP(ki2),stats,process_dir))
{
  (void) fprintf(stderr,"%s: lvstats Failed\n",program);
  return(0);
}

if (fname)
{
  (void) fprintf(printdev,"\nImage Statistics for Input Image 1 \n");

  if (MAP(ki2))
  {
    (void) fprintf(printdev,"Masking File Name: Input Image 2 \n");
  }
  (void)fprintf(printdev,"\n");
  
  for (p=0 ; p < num_bands; p++)
  {
    new_stats = stats[p];
    if (separate)
      (void)fprintf(printdev,"\n*******  Statistics for Band #%d  ****** \n\n",
                    p);

    (void) fprintf(printdev,"  Mean: %g\n",new_stats->mean);
    (void) fprintf(printdev,"  Variance: %g\n",new_stats->var);
    (void) fprintf(printdev,"  Std. Dev: %g\n",new_stats->sdev);
    (void) fprintf(printdev,"  RMS:  %g\n",new_stats->rms);
    (void) fprintf(printdev,"  Peaks:\n");

    if ((img1->startx >= 0) || (img1->starty >= 0))
    /* image is a subimage, so print out location of both it and its parent */
    {
      (void) fprintf(printdev,"    High: %g at (X,Y) subimage location: (%d,%d)",
                     new_stats->fmax,new_stats->maxx,new_stats->maxy);
      (void) fprintf(printdev,": image location: (%d,%d)\n",
                     (new_stats->maxx + img1->startx),
                     (new_stats->maxy + img1->starty));
      (void) fprintf(printdev,"    Low : %g at (X,Y) subimage location: (%d,%d)",
                     new_stats->fmin,new_stats->minx,new_stats->miny);
      (void) fprintf(printdev,": image location: (%d,%d)\n",
                     (new_stats->minx + img1->startx),
                     (new_stats->miny + img1->starty));
    }
    else         /* image is not a subimage */
    {
      (void) fprintf(printdev,"    High: %g at (X,Y) location: (%d,%d)\n",
                     new_stats->fmax,new_stats->maxx,new_stats->maxy);
      (void) fprintf(printdev,"    Low : %g at (X,Y) location: (%d,%d)\n",
                     new_stats->fmin,new_stats->minx,new_stats->miny);
    }

    (void) fprintf(printdev,"  Total integral under the image: %g\n",
                   new_stats->integ);
    (void) fprintf(printdev,"  Positive part of integral under the image: %g\n",
                   new_stats->pinteg);
    (void) fprintf(printdev,"  Negative part of integral under the image: %g\n",
                   new_stats->ninteg);
    (void) fprintf(printdev,"  Contributing points: %d\n",new_stats->cpts);
    (void) fprintf(printdev,"  Number of positive points in image: %d\n",
                   new_stats->pos);
    (void) fprintf(printdev,"  Number of negative points in image: %d\n",
                   new_stats->neg);
    if(new_stats->var == 0){
      (void) fprintf(printdev,"  Skewness:  Not calculated (variance = 0)\n");
      (void) fprintf(printdev,"  Kurtosis:  Not calculated (variance = 0)\n");
    }
    else {
      (void) fprintf(printdev,"  Skewness:  %g\n",new_stats->skewness);
      (void) fprintf(printdev,"  Kurtosis:  %g\n",new_stats->kurtosis);
    }
    if ( img1->data_storage_type == VFF_TYP_1_BYTE)
    {
      (void) fprintf(printdev,"  Entropy:  %g\n",new_stats->entropy);
      (void) fprintf(printdev,"  Contrast:  %g\n",new_stats->contrast);
    }
  }
}

  img3 = createimage((unsigned long) 1,                 /* col_size */
                     (unsigned long) num_bands,         /* row_size */
                     (unsigned long) VFF_TYP_FLOAT,     /* data_storage_type */
                     (unsigned long) 1,                 /* num_of_images */
                     (unsigned long) ALL_OPTS,          /* num_data_bands */
                     comment,                           /* comment */
                     (unsigned long) 0,                 /* map_row_size */
                     (unsigned long) 0,                 /* map_col_size */
                     (unsigned long) VFF_MS_NONE,       /* map_scheme */
                     (unsigned long) VFF_MAPTYP_NONE,   /* map_storage_type */
                     (unsigned long) VFF_LOC_IMPLICIT,  /* location_type */
                     (unsigned long) 0                  /* location_dim */
                    );

  if (img3 == NULL)
  {
    (void) fprintf(stderr,"%s: output file allocation failed\n",program);
    return(0);
  }

  (void)sprintf(comment,"Image Statistics for Input Image 1 \n");
  append_comment(img3->comment, comment);

  if(MAP(ki2))
  {
    (void)sprintf(comment,"Masking Input Image 2 \n");
    append_comment(img3->comment, comment);
  }

  append_comment(img3->comment, "\n");

  num_opts = 0;
    
  if(save_all || mean)
  {
    (void)sprintf(comment,"Band %2d: Mean\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || variance)
  {
    (void)sprintf(comment,"Band %2d: Variance\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || std_dev)
  {
    (void)sprintf(comment,"Band %2d: Std Dev\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || rms)
  {
    (void)sprintf(comment,"Band %2d: RMS\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || max_value )
  {
    (void)sprintf(comment,"Band %2d: Max Val\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || x_max)
  {
    (void)sprintf(comment,"Band %2d: X Pos of Max Val\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || y_max )
  {
    (void)sprintf(comment,"Band %2d: Y Pos of Max Val\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || min_value )
  {
    (void)sprintf(comment,"Band %2d: Min Val\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || x_min)
  {
    (void)sprintf(comment,"Band %2d: X Pos of Min Val\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || y_min )
  {
    (void)sprintf(comment,"Band %2d: Y Pos of Min Val\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || integral )
  {
    (void)sprintf(comment,"Band %2d: Total Integral\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || integral_pos )
  {
    (void)sprintf(comment,"Band %2d: Pos Part of Integ\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || integral_neg )
  {
    (void)sprintf(comment,"Band %2d: Neg Part of Integ\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || num_contrib )
  {
    (void)sprintf(comment,"Band %2d: Total Contrib Pts\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || num_positive )
  {
    (void)sprintf(comment,"Band %2d: Pos Contrib Pts\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || num_negative )
  {
    (void)sprintf(comment,"Band %2d: Neg Contrib Pts\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || skewness)
  {
    (void)sprintf(comment,"Band %2d: Skewness\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || kurtosis)
  {
    (void)sprintf(comment,"Band %2d: Kurtosis\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || entropy)
  {
    (void)sprintf(comment,"Band %2d: Entropy\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  if(save_all || contrast)
  {
    (void)sprintf(comment,"Band %2d: Contrast\n",num_opts);
    append_comment(img3->comment, comment);
    num_opts ++;
  }

  img3->num_data_bands = num_opts;

 /*
  * allocate for float array to contain stat data 
  */

  viff_out = (float *)malloc((unsigned int)(num_opts * num_bands 
              * sizeof(float)));
  if (viff_out == NULL)
  {
    (void)fprintf(stderr,"%s: cannot allocate for output info\n");
  }

  for (i=0; i<num_bands; i++)
  {
    opt_index = 0;
    new_stats = stats[i];

    if (save_all || mean )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->mean;
      opt_index ++;
    }

    if (save_all || variance )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->var;
      opt_index ++;
    }

    if (save_all || std_dev)
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->sdev;
      opt_index ++;
    }

    if (save_all || rms)
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->rms;
      opt_index ++;
    }

    if (save_all || max_value )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->fmax;
      opt_index ++;
    }

    if (save_all || x_max)
    {
      viff_out[i + (opt_index*num_bands)] = (float) new_stats->maxx;
      opt_index ++;
    }

    if (save_all || y_max )
    {
      viff_out[i + (opt_index*num_bands)] = (float) new_stats->maxy;
      opt_index ++;
    }

    if (save_all || min_value )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->fmin;
      opt_index ++;
    }

    if (save_all || x_min)
    {
      viff_out[i + (opt_index*num_bands)] = (float) new_stats->minx;
      opt_index ++;
    }

    if (save_all || y_min )
    {
      viff_out[i + (opt_index*num_bands)] = (float) new_stats->miny;
      opt_index ++;
    }

    if (save_all || integral )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->integ;
      opt_index ++;
    }

    if (save_all || integral_pos )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->pinteg;
      opt_index ++;
    }

    if (save_all || integral_neg )
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->ninteg;
      opt_index ++;
    }

    if (save_all || num_contrib )
    {
      viff_out[i + (opt_index*num_bands)] = (float) new_stats->cpts;
      opt_index ++;
    }

    if (save_all || num_positive )
    {
      viff_out[i + (opt_index*num_bands)] = (float) new_stats->pos;
      opt_index ++;
    }

    if (save_all || num_negative )
    {
      viff_out[i + (opt_index*num_bands)] = (float) new_stats->neg;
      opt_index ++;
    }

    if (save_all || skewness)
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->skewness;
      opt_index ++;
    }

    if (save_all || kurtosis)
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->kurtosis;
      opt_index ++;
    }

    if (save_all || entropy)
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->entropy;
      opt_index ++;
    }

    if (save_all || contrast)
    {
      viff_out[i + (opt_index*num_bands)] = new_stats->contrast;
      opt_index ++;
    }

  }  /* end of for loop throught image bands */


  img3->imagedata = (char *)viff_out;

for (i=0; i<num_bands; i++)       
{
  free((struct lvstat *)stats[i]);
}
free((struct lvstat **)stats);

   if (((*o1)=(AVSfield *)viff_to_field(img3))==NULL) return (0);
   return (1);
}
 
/* ***********************************************************************/
/* Initialization for modules contained in this file.                    */
/* ***********************************************************************/
int ((*mod_list[])()) = {
   dpeakpick_desc,
   dstats_desc,
   varviff_desc,
/* vcomment_desc, */
   vfileinfo_desc,
/* vheaded_desc, */
   vprdata_desc,
/* vprmap_desc, */
   vstats_desc,
};

#define NMODS (sizeof(mod_list) / sizeof(char *))

AVSinit_modules()
{
        AVSinit_from_module_list(mod_list, NMODS);
}

