 /*
  * Khoros: $Id: lvgpwl.c,v 1.3 1992/03/20 23:05:44 dkhoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: lvgpwl.c,v 1.3 1992/03/20 23:05:44 dkhoros Exp $";
#endif

 /*
  * $Log: lvgpwl.c,v $
 * Revision 1.3  1992/03/20  23:05:44  dkhoros
 * VirtualPatch5
 *
  */

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1992, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1992 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lvgpwl.c
 >>>>
 >>>>      Program Name: vgpwl
 >>>>
 >>>> Date Last Updated: Thu Mar  5 08:28:33 1992 
 >>>>
 >>>>          Routines: lvgpwl - the library call for vgpwl
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lvgpwl - library call for vgpwl
*
* Purpose:
*    
*    Generates a set of piecewise linear periodic image.
*    
*    

* Input:
*    
*    image          resulting image.
*    
*    rows           number of rows in image
*    
*    cols           number of columns in image
*    
*    fs             sampling frequency
*    
*    minimum        minimum value of function
*    
*    maximum        maximum value of function
*    
*    xperiod        X period of the function in seconds
*    
*    yperiod        Y period of the function in seconds
*    
*    xrise          X rise time in seconds
*    
*    yrise          Y rise time in seconds
*    
*    xfall          X fall time in seconds
*    
*    yfall          Y fall time in seconds
*    
*    xwidth         X width of pulse
*    
*    ywidth         Y width of pulse
*    
*    type           data type of output image.  This  must  be  either
*                   VFF_TYP_FLOAT or VFF_TYP_COMPLEX.
*    
*    

* Output:
*    
*    image          holds the result of the data generation.
*    
*    return Value:  1 on success, 0 on failure.
*    
*    

*
* Written By: Jeremy Worley
*    
*    Jeremy Worley 05 Mar 1992 08:26 MST
*              Fixed up an fprintf function that had too few arguments
*              based  on the format field.  Added rows,cols, and bands
*              to  the  list  of  explicitly  declared  arguments   to
*              lvgpwl().
*    
*    

****************************************************************/


/* -library_def */
int lvgpwl(image,rows,cols,bands,fs,minimum,maximum,xperiod,yperiod,
           xrise,yrise,xfall,yfall,xwidth,ywidth,type)
    struct xvimage **image;
    int    type,bands,rows,cols;
    double  fs,minimum,maximum,xperiod,yperiod,xrise,yrise,xfall,yfall,
           xwidth,ywidth;
/* -library_def_end */

/* -library_code */
{
    char *program="lvgpwl";
  char  *comment="signal ensemble courtesy of lvgpwl.";
  int   j,b;
  float *rpart,*ipart=NULL, *data;
  
  int   gsin();
  struct xvimage *createimage();

/*
** Immediately check bounds on arguments. 
*/

  if(rows<1 || cols<1 || bands<1){
     fprintf(stderr,"%s:  Image dimension out of bounds.\n",program);
     return(0);
  }else if(fs<=0.0){
     fprintf(stderr,"%s:  Sampling frequency out of bounds.\n",program);
     return(0);
  }else if(xperiod<=0.0 || yperiod<=0.0){
     fprintf(stderr,"%s:  Period out of bounds.\n",program);
     return(0);
  }else if(type!=VFF_TYP_FLOAT && type!=VFF_TYP_COMPLEX){
     fprintf(stderr,"%s:  Illegal image data type specified.\n",program);
     return(0);
  }else if(xrise < 0.0 || xfall < 0.0 || yrise < 0.0 || yfall < 0.0){
     fprintf(stderr,"%s:  Transition time must be non negative.\n",program);
     return(0);
  }

/*
** Generate image
*/
  
  *image = createimage(rows,cols,type,1,bands,comment,
                      (unsigned long)0, /* map_row_size */
                      (unsigned long)0, /* map_col_size */
                      (unsigned long)VFF_MS_NONE,
                      (unsigned long)VFF_MAPTYP_NONE,
                      (unsigned long)VFF_LOC_IMPLICIT,
                      (unsigned long)0);/* location dimension */

  if(*image==NULL){
     fprintf(stderr,"%s:  Unable to allocate memory for image.\n",program);
     return(0);
  }

  data = (float *)(*image)->imagedata;

/*
** malloc appropriate amount of memory for real part of signal.
** if there are any errors along the way, free up whatever memory was 
** allocated before exiting.
*/

  rpart = (float *)malloc(rows*cols*sizeof(float));
  if(rpart==NULL){
     fprintf(stderr,"%s:  Unable to malloc temporary workspace.\n",program);
     return(0);
  }

/*
** if this is a complex image, malloc appropriate amount of 
** memory for the imaginary part.  if there are any errors along the way, 
** free up whatever memory was allocated before exiting.
*/
 
  if(type==VFF_TYP_COMPLEX){
     ipart = (float *)malloc(rows*cols*sizeof(float));
     if(ipart==NULL){
        fprintf(stderr,"%s:  Unable to malloc temporary workspace.\n",program);
        free(rpart);
        return(0);
     }
  }

/*
** loop through all bands
*/

  for(b=0;b<bands;b++){

/*
** call low level sine wave generator for real part.
*/

     j=gpwl(rpart,rows,cols,fs,minimum,maximum,yperiod,xperiod,
            yrise,xrise,yfall,xfall,ywidth,xwidth);
     if(j==0){
        fprintf(stderr,"%s:  Error during signal generation\n",program);
        free(rpart);
        if(type==VFF_TYP_COMPLEX)free(ipart);
        return(0);
     }

/*
** call low level sine wave generator for imaginary part.
*/

     if(type==VFF_TYP_COMPLEX){
        j=gpwl(ipart,rows,cols,fs,minimum,maximum,yperiod,xperiod,
               yrise,xrise,yfall,xfall,ywidth,xwidth);
        if(j==0){
           fprintf(stderr,"%s:  Error during signal generation\n",program);
           free(rpart);
           free(ipart);
           return(0);
        }
     }

/*
** interlace real and imaginary parts if necessary
** reorganize data and place it in ensemble
*/

     if(type==VFF_TYP_COMPLEX){
        for(j=0;j<rows*cols;j++){
            *data++ = rpart[j];
            *data++ = ipart[j];
        }
     }else{
        bcopy(rpart,data,rows*cols*sizeof(float));
        data += rows*cols;
     }
  }

/*
** clean up mess
*/ 

  free(rpart);
  if(type==VFF_TYP_COMPLEX)free(ipart);

/*
** normal return
*/

  return(1);

}
/* -library_code_end */
