
#include <stdio.h>

#include <avs/avs.h>
#include <avs/field.h>

#include "vinclude.h"
#include "file_formats/fits.h"
#include "file_formats/pbm.h"
#include "file_formats/tga.h"
#include "file_formats/tiffio.h"
#include "file_formats/rast.h"
#include "proto.h"
#include "standard.h"

#include "avskhoros.h"

/* *****************************************/
/*  Module Descriptions                    */
/* *****************************************/

int fits2viff_desc ()
{
   int in_port, out_port, param;
   extern int fits2viff_compute();

   AVSset_module_name("fits2AVS",MODULE_DATA);

   ADD_FILE;

   AVScreate_output_port("image","field 2D");

   AVSadd_parameter("color","choice","no model",
	"no model,ntscRGB model,genericRGB model,cieRGB model,ntscCMY model,ntscYIQ model,HSV model,HLS model,IHS model,cieXYZ model,cieUVW model,cieucsUVW mod,cieucsSOW mod,cieucsLab mod,GENERIC model",
	",");

   AVSset_compute_proc(fits2viff_compute);
   return(1);
}

int mat2viff_desc ()
{
   int in_port, out_port, param;
   extern int mat2viff_compute();

   AVSset_module_name("mat2AVS",MODULE_DATA);

   ADD_FILE;

   AVScreate_output_port("image","field 2D");

   AVSset_compute_proc(mat2viff_compute);
   return(1);
}

int pbm2viff_desc ()
{
   int in_port, out_port, param;
   extern int pbm2viff_compute();

   AVSset_module_name("pbm2AVS",MODULE_DATA);

   ADD_FILE;
   AVScreate_output_port("Output VIFF Image","field 2D");

   AVSset_compute_proc(pbm2viff_compute);
   return(1);
}

int rast2viff_desc ()
{
   int in_port, out_port, param;
   extern int rast2viff_compute();

   AVSset_module_name("rast2AVS",MODULE_DATA);

   ADD_FILE;
   AVScreate_output_port("Output VIFF File ","field 2D");
   AVSadd_parameter("invert image","choice","photoPOSITIVE","photoPOSITIVE|photoNEGATIVE","|");

   AVSset_compute_proc(rast2viff_compute);
   return(1);
}

int tga2viff_desc ()
{
   int in_port, out_port, param;
   extern int tga2viff_compute();

   AVSset_module_name("tga2AVS",MODULE_DATA);

   ADD_FILE;
   AVScreate_output_port("Output VIFF file ","field 2D");
   AVSadd_parameter("byte skip","integer",0,0,512);

   AVSset_compute_proc(tga2viff_compute);
   return(1);
}

int tiff2viff_desc ()
{
   int in_port, out_port, param;
   extern int tiff2viff_compute();

   AVSset_module_name("tiff2AVS",MODULE_DATA);

   AVScreate_output_port("Input TIFF Image","field 2D");
   ADD_FILE;
   AVSadd_parameter("Verbose","boolean",0,0,1);

   AVSset_compute_proc(tiff2viff_compute);
   return(1);
}

int viff2fits_desc ()
{
   int in_port, out_port, param;
   extern int viff2fits_compute();

   AVSset_module_name("AVS2fits",MODULE_RENDER);

   AVScreate_input_port("Input VIFF file ","field 2D",REQUIRED);
   ADD_FILE;

   AVSset_compute_proc(viff2fits_compute);
   return(1);
}

int viff2mat_desc ()
{
   int in_port, out_port, param;
   extern int viff2mat_compute();

   AVSset_module_name("AVS2mat",MODULE_RENDER);

   AVScreate_input_port("Input File ","field 2D",REQUIRED);
   ADD_FILE;
   AVSadd_parameter("elements/line","integer",10,1,999);
   AVSadd_parameter("field size","integer",10,1,999);
   AVSadd_parameter("fraction digits","integer",6,1,999);
   AVSadd_parameter("justification","choice","right","right|left","|");
   AVSadd_parameter("bracket type","choice","square","square|triangular","|");
   AVSadd_parameter("delimeter","choice","space","space|comma","|");
   AVSadd_parameter("format","choice","int","int|float|scientific","|");
   AVSset_compute_proc(viff2mat_compute);
   return(1);
}

int viff2pbm_desc ()
{
   int in_port, out_port, param;
   extern int viff2pbm_compute();

   AVSset_module_name("AVS2pbm",MODULE_RENDER);

   AVScreate_input_port("Input viff image","field 2D",REQUIRED);
   ADD_FILE;
   AVSadd_parameter("format","choice","ASCII","ASCII|RAW","|");

   AVSset_compute_proc(viff2pbm_compute);
   return(1);
}

int viff2rast_desc ()
{
   int in_port, out_port, param;
   extern int viff2rast_compute();

   AVSset_module_name("AVS2rast",MODULE_RENDER);

   AVScreate_input_port("Input VIFF file   ","field 2D",REQUIRED);
   ADD_FILE;

   AVSadd_parameter("invert image","choice",
"photoPOSITIVE","photoPOSITIVE|photoNEGATIVE","|");

   AVSset_compute_proc(viff2rast_compute);
   return(1);
}

int viff2tiff_desc ()
{
   int in_port, out_port, param;
   extern int viff2tiff_compute();

   AVSset_module_name("AVS2tiff",MODULE_RENDER);

   AVScreate_input_port("Input VIFF Image    ","field 2D",REQUIRED);
   ADD_FILE;
   AVSadd_parameter("compression","choice","None","None,LZW,CCITTFAX3,CCITTRLEW,PACKBITS",",");
   AVSadd_parameter("verbose","boolean",0,0,1);

   AVSset_compute_proc(viff2tiff_compute);
   return(1);
}

int viff2xbm_desc ()
{
   int in_port, out_port, param;
   extern int viff2xbm_compute();

   AVSset_module_name("AVS2xbm",MODULE_RENDER);

   AVScreate_input_port("Input Image    ","field 2D",REQUIRED);
   ADD_FILE;
   AVSadd_parameter("description","string","avs image",NULL,NULL);
   AVSadd_parameter("invert image","choice","photoPOSITIVE","photoPOSITIVE|photoNEGATIVE","|");

   AVSset_compute_proc(viff2xbm_compute);
   return(1);
}

int xbm2viff_desc ()
{
   int in_port, out_port, param;
   extern int xbm2viff_compute();

   AVSset_module_name("xbm2AVS",MODULE_RENDER);

   ADD_FILE;
   AVScreate_output_port("Output Image","field 2D");
   AVSadd_parameter("photoNEGATIVE","boolean",0,0,1);

   AVSset_compute_proc(xbm2viff_compute);
   return(1);
}

/* *****************************************/
/* Module Compute Routines                 */
/* *****************************************/

int fits2viff_compute (AVSfield **o1,char *parm1,char *color)
{
   struct xvimage *ko1;
   struct fits *read_fits(), *fits_image;
   int value;

   fits_image=read_fits(parm1);

   if(fits_image == NULL)
   {
      (void) fprintf(stderr,"fits2viff: Image could not be read\n");
      return(0); 
   }
   
   value=lfits2viff(fits_image,&ko1,AVSchoice_number("color")); 

   if (((*o1)=(AVSfield *)viff_to_field(ko1))==NULL) return (0);
 
   return (value);
}
 
int mat2viff_compute (AVSfield **o1,char *fname)
{
   int value;
   struct xvimage *ko1;
 
   value=lmat2viff(&ko1,fname);

   if (((*o1)=(AVSfield *)viff_to_field(ko1))==NULL) return (0);
 
   return (value);
}
 
int pbm2viff_compute (AVSfield **o1, char *fname)
{
   int value;
   struct xvimage *ko1;
   struct pbm *pbm_image, *read_pbm();
 
   pbm_image = read_pbm(fname);
   if (pbm_image == NULL)
   {
      (void) fprintf(stderr,"pbm2viff: Could not read input image.\n");
      return(1);
   }

   value=lpbm2viff(pbm_image,&ko1);

   if (((*o1)=(AVSfield *)viff_to_field(ko1))==NULL) return (0);
 
   return (value);
}
 
int rast2viff_compute (AVSfield **o1,char *parm1,int parm2)
{
   int value;
   struct xvimage *ko1;
   struct rast *read_rast(), *rast_image;
 
   rast_image = read_rast(parm1);
   if (rast_image == NULL)
   {
      (void) fprintf(stderr,"rast2viff: Could not read input image.\n");
      exit(1);

   }

   value=lrast2viff(rast_image,&ko1,parm2);

   if (((*o1)=(AVSfield *)viff_to_field(ko1))==NULL) return (0);
 
   return (value);
}
 
int tga2viff_compute (AVSfield **o1,char *parm1,int parm2)
{
   int value;
   struct xvimage *ko1;
   tga *tga_image, *read_tga();

   int file; if ( (file = kopen(parm1, O_RDONLY)) < 0)
   {
      fprintf(stderr,"tga2viff: Unable to access file %s\n",parm1);
      return(0);
   }
   tga_image = read_tga(file,parm2);

   value=ltga2viff(tga_image,&ko1);

   if (((*o1)=(AVSfield *)viff_to_field(ko1))==NULL) return (0);
 
   return (value);
}
 
int tiff2viff_compute (AVSfield **o1,char *filename,int verbose)
{
   int value;
   struct xvimage *ko1;
   TIFF *tif;

   tif = TIFFOpen(filename, "r");
   if (tif == NULL)
   {
      fprintf(stderr,"tiff2viff: unable to read the TIFF image '%s'\n",filename);
      return(0);
   }
 
   value=ltiff2viff(tif,&ko1,verbose,filename);

   if (((*o1)=(AVSfield *)viff_to_field(ko1))==NULL) return (0);
 
   return (value);
}
 
int viff2fits_compute (AVSfield *i1,char *fname)
{
   int value;
   struct xvimage *ki1;
   struct fits     *fits_image;
 
   if (i1)
      if ((ki1=(struct xvimage *)field_to_viff(i1))==NULL) return(0);

   value=lviff2fits(ki1,&fits_image);

   if (!write_fits(fname,fits_image)) 
   {
      (void)fprintf(stderr,"write_fits Failed\n");
      return(0);
   }

   return (value);
}
 
int viff2mat_compute (AVSfield *i1,char *fname,int ele,int size,int digits,char *just,char *brack,char *delim,char *format)
{
   int value,errnum;
   struct xvimage *ki1;
   char   style[20], temp[20];
 
   if (i1)
      if ((ki1=(struct xvimage *)field_to_viff(i1))==NULL) return(0);

   /*
    *  Start of building the specified control string for lviff2mat. 
    */

   strcpy(style,"%");

   switch (AVSchoice_number("justification"))
   {
      case 0: 
         break;
      case 1: 
         strcat(style,"-");
         break;
   }

   (void)sprintf(temp,"%d",size);
   strcat(style,temp);

   strcat(style,".");
      
   if (AVSchoice_number("format"))
   {
      (void)sprintf(temp,"%d",digits);
      strcat(style,temp);
   }

   switch (AVSchoice_number("format"))
   {
      case 0: /* Integer style output requested. */
         strcat(style,"0f");
         break;
      case 1: /* Floating point style requested. */
         strcat(style,"f");
         break;
      case 2: /* Scientific notation requested. */
         strcat(style,"e");
         break;
      }

   value=lviff2mat(fname,ki1,style,ele,AVSchoice_number("delimeter"),AVSchoice_number("bracket type"),&errnum);

   return (value);
}
 
int viff2pbm_compute (AVSfield *i1,char *fname,char *format)
{
   int value;
   struct xvimage *ki1;
   struct pbm      *pbm;
 
   if (i1)
      if ((ki1=(struct xvimage *)field_to_viff(i1))==NULL) return(0);

   value=lviff2pbm(ki1, &pbm, AVSchoice_number("format"));

   write_pbm(fname, pbm);


   return (value);
}
 
int viff2rast_compute (AVSfield *i1,char *fname,int invert)
{
   int value;
   struct xvimage *ki1;
   struct rast *rast_image;
 
   if (i1)
      if ((ki1=(struct xvimage *)field_to_viff(i1))==NULL) return(0);

   value=lviff2rast(ki1, &rast_image, invert);

   write_rast(fname, rast_image);

   return (value);
}
 
int viff2tiff_compute (AVSfield *i1,char *fname,char *comp,int verbose)
{
   int value;
   struct xvimage *ki1;
 
   if (i1)
      if ((ki1=(struct xvimage *)field_to_viff(i1))==NULL) return(0);

   return (value);
}
 
int viff2xbm_compute (AVSfield *i1,char *fname,char *name,int invert)
{
   struct xvimage *ki1;
   FILE *fp;
 
   FILE_OPEN(fp,fname);

   if (i1)
      if ((ki1=(struct xvimage *)field_to_viff(i1))==NULL) return(0);

   if (! lviff2xbm(fp, ki1, name, invert))
   {
      (void) fprintf(stderr, "lviff2xbm Failed\n");
      return(0);
   }

   FILE_CLOSE(fp,fname);

   return (1);
}
 
int xbm2viff_compute (AVSfield **o1,char *fname,int invert)
{
   int value;
   struct xvimage *ko1;
   FILE *fp;

   FILE_OPEN(fp,fname);
 
   value=lxbm2viff(fp,&ko1,invert);

   if (((*o1)=(AVSfield *)viff_to_field(ko1))==NULL) return (0);
 
   FILE_CLOSE(fp,fname);
  
   return(value);
}
 
/********************************************************************/
/* Initialization for modules contained in this file.               */
/********************************************************************/
int ((*mod_list[])()) = {
   fits2viff_desc,
   mat2viff_desc,
   pbm2viff_desc,
   rast2viff_desc,
   tga2viff_desc,
   tiff2viff_desc,
   viff2fits_desc,
   viff2mat_desc,
   viff2pbm_desc,
   viff2rast_desc,
   viff2tiff_desc,
   viff2xbm_desc,
   xbm2viff_desc,
};

#define NMODS (sizeof(mod_list) / sizeof(char *))

AVSinit_modules()
{
        AVSinit_from_module_list(mod_list, NMODS);
}

