 /*
  * Khoros: $Id: lvsurf.c,v 1.2 1992/03/20 22:38:07 dkhoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: lvsurf.c,v 1.2 1992/03/20 22:38:07 dkhoros Exp $";
#endif

 /*
  * $Log: lvsurf.c,v $
 * Revision 1.2  1992/03/20  22:38:07  dkhoros
 * VirtualPatch5
 *
  */ 

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lvsurf.c
 >>>>
 >>>>      Program Name: vsurf
 >>>>
 >>>> Date Last Updated: Tue Apr 16 14:02:05 1991 
 >>>>
 >>>>          Routines: lvsurf - the library call for vsurf
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
static int compute_normal();
static int compute_angle();
static int find_norm_ul();
static int find_norm_ur();
static int find_norm_lr();
static int find_norm_ll();
static int crossp();
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lvsurf - library call for vsurf
*
* Purpose:
*    
*    Compute surface parameters (normal and angle)
*    
*    
* Input:
*    
*    img1           Pointer to FLOAT image used to  represent  a  sur-
*                   face.
*    
*    
* Output:
*    
*    norm           Pointer to image pointer  that  will  receive  the
*                   image  containing the normal vectors to each pixel
*                   on the surface.
*    
*    ang            Pointer to image pointer  that  will  receive  the
*                   image  containing  the  angle  between the surface
*                   normal and the reference plane normal.
*    
*    
*
* Written By: Scott Wilson
*    
*    7-Mar-91 Scott Wilson - Added transfer of pixel sizes from  input
*    to output
*    
*    
****************************************************************/


/* -library_def */
int
lvsurf(img1,img2,img3)
struct xvimage *img1, **img2, **img3;
/* -library_def_end */

/* -library_code */
  {
    /* GEOMETRY ASSUMPTIONS:

       1. X axis increases from left to right.
       2. Y axis increases from back to front.
       3. Z axis increases from bottom to top.

       Result: left handed coordinate system.
    */

    float *surface,*normal,*angle;
    struct xvimage *n,*a,*createimage();
    int i,j,k,l;
    int rows,cols;
    float pixsiz;

    /* Generate the desired surface profile */
    cols = img1->row_size;
    rows = img1->col_size;
    pixsiz = img1->pixsizx;

    surface = (float *)(img1->imagedata);

    /* Now compute surface normal */
    normal = (float *)malloc(rows*cols*3*sizeof(float));
    bzero(normal,rows*cols*3*sizeof(float));
    for (i=0; i<rows; i++)
      for (j=0; j<cols; j++)
        compute_normal(surface,j,i,rows,cols,pixsiz,normal);

    /* Now compute the angle between the entrance plane normal
       and the surface normal */
    angle = (float *)malloc(rows*cols*sizeof(float));
    bzero(angle,rows*cols*sizeof(float));
    for (i=0; i<rows; i++)
      for (j=0; j<cols; j++)
        compute_angle(surface,normal,j,i,rows,cols,angle);

    /* Fix up KHOROS headers for the images */
    n = createimage(rows,cols,VFF_TYP_FLOAT,
                    1,3,"Optic surface normals",
                    0,0,VFF_MS_NONE,VFF_MAPTYP_NONE,
                    VFF_LOC_IMPLICIT,0);
    n->imagedata = (char *)normal;
    n->pixsizx = img1->pixsizx;
    n->pixsizy = img1->pixsizy;
    *img2 = n;
    a = createimage(rows,cols,VFF_TYP_FLOAT,
                    1,1,"Optic surface normal angles",
                    0,0,VFF_MS_NONE,VFF_MAPTYP_NONE,
                    VFF_LOC_IMPLICIT,0);
    a->imagedata = (char *)angle;
    a->pixsizx = img1->pixsizx;
    a->pixsizy = img1->pixsizy;
    *img3 = a;

    return(1);
  }

static int compute_normal(surface,x,y,rows,cols,pixsiz,normal)
float *surface,*normal,pixsiz;
int x,y,rows,cols;
  {
    /* Compute the surface normal at each point in SURFACE and store
       this vector in the vector image NORMAL at the corresponding
       location. 
    */
    float norm1[3],norm2[3],norm3[3],norm4[3];

    if (x == 0 && y == 0)
      {
        /* Upper left corner */
        find_norm_ul(surface[PIXEL(x,y+1,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x+1,y,rows,cols)],pixsiz,norm1);
        normal[BPIXEL(0,x,y,rows,cols)] = norm1[0];
        normal[BPIXEL(1,x,y,rows,cols)] = norm1[1];
        normal[BPIXEL(2,x,y,rows,cols)] = norm1[2];
      }
    else if (x == cols-1 && y == 0)
      {
        /* Upper right corner */
        find_norm_ur(surface[PIXEL(x-1,y,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x,y+1,rows,cols)],pixsiz,norm1);
        normal[BPIXEL(0,x,y,rows,cols)] = norm1[0];
        normal[BPIXEL(1,x,y,rows,cols)] = norm1[1];
        normal[BPIXEL(2,x,y,rows,cols)] = norm1[2];
      }
    else if (x == cols-1 && y == rows-1)
      {
        /* Lower right corner */
        find_norm_lr(surface[PIXEL(x,y-1,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x-1,y,rows,cols)],pixsiz,norm1);
        normal[BPIXEL(0,x,y,rows,cols)] = norm1[0];
        normal[BPIXEL(1,x,y,rows,cols)] = norm1[1];
        normal[BPIXEL(2,x,y,rows,cols)] = norm1[2];
      }
    else if (x == 0 && y == rows-1)
      {
        /* Lower left corner */
        find_norm_ll(surface[PIXEL(x+1,y,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x,y-1,rows,cols)],pixsiz,norm1);
        normal[BPIXEL(0,x,y,rows,cols)] = norm1[0];
        normal[BPIXEL(1,x,y,rows,cols)] = norm1[1];
        normal[BPIXEL(2,x,y,rows,cols)] = norm1[2];
      }
    else if (y == 0)
      {
        /* Top edge */
        find_norm_ul(surface[PIXEL(x,y+1,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x+1,y,rows,cols)],pixsiz,norm1);
        find_norm_ur(surface[PIXEL(x-1,y,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x,y+1,rows,cols)],pixsiz,norm2);
        normal[BPIXEL(0,x,y,rows,cols)] = 0.5*(norm1[0]+norm2[0]);
        normal[BPIXEL(1,x,y,rows,cols)] = 0.5*(norm1[1]+norm2[1]);
        normal[BPIXEL(2,x,y,rows,cols)] = 0.5*(norm1[2]+norm2[2]);
      }
    else if (y == rows-1)
      {
        /* Bottom edge */
        find_norm_lr(surface[PIXEL(x,y-1,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x-1,y,rows,cols)],pixsiz,norm1);
        find_norm_ll(surface[PIXEL(x+1,y,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x,y-1,rows,cols)],pixsiz,norm2);
        normal[BPIXEL(0,x,y,rows,cols)] = 0.5*(norm1[0]+norm2[0]);
        normal[BPIXEL(1,x,y,rows,cols)] = 0.5*(norm1[1]+norm2[1]);
        normal[BPIXEL(2,x,y,rows,cols)] = 0.5*(norm1[2]+norm2[2]);
      }
    else if (x == 0)
      {
        /* Left edge */
        find_norm_ul(surface[PIXEL(x,y+1,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x+1,y,rows,cols)],pixsiz,norm1);
        find_norm_ll(surface[PIXEL(x+1,y,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x,y-1,rows,cols)],pixsiz,norm2);
        normal[BPIXEL(0,x,y,rows,cols)] = 0.5*(norm1[0]+norm2[0]);
        normal[BPIXEL(1,x,y,rows,cols)] = 0.5*(norm1[1]+norm2[1]);
        normal[BPIXEL(2,x,y,rows,cols)] = 0.5*(norm1[2]+norm2[2]);
      }
    else if (x == cols-1)
      {
        /* Right edge */
        find_norm_ur(surface[PIXEL(x-1,y,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x,y+1,rows,cols)],pixsiz,norm1);
        find_norm_lr(surface[PIXEL(x,y-1,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x-1,y,rows,cols)],pixsiz,norm2);
        normal[BPIXEL(0,x,y,rows,cols)] = 0.5*(norm1[0]+norm2[0]);
        normal[BPIXEL(1,x,y,rows,cols)] = 0.5*(norm1[1]+norm2[1]);
        normal[BPIXEL(2,x,y,rows,cols)] = 0.5*(norm1[2]+norm2[2]);
      }
    else
      {
        /* In the middle somewhere */
        find_norm_ul(surface[PIXEL(x,y+1,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x+1,y,rows,cols)],pixsiz,norm1);
        find_norm_ur(surface[PIXEL(x-1,y,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x,y+1,rows,cols)],pixsiz,norm2);
        find_norm_lr(surface[PIXEL(x,y-1,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x-1,y,rows,cols)],pixsiz,norm3);
        find_norm_ll(surface[PIXEL(x+1,y,rows,cols)],
                  surface[PIXEL(x,y,rows,cols)],
                  surface[PIXEL(x,y-1,rows,cols)],pixsiz,norm4);
        normal[BPIXEL(0,x,y,rows,cols)] = 0.25*(norm1[0]+norm2[0]+norm3[0]+norm4[0]);
        normal[BPIXEL(1,x,y,rows,cols)] = 0.25*(norm1[1]+norm2[1]+norm3[1]+norm4[1]);
        normal[BPIXEL(2,x,y,rows,cols)] = 0.25*(norm1[2]+norm2[2]+norm3[2]+norm4[2]);
      }
  }

static int find_norm_ul(a,b,c,size,norm)
float a,b,c,size,*norm;
  {
    /* Compute normal for upper left corner geometry */
    float v1[3],v2[3];

    /* Load the vectors */
    v1[0] = 0;      /* X component */
    v1[1] = size;   /* Y component */
    v1[2] = a-b;    /* Z component */
    v2[0] = size;   /* X component */
    v2[1] = 0;      /* Y component */
    v2[2] = c-b;    /* Z component */

    /* Obtain cross-product */
    crossp(v1,v2,norm);
  } 

static int find_norm_ur(a,b,c,size,norm)
float a,b,c,size,*norm;
  {
    /* Compute normal for upper right corner geometry */
    float v1[3],v2[3];

    /* Load the vectors */
    v1[0] = -size;   /* X component */
    v1[1] = 0;      /* Y component */
    v1[2] = a-b;    /* Z component */
    v2[0] = 0;      /* X component */
    v2[1] = size;   /* Y component */
    v2[2] = c-b;    /* Z component */

    /* Obtain cross-product */
    crossp(v1,v2,norm);
  } 

static int find_norm_lr(a,b,c,size,norm)
float a,b,c,size,*norm;
  {
    /* Compute normal for lower right corner geometry */
    float v1[3],v2[3];

    /* Load the vectors */
    v1[0] = 0;      /* X component */
    v1[1] = -size;   /* Y component */
    v1[2] = a-b;    /* Z component */
    v2[0] = -size;   /* X component */
    v2[1] = 0;      /* Y component */
    v2[2] = c-b;    /* Z component */

    /* Obtain cross-product */
    crossp(v1,v2,norm);
  }

static int find_norm_ll(a,b,c,size,norm)
float a,b,c,size,*norm;
  {
    /* Compute normal for lower left corner geometry */
    float v1[3],v2[3];

    /* Load the vectors */
    v1[0] = size;   /* X component */
    v1[1] = 0;      /* Y component */
    v1[2] = a-b;    /* Z component */
    v2[0] = 0;      /* X component */
    v2[1] = -size;   /* Y component */
    v2[2] = c-b;    /* Z component */

    /* Obtain cross-product */
    crossp(v1,v2,norm);
  }

#define MAG(x0,x1,x2) sqrt(x0*x0+x1*x1+x2*x2) 

static int crossp(v1,v2,norm)
float *v1,*v2,*norm;
  {
    float m;
    /* Compute cross product in a LEFT HANDED COORDINATE SYSTEM */
    /* Form cross product norm = v1 X v2 */
    norm[0] =  -v1[1]*v2[2]+v1[2]*v2[1];
    norm[1] =  -v1[0]*v2[2]+v2[0]*v1[2];
    norm[2] =  -v1[0]*v2[1]+v2[0]*v1[1];
    m = sqrt((double)(norm[0]*norm[0]+norm[1]*norm[1]+norm[2]*norm[2]));
    norm[0] /= m;
    norm[1] /= m;
    norm[2] /= m;
  }

static int compute_angle(surface,normal,x,y,rows,cols,angle)
float *surface,*normal,*angle;
int x,y,rows,cols;
  {
    /* Compute angle between surface normal and the surface reference plane */
    double ions[3],vec[3],dangle,dp;
    double inner();

    /* Load the ion source vector */
    ions[0] = 0.0;  /* X component */
    ions[1] = 0.0;  /* Y component */
    ions[2] = 1.0;  /* Z component */

    /* Load the surface normal vector */
    vec[0] = normal[BPIXEL(0,x,y,rows,cols)];
    vec[1] = normal[BPIXEL(1,x,y,rows,cols)];
    vec[2] = normal[BPIXEL(2,x,y,rows,cols)];

    dp = inner(ions,vec,3);
    dp = dp / (MAG(vec[0],vec[1],vec[2]) * MAG(ions[0],ions[1],ions[2]));
    dangle = acos(dp);
    angle[PIXEL(x,y,rows,cols)] = dangle*180.0/3.1415926; /* Degrees! */
  }
/* -library_code_end */
