/****************************************************************************
                          INTERNATIONAL AVS CENTRE
           (This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Centre, University of Manchester, the AVS Consortium and
the individual  submitting the module and files associated with said module
provide absolutely NO WARRANTY OF ANY KIND with respect to this software.  The
entire risk as to the quality and performance of this software is with the
user.  IN NO EVENT WILL The International AVS Centre, University of Manchester,
the AVS Consortium and the individual submitting the module and files
associated with said module BE LIABLE TO ANYONE FOR ANY DAMAGES ARISING FROM
THE USE OF THIS SOFTWARE, INCLUDING, WITHOUT LIMITATION, DAMAGES RESULTING FROM
LOST DATA OR LOST PROFITS, OR ANY SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@iavsc.org.

Please send AVS module bug reports to avs@iavsc.org.

***************************************************************************/
/*------------------------------------------------------------------------------
 *  border.c: add a border to an image
 * 
 *  AUTHOR: Chris Pudney (cpudney@alphapharm.uwa.edu.au)
 *			Biomedical Confocal Microscopy Research Centre
 *          Dept. Pharmacology, The University of Western Australia
 *			Nedlands, WA 6907, AUSTRALIA
 *
 *  VERSION: DEC ALPHA, OSF/1, AVS, cc
 *
 *  STARTED: 26/03/96
 *
 *  UPDATED: 29/03/96
 */

#include <stdio.h>
#include <malloc.h>
#include <string.h>

#include <avs/avs.h>
#include <avs/field.h>



char	bdErrStr[128];


/*------------------------------------------------------------------------------
 * Return an image (input) with a border added.  The value of border and
 * lower and upper widths must be specified.
 */

AVSfield *
border(
	AVSfield	*input,
	float	value,
	int	*lower,
	int	*upper)
{
	int	differ, *ind, ndim, type, *dims, i, go, veclen, lo, mid, hi, row, in;
	void	*inPtr, *outPtr;
	unsigned char *byte_outptr,*byte_inptr;
	short *short_outptr, *short_inptr;
	float *real_outptr, *real_inptr;
	double *doub_outptr, *doub_inptr;
	int *int_outptr, *int_inptr;
	AVSfield	*output;


	/* Initialise */
	ndim = input->ndim;
	veclen = input->veclen;
	type = input->type;
	lo = lower[0] * veclen;
	hi = upper[0] * veclen;
	mid = input->dimensions[0] * veclen;

	/* Assign new dimensions */
	dims = (int *)calloc(ndim, sizeof(int));
	ind = (int *)calloc(ndim, sizeof(int));
	if (!dims || !ind)
	{
		(void)sprintf(bdErrStr, "Could not allocate memory for border.");
		return (AVSfield *)NULL;

	} /* end if (!dims) */

	differ = 0;
	for (i = 0;
		i < ndim;
		i++)
	{
		dims[i] = input->dimensions[i] + lower[i] + upper[i];
		differ = differ || lower[i] || upper[i];

	} /* end for (i ... */
 

	/* Allocate output image */	
	output = (AVSfield *)AVSfield_alloc(input, dims);
	if (output)
	{
		/* Assign mimumin/max */
		if (differ && ((AVSfield_float *)output)->minimum &&
			((AVSfield_float *)output)->maximum)
		{
			for (i = 0;
				i < veclen;
				i++)
			{
				switch (type)
				{
				case AVS_TYPE_REAL:
					if (value < ((AVSfield_float *)output)->minimum[i])
					{
						((AVSfield_float *)output)->minimum[i] = value;
	
					} /* end if (value ... */
	
					else if (value > ((AVSfield_float *)output)->maximum[i])
					{
						((AVSfield_float *)output)->maximum[i] = value;
		
					} /* end else if (value ... */
					break;

				case AVS_TYPE_DOUBLE:
					if (value < ((AVSfield_double *)output)->minimum[i])
					{
						((AVSfield_double *)output)->minimum[i] = value;
	
					} /* end if (value ... */
	
					else if (value > ((AVSfield_double *)output)->maximum[i])
					{
						((AVSfield_double *)output)->maximum[i] = value;
		
					} /* end else if (value ... */
					break;

				case AVS_TYPE_INTEGER:
					if ((int)value < ((AVSfield_int *)output)->minimum[i])
					{
						((AVSfield_int *)output)->minimum[i] = (int)value;
	
					} /* end if ((int)value ... */
	
					else if ((int)value > ((AVSfield_int *)output)->maximum[i])
					{
						((AVSfield_int *)output)->maximum[i] = (int)value;
		
					} /* end else if ((int)value ... */
					break;

				case AVS_TYPE_SHORT:
					if ((short)value <
						((AVSfield_short *)output)->minimum[i])
					{
						((AVSfield_short *)output)->minimum[i] = (short)value;
	
					} /* end if ((short)value ... */
	
					else if ((short)value >
							((AVSfield_short *)output)->maximum[i])
					{
						((AVSfield_short *)output)->maximum[i] = (short)value;
		
					} /* end else if ((short)value ... */
					break;

				case AVS_TYPE_BYTE:
					if ((unsigned char)value <
						((AVSfield_char *)output)->minimum[i])
					{
						((AVSfield_char *)output)->minimum[i] =
							(unsigned char)value;
	
					} /* end if ((unsigned char)value ... */
	
					else if ((unsigned char)value >
							((AVSfield_char *)output)->maximum[i])
					{
						((AVSfield_char *)output)->maximum[i] =
							(unsigned char)value;
		
					} /* end else if ((unsigned char)value ... */
					break;

				} /* end switch (output ... */
			} /* for (i ... */
		} /* end if (input ... */


		/* Write points/extents */
		if (differ && output->points)
		{
			int	j, dim, loPts, hiPts, midPts, rowPts;
			float	bound, stride, *oPtr, *pPtr;


			oPtr = output->points;
			pPtr = input->points;
			switch (output->uniform)
			{
			case UNIFORM:
				for (i = 0;
					i < ndim;
					i++)
				{
					dim = input->dimensions[i];
					stride = (dim > 1) ?
						(pPtr[2*i+1] - pPtr[2*i]) / (dim - 1) : 0.;
					bound = oPtr[2*i] = pPtr[2*i] - lower[i] * stride;
					if (bound < output->min_extent[i])
					{
						output->min_extent[i] = bound;
 
					} /* end if (bound ... */
					
					else if (bound > output->max_extent[i])
					{
						output->max_extent[i] = bound;
 
					} /* end if (bound ... */

					bound = oPtr[2*i+1] = pPtr[2*i+1] + upper[i] * stride;
					if (bound < output->min_extent[i])
					{
						output->min_extent[i] = bound;
 
					} /* end if (bound ... */
					
					else if (bound > output->max_extent[i])
					{
						output->max_extent[i] = bound;
 
					} /* end if (bound ... */
				} /* end for (i ... */

				break;

			case RECTILINEAR:
				for (i = 0;
					i < ndim;
					i++)
				{
					dim = input->dimensions[i];
					if (dim)
					{
						bound = *pPtr;
						stride = (dim > 1) ? *(pPtr+1) - bound : 0.;
						for (j = lower[i];
							j;
							j--)
						{
							*oPtr++ = bound - (j * stride);
	
						} /* end for (j ... */
	
						bound = bound - (lower[i] * stride);
						if (bound < output->min_extent[i])
						{
							output->min_extent[i] = bound;
	 
						} /* end if (bound ... */
						
						else if (bound > output->max_extent[i])
						{
							output->max_extent[i] = bound;
	 
						} /* end if (bound ... */
	
						(void)memcpy((void *)oPtr, (void *)pPtr,
							sizeof(float) * dim);
						oPtr += dim;
						pPtr += dim;
	
						bound = *(pPtr-1);
						stride = (dim > 1) ? bound - *(pPtr - 2) : 0.;
						for (j = 1;
							j <= upper[i];
							j++)
						{
							*oPtr++ = bound + (j * stride);
	
						} /* end for (j ... */
	
						bound = bound + (upper[i] * stride);
						if (bound < output->min_extent[i])
						{
							output->min_extent[i] = bound;
	 
						} /* end if (bound ... */
						
						else if (bound > output->max_extent[i])
						{
							output->max_extent[i] = bound;
	 
						} /* end if (bound ... */
					} /* end if (input ... */
	
					else
					{
						oPtr += (lower[i] + upper[i]);

					} /* end else */
				} /* end for (i ... */

				break;

			case IRREGULAR:
				loPts = lower[0];
				hiPts = upper[0];
				midPts = input->dimensions[0];
				rowPts = dims[0];
				for (j = 0;
					j < output->nspace;
					j++)
				{
					for (i = 0;
						i < ndim;
						i++)
					{
						ind[i] = 0;
	
					} /* end for (i ... */	
	
					for (go = 1;
						go;)
					{
						in = 1;
						for (i = 1;
							i < ndim;
							i++)
						{
							if (ind[i] < lower[i] || ind[i] >= dims[i] - upper[i])
							{
								in = 0;
								break;
	
							} /* end if (ind ... */
						} /* end for (i ... */
	
						if (in)
						{
							(void)memset((void *)oPtr, 0, loPts * sizeof(float));
							oPtr += loPts;
							(void)memcpy((void *)oPtr, (void *)pPtr,
								midPts * sizeof(float));
							oPtr += midPts;
							pPtr += midPts;
							(void)memset((void *)oPtr, 0, hiPts * sizeof(float));
							oPtr += hiPts;
	
						} /* end if (in) */
	
						else
						{
							(void)memset((void *)oPtr, 0, rowPts * sizeof(float)); 
							oPtr += rowPts;
	
						} /* end else */
	
						go = 0;
						for (i = 1;
							i < ndim;
							i++)
						{
							ind[i]++;
							if (ind[i] >= dims[i])
							{
								ind[i] = 0;
	
							} /* end if (ind ... */
		
							else
							{
								go = 1;
								break;
	
							} /* end else */	
						} /* end for (i ... */
					} /* end for (go ... */
				} /* end for (j ... */

				/* Update extents. */
				for (i = 0;
					i < j;
					i++)
				{
					if (output->min_extent[i] > 0.)
					{
						output->min_extent[i] = 0.;

					} /* end if (output ... ) */

					else if (output->max_extent[i] < 0.)
					{
						output->max_extent[i] = 0.;

					} /* end if (output ... ) */
				} /* end for (i ... */

				break;

			} /* end switch (output ... */
		} /* end if (differ ... */


		/* Copy data from input to output */
		row = dims[0] * veclen;
		outPtr = (void *)((AVSfield_char *)output)->data;
		inPtr = (void *)((AVSfield_char *)input)->data;
		for (i = 0;
			i < ndim;
			i++)
		{
			ind[i] = 0;

		} /* end for (i ... */	

		for (go = 1;
			go;)
		{
			in = 1;
			for (i = 1;
				i < ndim;
				i++)
			{
				if (ind[i] < lower[i] || ind[i] >= dims[i] - upper[i])
				{
					in = 0;
					break;

				} /* end if (ind ... */
			} /* end for (i ... */


			switch (type)
			{
			case AVS_TYPE_BYTE:
			        byte_outptr=(unsigned char *)outPtr;
			        byte_inptr=(unsigned char *)inPtr;

				if (in)
				{
					/* Write a row: lower border, input, upper border) */
					for (i = 0;
						i < lo;
						i++)
					{
						*byte_outptr++ = (unsigned char)value;

					} /* end for (i ... */

					(void)memcpy((void *)byte_outptr, (void *)byte_inptr, mid * sizeof(unsigned char));
					byte_outptr += mid;
					byte_inptr += mid;

					for (i = 0;
						i < hi;
						i++)
					{
						*byte_outptr++ = (unsigned char)value;

					} /* end for (i ... */
				} /* end if (in) */

				else
				{
					/* Write a row's worth of border */
					for (i = 0;
						i < row;
						i++)
					{
						*byte_outptr++ = (unsigned char)value;

					} /* end for (i ... */
				} /* end else */

			        outPtr=(void *)byte_outptr;
			        inPtr=(void *)byte_inptr;

			        break;
			case AVS_TYPE_SHORT:
				short_outptr=(short *)outPtr;
				short_inptr=(short *)inPtr;

				if (in)
				{
					/* Write a row: lower border, input, upper border) */
					for (i = 0;
						i < lo;
						i++)
					{
						*short_outptr++ = (short)value;

					} /* end for (i ... */

					(void)memcpy(short_outptr, short_inptr, mid * sizeof(short));
					short_outptr += mid;
					short_inptr += mid;

					for (i = 0;
						i < hi;
						i++)
					{
						*short_outptr++ = (short)value;

					} /* end for (i ... */
				} /* end if (in) */

				else
				{
					/* Write a row's worth of border */
					for (i = 0;
						i < row;
						i++)
					{
						*short_outptr++ = (short)value;

					} /* end for (i ... */
				} /* end else */

				outPtr=(void *)short_outptr;
				inPtr=(void *)short_inptr;
				
					break;

			case AVS_TYPE_INTEGER:

				int_outptr=(int *)outPtr;
				int_inptr=(int *)inPtr;

				if (in)
				{
					/* Write a row: lower border, input, upper border) */
					for (i = 0;
						i < lo;
						i++)
					{
						*int_outptr++ = (int)value;

					} /* end for (i ... */

					(void)memcpy(int_outptr, int_inptr, mid * sizeof(int));
				        int_outptr += mid;
					int_inptr += mid;

					for (i = 0;
						i < hi;
						i++)
					{
						*int_outptr++ = (int)value;

					} /* end for (i ... */
				} /* end if (in) */

				else
				{
					/* Write a row's worth of border */
					for (i = 0;
						i < row;
						i++)
					{
						*int_outptr++ = (int)value;

					} /* end for (i ... */
				} /* end else */
			    
				outPtr=(void *)int_outptr;
				inPtr=(void *)int_inptr;

					break;

			case AVS_TYPE_REAL:

				real_outptr=(float *)outPtr;
				real_inptr=(void *)inPtr;

				if (in)
				{
					/* Write a row: lower border, input, upper border) */
					for (i = 0;
						i < lo;
						i++)
					{
						*real_outptr++ = value;

					} /* end for (i ... */

					(void)memcpy(real_outptr, real_inptr, mid * sizeof(float));
					real_outptr += mid;
					real_inptr += mid;

					for (i = 0;
						i < hi;
						i++)
					{
						*real_outptr++ = value;

					} /* end for (i ... */
				} /* end if (in) */

				else
				{
					/* Write a row's worth of border */
					for (i = 0;
						i < row;
						i++)
					{
						*real_outptr++ = value;

					} /* end for (i ... */
				} /* end else */

				outPtr=(void *)real_outptr;
				inPtr=(void *)real_inptr;

					break;

			case AVS_TYPE_DOUBLE:

				doub_outptr=(double *)outPtr;
				doub_inptr=(double *)inPtr;

				if (in)
				{
					/* Write a row: lower border, input, upper border) */
					for (i = 0;
						i < lo;
						i++)
					{
						*doub_outptr++ = value;

					} /* end for (i ... */

					(void)memcpy(doub_outptr, doub_inptr, mid * sizeof(double));
					doub_outptr += mid;
					doub_inptr += mid;

					for (i = 0;
						i < hi;
						i++)
					{
						*doub_outptr++ = value;

					} /* end for (i ... */
				} /* end if (in) */

				else
				{
					/* Write a row's worth of border */

					for (i = 0;
						i < row;
						i++)
					{
						*doub_outptr++ = value;

					} /* end for (i ... */
				} /* end else */
				
				outPtr=(void *)doub_outptr;
			        inPtr=(void *)doub_inptr;

					break;

			} /* end switch (type) */


			go = 0;
			for (i = 1;
				i < ndim;
				i++)
			{
				ind[i]++;
				if (ind[i] >= dims[i])
				{
					ind[i] = 0;

				} /* end if (ind ... */

				else
				{
					go = 1;
					break;

				} /* end else */	
			} /* end for (i ... */
		} /* end for (go */
	} /* end if (output) */

	else
	{
		(void)sprintf(bdErrStr, "Could not allocate output image.");
		
	} /* end else */
       
	dims=(void *)dims;
	ind=(void *)ind;
	free(dims);
	free(ind);
	return output;

} /* end border() */
