/****************************************************************************
                          INTERNATIONAL AVS CENTRE
           (This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Centre, University of Manchester, the AVS Consortium and
the individual  submitting the module and files associated with said module
provide absolutely NO WARRANTY OF ANY KIND with respect to this software.  The
entire risk as to the quality and performance of this software is with the
user.  IN NO EVENT WILL The International AVS Centre, University of Manchester,
the AVS Consortium and the individual submitting the module and files
associated with said module BE LIABLE TO ANYONE FOR ANY DAMAGES ARISING FROM
THE USE OF THIS SOFTWARE, INCLUDING, WITHOUT LIMITATION, DAMAGES RESULTING FROM
LOST DATA OR LOST PROFITS, OR ANY SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@iavsc.org.

Please send AVS module bug reports to avs@iavsc.org.

***************************************************************************/

/**********************************************************************

Copyright (C) 1995, 1996  E. O. Lawrence Berkeley National Laboratory.  
All Rights Reserved.  Permission to copy and modify this software and 
its documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution for 
joint exchange.   Thus it is experimental and scientific in nature, 
undergoing development, and is provided "as is" with no warranties of 
any kind whatsoever, no support, promise of updates or printed 
documentation.

This work is supported by the U. S. Department of Energy under contract 
number DE-AC03-76SF00098 between the U. S. Department of Energy and the 
University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory
                Berkeley, California 

  "this software is 100% hand-crafted by a human being in the USA"


//\/\\//\\//\\/\//\\/\\//\\//\\/\\/\//\\//\\//\\//\\//\\//\/\\//\\//\\

Mon Nov 13 12:45:49 PST 1995

this module accepts as input an LBL-AVS "VR device field" from a single
source, and acts as a gate for that field.  whenever a (user selected) button
event is detected in that field, the data in the input field will be copied
to it's output port.  when no button event is present, no data will be
copied downstream.

this module's first useful function was to serve as a gate between the
"division IPU" module and the "VR Transform Object" module.  used as such,
the result is that when the user depresses one of the buttons on the division
3dmouse, that the events would then actually cause the object to be moved
around.  when the user lets go of the button, the object would stop being
moved around.

button 4 corresponds to the "top" button (index finger) on the division
3d mouse.

when a user-selected button (or buttons) is/are detected, and the data
is passed downstream, those user-selected button values WILL BE ZEROED
OUT in the output field.  you can hack the code below if you don't like
this behavior (like you need the button info downstream).

*********************************************************************/


#include <stdio.h>
#include <string.h>
#include <avs/avs.h>
#include <avs/field.h>
#include <avs/geomdata.h>
#include <avs/udata.h>
#include "matrix.h"
#include "vr_util.h"

#define CHILL 1
#define WHACKED 0

#define BUTTON_PRESSED 0
#define BUTTON_BEING_PRESSED 1
#define BUTTON_RELEASED 2

static char DEVICE_INPUT_NAME[] = {"input_device_event_field"};
static int button_state;

static matrix4x4 accumulate_matrix;

typedef struct ui_state
{
    int have_input_device_data;
} ui_state;


AVSinit_modules()
{
    int vr_button_gate_acc();
    AVSmodule_from_desc(vr_button_gate_acc);
}

int
vr_button_gate_acc()
{
    int vr_button_gate_acc_func();
    int p;

    AVSset_module_name("button gate accumulate",MODULE_FILTER);

    AVScreate_input_port(DEVICE_INPUT_NAME,VR_FIELD_STRING,REQUIRED);
    AVScreate_output_port("outputfield",VR_FIELD_STRING);

    p=AVSadd_parameter("dummy1","string","Button Mask","","");
    AVSconnect_widget(p,"text");
    AVSadd_parameter_prop(p,"width","integer",4);

    p=AVSadd_parameter("button1","boolean",0,0,1);
    AVSconnect_widget(p,"toggle");
    p=AVSadd_parameter("button2","boolean",0,0,1);
    AVSconnect_widget(p,"toggle");
    p=AVSadd_parameter("button3","boolean",0,0,1);
    AVSconnect_widget(p,"toggle");
    p=AVSadd_parameter("button4","boolean",0,0,1);
    AVSconnect_widget(p,"toggle");
    p=AVSadd_parameter("button5","boolean",0,0,1);
    AVSconnect_widget(p,"toggle");

    AVSset_compute_proc(vr_button_gate_acc_func);
}


int
vr_button_gate_acc_func(inf,outf,d1,b1,b2,b3,b4,b5)
AVSfield *inf,**outf;
char *d1;
int b1,b2,b3,b4,b5;
{
    AVSfield template;
    int dims[1];
    int status;
    int have_button_event;
    int button1,button2,button3,button4,button5;
    int button_status_mask;
    ui_state new_state;
    float r[3],t[3];
    
    memset((char *)&template,0,sizeof(AVSfield));
    MAKE_VR_AVS_FIELD_TEMPLATE(&template);
    if (*outf)
	AVSfield_free(*outf);
    *outf = (AVSfield *)AVSfield_alloc(&template,VR_field_dims);
    memset((char *)((*outf)->field_union.field_data_float_u),0,
	   sizeof(float)*VR_field_dims[0]); /* clear all memory */


    
    do_state(&new_state);

    status = 0;
    if (new_state.have_input_device_data)
    {
	int have_button_event,top_button;
	
	TEST_BUTTONS_FLAG(have_button_event,inf);

	VR_TEST_BUTTON(VR_BUTTON1_MASK,button1,inf);
	VR_TEST_BUTTON(VR_BUTTON2_MASK,button2,inf);
	VR_TEST_BUTTON(VR_BUTTON3_MASK,button3,inf);
	VR_TEST_BUTTON(VR_BUTTON4_MASK,button4,inf);
	VR_TEST_BUTTON(VR_BUTTON5_MASK,button5,inf);

	button_status_mask = ((b1 && button1) ||
	    (b2 && button2) || (b3 && button3) ||
		(b4 && button4) || (b5 && button5));

	if ((have_button_event) && (button_status_mask))
	{
	    /* first copy over all data from input */
	    VR_COPY_ALL_DATA((*outf),inf);
	    
	    if (button_state == BUTTON_RELEASED)
	    {
		button_state = BUTTON_PRESSED;
		identity_4x4(&accumulate_matrix);
/*		fprintf(stderr,"\tbutton going down.\n"); */
	    }
	    else
	    {
		/* grab events, modify according to the accumulated
		 transformation matrix, and stuff them into the output
		 field. */
		VR_GET_ROT_VECTOR(r[0],r[1],r[2],inf);
		VR_GET_TRANS_VECTOR(t[0],t[1],t[2],inf);

		accumulate_events(&accumulate_matrix,t,r);

		VR_SET_ROT_VECTOR(r[0],r[1],r[2],(*outf));
		VR_SET_TRANS_VECTOR(t[0],t[1],t[2],(*outf));
		
/*		fprintf(stderr,"\tbutton is down.\n"); */
	    }
	    

	    /* then clear out selected button events */
    
	    if (b1)
		VR_SET_BUTTON(VR_BUTTON1_MASK,0,(*outf));
	    if (b2)
		VR_SET_BUTTON(VR_BUTTON2_MASK,0,(*outf));
	    if (b3)
		VR_SET_BUTTON(VR_BUTTON3_MASK,0,(*outf));
	    if (b4)
		VR_SET_BUTTON(VR_BUTTON4_MASK,0,(*outf));
	    if (b5)
		VR_SET_BUTTON(VR_BUTTON5_MASK,0,(*outf));

	    status = 1;
	}
	else
	{
	    button_state = BUTTON_RELEASED;
/*	    fprintf(stderr,"\tButton is up.\n"); */
	}
    }

    return(status);
}

int
do_state(this_state)
ui_state *this_state;
{
    /**
      * this routine is a throwback to a prototype which was built as a
      * coroutine.  by making the module a subroutine, fewer states in
      * the software are necessary, making for simplified code.
      * at present, all we do here is check to see if there's any new
      * data from the input device.
    **/
    this_state->have_input_device_data = AVSinput_changed(DEVICE_INPUT_NAME,0);
}

int
accumulate_events(m,t,r)
matrix4x4 *m;
float *t,*r;
{
    matrix4x4 tmp;
    vector4 v1,v2;
    

    /* take the inverse of the accumulated rotation matrix. */
    transpose_4x4(m,&tmp);

    /* run rotation values through the inverse matrix.  this gives us
     new values back in the un-rotated space.*/
    v1.v[0] = r[0];
    v1.v[1] = r[1];
    v1.v[2] = r[2];
    v1.v[3] = 0.;
    vector_matrix_mult_4(&v1,&tmp,&v2); 
    r[0] = v2.v[0];
    r[1] = v2.v[1];
    r[2] = v2.v[2];

    /* do the same with translation events, but use the forward
     transformation matrix.  */
    v1.v[0] = t[0];
    v1.v[1] = t[1];
    v1.v[2] = t[2];
    vector_matrix_mult_4(&v1,&tmp,&v2); 
    t[0] = v2.v[0];
    t[1] = v2.v[1];
    t[2] = v2.v[2];
    
    /* construct delta rotation matrix */
    build_composite_matrix(&tmp,r);

    /* and accumulate onto the end of the accumulated matrix. */
    mmul_4x4(m,&tmp);
}

#include <math.h>
#define DEGREES_TO_RADIANS(a) ((a)/180.0 * M_PI)

int
build_composite_matrix(matrix4x4 *m,float *r)
{
    /**
      * NOTES:
      *
      * what we're going to do here is build a transformaion matrix and
      * put it into "m".  this transformation matrix is formed by the absolute
      * rotation angles present in float r[3].  the way in which r[] is
      * interpreted is r[0] = azimuth, r[1] = elevation, r[2] = roll.
      * furthermore, we're making some assumptions with regard to the
      * fastrak device.  these are:
      *
      * 1. the alignment reference frame for all stations is the power on
      *    reference frame: a right handed coordinate systems with +z
      *    pointing up.
      *
      * 2. azimuth represents angular changes about the z axis of the
      *    alignment reference frame; elevation represents changes aboiut
      *    the y axis of the alignment reference frame; roll represents
      *    angular changes about the x-axis of the alignment reference
      *    frame.
      * 3. the angular values presented in r[] are taken in the order of
      *    azimuth, elevation then roll.  the transformation matrix we
      *    construct below will be computed in this order.
      *
      * the sequence of computations below is valid only if all these
      * assumptions are TRUE.
      *
      * the routine which is called subsequent to this routine,
      * "recompute_base_rotations" will recompute delta azimuth, elevation
      * and roll values making use of the same 3 fastrak assumptions as this
      * routine.
    **/
      
    matrix4x4 azimuth, elevation, roll;
    double t,c,s;
    extern double cos(),sin();
    int i;

    /* do azimuth - according to the fastrak manual, azimuth angles
     occur about the "z" axis.*/
    identity_4x4(&azimuth);
/*    t = DEGREES_TO_RADIANS(r[0]); */
    t = r[2];
    c = cos(t);
    s = sin(t);
    azimuth.m[0][0] = azimuth.m[1][1] = c;
    azimuth.m[0][1] = s;	
    azimuth.m[1][0] = -s;	

    /* do elevation */
/*    t = DEGREES_TO_RADIANS(r[1]); /* rotation about y axis */
    t = r[1];
    c = cos(t);
    s = sin(t);
    identity_4x4(&elevation);
    elevation.m[0][0] = elevation.m[2][2] = c;
    elevation.m[0][2] = -s;
    elevation.m[2][0] = s;

    /* do roll - according to the manual, roll occurs about the X axis. */
    identity_4x4(&roll);
/*    t = DEGREES_TO_RADIANS(r[2]); /* rotation about x axis */
    t = r[0];
    c = cos(t);
    s = sin(t);
    roll.m[1][1] = roll.m[2][2] = c;
    roll.m[1][2] = s;
    roll.m[2][1] = -s;

    identity_4x4(m);

    mmul_4x4(m,&azimuth);
    mmul_4x4(m,&elevation);
    mmul_4x4(m,&roll);


}

