/*******************************************************************************

 'blend.c' 
 Surface blending custom effect for SoftImage 3D
 Version 1.0
 
 Written by M.J.Evers
 (c) Computer Graphics Unit, Manchester Computing, 
 University of Manchester, 1996

 Modified: 15/8/96, 29/8/96, 4/9/96
           1/11/96    M.Preston. Converted to use SDK 1.1b2

 Parts of this custom effect were written by M. Preston and ...
 
*******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <SAA.h>

#include "convert.h"
#include "blending.h"


/****************************************/
/*           LOCAL CONSTANTS            */
/****************************************/

/*
** I've created a number of defines for the dialog item numbers and
** for the indexes in the argument array of the elements the user 
** has picked. This makes changing the pick order a lot easier
**
** In the case we have a variable nr of arguments, NB_ARGS
** specifies the maximum nr of arguments.
*/

#define NB_ARGS           12

#define ARG_SRF1          0
#define ARG_SRF2          (2 + (idata->nbIntmCurves) )
/* ^ the above define is perhaps a bit tricky I think */

#define DLG_OK            1
#define DLG_CANCEL        2
#define DLG_APPLY         11
#define DLG_SLIDER1       7
#define DLG_SLIDER2       10
#define DLG_TUNING1       6
#define DLG_TUNING2       9
#define DLG_EDIT1         5
#define DLG_EDIT2         8

#define DLG_TXTVW1        12
#define DLG_TXTVW2        14
#define DLG_TXTVW3        16
#define DLG_TXTVW4        18

#define DLG_INCRV1ID      20
#define DLG_OUTCRV1ID     21
#define DLG_INCRV2ID      22
#define DLG_OUTCRV2ID     23
#define DLG_NBARGS        38

#define DLG_P_INCRV       30
#define DLG_P_OUTCRV      31
#define DLG_CHK_U_DIR     32
#define DLG_CHK_CONST     29


/****************************************/
/*             LOCAL MACROS             */
/****************************************/

/*
** Some functions have arguments that are never used.  To prevent C++ compilers
** from issuing warnings about those arguments, the following macro can be
** used.
*/
#ifdef __cplusplus
#  define _UNUSED_ARG( x )
#else
#  define _UNUSED_ARG( x ) x
#endif

/****************************************/
/*             LOCAL TYPES              */
/****************************************/

/*
** The ProjectionCurvesInfo data structure contains 
** the number of proejection curves on a specific surface and 
** arrays of containing these curves and extracted version of them.
*/
typedef struct
{
  int nbCrv;                      /* number of curves */
  SAA_SubElem * projCurves;       /* projection curves of surface */
  SAA_Elem * extractedProjCurves; /* extracted projection curves of surface */
} ProjectionCurvesInfo; 

/*
** The following struct contains instance-specific information.
*/
typedef struct
{
  /*
  ** Saaphire functions return an SI_Error code.  The following field stores
  ** the last code that was returned so that we remember when things have gone
  ** wrong.
  */
  SI_Error result;

  /*
  ** Many Saaphire functions operate on a scene.  The following field stores
  ** the current scene so that we don't have to keep getting it.
  */
  SAA_Scene scene;

  /* 
  ** nbArgs is the number of arguments the user has picked;
  ** nbIntmCurves is the number of intermediate curves 
  */
  int nbArgs, nbIntmCurves;

  /*
  ** For each selected surface we have a structure containing
  ** info about the projection curves on it.
  ** surfaces[] is only used during the setup dialog phase, that is in
  ** the functions initInstanceData() and closeDialogCallback()
  */
  ProjectionCurvesInfo surfaces[2];
  
  /*
  ** preview is the previewing blend surface. This field is also
  ** only used during the setup dialog phase
  */
  SAA_Elem preview;  /***/

  /*
  ** surface<x>Curves contains the 2 selected projection curves
  ** for each surface
  ** element 0 is the inner curve, element 1 the outer curve
  */
  SAA_SubElem surface1Curves[2];
  SAA_SubElem surface2Curves[2];

  int in1, out1, in2, out2;

  /*
  ** r1 and r2 used to be the curvature parameters for the blend surface 
  ** I'm now using them as the twist. They will ultimately be passed to the
  ** SAA_nurbsCurveShiftParameterization()
  */
  int r1, r2;

  /*
  ** The following parameters contain the data regarding 
  ** the use of constant u/v value curves instead of 
  ** projection curves
  */
  SAA_Boolean constantParam, constU;
  float innerValue, outerValue;

  /*
  ** The updateFlag is used to prevent this effect
  ** from updating the blend surface too often:
  ** SI calls update for each result definition line
  ** in the .cus file. In this CE there are 2 result
  ** definitions. We toggle the flag in update() so
  ** that of each two calls of update() only one real
  ** update takes place.
  ** We do not use the SAA_customContextGetCurrentResult
  ** because it does not return the right indexes.
  */
  SAA_Boolean updateFlag;

} InstanceData;

/****************************************/
/*      LOCAL FUNCTION PROTOTYPES       */
/****************************************/

SI_Error initInstance( InstanceData ** idata );

SI_Error initInstanceData( SAA_CustomContext context );

SI_Error getProjectionCurves( SAA_Scene * scene, SAA_Elem * srf, 
			      ProjectionCurvesInfo * info );

void twistCurves(InstanceData *idata, SAA_Elem *curves, int num_curves);
void getCurves( InstanceData * idata, SAA_Elem * curves, 
		SAA_Elem * srf1, SAA_Elem * srf2 );

void do_blending2( SAA_Elem * curves, int nbCurves, 
		   PR_nurb * out, SAA_Scene * scene,
		   int r1, int r2 );

/****************************************/
/*           LOCAL VARIABLES            */
/****************************************/


/****************************************/
/*           GLOBAL FUNCTIONS           */
/****************************************/

/*
** The names of global (public) functions that are given to SOFTIMAGE|3D in
** .cus files must not be mangled by C++ compilers.  If this file is being
** compiled by a C++ compiler, prevent the compiler from mangling global
** function names.
*/
#ifdef __cplusplus
extern "C" {
#endif


SI_Error previewCallback (
  const SAA_CustomContext context,
  int dialogItemId,
  void * userData
 );

SI_Error toggleCallback (
  const SAA_CustomContext context,
  int dialogItemId,
  void * userData
 );

SI_Error cancelCallback (
  const SAA_CustomContext context,
  int dialogItemId,
  void * userData 
 );

SI_Error applyCallback (
  const SAA_CustomContext context,
  int dialogItemId,
  void *userData
 );

SI_Error selectCurveCallback (
  const SAA_CustomContext context,
  int dialogItemId,
  void * userData  
 );

SI_Error closeDialogCallback (
  const SAA_CustomContext context,
  int dialogItemId,
  void * userData
 );



/*******************************************************************************

   $$L init

   Saaphire effect initialization function. This function initialises the
   instance data as far as setup() has not done it and creates the blend 
   surface.

   Returned Value : SI_Error.

   NOTE: The init() function definition is SAAPHIRE version 1.0 style 
         instead of v1.1 style (that means there is no 'context' variable).
	 The reason for this is that setting the result models does not
	 work with the v1.1 context functions.
	 Lines of code that would be used if init() was v1.1 style have been
	 commented out.

*******************************************************************************/

_CUS_EXTERN SI_Error init
(
/*   SAA_CustomValueList valueList,
   int nbargs,
   SAA_Elem *args,
   int  nbres,
   SAA_Elem *resx,
   void **instdata
*/
  SAA_CustomContext context
)

{
   InstanceData	*idata;		/* This effect's instance data.		*/
   SI_Error	result, res;	/* Return value.			*/

   /* v1.1: */
   SAA_CustomValueList valueList;
   SAA_Elem *args;

   SAA_Elem *blendedSurface;
   PR_nurb prBlend;
   
   int i;

   SAA_Elem * extractedCurves, tmp1, tmp2;

   int nbCrv, inCrv, outCrv;
   SAA_SubElem * projCurves;
   SAA_Boolean constCrv;
   SAA_ModelType mdlType;
   SAA_Elem *resx;
   int num_res, num_args;

   /*
   ** Most initialization is done in the Setup() function
   ** The only thing we have to do here is to get the instance
   ** data and check if it is not NULL.
   ** We also have to check if the effect is created or if the scene is loaded
   ** from a database. In the latter case, the setup() function has not been
   ** called, so we need to create the instance data here.
   ** (it's getting a bit messy...)
   */

   /* First create space for the result */
   blendedSurface = (SAA_Elem *) malloc(sizeof(SAA_Elem));

   SAA_customContextGetCustomValueList( context, &valueList );

   SAA_customContextGetNbArguments( context, &num_args );

  args = (SAA_Elem *) malloc( num_args * sizeof(SAA_Elem));
   SAA_customContextGetArguments(context, num_args, args);

   SAA_customContextGetNbResults(context, &num_res);

   resx = (SAA_Elem *) malloc( num_res * sizeof(SAA_Elem));
/*    SAA_customContextGetResults(context, num_res, resx); */


   nrb_clear(&prBlend);

   if ( /*resx[0].elemid != -1*/ num_res !=0) {

     initInstance( &idata );

     /* Pass the instance data pointer to SAAPHIRE */
     result = SAA_customContextSetInstanceData(context, (void *) idata);

     /* *instdata = idata; */

     /* Get the number of arguments */
     idata->nbArgs = num_args;

     /* 
     ** Get the nr of arguments from the dialog box: SoftImage
     ** does not save the nr of arguments correctly 
     */
     /*idata->nbArgs = nbargs; */
/*     SAA_cusvalGetIntValue( valueList, "NBARGS", &idata->nbArgs ); */

     idata->nbIntmCurves = (idata->nbArgs / 2) - 2;

     /* 
     ** Retrieve the projection curves using the id fields in 
     ** the dialog.
     */
     SAA_cusvalGetIntValue( valueList, "INCRV1", &idata->in1 );
     SAA_cusvalGetIntValue( valueList, "OUTCRV1", &idata->out1 );

     inCrv = idata->in1;
     outCrv = idata->out1;

     /*
     ** Get an array with all the trim curves of surface 1
     */
     SAA_nurbsSurfaceGetNbTrimCurves( &idata->scene, &args[ ARG_SRF1 ], 
				      SAA_TRIMTYPE_PROJECTION, &nbCrv );


     /* Perhaps we should check if all the curves still exist on the 
	surfaces.
     */

     /*
     ** Allocate memory for the array of trim curves and 
     ** retrieve the projection curves
     */
#ifdef __cplusplus
     projCurves = new SAA_SubElem [ nbCrv ];
#else
     projCurves = (SAA_SubElem *) calloc( nbCrv, sizeof(SAA_SubElem) );
#endif
     assert ( projCurves != NULL );

     /*
     ** Search for the curves with the ids equal to the 
     ** stored ids and store these curves in the instance data
     */
     SAA_nurbsSurfaceGetTrimCurves(&idata->scene, &args[ ARG_SRF1 ], 
				   SAA_GEOM_DEFORMED, 0,
				   SAA_TRIMTYPE_PROJECTION, nbCrv,
				   projCurves );
     for( i=0; i < nbCrv; i++ ) {
       if ( inCrv == projCurves[i].id[0] ) {
	 idata->surface1Curves[0] = projCurves[i];
       }
       if ( outCrv == projCurves[i].id[0] ) {
	 idata->surface1Curves[1] = projCurves[i];
       }
     }

#ifdef __cplusplus
     delete [] projCurves;
#else
     free( projCurves );
#endif

     /*
     ** Do the same for surface 2
     */
     SAA_cusvalGetIntValue( valueList, "INCRV2", &idata->in2 );
     SAA_cusvalGetIntValue( valueList, "OUTCRV2", &idata->out2 );
     inCrv = idata->in2;
     outCrv = idata->out2;

     SAA_nurbsSurfaceGetNbTrimCurves( &idata->scene, &args[ ARG_SRF2 ], 
				      SAA_TRIMTYPE_PROJECTION, &nbCrv );
     
     /* 
     ** Check if we use projection curves or constant u / v parameter curves 
     */
     SAA_cusvalGetStateValue( valueList, "SRF2SWITCH", &constCrv );
     if ( !constCrv ) {
       /*
       ** Allocate memory for the array of trim curves and 
       ** retrieve the projection curves
       */

#ifdef __cplusplus
       projCurves = new SAA_SubElem [ nbCrv ];
#else
       projCurves = (SAA_SubElem *) calloc( nbCrv, sizeof(SAA_SubElem) );
#endif
       assert ( projCurves != NULL );

       /*
       ** Search for the curves with the ids equal to the 
       ** stored ids and store these curves in the instance data
       */
       SAA_nurbsSurfaceGetTrimCurves( &idata->scene, &args[ ARG_SRF2 ], 
				   SAA_GEOM_DEFORMED, 0,
				      SAA_TRIMTYPE_PROJECTION, nbCrv,
				      projCurves );
       for( i=0; i < nbCrv; i++ ) {
	 if ( inCrv == projCurves[i].id[0] ) {
	   idata->surface2Curves[0] = projCurves[i];
	 }
	 if ( outCrv == projCurves[i].id[0] ) {
	   idata->surface2Curves[1] = projCurves[i];
	 }
       }

#ifdef __cplusplus
       delete [] projCurves;
#else
       free( projCurves );
#endif
     }
   } else {

     result = SAA_customContextGetInstanceData(context, (void **) &idata);
     /* idata = (InstanceData *) *instdata; */

     if( idata != NULL ) {
       result = idata->result;
     } else {
       result = SI_ERR_CUSTOM_FATAL;
     }
   } 
   /* 
   ** Get the data from the dialog
   */
   SAA_cusvalGetIntValue( valueList, "SRF1PARAM", &idata->r1 );
   SAA_cusvalGetIntValue( valueList, "SRF2PARAM", &idata->r2 );
   SAA_cusvalGetFloatValue( valueList, "INCRVP", &idata->innerValue );
   SAA_cusvalGetFloatValue( valueList, "OUTCRVP", &idata->outerValue );
   SAA_cusvalGetStateValue( valueList, "CNSTDIR", &idata->constU );
   SAA_cusvalGetStateValue( valueList, "SRF2SWITCH", &idata->constantParam );

   /* 
   ** Initialise the NURBS procedure library
   */
   nrb_initialise();

   /* 
   ** We check the elemid field of the element in the result array.
   ** If it's -1, the result object needs to be created .
   ** Otherwise, the user has loaded a scene containing this
   ** custom effect and we should not create the object then.
   */
   SAA_customContextGetNbResults(context, &num_res);

   if ( !num_res) {

#ifdef __cplusplus
     extractedCurves = new SAA_Elem [ idata->nbIntmCurves + 4 ];
#else
     extractedCurves = (SAA_Elem *) calloc( idata->nbIntmCurves + 4, sizeof( SAA_Elem ) );
#endif

     /* Get the curves of both surfaces */
     getCurves( idata, extractedCurves, &args[ ARG_SRF1 ], 
		&args[ ARG_SRF2 ] );


     /* 
     ** Move the 2nd and 3rd curve to the end of the array
     ** and copy the intermediate curves in between.
     ** I use a temporary variable instead of direct assignment to 
     ** prevent overwriting the curves in the case there is just
     ** one intermediate curve.
     **
     */
     extractedCurves[ idata->nbIntmCurves + 3 ] = extractedCurves[ 3 ]; 
     extractedCurves[ idata->nbIntmCurves + 2 ] = extractedCurves[ 2 ];

     for( i=0; i < idata->nbIntmCurves; i++ ) {
       extractedCurves[ i + 2 ] = args[ i + 2 ];
     }
     
     twistCurves(idata, extractedCurves, idata->nbIntmCurves+4);

     do_blending2( extractedCurves, idata->nbIntmCurves + 4,
		   &prBlend, &idata->scene, idata->r1, idata->r2 );

     convertNURBSToSI( &prBlend, blendedSurface, &idata->scene );

     /*
     ** We don't close the surface here - it would take a lot of 
     ** code. The surface will be closed by the update() function, 
     ** which is always called directly after init() (as far as I
     ** can see).
     */

     /* 
     ** Destroy the extracted curves, but leave the intermediate 
     ** curves intact.
     */
     SAA_elementDestroy( &idata->scene, &extractedCurves[ 0 ] );
     SAA_elementDestroy( &idata->scene, &extractedCurves[ 1 ] );
     SAA_elementDestroy( &idata->scene, &extractedCurves[ idata->nbIntmCurves + 2 ] );
     SAA_elementDestroy( &idata->scene, &extractedCurves[ idata->nbIntmCurves + 3 ] );
#ifdef __cplusplus
     delete [] extractedCurves;
#else
     free( extractedCurves );
#endif

     SAA_elementSetName( &idata->scene, blendedSurface, "blend1");
   
     /*
     ** As init() is responsible for the result elements,
     ** we put the newly created spline in the context. 
     */
     /* resx[0] = blendedSurface;
     resx[1] = blendedSurface; */

     SAA_customContextSetResult(context, 0, blendedSurface); 
     /* SAA_customContextSetResult(context, 1, blendedSurface); */
   }

   /* Give the updateFlag an initial value */
   idata->updateFlag = TRUE;
    
   /*
   ** That's it.
   */
   if( idata != NULL )
   {
      idata->result = result;
   }

   return( result );
}

/*******************************************************************************

   $$L update

   Saaphire effect update function, executes the effect.

   Returned Value : SI_Error.

*******************************************************************************/

_CUS_EXTERN SI_Error update
(
 SAA_CustomContext context
)
{
   InstanceData	*idata;		/* This effect's instance data.		*/
   SI_Error	result;		/* Return value.			*/

   SAA_Elem blendedSurface[2];
   SAA_Elem * extractedCurves; 
   SAA_Elem tmp1, tmp2;

   PR_nurb prBlend;
   SAA_Elem args [ NB_ARGS ];

   int i, stepU, stepV;

   /*
   ** Check if init() couldn't allocate instance data.
   */
   result = SAA_customContextGetInstanceData(context, (void **) &idata);

   if( idata != NULL && result == SI_SUCCESS)
   {
     result = idata->result;
   } else {
     result = SI_ERR_CUSTOM_FATAL;
   }

   /*
   ** Toggle the update flag and perform the actual update only
   ** when the flag is true
   */
   idata->updateFlag = !idata->updateFlag;
   if ( idata->updateFlag ) {

     SAA_customContextGetResults( context, 2, blendedSurface );

     /* 
     ** Re-create the blending surface 
     */
     SAA_customContextGetArguments( context, idata->nbArgs, args );

#ifdef __cplusplus
     extractedCurves = new SAA_Elem [ idata->nbIntmCurves + 4 ];
#else
     extractedCurves = (SAA_Elem *) calloc( idata->nbIntmCurves + 4, sizeof( SAA_Elem ) );
#endif

     /* Get the curves for both surfaces */
     getCurves( idata, extractedCurves, &args[ ARG_SRF1 ],
		&args[ ARG_SRF2 ] );

     /* 
     ** Move the 2nd and 3rd curve to the end of the array
     ** and copy the intermediate curves in between.
     ** I use a temporary variable instead of direct assignment to 
     ** prevent overwriting the curves in the case there is just
     ** one intermediate curve.
     */


     /* 
     ** We have to swap inner and outer curves: 
     ** getCurves returns inner; outer, while do_blending2 expects 
     ** outer;inner, for the 1st surface 
     */
     tmp1 = extractedCurves[ 0 ];
     extractedCurves[ 0 ] = extractedCurves[ 1 ];
     extractedCurves[ 1 ] = tmp1;
     extractedCurves[ idata->nbIntmCurves + 3 ] = extractedCurves[ 3 ];
     extractedCurves[ idata->nbIntmCurves + 2 ] = extractedCurves[ 2 ];
     
     for( i=0; i < idata->nbIntmCurves; i++ ) {
       extractedCurves[ i + 2 ] = args[ i + 2 ];
     }
     
     twistCurves(idata, extractedCurves, idata->nbIntmCurves + 4);
     do_blending2( extractedCurves, idata->nbIntmCurves + 4,
		   &prBlend, &idata->scene, (double) idata->r1, 
		   (double) idata->r2 );

     /*
     ** Store the step values of the surface. They are reset
     ** when we call SAA_nurbsSurfaceEditSurface()
     */
     SAA_nurbsSurfaceGetStep( &idata->scene, &blendedSurface[0], &stepU, &stepV );

     /* 
     ** Copy the vertices from the newly created PRLIB NURBS to the 
     ** existing SI NURBS. copyPoints will take care of closing
     ** the surface. <- We do not close the surface any more.
     */
     copyPoints( &prBlend, &blendedSurface[0], &idata->scene, FALSE/*was TRUE*/, FALSE );

     /*
     ** Restore step values
     */
     SAA_nurbsSurfaceSetStep( &idata->scene, &blendedSurface[0], stepU, stepV );

     /* 
     ** Destroy the extracted curves, but leave the intermediate 
     ** curves intact.
     */
     SAA_elementDestroy( &idata->scene, &extractedCurves[ 0 ] );
     SAA_elementDestroy( &idata->scene, &extractedCurves[ 1 ] );
     SAA_elementDestroy( &idata->scene, &extractedCurves[ idata->nbIntmCurves + 2 ] );
     SAA_elementDestroy( &idata->scene, &extractedCurves[ idata->nbIntmCurves + 3 ] );

#ifdef __cplusplus
     delete [] extractedCurves;
#else
     free( extractedCurves );
#endif

     nrb_deallocatepts( &prBlend.pf_ppp );
     nrb_deallocateknots( &prBlend.pf_u );
     nrb_deallocateknots( &prBlend.pf_v );

   } 

   /*
   ** That's it.
   */
   if( idata != NULL )
   {
      idata->result = result;
   }
   return( result );
}

/*******************************************************************************

   $$L cleanup

   Saaphire effect cleanup function, frees instance data.

   Returned Value : SI_Error.

*******************************************************************************/

_CUS_EXTERN SI_Error cleanup
(
 SAA_CustomContext context
)
{
   InstanceData	*idata;		/* This effect's instance data.		*/
   SI_Error	result;		/* Return value.			*/

   /*
   ** Check if init() couldn't allocate instance data.
   */
   result = SAA_customContextGetInstanceData(context, (void **) &idata);

   if( idata != NULL && result == SI_SUCCESS ) {
      result = idata->result;
   } else {
      result = SI_ERR_CUSTOM_FATAL;
   }

   /*
   ** Free instance data.
   */
   if( idata != NULL )
   {
#ifdef __cplusplus
     delete idata;
#else
     free( idata );
#endif
   }

   /*
   ** That's it.
   */
   return( result );
}


/*
** setup() function sets up the 'setup' dialog
** That is: it en/disables items and installs callback functions
*/
_CUS_EXTERN SI_Error setup
(
 SAA_CustomContext context
)
{
   InstanceData	*idata;		/* This effect's instance data.		*/
   SI_Error	result;		/* Return value.			*/
   char s [81];
   int namesize, i;
   SAA_Boolean mode;

   /* 
   ** Do the initialisation of instance data and the extraction of 
   ** the projection curves, as setup() is the first function of this
   ** custom effect to be called
   */
   result = initInstanceData( context );

   if ( result == SI_SUCCESS ) 
     {
       result = SAA_customContextGetInstanceData(context, (void **) &idata);

       /*
       ** init the id field of the preview element to -1
       */
       idata->preview.elemid = -1; /***/

       /* 
       ** We disable the 'apply' button, as it has no function at the moment,
       ** and we enable the text views. We also enable the items for the 
       ** constant parameter curve selection.
       */
       SAA_dialogitemSetSensitivity( context, DLG_APPLY, FALSE ); /***/
       SAA_dialogitemSetSensitivity( context, DLG_TXTVW1, FALSE );
       SAA_dialogitemSetSensitivity( context, DLG_TXTVW2, FALSE );
       SAA_dialogitemSetSensitivity( context, DLG_CHK_CONST, FALSE );
       
       SAA_dialogitemGetStateValue( context, DLG_CHK_CONST, &mode );
       if (mode) { 
	 SAA_dialogitemSetSensitivity( context, DLG_P_INCRV, FALSE );
	 SAA_dialogitemSetSensitivity( context, DLG_P_OUTCRV, FALSE );
	 SAA_dialogitemSetSensitivity( context, DLG_CHK_U_DIR, FALSE );

	 SAA_dialogitemSetSensitivity( context, DLG_TXTVW3, TRUE );
	 SAA_dialogitemSetSensitivity( context, DLG_TXTVW4, TRUE );
       } else {
	 SAA_dialogitemSetSensitivity( context, DLG_P_INCRV, TRUE );
	 SAA_dialogitemSetSensitivity( context, DLG_P_OUTCRV, TRUE );
	 SAA_dialogitemSetSensitivity( context, DLG_CHK_U_DIR, TRUE );
	 
	 SAA_dialogitemSetSensitivity( context, DLG_TXTVW3, FALSE );
	 SAA_dialogitemSetSensitivity( context, DLG_TXTVW4, FALSE );
       }

       /* 
       ** As we do not know how to hide the id fields, we just
       ** disable them
       */
       SAA_dialogitemSetSensitivity( context, DLG_INCRV1ID, TRUE );
       SAA_dialogitemSetSensitivity( context, DLG_OUTCRV1ID, TRUE );
       SAA_dialogitemSetSensitivity( context, DLG_INCRV2ID, TRUE );
       SAA_dialogitemSetSensitivity( context, DLG_OUTCRV2ID, TRUE );
       SAA_dialogitemSetSensitivity( context, DLG_NBARGS, TRUE );

       /*
       ** Put the names (or indexes or whatever) of the projection curves
       ** in the text views
       */
       SAA_textlistClear( context, DLG_TXTVW1 );
       SAA_textlistClear( context, DLG_TXTVW2 );
       SAA_textlistClear( context, DLG_TXTVW3 );
       SAA_textlistClear( context, DLG_TXTVW4 );

       SAA_textlistSetSelectionMode( context, DLG_TXTVW1, SAA_SELECT_SINGLE );
       SAA_textlistSetSelectionMode( context, DLG_TXTVW2, SAA_SELECT_SINGLE );
       SAA_textlistSetSelectionMode( context, DLG_TXTVW3, SAA_SELECT_SINGLE );
       SAA_textlistSetSelectionMode( context, DLG_TXTVW4, SAA_SELECT_SINGLE );

       for( i=0; i<idata->surfaces[0].nbCrv; i++ ) {
	 /*
	 ** We put the names of the curves in both text views 
	 */
	 SAA_elementGetNameLength( &idata->scene, &idata->surfaces[0].extractedProjCurves[i], 
				   &namesize );
	 SAA_elementGetName( &idata->scene, &idata->surfaces[0].extractedProjCurves[i],
			     namesize+1, s );
	 
	 SAA_textlistCreateItem( context, DLG_TXTVW1, i, s );
	 SAA_textlistCreateItem( context, DLG_TXTVW2, i, s );
       }

       for( i=0; i<idata->surfaces[1].nbCrv; i++ ) {
	 /*
	 ** We put the names of the curves in both text views 
	 */
	 SAA_elementGetNameLength( &idata->scene, &idata->surfaces[1].extractedProjCurves[i],
				   &namesize );
	 SAA_elementGetName( &idata->scene, &idata->surfaces[1].extractedProjCurves[i],
			     namesize+1, s );
	 
	 SAA_textlistCreateItem( context, DLG_TXTVW3, i, s );
	 SAA_textlistCreateItem( context, DLG_TXTVW4, i, s );
       }

       /*
       ** Install a callback which highlights the curves that the user selects
       ** in the text views.
       */
       SAA_dialogitemAddCallback( context, DLG_TXTVW1, &selectCurveCallback, NULL );
       SAA_dialogitemAddCallback( context, DLG_TXTVW2, &selectCurveCallback, NULL );
       SAA_dialogitemAddCallback( context, DLG_TXTVW3, &selectCurveCallback, NULL );
       SAA_dialogitemAddCallback( context, DLG_TXTVW4, &selectCurveCallback, NULL );

       SAA_selectlistSetMode( &idata->scene, (SAA_SelectionMode) SAA_SELECT_SINGLE );

       /*
       ** Install a callback which deletes the curves in case 'cancel' 
       ** is pressed and that deletes the curves that were not chosen 
       ** in case 'ok' is pressed.
       */
       SAA_dialogitemAddCallback( context, DLG_OK, & closeDialogCallback, NULL );
       SAA_dialogitemAddCallback( context, DLG_CANCEL, & closeDialogCallback, NULL );
       SAA_dialogitemAddCallback( context, DLG_APPLY, & previewCallback, NULL ); /***/
   
       /*
       ** Install a callback to switch between projection curve mode and 
       ** constant parameter curve mode
       */
       SAA_dialogitemAddCallback( context, DLG_CHK_CONST, &toggleCallback, NULL );
     }

   return result;
}

_CUS_EXTERN SI_Error edit
(
 SAA_CustomContext context
)
{
   InstanceData	*idata;		/* This effect's instance data.		*/
   SI_Error	result;		/* Return value.			*/

   result = SAA_customContextGetInstanceData(context, (void **) &idata);
   /* *** check also return value of ..GetInstData */

   /* 
   ** We enable the apply button and disable the text views 
   ** (these are only used when creating the surface
   ** We also disable the items for constant parameter 
   ** curve selection.
   */
   SAA_dialogitemSetSensitivity( context, DLG_APPLY, FALSE );

   SAA_dialogitemSetSensitivity( context, DLG_TXTVW1, TRUE );
   SAA_dialogitemSetSensitivity( context, DLG_TXTVW2, TRUE );
   SAA_dialogitemSetSensitivity( context, DLG_TXTVW3, TRUE );
   SAA_dialogitemSetSensitivity( context, DLG_TXTVW4, TRUE );

   /*
   ** If the constant parameter curve option was not selected,
   ** we disable the corresponding items on the dialog
   */
   if( !idata->constantParam ) {
     SAA_dialogitemSetSensitivity( context, DLG_P_INCRV, TRUE );
     SAA_dialogitemSetSensitivity( context, DLG_P_OUTCRV, TRUE );
     SAA_dialogitemSetSensitivity( context, DLG_CHK_U_DIR, TRUE );
   } else {
     SAA_dialogitemSetSensitivity( context, DLG_P_INCRV, FALSE );
     SAA_dialogitemSetSensitivity( context, DLG_P_OUTCRV, FALSE );
     SAA_dialogitemSetSensitivity( context, DLG_CHK_U_DIR, FALSE );
   }

   SAA_dialogitemSetSensitivity( context, DLG_CHK_CONST, TRUE );

   /* 
   ** As we do not know how to hide the id fields, we just
   ** disable them
   */
   SAA_dialogitemSetSensitivity( context, DLG_INCRV1ID, TRUE );
   SAA_dialogitemSetSensitivity( context, DLG_OUTCRV1ID, TRUE );
   SAA_dialogitemSetSensitivity( context, DLG_INCRV2ID, TRUE );
   SAA_dialogitemSetSensitivity( context, DLG_OUTCRV2ID, TRUE );
   SAA_dialogitemSetSensitivity( context, DLG_NBARGS, TRUE );

   SAA_dialogitemAddCallback( context, DLG_CANCEL, &cancelCallback, NULL );
   SAA_dialogitemAddCallback( context, DLG_APPLY, &applyCallback, NULL );
   SAA_dialogitemAddCallback( context, DLG_OK, &applyCallback, NULL );
   SAA_dialogitemAddCallback( context, DLG_SLIDER1, &applyCallback, NULL );
   SAA_dialogitemAddCallback( context, DLG_SLIDER2, &applyCallback, NULL );
   SAA_dialogitemAddCallback( context, DLG_TUNING1, &applyCallback, NULL );
   SAA_dialogitemAddCallback( context, DLG_TUNING2, &applyCallback, NULL );

   return SI_SUCCESS;
}

/**********************************************/
/* SETUP DIALOG CALLBACK FUNCTIONS            */
/**********************************************/
/*
** The previewCallback is attached to the apply button in the setup
** phase. It creates a blend surface based on the curves chosen by the 
** user.
*/
SI_Error previewCallback (
  const SAA_CustomContext context,  /* current context */
  int _UNUSED_ARG( dialogItemId ),                 /* id of the dialog item */
  void * _UNUSED_ARG( userData )                  /* User  data */
 )
{
   InstanceData	* idata;
   SAA_Elem args [ NB_ARGS ];
   SAA_Elem * extractedCurves;
   PR_nurb temp, temp2;
   int nbIntmCurves, i, nbSel, result;
   SAA_Boolean constUV, mode, is_valid;
   float innerValue, outerValue;
   int r1, r2;
   int num_curves;

   result = SAA_customContextGetInstanceData(context, (void **) &idata);
   SAA_customContextGetArguments( context, idata->nbArgs, args );

   nbIntmCurves = ( idata->nbArgs / 2 ) - 2;

#ifdef __cplusplus
   extractedCurves = new SAA_Elem [ nbIntmCurves + 4 ];
#else
   extractedCurves = (SAA_Elem *) calloc( nbIntmCurves + 4, sizeof( SAA_Elem ));
#endif

   /*
   ** Get indexes of selected curves
   ** Use indexes in surfaces[].projCurves[] arrays 
   ** of the instance data and store these elements
   ** in the array ...
   */

   /* inner curve of surface 1 */
   SAA_textlistGetNbSelected( context, DLG_TXTVW1, &nbSel );
   if ( nbSel == 1 ) {
     SAA_textlistGetSelected( context, DLG_TXTVW1, 1, &i );
     idata->in1 = i;
     extractedCurves[1] = idata->surfaces[0].extractedProjCurves[i];
   } else {
     result = -1;
   }
   /* outer curve of surface 1 */
   SAA_textlistGetNbSelected( context, DLG_TXTVW2, &nbSel );
   if ( nbSel == 1 ) {
     SAA_textlistGetSelected( context, DLG_TXTVW2, 1, &i );
     idata->out1 = i;
     extractedCurves[0] = idata->surfaces[0].extractedProjCurves[i];
   } else {
     result = -1;
   }

   SAA_dialogitemGetStateValue( context, DLG_CHK_CONST, &constUV );

   if ( !constUV ) {
     /* use the specified projection curves */
     /* inner curve of surface 2 */
     SAA_textlistGetNbSelected( context, DLG_TXTVW3, &nbSel );
     if ( nbSel == 1 ) {
       SAA_textlistGetSelected( context, DLG_TXTVW3, 1, &i );
       idata->in2 = i;
       extractedCurves[ nbIntmCurves + 2 ] = idata->surfaces[1].extractedProjCurves[i];
     } else {
       result = -1;
     }
     /* outer curve of surface 2 */
     SAA_textlistGetNbSelected( context, DLG_TXTVW4, &nbSel );
     if ( nbSel == 1 ) {
       SAA_textlistGetSelected( context, DLG_TXTVW4, 1, &i );
       idata->out2 = i;
       extractedCurves[ nbIntmCurves + 3 ] = idata->surfaces[1].extractedProjCurves[i];
     } else {
       result = -1;
     }
   } else {
     /* 
     ** use the u or v curves. Note that this piece of code has been 
     ** copied and adapted from the getCurves() function
     */

     /* convert surface 2 to PR LIB format */
     convertNURBSFromSI( &args[ 2 + nbIntmCurves /* <- tricky */ ], &temp2, &idata->scene );
     clampNURBSsurface( &temp2, &temp );

     globalise( &args[ 2 + nbIntmCurves /* <- tricky */ ], &temp, &idata->scene );
     scaleKnotVector( &temp.pf_u, 1.0 );
     scaleKnotVector( &temp.pf_v, 1.0 );
    
     /* evaluate the surface and produce the NURBS */

     /* Check if we need to evaluate in the v direction, 
	and transpose the surface if this is the case */
     SAA_dialogitemGetStateValue( context, DLG_CHK_U_DIR, &mode );
     if ( !mode )
       nrb_transpose( &temp );

     SAA_dialogitemGetFloatValue( context, DLG_P_INCRV, &innerValue );
     nrb_evaluate( &temp, innerValue, &temp2 );
     convertNURBSToSI( &temp2, &extractedCurves[ nbIntmCurves + 2 ], &idata->scene );

     SAA_dialogitemGetFloatValue( context, DLG_P_OUTCRV, &outerValue );
     nrb_evaluate( &temp, outerValue, &temp2 );
     convertNURBSToSI( &temp2, &extractedCurves[ nbIntmCurves + 3 ], &idata->scene );
   }
     
   if ( result == SI_SUCCESS ) {
     /* 
     ** Store all intermediate curves in the array...
     */
     for( i=0; i < nbIntmCurves; i++ ) {
       extractedCurves[ i + 2 ] = args[ i + 2 ];
     } 

     /*
     ** Get r1 and r2 from the dialog
     */
     SAA_dialogitemGetIntValue( context, DLG_EDIT1, &idata->r1 );
     SAA_dialogitemGetIntValue( context, DLG_EDIT2, &idata->r2 );

     /* 
     ** Create the blend surface
     */
     twistCurves(idata, extractedCurves, nbIntmCurves + 4);
     do_blending2( extractedCurves, nbIntmCurves + 4,
		   &temp, &idata->scene, idata->r1, idata->r2 );

     /*
     ** If it already exists in the instance data,
     ** copy the points. 
     ** Otherwise, convert it to SI format and store it in
     ** instance data
     */
     if ( idata->preview.elemid == -1 ) {
       convertNURBSToSI( &temp, &idata->preview, &idata->scene );
     } else {
       copyPoints( &temp, &idata->preview, &idata->scene, FALSE, FALSE );
     }

     SAA_sceneRefresh( context );
   } else {
     SAA_statusBarSet( "Could not create preview surface",
		       SAA_WARNING_CODE );
   }

   /*
   ** Clean up 
   */
   if ( constUV ) {
     /* destroy the constant u/v curves that we have created above */
     SAA_elementDestroy( &idata->scene, &extractedCurves[ nbIntmCurves + 2 ] );
     SAA_elementDestroy( &idata->scene, &extractedCurves[ nbIntmCurves + 3 ] );
   }

#ifdef __cplusplus
   delete [] extractedCurves;
#else
   free( extractedCurves );
#endif

}

/*
** toggleCallback() is a callback function for the setup
** dialog and it is called when the check box for constant parameter
** curves is changed. It enables and disables specific items on the
** dialog.
*/
SI_Error toggleCallback (
  const SAA_CustomContext context,  /* current context */
  int _UNUSED_ARG( dialogItemId ),                 /* id of the dialog item */
  void * _UNUSED_ARG( userData )                  /* User  data */
 )
{
   InstanceData	*idata;
   SAA_Boolean mode;

   SAA_customContextGetInstanceData(context, (void **) &idata);
   /* *** check return value of ..GetInstData */

   SAA_dialogitemGetStateValue( context, DLG_CHK_CONST, &mode );

   if ( mode ) {
     /* 
     ** Disable text view for surface 2 and enable text edit items 
     ** for parameter values
     */
     SAA_dialogitemSetSensitivity( context, DLG_TXTVW3, TRUE );
     SAA_dialogitemSetSensitivity( context, DLG_TXTVW4, TRUE );

     SAA_dialogitemSetSensitivity( context, DLG_P_INCRV, FALSE );
     SAA_dialogitemSetSensitivity( context, DLG_P_OUTCRV, FALSE );
     SAA_dialogitemSetSensitivity( context, DLG_CHK_U_DIR, FALSE );
   } else {
     /* 
     ** Enable text view for surface 2 and disable text edit items 
     ** for parameter values
     */
     SAA_dialogitemSetSensitivity( context, DLG_TXTVW3, FALSE );
     SAA_dialogitemSetSensitivity( context, DLG_TXTVW4, FALSE );

     SAA_dialogitemSetSensitivity( context, DLG_P_INCRV, TRUE );
     SAA_dialogitemSetSensitivity( context, DLG_P_OUTCRV, TRUE );
     SAA_dialogitemSetSensitivity( context, DLG_CHK_U_DIR, TRUE );
   }
     
   return SI_SUCCESS;
}

/*
** selectCurveCallback() is a callback function for the setup
** dialog and it is called when a projection curve is selected
** in one of the text views. This function takes care of 
** highlighting the NURBS curve corresponding to the selected
** curve.
*/
SI_Error selectCurveCallback (
  const SAA_CustomContext context,  /* current context */
  int dialogItemId,                 /* id of the dialog item */
  void * _UNUSED_ARG( userData )                  /* User  data */
 )
{
   InstanceData	*idata;		/* This effect's instance data.		*/
   int i, nbSel, result;

   result = SAA_customContextGetInstanceData(context, (void **) &idata);
   /* *** check return value of ..GetInstData */

   /*
   ** Select the extracted trim curve that has been selected
   ** We check the number of selected items to avoid nasty messages
   ** from SoftImage...
   */
   SAA_textlistGetNbSelected( context, dialogItemId, &nbSel );
   if ( nbSel == 1 ) {
     SAA_textlistGetSelected( context, dialogItemId, 1, &i );

     switch ( dialogItemId ) {
     case DLG_TXTVW1 :
       SAA_selectlistSetElements( &idata->scene, TRUE, 1, 
				  &idata->surfaces[0].extractedProjCurves[i] );
       break;
     case DLG_TXTVW2 : 
       SAA_selectlistSetElements( &idata->scene, TRUE, 1, 
				  &idata->surfaces[0].extractedProjCurves[i] );
       break;
     case DLG_TXTVW3 : 
       SAA_selectlistSetElements( &idata->scene, TRUE, 1, 
				  &idata->surfaces[1].extractedProjCurves[i] );
       break;
     case DLG_TXTVW4 : 
       SAA_selectlistSetElements( &idata->scene, TRUE, 1, 
				  &idata->surfaces[1].extractedProjCurves[i] );
       break;
     }
   } else {
     SAA_selectlistClear( &idata->scene );
   }

   /*
   ** We refresh the scene so that the selected element will be
   ** visible to the user
   */
   SAA_sceneRefresh( context );

   return SI_SUCCESS;
}

/*
** closeDialogCallback() is the callback function for the setup 
** dialog. It is called when the dialog is closed and takes care
** of storing the selected projection curves in the instance
** data.
*/
SI_Error closeDialogCallback (
  const SAA_CustomContext context,  /* current context */
  int dialogItemId,                 /* id of the dialog item */
  void * _UNUSED_ARG( userData )                  /* User  data */
 )
{
   InstanceData	*idata;		/* This effect's instance data.		*/
   int i, j, nbSel, result;
   SAA_Boolean mode;

   result = SAA_customContextGetInstanceData(context, (void **) &idata);
   /* *** check return value of ..GetInstData */

   if ( dialogItemId == DLG_OK ) {
     /*
     ** 'ok' has been pressed: store the 4 selected projection curves.
     ** If no item is selected in a text view, we choose a default
     ** curve (index 0 for the inner and index 1 for the outer curve)
     */

     /* 
     ** Get inner curve of first surface 
     */
     SAA_textlistGetNbSelected( context, DLG_TXTVW1, &nbSel );
     if ( nbSel == 1 ) {
       SAA_textlistGetSelected( context, DLG_TXTVW1, 1, &i );
       idata->in1 = i;
     } else {
       SAA_statusBarSet( "Default inner curve selected for surface 1",
			 SAA_WARNING_CODE );
       i = 0;
     }
     idata->surface1Curves[0] = idata->surfaces[0].projCurves[ i ];

     /* 
     ** Get outer curve of first surface 
     */
     SAA_textlistGetNbSelected( context, DLG_TXTVW2, &nbSel );
     if ( nbSel == 1 ) {
       SAA_textlistGetSelected( context, DLG_TXTVW2, 1, &i );
       idata->out1 = i;
     } else {
       SAA_statusBarSet( "Default outer curve selected for surface 1",
			 SAA_WARNING_CODE );
       i = 1;
     }
     idata->surface1Curves[1] = idata->surfaces[0].projCurves[ i ];
     
     /* 
     ** Get inner curve of second surface 
     */
     if ( idata->surfaces[1].nbCrv > 0 ) {
       SAA_textlistGetNbSelected( context, DLG_TXTVW3, &nbSel );
       if ( nbSel == 1 ) {
	 SAA_textlistGetSelected( context, DLG_TXTVW3, 1, &i );
	 idata->in2 = i;
       } else {
	 SAA_statusBarSet( "Default inner curve selected for surface 2",
			   SAA_WARNING_CODE );
	 i = 0;
       }
       idata->surface2Curves[0] = idata->surfaces[1].projCurves[ i ];

       /*
       ** Get outer curve of second surface 
       */
       SAA_textlistGetNbSelected( context, DLG_TXTVW4, &nbSel );
       if ( nbSel == 1 ) {
	 SAA_textlistGetSelected( context, DLG_TXTVW4, 1, &i );
	 idata->out2 = i;
       } else {
	 SAA_statusBarSet( "Default outer curve selected for surface 2",
			   SAA_WARNING_CODE );
	 i = 1;
       }
       idata->surface2Curves[1] = idata->surfaces[1].projCurves[ i ];
     }

     /*
     ** Store the ids of the projection curves in text/edit fields
     ** in the dialog
     */
     SAA_dialogitemSetIntValue( context, DLG_INCRV1ID, 
				idata->surface1Curves[0].id[0] );
     SAA_dialogitemSetIntValue( context, DLG_OUTCRV1ID, 
				idata->surface1Curves[1].id[0] );
     SAA_dialogitemSetIntValue( context, DLG_INCRV2ID, 
				idata->surface2Curves[0].id[0] );
     SAA_dialogitemSetIntValue( context, DLG_OUTCRV2ID, 
				idata->surface2Curves[1].id[0] );
     /* Store the nr of arguments in the dialog box */
     SAA_dialogitemSetIntValue( context, DLG_NBARGS, idata->nbArgs );

     /* If the user has selected projection curves for the 
     ** second curves, check whether there is indeed at least
     ** one projection curve. If not, we cancel the surface 
     ** blending to prevent SoftImage from crashing
     */
     SAA_dialogitemGetStateValue( context, DLG_CHK_CONST, &mode );

     if ( ( !mode ) && ( idata->surfaces[1].nbCrv == 0 ) ) { 
       /* Give a message to the user */
       SAA_statusBarSet( "Surface should have at least one projection curve",
			 SAA_ERROR_CODE );
       result = SI_ERR_CUSTOM_FATAL;
     }
   }

   /* 
   ** Destroy the extracted curves and free the allocated memory
   */
   for( i=0; i<2; i++ ) {
     for( j=0; j<idata->surfaces[i].nbCrv; j++ ) {
       SAA_elementDestroy( &idata->scene, 
			   &idata->surfaces[i].extractedProjCurves[j] );
     }

#ifdef __cplusplus
     delete [] idata->surfaces[i].extractedProjCurves;
     delete [] idata->surfaces[i].projCurves;
#else
     free( idata->surfaces[i].extractedProjCurves );
     free( idata->surfaces[i].projCurves );
#endif
   }

   /*
   ** Destroy the preview curve if it exists
   */
   if (idata->preview.elemid != -1) {
     SAA_elementDestroy( &idata->scene, &idata->preview );
   }

   SAA_selectlistClear( &idata->scene );

   SAA_sceneRefresh( context );

   return result;
}

/*********************************************/
/* EDIT DIALOG CALLBACK FUNCTIONS            */
/*********************************************/

/*
** cancelCallback() is the callback function for the cancel button
** of the 'edit parameters'
** dialog. It restores the settings of the parameters to their old
** values and updates the blend surface. 
*/
SI_Error cancelCallback (
  const SAA_CustomContext context,  /* current context */
  int _UNUSED_ARG( dialogItemId ),                 /* id of the dialog item */
  void * _UNUSED_ARG( userData )                  /* User  data */
 )
{
   int result;
   InstanceData	*idata;		/* This effect's instance data.		*/
   SAA_CustomValueList valueList;

   result = SAA_customContextGetInstanceData(context, (void **) &idata);
   /* *** check return value of ..GetInstData */

   /* 
   ** update the r1 and r2 fields and the other parameters
   ** of the instance data
   ** we ignore the changes made in the dialog by 
   ** getting the values from the custom value 
   ** list 
   */
   SAA_customContextGetCustomValueList( context, &valueList );
   SAA_cusvalGetIntValue( valueList, "SRF1PARAM", &idata->r1 );
   SAA_cusvalGetIntValue( valueList, "SRF2PARAM", &idata->r2 );
   SAA_cusvalGetFloatValue( valueList, "INCRVP", &idata->innerValue );
   SAA_cusvalGetFloatValue( valueList, "OUTCRVP", &idata->outerValue );
   SAA_cusvalGetStateValue( valueList, "CNSTDIR", &idata->constU );

   /* We set updateFlag to FALSE so that update will perform 
      the actual update */
   idata->updateFlag = FALSE;
   update( context );

   SAA_sceneRefresh( context );

   return SI_SUCCESS;
}

/*
** applyCallback() is the callback function for the 'edit parameters'
** dialog. It applies the settings in the dialog items to the
** blend surface, and updates it.
*/
SI_Error applyCallback (
  const SAA_CustomContext context,    /* current context */
  int _UNUSED_ARG( dialogItemId ),         /* id of the dialog item */
  void * _UNUSED_ARG( userData )           /* User  data */
 )
{
   int result;
   InstanceData	*idata;		/* This effect's instance data.		*/

   result = SAA_customContextGetInstanceData(context, (void **) &idata);
   /* *** check return value of ..GetInstData */

   /* 
   ** Update the r1 and r2 fields and the constant u/v curve
   ** parameters in the instance data 
   */
   SAA_dialogitemGetIntValue( context, DLG_EDIT1, &idata->r1);
   SAA_dialogitemGetIntValue( context, DLG_EDIT2, &idata->r2);
   SAA_dialogitemGetFloatValue( context, DLG_P_INCRV, &idata->innerValue );
   SAA_dialogitemGetFloatValue( context, DLG_P_OUTCRV, &idata->outerValue );
   SAA_dialogitemGetStateValue( context, DLG_CHK_U_DIR, &idata->constU );

   /* We set updateFlag to FALSE so that update will perform 
      the actual update */
   idata->updateFlag = FALSE;
   update( context );
   SAA_sceneRefresh( context );

   return SI_SUCCESS;
}


#ifdef __cplusplus
}
#endif


/****************************************/
/*           LOCAL FUNCTIONS            */
/****************************************/

/* 
** initInstance() performs the initialisation of the instance data 
** for this custom effect. 
*/
SI_Error initInstance( InstanceData ** idata )
{
   SI_Error result;

   /*
   ** Initialize Saaphire.  This is always the first thing that a Saaphire
   ** effect should do.
   */

   result = SAA_sceneInit();
   if(result != SI_SUCCESS)

   /*
   ** Set the verify flags so that SI will verify everything.
   */
   SAA_verifyFlagSet( SAA_VERIFY_ALL, FALSE );

   /*
   ** Allocate instance-specific data for this effect instance.  Store the
   ** pointer in the context argument so that it gets passed to
   ** update() and cleanup().
   */

   if ( result == SI_SUCCESS ) {
#ifdef __cplusplus
     *idata = new InstanceData;
#else
     *idata = ( InstanceData * ) malloc( sizeof( InstanceData ) );
#endif

     if( *idata != NULL ) {
       result = SI_SUCCESS;
     } else {
       fprintf( stderr, "ERROR: initInstance(): Error allocating instance data memory." );
       result = SI_ERR_CUSTOM_FATAL;
     }
   }

   /* Store the current scene */
   if( result == SI_SUCCESS ) {
     result = SAA_sceneGetCurrent( & (*idata)->scene );
   }
   (*idata)->result = result;

   return result;
}


/* 
** getProjectionCurves() gets the projection curves of a given NURBS
** surface and stores them in the given ProjectionCurvesInfo
** struct.
*/
SI_Error getProjectionCurves( SAA_Scene * scene, SAA_Elem * srf, 
			      ProjectionCurvesInfo * info )
{
  SI_Error result;
  int nbCrv;

  SAA_nurbsSurfaceGetNbTrimCurves( scene, srf, 
				   SAA_TRIMTYPE_PROJECTION, &nbCrv );
     
  /* There should be at least one projection curve 
  **  ^ condition has been dropped to allow for constant u/v
  **    parameter curves
  */
  info->nbCrv = nbCrv;
  if (nbCrv >= 1) 
    {
      /* Allocate memory for the array of trim curves and 
	 for the array of the extracted trim curves */

#ifdef __cplusplus
      info->projCurves = new SAA_SubElem [ nbCrv ]; 
      info->extractedProjCurves = new SAA_Elem [ nbCrv ];
#else
      info->projCurves = (SAA_SubElem *) 
	  calloc( nbCrv, sizeof(SAA_SubElem) );
      info->extractedProjCurves = (SAA_Elem *) 
	  calloc( nbCrv, sizeof(SAA_Elem) );
#endif
      assert( (info->projCurves != NULL && 
	       info->extractedProjCurves != NULL) );
      
      SAA_nurbsSurfaceGetTrimCurves( scene, srf, 
				    SAA_GEOM_DEFORMED, 0,
				    SAA_TRIMTYPE_PROJECTION, 
				     nbCrv, info->projCurves );
       
      SAA_surfaceCurveExtract(scene, srf, nbCrv, info->projCurves,
			      info->extractedProjCurves);

      result = SI_SUCCESS;
    } 
  else 
    {
      info->projCurves = NULL; 
      info->extractedProjCurves = NULL;
      result = SI_SUCCESS;
    }

  return result;
}


/* 
** initInstanceData() performs the initialisation of the instance data 
** for this custom effect. It also extracts the projection curves 
** from the surfaces picked by the user and stores these in the 
** instance data 
*/
SI_Error initInstanceData
( 
   SAA_CustomContext context
)
{
  InstanceData	*idata;   /* This effect's instance data. */
  SI_Error	result;	  /* Return value.		  */
  SAA_ModelType mdlType;
  SAA_Elem args [ NB_ARGS ];
  int i;


  /* First, create the instance data */
  result = initInstance( &idata );

  if ( result == SI_SUCCESS ) {
    /* Pass the instance data pointer to SAAPHIRE */
    result = SAA_customContextSetInstanceData(context, (void *) idata);
  }

  /*
  ** Store the current scene in the instance-specific data.
  ** Get the models that have been selected.
  */
  if( result == SI_SUCCESS )
    {
      /* Get the array of arguments from the context */
      SAA_customContextGetNbArguments( context, &idata->nbArgs );
      SAA_customContextGetArguments( context, idata->nbArgs, args );
      idata->nbIntmCurves = (idata->nbArgs / 2) - 2;
      /* division by 2 because each argument occurs twice in .cus file
	 and - 2 to subtract the 2 surface arguments 
      */
      
      /* 
      ** We check if the last picked model is a NURBS surface and
      ** whether arguments 2 ... (nbArgs/2)-1 are NURBS curves.
      ** Note that each picked argument occurs twice in the argument
      ** list.
      */   
      for( i=0; i < idata->nbIntmCurves; i++ ) {
	SAA_modelGetType( &idata->scene, &args[ i+2 ], &mdlType );
	result = ( mdlType == SAA_MNCRV ) ? result : SI_ERR_CUSTOM_FATAL ;
      }

      SAA_modelGetType( &idata->scene, &args[ ARG_SRF2 ], &mdlType );
      result = ( mdlType == SAA_MNSRF ) ? result : SI_ERR_CUSTOM_FATAL ;

      if ( result == SI_SUCCESS ) {
	/*
	** Store the projection curves from the surfaces
	** in the instance data
	*/
	getProjectionCurves( &idata->scene, &args[ ARG_SRF1 ], 
			     &idata->surfaces[0] );
	getProjectionCurves( &idata->scene, &args[ ARG_SRF2 ], 
			     &idata->surfaces[1] );

	/*
	** Check if surface 1 has at least 1 projection curve
	** and cancel custom effect if not
	*/
	if (idata->surfaces[0].nbCrv == 0) {
	  /* Give a message to the user */
	  SAA_statusBarSet( "Surface 1 should have at least 1 projection curve",
			  SAA_ERROR_CODE );
	  result = SI_ERR_CUSTOM_FATAL;
	}
      } else {
	/* Give a message to the user */
	SAA_statusBarSet( "Invalid selection of surface or curves",
			  SAA_ERROR_CODE );
	
      }
   }
   idata->result = result;

   return result;
}

/*
** twistCurves() applies the twist factors, stored in idata->r1/2, to the first 2, and
** last 2 curves stored in `curves'. 
*/

void twistCurves(InstanceData *idata, SAA_Elem *curves, int num_curves)
{
  SAA_Boolean is_valid;
  int i;

  /* Twist the curve using idata->r1. We have to have a loop because in 
     theory the user could use the same curve for the inner & outer. MP */

  for(i=0; i<2; i++)
  {
    SAA_elementIsValid(&idata->scene, &curves[i], &is_valid);

    if(is_valid)
      SAA_nurbsCurveShiftParameterization(&idata->scene, &curves[i],
					  idata->r1);
  }

  /* Now the last 2 */

  for(i=(num_curves-2); i<(num_curves); i++)
  {
    SAA_elementIsValid(&idata->scene, &curves[i], &is_valid);

    if(is_valid)
      SAA_nurbsCurveShiftParameterization(&idata->scene, &curves[i],
					  idata->r2);
  }
}


/* 
** getCurves() extracts the curves from the given surfaces
** and returns them in curves. curves should be an array of 4
** elements. Elements 0 and 1 will become the extracted curves of
** surface 1; elements 2 and 3 will become the extracted curves of
** surface 2.
** (0 and 2: inner curves; 1 and 3: outer curves)
*/
void getCurves( InstanceData * idata, SAA_Elem * curves, 
		SAA_Elem * srf1, SAA_Elem * srf2 )
{
  int i;
  PR_nurb pr_srf, tmp_inner, tmp_outer, temp;
  SAA_Boolean closed[2];
  int num_curves;
  SAA_SubElem tmp_curves[2];
  SAA_SubElem *martin;

  /* Extract the projection curves of surface 1 */

  SAA_nurbsSurfaceGetNbTrimCurves(&idata->scene, srf1, SAA_TRIMTYPE_PROJECTION,
				  &num_curves);
  martin = (SAA_SubElem *) malloc(num_curves * sizeof(SAA_SubElem));

  SAA_nurbsSurfaceGetTrimCurves(&idata->scene, srf1, SAA_GEOM_DEFORMED, 0,
				SAA_TRIMTYPE_PROJECTION, num_curves, 
				martin);
  
  /* Now extract those ones which are actually the right ones */

  for(i=0; i<num_curves; i++)
  {
    if(idata->in1 == martin[i].id[0])
      SAA_surfaceCurveExtract(&idata->scene, srf1, 1, &martin[i],
			      curves);
    if(idata->out1 == martin[i].id[0])
      SAA_surfaceCurveExtract(&idata->scene, srf1, 1, &martin[i],
			      &curves[1]);
  }

  if ( !idata->constantParam ) {
    /* 
    ** Extract the projection curves of surface 2
    ** in the case the user has specified projection
    ** curves for surface 2
    */
    SAA_nurbsSurfaceGetNbTrimCurves(&idata->scene, srf2, SAA_TRIMTYPE_PROJECTION,
				    &num_curves);
    free(martin);
    martin = (SAA_SubElem *) malloc(num_curves * sizeof(SAA_SubElem));

    SAA_nurbsSurfaceGetTrimCurves(&idata->scene, srf2, SAA_GEOM_DEFORMED, 0,
				  SAA_TRIMTYPE_PROJECTION, num_curves, 
				  martin);
  
    /* Now extract those ones which are actually the right ones */
    
    for(i=0; i<num_curves; i++)
    {
      if(idata->in1 == martin[i].id[0])
	SAA_surfaceCurveExtract(&idata->scene, srf2, 1, &martin[i],
				&curves[2]);
      if(idata->out1 == martin[i].id[0])
	SAA_surfaceCurveExtract(&idata->scene, srf2, 1, &martin[i],
				&curves[3]);
    }

  } else {
    /* convert the surface to PR LIB format */
    convertNURBSFromSI( srf2, &temp, &idata->scene );
    clampNURBSsurface( &temp, &pr_srf );

    globalise( srf2, &pr_srf, &idata->scene );
    scaleKnotVector( &pr_srf.pf_u, 1.0 );
    scaleKnotVector( &pr_srf.pf_v, 1.0 );
    
    /* evaluate the surface and produce the NURBS */

    /* Check if we need to evaluate in the v direction, 
       and transpose the surface if this is the case */
    if ( ! idata->constU )
      nrb_transpose( &pr_srf );

    nrb_evaluate( &pr_srf, idata->innerValue, &tmp_inner );
    nrb_evaluate( &pr_srf, idata->outerValue, &tmp_outer );

    /* 
    ** For now we convert the curves back to SI format.
    ** It's not very efficient, as they will be converted back to
    ** NURBS PR LIB format, but doing it this way required
    ** less modification of the code (and it should work correctly).
    */
    convertNURBSToSI( &tmp_inner, &curves[2], &idata->scene );
    convertNURBSToSI( &tmp_outer, &curves[3], &idata->scene );
  }
}


/*
** do_blending2() performs the actual blending. It takes four
** curves in SoftImage format, a scene, and the two parameters
** for the 'blend' surface, and it creates a blend surface
** in NURBS Procedure library format.
** It expects an array of NURBS curves, which should contain at least 
** 4 curves. 
** The first and last curves are the 'outer' curves (indexes 0 and nbCurves-1),
** the second and second last curves are the 'inner' curves (indexes 1 and
** nbCurves-2). 
** The other curves, if any, are intermediate curves.
*/
void do_blending2( SAA_Elem * curves, int nbCurves,
		   PR_nurb * out, SAA_Scene * scene,
		   int r1, int r2 )
{
  int i;
  SAA_ModelType mdlType;
  PR_nurb * nurbs;

  assert( nbCurves >= 4 );
  
  /* Verify the model type of the parameters */
  for( i = 0; i < nbCurves; i++ ) {
    SAA_modelGetType( scene, &curves[i], &mdlType ); 
    assert (mdlType == SAA_MNCRV);
  }

  /* MJE 12/8: we should check if all the curves are closed */
  /*  SAA_verifyFlagSet( SAA_VERIFY_MODEL, FALSE );*/

#ifdef __cplusplus
  nurbs = new PR_nurb [ nbCurves ];
#else
  nurbs = (PR_nurb *) calloc( nbCurves, sizeof( PR_nurb ) );
#endif

  for( i = 0; i < nbCurves; i++ ) {
    /*
    ** Clear the PR nurbs 
    */
    nrb_clear( &nurbs[i] );

    /*
    ** Convert the SI NURBS to NURBS procedure library format 
    */
    convertNURBSFromSI( &curves[i], &nurbs[i], scene );

    /* clamp the curves */
    clampNURBScurve( &nurbs[i], &nurbs[i] );

    /*
    ** We scale the u knot vectors of all NURBS to [0.0 ... 1.0] 
    ** (which is more or less standard)
    */ 
    scaleKnotVector( &nurbs[i].pf_u, 1.0 );

    /*
    ** SI has given us the local coordinates of the vertices of the
    ** objects. We will 'globalise' them by getting the transformation 
    ** matrix of the center of the surfaces and transforming the NURBS PR
    ** lib surfaces.
    */   
    globalise( &curves[i], &nurbs[i], scene );
  }

  /*
  ** Create the blend surface
  */
  blender( nurbs, nbCurves, out);

  /* Clean up things */
  for (i=0; i<nbCurves; i++) {
    nrb_deallocatepts( &nurbs[i].pf_ppp );
    nrb_deallocateknots( &nurbs[i].pf_u );
    nrb_deallocateknots( &nurbs[i].pf_v );
  }

/*  SAA_verifyFlagSet( SAA_VERIFY_MODEL, TRUE );*/

#ifdef __cplusplus
  delete [] nurbs;
#else
  free( nurbs );
#endif

}
