/*******************************************************************

  NURBS blending code
  Version 1.0

 ******************************************************************
  Written/adapted by: M.J. Evers
  (c) Computer Graphics Unit, Manchester Computing, 
      
  Created:       31/7/1996       
  Last Modified: 08/08/1996, 15/8/96

 ******************************************************************/


#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

#include "blending.h"



void blender( PR_nurb * curves, int nbCurves,  
	      PR_nurb * result, double r1, double r2 )
/* Blender creates a surface from src and returns it in result.

   The array curves should contain at least 4 curves:
     curves[0]  =  dirCurve1 = outer curve of surface 1
     curves[1]  =  curve1 = inner curve of surface 1
     curves[nbCurves-2]  =  curve2 = inner curve of surface 2
     curves[nbCurves-1]  =  dirCurve2 = outer curve of surface 2

   The other curves are intermediate curves.

   The caller should allocate memory for result (that is, for the
   basic NURBS structure, not for the vertices and knot vectors...)
*/ 
{
  int i, j;
  int nbVerticesU, nbVerticesV;
  int nbKnotsU, nbKnotsV;
  Ppoint4 v1, v2, v3;

  nrb_clear( result );

  /*
  ** First, we normalise the 4 NURBS so that they all have the 
  ** same knot vector, the same number of vertices and the same order
  */
  normalise( curves, FALSE, nbCurves ); 

  /* 
  ** We now create a NURBS with the following properties:
  **  u-dir: order = 4, nr ctrl points = nbCurves
  **  v-dir: order and nr of ctrl points is equal to
  **         order and nr of ctrl points of NURBS curves
  */
  nbVerticesU = curves[0].pf_u.pf_n; 
  nbVerticesV = nbCurves;

  nbKnotsU = curves[0].pf_u.pf_nt;
  nbKnotsV = nbCurves + 4;

  nrb_init( curves[0].pf_u.pf_k, nbKnotsU, 4, nbKnotsV, result );

  /*
  ** Allocate space for the knots - not necessary, already done by nrb_init 
  ** and create a knot vector for the v direction 
  */
  nrb_makeknots( 0.0, 1.0, &result->pf_v );

  /*
  ** Copy the control points of the 'outer' curves
  ** that is, curves[0], which will become the 0th u curve of the surface 
  ** and curves[nbCurves-1], which will become the last u curve
  */
  for( j=0; j<nbVerticesU; j++ ) {
    result->pf_ppp->pts[ j ] = curves[0].pf_ppp->pts[j];
  }

  for( j=0; j<nbVerticesU; j++ ) {
    result->pf_ppp->pts[ (nbCurves-1)*nbVerticesU + j ] = 
      curves[ (nbCurves-1) ].pf_ppp->pts[j];
  }
  
  /*
  ** Now copy the control points of the 'inner' curves, 
  ** (curves[1] and curves[2]) using the parameters r1 and r2.
  */

  /* 
  ** curves[1], will become the 1st u curve of the surface
  */
  for( j=0; j<nbVerticesU; j++ ) {
    v1 = curves[1].pf_ppp->pts[j]; /* inner curve */
    v2 = curves[0].pf_ppp->pts[j]; /* outer curve */

    v3.x = v2.x + r1 * ( v1.x - v2.x );
    v3.y = v2.y + r1 * ( v1.y - v2.y );
    v3.z = v2.z + r1 * ( v1.z - v2.z );
    v3.w = v2.w + r1 * ( v1.w - v2.w );

    result->pf_ppp->pts[ nbVerticesU + j ] = v3;
  }

  /* 
  ** curves[nbCurves-2], will become the 2nd last u curve of the surface
  */
  for( j=0; j<nbVerticesU; j++ ) {
    v1 = curves[ (nbCurves-2) ].pf_ppp->pts[j];
    v2 = curves[ (nbCurves-1) ].pf_ppp->pts[j];

    v3.x = v2.x + r2 * ( v1.x - v2.x );
    v3.y = v2.y + r2 * ( v1.y - v2.y );
    v3.z = v2.z + r2 * ( v1.z - v2.z );
    v3.w = v2.w + r2 * ( v1.w - v2.w );

    result->pf_ppp->pts[ (nbCurves-2) * nbVerticesU + j ] = v3;
  }

  /* 
  ** Copy the points of all the intermediate curves 
  */
  for( i=2; i < (nbCurves-2) ; i++ ) {
    for( j=0; j<nbVerticesU; j++ ) {
      result->pf_ppp->pts[ i*nbVerticesU + j ] = curves[i].pf_ppp->pts[j];
    }
  }

  /* 
  ** Finally, copy the knots
  ** Note that we only have to copy the knots in the u direction:
  ** the knots in the v direction have already been created 
  */
  for( i=0; i<nbKnotsU; i++ ) {
    result->pf_u.pf_kk->knots[i] = curves[0].pf_u.pf_kk->knots[i];
  }
}


void normalise( PR_nurb * list, int direction, int num_sections )
/* Original version by M. Preston
   Adapted by M.J.Evers, 1996
   
   sections is an array of PR NURBS; num_sections is the number of elements
   in this array;
*/
/* This procedure normalises a section list,ie it ensures everything is 
   elevated to the correct order,and all sections have the correct 
   number of control points .

   If called with direction = TRUE it normalises everything in the x
   direction,and in the y if direction = FALSE. */
{
    int current, i; 
    PR_nurb temp_nurb, knot_nurb;
    int max_order;
    int *dummy;
    nrb_clear( &temp_nurb );

    max_order = list[ 0 ].pf_u.pf_k;
    for( current=1; current<num_sections; current++ )
      {
	if ( list[ current ].pf_u.pf_k > max_order )
	    max_order = list[ current ].pf_u.pf_k;
      }

    /* Now change required parts */
    for( current=0; current<num_sections; current++ )
      {
	while ( list[ current ].pf_u.pf_k < max_order )
	  {
	    nrb_clear( &temp_nurb );
	    nrb_elevate( FALSE, &list[ current ], &temp_nurb);
	   /**/ nrb_deallocateknots( &list[ current ].pf_u );
	   /**/ nrb_deallocateknots( &list[ current ].pf_v );
	   /**/ nrb_deallocatepts( &list[ current ].pf_ppp );
	    nrb_copy( &temp_nurb, &list[ current ]);
	   /**/ nrb_deallocateknots( &temp_nurb.pf_u );
	   /**/ nrb_deallocateknots( &temp_nurb.pf_v );
	   /**/ nrb_deallocatepts( &temp_nurb.pf_ppp );
	  }
      }
	
    /*
    ** Now initialise
    **  we want to use nrb_init, which sets the order, nr of knots and vertices, and
    **  allocates memory for the knot vectors and for the vertices 
    */
    temp_nurb.pf_u.pf_k = max_order;
    temp_nurb.pf_u.pf_n = list[0].pf_u.pf_n;
    temp_nurb.pf_u.pf_nt = temp_nurb.pf_u.pf_n + temp_nurb.pf_u.pf_k;
    temp_nurb.pf_u.pf_kk = NULL;

    temp_nurb.pf_v.pf_k = list[0].pf_v.pf_k;
    temp_nurb.pf_v.pf_n = list[0].pf_v.pf_n;
    temp_nurb.pf_v.pf_nt = temp_nurb.pf_v.pf_n + temp_nurb.pf_v.pf_k;
    temp_nurb.pf_v.pf_kk = NULL;

    temp_nurb.pf_ppp = NULL;

    nrb_allocateknots( &temp_nurb.pf_u );
    nrb_allocateknots( &temp_nurb.pf_v );
    nrb_allocatepts( (temp_nurb.pf_u.pf_nt * temp_nurb.pf_v.pf_nt) * 2,
		     &temp_nurb.pf_ppp );
    dummy = (int *) malloc(500);

    for( i=0; i < (temp_nurb.pf_u.pf_nt * temp_nurb.pf_v.pf_nt * 2); i++ )
      {
	temp_nurb.pf_ppp->pts[i] = ptk_point4( 0.0, 0.0, 0.0, 0.0 );
      }
    for(i=0; i < list[0].pf_u.pf_nt; i++)
      temp_nurb.pf_u.pf_kk->knots[i] =
	list[0].pf_u.pf_kk->knots[i];
    for(i=0; i<list[0].pf_v.pf_nt; i++)
      temp_nurb.pf_v.pf_kk->knots[i] =
	list[0].pf_v.pf_kk->knots[i];

    for( current=1; current < num_sections; current++ )
      {
	nrb_unionknots( &list[ current ].pf_u, &temp_nurb.pf_u, &temp_nurb.pf_u );
      }

/*    nrb_dump( stderr, &temp_nurb );*/

    /* When allocating the points we use the number of knots from
       temp_nurb (which is the total nr of knots after union) minus
       the order. We do not use the number of points, because it 
       is more or less undefined (because it is not used during
       the unioning of knots.
    */
    temp_nurb.pf_u.pf_n = (temp_nurb.pf_u.pf_nt - temp_nurb.pf_u.pf_k);
    temp_nurb.pf_v.pf_n = 1;

    nrb_allocatepts( 2*(temp_nurb.pf_u.pf_nt - temp_nurb.pf_u.pf_k + 100) * 
		     (temp_nurb.pf_v.pf_nt - temp_nurb.pf_v.pf_k + 100),
		     &temp_nurb.pf_ppp );

    for( current=0; current < num_sections; current++ )
      {
	/* Perform Oslo algorithm to increase pts */

       /**/ nrb_deallocateknots( &temp_nurb.pf_v );
	nrb_copyknots( &list[ current ].pf_v, &temp_nurb.pf_v );

        /*
	** The solution to the one frustrating bug: allocate memory for the
	** points. RefineSurface doesn't do that (nrb_osloc did...)
	*/
	RefineSurface( &list[ current ], &temp_nurb, TRUE );

       /**/ nrb_deallocateknots( &list[ current ].pf_u );
       /**/ nrb_deallocateknots( &list[ current ].pf_v );
       /**/ nrb_deallocatepts( &list[ current ].pf_ppp );

	nrb_clear( &list[ current ] );	
	nrb_copy( &temp_nurb, &list[ current ] );

     }
    /**/ nrb_deallocateknots( &temp_nurb.pf_u );
    /**/ nrb_deallocateknots( &temp_nurb.pf_v );
    /**/ nrb_deallocatepts( &temp_nurb.pf_ppp );
}

void clampNURBScurve( PR_nurb * src, PR_nurb * dest )
/* Clamps an unclamped NURBS curve according to method
** described in: Piegl & Tiller, The NURBS Book, p. 576
** The caller should allocate memory for dest.
*/
{
  /*
  ** Locals:
  **  p = degree of curve
  **  m = number of knots in src (note that m in the passage
  **    mentioned above is the index of the last knot, as far
  **    as I can see)
  **  r = total nr of knots and ctrl point added 
  **      ( = p*2 if u(p) and u(m-p) have multipl. 1 )
  */
  int p, m, r, i;  
  PR_nurb tmp1, tmp2;
  double u, a;

  p = src->pf_u.pf_k - 1;
  m = src->pf_u.pf_nt;
  /* for the moment we assume that knots u(p) and u(m-p) 
     have multiplicity 1 
  */
  r = p * 2;

  nrb_clear( &tmp1 );
  nrb_init( src->pf_u.pf_k, src->pf_u.pf_nt + p, 0, 1, &tmp1 );
  nrb_allocateknots(&tmp1.pf_u);
/*  nrb_allocateknots(&tmp1.pf_v);
    tmp1.pf_v.pf_kk->knots[0] = 0.0;*/
  nrb_allocatepts( src->pf_u.pf_n + p, &tmp1.pf_ppp );

  nrb_clear( &tmp2 );
  nrb_init( src->pf_u.pf_k, src->pf_u.pf_nt + r, 0, 1, &tmp2 );
  nrb_allocateknots(&tmp2.pf_u);
  nrb_allocateknots(&tmp2.pf_v);
  tmp1.pf_v.pf_kk->knots[0] = 0.0;
  nrb_allocatepts( src->pf_u.pf_n + r, &tmp2.pf_ppp );

  /* Insert knots equal to u(p) */
  u = src->pf_u.pf_kk->knots[p];
  InsertKnot( src->pf_ppp->pts, src->pf_u.pf_kk->knots, src->pf_u.pf_n,
	      src->pf_u.pf_k, tmp1.pf_ppp->pts, tmp1.pf_u.pf_kk->knots, 
	      u, 1, p, p );

  /* Insert knots equal to u(m-p) */
  u = src->pf_u.pf_kk->knots[ (m-1) - p ];
  InsertKnot( tmp1.pf_ppp->pts, tmp1.pf_u.pf_kk->knots, src->pf_u.pf_n+p,
	      src->pf_u.pf_k, tmp2.pf_ppp->pts, tmp2.pf_u.pf_kk->knots
	      , u, 1, p, (m-1)-p + p );
  /* The extra +p is because of in tmp1 already p knots
     have been inserted */

  /* Now discard knots 0..p-1 and knots (r + (m-1)-p + 1) .. m-1 
     and discard vertices 0..p-1 and vertices (r + (n-1)-p + 1) .. n-1*/
  /* That is, do not copy them to dest. */

  nrb_clear( dest );
  nrb_init( src->pf_u.pf_k, src->pf_u.pf_nt, 0, 1, dest );
  nrb_allocateknots(&dest->pf_u);
  nrb_allocateknots(&dest->pf_v);
  dest->pf_v.pf_kk->knots[0] = 0.0;
  nrb_allocatepts( src->pf_u.pf_n + r, &dest->pf_ppp );

  for( i=0; i < dest->pf_u.pf_n; i++ ) {
    dest->pf_ppp->pts[i] = tmp2.pf_ppp->pts[i + p];
  }

  /*
  ** While copying the knots, we translate them so that they
  ** will start at 0.0 
  */
  a = tmp2.pf_u.pf_kk->knots[p];
  for( i=0; i < dest->pf_u.pf_nt; i++ ) {
    dest->pf_u.pf_kk->knots[i] = tmp2.pf_u.pf_kk->knots[i + p] - a;
  }

  /* Deallocate tmp1 and tmp2 memory */
  nrb_deallocateknots( &tmp2.pf_u );
  nrb_deallocateknots( &tmp2.pf_v );
  nrb_deallocatepts( &tmp2.pf_ppp );

  nrb_deallocateknots( &tmp1.pf_u );
  nrb_deallocateknots( &tmp1.pf_v );
  nrb_deallocatepts( &tmp1.pf_ppp );
/*  nrb_dump( stderr, dest );*/
}


void clampNURBSsurface( PR_nurb * src, PR_nurb * dest )
/* Clamps an unclamped NURBS surface in the u direction
** according to method
** described in: Piegl & Tiller, The NURBS Book, p. 576
** The caller should allocate memory for dest.
*/
{
  /*
  ** Locals:
  **  p = degree of curve
  **  m = number of knots in src (note that m in the passage
  **    mentioned above is the index of the last knot, as far
  **    as I can see)
  **  r = total nr of knots and ctrl point added 
  **      ( = p*2 if u(p) and u(m-p) have multipl. 1 )
  */
  int p, m, r, i, j;  
  PR_nurb tmp1, tmp2;
  double u, a;

  p = src->pf_u.pf_k - 1;
  m = src->pf_u.pf_nt;
  /* for the moment we assume that knots u(p) and u(m-p) 
     have multiplicity 1 
  */
  r = p * 2;

  /* tmp1 and tmp2 are used to temporarily store the extended knot vector
     and the extended vertice vector */
  nrb_clear( &tmp1 );
  nrb_init( src->pf_u.pf_k, src->pf_u.pf_nt + p, 0, 1, &tmp1 );
  nrb_allocateknots(&tmp1.pf_u);
/*  nrb_allocateknots(&tmp1.pf_v);
    tmp1.pf_v.pf_kk->knots[0] = 0.0;*/
  nrb_allocatepts( src->pf_u.pf_n + p, &tmp1.pf_ppp );

  nrb_clear( &tmp2 );
  nrb_init( src->pf_u.pf_k, src->pf_u.pf_nt + r, 0, 1, &tmp2 );
  nrb_allocateknots(&tmp2.pf_u);
  nrb_allocateknots(&tmp2.pf_v);
  tmp1.pf_v.pf_kk->knots[0] = 0.0;
  nrb_allocatepts( src->pf_u.pf_n + r, &tmp2.pf_ppp );

  /* init dest */
  nrb_copy( src, dest );

  /* Clamp all u vertice vectors */
  for( j=0; j < src->pf_v.pf_n; j++ ) 
    {

      /* Insert knots equal to u(p) */
      u = src->pf_u.pf_kk->knots[p];
      InsertKnot( &src->pf_ppp->pts[ (j*src->pf_u.pf_n) ], 
		  src->pf_u.pf_kk->knots, src->pf_u.pf_n,
		  src->pf_u.pf_k, tmp1.pf_ppp->pts, tmp1.pf_u.pf_kk->knots, 
		  u, 1, p, p );


      /* Insert knots equal to u(m-p) */
      u = src->pf_u.pf_kk->knots[ (m-1) - p ];
      InsertKnot( tmp1.pf_ppp->pts, tmp1.pf_u.pf_kk->knots, src->pf_u.pf_n+p,
		  src->pf_u.pf_k, tmp2.pf_ppp->pts, tmp2.pf_u.pf_kk->knots
		  , u, 1, p, (m-1)-p + p );
      /* The extra +p is because of in tmp1 already p knots
	 have been inserted */

      /* the vertice vector of tmp2 contains the vertices we need plus
	 2*p vertices we want to discard. 
      */
      /* Now discard knots 0..p-1 and knots (r + (m-1)-p + 1) .. m-1 
	 and discard vertices 0..p-1 and vertices (r + (n-1)-p + 1) .. n-1*/
      /* That is, do not copy them to dest. */

      for( i=0; i < dest->pf_u.pf_n; i++ ) {
	dest->pf_ppp->pts[i + (j*src->pf_u.pf_n) ] = tmp2.pf_ppp->pts[i + p];
      }
    }

  /* the knot vector of tmp2 contains the knots we need plus
	 2*p knots we want to discard.
  */
  /*
  ** While copying the knots, we translate them so that they
  ** will start at 0.0 
  */
  a = tmp2.pf_u.pf_kk->knots[p];
  for( i=0; i < dest->pf_u.pf_nt; i++ ) {
    dest->pf_u.pf_kk->knots[i] = tmp2.pf_u.pf_kk->knots[i + p] - a;
  }

  /* Deallocate tmp1 and tmp2 memory */
  nrb_deallocateknots( &tmp2.pf_u );
  nrb_deallocateknots( &tmp2.pf_v );
  nrb_deallocatepts( &tmp2.pf_ppp );

  nrb_deallocateknots( &tmp1.pf_u );
  nrb_deallocateknots( &tmp1.pf_v );
  nrb_deallocatepts( &tmp1.pf_ppp );
/*  nrb_dump( stderr, dest );*/
}


void InsertKnot( Ppoint4 * ptsSrc, Pfloat * knotsSrc, int np, int order, 
		 Ppoint4 * ptsDest, Pfloat * knotsDest,
		 double u, int s, int r, int k )
/* 
** Inserts knot in the u direction.
** PR NURBS is assumed to be a curve!
** This algorithm has been adapted from:
**  Piegl & Tiller, The NURBS Book, p. 151
**
**  u = new knot value
**  s = multiplicity of knot
**  r = number of times to insert knot
**  k = index in knot vector of knot after which knot has to be
**      inserted
**
** Locals:
**  np = number of points of src
**  p  = degree
**
**  nq = number of points after insertion
**  mp = number of knots of src
** 
** NOTE: the caller should allocate *all* memory for dest, that
** includes enough memory for the points and the knots!
*/
{
  int mp, nq/*, np*/, p;
  int i, j, L;
  double alpha;
  Ppoint4 Rw[ 10 ];
  /* Actually, the size of Rw should be equal to the order (=p+1)
  ** of the NURBS curve, but 10 is ok I think.
  */

/*  np = src->pf_u.pf_n;
    p = src->pf_u.pf_k - 1;*/
  p = order - 1;

  mp = np+p+1; /*src->pf_u.pf_nt;*/
  nq = np+r; /*src->pf_u.pf_n + r;*/

  /* Load new knot vector */
  for( i=0; i<=k; i++ ) {
    knotsDest[i] = knotsSrc[i];
/*    dest->pf_u.pf_kk->knots[i] = src->pf_u.pf_kk->knots[i];*/
  }
  for( i=1; i<=r; i++ ) {
    knotsDest[ k+i ] = u;
/*    dest->pf_u.pf_kk->knots[ k+i ] = u;*/
  }
  for( i=k+1; i<=mp; i++ ) {
    knotsDest[ i+r ] = knotsSrc[i];
/*    dest->pf_u.pf_kk->knots[ i+r ] = src->pf_u.pf_kk->knots[i];*/
  }

  /* Save unaltered control points */
  for( i=0; i<=k-p; i++ ) {
    ptsDest[i] = ptsSrc[i];
/*    dest->pf_ppp->pts[i] = src->pf_ppp->pts[i];*/
  }
  for( i=k; i<=np; i++ ) {
    ptsDest[ i+r ] = ptsSrc[i];
/*    dest->pf_ppp->pts[ i+r ] = src->pf_ppp->pts[i];*/
  }
  for( i=0; i<=p-s; i++ ) {
    Rw[ i ] = ptsSrc[ k-p+i ];
/*    Rw[ i ] = src->pf_ppp->pts[ k-p+i ];*/
  }

  /* Insert the knot r times */
  for( j=1; j<=r; j++ ) 
    {
      L = k-p+j;
      for( i=0; i<=p-j-s; i++ )
	{
	  alpha = (u - knotsSrc[ L+i ]) /
	    (knotsSrc[ i+k+1 ] - knotsSrc[ L+i ]);
/*	  alpha = (u - src->pf_u.pf_kk->knots[ L+i ]) /
	    (src->pf_u.pf_kk->knots[ i+k+1 ] - src->pf_u.pf_kk->knots[ L+i ]);*/

	  Rw[ i ].x = alpha * Rw[ i+1 ].x + (1.0 - alpha) * Rw[ i ].x;
	  Rw[ i ].y = alpha * Rw[ i+1 ].y + (1.0 - alpha) * Rw[ i ].y;
	  Rw[ i ].z = alpha * Rw[ i+1 ].z + (1.0 - alpha) * Rw[ i ].z;
	  Rw[ i ].w = alpha * Rw[ i+1 ].w + (1.0 - alpha) * Rw[ i ].w;
	}
      ptsDest[ L ] = Rw[ 0 ];
      ptsDest[ k+r-j ] = Rw[ p-j ];
/*      dest->pf_ppp->pts[ L ] = Rw[ 0 ];
        dest->pf_ppp->pts[ k+r-j ] = Rw[ p-j ];*/
    }

  /* Load remaining control points */
  for( i=L+1; i<k; i++ ) {
    ptsDest[ i ] = Rw[ i-L ];
/*    dest->pf_ppp->pts[ i ] = Rw[ i-L ];*/
  }
}

void RefineCurve(PR_nurb * src, PR_nurb * dest, PR_knots * newKnots )
/*
** RefineCurve refines the curve src using the knot vector newKnots.
** The result is placed into dest.
** The caller should allocate memory for dest
*/
{
  int i, j, k;
  double u;
  PR_nurb temp1, temp2;

  nrb_copy( src, &temp1 );

  i = 0;
  j = 0;
  k = -1; /* index of knot after which a knot will be inserted */
  /* I think this algorithm does not work for the special case
     that knots need to be inserted at the front of the knot
     vector
     */
  while (i < src->pf_u.pf_nt) 
    {
      /* While there are knots in newKnots which are smaller
      ** than the current knot in the src knot vector, insert
      ** these knots
      */
    u = newKnots->knots[j]; 
    if ( u < src->pf_u.pf_kk->knots[i] ) {
      assert( k >= 0 ); 
      /* just in case the special case may occur */

      nrb_clear( dest );
      nrb_init( src->pf_u.pf_k, src->pf_u.pf_nt + 1, 0, 1, dest );
      nrb_allocateknots(&dest->pf_u);
      nrb_allocateknots(&dest->pf_v);
      dest->pf_v.pf_kk->knots[0] = 0.0;
      nrb_allocatepts( src->pf_u.pf_n + 1, &dest->pf_ppp );

      InsertKnot( temp1.pf_ppp->pts, temp1.pf_u.pf_kk->knots,
		  temp1.pf_u.pf_n, temp1.pf_u.pf_k,
		  temp2.pf_ppp->pts, temp2.pf_u.pf_kk->knots,
		  u, 0, 1, k ); 
      nrb_copy( &temp2, &temp1 );

      j++;
    } else if ( u == src->pf_u.pf_kk->knots[i] ) {
      /* knot is already in the knot vector */
      i++;
      j++;
    } else {    /* u > src->pf_u.pf_kk->knots[i] */
      /* Advance to next knot in src */
      i++;
    }
    k++;
      
  } 

  nrb_copy( &temp1, dest );
}

