/*******************************************************************
  Conversion library - 
     Contains functions to convert objects to and from the NURBS 
     procedure library format to SoftImage SAAPHIRE

     Version 1.0
 ******************************************************************
  Written by: M.J. Evers
      Created:       18/7/1996
      Last Modified: 25/7/1996, 13/8/96, 15/8/96, 20/8/96,
      21/8/96, 29/8/96

 ******************************************************************/

#include "convert.h"
#include <assert.h>


void convertNURBSToSI( PR_nurb * PR_src, SAA_Elem * SI_dest, 
		       SAA_Scene * scene )
{

/* The following variables have been introduced to pass
   the NURBS information to SAAPHIRE
*/
  int degree [2];
  SAA_Boolean closed [2];
  int nbCtrlVertices [2], nbVertices;
  SAA_DVector * ctrlVertices;
  int nbKnotsU, nbKnotsV;
  double * knotsU, * knotsV;

  int i;
  double a;

  /* Check if it's a surface or a curve. The NURBS procedure
     library uses the same data structure for curves and 
     surfaces, but SAAPHIRE has different objects for them,
  */
  if (PR_src->pf_v.pf_k > 0) {
    /* It's a surface */

    /* Copy the data across, first U */

    degree[0] = PR_src->pf_u.pf_k - 1;    
       /* -1 to convert from order to degree */
    nbCtrlVertices[0] = PR_src->pf_u.pf_n;
    closed[0] = FALSE;    /* assumed FALSE */
 
    degree[1] = PR_src->pf_v.pf_k - 1;    
       /* -1 to convert from order to degree */
    nbCtrlVertices[1] = PR_src->pf_v.pf_n;
    closed[1] = FALSE;    /* assumed FALSE */

    nbVertices = nbCtrlVertices[0] * nbCtrlVertices[1];

#ifdef __cplusplus
    ctrlVertices = new SAA_DVector [ nbVertices  ];
#else
    ctrlVertices =  (SAA_DVector *) 
      calloc( nbVertices, sizeof(SAA_DVector) );
#endif
    assert ( ctrlVertices != NULL );

/* 
   I assume here that the ordering of the points is the same
   for SI and the NURBS procedure library

   This turned out to be true.

   There is a bug in the SDK with regard to the weight:
   to correct for this, we divide x, y, z by the weight.
*/
    for( i = 0; i < nbVertices; i++ )
      {
	a = PR_src->pf_ppp->pts[i].w;

	ctrlVertices[ i ].x = PR_src->pf_ppp->pts[i].x / a;
	ctrlVertices[ i ].y = PR_src->pf_ppp->pts[i].y / a;
	ctrlVertices[ i ].z = PR_src->pf_ppp->pts[i].z / a;
	ctrlVertices[ i ].w = PR_src->pf_ppp->pts[i].w;
      }
   
    /* Next, we create the SI NURBS through SAAPHIRE */
    SAA_nurbsSurfaceCreate( scene, degree, SAA_PARAM_NON_UNIFORM, 
			    closed, nbCtrlVertices, ctrlVertices, 
			    SI_dest );

    /* The next thing to do is to get the knot vector and pass 
       it to SI using the SAA_nurbsSurfaceSetKnots() function
    */

    /* First, get the nr of knots in the u and v direction */
    SAA_nurbsSurfaceGetNbKnots( scene, SI_dest, &nbKnotsU, &nbKnotsV );

    /* Allocate memory for our knot vectors */
#ifdef __cplusplus
    knotsU = new double [ nbKnotsU ];
    knotsV = new double [ nbKnotsV ];
#else
    knotsU = (double *) calloc( nbKnotsU, sizeof(double) );
    knotsV = (double *) calloc( nbKnotsV, sizeof(double) );
#endif
    assert( (knotsU != NULL && knotsV != NULL) );

    /* 
    ** Copy the knots to knotsU and  knotsV
    ** I don't copy the first and last knot from the NURBS procedure
    ** library knots vector 
    */

    for(i=0; i<nbKnotsU; i++)
      knotsU[i] = PR_src->pf_u.pf_kk->knots[i+1];

    for(i=0; i<nbKnotsV; i++)
      knotsV[i] = PR_src->pf_v.pf_kk->knots[i+1];

    /* 
    ** Hypothesis: I think SI does not store the first and last knot, but
    **             assumes these to be equal to the second resp. 
    **	           second last knot (which is generally true for NURBS)
    ** This worked out correctly.
    */
  
    SAA_nurbsSurfaceSetKnots( scene, SI_dest, SAA_GEOM_ORIGINAL,	       
			      -1 /* dummy value */, nbKnotsU, nbKnotsV,
			      knotsU, knotsV );

    /* Clean up */
#ifdef __cplusplus
    delete [] knotsV;
    delete [] knotsU;
    delete [] ctrlVertices;
#else
    free(knotsV);
    free(knotsU);
    free(ctrlVertices); 
#endif 

  } else {
    /* It's a curve */

    /* Copy the data across */

    degree[0] = PR_src->pf_u.pf_k - 1;    
    /* -1 to convert from order to degree */
    nbVertices = PR_src->pf_u.pf_n;
    closed[0] = FALSE;
 
#ifdef __cplusplus
    ctrlVertices =  new SAA_DVector [ nbVertices ];
#else
    ctrlVertices =  (SAA_DVector *) 
	  	    calloc( nbVertices ,sizeof(SAA_DVector) );
#endif
    assert( ctrlVertices != NULL ); 

/* 
** I assume here that the ordering of the points is the same
** for SI and the NURBS procedure library
**
** This turned out to be true.
**
** There is a bug in the SDK with regard to the weight:
** to correct for this, we divide x, y, z by the weight.
*/
    for( i = 0; i < nbVertices; i++ )
      {
	a = PR_src->pf_ppp->pts[i].w;

	ctrlVertices[ i ].x = PR_src->pf_ppp->pts[i].x / a;
	ctrlVertices[ i ].y = PR_src->pf_ppp->pts[i].y / a;
	ctrlVertices[ i ].z = PR_src->pf_ppp->pts[i].z / a;
	ctrlVertices[ i ].w = PR_src->pf_ppp->pts[i].w;
      }
   
    /* Next, we create the SI NURBS through SAAPHIRE */
    SAA_nurbsCurveCreate( scene, degree[0], SAA_PARAM_NON_UNIFORM, 
			  closed[0], nbVertices, ctrlVertices, 
			  SI_dest );

    /* First, get the nr of knots in the u and v direction */
    SAA_nurbsCurveGetNbKnots( scene, SI_dest, &nbKnotsU );

    /* Allocate memory for our knot vectors */
#ifdef __cplusplus
    knotsU = new double [ nbKnotsU ];
#else
    knotsU = (double *) calloc( nbKnotsU, sizeof(double) );
#endif
    assert( knotsU != NULL );

    /* 
    ** Copy the knots to knotsU
    ** I don't copy the first and last knot from the NURBS procedure
    ** library knots vector 
    */
    for(i=0; i<nbKnotsU; i++)
      knotsU[i] = PR_src->pf_u.pf_kk->knots[i+1];

    /* 
    ** Hypothesis: I think SI does not store the first and last knot, but
    **             assumes these to be equal to the second resp. 
    **	           second last knot (which is generally true for NURBS)
    ** This worked out correctly.
    */
  
    SAA_nurbsCurveSetKnots( scene, SI_dest, SAA_GEOM_ORIGINAL,	       
	                    -1 /* dummy value */, nbKnotsU,
			    knotsU );

    /* Clean up */
#ifdef __cplusplus
    delete [] knotsU;
    delete [] ctrlVertices;
#else
    free(knotsU);
    free(ctrlVertices); 
#endif

  }
}

/*
** createPRNURBS creates a PR NURBS using all the information
** given and places it into PR_dest. The caller should allocate
** memory for PR_dest
*/
void createPRNURBS( PR_nurb * PR_dest, int orderU, int orderV, 
		    int nbVertices,
		    int nbKnotsU, int nbKnotsV, SAA_DVector * ctrlVertices, 
		    double * knotsU, double * knotsV )
{
  double a;
  int i;

  nrb_init( orderU, nbKnotsU, 
	    orderV, nbKnotsV, PR_dest );

  /* Allocate memory for the knot vectors and the control point vector */
  nrb_allocateknots(&PR_dest->pf_u);
  nrb_allocateknots(&PR_dest->pf_v);
  nrb_allocatepts( nbVertices, &PR_dest->pf_ppp );
    
  /* here also we have to correct for the bug, by 
     multiplying x, y, z with w */
  for( i = 0; i < nbVertices; i++ )
    {
      a = ctrlVertices[ i ].w;
      
      PR_dest->pf_ppp->pts[i].x = ctrlVertices[ i ].x * a;
      PR_dest->pf_ppp->pts[i].y = ctrlVertices[ i ].y * a;
      PR_dest->pf_ppp->pts[i].z = ctrlVertices[ i ].z * a;
      PR_dest->pf_ppp->pts[i].w = ctrlVertices[ i ].w;
    }

  /* copy the knots */
  for(i=0; i<nbKnotsU; i++)
    PR_dest->pf_u.pf_kk->knots[i] = knotsU[i];
  
  for(i=0; i<nbKnotsV; i++)
    PR_dest->pf_v.pf_kk->knots[i] = knotsV[i];

}

void convertCurveFromSI( SAA_Elem * SI_src, PR_nurb * PR_dest,
			 SAA_Scene * scene )
{
  int i, result;
  double a;
  int degreeU;
  int left, right, leftK, rightK, nbKnotsSrc;
  
  int nbVertU, nbVertices;
  SAA_DVector * ctrlVertices;

  int nbKnotsU;
  double * knotsU, knotsV;

  SAA_Boolean closed;
 
  /* get degree and nr of vertices */
  SAA_nurbsCurveGetDegree( scene, SI_src, &degreeU );
  SAA_modelGetNbVertices( scene, SI_src, &nbVertU );

  SAA_nurbsCurveGetClosed( scene, SI_src, &closed );
  if ( !closed ) { 
    /*
    ** Open curve - conversion is straightforward 
    */
    nbVertices = nbVertU;
    
#ifdef __cplusplus
    ctrlVertices = new SAA_DVector [ nbVertU ];
#else
    ctrlVertices = (SAA_DVector *) calloc( nbVertU, sizeof( SAA_DVector ) );
#endif
    assert( ctrlVertices != NULL);

    SAA_modelGetVertices( scene, SI_src, SAA_GEOM_ORIGINAL, 
			  -1 /* dummy value */, nbVertU, ctrlVertices );

    /* get knots */
    SAA_nurbsCurveGetNbKnots( scene, SI_src, &nbKnotsU );    
    nbKnotsU += 2;   /* +2 because of 2 'missing' knots in SI */

#ifdef __cplusplus
    knotsU = new double [ nbKnotsU ];
#else
    knotsU = (double *) calloc( nbKnotsU, sizeof( double ) );
#endif
    assert( knotsU != NULL );

    SAA_nurbsCurveGetKnots( scene, SI_src, SAA_GEOM_ORIGINAL,
			    -1 /* dummy value */, nbKnotsU-2, &knotsU[1] );

  } else {
    /* 
    ** Closed NURBS curve
    */

    /* nbVertU = # ctrl vertices in SI NURBS, nbVertices = # ctrl vertices
       in PR LIB NURBS */

    /*
    ** right and left are the number of points added to the left
    ** and the right of the vertices vector 
    */
    right = degreeU / 2;
    left = degreeU - right;
    nbVertices = nbVertU + left + right;

#ifdef __cplusplus
    ctrlVertices = new SAA_DVector [ nbVertices ];
#else
    ctrlVertices = (SAA_DVector *) calloc( nbVertices, 
					 sizeof( SAA_DVector ) );
#endif
    assert( ctrlVertices != NULL);

    SAA_modelGetVertices( scene, SI_src, SAA_GEOM_ORIGINAL, 
			  -1 /* dummy value */, nbVertU, &ctrlVertices[left] );

    /* get knots */
    SAA_nurbsCurveGetNbKnots( scene, SI_src, &nbKnotsSrc );

    /*
    ** rightK and leftK are the number of knots to be added left and
    ** right of the knot vector 
    */
    rightK = degreeU;
    leftK = degreeU;

    nbKnotsU = nbVertices + degreeU + 1;

#ifdef _cplusplus
    knotsU = new double [ nbKnotsU ];
#else
    knotsU = (double *) calloc( nbKnotsU, sizeof( double ));
#endif
    assert( (knotsU != NULL) );

    SAA_nurbsCurveGetKnots( scene, SI_src, SAA_GEOM_ORIGINAL,
			    -1 /* dummy value */, nbKnotsSrc, &knotsU[leftK] );

    /* Create additional vertices */
    for( i=0; i < left; i++ ) {
/*      assert( (i >= 0 && i<nbVertices && 
	       (nbVertU + i) >= 0 && (nbVertU + i) < nbVertices ) );*/
      ctrlVertices[ i ] = ctrlVertices[ (nbVertU + i) ];
      /* original index of source: (left + nbVertU - left + i) = (nbVertU + i) */
    }
    for( i=0; i < right; i++ ) {
/*      assert( ( (left + nbVertU + i) >= 0 && (left + nbVertU + i) < nbVertices
		&& (left + i) >= 0 && (left + i) < nbVertices ) );*/
      ctrlVertices[ left + nbVertU + i ] = ctrlVertices[ (left + i) ];
    }

    /* 
    ** Compute the new knots on the left
    ** we compute knots with indexes 1..leftK-1 
    */
    for( i=leftK - 1; i >= 1; i-- ) {
/*      assert( ( i>=0 && i<nbKnotsU && (i+1) >= 0 && (i+1) < nbKnotsU ) );
        assert( ( (nbKnotsSrc + i) >= 0 && (nbKnotsSrc + i) < nbKnotsU ) );
        assert( ( (nbKnotsSrc + i - 1) >= 0 && (nbKnotsSrc + i - 1) < nbKnotsU) );*/

      knotsU[ i ] = knotsU[ i+1 ] - 
	( knotsU[ nbKnotsSrc + i ] - 
	  knotsU[ nbKnotsSrc + i - 1] );
      /* 
      ** original indexes were:
      **  ( knotsU[ leftK + nbKnotsSrc - 1 - (leftK - 1 - i) ] - 
      **    knotsU[ leftK + nbKnotsSrc - 1 - (leftK - 1 - i) - 1] );
      ** This really needs some documentation
      */
    }

    /* 
    ** Compute the new knots on the right
    ** we compute knots with indexes 
    **  leftK+nbKnotsSrc..leftK+nbKnotsSrc+rightK-2 
    */
    for( i=0; i < (rightK - 1); i++ ) {
/*      assert( ( (leftK + nbKnotsSrc + i) >= 0 && (leftK + nbKnotsSrc + i) <
		nbKnotsU) );
        assert( ( (leftK + nbKnotsSrc + i-1) >= 0 &&
		(leftK + nbKnotsSrc + i - 1) < nbKnotsU ) );
        assert( ( (leftK + i + 1) >= 0 && (leftK + i + 1) < nbKnotsU ) );
        assert( ( (leftK + i) >= 0 && (leftK + i) < nbKnotsU ) );*/

      knotsU[ leftK + nbKnotsSrc + i ] = 
	knotsU[ leftK + nbKnotsSrc + i - 1 ] + 
	(knotsU[ leftK + i + 1] - knotsU[ leftK + i]);
      /* 
      ** original indexes were:
      ** knotsU[ (leftK - 1)+1 + nbKnotsSrc + i ] = 
      **   knotsU[ (leftK - 1)+1 + nbKnotsSrc + i - 1 ] + 
      **   (knotsU[ (leftK - 1)+1 + i + 1] - knotsU[ (leftK - 1)+1 + i]);
      ** This also really needs some documentation
      */
    }

    /* translate knots so that they are >= 0 */
    a = knotsU[1];
    for( i=1; i < nbKnotsU-1; i++ ) {
      assert( (i >= 0 && i < nbKnotsU ) );

      knotsU[i] -= a;
    }

    /* reset nbVertU to nbVertices */
/*    nbVertU = nbVertices;*/
  }

  knotsV = 0.0;

  /* make the first knot equal to the second and the last knot
     equal to the second last */
  knotsU[0] = knotsU[1];
  knotsU[nbKnotsU-1] = knotsU[nbKnotsU-2];

  /* 
  ** Now that we've retrieved the info from the SI NURBS,
  ** create the PR LIB NURBS and copy the info.
  */
  createPRNURBS( PR_dest, degreeU+1, 0, nbVertices, 
		 nbKnotsU, 1, ctrlVertices, knotsU,
		 &knotsV );

  /* cleanup */
#ifdef __cplusplus
  delete [] knotsU;
  delete [] ctrlVertices;
#else
  free( knotsU );
  free( ctrlVertices );
#endif

}

SI_Error convertSurfaceFromSI( SAA_Elem * SI_src, PR_nurb * PR_dest,
			       SAA_Scene * scene )
{
  int i, j, result;
  int degreeU, degreeV;
  int leftU, rightU, leftUK, rightUK, leftV, rightV, leftVK, rightVK;
  int nbKnotsUSrc, nbKnotsVSrc;
  double a;
  
  int nbVertU, nbVertV, nbVertices, nbVerticesSrc, nbVertUSrc, nbVertVSrc;
  SAA_DVector * ctrlVertices, * ctrlVerticesSrc;

  int nbKnotsU, nbKnotsV;
  double * knotsU, * knotsV;

  SAA_Boolean closedU, closedV;

  /* get degree and control vertices */
  SAA_nurbsSurfaceGetNbVertices( scene, SI_src, &nbVertU, &nbVertV );
  SAA_nurbsSurfaceGetDegree( scene, SI_src, &degreeU, &degreeV );

  SAA_nurbsSurfaceGetClosed( scene, SI_src, &closedU, &closedV );
  /* do something with it */
  if ( closedU && closedV ) 
    /*
    ** Surface is closed in u direction and closed in v direction
    */
    {
      /*
      ** right and left are the number of points added to the left
      ** and the right of the vertices vector 
      */
      nbVertUSrc = nbVertU;
      rightU = degreeU / 2;
      leftU = degreeU - rightU;
      nbVertU = nbVertUSrc + leftU + rightU;

      nbVertVSrc = nbVertV;
      rightV = degreeV / 2;
      leftV = degreeV - rightV;
      nbVertV = nbVertVSrc + leftV + rightV;

      nbVerticesSrc = nbVertUSrc * nbVertVSrc;
      nbVertices = nbVertU * nbVertV;

#ifdef __cplusplus
      ctrlVerticesSrc = new SAA_DVector [ nbVerticesSrc ]; 
      ctrlVertices = new SAA_DVector [ nbVertices ]; 
#else
      ctrlVerticesSrc = (SAA_DVector *) calloc( nbVerticesSrc, 
					     sizeof( SAA_DVector ) );
      ctrlVertices = (SAA_DVector *) calloc( nbVertices, 
					     sizeof( SAA_DVector ) );
#endif
      assert( ctrlVertices != NULL && ctrlVerticesSrc != NULL);
      
      SAA_modelGetVertices( scene, SI_src, SAA_GEOM_ORIGINAL, 
			    -1 /* dummy value */, nbVerticesSrc,
			    ctrlVerticesSrc );

      /* do the knot vectors */
      SAA_nurbsSurfaceGetNbKnots( scene, SI_src, &nbKnotsUSrc, &nbKnotsVSrc );

      /*
      ** rightK and leftK are the number of knots to be added left and
      ** right of the knot vector 
      */
      rightUK = degreeU;
      leftUK = degreeU;
      rightVK = degreeV;
      leftVK = degreeV;

      nbKnotsU = nbVertU + degreeU + 1;
      nbKnotsV = nbVertV + degreeV + 1;

#ifdef __cplusplus
      knotsU = new double [ nbKnotsU ];
      knotsV = new double [ nbKnotsV ];
#else
      knotsU = (double *) calloc( nbKnotsU, sizeof( double ) );
      knotsV = (double *) calloc( nbKnotsV, sizeof( double ) );
#endif
      assert( (knotsU != NULL && knotsV != NULL ) );

      SAA_nurbsSurfaceGetKnots( scene, SI_src, SAA_GEOM_ORIGINAL, 
				-1 /* dummy value */, nbKnotsUSrc, nbKnotsVSrc,
				&knotsU[leftUK], &knotsV[leftVK] );

      /* Copy all the points and create additional vertices in the
	 u-direction
      */
      for( j=leftV; j < leftV+nbVertVSrc; j++ ) {
	for( i=leftU; i < leftU+nbVertUSrc; i++ ) {
	  ctrlVertices[ i + ( j * nbVertU) ] = 
	    ctrlVerticesSrc[ (i - leftU) + ( (j - leftV) * nbVertUSrc) ];
	}
	/* create additional points */
	for( i=0; i < leftU; i++ ) {
	  ctrlVertices[ i + (j*nbVertU) ] = 
	    ctrlVertices[ (nbVertUSrc + i) + ( j*nbVertU) ];
	}
	for( i=0; i < rightU; i++ ) {
	  ctrlVertices[ leftU + nbVertUSrc + i + (j*nbVertU) ] = 
	    ctrlVertices[ (leftU + i) + (j*nbVertU) ];
	}
      }


      /* 
      ** And create the additional points in the v direction
      */

      for( j=0; j < nbVertU; j++ ) {

	/* create additional points */
	for( i=0; i < leftV; i++ ) {
	  ctrlVertices[ i*nbVertU + j ] = 
	    ctrlVertices[ (nbVertVSrc + i)*nbVertU + j ];
	}

	for( i=0; i < rightV; i++ ) {
	  ctrlVertices[ (leftV + nbVertVSrc + i)*nbVertU + j ] = 
	    ctrlVertices[ (leftV + i)*nbVertU + j ];
	}
      }

      /* 
      ** Compute the new u knots on the left
      ** we compute knots with indexes 1..leftK-1 
      */
      for( i=leftUK - 1; i >= 1; i-- ) {
	knotsU[ i ] = knotsU[ i+1 ] - 
	  ( knotsU[ nbKnotsUSrc + i ] - 
	    knotsU[ nbKnotsUSrc + i - 1] );
      }

      for( i=leftVK - 1; i >= 1; i-- ) {
	knotsV[ i ] = knotsV[ i+1 ] - 
	  ( knotsV[ nbKnotsVSrc + i ] - 
	    knotsV[ nbKnotsVSrc + i - 1] );
      }

      /* 
      ** Compute the new u knots on the right
      ** we compute knots with indexes 
      **  leftK+nbKnotsSrc..leftK+nbKnotsSrc+rightK-2 
      */
      for( i=0; i < (rightUK - 1); i++ ) {
	knotsU[ leftUK + nbKnotsUSrc + i ] = 
	  knotsU[ leftUK + nbKnotsUSrc + i - 1 ] + 
	  (knotsU[ leftUK + i + 1] - knotsU[ leftUK + i]);
      }

      for( i=0; i < (rightVK - 1); i++ ) {
	knotsV[ leftVK + nbKnotsVSrc + i ] = 
	  knotsV[ leftVK + nbKnotsVSrc + i - 1 ] + 
	  (knotsV[ leftVK + i + 1] - knotsV[ leftVK + i]);
      }

      /* translate knots so that they are >= 0 */
      a = knotsU[1];
      for( i=1; i < nbKnotsU-1; i++ ) {
	knotsU[i] -= a;
      }

      a = knotsV[1];
      for( i=1; i < nbKnotsV-1; i++ ) {
	knotsV[i] -= a;
      }

#ifdef __cplusplus
      delete [] ctrlVerticesSrc; /* 19/8/98 MJE: forgot a ; */
#else
      free( ctrlVerticesSrc );
#endif

      result = SI_SUCCESS;
    } 
  else if ( !closedU && !closedV ) 
    /*
    ** Surface is open in u direction and open in v direction
    */
    {
      nbVertices = nbVertU * nbVertV;
#ifdef __cplusplus
      ctrlVertices = new SAA_DVector [ nbVertices ]; 
#else
      ctrlVertices = (SAA_DVector *) calloc( nbVertices, 
					     sizeof( SAA_DVector ) );
#endif
      assert( ctrlVertices != NULL );
      
      SAA_modelGetVertices( scene, SI_src, SAA_GEOM_ORIGINAL, 
			    -1 /* dummy value */, nbVertices,
			    ctrlVertices );

     /* do the knot vectors */
      SAA_nurbsSurfaceGetNbKnots( scene, SI_src, &nbKnotsU, &nbKnotsV );
      nbKnotsU += 2; nbKnotsV += 2;  /* +2 because of 2 'missing' knots in SI */ 

#ifdef __cplusplus
      knotsU = new double [ nbKnotsU ];
      knotsV = new double [ nbKnotsV ];
#else
      knotsU = (double *) calloc( nbKnotsU, sizeof( double ) );
      knotsV = (double *) calloc( nbKnotsV, sizeof( double ) );
#endif
      assert( (knotsU != NULL && knotsV != NULL ) );

      SAA_nurbsSurfaceGetKnots( scene, SI_src, SAA_GEOM_ORIGINAL, 
				-1 /* dummy value */, nbKnotsU-2, nbKnotsV-2,
				&knotsU[1], &knotsV[1] );

      result = SI_SUCCESS;
    } 
  else if ( closedU && !closedV ) 
    /*
    ** Surface is closed in u direction and open in v direction
    */
    {
      /*
      ** right and left are the number of points added to the left
      ** and the right of the vertices vector 
      */
      nbVertUSrc = nbVertU;
      rightU = degreeU / 2;
      leftU = degreeU - rightU;
      nbVertU = nbVertUSrc + leftU + rightU;

      nbVerticesSrc = nbVertUSrc * nbVertV;
      nbVertices = nbVertU * nbVertV;
#ifdef __cplusplus
      ctrlVerticesSrc = new SAA_DVector [ nbVerticesSrc ]; 
      ctrlVertices = new SAA_DVector [ nbVertices ]; 
#else
      ctrlVerticesSrc = (SAA_DVector *) calloc( nbVerticesSrc, 
					     sizeof( SAA_DVector ) );
      ctrlVertices = (SAA_DVector *) calloc( nbVertices, 
					     sizeof( SAA_DVector ) );
#endif
      assert( ctrlVertices != NULL && ctrlVerticesSrc != NULL);
      
      SAA_modelGetVertices( scene, SI_src, SAA_GEOM_ORIGINAL, 
			    -1 /* dummy value */, nbVerticesSrc,
			    ctrlVerticesSrc );

      /* do the knot vectors */
      SAA_nurbsSurfaceGetNbKnots( scene, SI_src, &nbKnotsUSrc, &nbKnotsV );
      nbKnotsV += 2;  /* +2 because of 2 'missing' knots in SI */ 

      /*
      ** rightK and leftK are the number of knots to be added left and
      ** right of the knot vector 
      */
      rightUK = degreeU;
      leftUK = degreeU;

      nbKnotsU = nbVertU + degreeU + 1;

#ifdef __cplusplus
      knotsU = new double [ nbKnotsU ];
      knotsV = new double [ nbKnotsV ];
#else
      knotsU = (double *) calloc( nbKnotsU, sizeof( double ) );
      knotsV = (double *) calloc( nbKnotsV, sizeof( double ) );
#endif
      assert( (knotsU != NULL && knotsV != NULL ) );

      SAA_nurbsSurfaceGetKnots( scene, SI_src, SAA_GEOM_ORIGINAL, 
				-1 /* dummy value */, nbKnotsUSrc, nbKnotsV-2,
				&knotsU[leftUK], &knotsV[1] );

      /* Copy all the points and create additional vertices in the
	 u-direction
      */
      for( j=0; j < nbVertV; j++ ) {
	for( i=leftU; i < leftU+nbVertUSrc; i++ ) {
	  ctrlVertices[ i + (j*nbVertU) ] = 
	    ctrlVerticesSrc[ (i - leftU) + (j*nbVertUSrc) ];
	}
	/* create additional points */
	for( i=0; i < leftU; i++ ) {
	  ctrlVertices[ i + (j*nbVertU) ] = 
	    ctrlVertices[ (nbVertUSrc + i) + (j*nbVertU) ];
	  /* original index of source: (left + nbVertU - left + i) = (nbVertU + i) */
	}
	for( i=0; i < rightU; i++ ) {
	  ctrlVertices[ leftU + nbVertUSrc + i + (j*nbVertU) ] = 
	    ctrlVertices[ (leftU + i) + (j*nbVertU) ];
	}
      }

      /* 
      ** Compute the new knots on the left
      ** we compute knots with indexes 1..leftK-1 
      */
      for( i=leftUK - 1; i >= 1; i-- ) {
	knotsU[ i ] = knotsU[ i+1 ] - 
	  ( knotsU[ nbKnotsUSrc + i ] - 
	    knotsU[ nbKnotsUSrc + i - 1] );
	/* 
	** original indexes were:
	**  ( knotsU[ leftK + nbKnotsUSrc - 1 - (leftK - 1 - i) ] - 
	**    knotsU[ leftK + nbKnotsUSrc - 1 - (leftK - 1 - i) - 1] );
	** This really needs some documentation
	*/
      }

      /* 
      ** Compute the new knots on the right
      ** we compute knots with indexes 
      **  leftK+nbKnotsSrc..leftK+nbKnotsSrc+rightK-2 
      */
      for( i=0; i < (rightUK - 1); i++ ) {
	knotsU[ leftUK + nbKnotsUSrc + i ] = 
	  knotsU[ leftUK + nbKnotsUSrc + i - 1 ] + 
	  (knotsU[ leftUK + i + 1] - knotsU[ leftUK + i]);
	/* 
	** original indexes were:
	** knotsU[ (leftK - 1)+1 + nbKnotsUSrc + i ] = 
	**   knotsU[ (leftK - 1)+1 + nbKnotsUSrc + i - 1 ] + 
	**   (knotsU[ (leftK - 1)+1 + i + 1] - knotsU[ (leftK - 1)+1 + i]);
	** This also really needs some documentation
	*/
      }

      /* translate knots so that they are >= 0 */
      a = knotsU[1];
      for( i=1; i < nbKnotsU-1; i++ ) {
	knotsU[i] -= a;
      }

#ifdef __cplusplus
      delete [] ctrlVerticesSrc; /* 19/8/98 MJE: forgot a ; */
#else
      free( ctrlVerticesSrc );
#endif

      result = SI_SUCCESS;
    } 
  else   /* !closed U && closedV */
    /*
    ** Surface is open in u direction and closed in v direction
    */
    {
      /*
      ** right and left are the number of points added to the left
      ** and the right of the vertices vector 
      */
      nbVertVSrc = nbVertV;
      rightV = degreeV / 2;
      leftV = degreeV - rightV;
      nbVertV = nbVertVSrc + leftV + rightV;

      nbVerticesSrc = nbVertU * nbVertVSrc;
      nbVertices = nbVertU * nbVertV;
#ifdef __cplusplus
      ctrlVerticesSrc = new SAA_DVector [ nbVerticesSrc ]; 
      ctrlVertices = new SAA_DVector [ nbVertices ]; 
#else
      ctrlVerticesSrc = (SAA_DVector *) calloc( nbVerticesSrc, 
					     sizeof( SAA_DVector ) );
      ctrlVertices = (SAA_DVector *) calloc( nbVertices, 
					     sizeof( SAA_DVector ) );
#endif
      assert( ctrlVertices != NULL && ctrlVerticesSrc != NULL);
      
      SAA_modelGetVertices( scene, SI_src, SAA_GEOM_ORIGINAL, 
			    -1 /* dummy value */, nbVerticesSrc,
			    ctrlVerticesSrc );

      /* do the knot vectors */
      SAA_nurbsSurfaceGetNbKnots( scene, SI_src, &nbKnotsU, &nbKnotsVSrc );
      nbKnotsU += 2;  /* +2 because of 2 'missing' knots in SI */ 

      /*
      ** rightK and leftK are the number of knots to be added left and
      ** right of the knot vector 
      */
      rightVK = degreeV;
      leftVK = degreeV;

      nbKnotsV = nbVertV + degreeV + 1;

#ifdef __cplusplus
      knotsU = new double [ nbKnotsU ];
      knotsV = new double [ nbKnotsV ];
#else
      knotsU = (double *) calloc( nbKnotsU, sizeof( double ) );
      knotsV = (double *) calloc( nbKnotsV, sizeof( double ) );
#endif
      assert( (knotsU != NULL && knotsV != NULL ) );

      SAA_nurbsSurfaceGetKnots( scene, SI_src, SAA_GEOM_ORIGINAL, 
				-1 /* dummy value */, nbKnotsU-2, nbKnotsVSrc,
				&knotsU[1], &knotsV[leftVK] );

      /* Copy all the points and create additional vertices in the
	 u-direction
      */
      for( j=0; j < nbVertU; j++ ) {
	for( i=leftV; i < leftV+nbVertVSrc; i++ ) {
	  ctrlVertices[ i*nbVertU + j ] = 
	    ctrlVerticesSrc[ (i - leftV)*nbVertU + j ];
	}
	/* create additional points */
	for( i=0; i < leftV; i++ ) {
	  ctrlVertices[ i*nbVertU + j ] = 
	    ctrlVertices[ (nbVertVSrc + i)*nbVertU + j ];
	}
	for( i=0; i < rightV; i++ ) {
	  ctrlVertices[ (leftV + nbVertVSrc + i)*nbVertU + j ] = 
	    ctrlVertices[ (leftV + i)*nbVertU + j ];
	}
      }

      /* 
      ** Compute the new knots on the left
      ** we compute knots with indexes 1..leftK-1 
      */
      for( i=leftVK - 1; i >= 1; i-- ) {
	knotsV[ i ] = knotsV[ i+1 ] - 
	  ( knotsV[ nbKnotsVSrc + i ] - 
	    knotsV[ nbKnotsVSrc + i - 1] );
	/* 
	** original indexes were:
	**  ( knotsV[ leftK + nbKnotsVSrc - 1 - (leftK - 1 - i) ] - 
	**    knotsV[ leftK + nbKnotsVSrc - 1 - (leftK - 1 - i) - 1] );
	** This really needs some documentation
	*/
      }

      /* 
      ** Compute the new knots on the right
      ** we compute knots with indexes 
      **  leftK+nbKnotsSrc..leftK+nbKnotsSrc+rightK-2 
      */
      for( i=0; i < (rightVK - 1); i++ ) {
	knotsV[ leftVK + nbKnotsVSrc + i ] = 
	  knotsV[ leftVK + nbKnotsVSrc + i - 1 ] + 
	  (knotsV[ leftVK + i + 1] - knotsV[ leftVK + i]);
	/* 
	** original indexes were:
	** knotsV[ (leftK - 1)+1 + nbKnotsVSrc + i ] = 
	**   knotsV[ (leftK - 1)+1 + nbKnotsVSrc + i - 1 ] + 
	**   (knotsV[ (leftK - 1)+1 + i + 1] - knotsV[ (leftK - 1)+1 + i]);
	** This also really needs some documentation
	*/
      }

      /* translate knots so that they are >= 0 */
      a = knotsV[1];
      for( i=1; i < nbKnotsV-1; i++ ) {
	knotsV[i] -= a;
      }

#ifdef __cplusplus
      delete [] ctrlVerticesSrc;
#else
      free( ctrlVerticesSrc );
#endif

      result = SI_SUCCESS;
    }

  if ( result == SI_SUCCESS ) 
    {
  
      /* make the first knot equal to the second and the last knot
	 equal to the second last */
      knotsU[0] = knotsU[1];
      knotsU[nbKnotsU-1] = knotsU[nbKnotsU-2];

      knotsV[0] = knotsV[1];
      knotsV[nbKnotsV-1] = knotsV[nbKnotsV-2];

      /* 
      ** Now that we've retrieved the info from the SI NURBS,
      ** create the PR LIB NURBS and copy the info.
      */
      createPRNURBS( PR_dest, degreeU+1, degreeV+1, nbVertices, 
		     nbKnotsU, nbKnotsV, ctrlVertices, knotsU,
		     knotsV );

      /* cleanup */
#ifdef __cplusplus
      delete [] knotsV;
      delete [] knotsU;
      delete [] ctrlVertices;
#else
      free( knotsV );
      free( knotsU );
      free( ctrlVertices );
#endif
    }

  return result;
}

void convertNURBSFromSI( SAA_Elem * SI_src, PR_nurb * PR_dest,
			 SAA_Scene * scene )
{
  SAA_ModelType mdlType;

  /* Check if it's a surface or a curve. The NURBS procedure
     library uses the same data structure for curves and 
     surfaces, but SAAPHIRE has different objects for them,
  */

  SAA_modelGetType( scene, SI_src, &mdlType ); 

  if ( mdlType == SAA_MNSRF ) {
    /* It's a surface */

    convertSurfaceFromSI( SI_src, PR_dest, scene );

  } else if ( mdlType == SAA_MNCRV ) {
    /* It's a curve */

    convertCurveFromSI( SI_src, PR_dest, scene );
  }
}


void copyPoints( PR_nurb * src, SAA_Elem * dest, SAA_Scene * scene, 
		 SAA_Boolean closeU, SAA_Boolean closeV )
/* Copies the vertices and knots from the PR LIB NURBS src
   to the given SI NURBS surface dest. 
   The parameters closeU and closeV specify if the SI NURBS should be closed in
   the u resp. v direction.

   Note that copyPoints only works for NURBS surfaces.

   Note that the copying only affects the local coordinates of 
   the vertices of an object. The coordinates of its center remain
   unchanged...
*/
{
  int i;
  double a;

  SAA_DVector * vertices;
  int nbVertices[2], nbVert;

  double * knotsU, * knotsV;
  int nbKnotsU, nbKnotsV;

  SAA_Boolean closed [2];
  int degree [2];

  SAA_ModelType mdlType;

  /* 
  ** Check the parameters
  */
  SAA_modelGetType( scene, dest, &mdlType ); 
  assert( mdlType == SAA_MNSRF );

  /*
  ** copy the control vertices 
  */
  nbVertices[0] = src->pf_u.pf_n;
  nbVertices[1] = src->pf_v.pf_n;
  nbVert = nbVertices[0] * nbVertices[1];

#ifdef __cplusplus
  vertices = new SAA_DVector [ nbVert ];
#else
  vertices = (SAA_DVector *) calloc( nbVert, 
				     sizeof( SAA_DVector ) );
#endif
  assert( vertices != NULL );

  for( i=0; i < nbVert; i++ ) {
    a = src->pf_ppp->pts[i].w;

    vertices[ i ].x = src->pf_ppp->pts[i].x / a;
    vertices[ i ].y = src->pf_ppp->pts[i].y / a;
    vertices[ i ].z = src->pf_ppp->pts[i].z / a;
    vertices[ i ].w = src->pf_ppp->pts[i].w;
  }
    
  /* copy the knots */
  nbKnotsU = src->pf_u.pf_nt - 2;
  nbKnotsV = src->pf_v.pf_nt - 2;

#ifdef __cplusplus
  knotsU = new double [ nbKnotsU ];
  knotsV = new double [ nbKnotsV ];
#else
  knotsU = (double *) calloc( nbKnotsU, sizeof( double ) );
  knotsV = (double *) calloc( nbKnotsV, sizeof( double ) );
#endif
  assert( (knotsU != NULL && knotsV != NULL) );
   
  /* I don't copy the first and last knot from the NURBS procedure
     library knots vector */
  for(i=0; i<nbKnotsU; i++)
    knotsU[i] = src->pf_u.pf_kk->knots[i+1];

  for(i=0; i<nbKnotsV; i++)
    knotsV[i] = src->pf_v.pf_kk->knots[i+1];

  degree[0] = src->pf_u.pf_k - 1;
  degree[1] = src->pf_v.pf_k - 1;

  closed[0] = FALSE;
  closed[1] = FALSE;

  /*
  ** Copy the vertices using the edit function for SI NURBS 
  */
  SAA_nurbsSurfaceEdit( scene, dest, degree, SAA_PARAM_NON_UNIFORM,
			closed, nbVertices, vertices );

  /*
  ** Copy the knots
  */
  SAA_nurbsSurfaceSetKnots( scene, dest, SAA_GEOM_ORIGINAL,
			    0, nbKnotsU, nbKnotsV, knotsU, knotsV );
  /*
  ** Now we close the surface in the specified directions
  */
  if (!(!closeU && !closeV)) {
    closed[0] = closeU;
    closed[1] = closeV;

    SAA_nurbsSurfaceEdit( scene, dest, degree, SAA_PARAM_NON_UNIFORM,
			  closed, nbVertices, vertices );

  }

#ifdef __cplusplus   
  delete [] knotsV;
  delete [] knotsU;
  delete [] vertices;
#else
  free( knotsV );
  free( knotsU );
  free( vertices);
#endif

}

/*
** nurbsSurfaceSetClosed() sets the 'closed' attribute of a
** NURBS surface in a specific scene. closedU and closedV 
** specify whether to open or close the surface in the U and
** V directions.
*/
void nurbsSurfaceSetClosed( SAA_Scene * scene, SAA_Elem * srf,
			    SAA_Boolean closedU, SAA_Boolean closedV )
{
  SAA_DVector * ctrlVertices;
  int nbVertices[2], nbVert;

  SAA_Boolean closed [2];
  int degree [2];

  SAA_ModelType mdlType;

  /* 
  ** Check the parameters
  */
  SAA_modelGetType( scene, srf, &mdlType ); 
  assert( mdlType == SAA_MNSRF );

  /* get degree */
  SAA_nurbsSurfaceGetDegree( scene, srf, &degree[0], &degree[1] );

  /* get control vertices */
  SAA_nurbsSurfaceGetNbVertices( scene, srf, &nbVertices[0], &nbVertices[1] );

  nbVert = nbVertices[0] * nbVertices[1];

#ifdef __cplusplus
  ctrlVertices = new SAA_DVector [ nbVert ]; 
#else
  ctrlVertices = (SAA_DVector *) calloc( nbVert, 
					 sizeof( SAA_DVector ) );
#endif
  assert( ctrlVertices != NULL );

  SAA_modelGetVertices( scene, srf, SAA_GEOM_ORIGINAL, 
			-1 /* dummy value */, nbVert,
			ctrlVertices );

  closed[0] = closedU;
  closed[1] = closedV;

  SAA_nurbsSurfaceEdit( scene, srf, degree, SAA_PARAM_NON_UNIFORM,
			closed, nbVertices, ctrlVertices );

#ifdef __cplusplus   
  delete [] ctrlVertices;
#else
  free( ctrlVertices);
#endif

}

void dirtyCopy( SAA_Scene * scene, SAA_Elem * srf, SAA_Elem * dest )
/* 
   This function makes a copy of the NURBS surface srf and returns
   it in dest. Caller should allocate memory for dest.
*/
{
  int i;
  double a;

  SAA_DVector * ctrlVertices;
  int nbVertices[2], nbVert;

  double * knotsU, * knotsV;
  int nbKnotsU, nbKnotsV;

  SAA_Boolean closed [2];
  int degree [2];

  SAA_ModelType mdlType;

  /* 
  ** Check the parameters
  */
  SAA_modelGetType( scene, srf, &mdlType ); 
  assert( mdlType == SAA_MNSRF );

  /* get degree */
  SAA_nurbsSurfaceGetDegree( scene, srf, &degree[0], &degree[1] );

  /* get closed */
  SAA_nurbsSurfaceGetClosed( scene, srf, &closed[0], &closed[1] );

  /* get control vertices */
  SAA_nurbsSurfaceGetNbVertices( scene, srf, &nbVertices[0], &nbVertices[1] );

  nbVert = nbVertices[0] * nbVertices[1];

#ifdef __cplusplus
  ctrlVertices = new SAA_DVector [ nbVert ]; 
#else
  ctrlVertices = (SAA_DVector *) calloc( nbVert, 
					 sizeof( SAA_DVector ) );
#endif
  assert( ctrlVertices != NULL );

  SAA_modelGetVertices( scene, srf, SAA_GEOM_ORIGINAL, 
			-1 /* dummy value */, nbVert,
			ctrlVertices );


  /* get the knots */
  SAA_nurbsSurfaceGetNbKnots( scene, srf, &nbKnotsU, &nbKnotsV );

#ifdef __cplusplus
  knotsU = new double [ nbKnotsU ];
  knotsV = new double [ nbKnotsV ];
#else
  knotsU = (double *) calloc( nbKnotsU, sizeof( double ) );
  knotsV = (double *) calloc( nbKnotsV, sizeof( double ) );
#endif

  SAA_nurbsSurfaceGetKnots( scene, srf, SAA_GEOM_ORIGINAL, 
			    -1 /* dummy value */, nbKnotsU, nbKnotsV,
			    knotsU, knotsV );

  /*
  ** Create new NURBS surface
  */
  SAA_nurbsSurfaceCreate( scene, degree, SAA_PARAM_NON_UNIFORM,
			  closed, nbVertices, ctrlVertices, dest );

  /*
  ** Copy the knots
  */
  SAA_nurbsSurfaceSetKnots( scene, dest, SAA_GEOM_ORIGINAL,
			    0, nbKnotsU, nbKnotsV, knotsU, knotsV );

#ifdef __cplusplus   
  delete [] knotsV;
  delete [] knotsU;
  delete [] ctrlVertices;
#else
  free( knotsV );
  free( knotsU );
  free( ctrlVertices);
#endif

}


void dumpNURBS( FILE * f, SAA_Scene * scene, SAA_Elem * srf )
/* 
   Dumps info about the given NURBS onto the specified output
   stream.

   At the moment only NURBS curves are supported.
*/
{
  int i;
  double a;

  int nbVertices[2];

  double * knotsU, * knotsV;
  int nbKnotsU, nbKnotsV;

  SAA_Boolean closed [2];
  int degree [2];

  SAA_ModelType mdlType;

  /* 
  ** Check the parameters
  */
  SAA_modelGetType( scene, srf, &mdlType ); 
  if ( mdlType == SAA_MNCRV ) 
    { 
      /* CURVE */
      SAA_nurbsCurveGetDegree( scene, srf, &degree[0] );
      SAA_nurbsCurveGetClosed( scene, srf, &closed[0] );
      SAA_modelGetNbVertices( scene, srf, &nbVertices[0] );

      /* get the knots */
      SAA_nurbsCurveGetNbKnots( scene, srf, &nbKnotsU );

#ifdef __cplusplus
      knotsU = new double [ nbKnotsU ];
#else
      knotsU = (double *) calloc( nbKnotsU, sizeof( double ) );
#endif

      SAA_nurbsCurveGetKnots( scene, srf, SAA_GEOM_ORIGINAL, 
			      -1 /* dummy value */, nbKnotsU,
			      knotsU );

      /* print all the stuff */
      fprintf( f, "\nNURBS curve\nDegree = %i, # control vertices = %i\n",
	       degree[0], nbVertices[0] );
      fprintf( f, ( closed[0] ? "NURBS is closed\n" : "NURBS is open\n" ) );
      fprintf( f, "# knots = %i, the knots are: \n", nbKnotsU );

      fprintf( f, "%f, ", knotsU[0] );
      for( i=1; i<nbKnotsU; i++ ) {
	fprintf( f, "[%f] %f ", (knotsU[i] - knotsU[i-1]), knotsU[i] );
      }
      fprintf(f, "\n\n" );

#ifdef __cplusplus   
      delete [] knotsU;
#else
      free( knotsU );
#endif

    }
  else if ( mdlType == SAA_MNSRF ) 
    {
      /* SURFACE */
      SAA_nurbsSurfaceGetDegree( scene, srf, &degree[0], &degree[1] );
      SAA_nurbsSurfaceGetClosed( scene, srf, &closed[0], &closed[1] );
      SAA_nurbsSurfaceGetNbVertices( scene, srf, &nbVertices[0], &nbVertices[1] );

      /* get the knots */
      SAA_nurbsSurfaceGetNbKnots( scene, srf, &nbKnotsU, &nbKnotsV );

#ifdef __cplusplus
      knotsU = new double [ nbKnotsU ];
      knotsV = new double [ nbKnotsV ];
#else
      knotsU = (double *) calloc( nbKnotsU, sizeof( double ) );
      knotsV = (double *) calloc( nbKnotsV, sizeof( double ) );
#endif

      SAA_nurbsSurfaceGetKnots( scene, srf, SAA_GEOM_ORIGINAL, 
				-1 /* dummy value */, nbKnotsU, nbKnotsV,
				knotsU, knotsV );
      /* print all the stuff */
      fprintf( f, "\nNURBS surface\nDegree, U: %i, V: %i, # control vertices, U: %i, V: %i\n",
	       degree[0], degree[1], nbVertices[0], nbVertices[1] );
      fprintf( f, ( closed[0] ? "Closed in U\n" : "Open in U\n" ) );
      fprintf( f, ( closed[1] ? "Closed in V\n" : "Open in V\n" ) );
      fprintf( f, "# knots, U: %i, V: %i, the knots in U-dir are: \n", nbKnotsU, nbKnotsV );

      fprintf( f, "%f, ", knotsU[0] );
      for( i=1; i<nbKnotsU; i++ ) {
	fprintf( f, "[%f] %f ", (knotsU[i] - knotsU[i-1]), knotsU[i] );
      }

      fprintf( f, "\nthe knots in V-dir are: \n" );
      fprintf( f, "%f, ", knotsV[0] );
      for( i=1; i<nbKnotsV; i++ ) {
	fprintf( f, "[%f] %f ", (knotsV[i] - knotsV[i-1]), knotsV[i] );
      }
      fprintf(f, "\n\n" );

#ifdef __cplusplus   
      delete [] knotsV;
      delete [] knotsU;
#else
      free( knotsV );
      free( knotsU );
#endif

    }

}


void globalise( SAA_Elem * src, PR_nurb * dest, SAA_Scene * scene )
/* 
   This function 'globalises' the coordinates of dest (assuming that
   dest has the local coordinates of src)

*/
{
  Pmatrix3 matrix;
  Ppoint3 scale, rot, trans;

  SAA_modelGetScaling( scene, src, SAA_COORDSYS_GLOBAL, 
		       &scale.x, &scale.y, &scale.z );
  SAA_modelGetRotation( scene, src, SAA_COORDSYS_GLOBAL, 
			&rot.x, &rot.y, &rot.z );
  SAA_modelGetTranslation( scene, src, SAA_COORDSYS_GLOBAL, 
			   &trans.x, &trans.y, &trans.z );
  
  ptk_scale3(&scale, PTYPE_REPLACE, matrix);
  ptk_rotate3(rot.x, PTKEXAXIS, PTYPE_POSTCONCAT, matrix);
  ptk_rotate3(rot.y, PTKEYAXIS, PTYPE_POSTCONCAT, matrix);
  ptk_rotate3(rot.z, PTKEZAXIS, PTYPE_POSTCONCAT, matrix);
  ptk_shift3(&trans, PTYPE_POSTCONCAT, matrix);
  nrb_xform( dest, matrix );
}



void scaleKnotVector( PR_dir * knotVector, double newMax )
/* This function scales the given knot vector
   so that it will lie in the range 
   [0.0 ... newMax].
*/
{
  int i;
  double f;

  if ( knotVector->pf_kk != NULL ) {
    /* 
    ** Get the current upper limit of the u knot vector and divide 
    ** U by this. 
    */ 
    f = newMax / (knotVector->pf_kk->knots[ knotVector->pf_nt - 1 ]);

    for (i = 0; i < knotVector->pf_nt; i++) {
      knotVector->pf_kk->knots[ i ] *= f;
    }
  }
}

/*
 * Some function for reading/writing NURBS from/to files
 * in the format of the Manchester NURBS procedure library
 */


void readNURBS( const char * inputFile, SAA_Scene * scene )
/* read a NURBS from the file named inputFile, and 
   convert it to SAAPHIRE. The NURBS will be created in
   the given scene 
*/
{
  FILE *fp;	
  PR_nurb tmp_nurb;
  SAA_Elem SI_NURBS;

  fp = fopen( inputFile, "r");
  if ( fp == NULL )
  {
    printf( "readNURBS() - error opening input file\n" );
    return;
  } else {
    printf( "readNURBS() - input file opened succesfully\n" );
  }
  
  nrb_read(fp, &tmp_nurb);
  fclose(fp);

  convertNURBSToSI( &tmp_nurb, &SI_NURBS, scene );

}

void writeNURBS( const char * outputFile, SAA_Scene * scene,
		 SAA_Elem * SI_NURBS )
/* writes the given NURBS SI_NURBS from the given scene to the 
   file named outputFile.
*/
{
  FILE *fp;	
  PR_nurb tmp_nurb;

  nrb_clear( &tmp_nurb );

  convertNURBSFromSI( SI_NURBS, &tmp_nurb, scene );

  fp = fopen( outputFile, "w");
  if ( fp == NULL )
  {
    printf( "writeNURBS() - error opening output file\n" );
    return;
  } else {
    printf( "writeNURBS() - output file opened succesfully\n" );
  }
  
  nrb_write(fp, &tmp_nurb);
  fclose(fp);

  printf( "writeNURBS() - finished\n" );
}
