/*******************************************************************************

 'hyper.c' 
 HyperSurface animation custom effect for SoftImage 3D
 Version 1.0
 
 Written by M.J.Evers
 (c) Computer Graphics Unit, Manchester Computing, 
 University of Manchester, 1996

 Created: 19/8/96
 Modified: 19/8/96, 20/8/96, 21/8/96, 22/8/96, 29/8/96, 30/8/96

 Parts of this custom effect were written by M. Preston and ...
 
*******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <SAA.h>

#include "convert.h"
#include "blending.h"
#include "skinning.h"
#include "misc.h"

/****************************************/
/*           LOCAL CONSTANTS            */
/****************************************/

/*
** I've created a number of defines for the dialog item numbers and
** for the indexes in the argument array of the elements the user 
** has picked. This makes changing the pick order a lot easier
**
** In the case we have a variable nr of arguments, NB_ARGS
** specifies the maximum nr of arguments.
*/

#define NB_ARGS           20

#define DLG_OK            1
#define DLG_CANCEL        2
#define DLG_UPDATE        16
#define DLG_SET           33

#define DLG_SLIDER1       9
#define DLG_SLIDER2       13
#define DLG_TUNING1       8
#define DLG_TUNING2       12
#define DLG_EDIT1         19
#define DLG_EDIT2         20

#define DLG_AUTOFRM       21
#define DLG_TXTVW         29
#define DLG_WEIGHT        30

#define DLG_NBARGS        15
#define DLG_W1            34
#define DLG_W2            35
#define DLG_W3            36
#define DLG_W4            37
#define DLG_W5            38
#define DLG_W6            39
#define DLG_W7            40
#define DLG_W8            41
#define DLG_W9            42
#define DLG_W10           43


/****************************************/
/*             LOCAL MACROS             */
/****************************************/

/*
** Some functions have arguments that are never used.  To prevent C++ compilers
** from issuing warnings about those arguments, the following macro can be
** used.
*/
#ifdef __cplusplus
#  define _UNUSED_ARG( x )
#else
#  define _UNUSED_ARG( x ) x
#endif

/****************************************/
/*             LOCAL TYPES              */
/****************************************/

/*
** The following struct contains instance-specific information.
*/
typedef struct
{
  /*
  ** Saaphire functions return an SI_Error code.  The following field stores
  ** the last code that was returned so that we remember when things have gone
  ** wrong.
  */
  SI_Error result;

  /*
  ** Many Saaphire functions operate on a scene.  The following field stores
  ** the current scene so that we don't have to keep getting it.
  */
  SAA_Scene scene;

  int nbArgs;

  /*
  ** hyperSrf contains the HyperSurface for this effect
  */
  PR_nurb4 hyperSrf;

  /* 
  ** fStart, fEnd are the start and end frames of the hypersurface 
  ** animation. autoFrame indicates whether to use fStart and fEnd or
  ** to use the current start and end frames.
  */
  int fStart, fEnd;
  SAA_Boolean autoFrame;

  /*
  ** weights are the weight factors associated with the 
  ** sections
  */
  float weights[ (NB_ARGS/2) ];

  /*
  ** The updateFlag is used to prevent this effect
  ** from updating the hypersurface too often:
  ** SI calls update for each result definition line
  ** in the .cus file. In this CE there are 2 result
  ** definitions. We toggle the flag in update() so
  ** that of each two calls of update() only one real
  ** update takes place.
  ** We do not use the SAA_customContextGetCurrentResult
  ** because it does not return the right indexes.
  */
  SAA_Boolean updateFlag;

} InstanceData;

/****************************************/
/*      LOCAL FUNCTION PROTOTYPES       */
/****************************************/

SI_Error initInstance( InstanceData ** idata );

void DeleteSectionList( SectionRecord ** the_list );

void deallocate_nrb( PR_nurb * nrb );

void adjustWeights( PR_nurb * current, double f );

void createHyperSurface( SAA_Scene * scene, SAA_Elem * args, int nbSections,
			 float * weights, PR_nurb * out );


/****************************************/
/*           LOCAL VARIABLES            */
/****************************************/


/****************************************/
/*           GLOBAL FUNCTIONS           */
/****************************************/

/*
** The names of global (public) functions that are given to SOFTIMAGE|3D in
** .cus files must not be mangled by C++ compilers.  If this file is being
** compiled by a C++ compiler, prevent the compiler from mangling global
** function names.
*/
#ifdef __cplusplus
extern "C" {
#endif


SI_Error weightCallback (
  const SAA_CustomContext context,
  int dialogItemId,
  void *userData
 );

SI_Error updateCallback (
  const SAA_CustomContext context,
  int dialogItemId,
  void *userData
 );

SI_Error okCallback (
  const SAA_CustomContext context,
  int dialogItemId,
  void *userData
 );


/*******************************************************************************

   $$L init

   Saaphire effect initialization function. This function initialises the
   instance data as far as setup() has not done it and creates the blend 
   surface.

   Returned Value : SI_Error.

   NOTE: The init() function definition is SAAPHIRE version 1.0 style 
         instead of v1.1 style (that means there is no 'context' variable).
	 The reason for this is that setting the result models does not
	 work with the v1.1 context functions.
	 Lines of code that would be used if init() was v1.1 style have been
	 commented out.

*******************************************************************************/

_CUS_EXTERN SI_Error init
(
   SAA_CustomValueList valueList,
   int nbargs,
   SAA_Elem *args,
   int  nbres,
   SAA_Elem *resx,
   void **instdata

   /* v1.1: */ /* SAA_CustomContext context */
)

{
  InstanceData	*idata;		/* This effect's instance data.		*/
  SI_Error	result, res;	/* Return value.			*/

  /* v1.1: */
  /*   SAA_CustomValueList valueList;
  SAA_Elem args [ NB_ARGS ]; */

  SAA_Elem out;
  PR_nurb current;

  int i, nbSections;

  SAA_ModelType mdlType;
   
/* v1.1: *//*   SAA_customContextGetCustomValueList( context, &valueList ); */

  result = initInstance( &idata );

  /* Pass the instance data pointer to SAAPHIRE */
/* v1.1: *//*     result = SAA_customContextSetInstanceData(context, (void *) idata);*/
  *instdata = idata;

  /* 
  ** Get the nr of arguments and the weight factors from 
  ** the dialog box. 
  ** If the effect is created from scratch, the weight values are
  ** initialised during the setup dialog.
  */
  SAA_cusvalGetIntValue( valueList, "NBARGS", &idata->nbArgs );
  SAA_cusvalGetFloatValue( valueList, "W1", &idata->weights[0] );
  SAA_cusvalGetFloatValue( valueList, "W2", &idata->weights[1] );
  SAA_cusvalGetFloatValue( valueList, "W3", &idata->weights[2] );
  SAA_cusvalGetFloatValue( valueList, "W4", &idata->weights[3] );
  SAA_cusvalGetFloatValue( valueList, "W5", &idata->weights[4] );
  SAA_cusvalGetFloatValue( valueList, "W6", &idata->weights[5] );
  SAA_cusvalGetFloatValue( valueList, "W7", &idata->weights[6] );
  SAA_cusvalGetFloatValue( valueList, "W8", &idata->weights[7] );
  SAA_cusvalGetFloatValue( valueList, "W9", &idata->weights[8] );
  SAA_cusvalGetFloatValue( valueList, "W10", &idata->weights[9] );

  nbSections = idata->nbArgs / 2;

  /* 
  ** Initialise the NURBS procedure library
  */
  nrb_initialise();
  
  /* 
  ** Check if the user has picked at least 2 models 
  */
  if ( nbSections < 2 ) {
    result = SI_ERR_CUSTOM_FATAL;
  }

  if ( result == SI_SUCCESS ) {

    createHyperSurface( &idata->scene, args, nbSections, idata->weights, &idata->hyperSrf );

    /* 
    ** We check the elemid field of the element in the result array.
    ** If it's -1, the result object needs to be created .
    ** Otherwise, the user has loaded a scene containing this
    ** custom effect and we should not create the object then.
    */
    if ( resx[0].elemid == -1) {

      nrb_clear( &current );
      evaluate_4to3( &idata->hyperSrf, 0.0, &current );

      convertNURBSToSI( &current, &out, &idata->scene );
      SAA_elementSetName( &idata->scene, &out, "Hyper1");
   
      /*
      ** As init() is responsible for the result elements,
      ** we put the newly created spline in the context. 
      */
      resx[0] = out;
      resx[1] = out;
/* v1.1: use SAA_contextSetResult() */

      deallocate_nrb( &current );
    }

    /* Give the updateFlag an initial value */
    idata->updateFlag = TRUE;
  }

  /*
  ** That's it.
  */
  if( idata != NULL )
    {
      idata->result = result;
    }
  return( result );
}



/*******************************************************************************

   $$L update

   Saaphire effect update function, executes the effect.

   Returned Value : SI_Error.

*******************************************************************************/

_CUS_EXTERN SI_Error update
(
 SAA_CustomContext context
)
{
  InstanceData	*idata;		/* This effect's instance data.		*/
  SI_Error	result;		/* Return value.			*/

  SAA_Elem res[2];

  PR_nurb current;
  SAA_CustomValueList valueList;

  float t, weightFactor, timeFactor;
  int i, stepU, stepV;
  int fStart, fEnd, fCurrent;

  /*
  ** Check if init() couldn't allocate instance data.
  */
  result = SAA_customContextGetInstanceData(context, (void **) &idata);

  if( idata != NULL && result == SI_SUCCESS)
    {
      result = idata->result;
    } else {
      result = SI_ERR_CUSTOM_FATAL;
    }

  if ( result == SI_SUCCESS ) {

    /*
    ** Toggle the update flag and perform the actual update only
    ** when the flag is true
    */
    idata->updateFlag = !idata->updateFlag;
    if ( idata->updateFlag ) {
      /* 
      ** Get the data from the dialog
      */
      SAA_customContextGetCustomValueList( context, &valueList );

      SAA_cusvalGetIntValue( valueList, "STRTFRM", &idata->fStart );
      SAA_cusvalGetIntValue( valueList, "ENDFRM", &idata->fEnd );
      SAA_cusvalGetStateValue( valueList, "AUTOFRM", &idata->autoFrame );

      /* Get the 'custom' fcurve values for the current frame/time */
      SAA_cusvalGetFloatValue( valueList, "Time", &timeFactor );

      SAA_customContextGetResults( context, 2, res );

      t = timeFactor; 

      /* 
      ** If the user has specified start and end frame numbers, 
      ** clip the time to this time period.
      */
      if ( ! idata->autoFrame ) {
	SAA_sceneGetPlayCtrlCurrentFrame( &idata->scene, &fCurrent );
 
	if (  fCurrent < idata->fStart ) {
	  t = 0.0;
	} else if ( fCurrent > idata->fEnd ){
	  t = 1.0; 
	}
      }

      /* 
      ** Clip t to the range [0,1] 
      */
      if ( t < 0.0 ) t = 0.0;
      if ( t > 1.0 ) t = 1.0;

      /*
      ** Store the step values of the surface. They are reset
      ** when we call SAA_nurbsSurfaceEditSurface()
      */
      SAA_nurbsSurfaceGetStep( &idata->scene, &res[0], &stepU, &stepV );

      /*
      ** Evaluate the hypersurface at the current point in time
      ** and copy the points to the SI result model
      */
      nrb_clear( &current );
      evaluate_4to3( &idata->hyperSrf, t, &current );

      copyPoints( &current, &res[0], &idata->scene, FALSE, FALSE );

      /*
      ** Restore step values
      */
      SAA_nurbsSurfaceSetStep( &idata->scene, &res[0], stepU, stepV );
      
      deallocate_nrb( &current );
    }
  }

  if( idata != NULL )
    {
      idata->result = result;
    }
  return( result );
}

/*******************************************************************************

   $$L cleanup

   Saaphire effect cleanup function, frees instance data.

   Returned Value : SI_Error.

*******************************************************************************/

_CUS_EXTERN SI_Error cleanup
(
 SAA_CustomContext context
)
{
   InstanceData	*idata;		/* This effect's instance data.		*/
   SI_Error	result;		/* Return value.			*/

   /*
   ** Check if init() couldn't allocate instance data.
   */
   result = SAA_customContextGetInstanceData(context, (void **) &idata);

   if( idata != NULL && result == SI_SUCCESS ) {
      result = idata->result;
   } else {
      result = SI_ERR_CUSTOM_FATAL;
   }

   /*
   ** Free instance data.
   */
   if( idata != NULL )
   {
     /* deallocate the HyperSurface */
     deallocate_nrb( &idata->hyperSrf );
     
#ifdef __cplusplus
     delete idata;
#else
     free( idata );
#endif
   }

   return( result );
}


/*
** setup() function sets up the 'setup' dialog
** That is: it en/disables items and installs callback functions
*/
_CUS_EXTERN SI_Error setup
(
 SAA_CustomContext context
)
{
/*  SI_Error	result;*/

  /* 
  ** As we do not know how to hide the #args field and
  ** the weight factor fields, we just disable them.
  */
  SAA_dialogitemSetSensitivity( context, DLG_NBARGS, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W1, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W2, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W3, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W4, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W5, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W6, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W7, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W8, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W9, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W10, TRUE );

  SAA_dialogitemSetSensitivity( context, DLG_UPDATE, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_SET, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_WEIGHT, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_TXTVW, TRUE );
  SAA_textlistClear( context, DLG_TXTVW );
   
  SAA_dialogitemAddCallback( context, DLG_OK, &okCallback, NULL );

  return SI_SUCCESS;
}

_CUS_EXTERN SI_Error edit
(
 SAA_CustomContext context
)
{
  InstanceData	*idata;		/* This effect's instance data.		*/
  SI_Error	result;		/* Return value.			*/
  int i;
  char s[20];

  result = SAA_customContextGetInstanceData(context, (void **) &idata);
  /* *** check also return value of ..GetInstData */

  SAA_dialogitemSetSensitivity( context, DLG_SET, FALSE );
  SAA_dialogitemSetSensitivity( context, DLG_WEIGHT, FALSE );
  SAA_dialogitemSetSensitivity( context, DLG_TXTVW, FALSE );
  SAA_dialogitemSetSensitivity( context, DLG_UPDATE, FALSE );

  /*
  ** Add tags for the sections to the text view
  */
  SAA_textlistClear( context, DLG_TXTVW );
  for( i=0; i < idata->nbArgs/2; i++ ) {
    sprintf( s, "Section#%i", i );
    SAA_textlistCreateItem( context, DLG_TXTVW, i, s);
  }

  /* 
  ** As we do not know how to hide the #args field, we just
  ** disable it
  */
  SAA_dialogitemSetSensitivity( context, DLG_NBARGS, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W1, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W2, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W3, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W4, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W5, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W6, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W7, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W8, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W9, TRUE );
  SAA_dialogitemSetSensitivity( context, DLG_W10, TRUE );

  SAA_dialogitemAddCallback( context, DLG_UPDATE, &updateCallback, NULL );
  SAA_dialogitemAddCallback( context, DLG_SET, &weightCallback, NULL );
  SAA_dialogitemAddCallback( context, DLG_TXTVW, &weightCallback, NULL );

  return SI_SUCCESS;
}

/****************************************/
/*        CALLBACK FUNCTIONS            */
/****************************************/


SI_Error weightCallback (
  const SAA_CustomContext context,
  int dialogItemId,
  void * _UNUSED_ARG( userData )
 )
/* 
** weightCallback() is a callback function for the edit dialog
** and handles the viewing and changing
** of the weight factors 
*/
{
  InstanceData	*idata;
  int nbSel, i;
  
  SAA_customContextGetInstanceData(context, (void **) &idata);

  SAA_textlistGetNbSelected( context, DLG_TXTVW, &nbSel );
  if ( nbSel == 1 ) {
    SAA_textlistGetSelected( context, DLG_TXTVW, 1, &i );

    if ( dialogItemId == DLG_SET ) {
      SAA_dialogitemGetFloatValue( context, DLG_WEIGHT, &idata->weights[i] );
    } else {
      SAA_dialogitemSetFloatValue( context, DLG_WEIGHT, idata->weights[i] );
    }
  }
}


SI_Error updateCallback (
  const SAA_CustomContext context,
  int _UNUSED_ARG( dialogItemId ),
  void * _UNUSED_ARG( userData )
 )
/* 
** updateCallback() is callback function for the edit dialog 
** and re-creates the Hypersurface.
*/
{
  int nbSections;
  InstanceData	*idata;
  SAA_Elem args[ NB_ARGS ];

  SAA_customContextGetInstanceData(context, (void **) &idata);

  SAA_customContextGetArguments( context, idata->nbArgs, args );
  nbSections = idata->nbArgs / 2;

  createHyperSurface( &idata->scene, args, nbSections, idata->weights, &idata->hyperSrf );
  SAA_sceneRefresh( context );
  
  /* 
  ** Store the weight factors in the dialog, just in case they've been
  ** modified. It's currently the only way to save them. 
  */
  SAA_dialogitemSetFloatValue( context, DLG_W1, idata->weights[0] );
  SAA_dialogitemSetFloatValue( context, DLG_W2, idata->weights[1] );
  SAA_dialogitemSetFloatValue( context, DLG_W3, idata->weights[2] );
  SAA_dialogitemSetFloatValue( context, DLG_W4, idata->weights[3] );
  SAA_dialogitemSetFloatValue( context, DLG_W5, idata->weights[4] );
  SAA_dialogitemSetFloatValue( context, DLG_W6, idata->weights[5] );
  SAA_dialogitemSetFloatValue( context, DLG_W7, idata->weights[6] );
  SAA_dialogitemSetFloatValue( context, DLG_W8, idata->weights[7] );
  SAA_dialogitemSetFloatValue( context, DLG_W9, idata->weights[8] );
  SAA_dialogitemSetFloatValue( context, DLG_W10, idata->weights[9] );

  return SI_SUCCESS;
}


SI_Error okCallback (
  const SAA_CustomContext context,
  int _UNUSED_ARG( dialogItemId ),
  void * _UNUSED_ARG( userData )
 )
/* okCallback() is a callback function for the setup dialog 
** and saves the number of arguments and the weights in the dialog 
*/
{
  int nbArgs;

  /* 
  ** Store the number of arguments in the dialog. We cannot do
  ** this in init() because init() is v1.0 style and does not have 
  ** a context parameter.
  */
  SAA_customContextGetNbArguments( context, &nbArgs );
  SAA_dialogitemSetIntValue( context, DLG_NBARGS, nbArgs );

  /* 
  ** We also store the weight factors in the dialog. It's currently
  ** the only way to save them. 
  */
  SAA_dialogitemSetFloatValue( context, DLG_W1, 1.0 );
  SAA_dialogitemSetFloatValue( context, DLG_W2, 1.0 );
  SAA_dialogitemSetFloatValue( context, DLG_W3, 1.0 );
  SAA_dialogitemSetFloatValue( context, DLG_W4, 1.0 );
  SAA_dialogitemSetFloatValue( context, DLG_W5, 1.0 );
  SAA_dialogitemSetFloatValue( context, DLG_W6, 1.0 );
  SAA_dialogitemSetFloatValue( context, DLG_W7, 1.0 );
  SAA_dialogitemSetFloatValue( context, DLG_W8, 1.0 );
  SAA_dialogitemSetFloatValue( context, DLG_W9, 1.0 );
  SAA_dialogitemSetFloatValue( context, DLG_W10, 1.0 );
}

 
#ifdef __cplusplus
}
#endif


/****************************************/
/*           LOCAL FUNCTIONS            */
/****************************************/

/* 
** initInstance() performs the initialisation of the instance data 
** for this custom effect. 
*/
SI_Error initInstance( InstanceData ** idata )
{
   SI_Error result;

   /*
   ** Initialize Saaphire.  This is always the first thing that a Saaphire
   ** effect should do.
   */
   result = SAA_sceneInit();

   /*
   ** Set the verify flags so that SI will verify everything.
   */
   SAA_verifyFlagSet( SAA_VERIFY_ALL, FALSE );

   /*
   ** Allocate instance-specific data for this effect instance.  Store the
   ** pointer in the context argument so that it gets passed to
   ** update() and cleanup().
   */

   if ( result == SI_SUCCESS ) {
#ifdef __cplusplus
     *idata = new InstanceData;
#else
     *idata = ( InstanceData * ) malloc( sizeof( InstanceData ) );
#endif

     if( *idata != NULL ) {
       result = SI_SUCCESS;
     } else {
       fprintf( stderr, "ERROR: initInstance(): Error allocating instance data memory." );
       result = SI_ERR_CUSTOM_FATAL;
     }
   }

   /* Store the current scene */
   if( result == SI_SUCCESS ) {
     result = SAA_sceneGetCurrent( & (*idata)->scene );
   }
   (*idata)->result = result;

   return result;
}

void createHyperSurface( SAA_Scene * scene, SAA_Elem * args, int nbSections,
			 float * weights, PR_nurb4 * out )
/*
** This function creates a HyperSurface in PR LIB format, based on a 
** number of sections in SI format (sections).
** weights is an array of weight factors, one weight factor for each section
*/
{
  PR_nurb nurbs, temp;
  float t, delta;
  SectionRecord * sectionList;
  int i;
  SAA_Boolean closedU, closedV;

  /*
  ** The sections are uniformly distributed over the 
  ** time axis. delta is the time between 2 sections
  */
  t = 0.0;
  delta = 1.0 / (nbSections - 1);
  CreateSectionList( &sectionList );

  for( i = 0; i < nbSections; i++ ) 
    {
      nrb_clear( &nurbs );

      /*
      ** Convert the SI NURBS to NURBS procedure library format 
      */
      convertNURBSFromSI( &args[i], &nurbs, scene ); 

      /*
      ** Clamp the curve if necessary.
      */ 
      SAA_nurbsSurfaceGetClosed( scene, &args[i], &closedU, &closedV );
	
      if ( closedU ) {
	clampNURBSsurface( &nurbs, &temp );
	 deallocate_nrb( &nurbs );
	nrb_copy( &temp, &nurbs );
	 deallocate_nrb( &temp );
      }

      if ( closedV ) {
	nrb_transpose( &nurbs );
	clampNURBSsurface( &nurbs, &temp );
	 deallocate_nrb( &nurbs );
	nrb_copy( &temp, &nurbs );
	 deallocate_nrb( &temp );
        nrb_transpose( &nurbs );
      }

      /*
      ** SI has given us the local coordinates of the vertices of the
      ** objects. We will 'globalise' them by getting the transformation 
      ** matrix of the center of the surfaces and transforming the NURBS PR
      ** lib surfaces.
      */   
      globalise( &args[i], &nurbs, scene );

      /*
      ** We scale the u (and v) knot vectors of all NURBS to [0.0 ... 1.0] 
      ** (which is more or less standard)
      */
      scaleKnotVector( &nurbs.pf_u, 1.0 );
      scaleKnotVector( &nurbs.pf_v, 1.0 );

      /*
      ** adjust weights using the factors in the instance data 
      */
      adjustWeights( &nurbs, weights[i] );

      /*
      ** Add NURBS to the section list 
      */
      CreateNewSection( sectionList, &nurbs, t );
      t += delta;

      deallocate_nrb( &nurbs );
    }

  /*
  ** Now create the HyperSurface 
  */
  nrb_clear( out );
  skin_3to4( sectionList, out );

  /* Clean up things */
  DeleteSectionList( &sectionList );
}

