/* pngmeta.c - PNG meta data filter (optionally in SOIF format)
   
   For libpng 1.0 - version 0.85
   (C) Copyright Dave Beckett <D.J.Beckett@ukc.ac.uk>, University
   of Kent at Canterbury
   4th January 1996
   
   The function png_skip_till_end() is a modified version of
   png_read_end() from libpng 1.0 beta 2 version 0.85 by Guy Eric
   Schalnat, Group 42, Inc. and others.  Original copyright message:
   
   ----------------------------------------------------------------
   png.h - header file for png reference library

   libpng 1.0 beta 2 - version 0.85
   December 19, 1995

   Note: This is a beta version.  It reads and writes valid files
   on the platforms I have, but it has had limited portability
   testing.  Furthermore, you will may have to modify the
   includes below to get it to work on your system, and you
   may have to supply the correct compiler flags in the makefile.
   Read the readme.txt for more information, and how to contact
   me if you have any problems, or if you want your compiler/
   platform to be supported in the next official libpng release.

   See readme.txt for more information

   Copyright (c) 1995 Guy Eric Schalnat, Group 42, Inc.
	Contributing Authors:
      Dave Martindale
      Guy Eric Schalnat
      Paul Schmidt
      Tim Wegner

   The PNG Reference Library is supplied "AS IS". The Contributing Authors
   and Group 42, Inc. disclaim all warranties, expressed or implied,
   including, without limitation, the warranties of merchantability and of
   fitness for any purpose. The Contributing Authors and Group 42, Inc.
   assume no liability for damages, direct or consequential, which may
   result from the use of the PNG Reference Library.

   Permission is hereby granted to use, copy, modify, and distribute this
   source code, or portions hereof, for any purpose, without fee, subject
   to the following restrictions:
   1. The origin of this source code must not be misrepresented.
   2. Altered versions must be plainly marked as such and must not be
   misrepresented as being the original source.
   3. This Copyright notice may not be removed or altered from any source or
   altered source distribution.
	  
   The Contributing Authors and Group 42, Inc. specifically permit, without
   fee, and encourage the use of this source code as a component to
   supporting the PNG file format in commercial products. If you use this
   source code in a product, acknowledgment is not required but would be
   appreciated.
   */
   

#include <stdio.h>
#include <stdlib.h>
#define PNG_INTERNAL
#include "png.h"

#ifdef __TURBOC__
#include <mem.h>
#endif

/* defined so I can write to a file on gui/windowing platforms */
/*  #define STDERR stderr  */
#ifdef MSDOS
#define STDERR stdout	/* for DOS */
#else
#define STDERR stderr
#endif


/* read data, ignoring IDATs, till the end of the png file.
   
   Will not read past the end of the file, will verify the end is
   accurate, and will read any comments or time information at the
   end of the file, if info is not NULL. */
void
png_skip_till_end(png_structp png_ptr, png_infop info)
{
   png_byte chunk_start[8];
   png_uint_32 length;
   png_uint_32 crc;

  /* Skip IDAT chunks */
   do
     {
       png_crc_skip(png_ptr, png_ptr->idat_size);

       /* Calculate and check CRC on it */
       
       png_read_data(png_ptr, chunk_start, 4);
       crc = png_get_uint_32(chunk_start);
       if (((crc ^ 0xffffffffL) & 0xffffffffL) !=
	   (png_ptr->crc & 0xffffffffL))
	 png_error(png_ptr, "Bad CRC value");
       
       png_read_data(png_ptr, chunk_start, 8);
       png_ptr->idat_size = png_get_uint_32(chunk_start);
       png_reset_crc(png_ptr);
       png_calculate_crc(png_ptr, chunk_start + 4, 4);
     } while(!png_memcmp(chunk_start + 4, png_IDAT, 4));
   png_ptr->mode = PNG_AFTER_IDAT;
   length=png_ptr->idat_size;
   
   do
   {
      if (!png_memcmp(chunk_start + 4, png_IHDR, 4))
      {
         png_error(png_ptr, "invalid chunk after IDAT");
      }
      else if (!png_memcmp(chunk_start + 4, png_PLTE, 4))
      {
         png_error(png_ptr, "invalid chunk after IDAT");
      }
      else if (!png_memcmp(chunk_start + 4, png_gAMA, 4))
      {
         png_error(png_ptr, "invalid chunk after IDAT");
      }
      else if (!png_memcmp(chunk_start + 4, png_sBIT, 4))
      {
         png_error(png_ptr, "invalid chunk after IDAT");
      }
      else if (!png_memcmp(chunk_start + 4, png_cHRM, 4))
		{
         png_error(png_ptr, "invalid chunk after IDAT");
      }
      else if (!png_memcmp(chunk_start + 4, png_tRNS, 4))
      {
         png_error(png_ptr, "invalid chunk after IDAT");
      }
      else if (!png_memcmp(chunk_start + 4, png_bKGD, 4))
      {
         png_error(png_ptr, "invalid chunk after IDAT");
      }
      else if (!png_memcmp(chunk_start + 4, png_hIST, 4))
      {
         png_error(png_ptr, "invalid chunk after IDAT");
      }
      else if (!png_memcmp(chunk_start + 4, png_IDAT, 4))
      {
         if (length > 0 || png_ptr->mode != PNG_AT_LAST_IDAT)
				png_error(png_ptr, "too many IDAT's found");
      }
      else if (!png_memcmp(chunk_start + 4, png_pHYs, 4))
      {
         png_error(png_ptr, "invalid chunk after IDAT");
      }
      else if (!png_memcmp(chunk_start + 4, png_oFFs, 4))
      {
         png_error(png_ptr, "invalid chunk after IDAT");
      }
#if defined(PNG_READ_tIME_SUPPORTED)
      else if (!png_memcmp(chunk_start + 4, png_tIME, 4))
      {
         if (png_ptr->mode == PNG_BEFORE_IHDR ||
            png_ptr->mode == PNG_AFTER_IEND)
            png_error(png_ptr, "Out of Place tIME");

         if (info)
				png_handle_tIME(png_ptr, info, length);
         else
            png_crc_skip(png_ptr, length);
      }
#endif
#if defined(PNG_READ_tEXt_SUPPORTED)
      else if (!png_memcmp(chunk_start + 4, png_tEXt, 4))
      {
         if (png_ptr->mode == PNG_BEFORE_IHDR ||
            png_ptr->mode == PNG_AFTER_IEND)
            png_error(png_ptr, "Out of Place tEXt");

         if (info)
            png_handle_tEXt(png_ptr, info, length);
         else
            png_crc_skip(png_ptr, length);
      }
#endif
#if defined(PNG_READ_zTXt_SUPPORTED)
      else if (!png_memcmp(chunk_start + 4, png_zTXt, 4))
      {
         if (png_ptr->mode == PNG_BEFORE_IHDR ||
            png_ptr->mode == PNG_AFTER_IEND)
            png_error(png_ptr, "Out of Place zTXt");

         if (info)
            png_handle_zTXt(png_ptr, info, length);
         else
            png_crc_skip(png_ptr, length);
      }
#endif
      else if (!png_memcmp(chunk_start + 4, png_IEND, 4))
      {
         png_ptr->mode = PNG_AFTER_IEND;
      }
      else
	{
         if ((chunk_start[4] & 0x20) == 0)
            png_error(png_ptr, "Unknown Critical Chunk");

         png_crc_skip(png_ptr, length);
      }
      png_read_data(png_ptr, chunk_start, 4);
      crc = png_get_uint_32(chunk_start);
      if (((crc ^ 0xffffffffL) & 0xffffffffL) !=
	  (png_ptr->crc & 0xffffffffL))
	png_error(png_ptr, "Bad CRC value");
      if (png_ptr->mode == PNG_AT_LAST_IDAT)
	png_ptr->mode = PNG_AFTER_IDAT;
      
      /* Get next chunk */
      if(png_ptr->mode != PNG_AFTER_IEND)
	{
	  png_read_data(png_ptr, chunk_start, 8);
	  length = png_get_uint_32(chunk_start);
	  png_reset_crc(png_ptr);
	  png_calculate_crc(png_ptr, chunk_start + 4, 4);
	}
   } while (png_ptr->mode != PNG_AFTER_IEND);
}


void print_kv(FILE *fd, int output_soif, const char *field, const char *value) 
{
  static char bitbuf[25];
  
  /* SOIF: "KEY{SIZE}:\tVALUE\n" */
  if(output_soif)
    sprintf(bitbuf, "{%d}:\t", (int)strlen(value));
  else 
    strcpy(bitbuf, ": ");
  
  fprintf(fd, "%s%s%s\n", field, bitbuf, value);
}


/* MAIN BODY */
int main(int argc, char *argv[])
{
  png_struct png_ptr;
  png_info info_ptr;
  png_info end_info;
  FILE *fp;
  char *pngfile;
  int output_soif=0;
  int quiet=0;
  int usingfile=0;
  int i;
  const char *progname=argv[0];
  
  if(argc>1 && !strcmp(argv[1], "-soif")) {
    output_soif=1;
    argv++;
    argc--;
  }
  
  if(argc>1 && !strcmp(argv[1], "-quiet")) {
    quiet=1;
    argv++;
    argc--;
  }
  
  if (argc==1) {
    fp = stdin;
    pngfile = "stdin";
  } else if (argc==2) {
    pngfile = argv[1];
    fp = fopen(pngfile, "rb");
    if (!fp)
      {
	fprintf(STDERR, "%s: Could not open input file %s\n", progname, pngfile);
	fflush(STDERR);
	perror(NULL);
	return 1;
    }
    usingfile=1;
  } else {
    fprintf(STDERR, "USAGE: %s [-soif] [-quiet] [filename]\n", progname);
    return 1;
  }
  
  if (setjmp(png_ptr.jmpbuf))
    {
      fprintf(STDERR, "%s: libpng read error for %s\n", progname, pngfile);
      fclose(fp);
      return 1;
    }
  
  png_read_init(&png_ptr);
  png_info_init(&info_ptr);
  png_init_io(&png_ptr, fp);
  png_read_info(&png_ptr, &info_ptr);
  
  /* This is necessary to zero text pointers */
  png_info_init(&end_info);
  
  /* Local function */
  png_skip_till_end(&png_ptr, &end_info);
  
  if(!output_soif && !quiet)
    fprintf(stderr, "%s: PNG meta data for %s:\n", progname, pngfile);
  
  /* Print text keywords before IDAT */
  for (i = 0; i < info_ptr.num_text; i++)
    print_kv(stdout, output_soif, info_ptr.text[i].key, info_ptr.text[i].text);
  
  /* Print text keywords after IDAT */
  for (i = 0; i < end_info.num_text; i++)
    print_kv(stdout, output_soif, end_info.text[i].key, end_info.text[i].text);
  
  /* Cleanup */
  png_read_destroy(&png_ptr, &info_ptr, &end_info);

  fclose(fp);
  
  return 0;
}

