// ------------------------------------------------------------------------
// A simple SWIG Java module
//
// ------------------------------------------------------------------------

#include <ctype.h>

#include "swig.h"
#include "java.h"

char bigbuf[1024];

static char *usage = "\
Java Options\n\
     -module name     - Set name of module\n\
     -package name    - Set name of package\n\
     -shadow          - enable shadow classes\n\n";

char *SwigTcToJniType(DataType *t) {
  if(t->is_pointer == 1) {
	  switch(t->type) {
	    case T_INT:		return "jintArray";
	    case T_SHORT:	return "jshortArray";
	    case T_LONG:	return "jlongArray";
	    case T_CHAR:	return "jstring";
	    case T_FLOAT:	return "jfloatArray";
	    case T_DOUBLE:	return "jdoubleArray";
	    case T_UINT:	return "jintArray";
	    case T_ULONG:	return "jlongArray";
	    case T_UCHAR:	return "jbyteArray";
	    case T_SCHAR:	return "jbyteArray";
	    case T_BOOL:	return "jintArray";
	    case T_VOID:	
	    case T_USER:	return "jint";
	  }
  } else if(t->is_pointer > 1) {
    return "jintArray";
  } else {
	  switch(t->type) {
	    case T_INT: return "jint";
	    case T_SHORT: return "jshort";
	    case T_LONG: return "jlong";
	    case T_CHAR: return "jbyte";
	    case T_FLOAT: return "jfloat";
	    case T_DOUBLE: return "jdouble";
	    case T_UINT: return "jint";
	    case T_ULONG: return "jlong";
	    case T_UCHAR: return "jbyte";
	    case T_SCHAR: return "jbyte";
	    case T_BOOL: return "jint";
	    case T_VOID: return "void";
	    case T_USER: return "jint";
	  }
  }
  return NULL;
}

char *SwigTcToJavaType(DataType *t) {
  if(t->is_pointer == 1) {
	  switch(t->type) {
	    case T_INT:    return "int []";
	    case T_SHORT:  return "short []";
	    case T_LONG:   return "long []";
	    case T_CHAR:   return "String";
	    case T_FLOAT:  return "float []";
	    case T_DOUBLE: return "double []";
	    case T_UINT:   return "int []";
	    case T_ULONG:  return "long []";
	    case T_UCHAR:  return "byte []";
	    case T_SCHAR:  return "byte []";
	    case T_BOOL:   return "int []";
	    case T_VOID:
	    case T_USER:   return "int";
	  }
  } else if(t->is_pointer > 1) {
    return "int []";
  } else {
	  switch(t->type) {
	    case T_INT: return "int";
	    case T_SHORT: return "short";
	    case T_LONG: return "long";
	    case T_CHAR: return "byte";
	    case T_FLOAT: return "float";
	    case T_DOUBLE: return "double";
	    case T_UINT: return "int";
	    case T_ULONG: return "long";
	    case T_UCHAR: return "byte";
	    case T_SCHAR: return "byte";
	    case T_BOOL: return "int";
	    case T_VOID: return "void";
	    case T_USER: return "int";
	  }
  }
  return NULL;
}

char *SwigTcToJniScalarType(DataType *t) {
  if(t->is_pointer == 1) {
	  switch(t->type) {
	    case T_INT: return "Int";
	    case T_SHORT: return "Short";
	    case T_LONG: return "Long";
	    case T_CHAR: return "Byte";
	    case T_FLOAT: return "Float";
	    case T_DOUBLE: return "Double";
	    case T_UINT: return "Int";
	    case T_ULONG: return "Long";
	    case T_UCHAR: return "Byte";
	    case T_SCHAR: return "Byte";
	    case T_BOOL: return "Int";
            case T_VOID: return "Int"; 
	    case T_USER: return "Int";
	  }
  } else {
    return "Int";
  }

  return NULL;
}

char *JavaTypeFromTypemap(char *op, char *lang, DataType *t, char *pname) {
  char *tm;
  char *c = bigbuf;
  if(!(tm = typemap_lookup(op, lang, t, pname, "", ""))) return NULL;
  while(*tm && (isspace(*tm) || *tm == '{')) tm++;
  while(*tm && *tm != '}') *c++ = *tm++;
  *c='\0';

  return strdup(bigbuf);
}

char *makeValidJniName(char *name) {
  char *c = name;
  char *b = bigbuf;

  while(*c) {
    *b++ = *c;
    if(*c == '_') *b++ = '1';
    c++;
  }
  *b = '\0';

  return strdup(bigbuf);
}

// ---------------------------------------------------------------------
// JAVA::parse_args(int argc, char *argv[])
//
// Parse my command line options and initialize by variables.
// ---------------------------------------------------------------------

void JAVA::parse_args(int argc, char *argv[]) {

  // Look for certain command line options
  for (int i = 1; i < argc; i++) {
    if (argv[i]) {
      if (strcmp(argv[i],"-module") == 0) {
	if (argv[i+1]) {
	  set_module(argv[i+1],0);
	  mark_arg(i);
	  mark_arg(i+1);
	  i++;
	} else {
	  arg_error();
	}
      } else if (strcmp(argv[i],"-package") == 0) {
	if (argv[i+1]) {
	  package = new char[strlen(argv[i+1])+1];
          strcpy(package, argv[i+1]);
	  mark_arg(i);
	  mark_arg(i+1);
	  i++;
	} else {
	  arg_error();
	}
      } else if (strcmp(argv[i],"-shadow") == 0) {
	mark_arg(i);
        shadow = 1;
      } else if (strcmp(argv[i],"-help") == 0) {
	fprintf(stderr,"%s\n", usage);
      }
    }
  }

  // Set location of SWIG library
  strcpy(LibDir,"java");

  // Add a symbol to the parser for conditional compilation
  add_symbol("SWIGJAVA",0,0);

  // Add typemap definitions
  typemap_lang = "java";
}

// ---------------------------------------------------------------------
// void JAVA::parse()
//
// Start parsing an interface file for JAVA.
// ---------------------------------------------------------------------

void JAVA::parse() {

  fprintf(stderr,"Generating wrappers for Java\n");

  headers();       // Emit header files and other supporting code

  // Tell the parser to first include a typemap definition file

  if (include_file("typemaps.i") == -1) {
    fprintf(stderr,"Unable to find typemaps.i!\n");
    SWIG_exit(1);
  }
  yyparse();       // Run the SWIG parser
}

// ---------------------------------------------------------------------
// JAVA::set_module(char *mod_name,char **mod_list)
//
// Sets the module name.  Does nothing if it's already set (so it can
// be overriddent as a command line option).
//
// mod_list is a NULL-terminated list of additional modules.  This
// is really only useful when building static executables.
//----------------------------------------------------------------------

void JAVA::set_module(char *mod_name, char **mod_list) {
  if (module) return;
  module = new char[strlen(mod_name)+1];
  strcpy(module,mod_name);
}

// ---------------------------------------------------------------------
// JAVA::headers(void)
//
// Generate the appropriate header files for JAVA interface.
// ----------------------------------------------------------------------

void JAVA::headers(void)
{
  emit_banner(f_header);               // Print the SWIG banner message
  fprintf(f_header,"/* Implementation : Java*/\n\n");

  // Include header file code fragment into the output
  if (insert_file("header.swg",f_header) == -1) {
    fprintf(stderr,"Fatal Error. Unable to locate 'header.swg'.\n");
    SWIG_exit(1);
  }
}

// --------------------------------------------------------------------
// JAVA::initialize(void)
//
// Produces an initialization function.   Assumes that the init function
// name has already been specified.
// ---------------------------------------------------------------------

void JAVA::initialize() 
{
  if (!module) {
    fprintf(stderr,"SWIG : *** Error. No module name specified.\n");
    SWIG_exit(1);
  }

  if(package) {
    String s = package;
    s.replace(".", "_");
    s << "_";
    c_pkgstr = copy_string(s.get());

    String s2 = package;
    s2.replace(".", "/");
    s2 << "/";
    jni_pkgstr = copy_string(s2.get());
  } else {
    package = c_pkgstr = jni_pkgstr = "";
  }
    
  sprintf(bigbuf, "Java_%s%s_%%f", c_pkgstr, module);
  name_register("wrapper", copy_string(bigbuf));
  name_register("set", "set_%v");
  name_register("get", "get_%v");
  name_register("member", "%c_%f"); 
 
  // Generate the java class
  sprintf(bigbuf, "%s.java", module);
  if((f_java = fopen(bigbuf, "w")) == 0) {
    fprintf(stderr,"Unable to open %s\n", bigbuf);
    SWIG_exit(1);
  }

  if(package && *package)
	  fprintf(f_java, "package %s;\n", package);
  fprintf(f_java, "public class %s {\n", module);
}

// ---------------------------------------------------------------------
// JAVA::close(void)
//
// Wrap things up.  Close initialization function.
// ---------------------------------------------------------------------

void JAVA::close(void)
{
  // Finish off the java class
  fprintf(f_java, "}\n");
  fclose(f_java);
}

// ----------------------------------------------------------------------
// JAVA::create_command(char *cname, char *iname)
//
// Creates a JAVA command from a C function.
// ----------------------------------------------------------------------

void JAVA::create_command(char *cname, char *iname) {
}

// ----------------------------------------------------------------------
// JAVA::create_function(char *name, char *iname, DataType *d, ParmList *l)
//
// Create a function declaration and register it with the interpreter.
// ----------------------------------------------------------------------

void JAVA::create_function(char *name, char *iname, DataType *t, ParmList *l)
{
  String           source, target;
  char             *tm;
  String           cleanup, outarg;
  String	   body;

  // A new wrapper function object
  WrapperFunction  f;

  // Make a wrapper name for this function
  
  char *jniname = makeValidJniName(iname);
  char *wname = name_wrapper(jniname,package);
  free(jniname);
  char *jnirettype = JavaTypeFromTypemap("jni", typemap_lang, t, iname);
  if(!jnirettype) jnirettype = SwigTcToJniType(t);
  char *javarettype = JavaTypeFromTypemap("jtype", typemap_lang, t, iname);
  if(!javarettype) javarettype = SwigTcToJavaType(t);

  if(t->type != T_VOID || t->is_pointer) {
    f.add_local(jnirettype, "_jresult", "NULL");
  }

  fprintf(f_java, "  public final static native %s %s(", javarettype, iname);
  if(shadow && member_func) {
    String member_name = "";
    if(strcmp(iname, name_set(name_member(shadow_name, shadow_classname))) == 0)
	member_name << "set";
    else member_name << "get";
    member_name << (char) toupper((int) *shadow_name);
    member_name << shadow_name + 1;
    fprintf(f_shadow, "  public %s %s(", javarettype, (char *) member_name);
    body << tab4 << "return " << module << "." << iname << "(self()";
  }

  f.def << "JNIEXPORT " << jnirettype << " JNICALL " << wname << "(JNIEnv *jenv, jclass jcls ";

  // Emit all of the local variables for holding arguments.
  int pcount = emit_args(t,l,f);

  // Now walk the function parameter list and generate code to get arguments
  for (int i = 0; i < pcount ; i++) {
    Parm *p = l->get(i);         // Get the ith argument
    char *target_copy = NULL;
    char *target_length = NULL;
    char *local_i = NULL;
    source = "";
    target = "";

    // Produce string representation of source and target arguments
    source << "jarg" << i;
    target << "_arg" << i;

      char *jnitype = JavaTypeFromTypemap("jni", typemap_lang, p->t, p->name);
      if(!jnitype) jnitype = SwigTcToJniType(p->t);
      char *jtype = JavaTypeFromTypemap("jtype", typemap_lang, p->t, p->name);
      if(!jtype) jtype = SwigTcToJavaType(p->t);

      // Add to java function header
      if(shadow && member_func) {
        if(i > 0) {
          fprintf(f_shadow, "%s %s", jtype, (char *) source);
          if(i != pcount-1) fprintf(f_shadow, ", ");
          body << ", " << source;
        }
      }

      fprintf(f_java, "%s %s", jtype, (char *) source);
      if(i != pcount-1) fprintf(f_java, ", ");

      // Add to Jni function header
      f.def << ", " << jnitype << " " << source;
  
      // Get typemap for this argument
      tm = typemap_lookup("in",typemap_lang,p->t,p->name,source,target,&f);
      if (tm) {
	      f.code << tm << "\n";
	      f.code.replace("$arg",source);   // Perform a variable replacement
      } else {
        if(!p->t->is_pointer ||
	   (p->t->type == T_VOID && p->t->is_pointer == 1) ||
	   (p->t->type == T_USER && p->t->is_pointer == 1)) {
          f.code << tab4 << target << " = " << p->t->print_cast() << source << ";\n";
        } else {
          if(p->t->type == T_CHAR && p->t->is_pointer == 1) {
              f.code << tab4 << target << " = (char *) (*jenv)->GetStringUTFChars(jenv, " << source << ", 0);\n";
          } else {
            char *scalarType = SwigTcToJniScalarType(p->t);
            char *cptrtype = p->t->print_type();
            p->t->is_pointer--;
            char *basic_jnitype = (p->t->is_pointer > 0) ? "jint" : SwigTcToJniType(p->t);
            char *ctype = p->t->print_type();
            p->t->is_pointer++;

	    f.code << "if(sizeof(" << ctype << ") == sizeof(" << basic_jnitype << ")) {\n";
            f.code << tab4 << target << " = " << p->t->print_cast()
                 << "(*jenv)->Get" << scalarType << "ArrayElements(jenv," << source << ", 0);\n";
            f.code << "} else {\n";
            String basic_jniptrtype = basic_jnitype;
            basic_jniptrtype << "*";
            String source_length = "(*jenv)->GetArrayLength(jenv, ";
	    source_length << source << ")";

            target_copy = copy_string(f.new_local((char *) basic_jniptrtype, target, NULL));
            target_length = copy_string(f.new_local("jsize", target, source_length));
            if(local_i == NULL) local_i = copy_string(f.new_local("int", "i", NULL));

	    f.code << tab4 << target_copy << " = " << "(*jenv)->Get" << scalarType << "ArrayElements(jenv," << source << ", 0);\n";
	    f.code << tab4 << target << " = " << p->t->print_cast() << " malloc(" << target_length << " * sizeof(" << ctype << "));\n";
	    f.code << tab4 << "for(i=0; i<" << target_length << "; i++)\n";
            p->t->is_pointer--;
	    f.code << tab8 << target << "[i] = " << p->t->print_cast() <<
		target_copy << "[i];\n"; 
	    p->t->is_pointer++;
            f.code << "}\n\n";
          }
        }
      }

    // Check to see if there was any sort of a constaint typemap
    if ((tm = typemap_lookup("check",typemap_lang,p->t,p->name,source,target))) {
      // Yep.  Use it instead of the default
      f.code << tm << "\n";
      f.code.replace("$arg",source);
    }

    // Check if there was any cleanup code (save it for later)
    if ((tm = typemap_lookup("freearg",typemap_lang,p->t,p->name,source,target))) {
      // Yep.  Use it instead of the default
      cleanup << tm << "\n";
      cleanup.replace("$arg",source);
    }
    if ((tm = typemap_lookup("argout",typemap_lang,p->t,p->name,source,target))) {
      // Yep.  Use it instead of the default
      outarg << tm << "\n";
      outarg.replace("$arg",source);
    } else {
       if(p->t->is_pointer && p->t->type != T_USER &&  p->t->type != T_VOID) {
         if(p->t->type == T_CHAR && p->t->is_pointer == 1) {
            outarg << tab4 << "(*jenv)->ReleaseStringUTFChars(jenv, " << source << ", " << target << ");\n";
         } else {
            char *scalarType = SwigTcToJniScalarType(p->t);
            char *cptrtype = p->t->print_type();
            p->t->is_pointer--;
            char *basic_jnitype = (p->t->is_pointer > 0) ? "jint" : SwigTcToJniType(p->t);
            char *ctype = p->t->print_type();
            p->t->is_pointer++;

	    outarg << "if(sizeof(" << ctype << ") == sizeof(" << basic_jnitype << ")) {\n";
            outarg << tab4 << "(*jenv)->Release" << scalarType << "ArrayElements(jenv, " << source << ", (" << basic_jnitype << " *) " << target << ", 0);\n";
            outarg << "} else {\n";
	    outarg << tab4 << "for(i=0; i<" << target_length << "; i++)\n";
	    outarg << tab8 << target_copy << "[i] = (" << basic_jnitype << ") " << target << "[i];\n"; 
            outarg << tab4 << "(*jenv)->Release" << scalarType << "ArrayElements(jenv," << source << ", " << target_copy << ", 0);\n";
	    outarg << tab4 << "free(" << target << ");\n";
            outarg << "}\n\n";

            free(target_copy);
            free(target_length);
            free(local_i);
         }
       }
    }
  }

  fprintf(f_java, ");\n");
  if(shadow && member_func) {
    fprintf(f_shadow, ") {\n");
    body << ")";
    fprintf(f_shadow, "%s;\n  }\n\n", (char *) body);
  }
  f.def << ") {";

  // Now write code to make the function call

  emit_func_call(name,t,l,f);

  // Return value if necessary 

  if ((t->type != T_VOID) || (t->type == T_USER && t->is_pointer)) {
    if ((tm = typemap_lookup("out",typemap_lang,t,name,source,target))) {
      // Yep.  Use it instead of the default
      f.code << tm << "\n";
    }
  }

  // Dump argument output code;
  f.code << outarg;

  // Dump the argument cleanup code
  f.code << cleanup;

  // Look for any remaining cleanup

  if (NewObject) {
    if ((tm = typemap_lookup("newfree",typemap_lang,t,iname,"_result",""))) {
      f.code << tm << "\n";
    }
  }

  if(t->type != T_VOID || t->is_pointer) {
    if ((tm = typemap_lookup("ret",typemap_lang,t,iname,"_result","_jresult", NULL))) {
      f.code << tm << "\n";
    } else {
      if(!t->is_pointer || (t->type == T_USER && t->is_pointer == 1)) {
          f.code << tab4 << "_jresult = (" << jnirettype << ") _result;\n";
      } else {
        if(t->type == T_CHAR && t->is_pointer == 1) {
          f.code << tab4 << "_jresult = (jstring) (*jenv)->NewStringUTF(jenv, _result);\n";
        } else {
            fprintf(stderr,"%s : Line %d. No return typemap for datatype %s\n",
              input_file,line_number,t->print_type());

        }
      }
    }
  }

  // Wrap things up (in a manner of speaking)
  if(t->type != T_VOID)
    f.code << tab4 << "return _jresult;\n";
  f.code << "}\n";

  // Substitute the cleanup code (some exception handlers like to have this)
  f.code.replace("$cleanup",cleanup);
 
  // Emit the function
  
  f.print(f_wrappers);
  
  
  // If there's a documentation entry, produce a usage string
  
  if (doc_entry) {

    static DocEntry *last_doc_entry = 0;

    // Use usage as description
    doc_entry->usage << iname;

    // Set the cinfo field to specific a return type 

    if (last_doc_entry != doc_entry) {
      doc_entry->cinfo << "returns " << t->print_type();
      last_doc_entry = doc_entry;
    }
  }

}

// -----------------------------------------------------------------------
// JAVA::link_variable(char *name, char *iname, DataType *t)
//
// Create a JAVA link to a C variable.
// -----------------------------------------------------------------------

void JAVA::link_variable(char *name, char *iname, DataType *t)
{
  emit_set_get(name,iname, t);
}

// -----------------------------------------------------------------------
// JAVA::declare_const(char *name, char *iname, DataType *type, char *value)
// ------------------------------------------------------------------------

void JAVA::declare_const(char *name, char *iname, DataType *type, char *value) {
  char *tm;
  FILE *jfile;
  char *jname;

  if(shadow && member_func) {
    jfile = f_shadow;
    jname = shadow_name;
  } else {
    jfile = f_java;
    jname = name;
  }

  if ((tm = typemap_lookup("const",typemap_lang,type,name,name,iname))) {
    String str = tm;
    str.replace("$value",value);
    fprintf(jfile,"  %s\n\n", (char *) str);
  } else {
    if((type->is_pointer == 0)) {
      char *jtype = typemap_lookup("jtype", typemap_lang, type, name, name, iname);
      if(!jtype) jtype = SwigTcToJavaType(type);
      fprintf(jfile, "  public final static %s %s = %s;\n\n", jtype, jname, value);
    } else {
      if(type->type == T_CHAR && type->is_pointer == 1) {
        fprintf(jfile, "  public final static String %s = \"%s\";\n\n", jname, value);
      } else {
        emit_set_get(name,iname, type);
      }
    }
  }
}

void emit_shadow_banner(FILE *f) {
  fprintf(f, "/*\n");
  fprintf(f, " *\n");
  fprintf(f, " * This file was automatically generated by :\n");
  fprintf(f, " * Simplified Wrapper and Interface Generator (SWIG)\n");
  fprintf(f, " * Version 1.1  (Final)\n");
  fprintf(f, " *\n");
  fprintf(f, " * Portions Copyright (c) 1995-1997\n");
  fprintf(f, " * The University of Utah and The Regents of the University of California.\n");
  fprintf(f, " * Permission is granted to distribute this file in any manner provided\n");
  fprintf(f, " * this notice remains intact.\n");
  fprintf(f, " *\n");
  fprintf(f, " * Do not make changes to this file--changes will be lost!\n");
  fprintf(f, " *\n");
  fprintf(f, " */\n\n\n");
}

// ---------------------------------------------------------------------
// C++ Handling
//
// The following functions provide some support for C++ classes and
// C structs.
// ---------------------------------------------------------------------

void JAVA::cpp_open_class(char *classname, char *rename, char *ctype, int strip) {
  this->Language::cpp_open_class(classname,rename,ctype,strip);

  if(!shadow) return;

  if(rename)
    shadow_classname = copy_string(rename);
  else shadow_classname = copy_string(classname);

  sprintf(bigbuf, "%s.java", shadow_classname);
  if(!(f_shadow = fopen(bigbuf, "w"))) {
    fprintf(stderr, "Unable to create shadow class file: %s\n", bigbuf);
  }

  emit_shadow_banner(f_shadow);

  if(*package)
	fprintf(f_shadow, "package %s;\n\n", package);
  else fprintf(f_shadow, "import %s;\n\n", module);

  shadow_classdef = "";
  shadow_classdef << "public class " << shadow_classname << " %BASECLASS% " << "{\n";

  shadow_baseclass = "";
  shadow_classdef_emitted = 0;
  shadow_classdef << "  public int self = 0;\n\n";
  shadow_classdef << "  public int self() {\n" << tab4 << "return self;\n" << "  };\n\n";
  shadow_classdef << "  public " << shadow_classname << "(int obj) {\n" <<
	tab4 << "self = obj;\n  }\n\n";
}

void JAVA::emit_shadow_classdef() {
  if(*shadow_baseclass) {
    sprintf(bigbuf, "extends %s", shadow_baseclass);
    shadow_classdef.replace("%BASECLASS%", bigbuf);
  } else shadow_classdef.replace("%BASECLASS%", "");

  fprintf(f_shadow, (char *) shadow_classdef);
  shadow_classdef_emitted = 1;
}

void JAVA::cpp_close_class() {
  this->Language::cpp_close_class();
  if(!shadow) return;
  if(!shadow_classdef_emitted) emit_shadow_classdef();

  fprintf(f_shadow, "}\n");
  fclose(f_shadow);
  f_shadow = NULL;

  free(shadow_classname);
  shadow_classname = NULL;
}

void JAVA::cpp_member_func(char *name, char *iname, DataType *t, ParmList *l) {
  String           arg;
  String           nativecall  = "";

  this->Language::cpp_member_func(name,iname,t,l);

  if(!shadow) return;
  if(!shadow_classdef_emitted) emit_shadow_classdef();

  char *javarettype = JavaTypeFromTypemap("jtype", typemap_lang, t, iname);
  if(!javarettype) javarettype = SwigTcToJavaType(t);

  fprintf(f_shadow, "  public %s %s(", javarettype, iname);

  if((t->type != T_VOID || t->is_pointer))
    nativecall << "return ";
  nativecall << module << "." << name_member(iname, shadow_classname) << "(self()";

  int pcount = l->nparms;

  for (int i = 0; i < pcount ; i++) {
    Parm *p = l->get(i);         // Get the ith argument
    // Produce string representation of source and target arguments
    if(p->name && *(p->name))
	arg = p->name;
    else {
      arg = "arg";
      arg << i;
    }

      char *jtype = 0;
      if(p->t->type == T_USER && (jtype = (char *) shadow_classes.lookup(p->t->name))) {
        nativecall << ", " << arg << ".self()";
      } else nativecall << ", " << arg;

      if(!jtype) jtype = JavaTypeFromTypemap("jtype", typemap_lang, p->t, p->name);
      if(!jtype) jtype = SwigTcToJavaType(p->t);

      // Add to java function header
      fprintf(f_shadow, "%s %s", jtype, (char *) arg);
      if(i != pcount-1) {
        fprintf(f_shadow, ", ");
      }
  }

  fprintf(f_shadow, ") {\n");
  nativecall << ");\n";
  fprintf(f_shadow, "\t%s\n", (char *) nativecall);
  fprintf(f_shadow, "  }\n\n");
}

void JAVA::cpp_constructor(char *name, char *iname, ParmList *l) {
  this->Language::cpp_constructor(name,iname,l);

  if(!shadow) return;
  if(!shadow_classdef_emitted) emit_shadow_classdef();

  String nativecall = "";
  String arg;

  fprintf(f_shadow, "  public %s(", shadow_classname);

  nativecall << "self = " << module << "." << name_construct(shadow_classname) << "(";

  int pcount = l->nparms;

  for (int i = 0; i < pcount ; i++) {
    Parm *p = l->get(i);         // Get the ith argument
    // Produce string representation of source and target arguments
    if(p->name && *(p->name))
	arg = p->name;
    else {
      arg = "arg";
      arg << i;
    }

      char *jtype = JavaTypeFromTypemap("jtype", typemap_lang, p->t, p->name);
      if(!jtype) jtype = SwigTcToJavaType(p->t);

      // Add to java function header
      fprintf(f_shadow, "%s %s", jtype, (char *) arg);
      nativecall << arg;
      if(i != pcount-1) {
        nativecall << ", ";
        fprintf(f_shadow, ", ");
      }
  }

  fprintf(f_shadow, ") {\n");
  nativecall << ");\n";
  fprintf(f_shadow, "\t%s\n", (char *) nativecall);
  fprintf(f_shadow, "  }\n\n");
}

void JAVA::cpp_destructor(char *name, char *newname) {
  this->Language::cpp_destructor(name,newname);

  if(!shadow) return;
  if(!shadow_classdef_emitted) emit_shadow_classdef();

  char *realname = (newname) ? newname : name;

  fprintf(f_shadow, "  protected void finalize() {\n");
  fprintf(f_shadow, "\t%s.%s(self());\n", module, name_destroy(realname));
  fprintf(f_shadow, "  }\n\n");
}

void JAVA::cpp_class_decl(char *name, char *rename, char *type) {
  this->Language::cpp_class_decl(name,rename, type);

  if(!shadow) return;

  char *realname = (rename) ? rename : name;

  shadow_classes.add(name, copy_string(realname));
}

void JAVA::cpp_inherit(char **baseclass, int) {
  this->Language::cpp_inherit(baseclass, 0);

  if(!shadow) return;

  shadow_baseclass = copy_string(*baseclass);
}

void JAVA::cpp_variable(char *name, char *iname, DataType *t) {
  if(shadow && !shadow_classdef_emitted) emit_shadow_classdef();

  if(shadow) member_func = 1;
  shadow_name = copy_string((iname) ? iname : name);
  this->Language::cpp_variable(name, iname, t);
  member_func = 0;
}

void JAVA::cpp_declare_const(char *name, char *iname, DataType *type, char *value) {
  if(shadow && !shadow_classdef_emitted) emit_shadow_classdef();

  if(shadow) member_func = 1;
  shadow_name = copy_string((iname) ? iname : name);
  this->Language::cpp_declare_const(name, iname, type, value);
  member_func = 0;
}

