/****************************************************************************
                          INTERNATIONAL AVS CENTRE
           (This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Centre, University of Manchester, the AVS Consortium and
the individual  submitting the module and files associated with said module
provide absolutely NO WARRANTY OF ANY KIND with respect to this software.  The
entire risk as to the quality and performance of this software is with the
user.  IN NO EVENT WILL The International AVS Centre, University of Manchester,
the AVS Consortium and the individual submitting the module and files
associated with said module BE LIABLE TO ANYONE FOR ANY DAMAGES ARISING FROM
THE USE OF THIS SOFTWARE, INCLUDING, WITHOUT LIMITATION, DAMAGES RESULTING FROM
LOST DATA OR LOST PROFITS, OR ANY SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@iavsc.org.

Please send AVS module bug reports to avs@iavsc.org.

***************************************************************************/
/*   
     This software is copyright (C) 1991,  Regents  of  the
University  of  California.   Anyone may reproduce 3d_axis.c,
the software in this distribution, in whole or in part, pro-
vided that:

(1)  Any copy  or  redistribution  of  3d_axis.c  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;

(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
in  3d_axis.c  under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     3d_axis.c is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
The Regents of the University of California  shall  have  no
liability  with respect to the infringement of copyrights by
3d_axis.c, or any part thereof.     

*/

/*
 *  filename 3D_axis.c
 *  Wayne Hurlbert , LBL.
 *  May 1990
 *
 *  Generate 3D coordinate axis.
 *
 *  UNIFORM data sets are mapped into a 3D space of dimensions -1 to 1, so
 *  a line drawn from say, (-1,-1,-1) to (1,1,1), will go from the far lower
 *  left to the near upper right of the data set.  Here, (-1,-1,-1) is used
 *  as the origin of the 3D axis, but ticks and labels are based on the
 *  dimensions of the data set (0 - MAXX, 0 - MAXY, 0 - MAXZ).  These min and
 *  max values may be changed by the user, but such changes will only change
 *  ticks and labels, not the drawn axis.
 **
     UNIFORM fields are now mapped into a space corresponding to
     0..size-1 for each of the data dimensions.  This mapping corresponds
     to the AVS3 world of uniform fields. 

     Access to the world of AVS3 is enabled by using the compile
     flag -DAVS3.

     Significant reworking of the loops for tick marks, labels and
     meshes have been redone to reflect this change, and also to
     support having the "min" value greater than the "max" value.

     22 May 1991 w.bethel
 *
 *  RECTILINEAR and IRREGULAR data sets, and the axes generated here, are
 *  drawn in the world space of the coordinate info stored in the data set.
 *  The origin of the axis system is placed at the minimum coordinate value
 *  of each dimension.  Note that this will probably not coincide with the
 *  world origin (0,0,0), but it is usually the far lower left corner of the
 *  data set.  This means that changing the axis minimum and maximum values
 *  will change the positions of the drawn axis, but will not change label
 *  values or the screen location of any particular point.

     19 June 92

     Changed the 6 float parameters defining the axis extents into two
     typeins, each of which must consist of 3 floats.  Improved the
     initial entry into this module so that parm_set's in a saved network
     file will not be overwritten when the module executes for the first
     time.

     26 June 92

     Misc bugs fixed.


     1 Aug 92

     1. changes to accomodate 2D rectilinear and uniform fields. x/y
     coords are either supplied or computed.  z coordinates are computed
     from the data.

     2. changed the locator used to position the orthogonal planes so
     that it is now in the coordinate system of the data rather than some
     mysterious normalized space.

     3.  changed the tick size parameter so that its range gets reset based
     upon some function of the spatial extents of the data.


     6 Feb 95

     Major changes, converted to a 2d plotting routine.  The plot is scaled 
     and positioned by input sliders, and is non-transformable.  An optional
     scalar input draws a vertical bar on the plot representing the value
     of the input.  The xy data is drawn as a curve on the plot.  The data form
     is a 2D scalar uniform field with the 1st dimension holding the x values
     and the second the y values.

     mjr

     8 Dec 95
  
     Replace the frac macro to give correct behavior for the y axis values   mjr

    20 Dec 95

     Added rbg color input for the plot line and replaced the display
     of the minima and maxima                                     mjr

 */

#include <stdio.h>
#include <math.h>

#include <avs/vex.h>
#include <avs/flow.h>
#include <avs/field.h>
#include <avs/geom.h>
#include <avs/geomdata.h>

#define max(a, b) ((a) > (b) ? (a) : (b))
#define min(a, b) ((a) < (b) ? (a) : (b))

#define pos(val,minval,maxval,orig,ext) ((orig) + ((((val)-(minval))/((maxval)-(minval))))*(ext))

/*  the original frac macro is not correct, we need to 
    compute the fraction of the extent represented by val    mjr  

 define frac(val,minval,maxval,ext) (((val)-(minval))/((maxval)-(minval))*(ext))

*/

#define frac(val,minval,maxval,ext) (((val)/((maxval)-(minval)))*(ext))


#define MAX_TICK_PROPORTION 0.5
int first_time = TRUE;

#define AMNX "Plot Min"
#define AMXX "Plot Max"
#define INIT_STRING "null"

typedef    float XYZ[3];

/********** AVS initialization ************************************************/

AVSinit_modules()
{
    int axis_desc();
    AVSmodule_from_desc(axis_desc);
}


/******************************************************************************/

int axis_desc()
{
    int plot_compute();
    int parm;

    AVSset_module_name("2D plot", MODULE_MAPPER);

    AVScreate_input_port("Field Input", "field 2D uniform scalar float", REQUIRED);

    AVScreate_input_port("time step","integer", OPTIONAL);

    AVScreate_output_port("Plot", "geom");


    parm = AVSadd_float_parameter("x origin",-0.5,-1.,1.);
    AVSconnect_widget(parm,"slider");
    parm = AVSadd_float_parameter("y origin",-0.5,-1.,1.);
    AVSconnect_widget(parm,"slider");
    parm = AVSadd_float_parameter("x extent",1.,.01,2.);
    AVSconnect_widget(parm,"slider");
    parm = AVSadd_float_parameter("y extent",1.,.01,2.);
    AVSconnect_widget(parm,"slider");

    /*
     *  Set number of ticks per axis, and tick size.
     */
    parm = AVSadd_parameter("message1","string","Set ticks:", "","");
    AVSconnect_widget(parm,"text");
    AVSadd_parameter_prop(parm,"width","integer",4);
    parm = AVSadd_parameter("x axis ticks", "integer", 0, 0, 32);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("y axis ticks", "integer", 0, 0, 32);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_float_parameter("tick size", 0.01, 0.0, .25);
    AVSconnect_widget(parm,"slider");


    /*
     *  Turn planes on and off, and set the position of the plane in the
     *  3rd dimension. The plane mesh is determined by the tick spacing.
     */

    AVSadd_parameter("Show xy plane", "boolean", 0, 0, 1);


    /*
     *  Turn tick labels on and off, set font size, set precision, set
     *  number of ticks per label, and set which axis labels the origin.
     */
    parm = AVSadd_parameter("message3","string","Set labels:", "","");
    AVSconnect_widget(parm,"text");
    AVSadd_parameter_prop(parm,"width","integer",4);
    parm = AVSadd_float_parameter("font size", 0.07, 0.01, 0.1);
    AVSconnect_widget(parm,"slider");
 /*                                                                  */
 /*   02/06/95 NKG: change precision on label                        */
 /*   parm = AVSadd_parameter("label precision", "integer", 1, 0, 9);*/
 /*                                                                  */
    parm = AVSadd_parameter("x-label precision", "integer", 2, 0, 9);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("y-label precision", "integer", 2, 0, 9);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("font number", "integer", 1, 0, 21);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("x label density", "integer", 1, 1, 25);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("y label density", "integer", 1, 1, 25);
    AVSconnect_widget(parm,"islider");

    parm = AVSadd_parameter("origin labels","string","y","","");
    AVSconnect_widget(parm,"typein");
    AVSadd_parameter("Show labels","boolean", 0, 0, 1);


    /*
     *  Show the min and max values of the plot, as defined by the data
     *  set. 
     */

    parm = AVSadd_parameter("info","string_block","Axis limits, min  max","","");
    AVSconnect_widget(parm,"text_block");
    AVSadd_parameter_prop(parm,"width","integer",4);
    AVSadd_parameter_prop(parm,"height","integer",3);

    parm = AVSadd_float_parameter("plot R",1.,0.,1.);
    AVSconnect_widget(parm,"typein_real");
    AVSadd_parameter_prop(parm,"width","integer",2);

    parm = AVSadd_float_parameter("plot G",1.,0.,1.);
    AVSconnect_widget(parm,"typein_real");
    AVSadd_parameter_prop(parm,"width","integer",2);

    parm = AVSadd_float_parameter("plot B",1.,0.,1.);
    AVSconnect_widget(parm,"typein_real");
    AVSadd_parameter_prop(parm,"width","integer",2);


    AVSset_compute_proc(plot_compute);
}


/******************************************************************************/

plot_compute(inf, t_step, outg, xorig, yorig, xext, yext,
                         mess1, x_ticks, y_ticks, t_size,
                         xy_pl, 
                         mess3, font_size, x_precision, y_precision,
                                font_no,
                                x_lab_den, y_lab_den,
                                origin_labels, show_labels,
                                axis_info,R,G,B)
AVSfield_float *inf;
int     t_step;
GEOMedit_list **outg;
float *xorig, *yorig, *xext, *yext;  /* origin locations and extents */
char *mess1,*mess3;                  /* Dummy messages for titles.  */
int x_ticks,y_ticks;                 /* Number of ticks per axis.   */
float *t_size;                       /* Tick size.                  */
int xy_pl;                           /* Plane on/off switchs.       */
float *font_size;                    /* Size of label font.         */
int x_precision, y_precision;        /* Label precision. */
int font_no;                         /* mysterious avs font numbers 0..21 */
int show_labels;                     /* Turn labels on and off. */
char *origin_labels;                 /* Axis whose labels are at origin, */
int x_lab_den,y_lab_den;             /* Number of ticks per label.  */
char *axis_info;
float *R,*G,*B;                      /* R,G,B values for the line plot */
{
    GEOMobj *plot_obj, *line_obj, *mark_obj;
    GEOMobj *label_x_obj;
    GEOMobj *label_y_obj;
    int label_flg;
    float axis_pts[6];
    int i, size, num_ticks[2], which_planes[1], label_density[2];
    int precision[2];
    float where[1];
    float orig[2], ext[2];
    float *t;
    char buffer[128],*c1,*c2,*c3,*c4;
    double foo;
    extern double strtod();
    float min_extents[2],max_extents[2];
    int npoints;
    int two_d_field=0;
    double fake_zlimits[2];
    float computed_tick_size,temp,temp1,temp2;
    XYZ    *plot_line=NULL, *plot_colors=NULL;
    float  min[1],max[1];


    *outg     = (GEOMedit_list *)GEOMinit_edit_list(*outg);

    plot_obj    = GEOMcreate_obj(GEOM_POLYTRI, GEOM_NULL);
    line_obj    = GEOMcreate_obj(GEOM_POLYTRI, GEOM_NULL);
    mark_obj    = GEOMcreate_obj(GEOM_POLYTRI, GEOM_NULL);
    label_flg   = GEOMcreate_label_flags(font_no,0,0,0,GEOM_LABEL_CENTER,0);
    label_x_obj = GEOMcreate_label(GEOM_NULL,label_flg);
    label_flg   = GEOMcreate_label_flags(font_no,0,0,0,GEOM_LABEL_RIGHT,0);
    label_y_obj = GEOMcreate_label(GEOM_NULL,label_flg);
     

    /* Put this stuff in vectors for ease of passing. */
    num_ticks[0] = x_ticks;
    num_ticks[1] = y_ticks;

    AVSmodify_parameter("x label density",AVS_MAXVAL,NULL,NULL,x_ticks+1);
    AVSmodify_parameter("y label density",AVS_MAXVAL,NULL,NULL,y_ticks+1);


    where[0] = 0.0;
    which_planes[0] = xy_pl;
    label_density[0] = x_lab_den;
    label_density[1] = y_lab_den;

    precision[0] = x_precision;
    precision[1] = y_precision;

    orig[0] = *xorig;
    orig[1] = *yorig;
    ext[0] = *xext;
    ext[1] = *yext;

    AVSfield_get_minmax(inf,min,max);

    axis_pts[0] = I2D(inf,0,0);
    axis_pts[1] = I2D(inf,0,MAXY(inf)-1);

    axis_pts[2] = min[0];
    axis_pts[3] = max[0];

    axis_pts[4] = 0.0;
    axis_pts[5] = 0.0;

            
            /* now, sprintf the floats into a char buffer for use
               on setting the module widgets to reflect wtf is going on */
            
            sprintf(buffer,"Axis extents (min,max)\nX    %g    %g\nY    %g    %g",
                  axis_pts[0],axis_pts[1],axis_pts[2],axis_pts[3]);
            AVSmodify_parameter("info",AVS_VALUE,buffer,"","");


    make_axis(plot_obj,axis_pts, orig, ext);

    make_ticks(plot_obj,axis_pts,num_ticks,t_size,inf->uniform, orig, ext);

    make_planes(plot_obj,axis_pts,num_ticks,which_planes,where,
                inf->uniform, orig, ext);

    make_labels(axis_pts,num_ticks,inf->uniform, label_x_obj,
                label_y_obj,show_labels,precision,
                label_density,font_size,origin_labels, orig, ext);


    if(plot_line) free(plot_line);
    if(plot_colors) free(plot_colors);
    
    plot_line = (XYZ *)malloc(MAXY(inf)*sizeof(XYZ));
    plot_colors = (XYZ *)malloc(MAXY(inf)*sizeof(XYZ));

    for( i = 0; i < MAXY(inf); i++ )
     {
      plot_line[i][0] = pos(I2D(inf,0,i),axis_pts[0],axis_pts[1],orig[0],ext[0]);
      plot_line[i][1] = pos(I2D(inf,1,i),axis_pts[2],axis_pts[3],orig[1],ext[1]);
      plot_line[i][2] = 0.0;
      plot_colors[i][0] = *R;
      plot_colors[i][1] = *G;
      plot_colors[i][2] = *B;
     }

    GEOMadd_polyline(line_obj,plot_line,plot_colors,MAXY(inf)-1,GEOM_COPY_DATA);

    if( t_step )
    {
     plot_line[0][0] = pos(I2D(inf,0,t_step),axis_pts[0],axis_pts[1],orig[0],ext[0]);
     plot_line[0][1] = pos(I2D(inf,1,t_step),axis_pts[2],axis_pts[3],orig[1],ext[1]);
     plot_line[0][2] = 0.;
     plot_line[1][0] = pos(I2D(inf,0,t_step),axis_pts[0],axis_pts[1],orig[0],ext[0]);
     plot_line[1][1] = pos(axis_pts[2],axis_pts[2],axis_pts[3],orig[1],ext[1]);
     plot_line[1][2] = 0.;

     plot_colors[0][0] = 0.0;
     plot_colors[0][1] = *G;
     plot_colors[0][2] = *B;
     plot_colors[1][0] = 0.0;
     plot_colors[1][1] = *G;
     plot_colors[1][2] = *B;

     GEOMadd_disjoint_line(mark_obj,plot_line,plot_colors,2,GEOM_COPY_DATA);
    }


    
    /* Add into the edit list */
    GEOMedit_geometry(*outg, "3D_axis", plot_obj);
    GEOMedit_geometry(*outg, "3D_axis", line_obj);
    if( t_step ) GEOMedit_geometry(*outg, "3D_axis", mark_obj);
    GEOMedit_geometry(*outg, "3D_axis", label_x_obj);
    GEOMedit_geometry(*outg, "3D_axis", label_y_obj);

    /* now set it for non-transformable */

    GEOMedit_transform_mode(*outg,"3D_axis","locked",0);

    /* Now kill it all. */
    GEOMdestroy_obj(plot_obj);
    GEOMdestroy_obj(line_obj);
    if( t_step ) GEOMdestroy_obj(mark_obj);
    GEOMdestroy_obj(label_x_obj);
    GEOMdestroy_obj(label_y_obj);

    return(1);
}


/******************************************************************************/

make_axis(obj,axis_pts, orig, ext)
GEOMobj *obj;
float axis_pts[6];
float orig[2], ext[2];
/*
 *  This function loads 3D coordinates for 3 lines into axis_lines[].
 *  These lines will then be added to a list of ines for display
 *  by GEOMadd_disjoint_line().  axis_lines[] stores the lines as
 *  x, y, and z axis, where each axis is specified by the three
 *  coordinate values for its 2 endpoints (axis_lines[0][?] and
 *  axis_lines[1][?] make up the x axis).  Colors are stored in
 *  axis_color by rgb values in a similar way.
 */
{
    float axis_color[4][3],
          axis_lines[4][3];

    axis_lines[0][0] = pos(axis_pts[0],axis_pts[0],axis_pts[1],orig[0],ext[0]);    /* x axis */
    axis_lines[0][1] = pos(axis_pts[2],axis_pts[2],axis_pts[3],orig[1],ext[1]);
    axis_lines[0][2] = axis_pts[4];

    axis_lines[1][0] = pos(axis_pts[1],axis_pts[0],axis_pts[1],orig[0],ext[0]);
    axis_lines[1][1] = pos(axis_pts[2],axis_pts[2],axis_pts[3],orig[1],ext[1]);
    axis_lines[1][2] = axis_pts[4];

    axis_lines[2][0] = pos(axis_pts[0],axis_pts[0],axis_pts[1],orig[0],ext[0]);    /* y axis */
    axis_lines[2][1] = pos(axis_pts[2],axis_pts[2],axis_pts[3],orig[1],ext[1]);
    axis_lines[2][2] = axis_pts[4];

    axis_lines[3][0] = pos(axis_pts[0],axis_pts[0],axis_pts[1],orig[0],ext[0]);
    axis_lines[3][1] = pos(axis_pts[3],axis_pts[2],axis_pts[3],orig[1],ext[1]);
    axis_lines[3][2] = axis_pts[4];


    axis_color[0][0] = 0.0;
    axis_color[0][1] = 1.0;
    axis_color[0][2] = 0.0;

    axis_color[1][0] = 0.0;
    axis_color[1][1] = 1.0;
    axis_color[1][2] = 0.0;

    axis_color[2][0] = 0.0;
    axis_color[2][1] = 1.0;
    axis_color[2][2] = 0.0;

    axis_color[3][0] = 0.0;
    axis_color[3][1] = 1.0;
    axis_color[3][2] = 0.0;

    GEOMadd_disjoint_line(obj,axis_lines,axis_color,4,GEOM_COPY_DATA);
}

make_ticks(plot_obj,axis_pts,num_ticks,tick_size,data_type,orig,ext)
GEOMobj *plot_obj;
float axis_pts[6];
int num_ticks[2];
float *tick_size;
int data_type;
float orig[2],ext[2];
/*
 *  This function places ticks along each axis.  There are actually 2
 *  marks, one parallel to each of the other 2 axis.  We get the
 *  increment value and work our way along the axis making the 2 marks.
 *
 *  In the case of a UNIFORM data set, axis_pts[] contains the dimensions
 *  of the data set, but the actual plotting is normalized to the interval
 *  -1 to 1.  So we first save the dimensions and then for each axis get
 *  an increment for the dimensions, normalize the increment, and change
 *  axis_pts[] to normalized dimensions (-1 to 1).  (After each axis is
 *  done we restore axis_pts[] to the original dimensions so we can do
 *  the next one.)
 */
{
    float get_increment();
    float increment;
    float loc;
    float t_size = *tick_size;
    float axis_tick[4][3];
    float tick_color[4][3];
    int i;


/* Ticks on x axis. */

    if (num_ticks[0]) {
        increment = get_increment(num_ticks[0],axis_pts[0],axis_pts[1]);

        increment = frac(increment,axis_pts[0],axis_pts[1],ext[0]);

        set_tick_color(tick_color, 1.0, 0.0, 0.0);

        loc = axis_pts[0];

        loc = pos(loc,axis_pts[0],axis_pts[1],orig[0],ext[0]) + increment;

        for (i=0;i<=num_ticks[0];i++)
        {
/*            axis_tick[0][0] = loc;
            axis_tick[0][1] = pos(axis_pts[2],axis_pts[2],axis_pts[3],orig[1],ext[1]);
            axis_tick[0][2] = axis_pts[4] + t_size;

            axis_tick[1][0] = loc;
            axis_tick[1][1] = pos(axis_pts[2],axis_pts[2],axis_pts[3],orig[1],ext[1]);
            axis_tick[1][2] = axis_pts[4];     */

            axis_tick[0][0] = loc;
            axis_tick[0][1] = pos(axis_pts[2],axis_pts[2],axis_pts[3],orig[1],ext[1])+t_size;
            axis_tick[0][2] = axis_pts[4];

            axis_tick[1][0] = loc;
            axis_tick[1][1] = pos(axis_pts[2],axis_pts[2],axis_pts[3],orig[1],ext[1]);
            axis_tick[1][2] = axis_pts[4];

            GEOMadd_disjoint_line(plot_obj,axis_tick,tick_color,2,GEOM_COPY_DATA);
            loc += increment;
        }
    }

/* Ticks on y axis. */

    if (num_ticks[1]) {
        increment = get_increment(num_ticks[1],axis_pts[2],axis_pts[3]);

        increment = frac(increment,axis_pts[2],axis_pts[3],ext[1]);

        set_tick_color(tick_color, 0.0, 1.0, 0.0);
        loc = axis_pts[2];

        loc = pos(loc,axis_pts[2],axis_pts[3],orig[1],ext[1]) + increment;

        for (i=0;i<=num_ticks[1];i++)
        {
/*            axis_tick[0][0] = pos(axis_pts[0],axis_pts[0],axis_pts[1],orig[0],ext[0]);
            axis_tick[0][1] = loc;
            axis_tick[0][2] = axis_pts[4] + t_size;

            axis_tick[1][0] = pos(axis_pts[0],axis_pts[0],axis_pts[1],orig[0],ext[0]);
            axis_tick[1][1] = loc;
            axis_tick[1][2] = axis_pts[4];  */

            axis_tick[0][0] = pos(axis_pts[0],axis_pts[0],axis_pts[1],orig[0],ext[0])+t_size;
            axis_tick[0][1] = loc;
            axis_tick[0][2] = axis_pts[4];

            axis_tick[1][0] = pos(axis_pts[0],axis_pts[0],axis_pts[1],orig[0],ext[0]);
            axis_tick[1][1] = loc;
            axis_tick[1][2] = axis_pts[4];

            GEOMadd_disjoint_line(plot_obj,axis_tick,tick_color,2,GEOM_COPY_DATA);
            loc += increment;
        }
    }

}

make_planes(obj,axis_pts,mesh_size,which_planes,where,data_type,orig,ext)
GEOMobj *obj;
float axis_pts[6];
int mesh_size[2];       /* this is the number of ticks */
int which_planes[1];
float where[1];
int data_type;
float orig[2],ext[2];
/*
 *  This routine creates mesh planes by extending perpendiculars to the
 *  appropriate axis at tick mark locations.  This is all very similar
 *  to the make_ticks() function.
 */
{
    float get_increment();
    float increment;
    float loc;
    float mesh_line[2][3];
    float plane_color[2][3];
    int i;
    int ct;


/* xy plane */
    if (which_planes[0]) {
        increment = get_increment(mesh_size[0],axis_pts[0],axis_pts[1]);

        increment = frac(increment,axis_pts[0],axis_pts[1],ext[0]);

        set_plane_color(plane_color, 0.0, 1.0, 1.0);
        loc = axis_pts[0];

        loc = pos(loc,axis_pts[0],axis_pts[1],orig[0],ext[0]);

        for(ct=0;ct<=mesh_size[0]+1;ct++)
        {
            mesh_line[0][0] = loc;
            mesh_line[0][1] = pos(axis_pts[2],axis_pts[2],axis_pts[3],orig[1],ext[1]);
            mesh_line[0][2] = where[1];
            mesh_line[1][0] = loc;
            mesh_line[1][1] = pos(axis_pts[3],axis_pts[2],axis_pts[3],orig[1],ext[1]);
            mesh_line[1][2] = where[1];
            GEOMadd_disjoint_line(obj,mesh_line,plane_color,2,GEOM_COPY_DATA);
            loc += increment;
        }

        increment = get_increment(mesh_size[1],axis_pts[2],axis_pts[3]);

        increment = frac(increment,axis_pts[2],axis_pts[3],ext[1]);

        set_plane_color(plane_color, 1.0, 0.3, 0.3);

        loc = axis_pts[2];

        loc = pos(loc,axis_pts[2],axis_pts[3],orig[1],ext[1]);

        for (ct=0;ct<=mesh_size[1]+1;ct++)
        {
            mesh_line[0][0] = pos(axis_pts[0],axis_pts[0],axis_pts[1],orig[0],ext[0]);
            mesh_line[0][1] = loc;
            mesh_line[0][2] = where[1];
            mesh_line[1][0] = pos(axis_pts[1],axis_pts[0],axis_pts[1],orig[0],ext[0]);
            mesh_line[1][1] = loc;
            mesh_line[1][2] = where[1];
            GEOMadd_disjoint_line(obj,mesh_line,plane_color,2,GEOM_COPY_DATA);
            loc += increment;
        }
    }

}

make_labels(axis_pts,num_ticks,data_type,label_x_obj,label_y_obj,
            show_labels,precision,label_density,
            font_size,origin_labels,orig,ext)
float axis_pts[6];
int num_ticks[2];
int data_type;
GEOMobj *label_x_obj,*label_y_obj;
int precision[2];
int label_density[2];  /* Number of ticks per label. */
float *font_size;
int show_labels;       /* Turn labels on and off. */
char *origin_labels;   /* Axis whose label is at the origin. */
float  orig[2],ext[2];
/*
 *  This routine puts labels on the various axis, at tick points,
 *  if requested.  Labels may not occur at every tick, as determined
 *  by the label_density parameter.  This is all very similar to the
 *  make_ticks() function.
 *
 *  Note that for a UNIFORM data set we label the axis according to
 *  the dimensions of the data set, or the user supplied dimensions,
 *  as opposed to the (normalized) screen size of the data.  This
 *  amounts to using the local variables u_inc and u_loc to track the
 *  the values we need for the labels.
 *
 *  origin_labels is a string which we scan for the characters x,y,
 *  and z.  For each one found we will label the origin according
 *  to the min value of that axis.  So the origin can get pretty
 *  busy.  If a particular label is called for we start labeling
 *  at axis_pts[min], otherwise we start at axis_points[min] +
 *  increment.
 */
{
    float get_increment();
    float increment,u_inc;
    float loc,u_loc,val;

    float ref_pt[3];
    float offset[3];
    float color[3];
    char num[30];
    char arg[5];
    int i;
    int x_label,y_label;
    int ct;

    /* Find out which axis labels the origin. */
    x_label = y_label = FALSE;
    for (i = 0; i < strlen(origin_labels); ++i) {
        if (origin_labels[i] == 'x')
            x_label = TRUE;
        if (origin_labels[i] == 'y')
            y_label = TRUE;
    }




    color[0] = color[1] = color[2] = 1.0;

    /* Labels on x axis. */
    if (num_ticks[0] && show_labels) {

    /* Set the number of digits to the right of the decimal via precision. */
        arg[0] = '%';
        arg[1] = '.';
        arg[2] = precision[0] + 0x30;
        arg[3] = 'f';
        arg[4] = '\0';


        increment = u_inc = get_increment(num_ticks[0],axis_pts[0],axis_pts[1]);

        loc = axis_pts[0] + ((x_label) ? 0.0 : increment);

  /* 02-06-95 NKG: change good only when draw ratio is x-axis */
        val = loc;  
  /*    val = loc + 1; */

        loc = pos(loc,axis_pts[0],axis_pts[1],orig[0],ext[0]);

        for (ct=0;ct<=(num_ticks[0]/label_density[0]+(x_label?1:0)),
                  val <= axis_pts[1] ; ct++)
        {
            ref_pt[0] = loc;
            ref_pt[1] = pos(axis_pts[2],axis_pts[2],axis_pts[3],orig[1],ext[1]);
            ref_pt[2] = axis_pts[4];
            offset[0] = 0.0;
            offset[1] = -0.06;
            offset[2] = 0.0;

            sprintf(num,arg,val);
            GEOMadd_label(label_x_obj,num,ref_pt,offset,*font_size,color,-1);

            val += increment*label_density[0];
            loc += frac(increment * label_density[0],axis_pts[0],axis_pts[1],ext[0]);

        }
        /* do the last label */

        if (x_label && (loc <= (orig[0]+ext[0])) )
        {
            ref_pt[0] = loc;
            ref_pt[1] = pos(axis_pts[2],axis_pts[2],axis_pts[3],orig[1],ext[1]);
            ref_pt[2] = axis_pts[4];
            offset[0] = 0.0;
            offset[1] = -0.06;
            offset[2] = 0.0;

            val = axis_pts[1];
            sprintf(num,arg,val);

            GEOMadd_label(label_x_obj,num,ref_pt,offset,*font_size,color,-1);
        }                   
    }

/* Labels on y axis. */
    if (num_ticks[1] && show_labels) {

    /* Set the number of digits to the right of the decimal via precision. */
        arg[0] = '%';
        arg[1] = '.';
        arg[2] = precision[1] + 0x30;
        arg[3] = 'f';
        arg[4] = '\0';

        increment = u_inc = get_increment(num_ticks[1],axis_pts[2],axis_pts[3]);

        loc = axis_pts[2] + ((y_label) ? 0.0 : increment);
        val = loc;

        loc = pos(loc,axis_pts[2],axis_pts[3],orig[1],ext[1]);

        for (ct=0;ct<=(num_ticks[1]/label_density[1]+(y_label?1:0)),
                  val <= axis_pts[3]; ct++)
        {
            ref_pt[0] = pos(axis_pts[0],axis_pts[0],axis_pts[1],orig[0],ext[0]);
            ref_pt[1] = loc;
            ref_pt[2] = axis_pts[4];
            offset[0] = -0.02;
            offset[1] = -0.02;
            offset[2] = 0.0;

            sprintf(num,arg,val);

            GEOMadd_label(label_y_obj,num,ref_pt,offset,*font_size,color,-1);

            val += increment*label_density[1];
            loc += frac(increment * label_density[1],axis_pts[2],axis_pts[3],ext[1]);
        }
        /* do the last label */ 
        if (y_label && (loc <= (orig[1]+ext[1])) )
        {
            ref_pt[0] = pos(axis_pts[0],axis_pts[0],axis_pts[1],orig[0],ext[0]);
            ref_pt[1] = loc;
            ref_pt[2] = axis_pts[4];
            offset[0] = -0.02;
            offset[1] = -0.02;
            offset[2] = 0.0;

            val = axis_pts[3];
            sprintf(num,arg,val);
            GEOMadd_label(label_y_obj,num,ref_pt,offset,*font_size,color,-1);
        } 
    }

}


/************** Support stuff. **********************************************/

set_tick_color(tc,r,g,b)
float tc[4][3];
float r,g,b;
{
    tc[0][0] = tc [1][0] = tc[2][0] = tc[3][0] = r;
    tc[0][1] = tc [1][1] = tc[2][1] = tc[3][1] = g;
    tc[0][2] = tc [1][2] = tc[2][2] = tc[3][2] = b;
}

set_plane_color(pc,r,g,b)
float pc[2][3];
float r,g,b;
{
    pc[0][0] = pc [1][0] = r;
    pc[0][1] = pc [1][1] = g;
    pc[0][2] = pc [1][2] = b;
}

normalize(increment,axis_pts,min_pt,max_pt)
float *increment;
float axis_pts[6];
float min_pt,max_pt;
/*
 *  This is a uniform data set so we want to scale increment
 *  and set the axis dimensions back to -1,1.
 **** undone 22 may 1991
 */
{
    *increment = 2 * *increment / (max_pt - min_pt);
    axis_pts[0] = axis_pts[2] = -1.0;
    axis_pts[1] = axis_pts[3] = 1.0;
}


/************** Increment calculation. **************************************/

double mypow(),nicenum();

float get_increment(ticks,axis_min,axis_max)
int ticks;
float axis_min,axis_max;
{
    double range,d;
#if 0
    if (ticks <= (axis_max - axis_min)) {
        d = (double)((int)((axis_max - axis_min)/ticks + 0.5));
    }
    else {
        range = nicenum(axis_max - axis_min, 0);
        d = nicenum(range / (ticks - 1), 1);
    }
#endif
    d = (axis_max - axis_min)/(ticks+1);
    return (float)d;
}

double nicenum(x, round)
double x;
int round;
/*
 * nicenum: find a "nice" number approximately equal to x
 * round if round=1, ceil if round=0
 */
{
    long exp;
    double f, y;

    exp = floor(log10(x));
    f = x/mypow(10., exp);   /* fraction between 1 and 10 */
    if (round)
        if (f<1.5) y = 1.;
        else if (f<3.) y = 2.;
        else if (f<7.) y = 5.;
        else y = 10.;
    else
        if (f<=1.) y = 1.;
        else if (f<=2.) y = 2.;
        else if (f<=5.) y = 5.;
        else y = 10.;
    return y*mypow(10., exp);
}


double mypow(a, n)
double a;
register int n;
/*
 * mypow(a,n)=a^n for integer n
 * roundoff errors in pow were causing problems, so I wrote my own
 */
{
    double x;

    x = 1.;
    if (n>0)
        for (; n>0; n--)
            x *= a;
    else
        for (; n<0; n++)
            x /= a;
    return x;
}

