/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/* NOTE:  THIS MODULE AND SOURCE CODE IS FOR USE 
   WITH THE AVS SOFTWARE ENVIRONMENT ONLY */
/******************************************************/
/* Trace Render Module                                */
/******************************************************/
/*
 * trace_render - display system for a 1D or 2D stack of 
 *                signal waveforms
 *
 * Author: Danny Harvey,
 *         Dept. of Physics,
 *         Joint Seismic Program Center
 *         University of Colorado at Boulder
 *         Campus Box 390
 *         Boulder, Colorado 80309-0390
 *         Tel. 303-492-0270
 *
 * Revised and Extensively modified by: Ian Curington, AVS Inc.
 *
 * Revision History:
 *       1989    Original
 * 22 June 93    ianc: Major Conversion from Stellar/XFDI
 * 29 June 93    ianc: updated controls, moveout functions.
 *  1 July 93    ianc: added offscreen pixmap buffer
 *  2 July 93    ianc: speeded up expose event, added polar mode
 *  9 July 93    ianc: cleanup of unused variable
 *               ianc: extended to 1D signal
 *
 */

/******************************************************/
#include <stdio.h>
#include <avs/avs.h>
#include <avs/field.h>
/* IAC CODE CHANGE : #include <avs/avs_math.h> */
#include <avs/avs_math.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
/******************************************************/




/******************************************************/
/* GLOBAL VARIABLES                                   */
/******************************************************/

/* window default size, place */
#define	NP	1000
#define	XWIN	260
#define	YWIN	42
#define	WWIN    860	
#define	HWIN	500
#define	BWIN    5	


/* global X context */
Window window;
int screen;
int depth;
Pixmap pixmap;
Display *display;
GC gc, rgc;
XGCValues gcvalues;
int myexposeproc();

/* trace data structure attributes */

int ntr = 0;            /* number of traces */
int ntot = 0;           /* number of samples per trace */

/* input field pointer */
AVSfield_float *input = NULL;

/* polyline storage area */
XPoint *trace_list;
XPoint *mo_list;

float x_ampl, y_ampl;
float x_offset, y_offset;
float y_spacing;

float xmin, xmax, ymin, ymax;
float rmin, rmax;
int imin, imax;
float t0 = -10000000.0;
float tm = -10000000.0;
float matrix[16] =   {1.0, 0.0, 0.0, 0.0,
		      0.0, 1.0, 0.0, 0.0,
		      0.0, 0.0, 1.0, 0.0,
		     -1.0, 0.0, 0.0, 1.0};

/* Window actual size, place */
int xwin=XWIN;
int ywin=YWIN;
unsigned int wwin=(WWIN+2*BWIN);
unsigned int hwin=(HWIN+2*BWIN);
unsigned int bwin=BWIN;
int xpix=BWIN;
int ypix=BWIN;
int wpix=WWIN;
int hpix=HWIN;

/* transformations to pixel space */
float xp2ndc=2.0/WWIN;
float yp2ndc=2.0/HWIN;
float trscale=1.0;
float tr0=0.0;

/* starting interpretation values */
int   aspect;
int jumpscroll=0;
int moveout_lin=0;
int moveout_hyp=0;
int moveout_clr=0;
int moveout_sho=0;
int moveout_apl=0;
int send_out=0;
float moveout_t0=0.0;
float moveout_vel=6.0;
int icomp=3;
float *motime = NULL;
float *moredtime = NULL;
float *morange = NULL;
float max_angle=M_PI;

unsigned long Light_Blue;

/******************************************************/
/* Main Program Entry Point for Coroutine             */
/******************************************************/

main (argc, argv)

int argc;
char **argv;

{
	int x, y, w, h;
	XSetWindowAttributes xwa;
	XSizeHints xsh;
	XEvent event;
	int mask, x_event;
	int AVS_event, X_event;
	char *answer, *move_type;
	float *mo_t0, *mo_vel;
	int trace_desc();
        float *px_ampl, *py_ampl;
        float *px_offset, *py_offset;
        float *py_spacing;

        /******************************************************/
        /* Initialize  Coroutine                              */
        /******************************************************/
	AVScorout_init (argc, argv, trace_desc);
        AVScorout_set_sync(1);  /* Run synchronous with the flow-exec */


        /******************************************************/
        /* Get Current AVS Parmeters, Input data              */
        /******************************************************/
	AVScorout_input (&input,
                        &px_ampl, &py_ampl,
                        &px_offset, &py_offset,
                        &py_spacing,
                        &mo_t0, &mo_vel, 
			&answer, 
			&jumpscroll,
                        &move_type,
                        &moveout_clr,
			&moveout_sho,
                        &moveout_apl );

        x_ampl = *px_ampl;
        y_ampl = *py_ampl;
        x_offset = *px_offset;
        y_offset = *py_offset;
        y_spacing = *py_spacing;
	moveout_t0 = *mo_t0;
	moveout_vel = *mo_vel;
        aspect = AVSchoice_number("Aspect",answer)-1;

        if ( AVSchoice_number("Moveout-Type",move_type) == 2 )
        {
            moveout_lin = 0;
            moveout_hyp = 1;
        } else {
            moveout_lin = 1;
            moveout_hyp = 0;
        }

        /* allocate the memory the first time */
        if ( input ) make_traces (input);

        /******************************************************/
        /* Setup the X-Window Display                         */
        /******************************************************/
	display = XOpenDisplay (NULL);
	if (display == NULL) {
		fprintf (stderr, 
			"trace_render: Unable to connect to X server.\n");
		exit (1);
	}

	screen = DefaultScreen (display);
	depth = DisplayPlanes (display, screen);
	xwa.background_pixel = BlackPixel (display, screen);
	xwa.border_pixel = WhitePixel (display, screen);
	window = XCreateWindow (display, RootWindow(display, screen),
					xwin, ywin, wwin, hwin,
					bwin, depth,
					InputOutput,
					DefaultVisual(display, screen), 
					(CWBackPixel|CWBorderPixel), &xwa);
	if (window == NULL) {
		fprintf (stderr, "trace_render: Unable to create window.\n");
		exit (1);
	}

	pixmap = NULL;
	gcvalues.graphics_exposures = False;
	gc = XCreateGC (display, window, GCGraphicsExposures, &gcvalues);
	xsh.width = wwin;
	xsh.height = hwin;
	xsh.flags = USSize;
	XSetNormalHints (display, window, &xsh);
	XSetStandardProperties (display, window, argv[0], argv[0], None,
				argv, argc, &xsh);
	XSetBackground(display,gc, BlackPixel (display, screen));
	XSetForeground(display,gc, WhitePixel (display, screen));
        if ( depth == 8 )
          Light_Blue =  2 + 8 * 2 + 16 * 2;
        else if ( depth == 16 )
          Light_Blue =  10 + 16 * 10 + 16 * 16 * 20;
        else if ( depth == 24 )
          Light_Blue =  50 + 256 * 50 + 256 * 256 * 200;


        /* Create an offscreen pixmap */
        pixmap = XCreatePixmap(display,
                           DefaultRootWindow(display),
                           wwin, hwin, depth);


	XSelectInput (display, window, ExposureMask
					| KeyPressMask
					| ButtonPressMask
					| ButtonMotionMask
					| ButtonReleaseMask);
	XMapWindow (display, window);

        /* trigger an initial draw */
	wwin = 0;
	hwin = 0;
        mydraw();
        mypaint();
	XSync (display, False);

        /******************************************************/
        /* Primary Event Loop, wait for AVS or Window         */
        /******************************************************/
	while (1)
        {
		/* This will block until either X or AVS or both */
		mask = COROUT_WAIT;
		X_event = AVScorout_X_wait(display, NULL, &mask);

		if (mask)
                {
			AVScorout_input (&input,
                                &px_ampl, &py_ampl,
                                &px_offset, &py_offset,
                                &py_spacing,
                                &mo_t0, &mo_vel,
				&answer, 
			        &jumpscroll,
                                &move_type,
                                &moveout_clr,
			        &moveout_sho,
                                &moveout_apl );

                        x_ampl = *px_ampl;
                        y_ampl = *py_ampl;
                        x_offset = *px_offset;
                        y_offset = *py_offset;
                        y_spacing = *py_spacing;
			moveout_t0 = *mo_t0;
			moveout_vel = *mo_vel;
			AVS_event = 0;
                        aspect = AVSchoice_number("Aspect",answer)-1;

                        if ( AVSchoice_number("Moveout-Type",move_type) == 2 )
                        {
                            moveout_lin = 0;
                            moveout_hyp = 1;
                        } else {
                            moveout_lin = 1;
                            moveout_hyp = 0;
                        }

	        	/********************************/
		        /* Process an AVS trigger       */
       		 	/********************************/
			if (AVSinput_changed ("Input Trace", 0)) {
				make_traces(input);
				mydraw();
				mypaint();
			}
			if (AVSparameter_changed ("X Ampl")   ||
			    AVSparameter_changed ("Y Ampl")   ||
			    AVSparameter_changed ("X Offset") ||
			    AVSparameter_changed ("Y Offset") ||
			    AVSparameter_changed ("Y Spacing")|| 
			    AVSparameter_changed ("Aspect") )
       		        {
				mydraw();
				mypaint();
			}
			if (AVSparameter_changed ("Jump Scroll")) {
				jsproc (jumpscroll);
			}
			if (AVSparameter_changed ("Moveout-Type")) {
				moproc (moveout_lin);
			}
			if (AVSparameter_changed ("Moveout-Clr")) {
				moclrproc (moveout_clr);
			}
			if (AVSparameter_changed ("Show-Moveout")) {
				moshoproc (moveout_sho);
			}
			if (AVSparameter_changed ("Apply-Moveout")) {
				moaplproc (moveout_apl);
			}
			if (AVSparameter_changed ("T0")) {
				movt0proc ();
			}
			if (AVSparameter_changed ("Velocity")) {
				movt0proc ();
			}
		}
        	/********************************/
	        /* Process an X window trigger  */
       	 	/********************************/
		if (X_event) {
			do {
			XNextEvent (display, &event);
			switch (event.type) {
			case Expose:
				myexposeproc (window, &event);
				break;
			case KeyPress:
				break;
			case ButtonPress:
			case ButtonRelease:
			case MotionNotify:
				buttonmotionproc (window, &event);
				break;
			default:
				break;
			}
			} while (XPending (display));
		}
	}
}

/******************************************************/
/* AVS Description Routine                            */
/******************************************************/
int trace_desc()

{
	int  in_port, out_port;
	int  t0_param, vel_param;
        int  param;

	int quitproc();

	AVSset_module_name ("Trace Render", MODULE_RENDER);
	AVSset_destroy_proc (quitproc);

	in_port = AVScreate_input_port ("Input Trace",
                  "field uniform scalar float", REQUIRED);

        param = AVSadd_float_parameter ("X Ampl",
                 1.0, FLOAT_UNBOUND, FLOAT_UNBOUND );
	AVSadd_parameter_prop (param, "immediate", "boolean", 1);
        param = AVSadd_float_parameter ("Y Ampl",
                 1.0, FLOAT_UNBOUND, FLOAT_UNBOUND );
	AVSadd_parameter_prop (param, "immediate", "boolean", 1);
        param = AVSadd_float_parameter ("X Offset",
                 0.0, FLOAT_UNBOUND, FLOAT_UNBOUND );
	AVSadd_parameter_prop (param, "immediate", "boolean", 1);
        param = AVSadd_float_parameter ("Y Offset",
                 0.0, FLOAT_UNBOUND, FLOAT_UNBOUND );
	AVSadd_parameter_prop (param, "immediate", "boolean", 1);
        param = AVSadd_float_parameter ("Y Spacing",
                 10.0, FLOAT_UNBOUND, FLOAT_UNBOUND );
	AVSadd_parameter_prop (param, "immediate", "boolean", 1);

	t0_param = AVSadd_float_parameter ("T0", moveout_t0, 0.0, 1000.0);
	AVSconnect_widget(t0_param, "dial");
	AVSadd_parameter_prop (t0_param, "immediate", "boolean", 1);
	vel_param = AVSadd_float_parameter("Velocity",moveout_vel,0.01,20.0);
	AVSconnect_widget(vel_param, "dial");
	AVSadd_parameter_prop (vel_param, "immediate", "boolean", 1);

        AVSadd_parameter ("Aspect",
                "choice", "Vertical","Vertical#Horizontal#Polar", "#");

	AVSconnect_widget(
		AVSadd_parameter ("Jump Scroll", "boolean", jumpscroll, 0, 1), 
		"boolean");

        AVSadd_parameter ("Moveout-Type",
                "choice", "Moveout-Lin",
                "Moveout-Lin#Moveout-Hyp", "#");

	AVSconnect_widget(
		AVSadd_parameter ("Moveout-Clr", "boolean", moveout_clr, 0, 1), 
		"boolean");
	AVSconnect_widget(
		AVSadd_parameter ("Show-Moveout","boolean", moveout_sho, 0, 1), 
		"boolean");
	AVSconnect_widget(
		AVSadd_parameter ("Apply-Moveout","boolean",moveout_apl, 0, 1), 
		"boolean");
        return(1);
}

/******************************************************/
/* Utility to make Trace structure from Input Field   */
/******************************************************/
int make_traces(field)

AVSfield_float *field;    /* 1D or 2D uniform float */

{

	int i,k,n,itr;
	float max, min;

        /* number of traces */
        if ( field->ndim == 2 )
            ntr = field->dimensions[1];
        else
            ntr = 1;

        /* number of points per trace */
	ntot = field->dimensions[0];

        /* setup space for polyline lists */

/* IAC CODE CHANGE : 	if (trace_list) free (trace_list); */
	if (trace_list)  free(trace_list);
	trace_list = (XPoint *) malloc (ntr * ntot * sizeof(XPoint));


/* IAC CODE CHANGE : 	if (mo_list) free (mo_list); */
	if (mo_list)  free(mo_list);
	mo_list = (XPoint *) malloc (ntr * sizeof(XPoint));


/* IAC CODE CHANGE : 	if (motime) free (motime); */
	if (motime)  free(motime);
	motime = (float *) malloc (ntr * sizeof(float));

}

/******************************************************/
/*  Expose event processing                           */
/******************************************************/
int
myexposeproc(window, event)

Window window;
XExposeEvent *event;

{
	Window root;
	int x, y;
	unsigned int w, h, b, d;

	if (event->type != Expose) return;

	XGetGeometry (display, window, &root, &x, &y, &w, &h, &b, &d);
	xwin = x;
	ywin = y;
	if (w != wwin || h != hwin) {
		wwin = w;
		hwin = h;
		wpix = w - 2*BWIN;
		hpix = h - 2*BWIN;
		xp2ndc = 2.0 / wpix;
		yp2ndc = 2.0 / hpix;
		if (pixmap) XFreePixmap (display, pixmap);
		pixmap = XCreatePixmap (display, 
					window, wwin, hwin, depth);
		mydraw();
		mypaint();
	}
	if (window == window && event->count == 0) {
	/***	mydraw();   ***/
		mypaint();
	}
}

/******************************************************/
/*  button motion  event processing                   */
/******************************************************/
int
buttonmotionproc(window, event)

Window window;
XButtonEvent *event;

{
	static int button = 0;
	static int xdl, ydl;
	float xd, xd0, xs;
	float yd, yd0, ys;
	int i;
	XButtonEvent levent;
	float range, rslope;
	float a,b,toff;
	static float tfix;

	void xtrnmat();
	void ytrnmat();
	float xpix2uc();
	float ypix2uc();

	switch (event->type) {
	case MotionNotify:
		while (XPending(display)) {
			XNextEvent (display, &levent);
			if (levent.window == window 
					&& levent.type == MotionNotify) {
				event = &levent;
				continue;
			} else {
				XPutBackEvent (display, &levent);
				break;
			}
		}
		switch (button) {
		case Button1:
			if (jumpscroll) break;
/******
				xd = ydl-event->y;
				xd *= yp2ndc;
				for (i=0; i<ntr; i++) head[i]->matrix[13] += xd;
				tr0 = head[0]->matrix[13] - trscale;
 ****/
			xdl = event->x;
			ydl = event->y;
			mydraw();
			mypaint();
			break;
		case Button2:
			if (moveout_hyp) {
				xd = event->x-xdl;
/****
				xd *= xp2ndc/head[imax]->matrix[0];
				t0 += xd;
				b = t0*t0;
				a = (tm*tm-b)/(rmax*rmax);
				for (i=0; i<ntr; i++) {
					range = head[i]->range;
					toff = sqrt(a*range*range+b);
					toff -= tfix;
					head[i]->matrix[12] += 
				(toff - head[i]->mocor)*head[i]->matrix[0];
					head[i]->mocor = toff;
				}
 ****/
			} else {
				yd0 = hpix-ydl+ypix;
				yd0 = (yd0*yp2ndc-1.0-tr0)/trscale;
				yd = hpix-event->y+ypix;
				yd *= yp2ndc;
				trscale = (yd - 1.0 - tr0) / yd0;
/****
				for (i=0; i<ntr; i++) 
					head[i]->matrix[13] 
					= (head[i]->range-rmin)*trscale + tr0;
 ****/
			}
			xdl = event->x;
			ydl = event->y;
			mydraw();
			mypaint();
			break;
		case Button3:
			if (jumpscroll) break;
			yd0 = hpix-ydl+ypix;
/****
			yd0 = (yd0*yp2ndc-1.0-head[0]->matrix[13])/head[0]->matrix[5];
			yd = hpix-event->y+ypix;
			yd *= yp2ndc;
			ys = (yd - 1.0 - head[0]->matrix[13]) / yd0;
			for (i=0; i<ntr; i++) head[i]->matrix[5] = ys;
 ****/
			ydl = event->y;
			mydraw();
			mypaint();
			break;
		default:
			break;
		}
		break;
	case ButtonPress:
		button = event->button;
		xdl = event->x;
		ydl = event->y;
		if (moveout_hyp) {
			switch (button) {
			case Button1:
				if (tm < -1000000.0)
/****
					tm = xpix2uc (event->x-xpix, xp2ndc,
							head[imax]->matrix);
 ****/
				if (t0 < -1000000.0) t0 = tm;
				tfix = t0;
				break;
			case Button2:
				if (t0 < -1000000.0)
/****
					t0 = xpix2uc (event->x-xpix, xp2ndc,
							head[imin]->matrix);
 ****/
				if (tm < -1000000.0) tm = t0;
				tfix = tm;
				break;
			default:
				break;
			}
		}
		if (!jumpscroll) break;
		switch (button) {
		case Button1:
/****
				for (i=0; i<ntr; i++) {
					xd = ypix2uc (hpix-ydl+ypix, yp2ndc,head[i]->matrix);
					ytrnmat (0, yp2ndc, xd, head[i]->matrix);
				}
				tr0 = head[0]->matrix[13] - trscale;
 ****/
			mydraw();
			mypaint();
			break;
		case Button3:
/****
				for (i=0; i<ntr; i++) {
					xd = ypix2uc (0, yp2ndc,head[i]->matrix);
					ytrnmat (hpix-ydl+ypix, yp2ndc, xd, head[i]->matrix);
				}
				tr0 = head[0]->matrix[13] - trscale;
 ****/
			mydraw();
			mypaint();
			break;
		default:
			break;
		}
		break;
	case ButtonRelease:
		break;
	default:
		break;
	}
}

/******************************************************/
/*  mypaint - display the traces again                */
/******************************************************/
int
mypaint()

{
        XCopyArea(display,
                  pixmap,
                  window,
                  gc,
                  0, 0, wwin, hwin, 0, 0);

        XMapWindow(display, window);
        XSync(display, 0);

}

/******************************************************/
/*  destroy the application window                    */
/******************************************************/
int quitproc()

{
	XFreePixmap (display, pixmap);
	exit (0);
}

/******************************************************/
/*  Jump Scroll State control                         */
/******************************************************/
int jsproc(state)

int state;

{
}

/******************************************************/
/*  Move Out  control                                 */
/******************************************************/
int moproc(state)

int state;

{
}

/******************************************************/
/*  Move Out (Hyperbolic)  control                    */
/******************************************************/
int mohproc(state)

int state;

{
}

/******************************************************/
/*  Move Out Clear Function                           */
/******************************************************/
int moclrproc (state)

int state;

{
	int i;

	t0 = -10000000.0;
	tm = -10000000.0;
	mydraw();
	mypaint();
	moveout_clr = 0;
	AVSmodify_parameter ("Moveout-Clr", AVS_VALUE, 
						moveout_clr, NULL, NULL);
}

/******************************************************/
/*  Show Move Out Function                           */
/******************************************************/
int moshoproc (state)

int state;

{
	mydraw();
	mypaint();
}

/******************************************************/
/*  Apply Move Out Function                           */
/******************************************************/
int moaplproc (state)

int state;

{
	int i;
	float mocor;

        mydraw();
	mypaint();
}

/******************************************************/
/*  Save Move Out Function                           */
/******************************************************/
int soutproc (state, input)

int state;
AVSfield_float *input;

{

}

/******************************************************/
/*  Move Out T(zero) Function                         */
/******************************************************/
int movt0proc ()

{
	int i;

	if (moveout_lin) {
		for (i=0; i<ntr; i++) {
			motime[i] = moveout_t0 + (float)i/moveout_vel;
		}
		if (moveout_sho || moveout_apl) {
			mydraw();
			mypaint();
		}
	} else if (moveout_hyp) {
		for (i=0; i<ntr; i++) {
			motime[i] = sqrt(moveout_t0*moveout_t0 
			+ ((float)i * (float)i)/(moveout_vel*moveout_vel));
		}
		if (moveout_sho || moveout_apl) {
			mydraw();
			mypaint();
		}
	}
}

/******************************************************/
/*  Trace Graphics Drawing Function                   */
/******************************************************/
int mydraw()

{
	float xmn, xmx, ymn, ymx;
	int i;
        int itr;
        short ix, iy, irange;
        float vv, tt, itt, angle, radius;

        tt = moveout_t0*moveout_t0;
        vv = moveout_vel*moveout_vel;

        /* loop over traces, building the list */
        /* create ramp on x-coords */

	for (itr=0; itr<ntr; itr++)
        {

            /* as-is */
            if ( !moveout_apl )
            {
                irange = 0;
            }
            /* draw in linear moveout configuration */
            else if ( moveout_lin )
            {
                irange = (short)(((float)itr / moveout_vel + moveout_t0)
                                        * x_ampl - x_offset);
            }
            /* draw in hyperbolic moveout configuration */
            else if ( moveout_hyp )
            {
                 itt= (float)itr * (float)itr;
                 irange = (short)( sqrt( tt + itt / vv ));
            }


	    for (i=0; i<ntot; i++)
            {

              /* create ramp on x-coords */
              ix = (short)((float)i * x_ampl - x_offset)+irange;

              /* offset a signal, add signal to baseline */
              iy = (short)(((float)itr * y_spacing) - y_offset +
                   (y_ampl * input->data[itr * ntot + i]));


              if ( aspect == 1 )        /* horizontal */
              {
                trace_list[ itr * ntot + i ].x = ix;
                trace_list[ itr * ntot + i ].y = iy;
              } else if( aspect == 0 )  /* verticle */
              {
                trace_list[ itr * ntot + i ].y = ix;
                trace_list[ itr * ntot + i ].x = iy;
              } else if( aspect == 2 )  /* polar */
              {
                angle=max_angle*(float)ix/
                    ((float)(ntot-1)*x_ampl-x_offset+(float)irange);
                radius= (float)iy;
                trace_list[ itr * ntot + i ].x = (short)((cos(angle)*radius)+(wwin/2));
                trace_list[ itr * ntot + i ].y = (short)(sin(angle)*radius);
              }
            }
        }

        XSetForeground(display, gc, BlackPixel (display, screen));

        XFillRectangle(display, pixmap, gc, 0, 0, wwin, hwin);

        XSetForeground(display, gc, WhitePixel (display, screen));

	for (i=0; i<ntr; i++)
        {

             draw_2d_line_list ( &(trace_list[ i*ntot ]), ntot );

        }
	mymodraw();
}

/******************************************************/
/*  Move Out Graphics Drawing Function                */
/******************************************************/
int mymodraw()

{
	int i;
        short ix, iy;
        float vv, tt, itt;
        float angle, radius;
        float ixmax;

	if (moveout_sho && (moveout_lin || moveout_hyp)) {

		for (i=0; i<ntr; i++)
                {
                   iy = (short)(((float)i * y_spacing) - y_offset);
                   if ( moveout_lin )
                   {
			ix = (short)(((float)i / moveout_vel + moveout_t0)
                                       * x_ampl - x_offset);
			ixmax =(((float)(ntr-1) / moveout_vel + moveout_t0)
                                       * x_ampl - x_offset);
                   }
                   else if ( moveout_hyp )
                   {
                        tt = moveout_t0*moveout_t0;
                        vv = moveout_vel*moveout_vel;
			itt= (float)i * (float)i;
			ix = (short)( sqrt( tt + itt / vv ));
			itt= (float)ntr * (float)i;
			ixmax = (short)( sqrt( tt + itt / vv ));
                   }

                   if ( aspect == 1 )        /* horizontal */
                   {
			mo_list[i].x = ix;
			mo_list[i].y = iy;
                   }
                   else if( aspect == 0 )    /* verticle */
                   {
			mo_list[i].y = ix;
			mo_list[i].x = iy;
                   }
                   else if( aspect == 2 )    /* polar */
                   {
                     angle = max_angle*(float)ix/ixmax;
                     radius= (float)iy;
                     mo_list[i].x = (short)(cos(angle)*radius)+(wwin/2);
                     mo_list[i].y = (short)(sin(angle)*radius);
                   }
		}


	        XSetForeground(display,gc, Light_Blue );

                draw_2d_line_list ( mo_list, ntr );

	        XSetForeground(display,gc, WhitePixel (display, screen));
	}
}

/******************************************************/
/*  X Translate to Matrix  Function                   */
/******************************************************/
void
xtrnmat (xpx, xpix2ndc, xuc, matrx)

int xpx;
float xpix2ndc;
float xuc;
float *matrx;

/*
 *	xtrnmat will change a 4x4 XFDI coordinate transformation matrix to
 *	account for X translation.
 *
 *	Inputs  -	xpix	= An integer X-pixel coordinate in the
 *				  destination pixmap.
 *			xpix2ndc= The floating X-pixel to NDC scaling factor.
 *				  This is equal to two over the pixmap width.
 *			xuc	= A floating X user coordinate. The translation
 *				  is done such that this user coordinate is at
 *				  xpix.
 *
 *	Outputs -	matrix	= The 4x4 XFDI matrix is adjusted to effect the
 *				  X translation.
 */

{
	matrx[12] = (xpx*xpix2ndc - 1.0) - xuc*matrx[0];
}

/******************************************************/
/*  Y Translate to Matrix  Function                   */
/******************************************************/
void
ytrnmat (ypx, ypix2ndc, yuc, matrx)

int ypx;
float ypix2ndc;
float yuc;
float *matrx;

{
	matrx[13] = (ypx*ypix2ndc - 1.0) - yuc*matrx[5];
}

/******************************************************/
/*  X Pixel to user coordinate transformation         */
/******************************************************/
float
xpix2uc (xpx, xpix2ndc, matrx)

int xpx;
float xpix2ndc;
float *matrx;

/*
 *	xpix2uc will convert from X-pixel coordinates in a pixmap to
 *	floating user coordinates.
 *
 *	Inputs  -	xpix	= The X-pixel coordinate in the source pixmap.
 *			xpix2ndc= The floating X-pixel to NDC scaling factor.
 *				  This is equal to two over the pixmap width.
 *			matrix	= The 4x4 XFDI coordinate transformation 
 *				  matrix.
 *
 *	Returns -	The X user coordinate value corresponding to xpix.
 */

{
	float xuc;

	xuc = ((xpx*xpix2ndc - 1.0) - matrx[12]) / matrx[0];
	return (xuc);
}

/******************************************************/
/*  Y Pixel to user coordinate transformation         */
/******************************************************/
float
ypix2uc (ypx, ypix2ndc, matrx)

int ypx;
float ypix2ndc;
float *matrx;

{
	float yuc;

	yuc = ((ypx*ypix2ndc - 1.0) - matrx[13]) / matrx[5];
	return (yuc);
}

/******************************************************/
/*  Draw 2D line segments to the display window       */
/*   with local transformations                       */
/******************************************************/
draw_2d_line_list ( verts, n )
        XPoint *verts;
        int   n;
{

     XDrawLines( display, pixmap, gc, verts, n, CoordModeOrigin );

}


