/* mod_gen Version 1                                                     */
/* Module Name: "display_gaussian" (Input) (Subroutine)                  */
/* Authors: Mark Reed, NCSC, 919-248-1185                                */
/*         Ken Flurchick, NCSC, 919-248-1121                             */
/*         Lee Bartolotti, NCSC, 919-248-1185                            */
/* Date Created: February 1994                                           */
/*                                                                       */
/*                                                                       */
/* output 0 "out3d_fld" field 3D 3-space 3-vector rectilinear float      */
/* output 1 "out_geom" geom Polyhedron                                   */
/* param 0 "chkpt file" browser "0" "" "0"                               */
/* param 1 "Representation" radio_buttons "ball and stick"               */
/*	     "ball and stick:cpk:colored stick" ":"                      */
/* param 2 "Sphere Scale",typein_real 1.0 0.0 5.0                        */
/* param 3 "MO selected" typein_integer homo 1 norbs                     */
/* param 4 "Compute Choice" radio_buttons "None"                         */
/*           "MO:Density:None" ":"                                       */
/* param 5 "Grid Spacing" typein_real 0.400000 0.00000 100.000           */
/* param 6 "Extent" typein_real 4.00000 0.00000 50.000                   */


#include <stdio.h>
#include <avs/avs.h>
#include <avs/port.h>
#include <avs/field.h>
#include <avs/geom.h>


#include "string.h"
#include "math.h"
#include <stdlib.h>
#include <ctype.h>
#include <avs/flow.h>



#define MAXATOMS 200
#define NUMORBS 400
#define MAXPTS 100
#define AU 0.5291670E+0          /* units are angstrom/bohr */
#define AUINV 1.0/ 0.5291670E+0  /* units are bohr/angstrom */
#define NATOM_TYPE 	12
#define MAXBONDS 	8196
#define BONDLENGTH	1.65
#define BALL 		0
#define BALL_AND_STICK 	1
#define STICK 		2
#define COLORED_STICK 	3
#define RECLEN          80
#define C_rad 	1.85
#define H_rad 	1.20 
#define O_rad 	1.40
#define N_rad 	1.54
#define S_rad 	1.85
#define I_rad 	2.15
#define P_rad 	1.90      
#define Cl_rad 	1.81      
#define F_rad 	1.35
#define B_rad 	2.08
#define Zn_rad 	1.40
#define Other_rad  1.50
#define STICKSANDSPHERES_RADFACTOR   0.25   /* spheres for ball-and-stick */





/* ************************************************************************* */
/* ************************************************************************* */

/* ---------------------------------------- */
/*           Module Description             */
/* ---------------------------------------- */
int display_gaussian_desc()
{
  
  int in_port, out_port, param, iresult;
  extern int display_gaussian_compute();

  
  AVSset_module_name("display_gaussian", MODULE_DATA);
  


  /* Output Port Specifications              */
  out_port = AVScreate_output_port("out3d_fld", 
				   "field 3D scalar rectilinear float");

  out_port = AVScreate_output_port("out_geom", "geom");

  

  /* Parameter Specifications                */
  param = AVSadd_parameter("chkpt file", "string", "", "", "");
  AVSconnect_widget(param, "browser");

  param = AVSadd_parameter("Representation", "choice", "ball and stick",
		   "ball and stick:cpk:colored stick", ":"); 
  AVSconnect_widget(param, "radio_buttons");

  param = AVSadd_float_parameter("Sphere Scale", 1.00000, 0.00000, 5.00000);
  AVSconnect_widget(param, "typein_real");

  param = AVSadd_parameter("MO selected", "integer", 0, 0, 0);
  AVSconnect_widget(param, "typein_integer");

  param = AVSadd_parameter("Compute Choice", "choice", "None",
		   "MO:Density:None", ":"); 
  AVSadd_parameter_prop(param,"title","string","Compute");
  AVSconnect_widget(param, "radio_buttons");


  param = AVSadd_float_parameter("Grid Spacing", 0.400000, 0.00000, 
				 100.000);
  AVSconnect_widget(param, "typein_real");

  param = AVSadd_float_parameter("Extent", 4.00000, 0.00000, 50.00000);
  AVSconnect_widget(param, "typein_real");

  AVSset_compute_proc(display_gaussian_compute);

  return(1);
}




/* ************************************************************************* */
/* ************************************************************************* */

/* --------------------------------------------------------- */
/* Initialization for modules contained in this file.        */
/* --------------------------------------------------------- */
static int ((*mod_list[])()) = {
       display_gaussian_desc
     };
#define NMODS (sizeof(mod_list) / sizeof(char *))
       
     AVSinit_modules()
{
  AVSinit_from_module_list(mod_list, NMODS);
}


/* these are some global definitions */

char	type[MAXATOMS][4];
/* int	bonds[MAXBONDS][2]; */
float   bond_colors[MAXATOMS][3];
/* int	atomtype[MAXATOMS]; */
float	r_lst[MAXATOMS];
float	rgb_lst[MAXATOMS][3];
float	a_lst[MAXATOMS][3];
float	b_lst[4*MAXBONDS][3];
float   b_rgb_lst[4*MAXBONDS][3];
int	num_bonds;


/* table of Van der Waals radii per each of (NATOM_TYPE) atom types */
/* Note: the order is keyed to the values set in the host "makeatomtype" */

static float radius[NATOM_TYPE] =
{C_rad, O_rad, N_rad, H_rad, S_rad, P_rad, Cl_rad, I_rad, F_rad, B_rad,
Zn_rad, Other_rad};

static float atomcolors[][3] = {
	{0.0,1.0,0.0}, {1.0,0.0,0.0}, {0.0,0.0,1.0}, {1.0,1.0,1.0},
	{1.0,1.0,0.0}, {1.0,0.5,0.5}, {1.0,1.0,0.5}, {0.5,1.0,0.5},
	{0.0,1.0,1.0}, {0.5,0.0,0.5}, {1.0,0.0,1.0}, {1.0,0.0,1.0}
	};

  static   char  *atom_names[104] = { 
      "H ","HE","LI","BE","B ","C ","N ","O ","F ","NE","NA",
      "MG","AL","SI","P ","S ","CL","AR","K ","CA","SC","TI","V ",
      "CR","MN","FE","CO","NI","CU","ZN","GA","GE","AS","SE","BR",
      "KR","RB","SR","Y ","ZR","NB","MO","TC","RU","RH","PD","AG",
      "CD","IN","SN","SB","TE","I ","XE","CS","BA","LA","CE","PR",
      "ND","PM","SM","EU","GD","TB","DY","HO","ER","TM","YB","LU",
      "HF","TA","W ","RE","OS","IR","PT","AU","HG","TL","PB","BI",
      "PO","AT","RN","FR","RA","AC","TH","PA","U ","NP","PU","AM",
      "CM","BK","CF","ES","FM","MD","NO","LR","  "} ;



/* ************************************************************************* */
/* ************************************************************************* */

/* ---------------------------------------- */
/*         Module Compute Routine           */
/* ---------------------------------------- */
int display_gaussian_compute(out3d_fld, out_geom, chkptname, 
			  Geom_Representation, scale, mo, compute_choice,
			  gridspacing, extent)
     AVSfield_float **out3d_fld;
     GEOMedit_list *out_geom;
     char *chkptname;
     float *scale;
     int mo;
     char *compute_choice;
     char *Geom_Representation;
     float *gridspacing;
     float *extent;

{
  int dims1[3];
  static FILE *ifp = NULL;     /* pointer to the input chkpt file */ 
  static int natoms = 0;       /* number of atoms read in from .chk file */
  static int norbs = 0;        /* number of orbitals read in from .chk file */
  static int nelecs = 0;       /* num valence electrons read from .chk file */
  static int numgpx = MAXPTS;
  static int numgpy = MAXPTS;
  static int numgpz = MAXPTS;
  int nat[MAXATOMS];                /* atomic number of the atom */
  int numfullmo,halffullmo,homo;    /* occupation of mo's */
  int maxatoms = MAXATOMS;   /* define var s.t. it can be passed to fort pgm */
  int maxpts = MAXPTS;       /* define var s.t. it can be passed to fort pgm */
  double xyz[MAXATOMS][3],     /* location of atoms  */
        x[MAXPTS],        /* location of x,y,z grid coordinate pts */
        y[MAXPTS],
        z[MAXPTS],
        psi[MAXPTS][MAXPTS][MAXPTS],
                                    /* wave function for a molecular orbital */
        eigvec[NUMORBS*NUMORBS],    /* normalized eigenvectors */
        density[MAXPTS][MAXPTS][MAXPTS], /* density matrix defined on grid */
        sc;                         /* sphere scale value */
  float location[MAXATOMS][3];      /* float equiv of xyz to pass to */
				    /* coord_to_geom */


  int i,j,k,imo;                    /* counters */
  int moflag,densityflag,nocomputeflag;   /* display compute flags */
  static char basistype[20] = "  "; /* name of the basis type selected */



  void compute_gridpoints(/* int natoms,
		   double xyz[][3],
		   float *gridspacing,
		   float *extent,
		   double x[],double y[],double z[],
		   int *numgpxptr,
		   int *numgpyptr,
		   int *numgpzptr SRT */);
     
  void minmax (/* int natoms,double xyz[][3],double xyzmin[],
	     double xyzmax[] SRT */);

  void readinchkpt(/* FILE *ifp,
	    int *natomsptr,int *norbsptr,int *nelecsptr,
	    double xyz[][3],
	    double eigvec[],
	    int nat[],
	    char basistype[20] SRT */);

  void coord_to_geom(/* GEOMedit_list *output,char *rep,int file_changed,
		     int natoms,float location[][3],double scale SRT */);




  /* the compute routine performs 4 basic functions: */
  /* 1) read in new data from the a gaussian92 ascii checkpoint file, this */
  /*    is done in the function readinchkpt */
  /* 2) create and display the appropriate geometry - in coord_to_geom */
  /* 3) compute the grid used to display the field data - in compute_grid */
  /* 4) compute the appropriate field data, either the wave function for a */
  /*    specified MO or else the density. There are three basis sets */
  /*    supported, 631,321, and sto. The type is read in from the input file */
  /*    and the appropriate subroutine, mo631g, mo321, or stomo, is called. */
  /*                                                                        */
  /* It is not necessary to perform all of these functions for each execution */
  /* of the compute routine, therefore a series of if statements are used to */
  /* limit execution to only the appropriate code. */


  /* NOTE: this module is applicable to restricted Hartree-Fock (RHF) */
  /* calculations only. The authors will consider extending this to UHF */
  /* molecules in the future. */




  /* ---------------------------------------------------------------------- */
  /* ---------------------------------------------------------------------- */
  sc = *scale;
  if(sc == 0.0) sc=1.0;

  /* check to see if the input file has changed */
  if (AVSparameter_changed ("chkpt file")) {

    /* open input file */
    if ((ifp = fopen(chkptname,"r")) == NULL) {
      AVSwarning("Error opening input chkpt file.");
      return(0);
    }
    
    /* call function to read in data */
    readinchkpt(ifp,
	      &natoms,&norbs,&nelecs,
	      xyz,
	      eigvec,
	      nat,
	      basistype);


    /* compute number of filled mo's and set num of half filled = 0 or 1 */
    /* set mo and modify widget to reflect default value of homo */
    numfullmo = nelecs/2;  /* must be integer division */
    halffullmo = nelecs % 2;
    homo = numfullmo + halffullmo ;
    mo = homo;
    AVSmodify_parameter("MO selected",AVS_MAXVAL | AVS_MINVAL | AVS_VALUE,
                 homo,1,norbs);




  } /* end of if chkpt file changed */
    

  /* ----------------------------------------------------------------------- */

  /* write message to standard out if no input data file has been selected */
  if (ifp==NULL) {
    printf("NO DATA CHECKPOINT FILE SELECTED\n\n\n");
    return(0);
  }



  /* ----------------------------------------------------------------------- */
  /* if the input file, grid spacing or extent has changed then recompute */
  /* the grid and redo the geometry */
  if (
      AVSparameter_changed ("chkpt file") || 
      AVSparameter_changed ("Grid Spacing") ||
      AVSparameter_changed ("Extent")
      ) {
    

    /* call function to compute the grid */
    compute_gridpoints(natoms,
		       xyz,
		       gridspacing,
		       extent,
		       x,y,z,
		       &numgpx,&numgpy,&numgpz);



    /* call function to plot the geometry */
    for (i=0;i<natoms;i++){
      strcpy(type[i],atom_names[nat[i]-1]);
    }

    /* convert double to float then pass to coord_to_geom */
    for (i=0;i<natoms;i++){
      for (j=0;j<3;j++){
	location[i][j] = xyz[i][j];
      }
    }

    coord_to_geom(out_geom,Geom_Representation,1,natoms,location,sc); 

    /* set the flags s.t. proper choice will be computed */
    nocomputeflag = 0;

    if (!strcmp(compute_choice,"MO")) moflag = 1;
    else if (!strcmp(compute_choice,"Density")) densityflag = 1;
    else nocomputeflag = 1; 


  }   /* end of if chkpt file or extent or spacing parms changed */

  /* ----------------------------------------------------------------------- */
  /* redo geom if new button is selected */
  if (AVSparameter_changed("Representation") 
      || AVSparameter_changed("Sphere Scale") ) {
    coord_to_geom(out_geom,Geom_Representation,0,natoms,location,sc);
  }


  /* ----------------------------------------------------------------------- */
  /* set appropriate flags if a new MO is selected */ 
  if (AVSparameter_changed("MO selected") && !strcmp(compute_choice,"MO")) {
    moflag = 1;
    nocomputeflag = 0;
  }


  /* ----------------------------------------------------------------------- */
  /* set flag for the compute function selected */
  if (AVSparameter_changed("Compute Choice")) {  
    moflag = 0;
    densityflag = 0;
    nocomputeflag = 0;
    if (!strcmp(compute_choice,"MO")) moflag = 1;
    else if (!strcmp(compute_choice,"Density")) densityflag = 1;
    else nocomputeflag = 1;
  }



  /* ----------------------------------------------------------------------- */
  /* check to see if any flag is set, if so alloc space and at end fill grid */
 
  if (!nocomputeflag) {

    /* Free old field data    */
    if (*out3d_fld) AVSfield_free(*out3d_fld);
    /* Allocate space for new field output   */
    dims1[0] = numgpx;   
    dims1[1] = numgpy;   
    dims1[2] = numgpz;   
    *out3d_fld = (AVSfield_float *) 
      AVSdata_alloc("field 3D scalar rectilinear float",dims1);
    if (*out3d_fld == NULL) {
      AVSerror("Allocation of output field for MO failed.");
      return(0);
    }


    /* compute specified molecular orbital wave function if flag is set */
    if (moflag) {
      
      if (strstr(basistype,"6-")!=NULL) 
	mo631g_(basistype,&natoms,&norbs,&maxatoms,&maxpts,&mo,nat,
		eigvec,xyz,x,y,z,psi,&numgpx,&numgpy,&numgpz);
      else if (strstr(basistype,"3-")!=NULL) 
	mo321g_(basistype,&natoms,&norbs,&maxatoms,&maxpts,&mo,nat,
		eigvec,xyz,x,y,z,psi,&numgpx,&numgpy,&numgpz);
      else if (strstr(basistype,"ST")!=NULL)
	stomo_(basistype,&natoms,&norbs,&maxatoms,&maxpts,&mo,nat,
	       eigvec,xyz,x,y,z,psi,&numgpx,&numgpy,&numgpz);
      else {
	AVSerror("ERROR!! no match found for basis type\n");
	return(0);
      }


      /* stuff the psi into the output field */
      /* the I3D macro computes the location of the i,j,k'th element for you */
      /* it expects as its first arg a pointer to an avs field structure */
      for (i=0;i<numgpx;i++){
	for (j=0;j<numgpy;j++){
	  for (k=0;k<numgpz;k++){
	    I3D(*out3d_fld,i,j,k) = psi[i][j][k];
	  }
	}
      }
      
      
    } /* end of if mo flag */



  /* ----------------------------------------------------------------------- */
    /* compute density if flag is set */  
    if (densityflag) {
      
      /* zero the density array */
      for (i=0;i<numgpx;i++){
	for (j=0;j<numgpy;j++){
	  for (k=0;k<numgpz;k++){
	    density[i][j][k] = 0.000000;
	  }
	}
      }
      
      
      /* loop over all the OCCUPIED mo's */
      /* first compute the number of full mo's */
      /* then check to see if nelecs is odd and thus if a half full mo exists */
      for (imo=1;imo<=numfullmo;imo++){
	if (strstr(basistype,"6-")!=NULL) 
	  mo631g_(basistype,&natoms,&norbs,&maxatoms,&maxpts,&imo,nat,
		  eigvec,xyz,x,y,z,psi,&numgpx,&numgpy,&numgpz);
	else if (strstr(basistype,"3-")!=NULL) 
	  mo321g_(basistype,&natoms,&norbs,&maxatoms,&maxpts,&imo,nat,
		  eigvec,xyz,x,y,z,psi,&numgpx,&numgpy,&numgpz);
	else if (strstr(basistype,"ST")!=NULL)
	  stomo_(basistype,&natoms,&norbs,&maxatoms,&maxpts,&imo,nat,
		 eigvec,xyz,x,y,z,psi,&numgpx,&numgpy,&numgpz);
	else {
	  AVSerror("ERROR!! no match found for basis type\n");
	  return(0);
	}

	
	
	
	for (i=0;i<numgpx;i++){
	  for (j=0;j<numgpy;j++){
	    for (k=0;k<numgpz;k++){
	      density[i][j][k] = density[i][j][k]
		+ 2*psi[i][j][k]*psi[i][j][k];
	    }
	  }
	}
      }
      /* now if a half full mo exists include it in the sum */
      if (halffullmo) {
	if (strstr(basistype,"6-")!=NULL) 
	  mo631g_(basistype,&natoms,&norbs,&maxatoms,&maxpts,&imo,nat,
		  eigvec,xyz,x,y,z,psi,&numgpx,&numgpy,&numgpz);
	else if (strstr(basistype,"3-")!=NULL) 
	  mo321g_(basistype,&natoms,&norbs,&maxatoms,&maxpts,&imo,nat,
		  eigvec,xyz,x,y,z,psi,&numgpx,&numgpy,&numgpz);
	else if (strstr(basistype,"ST")!=NULL)
	  stomo_(basistype,&natoms,&norbs,&maxatoms,&maxpts,&imo,nat,
		 eigvec,xyz,x,y,z,psi,&numgpx,&numgpy,&numgpz);
	else {
	  AVSerror("ERROR!! no match found for basis type\n");
	  return(0);
	}

	
	for (i=0;i<numgpx;i++){
	  for (j=0;j<numgpy;j++){
	    for (k=0;k<numgpz;k++){
	      density[i][j][k] = density[i][j][k] + psi[i][j][k]*psi[i][j][k];
	    }
	  }
	}


      }
      
      /* stuff the density into the output field */
      /* the I3D macro computes the location of the i,j,k'th element for you */
      /* it expects as its first argument a ptr to an avs field structure */
      for (i=0;i<numgpx;i++){
	for (j=0;j<numgpy;j++){
	  for (k=0;k<numgpz;k++){
	    I3D(*out3d_fld,i,j,k) = density[i][j][k];
	  }
	}
      }
      
      
    }  /* end of if density flag */
    
    
    /* --------------------------------------------------------------------- */
    /* now stuff the grid points into the output field */
    /* since the field is rectilinear we need to specify one row of x's and */
    /* one column of y's and one of z's */
    for (i=0;i<numgpx;i++){
      RECT_X(*out3d_fld)[i] = x[i];
    }
    for (i=0;i<numgpy;i++){
      RECT_Y(*out3d_fld)[i] = y[i];
    }
    for (i=0;i<numgpz;i++){
      RECT_Z(*out3d_fld)[i] = z[i];
    }
  }  /* end of if a compute choice was selected */

  /* ----------------------------------------------------------------------- */
  /* set flags and return */
  moflag = 0;
  densityflag = 0;
  nocomputeflag = 1;


return(1);
}


/* ************************************************************************* */
/* ************************************************************************* */
void compute_gridpoints(
			natoms,
			xyz,
			gridspacing,
			extent,
			x,y,z,
			numgpxptr,
			numgpyptr,
			numgpzptr)

			int natoms;
			double xyz[][3];
			float *gridspacing;
			float *extent;
			double x[], y[], z[];
			int *numgpxptr;
			int *numgpyptr;
			int *numgpzptr;
     
     
{
  double xyzmin[3],                  /* min value of xyz respectively */
        xyzmax[3],                  /* max value of xyz respectively */
        xmin,ymin,zmin,
        xmax,ymax,zmax;
  int i;
  float extentau,gridau;            /* converted values of extent & spacing */
  
  
  /* This function computes the rectilinear field grid for the output field. */
  /* In fact, the grid spacing is uniform in all directions, and is specified */
  /* by the input parameter grid spacing. Note that the grid values are all */
  /* in AU(Bohr) and that gridspacing and extent are converted from Angstroms */
  /*  to AU. The grid is determined by finding the min and max atomic coords */
  /*  in each direction and then going beyond these values by the extent */
  /*  setting. Thus the number of grid points in each direction may vary. */

 
  printf ("ENTERING compute_gridpoints, computing grid values for cube\n");


  /* call function to find min and max values of xyz */
  minmax (natoms,xyz,xyzmin,xyzmax);
  

  /* note that extent and gridspacing are in angstroms */  
  extentau = *extent * AUINV;
  gridau = *gridspacing * AUINV;

  /* extend the min and max values */
  xmin = xyzmin[0]-extentau;
  xmax = xyzmax[0]+extentau;
  ymin = xyzmin[1]-extentau;
  ymax = xyzmax[1]+extentau;
  zmin = xyzmin[2]-extentau;
  zmax = xyzmax[2]+extentau;

  /* compute the number of grid points in each direction */
  *numgpxptr = ceil((xmax-xmin)/(gridau));
  *numgpyptr = ceil((ymax-ymin)/(gridau));
  *numgpzptr = ceil((zmax-zmin)/(gridau));
  
  /* assign the grid point arrays */
  x[0] = xmin;
  y[0] = ymin;
  z[0] = zmin;
  for (i=1;i<*numgpxptr;i++){
    x[i] = gridau+x[i-1];
  }
  for (i=1;i<*numgpyptr;i++){
    y[i] = gridau+y[i-1];
  }
  for (i=1;i<*numgpzptr;i++){
    z[i] = gridau+z[i-1];
  }


  printf("grid points computed\n");
  
  return;
}

/* ************************************************************************* */
/* ************************************************************************* */
void minmax ( natoms, xyz, xyzmin, xyzmax ) 
int natoms;
double xyz[][3];
double xyzmin[];
double xyzmax[];
/* this function finds the min and max values of the atomic locations */     
{
  int i,j;
  double minval,maxval;
  
  for (i=0;i<3;i++){  
    minval = 100.0e0;
    maxval = -100.0e0;
    for (j=0;j<natoms;j++){
      if(minval>xyz[j][i]) minval = xyz[j][i];
      if(maxval<xyz[j][i]) maxval = xyz[j][i];
    }
    xyzmin[i] = minval;
    xyzmax[i] = maxval;
  }
  
  return;
}


/* ********************************************************************** */
/* ********************************************************************** */
void readinchkpt(ifp,
		 natomsptr,
		 norbsptr,
		 nelecsptr,
		 xyz,
		 eigvec,
		 nat,
		 basistype)

		 FILE *ifp;
		 int *natomsptr;
		 int *norbsptr;
		 int *nelecsptr;
		 double xyz[][3];
		 double eigvec[];
		 int nat[];
		 char basistype[20];
{
  /* this function reads in the data from the .chk file and returns it to  */
  /* the calling routine */

  float vtmp[NUMORBS*NUMORBS],eigval[NUMORBS],atmchg[MAXATOMS];
  float dummy,evectors[NUMORBS][NUMORBS],xyzfloat[MAXATOMS][3];
  int natoms,norbs,icharg,multip,nae,nbe,ne,nbasis;
  int i,j,numvars,unpaired,npaired;
  int jmo,icnt;
  char dummyline[RECLEN],inpline[RECLEN],leadingjunk[RECLEN];
  char *basistypeptr,*anflag=0;
  int iuflag = 0;


/*
c retrieve coordinates and other useful info. from the checkpoint file.
*/
  /* fgets is used to insure an entire line is read */
  /* sscanf is used to process the line and read up to the first number */
  /* after reading in the appropriate number of data values,another fgets */
  /* is often done to flush the line */

  fgets(dummyline,RECLEN,ifp);
  puts(dummyline);


  fgets(inpline,RECLEN,ifp);
  puts(inpline);
  basistypeptr = strstr(inpline,"4-");
  if (basistypeptr==NULL) basistypeptr = strstr(inpline,"6-");
  if (basistypeptr==NULL) basistypeptr = strstr(inpline,"3-");
  if (basistypeptr==NULL) basistypeptr = strstr(inpline,"ST");
  if (basistypeptr==NULL) {
    printf("ERROR!! no match found for basis type\n");
    exit(0);
  }
  /* copy string to basistype, making sure it does not exceed lenght of 20 */
  /* and leaving room for null terminator */
/*   printf("basistypeptr = %s\n",basistypeptr); */
  strncpy(basistype,basistypeptr,19);
  strcat(basistype,"\0");
/*   printf("basistype = %s\n",basistype); */

  fgets(inpline,RECLEN,ifp);
  sscanf(inpline,"%[^0-9] %d",leadingjunk,&natoms);
/*   printf("ntaoms = %d\n",natoms); */

  fgets(inpline,RECLEN,ifp);
  sscanf(inpline,"%[^0-9] %d",leadingjunk,&icharg);
/*   printf("icharg = %d\n",icharg); */

  fgets(inpline,RECLEN,ifp);
  sscanf(inpline,"%[^0-9] %d",leadingjunk,&multip);
/*   printf("multip = %d\n",multip); */

  fgets(inpline,RECLEN,ifp);
  sscanf(inpline,"%[^0-9] %d",leadingjunk,&ne);
/*   printf("ne = %d\n",ne); */

  fgets(inpline,RECLEN,ifp);
  sscanf(inpline,"%[^0-9] %d",leadingjunk,&nae);
/*   printf("leader is:%s",leadingjunk); */
/*   printf("nae = %d\n",nae); */

  fgets(inpline,RECLEN,ifp);
  sscanf(inpline,"%[^0-9] %d",leadingjunk,&nbe);
/*   printf("leader is:%s",leadingjunk); */
/*   printf("nbe = %d\n",nbe); */

  fgets(inpline,RECLEN,ifp);
  sscanf(inpline,"%[^0-9] %d",leadingjunk,&nbasis);
/*   printf("leader is:%s",leadingjunk); */
/*   printf("nbasis = %d\n",nbasis); */

  /* read dummy lines, convert to lower case and search for the string: */
  /* atomic number */
  while (anflag==NULL) {
    if (fgets(dummyline,RECLEN,ifp)==NULL) {
      printf
   ("read error, perhaps EOF encountered while searching for atomic number\n");
      exit(0);
    }
    for (j=0;j<RECLEN;j++){
      if (isupper(dummyline[j])) dummyline[j] = tolower(dummyline[j]);
    }
    anflag = strstr(dummyline,"atomic number");
    puts(dummyline);
  }

  /* read atomic numbers */
  for (i=0;i<natoms;i++){
    fscanf(ifp,"%d",&nat[i]);
/*     printf("nat %d = %d\n",i,nat[i]); */
  }
  fgets(dummyline,RECLEN,ifp);


  /* read header then read nuclear charges */
  fgets(dummyline,RECLEN,ifp);
  puts(dummyline);
  for (i=0;i<natoms;i++){
    fscanf(ifp," %f ",&atmchg[i]);
/*     printf("atmchg %d = %f\n",i,atmchg[i]); */
  }
/*  fgets(dummyline,RECLEN,ifp);  */

  /* read header then read cartesian coordinates of atoms (in Angstroms) */
  fgets(dummyline,RECLEN,ifp);
  puts(dummyline);
  for (i=0;i<natoms;i++){
      fscanf(ifp,"%f %f %f ",&xyzfloat[i][0],&xyzfloat[i][1],&xyzfloat[i][2]);
      xyz[i][0] = xyzfloat[i][0];
      xyz[i][1] = xyzfloat[i][1];
      xyz[i][2] = xyzfloat[i][2];
/*     printf("xyz row %d = %f %f %f\n",i,xyz[i][0],xyz[i][1],xyz[i][2]); */
  }
/*  fgets(dummyline,RECLEN,ifp); */

  /* in the following cases, a header line must be read and a number */
  /* extracted which tells how many dummy vars to read */
  /* this is repeated 7 times */
  for (j=0;j<7;j++) {
    fgets(inpline,RECLEN,ifp);
    sscanf(inpline,"%[^0-9] %d",leadingjunk,&numvars);
/*     printf("leader is:%s:: numvars = %d\n",leadingjunk,numvars); */
    for (i=0;i<numvars;i++) {
      fscanf(ifp,"%f",&dummy);
    }
    fgets(inpline,RECLEN,ifp);
  }

  fgets(inpline,RECLEN,ifp);
  sscanf(inpline,"%[^0-9] %d",leadingjunk,&numvars);
/*   printf("numvars = %d\n",numvars); */
  for (i=0;i<numvars;i++) {
    fscanf(ifp,"%f",&eigval[i]);
  }
  fgets(inpline,RECLEN,ifp);

  /* read the data into a temp float array and then convert to double array */
  /* that is to be passed back */
  fgets(inpline,RECLEN,ifp);
  sscanf(inpline,"%[^0-9] %d",leadingjunk,&numvars);
/*   printf("numvars = %d\n",numvars); */
  for (i=0;i<numvars;i++) {
    fscanf(ifp,"%f",&vtmp[i]);
    eigvec[i] = vtmp[i];
  }
  fgets(inpline,RECLEN,ifp);
/*   printf("read in eigvec %f %f %f\n",eigvec[0],eigvec[1],eigvec[2]); */


/*
c determine homo, lumo and other useful info.
c iuflag = 1 if there are unpaired electrons.
c norbs is the total number of orbitals that will be given in the
c       psi1 input file.
c npaired is the number of paired electrons.
*/
  unpaired = multip-1;
  if (unpaired != 0) iuflag = 1;
  npaired = (ne-unpaired)/2;
  norbs = nbasis;


  jmo = -1;


  icnt = 0;
  for (i=0;i<nbasis;i++){
    for (j=0;j<nbasis;j++){
      evectors[j][i] = eigvec[icnt];
      icnt++;
    }
  }

 
  if (iuflag==1) {
    if (jmo > 0) {
      for (i=0;i<nbasis;i++) {
	vtmp[i] = evectors[i][jmo];
      }
    }
  }




/* end input  *************************** */


  /* FINISHED READING IN DATA FROM .CHK FILE */
  
  printf("finished reading in data\n");
  
  /* point pointers to addresses of vars so they will be returned */
  *natomsptr = natoms;
  *norbsptr = norbs;
  *nelecsptr = ne;
  
  return;
}



/* ********************************************************************** */
/* ********************************************************************** */
void coord_to_geom(output,rep,file_changed,natoms,location,scale)

GEOMedit_list *output; 
char *rep;
/* char atype[][4]; */
int file_changed;
int natoms;
float location[][3];
double scale;
{ 
  int	i;
  int rep_type;
  float bondlength;
  int	atom_count, bond_count;
  GEOMobj 	*atoms_obj, *bonds_obj;
  GEOMobj	*objs[2];
  float ballscale = STICKSANDSPHERES_RADFACTOR;
  int vcolors, a, b;
  int	bonds[MAXBONDS][2]; 
  int	atomtype[MAXATOMS];
  float *lcols;


  /* this routine produces the geometry output for the molecule */
  /* the geometry may be colored and sized based on atom type */


  /* -------------    start of routine    ------------- */
  printf ("\nENTERING coord_to_geom routine.\n");


  *output = GEOMinit_edit_list(*output); 

  if (!strcmp(rep,"cpk")) rep_type = BALL;
  else if (!strcmp(rep,"stick")) rep_type = STICK;
  else if (!strcmp(rep,"colored stick")) rep_type = COLORED_STICK;
  else rep_type = BALL_AND_STICK;


  if (rep_type == BALL) ballscale = 1.0;

  ballscale = ballscale*scale*AUINV;
  bondlength  = BONDLENGTH*AUINV;
  

  if (file_changed) {
     makeatomtype(type,natoms,atomtype);
     num_bonds = makebonds(location,type,natoms,MAXBONDS,bondlength,bonds);
  }



  if (rep_type == COLORED_STICK) {
     for (i = 0; i < natoms; i++) {
	bond_colors[i][0] = atomcolors[atomtype[i]][0];
	bond_colors[i][1] = atomcolors[atomtype[i]][1];
	bond_colors[i][2] = atomcolors[atomtype[i]][2];
     } 
     vcolors = 1;
  }
  else vcolors = 0;

  bond_count = 0;
  for(i=0;i<num_bonds;i++)
  {
	  a = bonds[i][0]; 
	  b = bonds[i][1];

	  b_lst[2*bond_count][0] = location[a][0];
	  b_lst[2*bond_count][1] = location[a][1];
	  b_lst[2*bond_count][2] = location[a][2];

	  if (vcolors) {
	     b_rgb_lst[2*bond_count][0] = bond_colors[a][0];
	     b_rgb_lst[2*bond_count][1] = bond_colors[a][1];
	     b_rgb_lst[2*bond_count][2] = bond_colors[a][2];

	     b_rgb_lst[2*bond_count+1][0] = bond_colors[a][0];
	     b_rgb_lst[2*bond_count+1][1] = bond_colors[a][1];
	     b_rgb_lst[2*bond_count+1][2] = bond_colors[a][2];
	  }

	  if (vcolors && (bond_colors[a][0] != bond_colors[b][0] ||
	      bond_colors[a][1] != bond_colors[b][1] ||
	      bond_colors[a][2] != bond_colors[b][2])) {

	     b_lst[2*bond_count+2][0] = b_lst[2*bond_count+1][0] = 
				(location[a][0] + location[b][0])/2.;
	     b_lst[2*bond_count+2][1] = b_lst[2*bond_count+1][1] = 
				(location[a][1] + location[b][1])/2.;
	     b_lst[2*bond_count+2][2] = b_lst[2*bond_count+1][2] = 
				(location[a][2] + location[b][2])/2.;

	     b_rgb_lst[2*bond_count+2][0] = bond_colors[b][0];
	     b_rgb_lst[2*bond_count+2][1] = bond_colors[b][1];
	     b_rgb_lst[2*bond_count+2][2] = bond_colors[b][2];

	     b_rgb_lst[2*bond_count+3][0] = bond_colors[b][0];
	     b_rgb_lst[2*bond_count+3][1] = bond_colors[b][1];
	     b_rgb_lst[2*bond_count+3][2] = bond_colors[b][2];

	     bond_count++;
	  }
	     
	  b_lst[2*bond_count+1][0] = location[b][0];
	  b_lst[2*bond_count+1][1] = location[b][1];
	  b_lst[2*bond_count+1][2] = location[b][2];
	  bond_count++;
  }
  
  atom_count = 0;

  for(i=0;i<natoms;i++)
  {   
      {   a_lst[atom_count][0]   = location[i][0];
	  a_lst[atom_count][1]   = location[i][1];
	  a_lst[atom_count][2]   = location[i][2];
	  rgb_lst[atom_count][0] = atomcolors[atomtype[i]][0];
	  rgb_lst[atom_count][1] = atomcolors[atomtype[i]][1];
	  rgb_lst[atom_count][2] = atomcolors[atomtype[i]][2];
	  r_lst[atom_count] = radius[atomtype[i]] * ballscale;
	  atom_count++;
      }
  }


  if ((rep_type == BALL || rep_type == BALL_AND_STICK) && atom_count) {
     atoms_obj = GEOMcreate_sphere(NULL,a_lst,r_lst,NULL,NULL,atom_count,0);  
     GEOMadd_float_colors(atoms_obj,rgb_lst,atom_count,0);
  }
  else atoms_obj = GEOMcreate_obj(GEOM_SPHERE,NULL);
  bonds_obj = GEOMcreate_obj(GEOM_POLYTRI,NULL);

  if (rep_type != BALL && bond_count) {
     if (vcolors) lcols = &b_rgb_lst[0][0];
     else lcols = NULL;
     GEOMadd_disjoint_line(bonds_obj,b_lst,lcols,2*bond_count,0);
  }

  objs[0] = atoms_obj;
  objs[1] = bonds_obj;



  GEOMset_extent(atoms_obj);
  GEOMset_extent(bonds_obj);
  GEOMunion_extents(atoms_obj,bonds_obj);


  GEOMedit_geometry(*output,"molecule",atoms_obj);
  GEOMedit_geometry(*output,"molecule",bonds_obj);
  /* Set the window -- this will transform the top so that stuff is visible 
  GEOMedit_window(*output,"molecule",atoms_obj->extent); */

  GEOMdestroy_obj(atoms_obj);
  GEOMdestroy_obj(bonds_obj);

  return;
}



/* ********************************************************************** */
/* ********************************************************************** */
makebonds(apos, atype, natoms, maxnbonds, threshold, bonds)
    float apos[][3];
    char      atype[][4];
    int       natoms;
    int       maxnbonds;
    float     threshold;
    int       bonds[][2]; 
{
    /*
     * Generate into bonds the indices into apos of neighbors. Returns number
     * of bonds. 
     *
     * This version is a simple-minded distance check with special case code to
     * prevent hydrogen over-connectivity. Mike Pique 
     */

    register int i, j;
    register int nbonds;
    register float dx, dy, dxy, dz;
    float     h_len = 1.4;

/*    if(units == BOHR) h_len = h_len/0.52914; */
    h_len = h_len/0.52914;  

    nbonds = 0;
    for (i = natoms - 1; i > 0; i--) {
	for (j = i - 1; j >= 0 && nbonds < maxnbonds; j--) {
	    /*
	     * The outer loop index 'i' is AFTER the inner loop 'j': 'i'
	     * leads 'j' in the list: since hydrogens traditionally follow
	     * the heavy atom they're bonded to, this makes it easy to quit
	     * bonding to hydrogens after one bond is made by breaking out of
	     * the 'j' loop when 'i' is a hydrogen and we make a bond to it.
	     * Working backwards like this makes it easy to find the heavy
	     * atom that came 'just before' the Hydrogen. mp 
	     */

	    /* never bond hydrogens to each other... */
	    if (atype[i][0] == 'H' && atype[j][0] == 'H')
		continue;

	    dx = apos[i][0] - apos[j][0];
	    if (dx < 0)
		dx = -dx;
	    if (dx > threshold)
		continue;
	    dy = apos[i][1] - apos[j][1];
	    if (dy < 0)
		dy = -dy;
	    if (dy > threshold)
		continue;
	    if ((dxy = hypot(dx, dy)) > threshold)
		continue;

	    dz = apos[i][2] - apos[j][2];
	    if (dz < 0)
		dz = -dz;
	    if (dz > threshold)
		continue;
	    if (hypot(dxy, dz) > threshold)
		continue;
            if (atype[i][0] == 'H' || atype[j][0] == 'H')
	      if(hypot(dxy,dz) > h_len )
		 continue;
	    bonds[nbonds][0] = j;
	    bonds[nbonds][1] = i;
#ifdef DEBUG
	    printf("bond[%d] %d to %d\n", nbonds, i, j);
#endif
	    nbonds++;
	    if (atype[i][0] == 'H')
		break;		/* only one bond per hydrogen */
	    if (atype[i][0] == 'h')
		break;		/* only one bond per hydrogen */
	}
    }

#ifdef NOT_NEEDED_ANYMORE
    if (1) {
	FILE *pf;
	char filename[128];
	int dummy[1];
        
	sprintf(filename, "FLEX.bnd");
	
	if ((pf = fopen(filename, "w"))) {
	    dummy[0] = nbonds;
	    fwrite((char *)dummy, sizeof(int), 1, pf);
	    fwrite((char *) bonds, sizeof(int), 2 * nbonds, pf);
	    fclose(pf);
	}
    }
#endif
    
    return nbonds;
}

/* ********************************************************************** */
/* ********************************************************************** */
int
makeatomtype(atype, natoms, atomtype)
    char      atype[][4];
int       natoms;
int       atomtype[];

{

    /*
     * Generate into atomtype the integer 0..7 atom type Returns number of
     * atoms	
     *
     */

    register int i;
    char      a, b;
    int       anum;

    for (i = 0; i < natoms; i++) {
	a = atype[i][0];
	if (islower(a))
	    a = toupper(a);
	switch (a) {
	  case 'H':
	    anum = 3;
	    break;
	  case 'C':
	    b = atype[i][1];
	    if (islower(b))
	       b = toupper(b);
	    if (b == 'L')
	       anum = 6;
            else
	       anum = 0;
	    break;
	  case 'O':
	    anum = 1;
	    break;
	  case 'N':
	    anum = 2;
	    break;
	  case 'S':
	    anum = 4;
	    break;
	  case 'P':
	    anum = 5;
	    break;
	  case 'I':
	    anum = 7;
	    break;
	  case 'F':
	    anum = 8;
	    break;
	  case 'B':
	    anum = 9;
	    break;
	  case 'Z':
	    anum = 10;
	    break;
	  default:
	    anum = 11;
	    break;
	}
	atomtype[i] = anum;
    }
    return natoms;
}

