/*****************************************************************************
                  INTERNATIONAL AVS CENTER
        (This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.



****************************************************************************/
/* Module: cylinder							   */
/* Author: Beth Hess under the supervision of Dr. Susanna Wei		   */
/*	   St. Joseph's University					   */
/* Note: This module is a modification of Cylinder written by Eric Sills,  */
/*       North Carolina Supercomputing Center				   */
/***************************************************************************/

/****************************/
/* Header File		    */
/****************************/

#include <stdio.h>
#include <avs/avs.h>
#include <avs/port.h>
#include <avs/geom.h>
#include <avs/avs_math.h>

/****************************/
/* Define Constants         */
/****************************/

#define MAXCYL 21			/* max number of cylinders */
static char version[] = "%W% %E%";  	/* module version string */
 
/****************************/
/* Module Description       */
/****************************/

int cylinder_desc()
{

	int in_port, out_port, param;
	int cylinder_compute();
	int cylinder_init();

	/* module name and type */
	AVSset_module_name("cylinder", MODULE_DATA);

	/* output port specifications */
	out_port = AVScreate_output_port("geo_out", "geom");

	/* Parameter Specifications */

	/* number of sides or rectangles in the cylinder */
	param = AVSadd_parameter("nsides", "integer", 20, 4, 72);
	AVSconnect_widget(param, "islider");

	/* file browser */
	param = AVSadd_parameter("data_file", "string","",""," ");
	AVSconnect_widget(param, "browser");

	/* read or write choice button */
	param = AVSadd_parameter("read_write", "choice", "write", "read write", " ");
	AVSconnect_widget(param, "radio_buttons");

	/* name of current cylinder */
	param = AVSadd_parameter("cylinder","integer",20,0,20);
	AVSconnect_widget(param,"typein_integer");

	/* color toggle */
	param = AVSadd_parameter("color_on", "boolean", 0, 0, 1);
	AVSconnect_widget(param,"toggle");

	/* x coordinate */	
	param = AVSadd_float_parameter("x", 0.0, -50.0, 50.0);
	AVSconnect_widget(param,"typein_real");

	/* y coordinate */
	param = AVSadd_float_parameter("y", 0.0, -50.0, 50.0);
	AVSconnect_widget(param,"typein_real");
	
	/* z coordinate */
	param = AVSadd_float_parameter("z", 0.0, -50.0, 50.0);
	AVSconnect_widget(param,"typein_real");

	/* cylinder radius */
	param = AVSadd_float_parameter("radius", 0.0, 0.0, 50.0);
	AVSconnect_widget(param,"typein_real");
 
	/* cylinder height */
	param = AVSadd_float_parameter("height", 0.0, 0.0, 50.0);
	AVSconnect_widget(param,"typein_real");

	/* x rotation */
	param = AVSadd_float_parameter("rot x", 0.0, -360.0, 360.0);
	AVSconnect_widget(param,"typein_real");

	/* y rotation */
	param = AVSadd_float_parameter("rot y", 0.0, -360.0, 360.0);
	AVSconnect_widget(param,"typein_real");

	/* declare the initialization function */
	AVSset_init_proc(cylinder_init);

	/* declare the computation function */
	AVSset_compute_proc(cylinder_compute);

	return(1);

} /***** End Module Description *****/


/***********************************/
/* Define cylinder structure       */
/***********************************/

struct cylinders {
	int color_on;
	float x, y, z;
	float r, h;
	float rx, ry;
} cylind[MAXCYL];


/***********************************/
/* Module Initialization           */
/***********************************/

cylinder_init() {
	int i;

/* set cylinder color, coordinates, radius, height and rotations to zero */
	for (i=0; i<MAXCYL; i++) 
	{
		cylind[i].color_on = 0;
		cylind[i].x = cylind[i].y = cylind[i].z = 0.0;
		cylind[i].r = cylind[i].h = 0.0;
		cylind[i].rx = cylind[i].ry = 0.0;
	}
}

	
/***********************************/
/* Module Compute Routine          */
/***********************************/

int cylinder_compute( geo_out, nsides, data_file, read_write, cylinder,color_on,
                      x, y, z, radius, height, rx, ry)
	GEOMedit_list *geo_out;
	int nsides;
	char *data_file, *read_write;
	int cylinder;
	int color_on;
	float *x, *y, *z;
	float *radius, *height;
	float *rx, *ry;

{

  	int i, idx, icyl, flags = 0;
	int num_vert;		/* number of vertices */
  	int *plist;		/* list of polygon vertex indices */
	int *itmp;
	float *verts;		/* vertex coordinates */
	float *norms;		/* array of normals, one per vertex */
  	float *theta, *tmp, *ntmp;
	float *colors;		/* array of RGB colors, one per vertex */
  	char objname[MAXCYL][80];	/* object name array */
	char *message;
  	FILE *fptr;		/* file pointer */
  	GEOMobj *obj[MAXCYL];	/* object array */


/* read cylinder information from file                                   */

/* select a new data file */
if (AVSparameter_changed("data_file", 0)) 
{
	if(!strcmp(read_write, "read")) 
	{

	/* attempt to read the data file */
		
		if (( fptr = fopen(data_file, "r")) == NULL) 
		{

		/* no read permission */
      			AVSwarning("cylinders: Error opening file %s",data_file);
      			return(1);
    		} 

  	cylinder = 0;

	/* read the data file */
  		while ((! feof(fptr)) && (cylinder < MAXCYL))
    		{
      			fscanf(fptr,"%f %f %f %f %f %f %f\n", &(cylind[cylinder].x), 
				&(cylind[cylinder].y), &(cylind[cylinder].z),
	      			&(cylind[cylinder].r), &(cylind[cylinder].h),
				&(cylind[cylinder].rx), &(cylind[cylinder].ry));

			/* set color to off */
			cylind[cylinder].color_on = 0;
      			cylinder++;
    		}

	/* modify parameter values*/

	AVSmodify_parameter("nsides",AVS_VALUE,nsides,0,0);
	AVSmodify_parameter("color_on",AVS_VALUE,cylind[cylinder].color_on,0,0);
	AVSmodify_float_parameter("x",AVS_VALUE,cylind[cylinder].x,0,0);
	AVSmodify_float_parameter("y",AVS_VALUE,cylind[cylinder].y,0,0);
	AVSmodify_float_parameter("z",AVS_VALUE,cylind[cylinder].z,0,0);
	AVSmodify_float_parameter("radius",AVS_VALUE,cylind[cylinder].r,0,0);
	AVSmodify_float_parameter("height",AVS_VALUE,cylind[cylinder].h,0,0);
	AVSmodify_float_parameter("rot x",AVS_VALUE,cylind[cylinder].rx,0,0);
	AVSmodify_float_parameter("rot y",AVS_VALUE,cylind[cylinder].ry,0,0);
}
	else 
	{
	/* attempt to write the data file */

	if (fptr = fopen(data_file,"r")) 
	{
	/* file already exists */

		fclose(fptr);
	/* warning: cancel or overwrite the existing file */
		message = AVSmessage(version, AVS_Warning, NULL,
			"Module Generator", "Cancel!Overwrite",
			"Overwrite File %s? \n",data_file);

		if (!strcmp(message, "Cancel")) 
		{
		/* cancel the write to file */
			AVSmodify_parameter("data_file", AVS_VALUE,
				NULL,NULL,NULL);
			return(0);
		
		}
	}
	if ((fptr = fopen(data_file, "w")) == NULL) 
	{
	/* no write permission */
		AVSwarning("cylinders: Error opening file %s",data_file);
		return(1);
	}
	/* write the data file */
	for (i=0; i<MAXCYL; i++) 
	{
		if (cylind[i].h > 0.0)
			fprintf(fptr, "%f %f %f %f %f %f %f\n", cylind[i].x,
        		   cylind[i].y, cylind[i].z, cylind[i].r, cylind[i].h,
			   cylind[i].rx, cylind[i].ry);
	}
	}
	fclose(fptr);
}

if (AVSparameter_changed("cylinder",0)) 
{
	for (i=0; i<MAXCYL; i++) 
		cylind[i].color_on = 0;
	
	AVSmodify_parameter("nsides",AVS_VALUE,nsides,0,0);
	AVSmodify_parameter("color_on",AVS_VALUE,cylind[cylinder].color_on, 0, 0);
	AVSmodify_float_parameter("x",AVS_VALUE,cylind[cylinder].x, 0, 0);
	AVSmodify_float_parameter("y",AVS_VALUE,cylind[cylinder].y, 0, 0);
	AVSmodify_float_parameter("z",AVS_VALUE,cylind[cylinder].z, 0, 0);
	AVSmodify_float_parameter("radius",AVS_VALUE,cylind[cylinder].r, 0, 0);
	AVSmodify_float_parameter("height",AVS_VALUE,cylind[cylinder].h, 0, 0);
	AVSmodify_float_parameter("rot x",AVS_VALUE,cylind[cylinder].rx, 0, 0);
	AVSmodify_float_parameter("rot y",AVS_VALUE,cylind[cylinder].ry, 0, 0);

}

/* set up the current cylinder */

nsides = nsides;
cylind[cylinder].color_on = color_on;
cylind[cylinder].x = *x;
cylind[cylinder].y = *y;
cylind[cylinder].z = *z;
cylind[cylinder].r = *radius; 
cylind[cylinder].h = *height;
cylind[cylinder].rx = *rx;
cylind[cylinder].ry - *ry;

/* Create the geo_out object                                             */

for (icyl = 0; icyl < MAXCYL; icyl++) 
{
    	if (cylind[icyl].h > 0.0) 
 	{   	
      		theta = (float *) ALLOC_LOCAL(nsides*sizeof(float));
      		for (i = 0; i < nsides; i++)
			theta[i] = (2*M_PI*i)/nsides;
		/* number of vertices - 4 for each rectangle */
      		num_vert = 4*nsides;
		/* allocate space for vertex coordinate array */
      		verts = (float *) ALLOC_LOCAL(3*num_vert*sizeof(float));
      		tmp = verts;
		/* allocate space for array of vertex normals */
      		norms = (float *) ALLOC_LOCAL(3*num_vert*sizeof(float));
      		ntmp = norms;
		/* allocate space for array of RGB colors */
		colors = (float *) ALLOC_LOCAL(3*num_vert*sizeof(float));

/* Determine vertices, normals and set color to red */
      	for (i = 0; i < nsides; i++)
	{
	  	idx = 3*i;
	  	verts[idx] = cylind[icyl].r * cos(theta[i]) + cylind[icyl].x;
	  	verts[idx+1] = cylind[icyl].r * sin(theta[i]) + cylind[icyl].y;
	  	verts[idx+2] = cylind[icyl].h/2. + cylind[icyl].z;

	  	norms[idx] = (verts[idx] - cylind[icyl].x)/cylind[icyl].r;
	  	norms[idx+1] = (verts[idx+1] - cylind[icyl].y)/cylind[icyl].r;
	  	norms[idx+2] = 0;

		colors[idx] = 1.0;
		colors[idx+1] = 0.0;
		colors[idx+2] = 0.0;
	}
      	for (i = 0; i < nsides; i++)
	{
	  	idx = 3*(i + nsides);
	  	verts[idx] = cylind[icyl].r * cos(theta[i]) + cylind[icyl].x;
	  	verts[idx+1] = cylind[icyl].r * sin(theta[i]) + cylind[icyl].y;
	  	verts[idx+2] = -cylind[icyl].h/2. + cylind[icyl].z;
      
	  	norms[idx] = (verts[idx] - cylind[icyl].x)/cylind[icyl].r;
	  	norms[idx+1] = (verts[idx+1] - cylind[icyl].y)/cylind[icyl].r;
	  	norms[idx+2] = 0;

		colors[idx] = 1.0;
		colors[idx+1] = 0.0;
		colors[idx+2] = 0.0;
	}
      	for (i = 0; i < nsides; i++)
	{
	  	idx = 3*(i + 2*nsides);
	  	verts[idx] = verts[idx - 6*nsides];
	  	verts[idx+1] = verts[idx + 1 - 6*nsides];
	  	verts[idx+2] = verts[idx + 2 - 6*nsides];

	  	norms[idx] = 0.;
	  	norms[idx+1] = 0.;
	  	norms[idx+2] = 1.;

		colors[idx] = 1.0;
		colors[idx+1] = 0.0;
		colors[idx+2] = 0.0;
	}
      	for (i = 0; i < nsides; i++)
	{
	  	idx = 3*(i + 3*nsides);
	  	verts[idx] = verts[idx - 6*nsides];
	  	verts[idx+1] = verts[idx + 1 - 6*nsides];
	  	verts[idx+2] = verts[idx + 2 - 6*nsides];
	  
	  	norms[idx] = 0.;
	  	norms[idx+1] = 0.;
	  	norms[idx+2] = -1.;

		colors[idx] = 1.0;
		colors[idx+1] = 0.0;
		colors[idx+2] = 0.0;
	}

/* allocate space for polygon vertex indices */
      	plist = (int *) ALLOC_LOCAL((nsides*5 + 2*(nsides + 1) + 1)*
				  sizeof(int)); 
      	itmp = plist;
      	for (i = 1; i < nsides; i++)
	{
	  	*itmp = 4;
	  	itmp++;
	  	*itmp = i;
	 	itmp++;
	  	*itmp = i + nsides;
	  	itmp++;
	  	*itmp = i + nsides + 1;
	  	itmp++;
	  	*itmp = i + 1;
	  	itmp++;
	}
      	*itmp = 4;
      	itmp++;
      	*itmp = nsides;
      	itmp++;
      	*itmp = 2*nsides;
      	itmp++;
      	*itmp = nsides+1;
      	itmp++;
      	*itmp = 1;
      	itmp++; 

      	*itmp = nsides;
      	itmp++;
      	for (i = 2*nsides + 1; i <= 3*nsides; i++)
	{
	  	*itmp = i;
	  	itmp++;
	}

      	*itmp = nsides;
      	itmp++;
      	for (i = 3*nsides + 1; i <= 4*nsides; i++)
	{
	  	*itmp = i;
	  	itmp++;
	}
    
      	*itmp = 0;

/* create a GEOM_POLYHEDRON                                              */
      	obj[icyl] = GEOMcreate_polyh(GEOM_NULL, verts, num_vert, plist, 
				      flags, GEOM_COPY_DATA);
/* name the object */
	sprintf(objname[icyl],"cylinder%i\0",icyl);

	if(cylind[icyl].color_on) {
	/* color the cylinder */
		GEOMadd_float_colors(obj[icyl], colors, num_vert,GEOM_COPY_DATA);
	}
	/* add array of normals to the cylinder */
      	GEOMadd_normals(obj[icyl], norms, num_vert, GEOM_COPY_DATA);

/* Create the GEOM Edit List for geo_out                                 */

      	if (icyl == 0)
	/* initialize the edit list */
		*geo_out = GEOMinit_edit_list(*geo_out);

/* add the current cylinder to the edit list */
      	GEOMedit_geometry(*geo_out, objname[icyl], obj[icyl]);

/* deallocate the memory associated with the current object */
      GEOMdestroy_obj(obj[icyl]); 

/* free locally allocated data */
      	FREE_LOCAL(theta);
      	FREE_LOCAL(verts);
      	FREE_LOCAL(plist);
      	FREE_LOCAL(norms);
	FREE_LOCAL(colors);
    }
}

	return(1);

} /***** End Module Compute Function *****/

 
/* ***********************************************************************/
/* Initialization for modules contained in this file.                    */
/* ***********************************************************************/
int ((*filter_mod_list[])()) = {
	cylinder_desc,
};
#define NMODS (sizeof(filter_mod_list) / sizeof(char *))

AVSinit_modules()
{
	AVSinit_from_module_list(filter_mod_list, NMODS);
	
} /***** End of Initialization *****/
 
