#include <stdio.h>

#include <avs/avs.h>
#include <avs/field.h>

#include "vinclude.h"
#include "file_formats/dlg.h"
#include "file_formats/elas.h"
#include "file_formats/big.h"
#include "file_formats/dem.h"
#include "proto.h"

#include "avskhoros.h"

char *program="remote_gis";
int ac=1;
char *av[] /* = NULL SRT is this OK? (did for IBM compilation ) */; 

/* *****************************************/
/*  Module Descriptions                    */
/* *****************************************/

int big2viff_desc ()
{
   int in_port, out_port, param;
   extern int big2viff_compute();

   AVSset_module_name("big2AVS",MODULE_DATA);

   AVScreate_output_port("Output VIFF file","field 2D");

   ADD_FILE;
   AVSadd_parameter("color","choice","no model","ntscRGB,genericRGB,cieRGB,ntscCMY,ntscYIQ,HSV,HLS,IHS,cieXYZ,cieUVW,cieucsUVW,cieucsSOW,cieucsLab,GENERIC",",");

   AVSset_compute_proc(big2viff_compute);
   return(1);
}

int dem2viff_desc ()
{
   int in_port, out_port, param;
   extern int dem2viff_compute();

   AVSset_module_name("dem2AVS",MODULE_DATA);

   AVScreate_output_port("Output VIFF file","field 2D");

   ADD_FILE;
   AVSadd_parameter("location","choice","No Location","Location,No Location",",");
   AVSadd_parameter("Subsampling","integer",1,1,25);

   AVSset_compute_proc(dem2viff_compute);
   return(1);
}

int dlg2viff_desc ()
{
   int in_port, out_port, param;
   extern int dlg2viff_compute();

   AVSset_module_name("dlg2AVS",MODULE_DATA);

   AVScreate_output_port("DLG Category #1","field 2D");
   AVScreate_output_port("DLG Category #2","field 2D");
   AVScreate_output_port("DLG Category #3","field 2D");
   AVScreate_output_port("DLG Category #4","field 2D");
   ADD_FILE;
   AVSadd_parameter("Columns","integer",1,1,512);
   AVSadd_parameter("location","choice","None","None,Sparse,Full",",");

   AVSset_compute_proc(dlg2viff_compute);
   return(1);
}

int elas2viff_desc ()
{
   int in_port, out_port, param;
   extern int elas2viff_compute();

   AVSset_module_name("elas2AVS",MODULE_DATA);

   AVScreate_output_port("Output VIFF file","field 2D");
   ADD_FILE;
   AVSadd_parameter("band number","integer",1,1,512);
   AVSadd_parameter("x","integer",1,1,512);
   AVSadd_parameter("y","integer",1,1,512);
   AVSadd_parameter("width","integer",1,1,512);
   AVSadd_parameter("height","integer",1,1,512);
   AVSadd_parameter("color","choice","no model","no model,ntscRGB,ntscCMY,ntscYIQ,HSV,HLS,IHS,cieRGB,cieXYZ,cieUVW,cieucsUVW,cieucsUVW,cieucsSOW,cieucsLab,cieucsLuv,GENERIC,genericRGB",",");

   AVSset_compute_proc(elas2viff_compute);
   return(1);
}

int elasinfo_desc ()
{
   int in_port, out_port, param;
   extern int elasinfo_compute();

   AVSset_module_name("elasinfo",MODULE_DATA);

   ADD_FILE;
   ADD_FILE;
   
   AVSset_compute_proc(elasinfo_compute);
   return(1);
}

int viff2big_desc ()
{
   int in_port, out_port, param;
   extern int viff2big_compute();

   AVSset_module_name("AVS2big",MODULE_RENDER);

   AVScreate_input_port("Input VIFF file","field 2D",REQUIRED);
   ADD_FILE;

   AVSset_compute_proc(viff2big_compute);
   return(1);
}

int vll2utm_desc ()
{
   int in_port, out_port, param;
   extern int vll2utm_compute();

   AVSset_module_name("vll2utm",MODULE_DATA);

   AVScreate_input_port("Input File ","field 2D",REQUIRED);
   AVScreate_output_port("Output File","field 2D");
   AVSadd_parameter("units","choice","degrees","degrees,seconds",",");

   AVSset_compute_proc(vll2utm_compute);
   return(1);
}

/* *****************************************/
/* Module Compute Routines                 */
/* *****************************************/

int big2viff_compute (AVSfield **o1,char *fname,char *color)
{
   int value;
   struct xvimage *ko1;
   struct big *read_big(), *big_image;

   big_image = read_big(fname);
   if(big_image == NULL)
   {
      (void) fprintf(stderr,"big2viff: Image could not be read\n"); 
      return(0);
   }

   value=(int)lbig2viff(big_image, &ko1,AVSchoice_number("color"));
 
   if (((*o1)=(AVSfield *)viff_to_field(ko1))==NULL) return (0);
 
   return (value);
}
 
int dem2viff_compute (AVSfield **o1,char *fname,char *loc,int sam)
{
   int value;
   struct xvimage *ko1;
   struct demimage *dem_img, *readdem();

   dem_img = readdem(fname);
   if (dem_img == NULL) 
   {
      fprintf (stderr,"error in dem2viff: could not read DEM file: %s\n",fname);
      return (0);
   }

   value=(int)ldem2viff(&ko1, dem_img,AVSchoice_number("location"),sam);
 
   if (((*o1)=(AVSfield *)viff_to_field(ko1))==NULL) return (0);
 
   return (value);
}
 
int dlg2viff_compute (AVSfield **o1,AVSfield **o2,AVSfield **o3,AVSfield **o4,char *fname,int col,char *loc)
{
   int value;
   int rows;
   double delta_x, delta_y;

   struct xvimage *ko1;
   struct xvimage *ko2;
   struct xvimage *ko3;
   struct xvimage *ko4;
   struct dlgimage *dlg_img, *readdlg();

   /*
    * Read the input dlg file.
    */
   dlg_img = readdlg (fname);
   if (dlg_img == NULL) 
   {
      fprintf (stderr, "error in dlg2viff: ");
      fprintf (stderr, "could not read dlg file: %s\n", fname);
      return (0);
   }

   /*
    * Make sure that the number of columns selected for the output image is
    * a multiple of 8.  This is necessary because the output is a bit image
    * and a row must have a whole number of bytes.
    */
   if (col%8 != 0) 
   {
      fprintf (stderr, "error in dlg2viff: ");
      fprintf (stderr, "number of columns must be a multiple of 8\n");
      return (0);
   }

   /*
    * Set up a ratio between the length and the area covered by the dlg
    * data to calculate the number of rows that the output image should
    * have.  This is done so the output dlg map will be correctly 
    * proportioned.
    */

   delta_x = dlg_img->corners[SW][INTERNAL_X]-dlg_img->corners[SE][INTERNAL_X];
   delta_x = fabs(delta_x);
   delta_y = dlg_img->corners[SW][INTERNAL_Y]-dlg_img->corners[NW][INTERNAL_Y];
   delta_y = fabs(delta_y);
   rows = (int) ((delta_y/delta_x) * col);
    /*
     * Set all the image pointers to NULL.  If they return from the library
     * with a NULL value it means there was not a category in the dlg file
     * to assign to them.
     */
   ko1=ko2=ko3=ko4=NULL;

   value=(int)ldlg2viff(dlg_img,rows,col,AVSchoice_number("location"),&ko1,&ko2,&ko3,&ko4);
 
   *o1=(AVSfield *)viff_to_field(ko1);
   *o2=(AVSfield *)viff_to_field(ko2);
   *o3=(AVSfield *)viff_to_field(ko3);
   *o4=(AVSfield *)viff_to_field(ko4);
 
   return (value);
}
 
int elas2viff_compute (AVSfield **o1,char *fname,int band,int x,int y,int width,int height,char *color)
{
   int value;
   struct xvimage *ko1;
   ELAS *elas_image, *open_elas();
   int elas_fid, subimg_flag;

   subimg_flag = FALSE;    /* initialize optional subimage flag to false */


  /*--------------------------------------------------------
   * First Open ELAS file to get a file descriptor, 
   *  then read ELAS header info
   *------------------------------------------------------ */

  elas_image = open_elas(fname, &elas_fid);
  if ( elas_image == NULL)
  {
     (void) fprintf(stderr,"%s: unable to read the ELAS image '%s'\n","elas2viff", fname);
     return(0);
  }

  /*-------------------------------------------------------------------
   * ERROR checking:
   *  If subimage coordinates are specified, ensure that all four 
   *  subimage parameters are specified, then check that the 
   *  requested location coordinates are within ELAS image bounds.
   *-------------------------------------------------------------------*/

   if (x || y || width || height)
   {
      if ( !(x && y && width && height) )
      {
         (void) fprintf(stderr,"%s: Must select all four subimage parameters.\n", "elas2viff");
         close(elas_fid);
         return(0);
      }

      subimg_flag = TRUE;     /* set optional subimage flag to true */

      if (x > (elas_image->last_elem_scan_line - elas_image->init_elem_scan_line))
      {
         (void) fprintf(stderr,"%s: input x-coordinate out of bounds.\n", "elas2viff");
         (void) fprintf(stderr,"ELAS image width = %d bytes.\n",elas_image->last_elem_scan_line);
         close(elas_fid);
         return(0);
      }
      if (y > (elas_image->last_line - elas_image->init_line))
      {
         (void) fprintf(stderr,"%s: input y-coordinate out of bounds.\n", "elas2viff");
         (void) fprintf(stderr,"ELAS image height = %d bytes.\n",elas_image->last_line);
         close(elas_fid);
         return(0);
      }
      if ((x + width) > (elas_image->last_elem_scan_line - elas_image->init_elem_scan_line))
      {
         (void) fprintf(stderr,"%s: requested width exceeds ELAS image bounds.\n", "elas2viff");
         (void) fprintf(stderr,"Use smaller x-coordinate and/or width values.\n");
         close(elas_fid);
         return(0);
      }
      if ((y + height) > (elas_image->last_line - elas_image->init_line))
      {
         (void) fprintf(stderr,"%s: requested height exceeds ELAS image bounds.\n", "elas2viff");
         (void) fprintf(stderr,"Use smaller y-coordinate and/or height values.\n");
         close(elas_fid);
         return(0);
      }
      if (band > elas_image->num_channels)
      {
         (void) fprintf(stderr,"%s: requested band number exceeds\n", "elas2viff");
         (void) fprintf(stderr,"the number of channels in the ELAS image.\n");
         (void) fprintf(stderr,"Number of channels in ELAS image = %d\n", elas_image->num_channels);
         close(elas_fid);
         return(0);
      }
   }

   if (band > elas_image->num_channels)
   {
      (void) fprintf(stderr,"%s: requested band number exceeds\n", "elas2viff");
      (void) fprintf(stderr,"the number of channels in the ELAS image.\n");
      (void) fprintf(stderr,"Number of channels in ELAS image = %d\n", elas_image->num_channels);
      close(elas_fid);
      return(0);
   }

   value=lelas2viff(elas_image, elas_fid, &ko1, band, x, y, width, height, AVSchoice_number("color"),subimg_flag);
 
   if (((*o1)=(AVSfield *)viff_to_field(ko1))==NULL) return (0);
 
   return (value);
}
 
int elasinfo_compute (char *in,char *out)
{
   int value;
   ELAS *elas_image, *open_elas();
   FILE *printdev;
   int elas_fid;

   /*--------------------------------------------------------
    * First Open ELAS file to get a file descriptor, 
    *  then read ELAS header info
    *------------------------------------------------------ */
   elas_image = open_elas(in, &elas_fid);
   if ( elas_image == NULL)
   {
      (void) fprintf(stderr,"%s: unable to read the ELAS image '%s'\n",
                   "elasinfo", in);
      return(0);
   }

   
   if (out)
   {
      if (! vwrite_ascii(out, &printdev))
      {
         (void)fprintf(stderr,"elasinfo: Can't open ascii outpur file.\n");
         close(elas_fid);
         return(0);
      }
   }
   else
      printdev = stdout;

   if (! lelasinfo(elas_image, elas_fid, in, printdev))
   {
      (void) fprintf (stderr, "lelasinfo Failed\n");
      close(elas_fid);
      return(0);
   }

   return (1);
}
 
int viff2big_compute (AVSfield *i1,char *fname)
{
   int value;
   struct xvimage *ki1;
   struct big *big_image;
 
   if (!AVSparameter_changed("filename")) return(0);

   if (i1)
      if ((ki1=(struct xvimage *)field_to_viff(i1))==NULL) return(0);

   value=(int)lviff2big(ki1,&big_image);
 
   if (!write_big (fname, big_image)) 
   {
       (void) fprintf (stderr, "write_big Failed\n");
       return(0);
   }

   return (value);
}
 
int vll2utm_compute (AVSfield *i1,AVSfield **o1,int param1)
{
   int value;
   struct xvimage *ki1;
   struct xvimage *ko1;

   if (i1)
      if ((ki1=(struct xvimage *)field_to_viff(i1))==NULL) return(0);

   value=(int)lvll2utm(ki1,AVSchoice_number("units"));

   ko1=ki1;
 
   if (((*o1)=(AVSfield *)viff_to_field(ko1))==NULL) return (0);
 
   return (value);
}
 
/* ***********************************************************************/
/* Initialization for modules contained in this file.                    */
/* ***********************************************************************/
int ((*mod_list[])()) = {
   big2viff_desc,
   dem2viff_desc,
   dlg2viff_desc,
   elas2viff_desc,
   elasinfo_desc,
   viff2big_desc,
   vll2utm_desc,
};

#define NMODS (sizeof(mod_list) / sizeof(char *))

AVSinit_modules()
{
        AVSinit_from_module_list(mod_list, NMODS);
}
