/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
#include <stdio.h>
#include <avs/avs.h>
#include <avs/port.h>
#include <avs/field.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/* Since all scratch files used by this module may be written into a common
area, all of them have the mode 0666 so that other users can overwrite them.
*/

#define HIDE "show"			/* "hide" or "show" */ 

#define IOERROR -1

#define MEAN_GRAPH 1
#define DISTRIBUTION_GRAPH 2
#define MIN_MED_MAX_GRAPH 3
#define THRESHOLD_GRAPH 4

#define NT input_field->dimensions[3]

#define MEANCOLOR0  350. 
#define MEANCOLOR1  176. 
#define MEANCOLOR2  1. 
#define THRESHCOLOR 125.
#define MAXCOLORRANGE 350.
#define MINCOLORRANGE 1.
#define REPLACE_PLOT 0
#define ADD_TO_PLOT 1
#define LINE 0
#define SCATTER 1
#define AREA 2
#define BAR 3
#define ASCII_DATA 1
#define SOLID 1
#define DASH 2
#define DOT 3

char *output_buffer;
char *error_buffer;
static char str[1024];

float bigxmax;
float bigymax;
float bigthresh;

int slice_plane;
char slice_axis[5];

static int graph_no = -1;

/* *****************************************/
/*  Module Description                     */
/* *****************************************/
int GRAPH2D_desc()
{

	int in_port, out_port, param;
	extern int GRAPH2D_compute();

	AVSset_module_name("GRAPH2D", MODULE_MAPPER);

	/* Input Port Specifications               */

	in_port = AVScreate_input_port("input_field", 
		"field 4D uniform 3-coordinate 1-vector float", REQUIRED);

	in_port = AVScreate_input_port("input_label", "string", REQUIRED);

	in_port = AVScreate_input_port("input_time", "string", REQUIRED);

	in_port = AVScreate_input_port("input_slice_axis", "string", REQUIRED);

	in_port = AVScreate_input_port("input_crop_vars", "string", REQUIRED);

	/* Parameter Specifications                */

	param = AVSadd_parameter("graph type", "choice", "mean graph", 
	"mean graph:distribution graph:min/med/max graph:threshold graph", ":");
	AVSconnect_widget(param, "radio_buttons");
	AVSadd_parameter_prop(param, "width", "integer", 4);

	/* Window number, x location, y location, width, height */

	param = AVSadd_parameter("window number", "integer", 1, 1, 4);
	AVSconnect_widget(param, "typein_integer");

	param = AVSadd_parameter("X window location", "integer", 300, 0, 1000);
	AVSconnect_widget(param, "typein_integer");

	param = AVSadd_parameter("Y window location", "integer", 50, 0, 1000);
	AVSconnect_widget(param, "typein_integer");

	param = AVSadd_parameter("window width", "integer", 450, 0, 1000);
	AVSconnect_widget(param, "typein_integer");

	param = AVSadd_parameter("window height", "integer", 450, 0, 1000);
	AVSconnect_widget(param, "typein_integer");

	/* X AXIS */

	param = AVSadd_float_parameter("X axis min", 0.0, FLOAT_UNBOUND, FLOAT_UNBOUND);
	AVSconnect_widget(param, "typein_real");

	param = AVSadd_float_parameter("X axis max", 10.0000, FLOAT_UNBOUND, FLOAT_UNBOUND);
	AVSconnect_widget(param, "typein_real");

	param = AVSadd_parameter("X axis tic placement",
		 "choice", "X axis tics outside", 
"X axis tics nowhere:X axis tics inside:X axis tics outside:X axis tics inside & outside",
		 ":");
	AVSconnect_widget(param, "radio_buttons");
	AVSadd_parameter_prop(param, "width", "integer", 4);

	param = AVSadd_parameter("X axis number of tics", "integer", 2, 0, 10);
	AVSconnect_widget(param, "typein_integer");

	param = AVSadd_parameter("X axis tic label precision", "integer", 0, 
		0, 6);
	AVSconnect_widget(param, "typein_integer");
	
	/* Y AXIS */

	param = AVSadd_float_parameter("Y axis min", 0.0, FLOAT_UNBOUND,
		 FLOAT_UNBOUND);
	AVSconnect_widget(param, "typein_real");

	param = AVSadd_float_parameter("Y axis max", 0.14000, FLOAT_UNBOUND, FLOAT_UNBOUND);
	AVSconnect_widget(param, "typein_real");

	param = AVSadd_parameter("Y axis tic placement",
		 "choice", "Y axis tics outside", 
"Y axis tics nowhere:Y axis tics inside:Y axis tics outside:Y axis tics inside & outside",
		 ":");
	AVSconnect_widget(param, "radio_buttons");
	AVSadd_parameter_prop(param, "width", "integer", 4);

	param = AVSadd_parameter("Y axis number of tics", "integer", 2, 0, 10);
	AVSconnect_widget(param, "typein_integer");

	param = AVSadd_parameter("Y axis tic label precision", "integer", 2, 
		0, 6);
	AVSconnect_widget(param, "typein_integer");
	
	param = AVSadd_float_parameter("threshold value", 0.120000, FLOAT_UNBOUND, 
		FLOAT_UNBOUND);
	AVSconnect_widget(param, "typein_real");
	param = AVSadd_parameter("percentile increment", "choice",
 "5 percentile", "1 percentile:5 percentile:10 percentile:20 percentile:25 percentile:50 percentile", ":");
	AVSconnect_widget(param, "radio_buttons");
	AVSadd_parameter_prop(param, "width", "integer", 2);
	AVSadd_parameter_prop(param, "columns", "integer", 2);

	param = AVSadd_parameter("output dir", "string", NULL, NULL, NULL);
	AVSconnect_widget(param, "string");
	AVSadd_parameter_prop(param, "width", "integer", 4);

	param = AVSadd_parameter("PostScript mode", "choice", 
		"PostScript COLOR mode",
		"PostScript COLOR mode:PostScript GREYSCALE mode",
		":");
	AVSconnect_widget(param, "radio_buttons");
	AVSadd_parameter_prop(param, "width", "integer", 4);

	param = AVSadd_parameter(
		"PostScript output file #", "integer", 1, 1, 99);
	AVSconnect_widget(param, "typein_integer");

	param = AVSadd_parameter("write PostScript", "oneshot", 0, 0, 1);
	AVSconnect_widget(param, "oneshot");

	param = AVSadd_parameter("autoscale", "boolean", 0, 0, 1);
	AVSconnect_widget(param, "toggle");

	param = AVSadd_parameter("legend position",
		"choice", "upper left legend", 
"lower left legend:lower right legend:upper right legend:upper left legend",
		":");
	AVSconnect_widget(param, "radio_buttons");
	AVSadd_parameter_prop(param, "width", "integer", 4);

	param = AVSadd_parameter("delete", "oneshot", 0, 0, 1);
	AVSconnect_widget(param, "oneshot");

	param = AVSadd_parameter("go", "oneshot", 0, 0, 1);
	AVSconnect_widget(param, "oneshot");

	param = AVSadd_parameter("autogo", "boolean", 0, 0, 1);
	AVSconnect_widget(param, "toggle");

        param = AVSadd_parameter("output table", "string",
                 " ", NULL, NULL);
        AVSconnect_widget(param, "text_browser");
        AVSadd_parameter_prop(param, "width", "integer", 4);
        AVSadd_parameter_prop(param, "height", "integer", 8);
 
        param = AVSadd_parameter("table file name", "string", NULL, NULL, NULL);
        AVSconnect_widget(param, "typein");
        AVSadd_parameter_prop(param, "width", "integer", 4);

	AVSset_compute_proc(GRAPH2D_compute);
	return(1);
}
 
/* *****************************************/
/* Module Compute Routine                  */
/* *****************************************/
int GRAPH2D_compute(input_field, input_label, input_time, 
input_slice_axis, input_crop_vars,	
graph_type, 
window_no, 
x_window_location, y_window_location, window_width, window_height,
xaxis_min, xaxis_max, xaxis_tic_placement, 
xaxis_number_tics, xaxis_tic_label_precision,
yaxis_min, yaxis_max, yaxis_tic_placement, 
yaxis_number_tics, yaxis_tic_label_precision,
threshold_value, percentile_increment, output_dir,
ps_mode, ps_fileno, ps, autoscale, legend_position, 
delete, go, autogo, output_table, table_file_name)
	AVSfield *input_field;
	char *input_label;
	char *input_time;
	char *input_slice_axis;
	char *input_crop_vars;
	char *graph_type;
	int window_no;
	int x_window_location;
	int y_window_location;
	int window_width;
	int window_height;
	float *xaxis_min;
	float *xaxis_max;
	char *xaxis_tic_placement;
	int xaxis_number_tics;
	int xaxis_tic_label_precision;
	float *yaxis_min;
	float *yaxis_max;
	char *yaxis_tic_placement;
	int yaxis_number_tics;
	int yaxis_tic_label_precision;
	float *threshold_value;
	char *percentile_increment;
	char *output_dir;
	char *ps_mode;
	int ps_fileno;
	int ps;
	int autoscale;
	char *legend_position;
	int delete;
	int go;
	int autogo;
	char *output_table;
	char *table_file_name;
{

#define MOST_TOP .9
#define MOST_BOTTOM .1
#define MINMEDMAX_TOP .825
#define MINMEDMAX_BOTTOM .275 

static int plottype[] = {SCATTER, SCATTER, LINE, AREA};
static int used[] = {0, 0, 0, 0};
static float legendx[] = {.15, .55, .55, .15};
static float legendy[] = {.275, .275, .825, .825};

/* RGB version of colors for mean plots with symbols controlled, leaving 
   the GRAPH VIEWER colors read from column 3 of the data file ignored

   Trying to match BLUE, GREEN, RED colors from the GRAPH VIEWER
   color spectrum selections of 350, 176, 1 for MEAN 1, MEAN 2, MEAN 3.

   This "control" symbol function was removed but might be needed for
   PostScript output. (?)
*/

static float red[] = {0.0, 0.0, 1.0};
static float green[] = {0.0, 1.0, 0.0};
static float blue[] = {1.0, 0.0, 0.0};

static int okay_to_check_table = 0;

int legend_no;
FILE *netfile;
struct stat statbuf;
int istat;
int fd;
int dist;

float gxmin, gxmax;
int gxtic, gxprec, gxpos;
float gymin, gymax;
int gytic, gyprec, gypos;

int cropx0, cropx1, cropy0, cropy1;
int cropz0, cropz1, cropt0, cropt1;

char *answer;

register int i, j;
register int win;

#define NETCMD fprintf(netfile, "%s\n", str);

#define CMD AVScommand("kernel", str, &output_buffer, &error_buffer);  

win = window_no - 1;

if (!strcmp(graph_type, "threshold graph"))
	AVSparameter_visible("threshold value", 1);
else
	AVSparameter_visible("threshold value", 0);

if (!strcmp(graph_type, "distribution graph"))
	AVSparameter_visible("percentile increment", 1);
else
	AVSparameter_visible("percentile increment", 0);

if (autoscale)
	{
	AVSparameter_visible("X axis min", 0);
	AVSparameter_visible("X axis max", 0);
	AVSparameter_visible("X axis tic placement", 0);
	AVSparameter_visible("X axis number of tics", 0);
	AVSparameter_visible("X axis tic label precision", 0);
	AVSparameter_visible("Y axis min", 0);
	AVSparameter_visible("Y axis max", 0);
	AVSparameter_visible("Y axis tic placement", 0);
	AVSparameter_visible("Y axis number of tics", 0);
	AVSparameter_visible("Y axis tic label precision", 0);
	}
else
	{
	AVSparameter_visible("X axis min", 1);
	AVSparameter_visible("X axis max", 1);
	AVSparameter_visible("X axis tic placement", 1);
	AVSparameter_visible("X axis number of tics", 1);
	AVSparameter_visible("X axis tic label precision", 1);
	AVSparameter_visible("Y axis min", 1);
	AVSparameter_visible("Y axis max", 1);
	AVSparameter_visible("Y axis tic placement", 1);
	AVSparameter_visible("Y axis number of tics", 1);
	AVSparameter_visible("Y axis tic label precision", 1);
	}

if ((ps) && (used[win]))
	{
	sprintf(str, "%s/pscript%d",  output_dir, win);
	if ((netfile = fopen(str, "w")) == NULL)
		{
		AVSerror("Cannot create xyplot postscript file %s", str);
		return(0);
		}
	if ((chmod(str, 0666)) == IOERROR)
		{
		fclose(netfile);
		AVSerror("Cannot set mode on xyplot postscript file %s", str);
		return(0);
		}
	if (!strcmp(ps_mode, "PostScript COLOR mode"))
		sprintf(str, "parm_set ITP%d:mode color",  win);
	else if (!strcmp(ps_mode, "PostScript GREYSCALE mode"))
		sprintf(str, "parm_set ITP%d:mode greyscale", win);
	NETCMD;

	sprintf(str, "parm_set ITP%d:filename %s/f%03d.ps", 
		win, output_dir, ps_fileno); NETCMD;

	fprintf(stderr, "creating PostScript file %s/f%03d.ps\n",
		output_dir, ps_fileno);

	sprintf(str, "graph_output_image"); NETCMD;

	sprintf(str, "script -close"); NETCMD;
	fclose(netfile);

	answer = AVSmessage("Version 1", AVS_Warning, NULL,
		"GRAPH2D", "Cancel!Continue",
	"Ready to generate output file %s/f%03d.ps in %s.\n\nContinue?",
		 output_dir, ps_fileno, ps_mode);
	 if (!strcmp(answer, "Continue"))
		 {
		sprintf(str, "script -play %s/pscript%d -echo no", 
			output_dir, win); CMD;

		AVSmodify_parameter("PostScript output file #",
			AVS_VALUE, ps_fileno+1, NULL, NULL);
		}
	return(1);
	}

if ((delete) && (used[win]))
	{
        sprintf(str, "net_clear -tag NET%d", win); CMD;
	used[win] = 0;
	return(0);
	}
	
if (	(!strcmp(input_slice_axis, "X")) ||
	(!strcmp(input_slice_axis, "Y")) ||
	(!strcmp(input_slice_axis, "Z")))
	sprintf(slice_axis, "%s", input_slice_axis);
else
	{
	AVSerror("Graph2D input must be sliced on X or Y or Z.");
	return(0);
	}

if ((go) || (autogo))
	{

	sscanf(input_crop_vars, "%d%d%d%d%d%d%d%d",
		&cropx0, &cropx1, &cropy0, &cropy1,
		&cropz0, &cropz1, &cropt0, &cropt1);

	if (!strcmp(input_slice_axis, "X"))
		{
		if (cropx0 == cropx1)
			slice_plane = cropx0;
		else
			{
			AVSerror(
			"Slice X incompatible with (%d,%d) crop variables",
			 cropx0, cropx1);
			return(0);
			}
		}
	if (!strcmp(input_slice_axis, "Y"))
		{
		if (cropy0 == cropy1)
			slice_plane = cropy0;
		else
			{
			AVSerror(
			"Slice Y incompatible with (%d,%d) crop variables",
			 cropy0, cropy1);
			return(0);
			}
		}
	if (!strcmp(input_slice_axis, "Z"))
		{
		if (cropz0 == cropz1)
			slice_plane = cropz0;
		else
			{
			AVSerror(
			"Slice Z incompatible with (%d,%d) crop variables",
			 cropz0, cropz1);
			return(0);
			}
		}

	/* Make sure output directory exists and is a valid UNIX directory */

	if ((output_dir) == NULL)
		{
		AVSerror("$NULL is not a valid output directory name.");
		return(0);
		}
	if ((fd = open(output_dir, O_RDONLY)) != IOERROR)
		{
		if ((istat = stat(output_dir, &statbuf)) != IOERROR)
                	{
                	if (!(statbuf.st_mode & S_IFDIR))
				{
                        	AVSerror("%s is not a directory.", output_dir);
				close(fd);
				return(0);
				}
			}
		else
			{
			AVSerror("Cannot get status of %s", output_dir);
			close(fd);
			return(0);
			}
		close(fd);
                }
	else
		{

                 answer = AVSmessage("Version 1", AVS_Warning, NULL,
                 "GRAPH2D", "Cancel!Create",
 		"%s output directory does not exist; create it?", output_dir);
                 if (!strcmp(answer, "Cancel"))
                         {
                         return(0);
                         }

		sprintf(str, "mkdir -p %s\n", output_dir);
		if (system(str) == -1)
			{
			AVSerror("%s cannot be created.", output_dir);
			return(0);
			}
		}
		
	graph_no = 0;
	if (graph_no > 2)
		{
		AVSerror("Maximum number of graphs on one plot = 3.");
		return(0);
		}
	legend_no = AVSchoice_number("legend position", legend_position) - 1;

	/***************************/
	/* Instead of following the old method of destroying the network
	   and making a new one each time a graph viewer window is needed,
	   now the network will only be recreated if the window has not
	   been previously set up or if it has been deleted.  Also, the
	   window will not be hidden as previously, and the user will see
	   the graphics changes as they are made.
	*/

	if (!used[win])
		{
	sprintf(str, "%s/net%d", output_dir, win);
	if ((netfile = fopen(str, "w")) == NULL)
		{
		AVSerror("Cannot open xyplot network file %s\n", str);
		return(0);
		}
	if ((chmod(str, 0666)) == IOERROR)
		{
		fclose(netfile);
		AVSerror("Cannot set mode on xyplot network file %s\n", str);
		return(0);
		}
        fprintf(netfile, "%s\n", "#!/usr/bin/avs -network");
        fprintf(netfile, "%s\n", "version 4.0 (25.137 DEC)");
 
        sprintf(str, "module \"graph viewer\" -alias GRAPH%d", win); NETCMD;

	sprintf(str, "module \"image to postscript\" -alias ITP%d", win);
	NETCMD;

	sprintf(str, "port_connect GRAPH%d:1 ITP%d:0\n", win, win); NETCMD;
 
        sprintf(str,
        "panel \"GRAPH%d!graph\" -w container -p ui -xy %d,%d -wh %d,%d -%s",
                win, x_window_location, y_window_location,
                window_width, window_height, HIDE); NETCMD;

	sprintf(str, "parm_set ITP%d:filename %s/f%03d.ps", 
		win, output_dir, ps_fileno); NETCMD;

	if (!strcmp(ps_mode, "PostScript COLOR mode"))
		sprintf(str, "parm_set ITP%d:mode color",  win);
	else if (!strcmp(ps_mode, "PostScript GREYSCALE mode"))
		sprintf(str, "parm_set ITP%d:mode greyscale", win);
	NETCMD;

        fclose(netfile);
 
        sprintf(str, "net_read %s/net%d -tag NET%d", output_dir, win, win); CMD;
		}
	/***************************/

	sprintf(str, "%s/script%d",  output_dir, win);
	if ((netfile = fopen(str, "w")) == NULL)
		{
		AVSerror("Cannot create xyplot script file %s.", str);
		return(0);
		}
	if ((chmod(str, 0666)) == IOERROR)
		{
		fclose(netfile);
		AVSerror("Cannot set mode on xyplot script file %s.", str);
		return(0);
		}

	sscanf(percentile_increment, "%d", &dist);
	i = AVSchoice_number("graph type", graph_type);
	if (!calculate_graph(input_field, i, win,
		 dist, *threshold_value, output_dir))
		{
		fclose(netfile);
		return(0);
		}
	if (autoscale) 
		{
		setxaxis(i, &gxmin, &gxmax, &gxpos, &gxtic, &gxprec);
		setyaxis(i, &gymin, &gymax, &gypos, &gytic, &gyprec);
		}
	else
		{
		gxmin = *xaxis_min;
		gxmax = *xaxis_max;
		gxpos = AVSchoice_number ("X axis tic placement",
			 xaxis_tic_placement) - 1;
		gxtic = xaxis_number_tics; 
		gxprec = xaxis_tic_label_precision;

		gymin = *yaxis_min;
		gymax = *yaxis_max;
		gypos = AVSchoice_number ("Y axis tic placement",
			 yaxis_tic_placement) - 1;
		gytic = yaxis_number_tics; 
		gyprec = yaxis_tic_label_precision;
		}

	sprintf(str, "graph_set_plot_size %d %d", window_width,
			window_height); NETCMD;

	sprintf(str, "graph_set_plot_mode %d", REPLACE_PLOT); NETCMD;
	sprintf(str, "graph_set_plot_style %d", plottype[i-1]); NETCMD;

	sprintf(str, "graph_set_data_format %d", ASCII_DATA); NETCMD;

	sprintf(str, "graph_set_two_column 1 2 3"); NETCMD;

	if (i == MIN_MED_MAX_GRAPH)
		{
		legendy[0] = legendy[1] = MINMEDMAX_BOTTOM;
		legendy[2] = legendy[3] = MINMEDMAX_TOP;
		}
	else
		{
		legendy[0] = legendy[1] = MOST_BOTTOM;
		legendy[2] = legendy[3] = MOST_TOP;
		}

	if (i == MEAN_GRAPH)
		{
		sprintf(str, "graph_set_scatter_info -graph %d 1 .18",
			graph_no); NETCMD;
		}
	else if (i == DISTRIBUTION_GRAPH)
		{
		sprintf(str, "graph_set_scatter_info 1 .18"); NETCMD;
		}

	sprintf(str, "graph_read_ascii_data %s/f1.w%d",
		 output_dir, win); NETCMD;

	if (i == MEAN_GRAPH)
		{
		sprintf(str, "graph_set_scatter_symbol -graph %d \"%c\"",
			graph_no, '*'); NETCMD;
		}
	else if (i == DISTRIBUTION_GRAPH)
		{
		if (!strcmp(ps_mode, "PostScript GREYSCALE mode"))
			{
			sprintf(str, "graph_set_scatter_symbol \"*\""); NETCMD;
			}
		}
	else if (i == MIN_MED_MAX_GRAPH)
		{
		sprintf(str, "graph_set_line_style -graph 0 %d", SOLID);
		NETCMD;
		}

/* Color the symbol if outputting in PostScript color mode; if greyscale,
   make the symbol white */

	if (i == MEAN_GRAPH)
		{
		if (!strcmp(ps_mode, "PostScript COLOR mode"))
			sprintf(str, "graph_set_plot_color -graph %d %g %g %g",
			graph_no,
			red[graph_no], green[graph_no], blue[graph_no]);
		else if (!strcmp(ps_mode, "PostScript GREYSCALE mode"))
			sprintf(str, "graph_set_plot_color -graph %d %g %g %g",
			graph_no, 1.0, 1.0, 1.0);
		NETCMD;
		}
	else if (i == MIN_MED_MAX_GRAPH)
		{
		if (!strcmp(ps_mode, "PostScript COLOR mode"))
			sprintf(str,
			"graph_set_plot_color -graph %d %g %g %g",
				0, red[0], green[0], blue[0]);
		else if (!strcmp(ps_mode, "PostScript GREYSCALE mode"))
			sprintf(str,
			"graph_set_plot_color -graph %d %g %g %g",
				0, 1.0, 1.0, 1.0);
		NETCMD;
		}
	else
		{
		if (!strcmp(ps_mode, "PostScript GREYSCALE mode"))
			sprintf(str, "graph_set_plot_color %g %g %g",
			1.0, 1.0, 1.0); NETCMD;
		}

	
	sprintf(str, "graph_set_xaxis 0 %.6f %.6f %d %d %d",
		gxmin, gxmax, gxpos, gxtic, gxprec); NETCMD;

	sprintf(str, "graph_set_yaxis 0 %.6f %.6f %d %d %d",
		gymin, gymax, gypos, gytic, gyprec); NETCMD;

	sprintf(str, "graph_set_plot_xlabel \"Hour\""); NETCMD;
	sprintf(str, "graph_set_plot_ylabel \"%s\"", " "); NETCMD;
	sprintf(str, "graph_set_plot_title \"%s\"", input_time); NETCMD;
	sprintf(str, "graph_set_legend_label -graph 0 \"  \""); NETCMD;
	sprintf(str, "graph_set_legend_label -graph 1 \"  \""); NETCMD;
	sprintf(str, "graph_set_legend_label -graph 2 \"  \""); NETCMD;

	switch(AVSchoice_number("graph type", graph_type))
		{
		case MEAN_GRAPH:
			{
			sprintf(str,
			"graph_set_legend_label -graph %d \"MEAN   %s\"",
		 	graph_no, input_label); NETCMD;
			break;
			}
		case DISTRIBUTION_GRAPH:
			{
			sprintf(str,
			"graph_set_legend_label \"DISTRIBUTION   %s\"",
			 	input_label); NETCMD;
			break;
			}
		case MIN_MED_MAX_GRAPH:
			{
			sprintf(str,
			"graph_set_legend_label -graph 0 \"MIN   %s\"",
			 	input_label); NETCMD;
			break;
			}
		case THRESHOLD_GRAPH:
			{
			sprintf(str,
		"graph_set_legend_label \"# CELLS  [%s]  > THRESHOLD %g\"",
			 	input_label, *threshold_value);
			NETCMD;
			break;
			}
		}

/* Font information is hard coded here for titles and labels:

parameter #	variable	range		meaning
---------------------------------------------------------------------------
1		font		0|1|2|3|4|5	Courier/Helvetica
						New-Century/Times
						Charter/Symbol
2		bold		0|1		false/true
3		italic		0|1		false/true
4		drop shadow	0|1		false/true
5		height		0.0 - 1.0
6		position 	0|1|2		center/left/right
7		red		0.0 - 1.0
8		green		0.0 - 1.0
9 		blue		0.0 - 1.0
---------------------------------------------------------------------------
Default selections made are:
	Helvetica, bold false, italic false, drop shadow false,
	height .11 or .12, position left justified, 
	red 1.0, green 1.0, blue 1.0 (for color = white)
*/
	sprintf(str,
		"graph_set_title_info 1 0 0 0 .12 0 1.0 1.0 1.0"); NETCMD;

	sprintf(str,
		"graph_set_xlabel_info 1 0 0 0 .12 0 1.0 1.0 1.0"); NETCMD;

	sprintf(str,
		"graph_set_ylabel_info 1 0 0 0 .12 0 1.0 1.0 1.0"); NETCMD;

	sprintf(str,
		"graph_set_ticlabel_info 1 0 0 0 .11 1 1.0 1.0 1.0"); NETCMD;

	sprintf(str, "graph_set_legend_pos %g %g",
			legendx[legend_no], legendy[legend_no]); NETCMD;

	if (i == MEAN_GRAPH)
		{
		sprintf(str,
	"graph_set_legend_label_info -graph %d 1 0 0 0 .11 1 1.0 1.0 1.0",
			graph_no); NETCMD;
		}
	else if (i == MIN_MED_MAX_GRAPH)
		{
		sprintf(str,
	"graph_set_legend_label_info -graph %d 1 0 0 0 .11 1 1.0 1.0 1.0", 0);
		NETCMD;
		}
	else
		{
		sprintf(str,
		"graph_set_legend_label_info 1 0 0 0 .11 1 1.0 1.0 1.0");
		NETCMD;
		}


	if (i == MIN_MED_MAX_GRAPH)
		{
		sprintf(str, "graph_set_plot_mode %d", ADD_TO_PLOT); NETCMD;	

		sprintf(str, "graph_set_data_format %d", ASCII_DATA); NETCMD;
		sprintf(str, "graph_set_two_column 1 2 3"); NETCMD;

		if (!strcmp(ps_mode, "PostScript COLOR mode"))
			{
			sprintf(str, "graph_set_plot_color -graph 1 %g %g %g",
			red[1], green[1], blue[1]);
			}
		if (!strcmp(ps_mode, "PostScript GREYSCALE mode"))
			{
			sprintf(str, "graph_set_plot_color -graph 1 %g %g %g",
			1.0, 1.0, 1.0);
			}
		NETCMD;

		sprintf(str, "graph_read_ascii_data %s/f2.w%d",
			output_dir, win); NETCMD;

		sprintf(str, "graph_set_line_style -graph 1 %d", DASH);
		NETCMD;

		sprintf(str, "graph_set_legend_label -graph 1 \"MED\""); NETCMD;

		sprintf(str,
	"graph_set_legend_label_info -graph %d 1 0 0 0 .11 1 1.0 1.0 1.0", 1);
		NETCMD;

		if (!strcmp(ps_mode, "PostScript COLOR mode"))
			{
			sprintf(str, "graph_set_plot_color -graph 1 %g %g %g",
			red[1], green[1], blue[1]);
			}
		if (!strcmp(ps_mode, "PostScript GREYSCALE mode"))
			{
			sprintf(str, "graph_set_plot_color -graph 1 %g %g %g",
			1.0, 1.0, 1.0);
			}
		NETCMD;

		sprintf(str, "graph_set_xaxis 0 %.6f %.6f %d %d %d",
			gxmin, gxmax, gxpos, gxtic, gxprec); NETCMD;

		sprintf(str, "graph_set_yaxis 0 %.6f %.6f %d %d %d",
			gymin, gymax, gypos, gytic, gyprec); NETCMD;
			
		sprintf(str, "graph_set_data_format %d", ASCII_DATA); NETCMD;
		sprintf(str, "graph_set_two_column 1 2 3"); NETCMD;

		if (!strcmp(ps_mode, "PostScript COLOR mode"))
			{
			sprintf(str, "graph_set_plot_color -graph 2 %g %g %g",
			red[2], green[2], blue[2]);
			}
		if (!strcmp(ps_mode, "PostScript GREYSCALE mode"))
			{
			sprintf(str, "graph_set_plot_color -graph 2 %g %g %g",
			1.0, 1.0, 1.0);
			}
		NETCMD;

		sprintf(str, "graph_read_ascii_data %s/f3.w%d",
			output_dir, win); NETCMD;

		sprintf(str, "graph_set_line_style -graph 2 %d", DOT);
		NETCMD;

		sprintf(str, "graph_set_legend_label -graph 2 \"MAX\""); NETCMD;

		sprintf(str,
	"graph_set_legend_label_info -graph %d 1 0 0 0 .11 1 1.0 1.0 1.0", 2);
		NETCMD;

		if (!strcmp(ps_mode, "PostScript COLOR mode"))
			{
			sprintf(str, "graph_set_plot_color -graph 2 %g %g %g",
			red[2], green[2], blue[2]);
			}
		if (!strcmp(ps_mode, "PostScript GREYSCALE mode"))
			{
			sprintf(str, "graph_set_plot_color -graph 2 %g %g %g",
			1.0, 1.0, 1.0);
			}
		NETCMD;

		sprintf(str, "graph_set_xaxis 0 %.6f %.6f %d %d %d",
			gxmin, gxmax, gxpos, gxtic, gxprec); NETCMD;

		sprintf(str, "graph_set_yaxis 0 %.6f %.6f %d %d %d",
			gymin, gymax, gypos, gytic, gyprec); NETCMD;

		}

	if (strcmp(HIDE, "show"))
		{
	sprintf(str,
	"panel \"GRAPH%d!graph\" -w container -p ui -xy %d,%d -wh %d,%d -show",
		win, x_window_location, y_window_location,
		window_width, window_height); NETCMD;
		}
	

	sprintf(str, "script -close"); NETCMD;
	fclose(netfile);
	sprintf(str, "script -play %s/script%d -echo no", output_dir, win); CMD;

        used[win] = 1;

	if (!table(output_table, table_file_name, output_dir, i, win, 
			gxprec, gyprec))
		return(0);

	okay_to_check_table = 1;

	return(1);
	}
if (AVSparameter_changed("table file name"))
	{
	if (okay_to_check_table)
		if (table(output_table, table_file_name, output_dir, i, win, 
				gxprec, gyprec))
			return(1);
	}
return(0);
}

 
/* ***********************************************************************/
/* Initialization for modules contained in this file.                    */
/* ***********************************************************************/
int ((*mod_list[])()) = {
	GRAPH2D_desc,
};
#define NMODS (sizeof(mod_list) / sizeof(char *))

AVSinit_modules()
{
	AVSinit_from_module_list(mod_list, NMODS);
}
 
static int intcompare(i,j)
int *i, *j;
{
return(*i - *j);
}

int calculate_graph(input_field, gtype, window_no, dist, 
	thresh_value, output_dir)
AVSfield_float *input_field;
int gtype;
int window_no;
int dist;
float thresh_value;
char *output_dir;
{

register int i, j;
register int n;
register int nthresh;

FILE *f1;
FILE *f2;
FILE *f3;

static float colorinc;
static float finc;
double accum;
double mean;
register int nmed;
int hour;
int hr;
int ndist;
float *fdata = NULL;
    

/* Create up to 3 ASCII data files containing summary information */

sprintf(str, "%s/f1.w%d", output_dir, window_no);
if ((f1 = fopen(str, "w")) == NULL)
	{
	AVSwarning("Cannot create f1 xyplot file %s\n.", str);
	return(0);
	}
if ((chmod(str, 0666)) == IOERROR)
	{
	fclose(f1);
	AVSwarning("Cannot set mode on f1 xyplot file %s\n.", str);
	return(0);
	}

if (gtype == MIN_MED_MAX_GRAPH)
	{
	sprintf(str, "%s/f2.w%d", output_dir, window_no);
	if ((f2 = fopen(str, "w")) == NULL)
		{
		AVSwarning("Cannot create f2 xyplot file %s\n.", str);
		return(0);
		}
	if ((chmod(str, 0666)) == IOERROR)
		{
		fclose(f1);
		fclose(f2);
		AVSwarning("Cannot set mode on f2 xyplot file %s\n.", str);
		return(0);
		}
	sprintf(str, "%s/f3.w%d", output_dir, window_no);
	if ((f3 = fopen(str, "w")) == NULL)
		{
		AVSwarning("Cannot create f3 xyplot file %s\n.", str);
		return(0);
		}
	if ((chmod(str, 0666)) == IOERROR)
		{
		fclose(f1);
		fclose(f2);
		fclose(f3);
		AVSwarning("Cannot set mode on f3 xyplot file %s\n.", str);
		return(0);
		}
	}

/* set zeros for all threshold values before 1st hour */

/*
if (gtype == THRESHOLD_GRAPH)
	for (i = 0; i < start_hour; ++i)
		fprintf(f1, "%d\t%d\t%.3f\n", i, 0, 0.0);
*/

ndist = 100/dist;
bigymax = 0.0;
bigthresh = 0.0;

for (hour = 0; hour < NT; ++hour)
	{
	hr = hour;
	n = 0;
	nthresh = 0;
	accum = 0.0;

	if (!strcmp(slice_axis, "X"))
		{
		for (j = 0; j < MAXZ(input_field); ++j)
			for (i = 0; i < MAXY(input_field); ++i)
				{
				accum += I4D(input_field,
					slice_plane, i, j, hour);
				++n;
				}
		}
	else if (!strcmp(slice_axis, "Y"))
		{
		for (j = 0; j < MAXZ(input_field); ++j)
			for (i = 0; i < MAXX(input_field); ++i)
				{
				accum += I4D(input_field,
					i, slice_plane, j, hour);
				++n;
				}
		}
	else if (!strcmp(slice_axis, "Z"))
		{
		for (j = 0; j < MAXY(input_field); ++j)
			for (i = 0; i < MAXX(input_field); ++i)
				{
				accum += I4D(input_field,
					i, j, slice_plane, hour);
				++n;
				}
		}
			
	if (gtype == MEAN_GRAPH)
		{
		mean = accum/((double) n);
		if (graph_no == 0)
			fprintf(f1, "%d\t%.8f\t%.3f\n", hr, mean, MEANCOLOR0);
		else if (graph_no == 1)
			fprintf(f1, "%d\t%.8f\t%.3f\n", hr, mean, MEANCOLOR1);
		else if (graph_no == 2)
			fprintf(f1, "%d\t%.8f\t%.3f\n", hr, mean, MEANCOLOR2);
		if (mean > bigymax)
			bigymax = mean;
		}

	if (fdata != NULL)

/* IAC CODE CHANGE : 		free((float *) fdata); */
		 free( fdata);
	if ((fdata = ((float *) malloc(sizeof(float) * n))) != NULL)	
		{
		n = 0;
		if (!strcmp(slice_axis, "X"))
			{
			for (j = 0; j < MAXZ(input_field); ++j)
				for (i = 0; i < MAXY(input_field); ++i)
					fdata[n++] = I4D(input_field,
						slice_plane, i, j, hour);
			}
		else if (!strcmp(slice_axis, "Y"))
			{
			for (j = 0; j < MAXZ(input_field); ++j)
				for (i = 0; i < MAXX(input_field); ++i)
					fdata[n++] = I4D(input_field,
						i, slice_plane, j, hour);
			}
		else if (!strcmp(slice_axis, "Z"))
			{
			for (j = 0; j < MAXY(input_field); ++j)
				for (i = 0; i < MAXX(input_field); ++i)
					fdata[n++] = I4D(input_field,
						i, j, slice_plane, hour);
			}
		qsort(fdata, n, sizeof(float), intcompare);
		if (gtype != MEAN_GRAPH)
			{
			if (fdata[n-1] > bigymax)
				bigymax = fdata[n-1];
			}

		finc = (((float) n) - 1.0)/((float) ndist);
		colorinc = (MAXCOLORRANGE - MINCOLORRANGE)/((float) ndist);
		if (gtype == DISTRIBUTION_GRAPH)
			{
			fprintf(f1, "%d\t%.8f\t%.3f\n",
				 hr, fdata[0], MAXCOLORRANGE); 
			for (i = 1; i < ndist; ++i)
				{
				j = finc * i;
				fprintf(f1, "%d\t%.8f\t%.3f\n", hr, fdata[j],
					MAXCOLORRANGE - (i * colorinc));
				}
			fprintf(f1, "%d\t%.8f\t%.3f\n",
	 			hr, fdata[n-1], MINCOLORRANGE); 
			}

		if (gtype == MIN_MED_MAX_GRAPH)
			{
			nmed = n/2;
			fprintf(f1, "%d\t%.8f\t%.3f\n", hr, fdata[0], 
				MAXCOLORRANGE);
			fprintf(f2, "%d\t%.8f\t%.3f\n", hr, fdata[nmed],
			 MAXCOLORRANGE - (MAXCOLORRANGE - MINCOLORRANGE)/2.0);
			fprintf(f3, "%d\t%.8f\t%.3f\n", hr, fdata[n-1],
				 MINCOLORRANGE);
			}
		if (gtype == THRESHOLD_GRAPH)
			{
			if (!strcmp(slice_axis, "X"))
				{
				for (j = 0; j < MAXZ(input_field); ++j)
					for (i = 0; i < MAXY(input_field); ++i)
						{
						if (I4D(input_field,slice_plane,
							i, j, hour)
							> thresh_value)
							++nthresh;
						}
				}
			if (!strcmp(slice_axis, "Y"))
				{
				for (j = 0; j < MAXZ(input_field); ++j)
					for (i = 0; i < MAXX(input_field); ++i)
						{
						if (I4D(input_field, i, 
							slice_plane, j, hour)
							> thresh_value)
							++nthresh;
						}
				}
			if (!strcmp(slice_axis, "Z"))
				{
				for (j = 0; j < MAXY(input_field); ++j)
					for (i = 0; i < MAXX(input_field); ++i)
						{
						if (I4D(input_field, i, j,
							slice_plane, hour)
							> thresh_value)
							++nthresh;
						}
				}
			if ((float) nthresh > bigthresh)
				bigthresh = nthresh;
		
			fprintf(f1, "%d\t%d\t%.3f\n", hr, nthresh, THRESHCOLOR);
			}

/* IAC CODE CHANGE : 		free((float *) fdata); */
		 free( fdata);
		}
	}
if (gtype == THRESHOLD_GRAPH)
	bigymax = bigthresh;
bigxmax = NT - 1;
	
fclose(f1);
if (gtype == MIN_MED_MAX_GRAPH)
	{
	fclose(f2); fclose(f3);
	}
return(1);
}

setxaxis(gtype, gxmin, gxmax, gxpos, gxtic, gxprec)
int gtype;
float *gxmin;
float *gxmax;
int *gxpos;
int *gxtic;
int *gxprec;
{
float x;

*gxmin = 0.0;
*gxmax = bigxmax;
*gxpos = 1;
*gxtic = 2;
*gxprec = 0;

AVSmodify_float_parameter("X axis min", AVS_VALUE, *gxmin, 0., 0.);
AVSmodify_float_parameter("X axis max", AVS_VALUE, *gxmax, 0., 0.);
AVSmodify_parameter("X axis tic placement", AVS_VALUE, 
	"X axis tics inside", NULL, NULL); 
AVSmodify_parameter("X axis number of tics", AVS_VALUE, *gxtic, NULL, NULL);
AVSmodify_parameter("X axis tic label precision", AVS_VALUE, 
	 *gxprec, NULL, NULL);
}
 
setyaxis(gtype, gymin, gymax, gypos, gytic, gyprec)
int gtype;
float *gymin;
float *gymax;
int *gypos;
int *gytic;
int *gyprec;
{
#define Y_PRECISION 4
#define Y_PRECISION_BAR 0
int y_prec;
float fround();

if (gtype != THRESHOLD_GRAPH)		
	y_prec = Y_PRECISION;       /* non-bar graph Y = float precision */ 
else
	y_prec = Y_PRECISION_BAR;  /* bar graph Y = int N without decimals */

*gymin = 0.0;
/*
*gymax = fround(bigymax, y_prec);
*/
*gymax = bigymax;
*gypos = 1;
*gytic = 2;
*gyprec = y_prec;

AVSmodify_float_parameter("Y axis min", AVS_VALUE, *gymin, 0., 0.);
AVSmodify_float_parameter("Y axis max", AVS_VALUE, *gymax, 0., 0.);
AVSmodify_parameter("Y axis tic placement", AVS_VALUE, 
	"Y axis tics inside", NULL, NULL); 
AVSmodify_parameter("Y axis number of tics", AVS_VALUE, *gytic, NULL, NULL);
AVSmodify_parameter("Y axis tic label precision", AVS_VALUE,
	*gyprec, NULL, NULL);
}

float fround(x, prec)
float x;
int prec;
{
float ftemp;
float factor;
int itemp;
register int i;

if (prec == 0)
	{
	ftemp = (float) ((int) (x + 0.5));
	}
else
	{
	factor = 10.0;
	for (i = 1; i < prec; ++i)
		factor = factor * 10.0;

	ftemp = x * factor + 0.5;
	itemp = (int) (ftemp);
	ftemp = (float) (itemp);
	ftemp = ftemp/factor;
	}
return(ftemp);
}

int table(output_table, table_file_name, output_dir, gtype, window_no,
	 xprec, yprec)
char *output_table;
char *table_file_name;
char *output_dir;
int gtype;
int window_no;
{
static char xformat[32];
static char yformat[32];
static char local_filename[256];

float x, y;

FILE *foutput;
FILE *finput;


xformat[0] = yformat[0] = '%';
xformat[1] = yformat[1] = '.';
sprintf(xformat+2, "%df", xprec);
sprintf(yformat+2, "%df", yprec);


if (table_file_name == NULL)
	sprintf(local_filename, "/tmp/table.%d", getpid());
else
	sprintf(local_filename, "%s", table_file_name);

if ((foutput = fopen(local_filename, "w")) == NULL)
	{
	AVSwarning("File %s cannot be created.", local_filename);
	return(0);
	}


sprintf(str, "%s/f1.w%d", output_dir, window_no);

if ((finput = fopen(str, "r")) == NULL)
	{
	fclose(foutput);
	AVSwarning("Cannot open xyplot file %s\n", str);
	return(0);
	}

while (NULL != fgets(str, 512, finput))
	{
	sscanf(str, "%g%g", &x, &y);
	fprintf(foutput, xformat, x);
	fprintf(foutput, "\t");
	fprintf(foutput, yformat, y);
	fprintf(foutput, "\n");
	}
fclose(finput);

if (gtype == MIN_MED_MAX_GRAPH)
	{
	sprintf(str, "%s/f2.w%d", output_dir, window_no);
	if ((finput = fopen(str, "r")) == NULL)
		{
		AVSwarning("Cannot open xyplot file %s\n", str);
		return(0);
		}

	while (NULL != fgets(str, 512, finput))
		{
		sscanf(str, "%g%g", &x, &y);
		fprintf(foutput, xformat, x);
		fprintf(foutput, "\t");
		fprintf(foutput, yformat, y);
		fprintf(foutput, "\n");
		}
	fclose(finput);
	sprintf(str, "%s/f3.w%d", output_dir, window_no);
	if ((finput = fopen(str, "r")) == NULL)
		{
		AVSwarning("Cannot open xyplot file %s\n", str);
		return(0);
		}

	while (NULL != fgets(str, 512, finput))
		{
		sscanf(str, "%g%g", &x, &y);
		fprintf(foutput, xformat, x);
		fprintf(foutput, "\t");
		fprintf(foutput, yformat, y);
		fprintf(foutput, "\n");
		}
	fclose(finput);
	}
fclose(foutput);

AVSmodify_parameter("output table", AVS_VALUE, local_filename, NULL, NULL);
AVSmodify_parameter("table file name", AVS_VALUE, local_filename, NULL, NULL);
return(1);
}
