/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/* 
 * KEYFRAME ANIMATOR:
 * created by Brian Kaplan
 *      kaplan@cica.indiana.edu
 *      Center for Innovative Computer Applications (CICA)
 *      Indiana University
 *
 * Copyright (c) 1993 Brian Kaplan, CICA, Indiana University
 * All rights reserved.
 *
 * This material was developed at the Center for Innovative Computer
 * Applications (CICA), Indiana University.  Permission to copy this
 * software or any portion of it, to redistribute it, and to use it for
 * any purpose and without fee is granted, subject to the following
 * restrictions and understandings.
 *
 * 1. Any copy made of this software must include this copyright notice
 * in full.  Any materials containing this software or any part of this
 * software must include this copyright notice in full.
 *
 * 2. Users of this software agree to make their best efforts (a) to
 * return to the Center for Innovative Computer Applications any improvements
 * or extensions that they make, so that these may be included in future
 * releases; and (b) to inform Indiana University of noteworthy uses of this
 * software.
 *
 * 3. All materials developed as a consequence of the use of this
 * software shall duly acknowledge such use, in accordance with the usual
 * standards of acknowledging credit in academic research.
 *
 * 4. Indiana University has made no warrantee or representation that the
 * operation of this software will be error-free, and Indiana University
 * is under no obligation to provide any services, by way of maintenance,
 * update, or otherwise.
 *
 * 5. In conjunction with products arising from the use of this material,
 * there shall be no use of the name of Indiana University or the Center
 * for Innovative Computer Applications nor of any adaptation thereof in
 * any advertising, promotional, or sales literature without prior written
 * consent from Indiana University in each case.
 *
 */

#include <stdio.h>
#include "Keyframe.h"

/************************
 * DATABASE DEFINITIONS *
 ************************/

/* structures to hold the animation information */
struct Objinfo {      /* object info that remains constant for the animation */
  char name[MAXNAMELENGTH];                             /* the object's name */
  char parent[MAXNAMELENGTH];                  /* the object's parent object */
};
struct Frameinfo {                   /* information which is frame-dependant */
  int keyframe;                                  /* is this frame a keyframe */
};

struct Objinfo *Objlist;                 /* stores information about objects */
struct Frameinfo *Framelist;              /* stores information about frames */
double ***Anim;               /* stores animation values for transformations */

int Totobjs;                     /* the total number of objects in animation */
int Totframes;                /* the total number of frames in the animation */
int Totchannels;            /* the total number of channels in the animation */

/* some non-global database routine templates */
int intiframelist(int);
int initobjlist(int);
int addframelist(int);
int addobjlist(int);
double ***Alloc3D(int, int, int);

/**************************************************************
 *  ROUTINES FOR HANDLING THE ANIMATION DATABASE              *
 **************************************************************/

int Anim_init(numframes,numobjs,numchannels)
/**************************************************************/
/* initializes the animation database                         */
/* numframes   = number of frames in the animation            */
/* numobjs     = number of objects                            */
/* numchannels = number of channels (animation parameters)    */
/**************************************************************/
     int numframes,numobjs,numchannels;
{
  int x,y;
  
  /* set the global variables */
  Totframes = numframes;
  Totobjs = numobjs;
  Totchannels = numchannels;

  fprintf(stderr,"ANIM:  Creating %d frames, %d object(s), %d channels...\n",
	  Totframes,Totobjs,Totchannels);

  /* initialize the Frame List (which contains information about frames) */
  if (initframelist(Totframes) < 0) return(-1);
  /* initialize the Object List (containing information about objects) */
  if (initobjlist(Totobjs) < 0) return(-1);

  /* initialize the Animation Storage (containing channel animation info) */ 
  Anim = Alloc3D(Totframes,Totobjs,Totchannels);
  if (Anim == NULL) return(-1);

  /* initialize channel info to default values */
  for (x=0; x<Totframes; x++)
    for (y=0; y<Totobjs; y++) {
      Anim[x][y][XTRN] = 0.0;
      Anim[x][y][YTRN] = 0.0;
      Anim[x][y][ZTRN] = 0.0;
      Anim[x][y][SIZE] = 1.0;
      Anim[x][y][QTRX] = 0.0;
      Anim[x][y][QTRY] = 0.0;
      Anim[x][y][QTRZ] = 0.0;
      Anim[x][y][QTRC] = 1.0;
    }

  return(0);
}

int initframelist(numframes)
/**************************************************************/
     int numframes;
{
  int i;

  Framelist = (struct Frameinfo *) malloc(numframes*sizeof(struct Frameinfo));
  if (Framelist == NULL) return(-1);

  for (i=0; i<numframes; i++) {
    /* set the keyframe info for the frame */
    Framelist[i].keyframe = 0;
  }

  return(0);
}

int initobjlist(numobjs)
/**************************************************************/
     int numobjs;
{
  int i;
  
  Objlist = (struct Objinfo *) malloc(numobjs*sizeof(struct Objinfo));
  if (Objlist == NULL) return(-1);

  for (i=0; i<numobjs; i++) {
    /* set the names for the objects */
    sprintf(Objlist[i].name,"%s-%d","EMPTY",i+1);

    /* set the objects parents */
    strcpy(Objlist[i].parent,"%top");
  }

  return(0);
}

void Anim_free()
/**************************************************************/
/* Frees the memory associated with the animation database.   */
/**************************************************************/
{
  free(Objlist);
  free(Framelist);
  free(Anim);
}

int Anim_addframes(n)
/**************************************************************/
/* add frames to the animation database                       */
/* n = number of frames to add to the database                */
/**************************************************************/
     int n;
{
  double ***temp;
  int x,y,z;

  fprintf(stderr,"ANIM:  Adding %d new frame(s)...\n",n);

  temp = Alloc3D(Totframes+n,Totobjs,Totchannels);
  if (temp == NULL) return(-1);

  for (x=0; x<Totframes; x++)
    for (y=0; y<Totobjs; y++)
      for(z=0; z<Totchannels; z++) temp[x][y][z] = Anim[x][y][z];
  free(Anim);
  Anim = temp;

  for (x=Totframes; x<(Totframes+n); x++)
    for (y=0; y<Totobjs; y++) {
      Anim[x][y][XTRN] = 0.0;
      Anim[x][y][YTRN] = 0.0;
      Anim[x][y][ZTRN] = 0.0;
      Anim[x][y][SIZE] = 1.0;
      Anim[x][y][QTRX] = 0.0;
      Anim[x][y][QTRY] = 0.0;
      Anim[x][y][QTRZ] = 0.0;
      Anim[x][y][QTRC] = 1.0;
    }
  
  /* note that addframelist depends on Totframes being equal to the */
  /* number of frames BEFORE the new frames were added */
  if (addframelist(n) < 0) return(-1);

  Totframes += n;

  return(0);
}

int addframelist(newframes)
/**************************************************************/
     int newframes;
{
  int i,j;
  struct Frameinfo *temp;
  
  j = Totframes + newframes;

  temp = (struct Frameinfo *) malloc(j*sizeof(struct Frameinfo));
  if (temp == NULL) return(-1);

  for (i=0; i<Totframes; i++) {
      temp[i].keyframe = Framelist[i].keyframe;
  }
  free(Framelist);
  Framelist = temp;

  for (i=Totframes; i<j; i++) {
    /* set the keyframe info for the new frame */
    Framelist[i].keyframe = 0;
  }

  return(0);
}

int Anim_addobjects(n)
/**************************************************************/
/* add objects to the animation database                      */
/* n = number of objects to add to the database               */
/**************************************************************/
     int n;
{
  double ***temp;
  int x,y,z;

  fprintf(stderr,"ANIM:  Adding %d new object(s)...\n",n);

  temp = Alloc3D(Totframes,Totobjs+n,Totchannels);
  if (temp == NULL) return(-1);

  for (x=0; x<Totframes; x++)
    for (y=0; y<Totobjs; y++)
      for(z=0; z<Totchannels; z++) temp[x][y][z] = Anim[x][y][z];
  free(Anim);
  Anim = temp;

  for (x=0; x<Totframes; x++)
    for (y=Totobjs; y<(Totobjs+n); y++) {
      Anim[x][y][XTRN] = 0.0;
      Anim[x][y][YTRN] = 0.0;
      Anim[x][y][ZTRN] = 0.0;
      Anim[x][y][SIZE] = 1.0;
      Anim[x][y][QTRX] = 0.0;
      Anim[x][y][QTRY] = 0.0;
      Anim[x][y][QTRZ] = 0.0;
      Anim[x][y][QTRC] = 1.0;
    }
  /* note that addobjlist depends on Totframes being equal to the */
  /* number of frames BEFORE the new frames were added */
  if (addobjlist(n) < 0) return(-1);

  Totobjs += n;

  return(0);
}


int addobjlist(newobjs)
/**************************************************************/
     int newobjs;
{
  int i,j;
  struct Objinfo *temp;
  
  j = Totobjs + newobjs;

  temp = (struct Objinfo *) malloc(j*sizeof(struct Objinfo));
  if (temp == NULL) return(-1);
  
  for (i=0; i<Totobjs; i++) {
      strcpy(temp[i].name, Objlist[i].name);
      strcpy(temp[i].parent, Objlist[i].parent);
  }
  free(Objlist);
  Objlist = temp;

  for (i=Totobjs; i<j; i++) {
    /* set the names for the objects */
    sprintf(Objlist[i].name,"%s-%d","EMPTY",i+1);

    /* set the objects parents */
    strcpy(Objlist[i].parent,"%top");
  }

  return(0);
}

double ***Alloc3D(x,y,z)
/**************************************************************/
     int x,y,z;
{
  double ***arr,*p,**q;
  int i,j;
  unsigned int totsiz;

/* use 8-byte alignment for now */
#define ALIGN_BNDRY 8

/* this is for perror */
extern int errno;

  /* allocate and return pointer to 3D double data,             */
  /* with header pointers.                                      */
  /* Declare your variable to point to the data with:           */
  /*         double ***arr;                                     */
  /* Call this function with the dimensions of the array:       */
  /*         arr = Alloc3D(10, 10, 10);                         */
  /* Use C array indexing to set and access elements, as in:    */
  /*         arr[xi][yi][zi]                                    */
  /* The data itself starts at address &arr[0][0][0]            */

  totsiz = x*sizeof(double**)+
           x*y*sizeof(double*)+
	   x*y*z*sizeof(double);

  /* round up to nearest ALIGN_BNDRY boundary by */
  /* adding (ALIGN_BNDRY - 1) and mask off low order bits. */
  /* Note that this only works when ALIGN_BNDRY is even power of 2!!! */
  totsiz = (totsiz+(ALIGN_BNDRY-1)) & ~(ALIGN_BNDRY-1);

#ifdef DEBUG_ALLOC3D
  printf("Alloc3D requesting %d bytes of memory.\n", totsiz);
#endif

  arr = (double ***)sbrk(totsiz);  /* ask for more resources */

  if (((int)arr) <= 0) {
    fprintf(stderr, "Alloc3D: sbrk returns < 0 on request of %d bytes!\n",
	    totsiz);
    perror("System error was ");
    return (NULL);
  }

  /* now make sure that arr is an aligned pointer */
  arr = (double ***)((((unsigned int)arr)+(ALIGN_BNDRY-1)) & ~(ALIGN_BNDRY-1));

  q = (double **)(arr+x);
  p = (double *)(q+x*y);
  for (i=0; i<x; i++) {
    arr[i] = q;
    q += y;
    for (j=0;j<y;j++)   {
      arr[i][j] = p;
      p += z;
    }
  }

  return(arr);
}

double ***OldAlloc3D(x,y,z)
/**************************************************************/
     int x,y,z;
{
  double ***arr,*p,**q;
  int i,j;

  /* allocate and return pointer to 3D double data,             */
  /* with header pointers.                                      */
  /* Declare your variable to point to the data with:           */
  /*         double ***arr;                                     */
  /* Call this function with the dimensions of the array:       */
  /*         arr = Alloc3D(10, 10, 10);                         */
  /* Use C array indexing to set and access elements, as in:    */
  /*         arr[xi][yi][zi]                                    */
  /* The data itself starts at address &arr[0][0][0]            */

  arr = (double ***)malloc((unsigned)(x*sizeof(double **)+
				      x*y*sizeof(double *)+
				      x*y*z*sizeof(double)));
  if (arr == NULL) return(arr);

  if (!arr) return(0);
  
  q = (double **)(arr+x);
  p = (double *)(q+x*y);
  for (i=0; i<x; i++) {
    arr[i] = q;
    q += y;
    for (j=0;j<y;j++)	{
      arr[i][j] = p;
      p += z;
    }
  }

  return(arr);
}

/***********************************************
 * FUNCTIONS FOR HANDLING THE KEYFRAME DATABASE *
 ***********************************************/

int NextKey(cframe,end)
     int cframe,end;
/**************************************************************/
/* Does a search for a keyframe FOLLOWING the specified frame */
/* up to the end of the animation window (specified by end).  */
/* Frame end is always considered a keyframe.                 */
/* If end > the number of frames in the animation             */
/* it will be reset to the last frame in the animation.       */
/* Note:  specifying cframe as 0 will return the first        */
/*        keyframe in the animation window.                   */
/* returns -1 if cframe is less than 0 or greater than the    */
/*        end window.                                         */
/* returns the end window if cframe equals the end window.    */
/**************************************************************/
{
  if (end > Totframes) end = Totframes;
  if (cframe < 0 || cframe > end) return(-1);

  cframe++;
  if (cframe > end) return(end);
  
  while(Framelist[cframe-1].keyframe == 0) {
    cframe++;
    if (cframe > end) return(end);
  }

  return(cframe);
}

int PrevKey(start,cframe)
     int start,cframe;
/**************************************************************/
/* Does a backward search for a keyframe PRECEEDING the       */
/* specified frame to the start of the animation window       */
/* (specified by start).  start is always seen as a keyframe. */
/* If start < 1, it will be reset to the first frame          */
/* in the animation.                                          */
/* Note:  specifying cframe as Totframes+1 will return the    */
/*        last keyframe in the animation window.              */
/* returns -1 if cframe is greater than the number of frames  */
/*        in the animation or less than the start window.     */
/* returns the start window if cframe = the start window.     */
/**************************************************************/
{
  if (start < 1) start = 1;
  if (cframe > (Totframes+1) || cframe < start) return(-1);

  cframe--;
  if (cframe < start) return(start);
  
  while(Framelist[cframe-1].keyframe == 0) {
    cframe--;
    if (cframe < start) return(start);
  }

  return(cframe);
}

int GetNumobjs()
{
    return(Totobjs);
}

int GetNumframes()
{
    return(Totframes);
}

int GetNumchannels()
{
    return(Totchannels);
}

void SetKey(frame, key)
    int frame;
    int key;
{
    Framelist[frame-1].keyframe = key;
}

int GetKey(frame)
    int frame;
{
    return(Framelist[frame-1].keyframe);
}

void SetName(obj, name)
    int obj;
    char *name;
{
    strcpy(Objlist[obj-1].name, name);
}

char *GetName(obj)
    int obj;
{
    return(Objlist[obj-1].name);
}

void SetParent(obj, parent)
    int obj;
    char *parent;
{
    strcpy(Objlist[obj-1].parent, parent);
}

char *GetParent(obj)
    int obj;
{
    return(Objlist[obj-1].parent);
}

void SetChan(frame, obj, chan, val)
/**************************************************************/
/* Sets the value of a channel at a given frame for a given   */
/* object.                                                    */
/**************************************************************/
    int frame;
    int obj;
    int chan;
    double val;
{
    Anim[frame-1][obj-1][chan] = val;    
}

double GetChan(frame, obj, chan)
/**************************************************************/
/* Returns the value of a channel at a given frame for a      */
/* given object.                                              */
/**************************************************************/
    int frame;
    int obj;
    int chan;
{
    return(Anim[frame-1][obj-1][chan]);    
}

void SetRot(frame, obj, q)
/**************************************************************/
/* Sets the rotation at frame frame for object obj to the     */
/* quaternion in q.                                           */
/**************************************************************/
    int frame;
    int obj;
    double q[4];
{
    frame--;
    obj--;
    
    Anim[frame][obj][QTRX] = q[0];
    Anim[frame][obj][QTRY] = q[1];
    Anim[frame][obj][QTRZ] = q[2];
    Anim[frame][obj][QTRC] = q[3];
}

void GetRot(q, frame, obj)
/**************************************************************/
/* Get the rotation at frame frame for object obj into the    */
/* quaternion in q.                                           */
/**************************************************************/
    int frame;
    int obj;
    double q[4];
{
    frame--;
    obj--;
    
    q[0] = Anim[frame][obj][QTRX];
    q[1] = Anim[frame][obj][QTRY];
    q[2] = Anim[frame][obj][QTRZ];
    q[3] = Anim[frame][obj][QTRC];
}

void SetTrans(frame, obj, t)
/**************************************************************/
/* Sets the translation at frame frame for object obj to the  */
/* translation in t.                                          */
/**************************************************************/
    int frame;
    int obj;
    double t[3];
{
    frame--;
    obj--;
    
    Anim[frame][obj][XTRN] = t[0];
    Anim[frame][obj][YTRN] = t[1];
    Anim[frame][obj][ZTRN] = t[2];
}

void GetTrans(t, frame, obj)
/**************************************************************/
/* Gets the translation at frame frame for object obj         */
/**************************************************************/
    int frame;
    int obj;
    double t[3];
{
    frame--;
    obj--;
    
    t[0] = Anim[frame][obj][XTRN];
    t[1] = Anim[frame][obj][YTRN];
    t[2] = Anim[frame][obj][ZTRN];
}

void SetSize(frame, obj, s)
/**************************************************************/
/* Sets the size at frame frame for object obj to the         */
/* size in s.                                                 */
/**************************************************************/
    int frame;
    int obj;
    double s;
{
    Anim[frame-1][obj-1][SIZE] = s;
}

void GetSize(s, frame, obj)
/**************************************************************/
/* Gets the size at frame frame for object obj                */
/**************************************************************/
    int frame;
    int obj;
    double s[3];
{
    s[0] = s[1] = s[2] = Anim[frame-1][obj-1][SIZE];
}

