/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/*
 * Name:
 * Marker_Select - Define and allow Pick Selection of Markers
 *
 * Type: Mapper, Image viewer upstream control module
 *
 * Language:  K&R C
 *
 * Author:    Ian Curington, AVS/UNIRAS Ltd. UK
 *
 * Description:
 *
 * Allow marker pick identification and display 
 * from an input marker list and the image viewer
 * system. Takes as input an AVS field with
 * a list of coordinates for markers, displays them
 * over an image (map) in the image viewer, and
 * reacts to mouse picks on these markers, sending
 * the pick selection to the output port.
 * Markers are defined by geometric polyline vector lists,
 * not ROI pixel masks.
 *
 * The input marker field is allowed to be either fully
 * specified, or abreviated, with only X, Y, and PID.
 * If a full marker field is specified, then you can
 * externally control position, shape, size, etc.
 * Any marker field sent out on the output port can
 * be read back in to the input field port. This way
 * state save and restore are possible. New marker
 * lists can be made using the "marker_pick" module.
 * This module does not allow edition of position or shape,
 * just allows query picking of pre-defined markers.
 *
 * Two integer output ports provide information on the picked
 * marker, the index number and the type or PID tag
 * associated with that marker. These ports only change when
 * a valid pick is made, otherwise they have values of
 * -1, and -1, indicating no valid pick selection has been made.
 *
 * Inputs:
 *   Input Image: Reference image, same as that sent for
 *                display in the image viewer, used
 *                for feedback path.
 *   Mouse Info:  Part of upstream pick feedback communication
 *                with the image viewer
 *   IV Struct:   Part of upstream communications with the image viewer.
 *   Poly_Marker: Input field,
 *                  for marker pick list definition.
 *                This field is 1D uniform integer N-vector.
 *                If N-vector is 3, then each marker is assumed to be
 *                at an X, Y position, in pixels, relative to the top
 *                left corner of the image. The third value is an
 *                arbitrary PID tag value, that is simply displayed
 *                and passed out on an integer output port when
 *                a marker is picked with the mouse.
 *                If the field N-vector size is greater than 3, then
 *                it is assumed to be a full marker spec as defined
 *                in "marker.h", usually 7. This can fully specifiy
 *                variable marker shape, size, position, and hidden
 *                stroke vector used in the shape creation.
 *
 * Outputs:
 *   Poly_Marker_Output:
 *                This contains a full marker list, as a mirror to
 *                to the input list. It is always full 7-vector,
 *                even if the input port is the 3-vector format.
 *                This list can contain one additional marker not
 *                specified on the input list - to show an "X" over
 *                the top of the picked marker for display feedback
 *                for a successful pick operation.
 *                This port can be saved to a file with the
 *                "write field" module, and later restored with
 *                "read_field" on the input port during a later session.
 *
 *  IV Draw Struct:
 *                Part of the image viewer upstream communication
 *                system.
 *
 *  Pick-Index:   Integer output port with the index value of the
 *                picked marker. If no pick is valid, this port = -1.
 *  Pick-PID:     Integer output port with any PID tag value associated
 *                at the input port for the picked marker. This
 *                can be used by other modules in the network to
 *                provide additional query or more detailed information
 *                or dynamic behavior resulting from the pick operation.
 *
 * Parameters:
 *  Set Pick Mode: A one-shot button that registers the pick operation
 *                mode with the image viewer, in the case that multiple
 *                modules are also requesting pick communication with the
 *                image viewer.
 *
 *  Pick Status Text Block:
 *                This is a square read-only text block used to
 *                show annotation about the picked marker.
 *
 *  Marker Size:
 *                An integer slider from 1 to 10 (default=2) that
 *                controls the relative size of the markers
 *                when the input port is in the short X, Y, PID mode.
 *                When the full marker list is presented at the input
 *                port, the size is specified per marker, so this
 *                slider has no effect.
 *
 * --------------------------
 *
 * Adapted from ip_read_line,
 * by Ian Curington
 *
 * Revision: 
 * 18 March 93 ianc - Original (Ensign Demo)
 * 22 March 93 ianc - inverted output port Y
 *                  - fixed buffer update bug
 *  7 Feb   94 ianc - adapted from Polyline
 *  8 Feb   94 ianc - updated output firing guard, fill style
 * 27 May   94 ianc - derived from marker_pick,
 *                    changed over to select for Oracle project.
 *  6 June  94 ianc - modified functionality for Oracle project.
 *  9 June  94 ianc - additional flow control work
 * 10 June  94 ianc - int output ports and scale slider  added
 * 20 June  94 ianc - update of pick search algorithm
 */

/*
                  Copyright (c) 1992, 1993, 1994 by
                  Advanced Visual Systems Inc.
                  All Rights Reserved
      
      This software comprises unpublished confidential information of
      Advanced Visual Systems Inc. and may not be used, copied or made
      available to anyone, except in accordance with the license
      under which it is furnished.
      
*/

/**********************************************************/
#include <stdio.h>
/* IAC CODE CHANGE : #include <math.h> */
#include <avs/avs_math.h>

#include <avs/avs.h>
#include <avs/field.h>
#include <avs/udata.h>
#include <avs/image_draw.h>
#include <avs/image_upstrm.h>

#include "marker.h"

#define BUFSIZE 1024
#define CLI_BUFFER_SIZE BUFSIZE*32
#define MAX_POLYLINE 500
#define CMD_STR \
 "%s -scene_id %d -image_name \"%s\" -module $Module %f %f %f %f"

/*
 * set module default marker and hilite types
 */
#define DEFAULT_MARKER MARKER_DIAMOND
#define DEFAULT_HILITE MARKER_X
#define DEFAULT_FILL      1
#define DEFAULT_SIZE      4
#define DEFAULT_HI_SIZE   2

/**********************************************************/

typedef struct
{
  char buffer[CLI_BUFFER_SIZE], *bptr;
} CLIbuffer;

static CLIbuffer commandBuffer;
static int       bufferInited = 0;
static char      imageName[80];

/**********************************************************/
/* Module Description Section */
/**********************************************************/
marker_select_desc()
{
  int port, param;
  char spath[128];
  extern char *AVSpath;
  int marker_select_compute();
    
   /***********************/
   /* General Info        */
   /***********************/
  AVSset_module_name("Marker Select", MODULE_MAPPER);

  /* can't be REENTRANT? (compute function has statics) */
  AVSset_module_flags(COOPERATIVE);

   /***********************/
   /* Input Ports         */
   /***********************/
  AVScreate_input_port("input", "field 2D", REQUIRED);

  port = AVScreate_input_port("Image Viewer Id", "struct image_viewer_id",
                        INVISIBLE | OPTIONAL);
  AVSset_input_class(port, "Image Draw:image_viewer_id");

  port = AVScreate_input_port("Mouse Info","struct mouse_info",
                        INVISIBLE | OPTIONAL);
  AVSset_input_class(port, "Image Draw:mouse_info");

  AVScreate_input_port("Polyline Input", MARKER_FIELD_I_SPEC, OPTIONAL);

   /***********************/
   /* Output Ports        */
   /***********************/
  AVScreate_output_port("Polyline Output", MARKER_FIELD_SPEC);

  AVScreate_output_port("Image Draw", "struct image_draw");

  port =
     AVScreate_output_port(/*name*/ "Pick-Index", /*type*/"integer");

  port =
     AVScreate_output_port(/*name*/ "Pick-PID", /*type*/"integer");


   /***********************/
   /* Parameters          */
   /***********************/
  param = AVSadd_parameter("set pick mode", "oneshot", 0, 0, 1);
  AVSconnect_widget(param, "oneshot");

  param=
    AVSadd_parameter("Selected Marker","string_block",
                     " === Selected Marker Information ===\n","","");
    AVSadd_parameter_prop(param,"height","integer",8);
    AVSadd_parameter_prop(param,"width","integer",4);

  param = AVSadd_parameter("Marker Size", "integer", 2, 1, 10);
  AVSconnect_widget(param, "islider");

   /***********************/
   /* Functions / Classes */
   /***********************/
  AVSset_compute_proc(marker_select_compute);

  /* collect runtime configurable data type */
  if (AVSpath != NULL)
    strcpy(spath, AVSpath);
  else
    strcpy(spath, "/usr/avs");
  strcat(spath, "/include/image_draw.h");

  AVSload_user_data_types(spath);
}

/**********************************************************/
/* Module Computational Section */
/**********************************************************/
int marker_select_compute(input,
                     iv_id,
                     mouseInfo,
                     poly_in,
                     poly,
                     im_draw,
                     pick_index,
                     pick_pid,
                     mode,
                     markerInfo,
                     marker_size )


    AVSfield        *input;      /* input image */
    image_viewer_id *iv_id;      /* image viewer feedback */
    mouse_info      *mouseInfo;  /* mouse status feedback */
    AVSfield_int    *poly_in;    /* input marker_select field */
    AVSfield_int    **poly;      /* output marker_select field */
    image_draw      **im_draw;   /* output draw commands (unused) */
    int             *pick_index; /* output picked index */
    int             *pick_pid;   /* output picked  */
    int             mode;        /* set mouse active mode for this module */
    char            *markerInfo;     /* marker information */
    int             marker_size; /* size scale factor control */

{
  char      command[BUFSIZE], label[256];
  CLIbuffer *clibuf;
  int       i, j, data_type, mask, end, npixels, status;
  float     length, min, max;
  char      *in, *out, *line_buffer;
  short     *in_short, *out_short;
  float     *in_float, *out_float;
  int       dims[1];
  int       marker_type;
  int       marker_fill;
  int       x, y, sz;
  int       search_dist[ MAX_POLYLINE ];
  int       nearest;

  /********************************************************/
  /* statics for remembering last line, and marker_select */
  /********************************************************/
  static int       begin = 1;
  static int       startX, startY, endX, endY;
  static int       marker_select_buf[MARKER_VECLEN][MAX_POLYLINE];
  static int       num_markers;
  static int       num_marker_selects;
  static int       picked_marker;
  static int       stat_pick_index, stat_pick_pid;


  /*
   * ======================================================
   * Guard, Special Exit to keep it from running in an 
   * endless loop - if ONLY the input image changed,
   * do nothing.
   * This module is designed to be hooked into a figure 8
   * with the marker_draw module.
   */
  if (AVSinput_changed("input", 0) &&
      !AVSinput_changed("Image Viewer Id", 0) &&
      !AVSinput_changed("Mouse Info", 0) &&
      !AVSinput_changed("Polyline Input", 0))
  {
      return(0);
  }


  /*
   * ======================================================
   * Build Output drawing command list
   * not used, but causes upstream events to be registered.
   */
  if (*im_draw == NULL)
    *im_draw = (image_draw *) AVSdata_alloc("struct image_draw", 0);

  if (input->uniform != UNIFORM)
  {
    AVSerror("Incompatible data type: can only process 2D uniform fields");
    return(0);
  }


  /*
   * ======================================================
   * Pre-Initialize pick output integer ports
   *
   */
   stat_pick_index = stat_pick_pid = -1;
   *pick_index = stat_pick_index;
   *pick_pid   = stat_pick_pid;

  /*
   * ======================================================
   * Initialize CLI command buffer
   * requests are blocked up for efficiency
   */
  clibuf = &commandBuffer;

  if (!bufferInited)
  {
    clibuf->bptr = clibuf->buffer;
    bufferInited = 1;
  }

  /*
   * ======================================================
   * This module will do nothing unless
   * it has upstream feedback
   */
  if (iv_id == NULL)
  {
    /* not connected to image viewer */
    imageName[0] = '\0';
    return(0);
  }

  /*
   * ======================================================
   * Process the attention command button
   */
  if (AVSparameter_changed("set pick mode"))
  {
    if (imageName[0] != '\0')
    {
      sprintf(command, "image_set_pick_mode \"marker_select\"");
      add_command(command, clibuf, 1);
    }
    /**
    AVSmark_output_unchanged("Polyline Output");
     **/
  }

  /*
   * ======================================================
   * Pre-initialize marker buffer if Optional input field
   *  Keep an internal list, but copy it from the
   *  input field each time, to make sure we are uptodate
   *  this will allow animation in the future.
   *  This supports full external control of marker or
   *  a simplified field with just position and PID.
   */
  if ( poly_in )
  {
     for (i=0; i<MAXX(poly_in); i++)
     {
        if ( poly_in->veclen == MARKER_VECLEN )
        {
           for (j=0; j<MARKER_VECLEN; j++)
           {
              marker_select_buf[j][i] = poly_in->data[i*MARKER_VECLEN+j];
           }
        }
        else if (poly_in->veclen >=3 )
        {
            /*
             * override marker attributes, setup defaults
             * for Oracle project
             */
           /* X coord */
           marker_select_buf[0][i] = poly_in->data[i*(poly_in->veclen)+0];
           /* Y coord */
           marker_select_buf[1][i] = poly_in->data[i*(poly_in->veclen)+1];

           /* Preset attributes */
           marker_select_buf[2][i] = DEFAULT_MARKER;
           marker_select_buf[3][i] = DEFAULT_SIZE * marker_size;
           marker_select_buf[4][i] = DEFAULT_FILL;
           marker_select_buf[5][i] = marker_select_buf[0][i] ;
           marker_select_buf[6][i] = marker_select_buf[1][i] ;
        }
        else
        {
            AVSerror("Incompatible Marker list veclen");
            return(0);
        }
     }
     num_markers        = MAXX(poly_in);
     num_marker_selects = MAXX(poly_in);
  }
  else
  {
     num_markers        = 0;
     num_marker_selects = 0;
  }


  /*
   * ======================================================
   * check buffer size
   */
  if ( num_marker_selects+1 >= MAX_POLYLINE )
  {
      AVSerror("marker_select buffer is full, no more room.");
      return(0);
  }

  /*
   * ======================================================
   * Process any changes of Image Viewer scene context
   * using upstream connection
   * This registers the fact that we want mouse event
   * control in the image viewer.
   */

  if (AVSinput_changed("Image Viewer Id", 0))
  {
    sprintf(command,
    "image_wakeup_when_defined -scene_id %d -mesh_id %d -module \"$Module\"",
       iv_id->scene_id, input->mesh_id);
    add_command(command, clibuf, 1);
  }

  /*
   * ======================================================
   * Process Mouse Click Events 
   * using upstream connection
   */
  if (mouseInfo && AVSinput_changed("Mouse Info", 0))
  {
    if (mouseInfo->nEvents == 0)
    {
      /*
       * ======================================================
       * Get Setup to receive events for the new image
       *   a new image - stop events for old, then
       *   add the new one. 
       */ 
      if (imageName[0] != '\0')
      {
          sprintf(command,
          "%s %s %d %s \"%s\" -module $Module -func_id \"%s\"",
              "image_stop_events",
              "-scene_id",
              iv_id->scene_id,
              "-image_name",
              imageName,
              "marker_select");
          add_command(command, clibuf, 1);
      }

      /*
       * ======================================================
       * request pick events from a new image
       * register request for events with the image viewer
       */
      strcpy(imageName, mouseInfo->image_name);
      mask = IMAGE_Button1Mask | IMAGE_PressMask | IMAGE_ReleaseMask |
                                 IMAGE_MotionMask;
      sprintf(command,
      "%s %d %s \"%s\" %s \"marker_select\" -mask %d",
          "image_select_events -scene_id",
          iv_id->scene_id,
          "-image_name",
          imageName,
          "-module $Module -func_id",
          mask);
      add_command(command, clibuf, 1);

      sprintf(command, "image_set_pick_mode \"marker_select\"");
      add_command(command, clibuf, 1);
    }
    else  /* mouse events in the queue */
    {
      /* got mouse events to deal with */
      end = 0;

      /*
       * ======================================================
       * loop over all events
       *  in this version we are only interested in
       *  the mouse down pick event, but the following code
       *  will also detect motion and mouse up gestures.
       */
      for (i = 0; i < mouseInfo->nEvents; i++)
      {
        if (((mouseInfo->buttonMask[i] & IMAGE_Button1Mask) &&
           (mouseInfo->buttonMask[i] & IMAGE_PressMask)))
        {
            /* button down defines the start point */
            startX = (int) mouseInfo->image_x[i];
            if (startX < 0)
              startX = 0;
            else if (startX > MAXX(input))
              startX = MAXX(input);

            startY = (int) mouseInfo->image_y[i];
            if (startY < 0)
              startY = 0;
            else if (startY > MAXY(input))
              startY = MAXY(input);

            begin = 1;
            end   = 0;
        }
        /*
         * Button Release detected
         */
        else if ((mouseInfo->buttonMask[i] & IMAGE_Button1Mask) &&
                 (mouseInfo->buttonMask[i] & IMAGE_ReleaseMask))
        {
            begin = 0;
            end   = 1;
        }
        else
        {
            /* most likely motion */
            begin = 0;
            end   = 0;
        }
      }

    }  /* end of mouse event to process */

    /*
     * Critical speed get-out claus:
     * Skip everything if it was a mouse event
     * we did not want to process
     * Global exit!
     */
    if ( !begin ) return(0);

  }  /* end of mouse info changed test */
      



  /*
   * ======================================================
   * process picking up of existing marker 
   * search list for match
   * look for a mouse pick inside of a search radius 
   * of the marker
   */
  if ( begin )
  {
	  /*
	   * find the (squared euclidean)
	   * distance from pick point to each marker
	   * and save in a list.
	   */
          for ( i=0; i < num_marker_selects ; i++ )
          {
              x    = marker_select_buf[0][i];
              y    = marker_select_buf[1][i];
	      search_dist[i] =  ( startX - x ) * ( startX - x ) +
				( startY - y ) * ( startY - y );
          }

	  /*
	   * search the list for the nearest marker,
	   * and throw out if too far away
	   */
          nearest = 80000; /* start in outer space */
          picked_marker = -1;
          for ( i=0; i < num_marker_selects ; i++ )
          {
	      if ( search_dist[i] < nearest )
	      {
		  nearest = search_dist[i];
		  picked_marker = i;
	      }
          }
          /* 20 pixel pick radius */
	  if ( picked_marker >= 0 &&
	       nearest > 400 ) picked_marker = -1;

	  /***
	   < old code >
          picked_marker = -1;
          for ( i=0; i < num_marker_selects ; i++ )
          {
              x    = marker_select_buf[0][i];
              y    = marker_select_buf[1][i];
              sz   = marker_select_buf[3][i] * 3;
              if ( startX > x-sz && startX < x+sz &&
                   startY > y-sz && startY < y+sz )
                  picked_marker = i;
          }
	  ***/

          begin = 0; /* toggle begin mode off for next time! */
  }


  /*
   * ======================================================
   * We have a pick! process it...
   * Highlight the picked marker with a new
   * overlay marker
   */
  if ( picked_marker >= 0 )
  {
          /*
           * add marker point to list
           */
          marker_fill = DEFAULT_FILL;
          marker_type = DEFAULT_HILITE;
          npixels     = DEFAULT_HI_SIZE * marker_size;
          startX      = marker_select_buf[0][picked_marker];
          startY      = marker_select_buf[1][picked_marker];
          endX        = startX;
          endY        = startY;

          /* Assemble new data structures */
          marker_select_buf[0][num_marker_selects] = startX;
          marker_select_buf[1][num_marker_selects] = startY;
          marker_select_buf[2][num_marker_selects] = marker_type;
          marker_select_buf[3][num_marker_selects] = npixels;
          marker_select_buf[4][num_marker_selects] = marker_fill;
          marker_select_buf[5][num_marker_selects] = endX;
          marker_select_buf[6][num_marker_selects] = endY;
          num_marker_selects = num_markers +1;

          /*
           * process output scatter field for marker_select
           */
          fill_field ( poly,
                       marker_select_buf,
                       num_marker_selects,
                       MAXY(input) );

          stat_pick_index = picked_marker;
          stat_pick_pid   = poly_in->data[
                              picked_marker * poly_in->veclen + 2 ];


  } /* end pick case */
  else
  /*
   * remove any previous pick hi-light
   * Nothing was picked, search failed
   */
  {
          num_marker_selects = num_markers;
          fill_field ( poly,
                       marker_select_buf,
                       num_marker_selects,
                       MAXY(input) );

          stat_pick_index = stat_pick_pid = -1;
          AVSmark_output_unchanged("Pick-Index");
          AVSmark_output_unchanged("Pick-PID");

  } /* end pick case */

  /*
   * Update the display Panel
   */
  display_info (marker_select_buf, picked_marker, stat_pick_pid);

  /*
   * Pass ints to output port
   */

   *pick_index = stat_pick_index;
   *pick_pid   = stat_pick_pid;

  /*
   * Good Return, process outputs
   */

  return(1);
}

/**********************************************************/
/* For CLI performance, buffer up CLI requests */
/**********************************************************/
add_command(command, clibuf, flush)
char        *command;
CLIbuffer *clibuf;
int       flush;
{
  int len;

  len = strlen(command) + 1;
  if ((CLI_BUFFER_SIZE - (clibuf->bptr - clibuf->buffer)) < len)
    flush_command_buffer(clibuf);

  strcpy(clibuf->bptr, command);
  clibuf->bptr += len;
  *(clibuf->bptr - 1) = '\n';

  if (flush)
    flush_command_buffer(clibuf);
} 

/**********************************************************/
/* For CLI performance, Flush out the CLI buffer  */
/**********************************************************/
flush_command_buffer(clibuf)
CLIbuffer *clibuf;
{
  char *dummy;

  if (clibuf->bptr != clibuf->buffer)
  {
    *(clibuf->bptr) = '\0';
    AVScommand("kernel", clibuf->buffer, &dummy, &dummy);
    clibuf->bptr = clibuf->buffer;
  }
}
/**********************************************************/
/* Allocate and fill the output marker polyline field     */
/**********************************************************/
fill_field( poly, marker_select_buf, num_marker_selects, y_size )

    AVSfield_int  **poly;      /* output marker_select field */
    int       marker_select_buf[2][MAX_POLYLINE];
    int       num_marker_selects;
    int       y_size;
{
    int dims[3];
    int i, veclen;
    int x, y, type, sz, fill, endx, endy;



    dims[0] = num_marker_selects;
    if ( *poly ) AVSfield_free( *poly );
    *poly = (AVSfield_int *) AVSdata_alloc( MARKER_FIELD_SPEC, dims);

    veclen = (*poly)->veclen;

    for ( i=0; i< dims[0]; i++ )
    {
        x    = marker_select_buf[0][i];
        y    = marker_select_buf[1][i];
        type = marker_select_buf[2][i];
        sz   = marker_select_buf[3][i];
        fill = marker_select_buf[4][i];
        endx = marker_select_buf[5][i];
        endy = marker_select_buf[6][i];

        (*poly)->data[i*veclen+0] = x;
        (*poly)->data[i*veclen+1] = y;
        (*poly)->data[i*veclen+2] = type;
        (*poly)->data[i*veclen+3] = sz;
        (*poly)->data[i*veclen+4] = fill;
        (*poly)->data[i*veclen+5] = endx;
        (*poly)->data[i*veclen+6] = endy;
    }

    AVSfield_set_labels (*poly,
     "X#Y#TYP#SZ#FILL#ENDX#ENDY", "#");
}

/**********************************************************/
/* Display Pick information in a Text Panel               */
/**********************************************************/

#define SCAT  strcat ( str, strt );

display_info (marker_select_buf, picked_marker, pid )
int       marker_select_buf[MARKER_VECLEN][MAX_POLYLINE];
int       picked_marker;
int       pid;
{
  int i;
  char str[500], strt[80];

  if (picked_marker >= 0)
  {
     sprintf(str, " === Selected Marker Information ===\n");
     sprintf(strt," Marker Index ID = %d\n", picked_marker); SCAT
     sprintf(strt," Marker PID Type = %d\n", pid          ); SCAT
     sprintf(strt," Start X = %d\n", marker_select_buf[0][picked_marker]); SCAT
     sprintf(strt," Start Y = %d\n", marker_select_buf[1][picked_marker]); SCAT
     sprintf(strt," Type    = %d\n", marker_select_buf[2][picked_marker]); SCAT
     sprintf(strt," Size    = %d\n", marker_select_buf[3][picked_marker]); SCAT
     sprintf(strt," Fill    = %d\n", marker_select_buf[4][picked_marker]); SCAT
     sprintf(strt," End X   = %d\n", marker_select_buf[5][picked_marker]); SCAT
     sprintf(strt," End Y   = %d\n", marker_select_buf[6][picked_marker]); SCAT
  }
  else
  {
     sprintf(str, " === Selected Marker Information ===\n");
     sprintf(strt,"No marker selected\n");         SCAT
  }
  AVSmodify_parameter("Selected Marker",AVS_VALUE,str,NULL,NULL);

}

/**********************************************************/
/* Init Module Entry Point for AVS kernel */
/**********************************************************/

#ifdef sep_exe
AVSinit_modules()
{
  AVSmodule_from_desc(marker_select_desc);
}
#endif


