/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/******************************************************************/
/*
 * UCD Node Contour
 *
 *   Create an array of floating point RGB values, one color for 
 *   each node in an unstructured cell data set.
 *   (Replacement module for ucd contour)
 *
 *   This module is a replacement for ucd-contour as supplied by AVS Inc.
 *   AVS Inc's ucd-contour module colors out-of-range data black,
 *   rather than clamping it to min and max values as the user often
 *   prefers.  This module is intended as a functional equivalent of
 *   ucd-contour, but which clamps out of range values to min and max 
 *   colors.
 *
 *   Author: Woody Muller, Becton Dickinson Research Center, RTP, NC
 *
 * Revision:
 * 6-July-1994   Original (wm)
 *
 *
/******************************************************************/

#include <stdio.h>
/* IAC CODE CHANGE : #include <avs_math.h> */
#include <avs/avs_math.h>
#include <avs/avs.h>
#include <avs/ucd_defs.h>
#include <avs/colormap.h>
#include <avs/field.h>
#include <avs/port.h>

static AVScolormap *s_cmap;
static char s_labels[UCD_LABEL_LEN];
static char s_delim[2];
static int s_dtype = 0;
static int *s_node_components;
static int s_num_node_components;
static int s_node_veclen;
static UCD_structure *s_ucd;
static int s_num_nodes;
static float *s_node_data;
static float s_rgbtable[256][3];
static int s_been_here = 0;
static int *s_offsets = NULL;
static float *s_mindy = NULL;
static float *s_maxie = NULL;

/********************************************************************/
/********************************************************************/

ucd_new_contour()
    {
    int ucd_new_contour_compute();
    int set_bw_colors();
    int parm;

    AVSset_module_name ("UCD Node Contour", MODULE_MAPPER);

    /* input ucd structure */
    AVScreate_input_port ("Input UCD", "ucd", REQUIRED);

    /* input port for the color map */
    AVScreate_input_port("Colormap", "colormap", OPTIONAL);

    parm = AVSadd_parameter("Node Data Type", "choice", "",
	"<data 1>.<data 2>.<data 3>.<data 4>.<data 5>.<data 6>.<data 7>",
	".");

    /* output port for the color vector */
    AVScreate_output_port ("Output Colors", "field 1D 3-vector float");

    AVSset_compute_proc(ucd_new_contour_compute);

    AVSset_init_proc(set_bw_colors);
    }

/********************************************************************/
/********************************************************************/

ucd_new_contour_compute(input_ucd, cmap, colorvec, node_dtype_string)
    UCD_structure *input_ucd;
    AVScolormap *cmap;
    AVSfield_float **colorvec;
    char *node_dtype_string;

    {
    AVSfield_float *field;
    int dims[1];
    char model_name[256];
    int data_veclen, name_flag, num_cells, cell_veclen;
    int util_flag;
    int jj, kk;
    float delta;
    int offset;
    float f_index, val;
    int i_index;

    if (*colorvec)
	AVSdata_free("field", *colorvec);

    s_cmap = cmap;
    s_ucd = input_ucd;	   

    if (AVSinput_changed("Colormap", 0))
	translate_colormap();

    if (AVSinput_changed("Input UCD", 0))
	{
	if (!UCDstructure_get_header (input_ucd, model_name, 
				&data_veclen, &name_flag, 
			        &num_cells, &cell_veclen, 
				&s_num_nodes, &s_node_veclen, 
			        &util_flag))
	    {
	    AVSerror("Error in UCD Node Contour: can't get UCD header.\n"); 
	    return (0);
	    }

	if (s_node_veclen == 0)
	    {
	    AVSerror("Error UCD Node Contour: no node data\n"); 
	    return (0);
	    }

	if (!UCDstructure_get_node_data(input_ucd, &s_node_data))
	    {
	    AVSerror ("UCD Node Contour: can't get node data.\n"); 
	    return(0);
	    }

	if (s_maxie)
	    FREE_LOCAL(s_maxie);
	if (s_mindy)
	    FREE_LOCAL(s_mindy);
	s_mindy = (float *)ALLOC_LOCAL(s_node_veclen * sizeof(float));
	s_maxie = (float *)ALLOC_LOCAL(s_node_veclen * sizeof(float));

	if (!UCDstructure_get_node_minmax(input_ucd, s_mindy, s_maxie))
	    {
	    AVSerror ("UCD Node Contour: can't get node minmax.\n"); 
	    return(0);
	    }

	if (!UCDstructure_get_node_components(input_ucd, &s_node_components))
	    {
	    AVSerror ("UCD Node Contour: can't get node components.\n"); 
	    return(0);
	    }

	count_node_components();
	figure_offsets();

	make_data_choice();

	}

    offset = 0;
    if (AVSparameter_changed("Node Data Type"))
	{
	char *mycopy, *ptr, *strstr();
	int kk;

	mycopy = ALLOC_LOCAL(strlen(s_labels) + 1);
	strcpy(mycopy, s_labels);
	ptr = strstr(mycopy, node_dtype_string);
	ptr[0] = '\0';

	s_dtype = 0;
	for (kk = 0; kk < strlen(mycopy); kk++)
	    if (mycopy[kk] == s_delim[0])
		s_dtype++;
	FREE_LOCAL(mycopy);
	
	offset = s_offsets[s_dtype] * s_num_nodes;
	}

    dims[0] = s_num_nodes;
    field = (AVSfield_float *)AVSdata_alloc("field 1D 3-vector float", dims);

    delta = s_maxie[s_dtype] - s_mindy[s_dtype];

    if (delta < .0000001)
        for (kk = 0; kk < 3 * s_num_nodes; kk++)
	    field->data[kk] = 1.0;
    else
	{
	for (jj = 0; jj < s_num_nodes; jj++)
	    {
	    if (s_node_data[jj] < s_mindy[s_dtype])
		{
		field->data[jj * 3] = s_rgbtable[0][0];
		field->data[jj * 3 + 1] = s_rgbtable[0][1];
		field->data[jj * 3 + 2] = s_rgbtable[0][2];
		}
	    else
	    if (s_node_data[jj] > s_maxie[s_dtype])
		{
		field->data[jj * 3] = s_rgbtable[255][0];
		field->data[jj * 3 + 1] = s_rgbtable[255][1];
		field->data[jj * 3 + 2] = s_rgbtable[255][2];
		}
	    else
		{
		val = (s_node_data[offset + jj] - s_mindy[s_dtype]) / delta;
		f_index = val * 255.9999;
		i_index = f_index;
		field->data[jj * 3] = s_rgbtable[i_index][0];
		field->data[jj * 3 + 1] = s_rgbtable[i_index][1];
		field->data[jj * 3 + 2] = s_rgbtable[i_index][2];
		}
	    }
	}

    *colorvec = field;

    return(1);
    }

/********************************************************************/
/********************************************************************/

AVSinit_modules()
    {
    AVSmodule_from_desc(ucd_new_contour);
    }

/********************************************************************/
/*************************************************************/

/* this conversion routine is from page 616 of Foley and Van Dam */

hsv_to_rgb(h, s, v, col)
    float h, s, v;
    float *col;

    {
    int i;
    float f, p, q, t;

    if (h >= 1.0) h = h - 1.0;
    h = 360.0 * h;

    if (s < 0.000001)
	{
	col[0] = col[1] = col[2] = v;
	}
    else
	{
	h = h / 60;
	i = h;
	f = h - i;
	p = v * (1 - s);
	q = v * (1 - (s * f));
	t = v * (1 - (s * (1 - f)));

	switch (i)
	    {
	    case 0:
		col[0] = v;
		col[1] = t;
		col[2] = p;
		break;
	    case 1:
		col[0] = q;
		col[1] = v;
		col[2] = p;
		break;
	    case 2:
		col[0] = p;
		col[1] = v;
		col[2] = t;
		break;
	    case 3:
		col[0] = p;
		col[1] = q;
		col[2] = v;
		break;
	    case 4:
		col[0] = t;
		col[1] = p;
		col[2] = v;
		break;
	    case 5:
		col[0] = v;
		col[1] = p;
		col[2] = q;
		break;
	    }
	}
    }    

/*************************************************************/
/*************************************************************/

count_node_components()
    {
    int kk;

    kk = 0;
    s_num_node_components = 0;
    while (kk < s_node_veclen)
	{
	kk += s_node_components[s_num_node_components];
	s_num_node_components++;
	}
    }

/*************************************************************/
/*************************************************************/

figure_offsets()
    {
    int kk, offset;

    if (s_offsets)
	FREE_LOCAL(s_offsets);
    s_offsets = (int *)ALLOC_LOCAL(s_node_veclen * sizeof(int));

    offset = 0;
    for (kk = 0; kk < s_num_node_components; kk++)
	{
	s_offsets[kk] = offset;
	offset += s_node_components[kk];
	}
    }

/*************************************************************/
/*************************************************************/

make_data_choice()
    {
    char *labels2, *labels3, *ptr, *strchr(), *first;
    int kk;
    int first_one;

    if (!UCDstructure_get_node_labels(s_ucd, s_labels, s_delim))
	{
	AVSerror ("UCD Node Contour: can't get node labels.\n"); 
	return(0);
	}

    labels2 = ALLOC_LOCAL(strlen(s_labels) + 1);
    strcpy(labels2, "");
    labels3 = ALLOC_LOCAL(strlen(s_labels) + 1);
    first = ALLOC_LOCAL(strlen(s_labels) + 1);

    first_one = 1;

    for (kk = 0; kk < s_num_node_components; kk++)
	{
	if (s_node_components[kk] == 1)
	    {
	    if (first_one)
		{
		UCDstructure_get_node_label(s_ucd, kk, first);
		first_one = 0;
		}
	    else
		strcat(labels2, s_delim);
	    UCDstructure_get_node_label(s_ucd, kk, labels3);
	    strcat(labels2, labels3);
	    }
	}
    AVSmodify_parameter("Node Data Type", 
			AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
			first, labels2, s_delim);			
    FREE_LOCAL(labels2);
    FREE_LOCAL(labels3);
    FREE_LOCAL(first);
    }

/******************************************************************/
/******************************************************************/

set_bw_colors()
    {
    float ff, kerchunk;
    int kk;
    
    kerchunk = 1. / 256.;
    ff = 0.0;
    for (kk = 0; kk < 256; kk++)
	{
	ff += kerchunk;
	s_rgbtable[kk][0] = ff;
	s_rgbtable[kk][1] = ff;
	s_rgbtable[kk][2] = ff;
	}
    }

/******************************************************************/
/******************************************************************/

translate_colormap()
    {
    int kk;
    
    if (s_cmap)
	for (kk = 0; kk < 256; kk++)
	    hsv_to_rgb(s_cmap->hue[kk], s_cmap->saturation[kk],
	    		s_cmap->value[kk], &s_rgbtable[kk][0]);
    else
	set_bw_colors();
    }

/******************************************************************/
/******************************************************************/

lookup_rgb(val, rgb)
    float val;
    float *rgb;

    {
    float f_index;
    int i_index;

    f_index = val * 255.9999;
    i_index = f_index;

    }

/******************************************************************/

