/*****************************************************************************
*
*                         NCSA DTM version 2.0
*                               June 10, 1991
*
* NCSA DTM Version 2.0 source code and documentation are in the public
* domain.  Specifically, we give to the public domain all rights for future
* licensing of the source code, all resale rights, and all publishing rights.
*
* We ask, but do not require, that the following message be included in all
* derived works:
*
* Portions developed at the National Center for Supercomputing Applications at
* the University of Illinois at Urbana-Champaign.
*
* THE UNIVERSITY OF ILLINOIS GIVES NO WARRANTY, EXPRESSED OR IMPLIED, FOR THE
* SOFTWARE AND/OR DOCUMENTATION PROVIDED, INCLUDING, WITHOUT LIMITATION,
* WARRANTY OF MERCHANTABILITY AND WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE
*
*****************************************************************************/

/***************************************************************************
**
** dtm.c	Contains the user routines for the DTM library.  These
**		routines provide inter-machine message passing for IPC.
**		Some of the features include:
**		  - automatic type conversion
**		  - synchronous and asynchronous operation
**		  - flexible message format, which is user configurable
**		  - high-level message types (SDS, RIS, SDL)
**
***************************************************************************/



/*
 * $Log: dtm.c,v $
 * Revision 1.9  1991/09/13  17:34:10  sreedhar
 * changes for DTMSYNC, DTMNOSYNC quality of service
 *
 * Revision 1.7  1991/08/15  18:57:06  sreedhar
 * Changes for logical portname version
 *
 * Revision 1.5  1991/06/11  15:19:04  sreedhar
 * WRITEMSG option - sequence start, end of message also put in same writev call.
 *
 * Revision 1.4  1991/06/07  15:58:10  sreedhar
 * Changes for "Sequence start" message
 *
 * Revision 1.3  1991/05/30  15:51:31  sreedhar
 * Changes for readMsg/writeMsg internal release
 *
 * Revision 1.2  1990/12/19  17:28:50  jefft
 * DTMerrno is now cleared at the beginning of each DTM routine.  This
 * allows application to continue when possible.
 *
 * Revision 1.1  90/11/08  16:12:20  jefft
 * Initial revision
 * 
 */

/*	+++++ System call - other points +++++ 
	
	i) Select with multiple fds.
	ii)iov buffer in dtm_writev_buffer.
*/
/*
	Overview of DTM message transfer protocol.

	1. TCP connection is established between sender and receiver.

	2. Sender sends the "sequence starts" message.

	3. Sender awaits "ack from seq start" from the receiver.
	   Receipt of ack by sender guarantees the sender that
	   receiver will definitely accept atleast the first user message 
	   sent by the sender.  Sender can then send as many user messages
	   as it wants to ( they would be accepted by receiver provided no
	   new connection request bumps current one ).  

	4. Sender sends the user's header message and user data messages. 

	3. Receiver will keep accepting user messages on current 
	   connection unless a new connection request is received, which
	   would be accepted after bumping the current connection.

	4. Sender would send "Messages over" message after it sends all user
	   messages.  Receiver would accept same.

	Graphic picture ( assumme no bumping ) -

	Sender			Receiver

		Connect request			
		-------------->			

		Sequence starts		|
		-------------->		|
					|
		Ack for seq start	|
		<----------------	|
					|	
		User header		|	
		-------------->		|
					|	--> a sequence of
					|	    BEGINWRITE,
					|	    WRITEDATASET,
		User message 1		|	    ENDWRITE
		-------------->		|
		...............		|	    or equivalently,
		...............		|	    WRITEMSG
					|
		User message n		|
		-------------->		|
					|
		Messages over		|
		-------------->		|
					|

		...............			More sequences as above
		...............


	A "sequence starts" message can be sent in availWrite or 
	beginWrite. 

	A connection is broken only when -

	1. Receiver 'bump's it by accepting some other new connection.
	2. When no "Ack for header" is received or a write fails.

	Note that the "ack for header", "message over" and "sequence starts"
	messages are called called "ack" in DTM terminology ( send_ack, 
	recv_ack calls used for all these ).
*/


#include	<stdio.h>
#include	<string.h>
#include	<sys/types.h>
#include	<sys/uio.h>
#include	<sys/time.h>
#include	<netinet/in.h>

#ifdef	RS6000
#include	<sys/select.h>
#endif

#ifdef	macintosh
#include	"DTMMac.h"
#include	<MacSockDefs.h>
#include	<socket.ext.h>
#endif

#include	"dtm.h"
#include	"dtmint.h"
#include	"debug.h"

DTMPORT		*DTMpt[MAX_DTM_PORTS];	/* DTM port table */
char		*dtm_discard;

/*
	Function to write -
								CALLTYPE

	1. complete user header/data and end the write 		WRITEMSG
	2. user header 						BEGINWRITE
	3. user data						WRITEDATASET
	4. end of write						ENDWRITE
	 
	Return values	: DTMERROR	on error
			  DTMNOERR	on success.

	Notes	:	This is an all purpose function. It is invoked
			to do a composite write ( of header/data ),
			to begin a write, to write a message as well
			as to end a write.

			Somewhat clumsy function, but centralises all
			activities related to a write - very useful
			when source enhancements are made to DTM library.

			In general, it should be obvious which sections
			of the code apply to which calltype without 
			really looking at the 'if' expression.

			e.g. "Check for client initialisation" section 
			     is only relevant for BEGINWRITE and WRITEMSG 
			     calltypes. 
*/

static	int	intWriteMsg( p, hdr, hdrsize, data, datasize, datatype, calltype )
int	p ;
char	*hdr ;
int	hdrsize ;
VOIDPTR	data ;
int	datasize ;
DTMTYPE	datatype ;
register CALLTYPE	calltype ;
{
register DTMPORT	*pp;
register Outport	*pcur ;
Outport	 *pend ;
int	err_count ;

  	DBGINT( "intWriteMsg called, type = %d\n", calltype );
  	DTMerrno = DTMNOERR;

  	/* 	Check for port initialization 	*/

  	if( ( pp = DTMpt[p] ) == NULL )
	{
    		DTMerrno = DTMPORTINIT;
    		DTMERR( "intWriteMsg: port has not been initialized.");
    		return DTMERROR;
    	}

	/*	In BEGINWRITE, acquire list of associated output ports	*/

	if( (calltype == BEGINWRITE) || (calltype == WRITEMSG) )
	{
#ifdef	DEBUGMSG
		if( (pp -> count = dtm_nrecv_list( pp -> sockfd, &pp -> out, pp -> count, pp -> portname )) == 0 )
#else
		if( (pp -> count = dtm_nrecv_list( pp -> sockfd, &pp -> out, pp -> count )) == 0 )
#endif
		{
			DTMerrno = DTMPORTINIT ;
    			DBGFLOW( "intWriteMsg: no out port list.");
    			return DTMERROR;
		}
		else
		{
			DBGINT( "intWriteMsg: %d TCP ports with the outport\n",
				pp -> count );	
		}
	}

	/*	For all ports in list, do relevant operations	*/

	pcur = pp -> out - 1 ;
	pend = pcur + pp -> count ;
	err_count = 0 ;

	while( ++pcur <= pend )
	{

	DBGINT( "intWriteMsg: pcur -> sockaddr.sin_addr.s_addr = %x\n",
			pcur -> sockaddr.sin_addr.s_addr );
	DBGINT( "intWriteMsg: pcur -> sockaddr.sin_port = %d\n",
			pcur -> sockaddr.sin_port );

	/* 	Check for client initialization 	*/

	if( (calltype == BEGINWRITE) || (calltype == WRITEMSG) ) 
	{
		if( pcur -> connfd == -1 )
		{
			if( dtm_connect( &pcur -> sockaddr, &pcur -> connfd ) == -1 )
			{
				DBGFLOW( "intWriteMsg: dtm_connect fails \n" );
				++err_count ;
				continue ; 
			}
		} 
	}

	/*	Check whether any connection exists		*/

	if( (calltype == WRITEDATASET) || (calltype == ENDWRITE) )
	{
		if( pcur -> connfd == -1 )
		{
			DBGFLOW( "intWriteMsg: no connection exists\n" );
			++err_count ;
			continue ;
		}
	}

  	/* 	Convert user data to standard format 	*/

	if((calltype == WRITEDATASET) || (calltype == WRITEMSG))
	{
  		datasize = (*DTMconvertRtns[(int)datatype])( DTMSTD, data, datasize );
	}

  	/* 
		Write appropriately -

		Calltype		Write

		BEGINWRITE		. sequence starts ( if needed )
					. size of header 
					. header

		WRITEDATASET		. size of user data 
					. user data

		ENDWRITE		nothing (send "message(s) over" ack)
		
		WRITEMSG		. sequence starts ( if needed )
					. size of header
					. header
					. size of user data
					. user data

		In BEGINWRITE & WRITEMSG, additionally accept ack
		for sequence starts.  The ack assures sender that 
		atleast one user message will be accepted before a new 
		connection is allowed to bump this one.
	*/

	{
	struct	iovec	iov[ 6 ] ;
	int	htmp ;
	int	dtmp ;
	int	atmp ;
	int	etmp ;
	int	status ;
	int	iovlen ;
	int	iovsize ;
	int	i ;

		iovsize = 0 ;
		iovlen = 0 ;

		/*	Send "Sequence starts" message	*/

		i = 0 ;
		if( (calltype == BEGINWRITE) || (calltype == WRITEMSG) )
		{
			if( !( pcur -> availwrite ) )
			{
				if( !( pcur -> seqstart ) )
				{
					atmp = 0 ;	
					STDINT( atmp );

					iov[ i ].iov_base = (char *)&atmp ;
					iov[ i ].iov_len = 4 ;
					i += 1 ;

					iovsize += 4 ;
					iovlen += 1 ;
				
					pcur -> seqstart = TRUE ;
				}
			}

			/*	Prepare header size and header	*/

			htmp = hdrsize ;
			STDINT( htmp );
		
			iov[ i ].iov_base = (char *)&htmp ;
			iov[ i ].iov_len = 4 ;
			i += 1 ;
			iov[ i ].iov_base = hdr ;
			iov[ i ].iov_len = hdrsize ;
			i += 1 ;

			iovsize += 4 + hdrsize ;
			iovlen += 2 ;
		}

		/* 	Prepare data size and data	*/

		if( (calltype == WRITEDATASET) || (calltype == WRITEMSG) )
		{
			dtmp = datasize ;
			STDINT( dtmp );

			iov[ i ].iov_base = (char *)&dtmp ;
			iov[ i ].iov_len = 4 ;
			i += 1 ;
			iov[ i ].iov_base = (char *)data ;
			iov[ i ].iov_len = datasize ;
			i += 1 ;

			iovsize += 4 + datasize ;
			iovlen += 2 ;
		}

		if( (calltype == ENDWRITE) || (calltype == WRITEMSG) )
		{
			etmp = 0 ;
			STDINT( etmp );

			iov[ i ].iov_base = (char *)&etmp ; 
			iov[ i ].iov_len = 4 ; 
			i += 1 ;
			
			iovsize += 4 ;
			iovlen += 1 ;
		}

		DBGINT( "Iovlen = %d\n", iovlen );
		DBGINT( "Before write pcur -> connfd = %d\n", pcur -> connfd );
		status = dtm_writev_buffer( pcur -> connfd, iov, iovlen, 
				iovsize, NULL, 0 );
		DBGINT( "dtm_writev_buffer - status = %d\n", status); 
		if( status < 0 )
		{		
			DBGINT( "dtm_writev_buffer - errno = %d\n", errno );
			if( DTMerrno == DTMEOF )
			{
				dtm_end_connect( pcur -> connfd );
				pcur -> seqstart = FALSE ;
				pcur -> availwrite = FALSE ;
				pcur -> connfd = -1 ;
			}
			++err_count ;
			continue ;
		}

		/*	For WRITEDATASET, just return status of write	*/

		if( calltype == WRITEDATASET )
		{
			continue ;
		}
	}

	/*	Await ack for beginwrite and writemsg if availwrite is false	*/

	if( (calltype == BEGINWRITE) || (calltype == WRITEMSG) )
	{
		if( !( pcur -> availwrite ) )
		{
		int	tmp ;

			if( (pp -> qservice == DTMSYNC) || 
			    ( (pp -> qservice == DTMNOSYNC) && 
			      (dtm_select( pcur -> connfd, &tmp, 0 ) == TRUE )
			    )
			)
			{
  				if( dtm_recv_ack( pcur -> connfd, &tmp ) == -1 )
				{
					DBGFLOW( "Incorrect ack for header\n" );
					dtm_end_connect( pcur -> connfd ) ;
					pcur -> seqstart = FALSE ;
					pcur -> availwrite = FALSE ;
					pcur -> connfd = -1 ;
					++err_count ;
					continue ;
				}	
			}

			pcur -> seqstart = FALSE ;
			pcur -> availwrite = TRUE ;
		}
	}

	if( (calltype == ENDWRITE) || (calltype == WRITEMSG) )
	{
		pcur -> seqstart = FALSE ;
		pcur -> availwrite = FALSE ;
	}

	}	/* end while	*/

	return ( err_count == pp -> count ) ? DTMERROR : DTMNOERR ;
}

/*
	Function to test 

	a) for existence of a new connection or a new 
	   message header to be read on a set of DTM ports OR
	b) for whether data is available to be read on a 
	   set of sockets.

	Return	values	:	TRUE		if atleast a DTM port or
						socket has something to be
						read.
				FALSE		if no DTM port or socket
						has something to be read.
				DTMERROR	if select system call returns
						error.

				Each port has status field. Possible values
				for status field are - 

				TRUE		something available to be read.
				FALSE		nothing available to be read.
				DTMERROR	port not initialised.
*/

int	DTMselectRead( dtmset, dtmnum, sockset, socknum, period )
Dtm_set	*dtmset ;		/* Pointer to set of DTM ports 	*/
int	dtmnum ;		/* Number of DTM ports to be checked for */
Sock_set	*sockset ;	/* Pointer to set of sockets	*/
int	socknum ;		/* Number of sockets to be checked for	*/
int	period ;
{
fd_set	readmask ;
fd_set	*fchk = &readmask ;
int	nf ;
int	index ;
Dtm_set		*p1 ;
Sock_set	*p2 ;
struct	timeval	timeout ;

	DBGFLOW( "DTMselectRead called\n" );
	timeout.tv_sec = period/1000 ;
	timeout.tv_usec = (period - (period/1000)*1000)*1000 ;

	FD_ZERO( fchk );

	/*	Set up DTM ports to be selected on	*/

	for( p1 = dtmset, index = 0 ; index < dtmnum ; index++, p1++ )
	{
	register	DTMPORT	*pp ; 
	
		/* 
			Select status is error if port entry is NULL or
			port is not initialised. 
		*/

		if( (pp = DTMpt[ p1 -> port ]) == NULL )
		{
			p1 -> status = DTMERROR ;
			continue ;
		}

		/* look for new connection request */

		FD_SET( pp -> sockfd, fchk );     	

		/* look for data in existing connection (if it exists) */

		if( pp -> connfd != -1 )
		{

			FD_SET( pp -> connfd, fchk );	
		}
		p1 -> status = FALSE ;
	}

	/*	Set up socket fds to be selected on	*/

	for( p2 = sockset, index = 0 ; index < socknum ; index++, p2++ )
	{
		FD_SET( p2 -> sockfd, fchk );
		p2 -> status = FALSE ;
	}

	nf = select( FD_SETSIZE, fchk, (fd_set *)0, (fd_set *)0, &timeout );	

	/* Select returns error	*/

	if( nf < 0 )
	{
	extern	int	errno ;

		DBGINT( "DTMselectRead: select error %d \n", errno  );
		DTMerrno = DTMSELECT ;
		return DTMERROR ;
	}

	/* None of the DTM ports or sockets have anything to be read	*/

	if( nf == 0 )
	{
		DBGFLOW( "DTMselectRead: Nothing to read\n" );
		return FALSE ;
	} 

	/*	Check whether any DTM port has something to be read */

	for( p1 = dtmset, index = 0 ; index < dtmnum ; index++, p1++ )
	{
	register	DTMPORT	*pp ;

		if( (pp = DTMpt[ p1 -> port ]) != NULL )
		{
			p1 -> status = 	FD_ISSET( pp -> sockfd, fchk ) ? TRUE : FALSE ; 
			if( pp -> connfd != -1 )
			{
				p1 -> status = p1 -> status || ( FD_ISSET( pp -> connfd, fchk ) ? TRUE : FALSE ) ;
			}
			
		}
	}

	/*	Check whether any socket has something to be read */

	for( p2 = sockset, index = 0 ; index < socknum ; index++, p2++ )
	{
		p2 -> status = FD_ISSET( p2 -> sockfd, fchk ) ? TRUE : FALSE ;
	}

	return TRUE ;
}

/*
	Function to test for existence of a new connection or
	a new message header to be read on a given DTM port.

	Return	values	:	TRUE		if either new connection 
						request or something new 
						to be read is available on
						given port.
				DTMERROR	on select error.	
				FALSE		otherwise.

	Notes	:

		DTMavailRead is basically call to DTMselectRead for 
		given port with 0 timeout.
*/

int	DTMavailRead( p )
int	p ;
{
Dtm_set	dtmset ;
int	dtmnum ;
Sock_set	sockset ;
int	socknum ;

	DBGFLOW( "DTMavailRead called\n" );
	dtmnum = 1 ;
	dtmset.port = p ;

	socknum = 0 ;
 
	return DTMselectRead( &dtmset, dtmnum, &sockset, socknum, 0);
}

/*
	Function to begin reading on a DTM port. The header of the
	message is read.

	Return	values	:	>= 0 		on success.
				DTMERROR	on some error.	
*/

int	DTMbeginRead( p, header, size )
int	p ;
int	size ;
char	*header ;
{
int	count;
register DTMPORT	*pp;

  	DBGFLOW( "DTMbeginRead called.\n" );
  	DTMerrno = DTMNOERR;

  	/* check for port initialization */

  	if( (pp = DTMpt[p]) == NULL )
	{
    		DTMERR("DTMbeginRead: port has not been initialized.");
    		DTMerrno = DTMPORTINIT;
    		return DTMERROR;
    	}

 	/* 
		If no connection exists, await and accept a connection.
		If a connection exists, await either -

		i)  A new connection ( and accept it ) or
		ii) a write on existing connection 	

		On accepting a connection, send an ack.
	*/

	if( pp -> connfd == -1 )
	{
	struct	timeval	timeout ;

		/* Wait only 2 minutes for the first connection request */

		timeout.tv_sec = 120 ;
		timeout.tv_usec = 0 ;

		/* No connection yet, await one and accept */

		DBGFLOW( "DTMbeginRead: First connection\n" );

		DBGINT( "DTMbeginRead: pp -> sockfd = %d\n",
			pp -> sockfd );
    		if( (pp -> connfd = dtm_accept( pp->sockfd, &pp->sockaddr, &timeout )) == -1 )
		{
			DTMerrno = DTMTIMEOUT ;
      			return DTMERROR;
		}
    	}
  	else 
	{
		/*  
		    At this point, await either 

		    i)  a new connection request on pp -> sockfd or
		    ii) a write on the existing connection,i.e., pp -> connfd.

			+++++ System call - call indirectly +++++
		*/

		{
		fd_set	readmask ;
		fd_set	*fchk = &readmask ;
		int	nf ;

			FD_ZERO( fchk );

			FD_SET( pp -> sockfd, fchk );
			FD_SET( pp -> connfd, fchk );

			DBGFLOW( "Awaiting new connection/header\n" );
			nf = select( FD_SETSIZE, fchk, (fd_set *)0, (fd_set *)0, NULL );	
			DBGINT( "Select returns %d\n", nf );

			if( nf < 0 )
			{
			extern	int	errno ;

				/* select returns error */

				DBGINT( "Select error %d\n", errno );
				DTMerrno = DTMSELECT ;
				return DTMERROR ;	
			}
			
			if( FD_ISSET( pp -> sockfd, fchk ) )
			{
				/* 
				   New connection request - end old one,
				   accept new one, send connect ack.
				*/
    			
				DBGFLOW( "New connection request\n" );
				DBGINT( "Ending %d\n", pp -> connfd);
				dtm_end_connect( pp->connfd );	

				/* Note NULL passed for timeout, since connection definitelt exists */
				if( (pp->connfd = dtm_accept( pp->sockfd, &pp->sockaddr, 0 )) == -1 )
				{
      					return DTMERROR;
				}
				DBGINT( "New connection is %d\n", pp -> connfd );
			}			
			else
			{
				if( FD_ISSET( pp -> connfd, fchk ) )
				{
					/* "Sequence starts" available for reading on old connection */

					DBGINT( "'Sequence starts' on existing connection %d\n", pp -> connfd );
				}

			}
		}
	}

	/* await/accept "Sequence starts" message	*/

	DBGINT( "Accepting 'sequence starts' on %d\n", pp -> connfd );
	{
	int	ack ;

		if( dtm_recv_ack( pp -> connfd, &ack ) == -1 )
		{
			pp -> connfd = -1 ;
			return DTMERROR ;
		}  
		else
		{
			if( ack != 0 )
			{
				DBGFLOW( "Something other than 'sequence start' received\n" );
				return DTMERROR ;
			}

			pp->blocklen = -1;
    			dtm_send_ack( pp ->connfd, 0 );
		}
	}

	/* await/accept header */

	DBGINT( "Accepting header on %d\n", pp -> connfd ) ;
  	if( (count = dtm_recv_header( pp-> connfd, header, size )) < 0 )
	{
		DBGINT( "Recv header error = %d\n", errno );
    		if( DTMerrno != DTMHEADER )
		{
      			pp->connfd = -1;
		}
	}

  	return count;
}

/*
	Function to read user messages. 

	Return	values	: 	number of bytes read,	on success
				DTMERROR		on error
*/

int DTMreadDataset(p, ds, size, type)
int	p ;
int	size ;
VOIDPTR	ds;
DTMTYPE	type;
{
  	DBGFLOW("DTMreadDataset called.\n");
  	DTMerrno = DTMNOERR;

  	/* check if DTMbeginRead has been called */

  	if( DTMpt[p]->connfd == -1 )
	{
    		DTMerrno = DTMCALL;
    		DTMERR("DTMreadDataset: DTMbeginRead required before DTMreadDataset.");
    		return DTMERROR;
    	}

  	/* determine max number of bytes that can be read */

	size = (*DTMconvertRtns[(int)type])(DTMSTD, NULL, size);

  	/* fill buffer from network */

	size = dtm_read_buffer(DTMpt[p]->connfd, &DTMpt[p]->blocklen, ds, size);

  	/* convert dataset to local representation */

  	return (*DTMconvertRtns[(int)type])(DTMLOCAL, ds, size);
}

/*
	Function to end reading user messages.

	Return	values	:	0		on no error
				DTMERROR	on error
*/

int 	DTMendRead( p )
int	p ;
{

  	DBGFLOW("DTMendRead called.\n");
  	DTMerrno = DTMNOERR;

  	/* check that DTMbeginRead has been called */

  	if( DTMpt[p]->connfd == -1 )
	{
    		DTMerrno = DTMCALL;
    		DTMERR("DTMend called without DTMbeginRead.");
    		return DTMERROR;
    	}

  	/* discard any remaining data */

  	while( dtm_read_buffer(DTMpt[p]->connfd, &DTMpt[p]->blocklen,
      		dtm_discard, DISCARDSIZE) > 0 ) ;

  	return 0 ; 
}

/*
	Function to combine reading of header/data.
	
	Return	values	:	number of bytes read,	on success
				DTMERROR		on error	

	Notes	: This function is really there for completeness
		  sake ( it complements DTMwriteMsg ). It is not
		  very clear how a user can use it effectively
		  ( since he has to know data size beforehand,
		    in which case he need not have a header ).
		
		  Hence, implementation of this function is to 
		  just call beginRead, readDataset and endRead
		  in that order.
*/

int	DTMreadMsg( p, hdr, hdrsize, data, datasize, datatype )
int	p ;
char	*hdr ;
int	hdrsize ;
VOIDPTR	*data ;
int	datasize ;
int	datatype ;
{
int	count ;

	DTMerrno = DTMNOERR ;

	if( DTMbeginRead( p, hdr, hdrsize ) == DTMERROR )
	{ 
		return DTMERROR ;
	}

	if( (count = DTMreadDataset( p, data, datasize, datatype )) == DTMERROR )
	{
		return DTMERROR ;
	} 

	if( DTMendRead( p ) == DTMERROR )
	{
		return DTMERROR ;
	}

	return count ;
}

/*
	Function to test whether a subsequent beginWrite( or writeMsg )
	would (definitely) succeed or not.

	Return	values	: 	TRUE		if subsequent write will
						succeed.
				DTMERROR	if port is not initialised or
						server ( UDP/TCP ) port not
						yet acquired etc.
*/

int	DTMavailWrite( p )
int	p ;
{
register DTMPORT	*pp;
int	rstatus ;
register Outport	*pcur ;
Outport	*pend ;
int	err_count ;

  	DBGFLOW( "DTMavailWrite called.\n" );
  	DTMerrno = DTMNOERR;

  	/* check for port initialization */

  	if( (pp = DTMpt[p]) == NULL )
	{
    		DTMerrno = DTMPORTINIT;
    		DTMERR("DTMavailWrite: port has not been intialized.");
    		return DTMERROR;
    	}

	/*	Acquire list of associated output ports	*/

#ifdef	DEBUGMSG
	if( (pp -> count = dtm_nrecv_list( pp -> sockfd, &pp -> out, pp -> count, pp -> portname )) == 0 )
#else
	if( (pp -> count = dtm_nrecv_list( pp -> sockfd, &pp -> out, pp -> count )) == 0 )
#endif
	{
		DTMerrno = DTMPORTINIT ;
    		DTMERR( "DTMavailWrite: no out port list.");
    		return DTMERROR;
	}

	/*	For all ports in list, do relevant operations	*/

	pcur = pp -> out - 1 ;
	pend = pcur + pp -> count ;
	err_count = 0 ;

	rstatus = TRUE ;
	while( ++pcur <= pend )
	{

		/* 	Check for client initialization 	*/

		if( pcur -> connfd == -1 )
		{
			if( dtm_quick_connect( &pcur -> sockaddr, &pcur -> connfd ) == -1 )
			{
				++err_count ;
				continue ;
			}
		}

		DBGINT( "DTMavailWrite: availWrite = %d\n", pcur -> availwrite );
		DBGINT( "DTMavailWrite: seqstart = %d\n", pcur -> seqstart );
		if( !(pcur -> availwrite) )
		{
			if( !(pcur -> seqstart) )
			{
				/* Send "Sequence starts" message. */
	
				if( dtm_send_ack( pcur -> connfd, 0 ) == -1 )
				{
					if( DTMerrno == DTMEOF )
					{
						dtm_end_connect( pcur -> connfd );
					 	pcur -> connfd = -1 ;
					}
					++err_count ;
					continue ;
				}
				pcur -> seqstart = TRUE ;
			}
	
			/* Look for "ack for seq start" */

			/*   +++++ System call - call indirectly +++++ quick_select */ 

			{
			fd_set	readmask ;
			fd_set	*fchk = &readmask ;
			struct	timeval	timeout ;
			int	ack ;
			int	nf ;
			int	status ;
		
				FD_ZERO( fchk );

				FD_SET( pcur -> connfd, fchk );
				timeout.tv_sec = 0 ;
				timeout.tv_usec = 0 ;

				nf = select( FD_SETSIZE, fchk, (fd_set *)0, (fd_set *)0, 
							&timeout );	

				if( nf < 0 )
				{
				extern	int	errno ;

					dtm_end_connect( pcur -> connfd );
					pcur -> seqstart = FALSE ;
					pcur -> availwrite = FALSE ;
					pcur -> connfd = -1 ;
					DTMerrno = DTMSELECT ;
					++err_count ;
					continue ;
				}

				/* No ack yet	*/

				if( nf == 0 ) 
				{
					if( pp -> qservice == DTMNOSYNC )
					{
						pcur -> seqstart = FALSE ;
						pcur -> availwrite = TRUE ;
					}
					else
					{
						rstatus = FALSE ;
					}
					continue ;
				}

				/* Discard any queued acks */

  				if( (status = dtm_recv_ack( pcur -> connfd, &ack )) == -1 )
				{
					DBGFLOW( "Incorrect ack for header\n" );
					dtm_end_connect( pcur -> connfd ) ;
					pcur -> seqstart = FALSE ;
					pcur -> availwrite = FALSE ;
					pcur -> connfd = -1 ;
					++err_count ;
					continue ;
				}	

				/* port is available for write */

				pcur -> seqstart = FALSE ;
				pcur -> availwrite = TRUE ;
			}
		}
	}	/* end while */

	return ( err_count == pp -> count ) ? DTMERROR : rstatus ;
}

/*
	Function to write user's header.

	Return	values	: same as intWriteMsg().
*/

int	DTMbeginWrite( p, header, size )
int	p ; 
int	size ;
char	*header;
{
	return	intWriteMsg( p, header, size, 0, 0, 0, BEGINWRITE );
}

/*
	Function to write user's data.

	Return	values	: same as intWriteMsg().
*/

int DTMwriteDataset(p, ds, size, type)
int	p, size;
VOIDPTR	ds;
DTMTYPE	type;
{
	return	intWriteMsg( p, 0, 0, ds, size, type, WRITEDATASET ); 
}

/*
	Function to end user's write.

	Return	values	: same as intWriteMsg().
*/

int 	DTMendWrite( p )
int	p;
{
	intWriteMsg( p, 0, 0, 0, 0, 0, ENDWRITE );
}

int	DTMsizeof( type )
DTMTYPE	type;
{
int	size = 1;

  	return (*DTMconvertRtns[(int)type])(DTMSTD, NULL, size);
}

/*
	Function to do a complete/composite write in which
	the user header/data is written and the write ended.

	Return	values	: same as intWriteMsg().
*/

int	DTMwriteMsg( p, hdr, hdrsize, data, datasize, datatype )
int	p ;
char	*hdr ;
int	hdrsize ;
VOIDPTR	data ;
int	datasize ;
DTMTYPE	datatype ;
{
	return intWriteMsg( p, hdr, hdrsize, data, datasize, datatype, WRITEMSG );
}
