/*****************************************************************************
*
*                         NCSA DTM version 2.0
*                               June 10, 1991
*
* NCSA DTM Version 2.0 source code and documentation are in the public
* domain.  Specifically, we give to the public domain all rights for future
* licensing of the source code, all resale rights, and all publishing rights.
*
* We ask, but do not require, that the following message be included in all
* derived works:
*
* Portions developed at the National Center for Supercomputing Applications at
* the University of Illinois at Urbana-Champaign.
*
* THE UNIVERSITY OF ILLINOIS GIVES NO WARRANTY, EXPRESSED OR IMPLIED, FOR THE
* SOFTWARE AND/OR DOCUMENTATION PROVIDED, INCLUDING, WITHOUT LIMITATION,
* WARRANTY OF MERCHANTABILITY AND WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE
*
*****************************************************************************/

/**********************************************************************
**
** dtminit.c - contains routines to initialize the dtm routines
**	based on command line options.
**
**********************************************************************/

/*
 * $Log: dtminit.c,v $
 * Revision 1.13  1991/09/18  15:28:11  jplevyak
 * Added some external definitions for shared functions.
 *
 * Revision 1.12  91/09/16  11:25:37  jplevyak
 * Fix bug, use of uninitialized register variable in function
 * DTMdestroyPort
 * 
 * Revision 1.11  91/09/13  20:09:31  sreedhar
 * supporting :9900, absence of env variable
 * 
 * Revision 1.10  1991/09/13  18:57:13  sreedhar
 * removed DTMinit() fn., added qservice in some places
 *
 * Revision 1.9  1991/08/15  18:56:19  sreedhar
 * Changes for logical portname version
 *
 * Revision 1.7  1991/06/11  15:34:15  sreedhar
 * quality of service parameter for future use
 *
 * Revision 1.6  1991/06/11  15:18:36  sreedhar
 * disclaimer added, availwrite/seqstart flags inited.
 *
 * Revision 1.5  1991/06/07  16:05:21  sreedhar
 * *colon = '\0' removed, it writes into user buffer
 *
 * Revision 1.4  1991/01/09  16:50:34  jefft
 * added include sys/include.h
 *
 * Revision 1.3  91/01/09  14:10:04  jefft
 * Now ignoring SIGPIPE signals.
 * 
 * Revision 1.2  90/11/21  10:53:08  jefft
 * Modified DTMgetPortAddr to return IP address instead of hostname.
 * 
 * Revision 1.1  90/11/08  16:21:54  jefft
 * Initial revision
 * 
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#ifndef	macintosh
#  include <sys/signal.h>
#else
#  include "DTMMac.h"
#endif
#include <netinet/in.h>
#include <malloc.h>

#include "dtm.h"
#include "dtmint.h"
#include "debug.h"


/*
	STATIC VARIABLES
*/

static int	DTMportCount = 0;
static int	initialized = FALSE;


/*
	STATIC FUNCTIONS
*/

/*
	 init_port()
	 Allocate and intialize port p.
*/

static 	int init_port( p, porttype, qservice )
int	p;
int	porttype ;
int	qservice ;
{
	register DTMPORT *pp ;

	DBGFLOW( "init_port called\n" );

	/* allocate port structure */

  	if( (pp = DTMpt[p] = (DTMPORT *)malloc(sizeof (DTMPORT))) == NULL )
	{
    		DTMerrno = DTMMEM;
    		DBGFLOW("DTMmakePort: could not allocate DTMPORT structure.");
    		return -1;
	}
	bzero( pp, sizeof( DTMPORT ) );

	pp -> porttype = porttype ;
	pp -> qservice = qservice ;

	/*	Input port data init	*/

	pp -> connfd = -1 ;

	/*	Output port data init	*/

	pp -> count = 0 ;
	pp -> out = NULL ;	

  	return 0;
}

static int 	DTMmakePort( portname, porttype, qservice )
	char	*portname ;
	int	porttype ;
	int	qservice ;
{
int	port ;

	DBGFLOW("DTMmakePort called.\n");

  	/* check for library initialization */
  	if( !initialized )
	{
    		/* create discard buffer */
    		if ((dtm_discard = (char *)malloc(DISCARDSIZE)) == NULL)  {
      			DTMerrno = DTMMEM;
      			DTMERR("DTMmakePort: insufficient memory for dicard buffer.");
      			return DTMERROR;
      		}

#ifndef	macintosh
    		/* ignore SIGPIPE signals, handled by dtm_write call */
    		signal(SIGPIPE, SIG_IGN);
#endif

    		initialized = TRUE;
    	}
    
  	/* find first open DTM port */
  	for (port=1; port < MAX_DTM_PORTS; port+=1)
	{
    		/* intialize port */
		if (DTMpt[port] == NULL)
		{
			if(init_port( port, porttype, qservice ) == DTMERROR)
				return DTMERROR;

			strncpy( DTMpt[ port ] -> portname, portname, 
					(PNAMELEN - 1) );
			DTMpt[ port ] -> portname[ PNAMELEN - 1 ] = '\0' ;

			DTMportCount += 1;
			return port;
		}
	}

  	/* no open DTM ports available */
  	DTMerrno = DTMNOPORT;
  	return DTMERROR;
}

/*	qservice parameter is for quality of service, for future use */

/*
	GLOBAL FUNCTIONS
*/

int DTMmakeInPort(portname, qservice )
char	*portname;
int	qservice ;
{
	int	p;
	int	fLogicalName = FALSE;

	DBGFLOW("DTMmakeInPort called.\n");

  	if( (p = DTMmakePort(portname, INPORTTYPE, qservice )) == DTMERROR )
    		return DTMERROR;

	if( init_sockaddr( &DTMpt[ p ] -> sockaddr, DTMpt[ p ] -> portname,
			&fLogicalName ) == -1 ) 
		return DTMERROR ;

	if( (DTMpt[p] -> sockfd = dtm_socket_init( &DTMpt[p] -> sockaddr, 
			INPORTTYPE, fLogicalName )) == -1 )
	{
		free( DTMpt[ p ] ) ;
		DTMpt[p] = NULL;
  		DTMportCount -= 1;
		return DTMERROR ;
	}

	DBGINT( "DTMmakeInPort: sockfd = %d\n", DTMpt[ p ] -> sockfd ) ;

	if( fLogicalName ) {
		int	fd ;
		struct	sockaddr_in addr ;
		char	*naddr ;
		extern	char	*dtm_get_naddr() ;
		extern	char	*dtm_get_refname() ;

		naddr = dtm_get_naddr( &addr, &fd );
		if( dtm_nsend_sockaddr( fd, naddr, dtm_get_refname(), portname, 
				&DTMpt[ p ] -> sockaddr ) < 0 ) {
			close( DTMpt[ p ] -> sockfd ) ;
			free( DTMpt[ p ] ) ;
			DTMpt[p] = NULL;
  			DTMportCount -= 1;
			return DTMERROR ;
		}
	}

  	return p;
}

int DTMmakeOutPort(portname, qservice )
char	*portname;
int	qservice ;
{
	int		p;
	int		fLogicalName = TRUE;
	struct	sockaddr_in	tcpaddr;

  	DBGFLOW("DTMmakeOutPort called.\n");

  	if( (p = DTMmakePort( portname, OUTPORTTYPE, qservice )) == -1)
		return DTMERROR;

	if( init_sockaddr( &tcpaddr, DTMpt[ p ] -> portname, &fLogicalName ) 
			== -1 ) 
		return DTMERROR ;

	if( !fLogicalName ) {

		Port	port ;
		Outport	*outp ;

		DBGINT( "DTMmakeOutPort: Physical TCP portname - %x ", 
					ntohl( tcpaddr.sin_addr.s_addr ));
		DBGINT( "%d\n", ntohs( tcpaddr.sin_port ));

		/* 
		   Case where physical (TCP) portname has been supplied
		   by user. 
		*/
		port.portid = tcpaddr.sin_port ;
		port.nethostid = tcpaddr.sin_addr.s_addr ;	

		if( (outp = (Outport *)malloc( sizeof( Outport ) )) == NULL )
		{
			DTMerrno = DTMMEM ;
			return DTMERROR ;
		} 
		dtm_add_newbuf( &port, outp );

		DTMpt[ p ] -> count = 1 ;
		DTMpt[ p ] -> out = outp ;	
	}	

	if( (DTMpt[p] -> sockfd = dtm_socket_init( &DTMpt[p] -> sockaddr,
		OUTPORTTYPE, fLogicalName )) == -1 ) {

		/*
			We cannot just free this port as the liked outport may
			still exist.
					- john 
		*/	
			free( DTMpt[ p ] ) ;
			DTMpt[p] = NULL;
			DTMportCount -= 1;
			return DTMERROR ;
	}

	if( fLogicalName ) {
		int	fd ;
		struct	sockaddr_in addr ;
		char	*naddr ;

		naddr = dtm_get_naddr( &addr, &fd );
		if( dtm_nsend_sockaddr( fd, naddr, dtm_get_refname(), 
				portname, &DTMpt[ p ] -> sockaddr ) < 0 )
		{
			close( DTMpt[ p ] -> sockfd ) ;
			free( DTMpt[ p ] ) ;
			DTMpt[p] = NULL;
  			DTMportCount -= 1;
			return DTMERROR ;
		}
	}

	return p ;
}


int DTMdestroyPort(p)
int	p;
{
register DTMPORT *pp ;

	DBGFLOW("DTMdestroyPort called.\n");

  	/* check if port has been made */

  	if(( pp = DTMpt[p] ) == NULL)
	{
    		DTMerrno = DTMPORTINIT;
    		DTMERR("DTMdestroyPort: port has not been made.");
    		return -1;
    	}

  	/* close main socket */

  	if (pp->sockfd != -1)
    		close(pp->sockfd);

  	/* close connections */

	if( pp -> porttype == INPORTTYPE )
	{
		if( pp -> connfd != -1 ) close( pp -> connfd );
	}
	else
	{
		register  Outport *pcur ;
		Outport	*pend; 

		pcur = pp->out;
		pend = pcur + pp->count ;
 
		if ( pcur != NULL )
		{
			while( pcur < pend )
			{
				if( pcur->connfd != -1 )
					close( pcur->connfd ) ;
				++pcur ;
			}		
			free ( pp->out ) ;
		}
	}
 
  	/* free space allocated for port */

  	free( pp );
	DTMpt[p] = NULL;

  	/* decrement port count */

  	DTMportCount -= 1;

  	return 0;
}

int DTMgetPortAddr(p, addr, len)
int	p, len;
char	*addr;
{
char	pnum[10];

  	DBGFLOW("DTMgetPortAddr called.\n");

  	/* check if port has been made */
  	if (DTMpt[p] == NULL)
	{
    		DTMerrno = DTMPORTINIT;
    		DTMERR("DTMgetPortAddr: port has not been made.");
    		return DTMERROR;
    	}

  	if (dtm_get_ipaddr(addr) == 0)
	{
    		DTMerrno = DTMHOST;
    		return DTMERROR;
    	}

  	sprintf(pnum, ":%d", ntohs( DTMpt[p]->sockaddr.sin_port ) );
  	strcat(addr, pnum);

  	return 0;
}

int DTMgetPortName(p, name)
  int	p;
  char	*name;
{

  if (DTMpt[p] == NULL)  {
    DTMerrno = DTMPORTINIT;
    return -1;
    }

  sprintf(name, "%s", DTMpt[p]->portname );
  return 0;
}
