
/* NOTE:  THIS MODULE AND SOURCE CODE IS FOR USE 
   WITH THE AVS SOFTWARE ENVIRONMENT ONLY */
/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/* 
 * Carpet - Create a carpet plot from 2D scalar data
 *
 * Derived from the concept of field to mesh,
 * with more specific assumptions about the input field,
 * and alternative to the colour scheme.
 *
 * version 3 !
 *
 * Author: Ian Curington, AVS Inc.
 *
 * Revision History:
 * 10 Jan 92  ianc - Original
 * 11 Jan 92  ianc - tidy up, sent to G. Walker, BT Labs
 *  7 Sep 93  ianc - upgrade to irregular 3space fields for
 *                   better hole algorithm
 *
 * ----------------- man page -------------------
 *
 * Module: "carpet"
 * 
 * Type:   mapper
 * 
 * Author: Ian Curington, Advanced Visual Systems, Inc.
 * 
 * Source: carpet.c
 * 
 * Description:
 * ------------
 *   A Carpet Plot surface is made showing surface height variation
 *   from one field, with color variations from data in a second field.
 *   Each cell is drawn as per field to mesh, (linear interpolation),
 *   or in block mode with descrete non-interpolated colors.
 *   It will optionally leave holes in the carpet
 *   where the data goes out of range for the colormap, allowing
 *   mask values to control cell visibility, or region boundaries.
 * 
 * Inputs:
 *   Field 1:
 *     "field 2D uniform/irregular scalar float"
 *     This controls the height map of the surface.
 *     It also controls the color variation, if no
 *     field 2 is present.
 *     Can be 2-space or 3-space.
 * 
 *   Field 2:
 *     If present, this field overrides the color, so
 *     two independent variables can be displayed in the same plot.
 *     This field must match the X and Y dimensions of field 1.
 * 
 *   Colormap:
 *     This controls the surface color variation applied to the
 *     input scalar field. The hi-low dials of the colormap field
 *     are used to match the field. These controls can control
 *     cell visibility when in "leave holes" mode.
 * 
 * Parameters:
 * -----------
 * 
 * Z-Scale
 *   This controls the relative offset from the base plane of each
 *   cell, creating the wavy carpet effect.
 * 
 * Smooth-Block
 *   In smooth mode, it builds the same structure as field_to_mesh,
 *   a fully connected lattice of points, with linear interpolation
 *   of both color and shape. In block mode, the shape is the same,
 *   but the color is broadcast to all vertices of each face from the
 *   first corner, ignoring data in the last row and column.
 *   It also generates a lot more polygons in block mode.
 * 
 * Flip Normal
 *   This controls a 180 degree flip in the surface normal,
 *   which will effect the lighting on the surface.
 * 
 * Leave Holes
 *   This will allow drop outs in the field where the 2nd field
 *   values go out of range for the colormap.
 * 
 * Example Network:
 * ----------------
 * 
 *                   Read Image (=avs.x)
 *                      |
 *                   Downsize    = 12
 *                      |
 *                   field to float
 *                   |         |
 *       extract scalar      extract scalar
 *                   |         |
 * gen colormap      |         |
 *        |          |         |
 *        |          |         |
 *        --------   |  -------|  
 *               |   |  |
 *                carpet
 *                   |
 *                geometry viewer
 *  
 * 
 * 
 */

/**********************************************************************/

#include <stdio.h>

#include <avs/avs.h>
#include <avs/field.h>
#include <avs/geom.h>
#include <avs/colormap.h>


/**********************************************************************/
/* Module Description Section */

carpet_desc()
{
      int carpet_compute();

      AVSset_module_name("carpet", MODULE_MAPPER);
      AVSset_module_flags( COOPERATIVE | REENTRANT);
      AVScreate_input_port("Field Input",
             "field 2D scalar float", REQUIRED);
      AVScreate_input_port("Field Input",
             "field 2D scalar float", OPTIONAL);
      AVScreate_input_port("Input Colormap", "colormap", REQUIRED);
      AVScreate_output_port("geometry", "geom");
      AVSadd_float_parameter("Z scale", 0.0, FLOAT_UNBOUND, FLOAT_UNBOUND);
      AVSadd_parameter("method", "choice", "Smooth","Smooth#Block", "#");
      AVSadd_parameter("flip normals", "boolean", 0, 0, 1 );
      AVSadd_parameter("leave holes", "boolean", 0, 0, 1 );
      AVSadd_parameter("On/Off", "boolean", 1, 0, 1 );
      AVSset_compute_proc(carpet_compute);
}

/**********************************************************************/
/* Module Compute Function */

carpet_compute(input, input2, colormap, output,
               zscale, method_str, flip, holes, on_off )

    AVSfield *input;
    AVSfield *input2;
    AVScolormap *colormap;
    GEOMedit_list *output;
    float *zscale;
    char  *method_str;
    int   flip;
    int   holes;
    int   on_off;
{
   GEOMobj *obj;
   float        *fmesh, min, max, min2, max2;
   unsigned int color_lut[BYTE_TABLE], *colors;
   int          size, i, j, k, indexer;
   int          axis;
   int          ii;
   float        extent[6], min_extent[3], max_extent[3];
   float        tx, ty, tz;
   int          method;
   int          i1, i2, i3, i4;
   int          active;

/*
 * B O D Y
 */
     /* if Off, take short cut and return, killing output geom */
     if ( !on_off )
     {
         obj = GEOMcreate_obj ( GEOM_POLYTRI, GEOM_NULL );
         *output = GEOMinit_edit_list(*output);
         GEOMedit_geometry(*output,"carpet",obj);
         GEOMdestroy_obj(obj);
         return(1);
     }

     /* check 2nd field, make sure it matches */
     if ( input2 ) 
     {
         if ( MAXX(input) != MAXX(input2) )
         {
             AVSerror("carpet: X dim of 2nd field doesn't match.");
             return(0);
         }

         if ( MAXY(input) != MAXY(input2) )
         {
             AVSerror("carpet: X dim of 2nd field doesn't match.");
             return(0);
         }
     }


     method = AVSchoice_number("method",method_str);

     size = MAXX(input) * MAXY(input);


     /*
      * use table driven color table for speed
      */
     colors = (unsigned int *) ALLOC_LOCAL (size * sizeof(unsigned int));
     for (i=0; i<colormap->size; i++) 
         PIXELhsva_to_pixel (&color_lut[i], colormap->hue[i], 
                  colormap->saturation[i],
                  colormap->value[i], colormap->alpha[i]);


     /*
      * Build the Mesh from the input field data
      */
     axis = 2;   /* default axes */

     fmesh = (float *) ALLOC_LOCAL (3 * size * sizeof(float));

   if ( input->uniform == UNIFORM )
   {
     if (input->points[0] == input->points[1])
        axis = 0;
     else if (input->points[2] == input->points[3])
        axis = 1;
     else if (input->points[4] == input->points[5])
        axis = 2;
     for (i=0; i<MAXY(input); i++) {
        for (j=0; j<MAXX(input); j++) {
         if (input->nspace == 3) {       /* slices comes from a volume */
             if (axis == 0) {        /* X Axis */
               UTILget_world_coords (&tx, &ty, &tz, (int)(input->points[0]),
                               j, i, input);
               fmesh[(i*MAXX(input)+j)*3+0] = tx;               
               fmesh[(i*MAXX(input)+j)*3+1] = ty;        
               fmesh[(i*MAXX(input)+j)*3+2] = tz;
             }  /* end if */ 
             else if (axis == 1) {  /* Y Axis */
               UTILget_world_coords (&tx, &ty, &tz, j, 
                               (int) (input->points[2]), i, input);
               fmesh[(i*MAXX(input)+j)*3+0] = tx; 
               fmesh[(i*MAXX(input)+j)*3+1] = ty;
               fmesh[(i*MAXX(input)+j)*3+2] = tz;
             }  /* end else if */ 
             else  {                /* Z Axis */
               UTILget_world_coords (&tx, &ty, &tz, j, i, 
                               (int) (input->points[4]), input);
               fmesh[(i*MAXX(input)+j)*3+0] = tx;
               fmesh[(i*MAXX(input)+j)*3+1] = ty;
               fmesh[(i*MAXX(input)+j)*3+2] = tz;
             }  /* end else */ 
         } /* end if for 3-space */
         else {                              /* 2D images */
              fmesh[(i*MAXX(input)+j)*3+0] = (float) j;
              fmesh[(i*MAXX(input)+j)*3+1] = (float)(MAXY(input)-i-1.0);
            fmesh[(i*MAXX(input)+j)*3+2] = 0.0;
            axis = 2;
         }  /* end else for images */
        }  /* end for j */
     }  /* end for i */

   }
   else if ( input->uniform == IRREGULAR && input->nspace == 3 )
   {
     k = 0;
     for (i=0; i<MAXY(input); i++) {
        for (j=0; j<MAXX(input); j++) {
               fmesh[(i*MAXX(input)+j)*3+0] = input->points[0*size+k];
               fmesh[(i*MAXX(input)+j)*3+1] = input->points[1*size+k];        
               fmesh[(i*MAXX(input)+j)*3+2] = input->points[2*size+k];
               k++;
        }  /* end for j */
     }  /* end for i */
   }   /* end of uniform-irregular test */
   else
   {
      AVSerror("carpet: field type not compatable, type=%d, nspace=%d",
                input->uniform, input->nspace );
      return(0);
   }

     /*
      * scan for min-max, on both ports
      */
     min = max = input->field_data_float[0];
     if ( input2 )
         min2 = max2 = input2->field_data_float[0];
     for (i=0; i<size; i++)
     {
         min = (input->field_data_float[i] < min ) ?
                  input->field_data_float[i] : min;
         max = (input->field_data_float[i] > max ) ?
                  input->field_data_float[i] : max;
         if ( input2 )
         {
         min2 = (input2->field_data_float[i] < min2 ) ?
                  input2->field_data_float[i] : min2;
         max2 = (input2->field_data_float[i] > max2 ) ?
                  input2->field_data_float[i] : max2;
         }
     }

     /*
      * offset the mesh geometry with scalar warp for first input port
      */
     if (max == min)
     {
         for (i=0; i<MAXY(input); i++) 
           for (j=0; j<MAXX(input); j++) 
             fmesh[(i*MAXX(input)+j)*3+axis] += *zscale;
     } else {
         for (i=0; i<MAXY(input); i++) 
           for (j=0; j<MAXX(input); j++) 
             fmesh[(i*MAXX(input)+j)*3+axis] += *zscale *
             ((float)(input->field_data_float[i*MAXX(input)+j]) - min) /
                 (max-min);
     }

     /*
      * Get Colors
      */
     if ( input2 )   /* get color from 2nd port, if possible */
     {
         if (max2 == min2)
         {
             for (i=0; i<MAXY(input2); i++) 
             for (j=0; j<MAXX(input2); j++) 
               colors[i*MAXX(input2)+j] = color_lut[0];
         } else {
             for (i=0; i<MAXY(input2); i++) {
               for (j=0; j<MAXX(input2); j++) {
                   indexer = PIXELget_index(colormap, 
                             input2->field_data_float[i*MAXX(input)+j]);
                   colors[i*MAXX(input)+j] = color_lut[indexer];
               }
             }
         }
     }
     else    /* color and height off the same value, on input 1 */
     {
         if (max == min)
         {
             for (i=0; i<MAXY(input); i++) 
             for (j=0; j<MAXX(input); j++) 
               colors[i*MAXX(input)+j] = color_lut[0];
         } else {
             for (i=0; i<MAXY(input); i++) {
               for (j=0; j<MAXX(input); j++) {
                   indexer = PIXELget_index(colormap, 
                             input->field_data_float[i*MAXX(input)+j]);
                   colors[i*MAXX(input)+j] = color_lut[indexer];
               }
             }
         }
     }

         obj = GEOMcreate_obj ( GEOM_POLYHEDRON, GEOM_NULL );
         for (i=0; i<MAXY(input)-1; i++)
         {
           for (j=0; j<MAXX(input)-1; j++)
           {
               i1 = (i*MAXX(input)+j);
               i2 = ((i+1)*MAXX(input)+j);
               i3 = ((i+1)*MAXX(input)+j+1);
               i4 = ((i)*MAXX(input)+j+1);

               if (input2)
               {
                if ( method == 2 )
                    active=check_range (input2->field_data_float[i1],
                                  colormap->lower, colormap->upper ); 
                else
                {
                    active=check_range (input2->field_data_float[i1],
                                  colormap->lower, colormap->upper ) &&
                           check_range (input2->field_data_float[i2],
                                  colormap->lower, colormap->upper ) &&
                           check_range (input2->field_data_float[i3],
                                  colormap->lower, colormap->upper ) &&
                           check_range (input2->field_data_float[i4],
                                  colormap->lower, colormap->upper ); 
                }
               }
               else
               {
                if ( method == 2 )
                    active=check_range (input->field_data_float[i1],
                                  colormap->lower, colormap->upper );
                else
                {
                    active=check_range (input->field_data_float[i1],
                                  colormap->lower, colormap->upper ) &&
                           check_range (input->field_data_float[i2],
                                  colormap->lower, colormap->upper ) &&
                           check_range (input->field_data_float[i3],
                                  colormap->lower, colormap->upper ) &&
                           check_range (input->field_data_float[i4],
                                  colormap->lower, colormap->upper ); 
                }
               }

                 if ( holes == 0 || holes == 1 && active )
                   if ( method == 1 )
                   {
                       draw_a_quad (obj,
                            fmesh[i1*3], fmesh[i1*3+1],fmesh[i1*3+2],
                            fmesh[i2*3], fmesh[i2*3+1],fmesh[i2*3+2],
                            fmesh[i3*3], fmesh[i3*3+1],fmesh[i3*3+2],
                            fmesh[i4*3], fmesh[i4*3+1],fmesh[i4*3+2],
                            colors[i1], colors[i2], colors[i3], colors[i4] );
                   }
                   else if ( method == 2 )
                   {
                       draw_a_quad (obj,
                            fmesh[i1*3], fmesh[i1*3+1],fmesh[i1*3+2],
                            fmesh[i2*3], fmesh[i2*3+1],fmesh[i2*3+2],
                            fmesh[i3*3], fmesh[i3*3+1],fmesh[i3*3+2],
                            fmesh[i4*3], fmesh[i4*3+1],fmesh[i4*3+2],
                            colors[i1], colors[i1], colors[i1], colors[i1] );
                   }
           }
         }

         GEOMgen_normals(obj, 0); 
         GEOMcvt_polyh_to_polytri(obj, GEOM_SURFACE   |
                                       GEOM_WIREFRAME |
                                       GEOM_NO_CONNECTIVITY );


     if ( flip ) 
        GEOMflip_normals(obj);
   
     *output = GEOMinit_edit_list(*output);
     GEOMedit_geometry(*output,"carpet",obj);

     if (input->nspace < 3)
         GEOMedit_transform_mode(*output, "carpet", "normal",0); 
     else
         GEOMedit_transform_mode(*output, "carpet", "parent",0);
     GEOMdestroy_obj(obj);

     FREE_LOCAL(fmesh);
     if (colormap) 
       FREE_LOCAL(colors);

     return(1);
}

/**********************************************************************/
/* draw one quad in one color at specified corners */

draw_a_quad (obj, x1, y1, z1,
                  x2, y2, z2,
                  x3, y3, z3,
                  x4, y4, z4,
                  color1, color2, color3, color4 )
    GEOMobj       *obj;
    float         x1, y1, z1,
                  x2, y2, z2,
                  x3, y3, z3,
                  x4, y4, z4;
    unsigned int  color1, color2, color3, color4;
{
       float verts[ 4 * 3 ];
       int   cols [ 4 ];

         /* draw quad */
         verts[ 0 * 3 + 0 ] = x1;
         verts[ 0 * 3 + 1 ] = y1;
         verts[ 0 * 3 + 2 ] = z1;
         cols [ 0 ]         = color1;

         verts[ 1 * 3 + 0 ] = x4;
         verts[ 1 * 3 + 1 ] = y4;
         verts[ 1 * 3 + 2 ] = z4;
         cols [ 1 ]         = color4;

         verts[ 2 * 3 + 0 ] = x3;
         verts[ 2 * 3 + 1 ] = y3;
         verts[ 2 * 3 + 2 ] = z3;
         cols [ 2 ]         = color3;

         verts[ 3 * 3 + 0 ] = x2;
         verts[ 3 * 3 + 1 ] = y2;
         verts[ 3 * 3 + 2 ] = z2;
         cols [ 3 ]         = color2;


       GEOMadd_disjoint_polygon ( obj, verts, NULL, NULL, 4,
                                  GEOM_NOT_SHARED, GEOM_COPY_DATA);
       GEOMadd_int_colors(obj, cols, 4, GEOM_COPY_DATA);

}


/**********************************************************************/
/* check for data within range or not */

int check_range( val, low, hi )
    float val, hi, low;
{
    if ( val < low )
        return(0);
    else if ( val > hi )
        return(0);
    else
        return(1);
}



/**********************************************************************/

#ifdef SEP_EXE
/*
 * initialization for modules contained in this file.
 */

int ((*mapper_mod_list[])()) = {
  carpet_desc,
};

#define NMODS sizeof(mapper_mod_list) / sizeof(char *) 

AVSinit_modules ()
{
      AVSinit_from_module_list(mapper_mod_list, NMODS);
}
#endif

