/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/* IAC CODE CHANGE : INSERT INCLUDE */
#include <values.h>
/*
 * Author: Massimiliano Bavelloni
 * Date Created: Fri Jul 13 10:45:07 1994
 *
 * Draw geometric cone around 3D scattered points in order to 
 * display astronomical points (galaxies). Galaxy coordinates 
 * have been already converted from Spherical Coordinates into 
 * Cartesian Coordinates.
 * In the computational section coordinates are converted into 
 * Spherical ones and then converted again in Cartesian coordinates.
 * The cone is always centered in origin (x=0, y=0, z=0).
 * Module draws axes, sets color of cone.
 *
 * This module can also be used with other kind of data (not only
 * galaxies).
 * May be used with any kind of scattered points in a 
 * 3D space.
 *

/*
 *-----------------------------------------------------------*
 *                                                           *
 *              Cone_Bounding    module                      *
 *                                                           *
 *------------------------------------------------------------
 */


#include <stdio.h>


#include <avs/avs.h>
/* IAC CODE CHANGE : #include <avs/avs_math.h> */
#include <avs/avs_math.h>
#include <avs/geom.h>
#include <avs/udata.h>
#include <avs/field.h>



/*
 * COSTANTS
 */
#define MAXFLOAT               ((float)3.40282346638528860e+38)


#define TRUE                   1
#define FALSE                  0

#define AND                    &&
#define OR                     ||
#define NOT                    !


#define RED                    0
#define GREEN                  1
#define BLUE                   2

#define COLORS                 3
#define DEFAULT_COLOR          0.95
#define MIN_VAL_COLOR          0.0
#define MAX_VAL_COLOR          1.0


#define X0                     0.0
#define Y0                     0.0
#define Z0                     0.0
#define VERT_0                 0
#define VERT_1                 1
#define VERT_2                 2
#define X_CRD                  0
#define Y_CRD                  1
#define Z_CRD                  2


#define R_MIN                  0.0

#define FIRST_PT               0


#define NUM_VERTS              3
#define NUM_VERTS_LINE         2
#define NUM_VERT_AXIS          2

#define NUM_CRD                3

#define NUM_POLY_FACE          8


#define INIT_VAL_N_TICKS       100.0
#define MIN_VAL_N_TICKS        1.0
#define MAX_VAL_N_TICKS        500.0


#define X_MIN_POS              0
#define X_MAX_POS              1
#define Y_MIN_POS              2
#define Y_MAX_POS              3
#define Z_MIN_POS              4
#define Z_MAX_POS              5


#define DIM_RANGE_XYZ          6




/*
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 *----------------------------------------------------------*
 *              ***   DESCRIPTION AREA    ***               *
 *----------------------------------------------------------*
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 */


/*
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 *                                                          *
 *           ***    cone_bounding_desc   ***                *
 *                                                          *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 */
 
/*  
 * AVS description function, define: 
 * Input Ports, Output Ports, and widget parameters
 */
int 
cone_bounding_desc()
 {
	extern cone_bounding_compute();
	int param;
	int in_port;
	int out_port;

	/* 
	 * name on the box
	 */
	AVSset_module_name("cone bounding", MODULE_MAPPER);

	/*
	 * 
	 * Input Definition Port
	 *
	 */
	 
	/*
	 * input ucd structure to draw
	 */
	in_port = AVScreate_input_port("Input", 
		"field 1D 3-space irregular float", REQUIRED);




	/*
	 * 
	 * Output Definition Port
	 *
	 */
	 
	/*
	 * output float array of xyz coordinates cone range
	 */
	out_port = AVScreate_output_port("xyz crd cone range", 
		"field 1D uniform real");


	/* 
	 * output geometry to draw
	 */
	out_port = AVScreate_output_port("cone_bounding", "geom");




	/*
	 *
	 * Parameter Definition Area
	 *
	 */
	 
	/*
	 * modify red color index of cone bound
	 */
	param = AVSadd_float_parameter("RED_bound", DEFAULT_COLOR, 
		MIN_VAL_COLOR, MAX_VAL_COLOR);
	AVSconnect_widget(param, "dial"); 


	/* 
	 * modify green color index of cone bound
	 */
	param = AVSadd_float_parameter("GREEN_bound", DEFAULT_COLOR, 
		MIN_VAL_COLOR, MAX_VAL_COLOR);
	AVSconnect_widget(param, "dial");


	/* 
	 * modify blue color index of cone bound
	 */
	param = AVSadd_float_parameter("BLUE_bound", DEFAULT_COLOR, 
		MIN_VAL_COLOR, MAX_VAL_COLOR);
	AVSconnect_widget(param, "dial"); 


	/*
	 * select number of segments to approximate arc 
	 */
	param = AVSadd_float_parameter("N_Ticks", INIT_VAL_N_TICKS, 
		MIN_VAL_N_TICKS, FLOAT_UNBOUND);
	AVSconnect_widget(param, "dial"); 
	
	



	/*
	 *
	 * INITIALIZE PROCESS
	 *
	 */
	 
	/* 
	 * routine pointers 
	 */
	AVSset_compute_proc(cone_bounding_compute);


 } /* cone_bounding_desc */






/*
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 *----------------------------------------------------------*
 *         ***   AVS COMPUTE ROUTINE AREA    ***            *
 *----------------------------------------------------------*
 *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */


/*
 **********************************************************
 * +++++++++++  Module  compute routine  ++++++++++++++++ *
 **********************************************************
 */
/*
 * Establish output geometry object
 * display cone bound
 * complete the geometry output process
 * paramters:
 *	(i) input: AVS field 1D 3-space irregular float 
 *				(scatter dots) 
 *	(o) out_cone_range: coordinate range structure 
 *	(o) output: geometry output list
 *	(i) Red_bound: red index of RGB color of cone
 *	(i) Green_bound: green index of RGB color of cone
 *	(i) Blue_bound: blue index of RGB color of cone
 *	(i) N_Ticks: number of segments to approximate arc of 
 *			circle
 *
 */
int 
cone_bounding_compute(input, out_cone_range, output, RED_bound, 
	GREEN_bound, BLUE_bound, N_Ticks)
AVSfield_float *input;
AVSfield_float **out_cone_range;
GEOMedit_list **output;
float *RED_bound;
float *GREEN_bound;
float *BLUE_bound;
float *N_Ticks;
{
	GEOMobj *obj;
	int num_bounds;
	int dims;
	char tmp[256], m_field[256];
	
	
	/*
	 * array of pointer of min and max coordinates of cone
	 */
	sprintf(m_field, "field 1D 1-space 0-vector uniform real");
	dims = DIM_RANGE_XYZ;
	(*out_cone_range) = (AVSfield_float *) AVSdata_alloc(m_field, 
		&dims);

	


	/* 
	 * number of objects to draw 
	 */
	num_bounds = input->dimensions[0];

	/* 
	 * establish the output geometry object 
	 */
	*output = (GEOMedit_list *)GEOMinit_edit_list(*output);
	obj = GEOMcreate_obj(GEOM_POLYHEDRON, NULL);

	
	/*
	 * draw the bounding cone
	 */
	display_cone_bounds(input, num_bounds, (*N_Ticks), 
		(*RED_bound), (*GREEN_bound), (*BLUE_bound), 
		obj, out_cone_range);
	


	/*
	 * 
	 * Send geometry to output port 
	 *
	 */
	 
	/*
	 * generate surface normals for object
	 *
	 */ 
	GEOMgen_normals(obj, 0);
	
	/*
	 * covert polyhedron to a polytriangle strip for
	 * more efficient rendering
	 */
	GEOMcvt_polyh_to_polytri(obj, GEOM_NO_CONNECTIVITY | 
		GEOM_SURFACE | GEOM_WIREFRAME);

	/* 
	 * set reference to the object
	 */
	GEOMedit_geometry(*output, "cone_bounding", obj);
	
	/*
	 * destroy object
	 */
	GEOMdestroy_obj(obj);

	/*
	 * set transformation mode for the object
	 */
	GEOMedit_transform_mode(*output, "cone_bounding", "parent", 
		BUTTON_UP);
	
	/*
	 * set render mode: "Line" represent dijointed lines
	 * connecting vertex of the object
	 */
	GEOMedit_render_mode(*output, "cone_bounding", "lines");

	/* AVSdata_free("field", *out_cone_range); */

	return(TRUE);
} /* cone_bounding_compute */




/*
 *************************************************************
 * Initialization for modules contained in this file          
 *************************************************************
 */

static int ((*mod_list[])()) = {
	cone_bounding_desc
};

#define NMODS (sizeof(mod_list) / sizeof(char *))



AVSinit_modules()
{

	AVSinit_from_module_list(mod_list, NMODS);
} /* AVSinit_modules */



/*
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 *---------------------------------------------------------*
 *     ***   UTILITY USER DEFINED FUNCTION AREA   ***      *
 *---------------------------------------------------------*
 *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 */



/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 *                                                          *
 *        *** convXYZToRaDec ***                            *
 *                                                          *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 */

/*
 * Compute conversion from Cartesian Coordinates to 
 * Spherical (Equatorial) Coordinates.
 * parameters:
 *	(i) x: x coordinate of object (galaxy)
 *	(i) y: y coordinate of object (galaxy)
 *	(i) z: z coordinate of object (galaxy)
 *	(o) ra: right ascension of object (galaxy)
 *	(o) dec: declination of object (galaxy)
 *	(o) r: distance of object (galaxy)
 */
int
convXYZToRaDec(x, y, z, ra, dec, r)
float x;
float y;
float z;
float *ra;
float *dec;
float *r;
{

	/*
	 * compute distance
	 */
	(*r) = sqrt( x*x + y*y + z*z );
	
	/*
	 * compute ra
	 */
	if (x >= 0.0)  
		if ( y >= 0.0 )  
			(*ra) = atan(y/x); 
		else 
			(*ra) = (2.0 * M_PI) + atan(y/x);
	else 
		(*ra) = M_PI + atan(y/x);
	
	if ( (*ra) > M_PI ) 
		(*ra) = -(( 2.0 * M_PI ) - (*ra));
	
	
	/*
	 * compute dec
	 */
	if (z > 0.0)
		(*dec) = M_PI_2 - atan(sqrt( x*x + y*y ) / z);
	else
		if (z < 0.0)
			(*dec) = -M_PI_2 - atan(sqrt( x*x + y*y ) / z);
	
		else
			/*
			 * if z == 0.0 --> dec = 0.0
			 */
			(*dec) = 0.0;
		
	
	
	return(TRUE);
} /* convXYZToRaDec */




/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 *                                                          *
 *        *** convRaDecToXYZ ***                            *
 *                                                          *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 */
/*
 * Compute conversion from Spherical (Equatorial) 
 * Coordinates to Cartesian Coordinates
 * parameters:
 *	(i) ra: right ascension of object (galaxy)
 *	(i) dec: declination of object (galaxy)
 *	(i) r: distance of object (galaxy)
 *	(o) x: x coordinate of object (galaxy)
 *	(o) y: y coordinate of object (galaxy)
 *	(o) z: z coordinate of object (galaxy)
 */
int
convRaDecToXYZ(ra, dec, r, x, y, z)
float ra;
float dec;
float r;
float *x;
float *y;
float *z;
{
	(*x) = r*cos(dec)*cos(ra);
	(*y) = r*cos(dec)*sin(ra);
	(*z) = r*sin(dec);

	return(TRUE);
} /* convXYZtoRaDec */



/*
 *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 *                                                           *
 *                   *** maxValue ***                        *
 *                                                           *
 *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 */
/*
 * Compute the max value of x, y, z
 * parameters:
 *	(o) x_max: max value of x
 *	(o) y_max: max value of y
 *	(o) z_max: max value of z
 *	(i) x_tmp: actual value of x
 *	(i) y_tmp: actual value of y
 *	(i) z_tmp: actual value of z
 */
int
maxValue(x_max, y_max, z_max, x_tmp, y_tmp, z_tmp)
float *x_max;
float *y_max;
float *z_max;
float x_tmp;
float y_tmp;
float z_tmp;
{

	if ((*x_max) < x_tmp)
		(*x_max) = x_tmp;
	
	if ((*y_max) < y_tmp)
		(*y_max) = y_tmp;
	
	if ((*z_max) < z_tmp)
		(*z_max) = z_tmp;
	
	
	return(TRUE);
} /* maxValue */ 



/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 *                                                          *
 *                     *** minValue ***                     *
 *                                                          *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 */
/*
 * Compute the min value of x, y, z
 * parameters:
 *	(o) x_min: min value of x
 *	(o) y_min: min value of y
 *	(o) z_min: min value of z
 *	(i) x_tmp: actual value of x
 *	(i) y_tmp: actual value of y
 *	(i) z_tmp: actual value of z
 */
int
minValue(x_min, y_min, z_min, x_tmp, y_tmp, z_tmp)
float *x_min;
float *y_min;
float *z_min;
float x_tmp;
float y_tmp;
float z_tmp;
{

	if ((*x_min) > x_tmp)
		(*x_min) = x_tmp;
	
	if ((*y_min) > y_tmp)
		(*y_min) = y_tmp;
	
	if ((*z_min) > z_tmp)
		(*z_min) = z_tmp;

	
	return(TRUE);
} /* maxValue */



/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 *                                                          *
 *               *** maxMinRaDec ***                        *
 *                                                          *
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 */

/*
 * Compute max and min values of Ra and Dec and
 * max value of r (distance)
 * 
 * parameters:
 *	(o) max_ra: max value of ra
 *	(o) min_ra: min value of ra
 *	(o) max_dec: max value of dec
 *	(o) min_dec: min value of dec
 *	(o) max_r: max value of distance
 *	(i) ra: current value of ra
 *	(i) dec: current value of dec
 *	(i) r: current distance
 *
 */
int
maxMinRaDec(max_ra, min_ra, max_dec, min_dec, max_r, 
	ra, dec, r)
float *max_ra;
float *min_ra;
float *max_dec;
float *min_dec;
float *max_r;
float ra;
float dec;
float r;
{
	if ( (*max_ra) < ra ) 
		(*max_ra) = ra;
		
	if ( (*min_ra) > ra ) 
		(*min_ra) = ra;
		
	if ( (*max_dec) < dec )  
		(*max_dec) = dec;
		
	if ( (*min_dec) > dec )  
		(*min_dec) = dec;

	if ( (*max_r) < r ) 
		(*max_r) = r;
	
	return(TRUE);
} /* maxMinRaDec */





/*
 * Open file, useful to check the converted values of 
 * coordinates
 * parameters:
 *	(o) fp: file pointer
 *	(i) str: filename
 *
 */
int
debugOpenFile(fp, str)
FILE **fp;
char *str;
{
	char filename[15];
	char strin[256];

	strcpy(filename, str);
	if ((*fp = fopen(filename, "w")) != NULL) {
		sprintf(strin," %s %s %s     %s %s %s   %s %s %s   %s", 
			"RA", "DEC", "Z" , "X", "Y", "Z", 
			"RA", "DEC", "Z", " \n\n");
		fputs(strin, (*fp));
		
		return(TRUE);
	} else 
		return(FALSE);
		
} /* debugOpenFile */



/*
 * Close file 
 * parameter:
 *	(i) fp: file pointer
 *
 */
int
debugCloseFile(fp)
FILE *fp;
{
	fclose(fp);
	
	return(TRUE);
} /* debugCloseFile */



/*
 * Write into debug file
 * parameters:
 *	(i) fp_debug: file pointer
 *	(i) strin: info to write
 *
 */
int
debugWrtFile2(fp_debug, strin)
FILE *fp_debug;
char strin[256];
{
	char buffer_str[256];


	sprintf(buffer_str, "%s \n", strin);
	fputs(buffer_str, fp_debug);

	return(TRUE);
} /* debugWrtFile */




/*
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 *                                                      *
 *       ***   draw axis   ***                          *
 *                                                      *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 */ 
/*
 * Draw axes: x red, y green, z blue. Axes length related 
 * with max coordinate value.
 * parameters:
 *	(o) obj: output geometry object
 *	(i) x_max: max x coordinate value
 *	(i) y_max: max y coordinate value
 *	(i) z_max: max z coordinate value
 *
 */
int 
draw_axis(obj, x_max, y_max, z_max)
GEOMobj* obj;
float x_max;
float y_max;
float z_max;
{
	float axis[NUM_VERT_AXIS * NUM_CRD];
	float cols[NUM_VERT_AXIS][COLORS];
	int i, j;
	
	
	/* 
	 * set initial color of x axis as red
	 */
	axis[0] = 0.0;
	axis[1] = 0.0;
	axis[2] = 0.0;
	axis[3] = x_max;
	axis[4] = 0.0;
	axis[5] = 0.0;
	
	for (i = 0; i < NUM_VERT_AXIS; i++) {
		cols[ i ][ RED ]   = 1.0; 
		cols[ i ][ GREEN ] = 0.0;
		cols[ i ][ BLUE ]  = 0.0;
	}
	
	/* 
	 * establish geometry object for axis
	 */
	GEOMadd_disjoint_polygon(obj, axis, NULL, NULL, 
		NUM_VERT_AXIS, GEOM_NOT_SHARED, 
		GEOM_COPY_DATA);
	GEOMadd_float_colors(obj, cols, NUM_VERT_AXIS, 
		GEOM_COPY_DATA);


	
	/* 
	 * set initial color of y axis as green
	 */
	axis[0] = 0.0;
	axis[1] = 0.0;
	axis[2] = 0.0;
	axis[3] = 0.0;
	axis[4] = y_max;
	axis[5] = 0.0;

	for (i = 0; i < NUM_VERT_AXIS; i++) {
		cols[ i ][ RED ] = 0.0; 
		cols[ i ][ GREEN ] = 1.0;
		cols[ i ][ BLUE ] = 0.0;
	}

	/* 
	 * establish geometry object for axis
	 */
	GEOMadd_disjoint_polygon(obj, axis, NULL, NULL, 
		NUM_VERT_AXIS, GEOM_NOT_SHARED,
		GEOM_COPY_DATA);
	GEOMadd_float_colors(obj, cols, NUM_VERT_AXIS, 
		GEOM_COPY_DATA);



	/* 
	 * set initial color of z axis as yellow
	 */
	axis[0] = 0.0;
	axis[1] = 0.0;
	axis[2] = 0.0;
	axis[3] = 0.0;
	axis[4] = 0.0;
	axis[5] = z_max;

	for (i = 0; i < NUM_VERT_AXIS; i++) {
		cols[ i ][ RED ] = 1.0; 
		cols[ i ][ GREEN ] = 1.0;
		cols[ i ][ BLUE ] = 0.0;
	}

	/* 
	 * establish geometry object for axis
	 */
	GEOMadd_disjoint_polygon(obj, axis, NULL, NULL, 
		NUM_VERT_AXIS, GEOM_NOT_SHARED,
		GEOM_COPY_DATA);
	GEOMadd_float_colors(obj, cols, NUM_VERT_AXIS, 
		GEOM_COPY_DATA);

	
	return(TRUE);
} /* draw_axis */



/*
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 *                                                      *
 *        ***   create_bounds   ***                     *
 *                                                      *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 */
/* 
 * Compute the segments to approximate the arces of circle
 * uppend this object at geometry object list, color of cone
 * and compute the 4 face of cone. The min value of r is 0.
 * parameters:
 *	(i)   obj: geometry object
 *	(i)   ra_min: min value of ra
 *	(i)   ra_max: max value of ra
 *	(i)   dec_min: min value of dec
 *	(i)   dec_max: max value of dec
 *	(i)   r_max: max value of distance
 *	(i/o) x_min: min value of x Cartesian coordinates
 *	(i/o) x_max: max value of x Cartesian coordinates
 *	(i/o) y_min: min value of y Cartesian coordinates
 *	(i/o) y_max: max value of y Cartesian coordinates
 *	(i/o) z_min: min value of z Cartesian coordinates
 *	(i/o) z_max: max value of z Cartesian coordinates
 *	(i)   color: array of colours
 *	(i)   n_ticks: number of segments to approximate 
 				the arc of circle
 */
int  
create_bounds (obj, ra_min, ra_max, dec_min, dec_max, r_max, 
	       x_min, x_max, y_min, y_max, z_min, z_max, 
	       color, n_ticks)
GEOMobj *obj;
float ra_min;
float ra_max;
float dec_min;
float dec_max;
float r_max;
float *x_min;
float *x_max;
float *y_min;
float *y_max;
float *z_min;
float *z_max;
float *color;
float n_ticks;
{
	float x0_new, y0_new, z0_new;
	float xc, yc, zc;
	float x1, y1, z1;
	float x2, y2, z2;
	float x_st, y_st, z_st;
	float mid_ra, mid_dec;
	float step_ra, step_dec;
	float old_ra, old_dec;
	float old_x, old_y, old_z;
	float tmp_ra, tmp_dec;
	float base[NUM_POLY_FACE * NUM_CRD];
	float face[NUM_VERTS * NUM_CRD];
	float tick[NUM_VERTS * NUM_CRD];
	float cols[NUM_POLY_FACE][COLORS];
	int i;
	char tmp[50]; 





	/* 
	 * copy same colours to all sides 
	 */
	for (i=0; i<4; i++) {
		cols[i][RED] = color[RED];
		cols[i][GREEN] = color[GREEN];
		cols[i][BLUE] = color[BLUE];
	} /* end for */


	/*
	 * Vertex of cone in the origin (0,0,0)
	 */
	x0_new = X0;
	y0_new = Y0;
	z0_new = Z0;

	
	/*
	 * length of arc step
	 */
	step_ra =  (ra_max - ra_min) / n_ticks;
	step_dec = (dec_max - dec_min) / n_ticks;
	
	
	/*
	 *
	 * Bound, Arc of RA with Max DEC Coordinate, N. 1
	 *
	 */
	old_ra = ra_min;
	old_dec = dec_max;
	
	/*
	 * calculate Cartesian coordinates corresponding to
	 * Spherical coordinates with max distance
	 */
	convRaDecToXYZ(old_ra, old_dec, r_max, &old_x, &old_y, 
		&old_z);

	/*
	 * approximate arc of circle with a lot of segments
	 */
	for (tmp_ra = ra_min + step_ra; tmp_ra <= ra_max ; 
		tmp_ra = tmp_ra + step_ra) {

		/*
		 * first vertex of segment  
		 */
		tick[VERT_0 * NUM_CRD + X_CRD] = old_x;
		tick[VERT_0 * NUM_CRD + Y_CRD] = old_y;
		tick[VERT_0 * NUM_CRD + Z_CRD] = old_z;


		/*
		 * calculate Cartesian coordinates of second 
		 * vertex of segment
		 */
		convRaDecToXYZ(tmp_ra, old_dec, r_max, &x_st, 
			&y_st, &z_st);
		/*
		 * store max and min values
		 */
		maxValue(x_max, y_max, z_max, x_st, y_st, z_st);
		minValue(x_min, y_min, z_min, x_st, y_st, z_st);

		/*
		 * set second vertex of segment
		 */
		tick[VERT_1 * NUM_CRD + X_CRD ] = x_st;
		tick[VERT_1 * NUM_CRD + Y_CRD ] = y_st;
		tick[VERT_1 * NUM_CRD + Z_CRD ] = z_st;

		/*
		 * append tick object to geometry objects
		 * and define color
		 */
		GEOMadd_disjoint_polygon(obj, tick, NULL, NULL, 
			NUM_VERTS_LINE, GEOM_NOT_SHARED, 
			GEOM_COPY_DATA);
		GEOMadd_float_colors(obj, cols, NUM_VERTS_LINE, 
			GEOM_COPY_DATA);
		
		/* 
		 * store current position
		 */
		old_x = x_st;
		old_y = y_st;
		old_z = z_st;

	} /* for */


	/* 
	 * append part of segment if necessary
	 */
	if (tmp_ra >= ra_max) {
	
		tick[VERT_0 * NUM_CRD + X_CRD] = old_x;
		tick[VERT_0 * NUM_CRD + Y_CRD] = old_y;
		tick[VERT_0 * NUM_CRD + Z_CRD] = old_z;

		convRaDecToXYZ(ra_max, old_dec, r_max, &x_st, 
			&y_st, &z_st);

		tick[VERT_1 * NUM_CRD + X_CRD ] = x_st;
		tick[VERT_1 * NUM_CRD + Y_CRD ] = y_st;
		tick[VERT_1 * NUM_CRD + Z_CRD ] = z_st;

		GEOMadd_disjoint_polygon(obj, tick, NULL, NULL, 
			NUM_VERTS_LINE, GEOM_NOT_SHARED, 
			GEOM_COPY_DATA);
		GEOMadd_float_colors(obj, cols, NUM_VERTS_LINE, 
			GEOM_COPY_DATA);

	} /* if */


	/* 
	 *
	 * Bound, Arc of RA with Min DEC coordinate. N. 2
	 *
	 */
	old_ra = ra_min;
	old_dec = dec_min;
	
	/*
	 * calculate Cartesian coordinates corresponding to
	 * Spherical coordinates with max distance
	 */
	convRaDecToXYZ(old_ra, old_dec, r_max, &old_x, &old_y, 
		&old_z);
	
	/*
	 * approximate arc of circle with a lot of segments
	 */
	for (tmp_ra = ra_min + step_ra; tmp_ra <= ra_max ; 
		tmp_ra = tmp_ra + step_ra) {

		/*
		 * first vertex of segment
		 */
		tick[VERT_0 * NUM_CRD + X_CRD] = old_x;
		tick[VERT_0 * NUM_CRD + Y_CRD] = old_y;
		tick[VERT_0 * NUM_CRD + Z_CRD] = old_z;


		/*
		 * calculate Cartesian coordinates of second 
		 * vertex of segment
		 */
		convRaDecToXYZ(tmp_ra, old_dec, r_max, &x_st, 
			&y_st, &z_st);
		/*
		 * store max and min values
		 */
		maxValue(x_max, y_max, z_max, x_st, y_st, z_st);
		minValue(x_min, y_min, z_min, x_st, y_st, z_st);

		/*
		 * set second vertex of segment
		 */
		tick[VERT_1 * NUM_CRD + X_CRD ] = x_st;
		tick[VERT_1 * NUM_CRD + Y_CRD ] = y_st;
		tick[VERT_1 * NUM_CRD + Z_CRD ] = z_st;

		/*
		 * append tick object to geometry objects
		 * and define color
		 */
		GEOMadd_disjoint_polygon(obj, tick, NULL, NULL, 
			NUM_VERTS_LINE, GEOM_NOT_SHARED, 
			GEOM_COPY_DATA);
		GEOMadd_float_colors(obj, cols, NUM_VERTS_LINE, 
			GEOM_COPY_DATA);

		/* 
		 * store current position
		 */
		old_x = x_st;
		old_y = y_st;
		old_z = z_st;

	} /* for */
	
	/* 
	 * append part of segment if necessary
	 */
	if (tmp_ra >= ra_max) {
		tick[VERT_0 * NUM_CRD + X_CRD] = old_x;
		tick[VERT_0 * NUM_CRD + Y_CRD] = old_y;
		tick[VERT_0 * NUM_CRD + Z_CRD] = old_z;

		convRaDecToXYZ(ra_max, old_dec, r_max, &x_st, 
			&y_st, &z_st);

		tick[VERT_1 * NUM_CRD + X_CRD ] = x_st;
		tick[VERT_1 * NUM_CRD + Y_CRD ] = y_st;
		tick[VERT_1 * NUM_CRD + Z_CRD ] = z_st;

		GEOMadd_disjoint_polygon(obj, tick, NULL, NULL, 
			NUM_VERTS_LINE, GEOM_NOT_SHARED, 
			GEOM_COPY_DATA);
		GEOMadd_float_colors(obj, cols, NUM_VERTS_LINE, 
			GEOM_COPY_DATA);

	} /* if */




	/*
	 *
	 * Bound, Arc of DEC with Min RA coordinate. N. 3
	 *
	 */
	old_ra = ra_min;
	old_dec = dec_min;
	/*
	 * calculate Cartesian coordinates corresponding to
	 * Spherical coordinates with max distance
	 */
	convRaDecToXYZ(old_ra, old_dec, r_max, &old_x, &old_y, 
		&old_z);

	/* 
	 * approximate arc of circle with a lot of segments
	 */	
	for (tmp_dec = dec_min + step_dec; tmp_dec <= dec_max; 
		tmp_dec = tmp_dec + step_dec) {

		/*
		 * first vertex of segment
		 */
		tick[VERT_0 * NUM_CRD + X_CRD] = old_x;
		tick[VERT_0 * NUM_CRD + Y_CRD] = old_y;
		tick[VERT_0 * NUM_CRD + Z_CRD] = old_z;

		/*
		 * calculate Cartesian coordinates of second
		 * vertex of segment
		 */
		convRaDecToXYZ(old_ra, tmp_dec, r_max, &x_st, 
			&y_st, &z_st);
		/*
		 * store max and min values
		 */
		maxValue(x_max, y_max, z_max, x_st, y_st, z_st);
		minValue(x_min, y_min, z_min, x_st, y_st, z_st);

		/*
		 * set second vertex of segment
		 */
		tick[VERT_1 * NUM_CRD + X_CRD ] = x_st;
		tick[VERT_1 * NUM_CRD + Y_CRD ] = y_st;
		tick[VERT_1 * NUM_CRD + Z_CRD ] = z_st;

		/*
		 * append tick object to geometry objects
		 * and define color
		 */
		GEOMadd_disjoint_polygon(obj, tick, NULL, NULL, 
			NUM_VERTS_LINE, GEOM_NOT_SHARED, 
			GEOM_COPY_DATA);
		GEOMadd_float_colors(obj, cols, NUM_VERTS_LINE, 
			GEOM_COPY_DATA);

		/*
		 * store current position
		 */
		old_x = x_st;
		old_y = y_st;
		old_z = z_st;

	} /* for */
	
	/*
	 * append part of segment if necessary
	 */
	if (tmp_dec >= dec_max) {
		tick[VERT_0 * NUM_CRD + X_CRD] = old_x;
		tick[VERT_0 * NUM_CRD + Y_CRD] = old_y;
		tick[VERT_0 * NUM_CRD + Z_CRD] = old_z;

		convRaDecToXYZ(old_ra, dec_max, r_max, &x_st, 
			&y_st, &z_st);

		tick[VERT_1 * NUM_CRD + X_CRD ] = x_st;
		tick[VERT_1 * NUM_CRD + Y_CRD ] = y_st;
		tick[VERT_1 * NUM_CRD + Z_CRD ] = z_st;

		GEOMadd_disjoint_polygon(obj, tick, NULL, NULL, 
			NUM_VERTS_LINE, GEOM_NOT_SHARED, 
			GEOM_COPY_DATA);
		GEOMadd_float_colors(obj, cols, NUM_VERTS_LINE, 
			GEOM_COPY_DATA);

	} /* if */



	/* 
	 *
	 * Bound, Arc of DEC with Max RA. N. 4
	 *
	 */
	old_ra = ra_max;
	old_dec = dec_min;
	/*
	 * calculate Cartesian coordinates corresponding to
	 * Spherical coordinates with max distance
	 */
	convRaDecToXYZ(old_ra, old_dec, r_max, &old_x, &old_y, 
		&old_z);

	/*
	 * approximate arc of circle with a lot of segments
	 */
	for (tmp_dec = dec_min + step_dec; tmp_dec <= dec_max; 
		tmp_dec = tmp_dec + step_dec) {

		/*
		 * first vertex of segment
		 */
		tick[VERT_0 * NUM_CRD + X_CRD] = old_x;
		tick[VERT_0 * NUM_CRD + Y_CRD] = old_y;
		tick[VERT_0 * NUM_CRD + Z_CRD] = old_z;

		/*
		 * calculate Cartesian coordinates of second
		 * vertex of segment
		 */
		convRaDecToXYZ(old_ra, tmp_dec, r_max, &x_st, 
			&y_st, &z_st);
		/*
		 * store max and min values
		 */
		maxValue(x_max, y_max, z_max, x_st, y_st, z_st);
		minValue(x_min, y_min, z_min, x_st, y_st, z_st);

		/*
		 * set second vertex of segment
		 */
		tick[VERT_1 * NUM_CRD + X_CRD ] = x_st;
		tick[VERT_1 * NUM_CRD + Y_CRD ] = y_st;
		tick[VERT_1 * NUM_CRD + Z_CRD ] = z_st;

		/*
		 * append tick object to geometry objects and
		 * define color
		 */
		GEOMadd_disjoint_polygon(obj, tick, NULL, NULL, 
			NUM_VERTS_LINE, GEOM_NOT_SHARED, 
			GEOM_COPY_DATA);
		GEOMadd_float_colors(obj, cols, NUM_VERTS_LINE, 
			GEOM_COPY_DATA);

		/*
		 * store current position
		 */
		old_x = x_st;
		old_y = y_st;
		old_z = z_st;

	} /* for */
	
	/* 
	 * append part of segment if necessary
	 */
	if (tmp_dec >= dec_max) {
		tick[VERT_0 * NUM_CRD + X_CRD] = old_x;
		tick[VERT_0 * NUM_CRD + Y_CRD] = old_y;
		tick[VERT_0 * NUM_CRD + Z_CRD] = old_z;

		convRaDecToXYZ(old_ra, dec_max, r_max, &x_st, 
			&y_st, &z_st);

		tick[VERT_1 * NUM_CRD + X_CRD ] = x_st;
		tick[VERT_1 * NUM_CRD + Y_CRD ] = y_st;
		tick[VERT_1 * NUM_CRD + Z_CRD ] = z_st;

		GEOMadd_disjoint_polygon(obj, tick, NULL, NULL, 
			NUM_VERTS_LINE, GEOM_NOT_SHARED, 
			GEOM_COPY_DATA);
		GEOMadd_float_colors(obj, cols, NUM_VERTS_LINE, 
			GEOM_COPY_DATA);

	} /* if */




	/* 
	 * central point of cone base 
	 */
	mid_ra = ra_min + ((ra_max - ra_min) / 2.0);
	mid_dec = dec_min + ((dec_max - dec_min) / 2.0);

	convRaDecToXYZ(mid_ra, mid_dec, r_max, &xc, &yc, &zc);


	
	
	
	
	
	
	/* 
	 *
	 * Draw 1st face, (triangle), of cone
	 *
	 */
	x0_new = X0;
	y0_new = Y0;
	z0_new = Z0;

	/*
	 * find 1st vertex of cone base
	 */
	convRaDecToXYZ(ra_max, dec_max, r_max, &x_st, &y_st, 
		&z_st);
	
	face[VERT_0 * NUM_CRD + X_CRD] = x0_new;
	face[VERT_0 * NUM_CRD + Y_CRD] = y0_new;
	face[VERT_0 * NUM_CRD + Z_CRD] = z0_new;

	face[VERT_1 * NUM_CRD + X_CRD ] = x_st;
	face[VERT_1 * NUM_CRD + Y_CRD ] = y_st;
	face[VERT_1 * NUM_CRD + Z_CRD ] = z_st;

	/*
	 * find 2nd vertex of cone base
	 */
	convRaDecToXYZ(ra_min, dec_max, r_max, &x2, &y2, &z2);


	face[VERT_2 * NUM_CRD + X_CRD] = x2;
	face[VERT_2 * NUM_CRD + Y_CRD] = y2;
	face[VERT_2 * NUM_CRD + Z_CRD] = z2;

	
	/*
	 * append 1st face of cone to geometry objects
	 * and define color
	 */
	GEOMadd_disjoint_polygon(obj, face, NULL, NULL, 
		NUM_VERTS_LINE, GEOM_NOT_SHARED, 
		GEOM_COPY_DATA);
	GEOMadd_float_colors(obj, cols, NUM_VERTS_LINE, 
		GEOM_COPY_DATA);





	
	
	
	
	/*
	 *
	 * Draw 2nd face of cone
	 *
	 */
	x1 = x2;
	y1 = y2;
	z1 = z2;

	face[VERT_0 * NUM_CRD + X_CRD] = x0_new;
	face[VERT_0 * NUM_CRD + Y_CRD] = y0_new;
	face[VERT_0 * NUM_CRD + Z_CRD] = z0_new;

	face[VERT_1 * NUM_CRD + X_CRD] = x1;
	face[VERT_1 * NUM_CRD + Y_CRD] = y1;
	face[VERT_1 * NUM_CRD + Z_CRD] = z1;


	/*
	 * find 3rd vertex of cone base
	 */
	convRaDecToXYZ(ra_min, dec_min, r_max, &x2, &y2, &z2);

	face[VERT_2 * NUM_CRD + X_CRD] = x2;
	face[VERT_2 * NUM_CRD + Y_CRD] = y2;
	face[VERT_2 * NUM_CRD + Z_CRD] = z2;


	/*
	 * append 2nd face to geometry objects and
	 * define color
	 */
	GEOMadd_disjoint_polygon(obj, face, NULL, NULL, 
		NUM_VERTS_LINE, GEOM_NOT_SHARED, 
		GEOM_COPY_DATA);
	GEOMadd_float_colors(obj, cols, NUM_VERTS_LINE, 
		GEOM_COPY_DATA);



	
	
	
	
	/*
	 *
	 * Draw 3rd face of cone
	 *
	 */
	x1 = x2;
	y1 = y2;
	z1 = z2;

	face[VERT_0 * NUM_CRD + X_CRD] = x0_new;
	face[VERT_0 * NUM_CRD + Y_CRD] = y0_new;
	face[VERT_0 * NUM_CRD + Z_CRD] = z0_new;

	face[VERT_1 * NUM_CRD + X_CRD] = x1;
	face[VERT_1 * NUM_CRD + Y_CRD] = y1;
	face[VERT_1 * NUM_CRD + Z_CRD] = z1;


	/*
	 * find 4th vertex of cone base
	 */
	convRaDecToXYZ(ra_max, dec_min, r_max, &x2, &y2, &z2);

	face[VERT_2 * NUM_CRD + X_CRD] = x2;
	face[VERT_2 * NUM_CRD + Y_CRD] = y2;
	face[VERT_2 * NUM_CRD + Z_CRD] = z2;


	/* 
	 * append 3rd face to geometry objects and
	 * define color
	 */
	GEOMadd_disjoint_polygon(obj, face, NULL, NULL, 
		NUM_VERTS_LINE, GEOM_NOT_SHARED, 
		GEOM_COPY_DATA);
	GEOMadd_float_colors(obj, cols, NUM_VERTS_LINE, 
		GEOM_COPY_DATA);



	
	
	
	
	/*
	 *
	 * Draw 4th face of cone
	 *
	 */
	
	/*
	 * store 4th vertex of cone base
	 */
	x1 = x2;
	y1 = y2;
	z1 = z2;

	face[VERT_0 * NUM_CRD + X_CRD] = x0_new;
	face[VERT_0 * NUM_CRD + Y_CRD] = y0_new;
	face[VERT_0 * NUM_CRD + Z_CRD] = z0_new;


	face[VERT_1 * NUM_CRD + X_CRD] = x1;
	face[VERT_1 * NUM_CRD + Y_CRD] = y1;
	face[VERT_1 * NUM_CRD + Z_CRD] = z1;


	face[VERT_2 * NUM_CRD + X_CRD] = x_st;
	face[VERT_2 * NUM_CRD + Y_CRD] = y_st;
	face[VERT_2 * NUM_CRD + Z_CRD] = z_st;


	/*
	 * append last face to geometry objects and
	 * define color
	 */
	GEOMadd_disjoint_polygon(obj, face, NULL, NULL, 
		NUM_VERTS_LINE, GEOM_NOT_SHARED, 
		GEOM_COPY_DATA);
	GEOMadd_float_colors(obj, cols, NUM_VERTS_LINE, 
		GEOM_COPY_DATA);


	return(TRUE);
} /* create_bounds */





/*
 *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 *                                                           *
 *              *** display_cone_bounds ***                  *
 *                                                           *
 *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*
 */
 /*
  * Set color, find number of geometry objects bounded by 
  * cone, find the range in Spherical coordinates and in 
  * Cartesian coordinates of this objects then call cone_bounds 
  * in order to display the cone.
  *
  * parameters:
  *	(i) input: galaxies
  *	(i) dimension: number of astronomical objects
  *	(i) n_ticks1: number of segment to approximate an arc 
  *			of circle
  *	(i) red_b: rgb index for the color of cone
  *	(i) green_b: rgb index for the color of cone
  *	(i) blue_b: rgb index for the color of cone
  *	(o) obj: list of geometry objects to display
  *	(o) out_cone_range: dimension range of cone
  *
  */  
int
display_cone_bounds(input, dimension, n_ticks1, red_b, 
	green_b, blue_b, obj, out_cone_range)
AVSfield_float *input;
int dimension;
float n_ticks1;
float red_b;
float green_b;
float blue_b;
GEOMobj *obj;
AVSfield_float **out_cone_range;
{
 
	float colour[3];
	float x, y, z;
	float ra, dec, r;
	float x1, y1, z1;
	float x_1st_pt, y_1st_pt, z_1st_pt;
	float x_last_pt, y_last_pt, z_last_pt;
	float ra_1st_pt, dec_1st_pt, r_1st_pt;
	float ra_last_pt, dec_last_pt, r_last_pt; 
	float ra_min, ra_max;
	float dec_min, dec_max;
	float r_max;
	float x_max, x_min;
	float y_max, y_min;
	float z_min, z_max;
	int cell;
	char debug[256];
	FILE *fp_dbg;



	/* debugOpenFile(&fp_dbg,"debug.dbg"); */
	
	/*
	 * initialize Spherical coordinates
	 */
	ra = R_MIN;
	dec = R_MIN;
	z = R_MIN;
       
	/*
	 * white: default color of cone
	 */
	colour[RED] = red_b;
	colour[GREEN] = green_b;
	colour[BLUE] = blue_b;



	/*
	 * reset coordinate range, Cartesian and Spherical
	 * coordinates
	 */
	x_max = -MAXFLOAT;
	y_max = -MAXFLOAT;
	z_max = -MAXFLOAT;
	
	x_min = MAXFLOAT;
	y_min = MAXFLOAT;
	z_min = MAXFLOAT;
	
	ra_min = MAXFLOAT;
	dec_min = MAXFLOAT;
	
	ra_max = -MAXFLOAT;
	dec_max = -MAXFLOAT;
	r_max = 0.0;



	/* sprintf(debug," x    y   z  ***   ra    dec  r  \n");
	   debugWrtFile2(fp_dbg, debug); 
	 */



	/*
	 * store coordinates of 1st object
	 */
	x_1st_pt = input->points[X_CRD * dimension + FIRST_PT];
	y_1st_pt = input->points[Y_CRD * dimension + FIRST_PT];
	z_1st_pt = input->points[Z_CRD * dimension + FIRST_PT];

	/*
	 * store coordinates of last object
	 */
	x_last_pt = input->points[X_CRD * dimension + 
		(dimension - 1)];
	y_last_pt = input->points[Y_CRD * dimension + 
		(dimension - 1)];
	z_last_pt = input->points[Z_CRD * dimension + 
		(dimension - 1)];

	/*
	 * find Spherical coordinates of 1st and last object
	 */
	convXYZToRaDec(x_1st_pt, y_1st_pt, z_1st_pt, &ra_1st_pt, 
		&dec_1st_pt, &r_1st_pt);
	convXYZToRaDec(x_last_pt, y_last_pt, z_last_pt, 
		&ra_last_pt, &dec_last_pt, &r_last_pt);


	/* 
	 * find max and min values of all astronomical objects
	 */
	for (cell = 0; cell < dimension; cell++) {
	
		x = input->points[X_CRD * dimension + cell];
		y = input->points[Y_CRD * dimension + cell];
		z = input->points[Z_CRD * dimension + cell];

		/*
		 * convert into Spherical coordinates
		 */
		convXYZToRaDec(x, y, z, &ra, &dec, &r);

		/* 
		 * debug info
		 */
		/* convRaDecToXYZ(ra, dec, r, &x1, &y1, &z1); */
		/* sprintf(debug,"%f %f %f *** %f %f %f *** \n", 
			x, y, z, ra, dec); */
		/* debugWrtFile2(fp_dbg,debug); */


		/*
		 * find max and min values for Cartesian 
		 * coordinates 
		 */
		maxValue(&x_max, &y_max, &z_max, x, y, z);
		minValue(&x_min, &y_min, &z_min, x, y, z);


		/* 
		 * find range for Spherical coordinates
		 */ 
		maxMinRaDec(&ra_max, &ra_min, &dec_max, &dec_min,
			&r_max, ra, dec, r);
		
		

       	} /* end for */



	/*
	 * find and draw cone bound and cone base
	 */
	create_bounds(obj, ra_min, ra_max, dec_min, dec_max, 
		r_max, &x_min, &x_max, &y_min, &y_max, &z_min, 
		&z_max, colour, n_ticks1);
	
	/*
	 * Trace axes, related to Cartesian coordinate range
	 */
	draw_axis(obj, x_max, y_max, z_max);

	/*
	 * set output Cartesian coordinates range
	 */
	(*out_cone_range)->points[X_MIN_POS] = x_min;
	(*out_cone_range)->points[X_MAX_POS] = x_max;
	(*out_cone_range)->points[Y_MIN_POS] = y_min;
	(*out_cone_range)->points[Y_MAX_POS] = y_max;
	(*out_cone_range)->points[Z_MIN_POS] = z_min;
	(*out_cone_range)->points[Z_MAX_POS] = z_max;


	/* debugCloseFile(fp_dbg); */
	
	return(TRUE);
} /* display_cone_bounds */
