/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/******************************************************************************/
/*
 *  Field-o-matic!!  It slices!  It dices!
 *
 *  The Field-o-matic function provides the cabability of slicing a 3D field
 *  to produce a 2D field.  The resulting 2D field may then also be cropped.
 *
 */



#include	"avs_utils.h"


static char	input_field[]		= "Input Field";
static char	input_colormap[]	= "Input Colormap";
static char	output_field[]		= "Output Field";
static char     output_geom[]		= "Output Geom";

static char	parm_show_zone[]	= "show slice zone";
static char	parm_show_pos[]		= "show slice position";
static char	parm_slice_axis[]	= "slice axis";
static char	parm_slice_coord[]	= "slice coord";
static char	parm_crop_slice[]	= "crop slice";
static char	parm_zone_cen_h[]	= "zone center h";
static char	parm_zone_cen_v[]	= "zone center v";
static char	parm_zone_rad_h[]	= "zone radius h";
static char	parm_zone_rad_v[]	= "zone radius v";

static char	choices_slice[3][6]	= {"I", "I!J", "I!J!K"};



field_o_matic_desc ()
{
    int		field_o_matic_compute ();
    int		out_port, iparm;

    AVSset_module_name ("field o matic", MODULE_MAPPER);


    AVScreate_input_port (input_field, "field", REQUIRED);
    AVScreate_input_port (input_colormap, "colormap", OPTIONAL);


    /*out_port = */ AVScreate_output_port (output_field,
                                      "field float irregular");
/*  Do NOT free output field
    AVSautofree_output (out_port);
*/
    out_port = AVScreate_output_port (output_geom, "geom");
    AVSautofree_output (out_port);


    iparm = AVSadd_parameter (parm_show_zone, "boolean", TRUE, FALSE, TRUE);
    AVSadd_parameter_prop (iparm, "width", "integer", 3);
    iparm = AVSadd_parameter (parm_show_pos, "boolean", TRUE, FALSE, TRUE);
    AVSadd_parameter_prop (iparm, "width", "integer", 3);

    AVSadd_parameter (parm_slice_coord, "integer", 1, 1, 999);
    iparm = AVSadd_parameter (parm_slice_axis, "choice", "K",
                              choices_slice[2], CHOICE_DELIMITER_STR);
    AVSadd_parameter_prop (iparm, "width", "integer", 1);
    AVSadd_parameter_prop (iparm, "columns", "integer", 1);

    iparm = AVSadd_parameter (parm_crop_slice, "boolean", FALSE, FALSE, TRUE);
    AVSadd_parameter_prop (iparm, "width", "integer", 3);
    AVSadd_parameter (parm_zone_cen_h, "integer", 1, 1, 999);
    AVSadd_parameter (parm_zone_cen_v, "integer", 1, 1, 999);
    AVSadd_parameter (parm_zone_rad_h, "integer", 1, 0, 999);
    AVSadd_parameter (parm_zone_rad_v, "integer", 1, 0, 999);

    AVSset_compute_proc (field_o_matic_compute);

    return;
}



/******************************************************************************/
/*
 *  Inform AVS (the main procedure) of this module.
 *
 */


AVSinit_modules()
{
    AVSmodule_from_desc (field_o_matic_desc);
    return ;
}



field_o_matic_compute (in_field, in_cmap, out_field, out_geom,
                       show_zone, show_pos,
                       i_slice_coord, p_slice_axis,
                       crop_slice,
                       i_zone_cen_h, i_zone_cen_v,
                       i_zone_rad_h, i_zone_rad_v)

AVSfield	*in_field;
AVScolormap	*in_cmap;
AVSfield	**out_field;
GEOMedit_list	*out_geom;
int		show_zone, show_pos;
int		i_slice_coord;
char		*p_slice_axis;
int		crop_slice;
int		i_zone_cen_h, i_zone_cen_v;
int		i_zone_rad_h, i_zone_rad_v;
{
    static char		zone_obj_name[] = "field_slice_zone";
    static char		pos_obj_name[]  = "field_slice_position";

    static AVSfield	*out_fld = NULL;

    static int		old_ndim = -1;
    static int		old_axis = -1;
    static int		old_pos[3][5] =	{ 0, -1, -1, -1, -1,
					  0, -1, -1, -1, -1,
					  0, -1, -1, -1, -1};

    register int	i, j, k, n;
    register int	mx, my, mz;
    register float	*pd, offset[3];
    register FLOAT3	*pv, xyz;
    int			ndim, nx, ny, nz, nvec;
    int			new_field, new_slice;
    int			iaxis, islice;
    int			icenter_h, icenter_v, iradius_h, iradius_v;
    int			nh, ih1, ih2, nv, jv1, jv2;
    int			ndims, dims[2], nspace;
    float		*fld_dat, *fld_pts, *fld_ptx, *fld_pty, *fld_ptz;
    FLOAT3		*verts, *colors;
    GEOMobj		*zone_obj, *pos_obj;



    *out_geom = GEOMinit_edit_list (*out_geom);



    iaxis     = AVSchoice_number (parm_slice_axis, p_slice_axis) - 1;

    new_field = AVSinput_changed (input_field, 0);

    new_slice = AVSparameter_changed (parm_crop_slice)  ||
                AVSparameter_changed (parm_slice_axis)  ||
                AVSparameter_changed (parm_slice_coord) ||
                AVSparameter_changed (parm_zone_cen_h)  ||
                AVSparameter_changed (parm_zone_cen_v)  ||
                AVSparameter_changed (parm_zone_rad_h)  ||
                AVSparameter_changed (parm_zone_rad_v);


    if (in_field == NULL) return (1);

    UTILS_field_dims_get (in_field, &ndim, &nx, &ny, &nz);
    nvec = (in_field->veclen > 1) ? in_field->veclen : 1;

    if ((ndim != old_ndim) || (AVSparameter_changed (parm_crop_slice)))
    {
        i = j = crop_slice;
        if (ndim < 3) j = FALSE;
        if (ndim < 2)
        {
            i          = FALSE;
            show_zone  = FALSE;
            show_pos   = FALSE;
            crop_slice = FALSE;
        }
        AVSparameter_visible (parm_show_zone,  (ndim >= 2));
        AVSparameter_visible (parm_show_pos,   (ndim >= 2));
        AVSparameter_visible (parm_crop_slice, (ndim >= 2));
        AVSparameter_visible (parm_zone_cen_h, i);
        AVSparameter_visible (parm_zone_cen_v, j);
        AVSparameter_visible (parm_zone_rad_h, i);
        AVSparameter_visible (parm_zone_rad_v, j);
    }

    if (ndim != old_ndim)
    {
        old_ndim = ndim;

        if (iaxis >= ndim)
        {
            iaxis           = ndim - 1;
            p_slice_axis[0] = 'I' + iaxis;
        }
        AVSmodify_parameter (parm_slice_axis, AVS_VALUE | AVS_MINVAL,
                             p_slice_axis, choices_slice[ndim-1], NULL);
    }


    islice    = i_slice_coord - 1;
    icenter_h = i_zone_cen_h - 1;
    icenter_v = i_zone_cen_v - 1;
    iradius_h = i_zone_rad_h;
    iradius_v = i_zone_rad_v;

    if (old_axis != -1)
    {
        if ((iaxis != old_axis) ||
            ((crop_slice) && (AVSparameter_changed (parm_crop_slice))))
        {
            if (iaxis != old_axis) islice = old_pos[iaxis][0];
            icenter_h = old_pos[iaxis][1];
            icenter_v = old_pos[iaxis][2];
            iradius_h = old_pos[iaxis][3];
            iradius_v = old_pos[iaxis][4];
        }
    }

    old_axis          = iaxis;
    old_pos[iaxis][0] = islice;

    if (crop_slice)
    {
        old_pos[iaxis][1] = icenter_h;
        old_pos[iaxis][2] = icenter_v;
        old_pos[iaxis][3] = iradius_h;
        old_pos[iaxis][4] = iradius_v;
    }



    k = islice;
    n = nvec;

    UTILS_field_pt_init (in_field);

    switch (iaxis)
    {
        case X_COORD :

            slice_setup (crop_slice, nx, ny, nz, &islice,
                         &icenter_h, &icenter_v, &iradius_h, &iradius_v,
                         &nh, &ih1, &ih2, &nv, &jv1, &jv2, &nvec,
                         &verts, &fld_dat, &fld_pts);

            pd = fld_dat, pv = verts;
            for (j = jv1; j <= jv2; j++)
            {
                for (i = ih1; i <= ih2; i++, pd += n, pv++)
                {
                    UTILS_field_pt_vec (k, i, j, pd);
                    UTILS_field_pt_xyz (k, i, j, pv);
                }
            }
            break;

        case Y_COORD :

            slice_setup (crop_slice, ny, nx, nz, &islice,
                         &icenter_h, &icenter_v, &iradius_h, &iradius_v,
                         &nh, &ih1, &ih2, &nv, &jv1, &jv2, &nvec,
                         &verts, &fld_dat, &fld_pts);

            pd = fld_dat, pv = verts;
            for (j = jv1; j <= jv2; j++)
            {
                for (i = ih1; i <= ih2; i++, pd += n, pv++)
                {
                    UTILS_field_pt_vec (i, k, j, pd);
                    UTILS_field_pt_xyz (i, k, j, pv);
                }
            }
            break;

        case Z_COORD :

            slice_setup (crop_slice, nz, nx, ny, &islice,
                         &icenter_h, &icenter_v, &iradius_h, &iradius_v,
                         &nh, &ih1, &ih2, &nv, &jv1, &jv2, &nvec,
                         &verts, &fld_dat, &fld_pts);

            pd = fld_dat, pv = verts;
            for (j = jv1; j <= jv2; j++)
            {
                for (i = ih1; i <= ih2; i++, pd += n, pv++)
                {
                    UTILS_field_pt_vec (i, j, k, pd);
                    UTILS_field_pt_xyz (i, j, k, pv);
                }
            }
            break;
    }

    if ((n = nh * nv) <= 0) return (1);



    if (new_field || new_slice)
    {
        if (out_fld != NULL) AVSfield_free (out_fld);

        ndims  = ndim - 1, dims[0] = nh, dims[1] = nv;
        nspace = 3;

        out_fld = UTILS_field_create (ndims, dims, nspace, nvec,
                                      AVS_TYPE_REAL, IRREGULAR);

        pd      = (float *) verts;
        fld_ptx = fld_pts, fld_pty = fld_ptx + n, fld_ptz = fld_pty + n;
        for (i = 0; i < n; i++)
        {
            *(fld_ptx++) = *(pd++);
            *(fld_pty++) = *(pd++);
            *(fld_ptz++) = *(pd++);
        }
        UTILS_field_copy_mem (out_fld, fld_dat, fld_pts);
        free (fld_dat), free (fld_pts);

        UTILS_field_extents_find (out_fld);
        UTILS_field_copy_strings (in_field, out_fld);

        *out_field = out_fld;
    }
    else
    {
        AVSmark_output_unchanged (output_field);
    }



    zone_obj = GEOMcreate_mesh (GEOM_NULL, verts, nh, nv, GEOM_COPY_DATA);

    if (!show_zone)
    {
        GEOMedit_geometry (*out_geom, zone_obj_name, zone_obj);
        GEOMedit_visibility (*out_geom, zone_obj_name, GEOM_EDIT_DELETE);
    }

    else		/* Show slice zone	*/
    {
        colors = GEOM_NULL;
        if (nvec == 1)
            UTILS_color_rgb_from_field (in_cmap, out_fld, &colors);
        GEOMadd_float_colors (zone_obj, colors, n, GEOM_COPY_DATA);

        GEOMgen_normals (zone_obj, 0);
        GEOMcvt_mesh_to_polytri (zone_obj, GEOM_SURFACE | GEOM_WIREFRAME);

        GEOMedit_geometry (*out_geom, zone_obj_name, zone_obj);
    }
    GEOMdestroy_obj (zone_obj);



    pos_obj = GEOMcreate_obj (GEOM_POLYTRI, GEOM_NULL);

    if (!show_pos)
    {
        GEOMedit_geometry (*out_geom, pos_obj_name, pos_obj);
        GEOMedit_visibility (*out_geom, pos_obj_name, GEOM_EDIT_DELETE);
    }

    else		/* Show slice position	*/
    {
        i  = icenter_h;
        j  = icenter_v;
        k  = islice;
        mx = nx - 1, my = ny - 1, mz = nz - 1;

        UTILS_field_offsets_from_surf (out_fld, offset);

        UTILS_field_pt_init (in_field);

        switch (iaxis)
        {
            case X_COORD :

                line_add (k, k, i, i, 0, mz, pos_obj, offset);
                line_add (k, k, 0, my, j, j, pos_obj, offset);
                edges_add (k, k, 0, my, 0, mz, pos_obj, offset);
                edges_add (k, k, ih1, ih2, jv1, jv2, pos_obj, offset);
                break;

            case Y_COORD :

                line_add (i, i, k, k, 0, mz, pos_obj, offset);
                line_add (0, mx, k, k, j, j, pos_obj, offset);
                edges_add (0, mx, k, k, 0, mz, pos_obj, offset);
                edges_add (ih1, ih2, k, k, jv1, jv2, pos_obj, offset);
                break;

            case Z_COORD :

                line_add (i, i, 0, my, k, k, pos_obj, offset);
                line_add (0, mx, j, j, k, k, pos_obj, offset);
                edges_add (0, mx, 0, my, k, k, pos_obj, offset);
                edges_add (ih1, ih2, jv1, jv2, k, k, pos_obj, offset);
                break;
        }

        GEOMedit_geometry (*out_geom, pos_obj_name, pos_obj);
    }
    GEOMdestroy_obj (pos_obj);

    return (1);
}



/******************************************************************************/

slice_setup (crop_slice, max_slice, max_h, max_v,
             p_islice, p_icen_h, p_icen_v, p_irad_h, p_irad_v,
             p_nh, p_ih1, p_ih2, p_nv, p_jv1, p_jv2, p_veclen,
             p_verts, p_fld_d, p_fld_p)

int		crop_slice, max_slice, max_h, max_v;
int		*p_islice, *p_icen_h, *p_icen_v, *p_irad_h, *p_irad_v;
int		*p_nh, *p_ih1, *p_ih2, *p_nv, *p_jv1, *p_jv2, *p_veclen;
FLOAT3		**p_verts;
float		**p_fld_d, **p_fld_p;
{
    int		n, mh, mv;

    mh = max_h - 1, mv = max_v - 1;

    *p_islice = keep_in_bounds (*p_islice, max_slice);

    AVSmodify_parameter (parm_slice_coord, AVS_VALUE | AVS_MAXVAL,
                         (*p_islice+1), 1, max_slice);

    if (crop_slice)
    {
        *p_icen_h = keep_in_bounds (*p_icen_h, max_h);
        *p_icen_v = keep_in_bounds (*p_icen_v, max_v);
        *p_irad_h = keep_in_bounds (*p_irad_h, max_h);
        *p_irad_v = keep_in_bounds (*p_irad_v, max_v);

        *p_ih1 = *p_icen_h - *p_irad_h;
        if (*p_ih1 < 0) *p_ih1 = 0;
        *p_ih2 = *p_icen_h + *p_irad_h;
        if (*p_ih2 >= max_h) *p_ih2 = mh;
        *p_jv1 = *p_icen_v - *p_irad_v;
        if (*p_jv1 < 0) *p_jv1 = 0;
        *p_jv2 = *p_icen_v + *p_irad_v;
        if (*p_jv2 >= max_v) *p_jv2 = mv;

        if (crop_slice > 0)
        {
            if (max_h > 1)
            {
                AVSmodify_parameter (parm_zone_cen_h, AVS_VALUE | AVS_MAXVAL,
                                     (*p_icen_h+1), 1, max_h);
                AVSmodify_parameter (parm_zone_rad_h, AVS_VALUE | AVS_MAXVAL,
                                     *p_irad_h, 0, mh);
            }

            if (max_v > 1)
            {
                AVSmodify_parameter (parm_zone_cen_v, AVS_VALUE | AVS_MAXVAL,
                                     (*p_icen_v+1), 1, max_v);
                AVSmodify_parameter (parm_zone_rad_v, AVS_VALUE | AVS_MAXVAL,
                                     *p_irad_v, 0, mv);
            }
        }
    }
    else
    {
        *p_icen_h = *p_irad_h = max_h / 2;
        *p_icen_v = *p_irad_v = max_v / 2;

        *p_ih1 = 0, *p_ih2 = mh;
        *p_jv1 = 0, *p_jv2 = mv;
    }

    *p_nh = *p_ih2 - *p_ih1 + 1;
    *p_nv = *p_jv2 - *p_jv1 + 1;

    n = (*p_nh) * (*p_nv);

    *p_verts = (FLOAT3 *) malloc (n * sizeof (FLOAT3));
    *p_fld_d = (float *) malloc (n * (*p_veclen) * sizeof (float));
    *p_fld_p = (float *) malloc (n * 3 * sizeof (float));

    return;
}



/******************************************************************************/

int keep_in_bounds (i, imax)

int		i, imax;
{

    if (i >= imax) return (imax - 1);
    if (i < 0) return (imax / 2);

    return (i);
}



/******************************************************************************/

edges_add (ix1, ix2, iy1, iy2, iz1, iz2, p_obj, offset)

int		ix1, ix2, iy1, iy2, iz1, iz2;
GEOMobj		*p_obj;
float		*offset;
{

    if (ix1 != ix2)
    {
        line_add (ix1, ix2, iy1, iy1, iz1, iz1, p_obj, offset);
        line_add (ix1, ix2, iy2, iy2, iz2, iz2, p_obj, offset);
    }

    if (iy1 != iy2)
    {
        line_add (ix1, ix1, iy1, iy2, iz1, iz1, p_obj, offset);
        line_add (ix2, ix2, iy1, iy2, iz2, iz2, p_obj, offset);
    }

    if (iz1 != iz2)
    {
        line_add (ix1, ix1, iy1, iy1, iz1, iz2, p_obj, offset);
        line_add (ix2, ix2, iy2, iy2, iz1, iz2, p_obj, offset);
    }

    return;
}




/******************************************************************************/

line_add (ix1, ix2, iy1, iy2, iz1, iz2, p_obj, offset)

int		ix1, ix2, iy1, iy2, iz1, iz2;
GEOMobj		*p_obj;
float		*offset;
{
    register int	i, n;
    register FLOAT3	*line_pts;

    n        = 0;
    line_pts = NULL;

    if (ix1 != ix2)
    {
        n = ix2 - ix1 + 1;
        line_pts = (FLOAT3 *) malloc (n * sizeof (FLOAT3));

        for (i = ix1, n = 0; i <= ix2; i++, n++)
            UTILS_field_pt_xyz (i, iy1, iz1, (line_pts+n));
    }

    if (iy1 != iy2)
    {
        n = iy2 - iy1 + 1;
        line_pts = (FLOAT3 *) malloc (n * sizeof (FLOAT3));

        for (i = iy1, n = 0; i <= iy2; i++, n++)
            UTILS_field_pt_xyz (ix1, i, iz1, (line_pts+n));
    }

    if (iz1 != iz2)
    {
        n = iz2 - iz1 + 1;
        line_pts = (FLOAT3 *) malloc (n * sizeof (FLOAT3));

        for (i = iz1, n = 0; i <= iz2; i++, n++)
            UTILS_field_pt_xyz (ix1, iy1, i, (line_pts+n));
    }

    if (n > 0)
    {
        for (i = 0; i < n; i++)
        {
            line_pts[i][0] += offset[0];
            line_pts[i][1] += offset[1];
            line_pts[i][2] += offset[2];
        }
        GEOMadd_polyline (p_obj, line_pts, GEOM_NULL, n, GEOM_COPY_DATA);

        free (line_pts);
    }


    return;
}
