/*******************************************************************************
 *
 *  These are a set of utility geometry object routines that can be used by 
 *  user-written AVS modules.
 *
 *	29 Mar 91  Phil McDonald, NOAA/ERL/FSL	Original version.
 *
 *	26 Aug 91  Phil McDonald		Add edlist routines.
 *
 *	04 Dec 91  Phil McDonald		Add edlist_visibility_set.
 *
 *	17 Mar 92  Phil McDonald		Add destroy funcs.
 *
 *	19 Mar 92  Phil McDonald		Copy and destroy funcs moved
 *						to geom_copy and geom_destroy.
 *
 *	20 Mar 92  Phil McDonald		Add polyline funcs.
 *
 *	16 Feb 93  Phil McDonald		Remove "norm" funcs.  Add
 *						freeze funcs.
 *
 ******************************************************************************/



#include	"avs_utils.h"



/*******************************************************************************
 *
 *  Mark all of the objects in the edit list of children as children of 
 *  the parent object named in the parent edit list.  If no parent edit list
 *  or no parent name is given, the children are adopted by the top level
 *  object.
 *
 */


void	UTILS_geom_edlist_adopt (parent_edlist, parent_name, children_edlist)

GEOMedit_list	parent_edlist;
char		*parent_name;
GEOMedit_list	children_edlist;
{
    register int	has_parent;
    register char	*name;
    register GEOMedit	*list, *list2;

    if (children_edlist == GEOM_NULL) return;

    if (parent_edlist == GEOM_NULL)

        name = parent_name;

    else
    {
        name = parent_edlist->l->name;

        list = parent_edlist->l;
        while (list->next != GEOM_NULL) list = list->next;
        list->next = children_edlist->l;
    }

    list = children_edlist->l;
    while (list != GEOM_NULL)
    {
        if (list->type == GEOM_EDIT_GEOMETRY)
        {
            has_parent = FALSE;
            list2      = children_edlist->l;
            while (list2 != GEOM_NULL)
            {
                if (list2->type == GEOM_EDIT_PARENT)
                {
                    if (strcmp (list->name, list2->name) == 0)
                    {
                        has_parent = TRUE;
                        list2      = GEOM_NULL;
                    }
                }
                if (list2 != GEOM_NULL) list2 = list2->next;
            }
            if (!has_parent)
                GEOMedit_parent (children_edlist, list->name, name);
        }
        list = list->next;
    }

    return;
}



/*******************************************************************************
 *
 *  Mark for deletion all of the objects in the specified edit list.
 *
 */

void	UTILS_geom_edlist_delete (edlist)

GEOMedit_list	edlist;
{
    GEOMedit		*list;

    if (edlist != GEOM_NULL)
    {
        list = edlist->l;
        while (list != GEOM_NULL)
        {
            if (list->type == GEOM_EDIT_GEOMETRY)
                 GEOMedit_visibility (edlist, list->name, GEOM_EDIT_DELETE);
            list = list->next;
        }
    }

    return;
}



/*******************************************************************************
 *
 *  Set all of the object names in an edit list to the specified name.
 *
 */

void	UTILS_geom_edlist_name_set (edlist, new_name)

GEOMedit_list	edlist;
char		*new_name;
{
    GEOMedit		*list;

    if (edlist != GEOM_NULL)
    {
        list = edlist->l;
        while (list != GEOM_NULL)
        {
            if (list->name != GEOM_NULL) free (list->name);
            list->name = new_name;
            list       = list->next;
        }
    }

    return;
}



/*******************************************************************************
 *
 *  Set the visibility of an edit list.
 *
 */

void	UTILS_geom_edlist_visibility_set (edlist, vis)

GEOMedit_list	edlist;
int		vis;
{
    register GEOMedit	*list;

    if (edlist == GEOM_NULL) return;

    list = edlist->l;
    while (list != GEOM_NULL)
    {
        if (list->type == GEOM_EDIT_VISIBILITY) *(int *)list->data = vis;
        list = list->next;
    }

    return;
}




/*******************************************************************************
 *
 *  Initialize geometry freezing parameters.
 *
 */

float	*UTILS_geom_freeze_check (freeze, upstream_xform, edlist)

int			freeze;
upstream_transform	*upstream_xform;
GEOMedit_list		edlist;
{
    static int		not_notified = TRUE;

    int			buttons;

    if (freeze)
    {
        if (freeze_xform == NULL)
        {
            if (not_notified)
            {
                buttons = BUTTON_UP | BUTTON_DOWN;

                not_notified = FALSE;
            }
            else
            {
                buttons = BUTTON_UP;

                if (upstream_xform != NULL)
                {
                    if (strcmp (upstream_xform->object_name, "top") == 0)
                    {
                        freeze_xform = &freeze_mat[0][0];
                        mat_identity (freeze_xform);

                        frozen_xform = &frozen_mat[0][0];
                        mat_copy (frozen_xform, upstream_xform->msxform);
                    }
                }
            }

            if (edlist != GEOM_NULL)
            {
                GEOMedit_transform_mode (edlist, "%top", "notify", buttons);
            }
        }
        else
        {
            mat_inverse  (freeze_xform, upstream_xform->msxform);
            mat_multiply (frozen_xform, freeze_xform, freeze_xform);
        }
    }
    else
    {
        if (freeze_xform != NULL)
        {
            freeze_xform = frozen_xform = NULL;

            not_notified = TRUE;

            if (edlist != GEOM_NULL)
            {
                GEOMedit_transform_mode (edlist, "%top", "normal", 0);
            }
        }
    }


    return (freeze_xform);
}



/*******************************************************************************
 *
 *  Convert the coordinates of a point to their frozen values.
 *
 */

void		 UTILS_geom_freeze_xyz (xyz, user_xform)

FLOAT3		xyz;
float		*user_xform;
{

    float	*xform;
    FLOAT3	tmp;

    if (xyz != NULL)
    {
        xform = (user_xform != NULL) ? user_xform : freeze_xform;

        if (xform != NULL)
        {
            UTILS_float3_set (0.0, 0.0, 0.0, tmp);
            mat_vecmul (tmp, xform);
            mat_vecmul (xyz, xform);
            UTILS_float3_sub (xyz, tmp);
        }
    }

    return;
}



/*******************************************************************************
 *
 *  Add a point on a polyline to a polyline object.
 *
 */

int	UTILS_geom_polyline_add_pt (line_obj, xyz, colors)

GEOMobj		*line_obj;
FLOAT3		xyz, *colors;
{
    static int		npts = 0;
    static FLOAT3	*pts = NULL, *clrs = NULL;
    static GEOMobj	*obj = NULL;


    if (xyz == NULL)
    {
        if (npts > 1)
            GEOMadd_polyline (obj, pts, clrs, npts, GEOM_COPY_DATA);
        if (line_obj != NULL)
        {
            if (pts == NULL)
                pts = (FLOAT3 *) malloc (GEOM_MAX_VERTS * sizeof (FLOAT3));
        }
        else
        {
            if (pts != NULL) free (pts);
            pts = NULL;
        }
        obj  = line_obj;
        clrs = colors;
        npts = 0;
    }
    else
    {
        UTILS_float3_copy (xyz, (pts+npts));
        npts++;
        if (npts >= GEOM_MAX_VERTS)
        {
            GEOMadd_polyline (obj, pts, clrs, npts, GEOM_COPY_DATA);
            UTILS_float3_copy ((pts+npts-1), pts);
            npts = 1;
        }
    }


    return (npts);
}



/*******************************************************************************
 *
 *  Create a polyline object from vertices in and ASCII text file.
 *
 */

GEOMobj		*UTILS_geom_polyline_from_text_file (file_name, colors)

char		*file_name;
FLOAT3		*colors;
{

    int			ifrst, ichr;
    FLOAT3		xyz;
    FILE		*vec_file;
    GEOMobj		*obj;

    vec_file = UTILS_file_name_check (&file_name, "r",
                                      "The ASCII vector file");

    if (vec_file == NULL) return (GEOM_NULL);

    obj = GEOMcreate_obj (GEOM_POLYTRI, GEOM_NULL);

    while (fscanf (vec_file, "%d", &ifrst) != EOF)
    {
        if (ifrst) UTILS_geom_polyline_add_pt (obj, NULL, colors);

        fscanf (vec_file, "%f%f", &xyz[0], &xyz[1]);

        xyz[2] = 0.0;
        ichr = fgetc (vec_file);
        while (ichr != '\n')
        {
            ichr = fgetc (vec_file);
            if (isgraph (ichr))
            {
                fseek (vec_file, -1L, 1);
                fscanf (vec_file, "%f", &xyz[2]);
                ichr = 0;
            }
        }

        UTILS_geom_polyline_add_pt (obj, xyz, colors);
    }
    UTILS_geom_polyline_add_pt (NULL, NULL, NULL);

    fclose (vec_file);



    return (obj);
}



/*******************************************************************************
 *
 *  Extract the vertices of a triangle from the vertices of a polytriangle.
 *
 */

void	UTILS_geom_polytri_tri_get (pverts_in, pcolors_in, pn_in, pi_in,
                                    pverts_out, pcolors_out, pn_out)

FLOAT3		*pverts_in, *pcolors_in, *pverts_out, *pcolors_out;
int		*pn_in, *pi_in, *pn_out;

{

    register FLOAT3	*pv_in, *pv_out, *pv_o, *pc_in, *pc_out;
    register int	i_in, n_in, n_out, n_get;

    i_in  = *pi_in;
    n_in  = *pn_in;

    pv_in  = pverts_in + i_in;
    pv_out = pverts_out;
    pv_o   = NULL;
    pc_in  = NULL;
    if ((pcolors_in != NULL) && (pcolors_out != NULL))
    {
        pc_in  = pcolors_in + i_in;
        pc_out = pcolors_out;
    }

    n_out = 0;
    n_get = 3;
    while (n_out < n_get)
    {
        if (UTILS_float3_are_same (pv_in, pv_o))
        {
            n_get = n_out;
            i_in--;
        }
        else
        {
            UTILS_float3_copy (pv_in, pv_out);
            pv_o = pv_out;
            pv_out++;
            n_out++;
            if (n_out == 3) i_in -= 2;
            if (pc_in != NULL)
            {
                UTILS_float3_copy (pc_in, pc_out);
                pc_out++;
            }
        }
        pv_in++;
        if (pc_in != NULL) pc_in++;
        i_in++;
        if (i_in == n_in) n_get = 0;
    }

    n_out = (n_out != 3) ? 0 :
            (UTILS_float3_are_same (pverts_out, (pverts_out+2))) ? 0 : n_out;

    *(pi_in)  = i_in;
    *(pn_out) = n_out;

    return;
}
