/****************************************************************************/
/*
 *  These functions can be used to print station data and read from FSL
 *  Aviation Division ASCII files.
 */



#include	"sta_data.h"



/****************************************************************************/
/*
 *  Print profiler station data to a file.
 */

int	sta_data_print_prf (file, sta_id, sta_name, sta_nlev, sta_data)

FILE	*file;
int	sta_id;
char	*sta_name;
int	sta_nlev;
float	*sta_data;
{

    register int	i = 0, j;

    fprintf (file, "%d %s \n", sta_id, sta_name);
    for (i = 0, j = 0; j < sta_nlev; j++)
    {
        fprintf (file, "\t\t%6.0f", sta_data[i++]);
        fprintf (file, "%6.1f%6.1f%6.1f", sta_data[i++],
                sta_data[i++], sta_data[i++]);
        fprintf (file, "%6.2f%6.2f%6.2f", sta_data[i++],
                sta_data[i++], sta_data[i++]);
        fprintf (file, "\n");
    }


    return (1);
}



/****************************************************************************/
/*
 *  Print SAO (surface) station data to a file.
 */

int	sta_data_print_sao (file, sta_id, sta_name, sta_nlev, sta_data)

FILE	*file;
int	sta_id;
char	*sta_name;
int	sta_nlev;
float	*sta_data;
{

    register int	i = 0, j;
    register char	str[6];

    fprintf (file, "%d %s %d \n", sta_id, sta_name, sta_nlev);
    fprintf (file, "\t\twind:  %6.1f %6.1f %6.0f %6.0f \n",
             sta_data[i++], sta_data[i++],
             sta_data[i++], sta_data[i++]);
    fprintf (file, "\t\ttemp:  %6.1f %6.1f \n",
             sta_data[i++], sta_data[i++]);
    fprintf (file, "\t\tpres:  %6.2f \n", sta_data[i++]);

    sta_data_str_unpack ((sta_data+(i++)), str);
    fprintf (file, "\t\tprec:  %6s %6.2f %6.2f \n",
             str,
             sta_data[i++],
             sta_data[23]);

    sta_data_str_unpack ((sta_data+10), str);
    fprintf (file, "\t\tvis :  %6.0f %6s \n",
             sta_data[10],
             str);
    i = 11;

    fprintf (file, "\t\tceil:  %6.0f\n", sta_data[i++]);
    for (j = 0; j < 5; j++)
    {
        sta_data_str_unpack ((sta_data+i+1), str);
        fprintf (file, "\t\tcld%d:  %6.0f %6s \n",
                 j, sta_data[i++],
                 str);
        if (sta_data[++i] == -99.0) j = 5;
    }
    i = 22;
    fprintf (file, "\t\tcldt:  %6.0f \n", sta_data[i++]);


    return (1);
}



/****************************************************************************/
/*
 *  Print upper air station data to a file.
 */

int	sta_data_print_upa (file, sta_id, sta_name, sta_nlev, sta_data)

FILE	*file;
int	sta_id;
char	*sta_name;
int	sta_nlev;
float	*sta_data;
{

    register int	i = 0, j;

    fprintf (file, "%d %s \n", sta_id, sta_name);
    for (i = 0, j = 0; j < sta_nlev; j++)
    {
        fprintf (file, "\t\t%6.0f%2.0f%6.0f", sta_data[i++],
                sta_data[i++], sta_data[i++]);
        fprintf (file, "%6.1f%6.1f", sta_data[i++],
                sta_data[i++]);
        fprintf (file, "%6.1f%6.1f", sta_data[i++],
                sta_data[i++]);
        fprintf (file, "\n");
    }


    return (1);
}



/****************************************************************************/
/*
 *  Read a data atom from the current station data file.
 */

int	sta_data_read_atom (nchar, p_data)

int		nchar;
float		*p_data;
{
    static char		f_fmt[] = "%f";
    static char		c_fmt[] = "%*c%xc";

    register int	iret = 0;
    register char	*p_c;

    if (p_data != NULL)
    {
        if (nchar <= 0)
        {
            *p_data = 0.0;
            iret    = fscanf (sta_data_file, f_fmt, p_data);
        }
        else
        {
            if (nchar <= 9)
            {
                p_c        = (char *) malloc (nchar + 1);
                c_fmt[4]   = nchar + 48;
                iret       = fscanf (sta_data_file, c_fmt, p_c);
                p_c[nchar] = '\0';
                sta_data_str_pack (p_c, p_data);
                free (p_c);
            }
        }
        if (iret == EOF) iret = 0;
    }


    return (iret);
}



/****************************************************************************/
/*
 *  Close the current station data file.
 */

int	sta_data_read_done ()

{

    if (sta_data_file != NULL)
    {
        fclose (sta_data_file);
        sta_data_file = NULL;
    }


    return (1);
}



/****************************************************************************/
/*
 *  Read through the newline at the end of the current record in the current
 *  station data file.
 */

int	sta_data_read_endl ()

{
    register int	i, iret = 1;

    do
    {
        i = fgetc (sta_data_file);
        if (i == EOF)
        {
            i    = '\n';
            iret = EOF;
        }
    } while (i != '\n');


    return (iret);
}



/****************************************************************************/
/*
 *  Initialize reading from a station data file.  If file_name is not null,
 *  try to open the file, otherwise, rewind the current station file.
 */

int	sta_data_read_init (file_name)

char	*file_name;
{
    register int	i;
    register char	*ext;

    if (file_name == NULL)
    {
        if (sta_data_file == NULL)

            sta_data_type = STA_DATA_TYPE_BAD;

        else
        {
            rewind (sta_data_file);
            sta_data_read_endl ();
        }
        return (sta_data_type);
    }

    sta_data_type = STA_DATA_TYPE_BAD;

    if ((sta_data_file = fopen (file_name, "r")) == NULL)
        return (sta_data_type);

    if ((i = strlen (file_name) - 4) > 0)
    {
        ext = file_name + i;
        sta_data_type = (!strcmp (ext, STA_DATA_EXT_SAO)) ? STA_DATA_TYPE_SAO :
                        (!strcmp (ext, STA_DATA_EXT_PRF)) ? STA_DATA_TYPE_PRF :
                        (!strcmp (ext, STA_DATA_EXT_UPA)) ? STA_DATA_TYPE_UPA :
                        STA_DATA_TYPE_BAD;
    }

    if (sta_data_type == STA_DATA_TYPE_BAD) sta_data_read_done ();
        

    return (sta_data_type);
}



/****************************************************************************/
/*
 *  Read the station id, name, and number of levels from the current station
 *  data file.
 */

int	sta_data_read_name (p_sta_id, p_sta_name, p_sta_nlev)

int		*p_sta_id;
char		*p_sta_name;
int		*p_sta_nlev;
{
    int		i;

    if (sta_data_file == NULL) return (0);
    if (p_sta_id      == NULL) return (0);
    if (p_sta_name    == NULL) return (0);
    if (p_sta_nlev    == NULL) return (0);

    *p_sta_id     = -1;
    p_sta_name[0] = '\0';
    *p_sta_nlev   = 0;

    if ((i = fscanf (sta_data_file, "%d%s", p_sta_id, p_sta_name)) != 2)
        return (EOF);

    *p_sta_nlev = 1;

    if (sta_data_type != STA_DATA_TYPE_SAO)
    {
        if ((i = fscanf (sta_data_file, "%d", p_sta_nlev)) != 1) return (EOF);
        if (sta_data_read_endl () == EOF) return (EOF);
        i = 3;
    }


    return (i);
}



/****************************************************************************/
/*
 *  Read the data for the next profiler station from the current station
 *  data file.
 */

int	sta_data_read_prf (p_sta_id, p_sta_name, p_sta_nlev, p_p_sta_data)

int	*p_sta_id;
char	*p_sta_name;
int	*p_sta_nlev;
float	**p_p_sta_data;
{

    register int	i, n, iret;
    register float	*p_dat, *p_data;

    *p_p_sta_data = NULL;

    if ((iret = sta_data_read_name (p_sta_id, p_sta_name, p_sta_nlev)) <= 0)
        return (iret);

    n     = *p_sta_nlev;
    p_dat = p_data = (float *) calloc ((n * STA_DATA_NDAT_PRF), sizeof (float));

    for (i = 0; i < n; i++)
    {
        sta_data_read_atom (0, p_dat++);		/* ht		*/
        sta_data_read_atom (0, p_dat++);		/* wind_u	*/
        sta_data_read_atom (0, p_dat++);		/* wind_v	*/
        sta_data_read_atom (0, p_dat++);		/* wind_w	*/
        sta_data_read_atom (0, p_dat++);		/* mom2_e	*/
        sta_data_read_atom (0, p_dat++);		/* mom2_n	*/
        sta_data_read_atom (0, p_dat++);		/* mom2_z	*/

        if ((iret = sta_data_read_endl ()) == EOF) i = n;
    }

    if (iret == EOF)
        free (p_data);
    else
        *p_p_sta_data = p_data;


    return (iret);
}



/****************************************************************************/
/*
 *  Read the data for the next SAO (surface) station from the current station
 *  data file.
 */

int	sta_data_read_sao (p_sta_id, p_sta_name, p_sta_nlev, p_p_sta_data)

int	*p_sta_id;
char	*p_sta_name;
int	*p_sta_nlev;
float	**p_p_sta_data;
{

    register int	i, iret;
    register float	*p_dat, *p_data;

    *p_p_sta_data = NULL;

    if ((iret = sta_data_read_name (p_sta_id, p_sta_name, p_sta_nlev)) <= 0)
        return (iret);

    p_dat = p_data = (float *) calloc (STA_DATA_NDAT_SAO, sizeof (float));

    sta_data_read_atom (0, p_dat++);			/* wind_u	*/
    sta_data_read_atom (0, p_dat++);			/* wind_v	*/
    sta_data_read_atom (0, p_dat++);			/* wind_dir	*/
    sta_data_read_atom (0, p_dat++);			/* wind_spd	*/
    sta_data_read_atom (0, p_dat++);			/* temp_f	*/
    sta_data_read_atom (0, p_dat++);			/* temp_dew_f	*/
    sta_data_read_atom (0, p_dat++);			/* press_in_hg	*/

    sta_data_read_atom (5, p_dat++);			/* prec_type	*/
    sta_data_read_atom (0, p_dat++);			/* prec_1hr	*/

    sta_data_read_atom (0, p_dat++);			/* vis		*/
    sta_data_read_atom (5, p_dat++);			/* vis_obst	*/

    sta_data_read_atom (0, p_dat++);			/* ceil_ht	*/
    for (i = 0;	i < 5; i++)
    {
        sta_data_read_atom (0, p_dat++);		/* cloud_ht	*/
        sta_data_read_atom (4, p_dat++);		/* cloud_amt	*/
    }
    sta_data_read_atom (0, p_dat++);			/* cloud_top	*/

    *p_dat = -9.99;
    iret   = 1;
    do
    {
        i = fgetc (sta_data_file);
        if (i == EOF)
        {
            iret = EOF;
            i    = '\n';
        }
        else if (isgraph (i))
        {
            fseek (sta_data_file, -1L, 1);
            sta_data_read_atom (0, p_dat++);		/* prec_3hr	*/
        }
    } while (i != '\n');

    if (iret == EOF)
    {
        free (p_data);
    }
    else
        *p_p_sta_data = p_data;


    return (iret);
}



/****************************************************************************/
/*
 *  Read the data for the next upper air station from the current station
 *  data file.
 */

int	sta_data_read_upa (p_sta_id, p_sta_name, p_sta_nlev, p_p_sta_data)

int	*p_sta_id;
char	*p_sta_name;
int	*p_sta_nlev;
float	**p_p_sta_data;
{

    register int	i, n, iret;
    register float	*p_dat, *p_data;

    *p_p_sta_data = NULL;

    if ((iret = sta_data_read_name (p_sta_id, p_sta_name, p_sta_nlev)) <= 0)
        return (iret);

    n     = *p_sta_nlev;
    p_dat = p_data = (float *) calloc ((n * STA_DATA_NDAT_UPA), sizeof (float));

    for (i = 0; i < n; i++)
    {
        sta_data_read_atom (0, p_dat++);		/* press_mb	*/
        sta_data_read_atom (0, p_dat++);		/* mand_flg	*/
        sta_data_read_atom (0, p_dat++);		/* ht_geop	*/
        sta_data_read_atom (0, p_dat++);		/* temp_c	*/
        sta_data_read_atom (0, p_dat++);		/* temp_dew_c	*/
        sta_data_read_atom (0, p_dat++);		/* wind_u	*/
        sta_data_read_atom (0, p_dat++);		/* wind_v	*/

        if ((iret = sta_data_read_endl ()) == EOF) i = n;
    }

    if (iret == EOF)
        free (p_data);
    else
        *p_p_sta_data = p_data;


    return (iret);
}



/****************************************************************************/
/*
 *  Pack a string (6 or fewer characters long) into a 4-byte variable.
 */

int	sta_data_str_pack (str, p_data)

char	*str;
long	*p_data;
{

    register int	i = 0, ishift = 0, ichr;

    *p_data = 0;

    while ((ichr = str[i++]) != '\0')
    {
        ichr = (ichr < 33) ? 0x1f :
               (ichr < 64) ? ichr - 16 :
               (ichr > 96) ? ichr - 96 : ichr - 64;

        *p_data |= ichr << ishift;

        ishift += 5;
    }


    return (1);
}



/****************************************************************************/
/*
 *  Unpack a string (6 or fewer characters long) from a 4-byte variable.
 */

int	sta_data_str_unpack (p_data, str)

long	*p_data;
char	*str;
{

    register int	i = 0, ichr;
    register long	data;

    data = *p_data;

    while (data != 0)
    {
        ichr     = data & 0x1f;
        str[i++] = (ichr == 0x1f) ? 32 : (ichr < 27) ? ichr + 64 : ichr + 16;

        data >>= 5;
    }

    while ((--i >= 0) && (!isgraph (str[i])));

    str[++i] = '\0';


    return (1);
}
