/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/*	field_to_map: reads a uniform field draws a boundary, annotates and 
		      outputs as a geometry.

	Author:	Andrew Bingham
		Department of Earth Sciences
		University of Oxford 
		1991.

		andyb@uk.ac.ox.earth
*/

#include <stdio.h>
/* IAC CODE CHANGE : #include <math.h> */
#include <avs/avs_math.h>
#include <avs/avs.h>
#include <avs/port.h>
#include <avs/field.h>
#include <avs/geom.h>
#include <avs/colormap.h>

#define LINEAR 0
#define MERCATOR 1
#define DTR (M_PI/180.0)

static int no_vertex;
static float *point_list;

void get_extents();
void get_data();
void get_colour_scale();
void get_cols();
void get_lower_boundary();
void get_upper_boundary();
void get_left_boundary();
void get_right_boundary();
float project_x(), project_y();
void hsva_to_argb();
int AddPoint();

/*****************************************/
/*  Module Specification             	 */
/*****************************************/
field_2_map_spec () 
{
	int field_2_map_compute();

	AVSset_module_name("field to map", MODULE_MAPPER);

	/* Input Port Specifications */
	AVScreate_input_port("input", 
			"field 2D scalar float 2-coord uniform", 
			REQUIRED);

	AVScreate_input_port("colourmap", "colormap", REQUIRED);

	/* Output Port Specifications */
	AVScreate_output_port("map", "geom");

	/* Parameter Specifications */
	AVSconnect_widget(AVSadd_parameter("number x ticks","integer", 2, 2, 20),"idial");
	AVSconnect_widget(AVSadd_parameter("number y ticks","integer", 2, 2, 20),"idial");
	AVSconnect_widget(AVSadd_float_parameter("z scale", 0.1, FLOAT_UNBOUND, FLOAT_UNBOUND),"dial");
	AVSconnect_widget(AVSadd_float_parameter("boundary width", 1.0, 0.0, 10.0),"dial");
	AVSconnect_widget(AVSadd_float_parameter("text height", 0.05, 0.0, 1.0),"dial");
	AVSconnect_widget(AVSadd_parameter("mode", "choice", "linear", "linear!mercator", "!"), "radio_buttons");

	AVSset_compute_proc(field_2_map_compute);
}


/*****************************************/
/* Module Compute Routine 	         */
/*****************************************/
int field_2_map_compute(input, cmap, list, no_x_ticks, no_y_ticks, zscale, 
			boundary_w, text_h, mode)
AVSfield_float *input;
AVScolormap *cmap;
GEOMedit_list *list;
int no_x_ticks, no_y_ticks;
float *zscale;
float *boundary_w;
float *text_h;
char *mode;
{
	GEOMobj *objD, *objB0, *objB1, *objB2, *objB3, *objT, *objL0, *objL1;
	GEOMobj *objC, *objCL;
	float extent[6], *data, rgb[256*3], *bufcols, *buffer, scale=1.0;
	float lower, upper, hue[256], sat[256], value[256], alpha[256];
	float data_val[256*2*3], rgb_val[256*2*3], dx, dy;
	int size, flags, project, i, font=5;
	float offset[3], ref[3], xdeg, ydeg, x, y, b_w, lat;
	char text[5], *outbuf, *errbuf;

	if (!strcmp(mode,"linear"))		project = LINEAR;
	else if (!strcmp(mode,"mercator"))	project = MERCATOR;

	b_w = (*boundary_w)*DTR;
	
	get_extents(extent, input, scale, b_w, *text_h, project);

	data = (float *)malloc(MAXX(input)*MAXY(input)*3*sizeof(float));
	get_data(input, data, scale, zscale, project);

	/* Create the data object */
	objD = GEOMcreate_mesh(extent, data, MAXX(input), MAXY(input),
				GEOM_COPY_DATA);
	GEOMgen_normals(objD,0);

	/* read colormap and convert hsv model to argb model */
	AVScolormap_get(cmap,256,&size,&lower,&upper,hue,sat,value,alpha);
	hsva_to_argb(cmap, rgb);
	bufcols = (float *)malloc(3*sizeof(float)*MAXX(input)*MAXY(input));
	get_cols(input, rgb, bufcols, lower, upper);
	GEOMadd_float_colors(objD,bufcols,MAXX(input)*MAXY(input),
				GEOM_COPY_DATA);

	
	/* create boundary */
	buffer = (float *)malloc(4*3*sizeof(float));
	get_lower_boundary(input, buffer, scale, b_w, project);
	objB0 = GEOMcreate_mesh(extent,buffer,2,2,GEOM_COPY_DATA);
	GEOMgen_normals(objB0,0);

	get_upper_boundary(input, buffer, scale, b_w, project);
	objB1 = GEOMcreate_mesh(extent,buffer,2,2,GEOM_COPY_DATA);
	GEOMgen_normals(objB1);

	get_left_boundary(input, buffer, scale, b_w, project);
	objB2 = GEOMcreate_mesh(extent,buffer,2,2,GEOM_COPY_DATA);
	GEOMgen_normals(objB2);

	get_right_boundary(input, buffer, scale, b_w, project);
	objB3 = GEOMcreate_mesh(extent,buffer,2,2,GEOM_COPY_DATA);
	GEOMgen_normals(objB3);

	/* create tick marks and label */
	no_vertex = 0;
	objT = GEOMcreate_obj(GEOM_POLYTRI, extent);

	dx = (float)fabs(input->points[1] - input->points[0]) / 
						(float)(no_x_ticks - 1);
	dy = (float)fabs(input->points[2] - input->points[3]) / 
						(float)(no_y_ticks - 1);

	point_list = (float *)
		malloc(3*4*((no_x_ticks)+(no_y_ticks))*sizeof(float));

	flags = GEOMcreate_label_flags(font,0,1,1,GEOM_LABEL_CENTER,0);
	objL0 = GEOMcreate_label(extent, flags);	
	offset[0] = offset[1] = offset[2] = 0.0;
	xdeg = input->points[0];
	ydeg = input->points[3];
	for (i=0; i<no_x_ticks; i++){
		x = project_x(xdeg, scale, project);
		y = project_y(ydeg, scale, project);
		AddPoint(x, y, 0.0);
		AddPoint(x, y-b_w, 0.0);
		sprintf(text,"%3.1f", xdeg);
		ref[0] = x;
		ref[1] = y-b_w-(*text_h);
		ref[2] = 0.0;
		GEOMadd_label(objL0,text,ref,offset,(*text_h),GEOM_NULL,-1);
		y = project_y(input->points[2], scale, project);
		AddPoint(x, y, 0.0);
		AddPoint(x, y+b_w, 0.0);
		xdeg += dx;
	}
		
	flags = GEOMcreate_label_flags(font,0,1,1,GEOM_LABEL_RIGHT,0);
	objL1 = GEOMcreate_label(extent, flags);
	xdeg = input->points[0];
	ydeg = input->points[3];
	for (i=0; i<no_y_ticks; i++){
		x = project_x(xdeg,scale, project);
		y = project_y(ydeg,scale, project);
		AddPoint(x, y, 0.0);
		AddPoint(x-b_w, y, 0.0);
		sprintf(text,"%3.1f", ydeg);
		ref[0] = x - b_w; 
		ref[1] = y;
		ref[2] = 0.0;
		GEOMadd_label(objL1,text,ref,offset,(*text_h),GEOM_NULL,-1);
		x = project_x(input->points[1],scale, project);
		AddPoint(x, y, 0.0);
		AddPoint(x+b_w, y, 0.0);
		ydeg += dy;
	}

	GEOMadd_disjoint_line(objT,point_list,GEOM_NULL,
				no_vertex,GEOM_COPY_DATA);


	/* create colour scale */
	get_colour_scale(input, data_val, rgb_val, buffer, rgb, lower, upper, scale, b_w, project);
	objC = GEOMcreate_mesh(extent,data_val,2,256,GEOM_COPY_DATA);
	GEOMgen_normals(objC,0);
	GEOMadd_float_colors(objC,rgb_val,256*2,GEOM_COPY_DATA);

	flags = GEOMcreate_label_flags(font,0,1,1,GEOM_LABEL_LEFT,0);
	objCL = GEOMcreate_label(extent, flags);

	x = project_x(input->points[1], scale, project);
	y = project_y(input->points[3], scale, project);
	sprintf(text,"%3.1f", lower);
	ref[0] = x + 3.0*b_w;
	ref[1] = y;
	ref[2] = 0.0;
	GEOMadd_label(objCL,text,ref,offset,(*text_h)*3.0/4.0,GEOM_NULL,-1);
	
	lat = (input->points[2] - input->points[3])/2.0 +
			input->points[3];
	y = project_y(lat, scale, project);
	sprintf(text,"%3.1f", upper);
	ref[0] = x + 3.0*b_w;
	ref[1] = y;
	ref[2] = 0.0;
	GEOMadd_label(objCL,text,ref,offset,(*text_h)*3.0/4.0,GEOM_NULL,-1);
	

	/* Create the GEOM Edit List */
	*list = GEOMinit_edit_list(*list);

	GEOMedit_geometry(*list, "data", objD);
	GEOMdestroy_obj(objD);

	GEOMedit_geometry(*list, "boundary", objB0);
	GEOMdestroy_obj(objB0);

	GEOMedit_geometry(*list, "boundary", objB1);
	GEOMdestroy_obj(objB1);

	GEOMedit_geometry(*list, "boundary", objB2);
	GEOMdestroy_obj(objB2);

	GEOMedit_geometry(*list, "boundary", objB3);
	GEOMdestroy_obj(objB3);

	GEOMedit_geometry(*list, "boundary", objT);
	GEOMdestroy_obj(objT);

	GEOMedit_geometry(*list, "label", objL0);
	GEOMdestroy_obj(objL0);

	GEOMedit_geometry(*list, "label", objL1);
	GEOMdestroy_obj(objL1);

	GEOMedit_geometry(*list, "colour_scale", objC);
	GEOMdestroy_obj(objC);


	GEOMedit_geometry(*list, "colour_scale", objCL);
	GEOMdestroy_obj(objCL);

	GEOMedit_render_mode(*list,"boundary","lines");
	GEOMedit_window(*list, "data", extent);	


/* IAC CODE CHANGE : 	free(data); */
	 free(data);

/* IAC CODE CHANGE : 	free(bufcols); */
	 free(bufcols);

/* IAC CODE CHANGE : 	free(buffer); */
	 free(buffer);

/* IAC CODE CHANGE : 	free(point_list);  */
	 free(point_list); 
 
   	AVScommand("kernel","present \"Geometry Viewer\"",&outbuf,&errbuf);
 
	return(1);
}


/******************************************************/
/* initialization for modules contained in this file. */
/******************************************************/
int ((*filter_mod_list[])()) = {
	field_2_map_spec,
};

#define NMODS (sizeof(filter_mod_list) / sizeof(char *))

AVSinit_modules()
{
	AVSinit_from_module_list(filter_mod_list, NMODS);
}

void get_extents(extent, input, scale, b_w, text_h, project)
float *extent;
AVSfield_float *input;
float scale, b_w, text_h;
int project;
{
	float text_w=0.1; /* fudge offset to allow for the text on y axis */

	extent[0] = project_x(input->points[0],scale,project) - b_w - text_w;
	extent[1] = project_x(input->points[1],scale,project) + b_w + 
								3.0*text_w;
	extent[2] = project_y(input->points[3],scale,project) - b_w - text_h;
	extent[3] = project_y(input->points[2],scale,project) + b_w;
	extent[4] = extent[5] = 0.0;
}

void get_data(input, s_data, scale, zscale, project)
AVSfield_float *input;
float *s_data;
float scale, *zscale;
int project;
{
	float *data, l_lon, l_lat, dx, dy, lon, lat, zmin, zmax;
	int i, j, index;

	zmax = zmin = 0.0;
	data = s_data;
	l_lon = input->points[0];
	l_lat = input->points[3];
	dx = (input->points[1]- input->points[0])/(float)MAXX(input);
	dy = (input->points[2]- input->points[3])/(float)MAXY(input);

	zmax = zmin = input->data[0];
	for (i=0; i<MAXX(input)*MAXY(input); i++){
		if (input->data[i] > zmax)  zmax = input->data[i];
		if (input->data[i] < zmin)  zmin = input->data[i];
	}
	
	for (j=0; j<MAXY(input); j++)
		for (i=0; i<MAXX(input); i++) {
			lon = l_lon + (float)i*dx;
			lat = l_lat + (float)j*dy;
	    		*data++ = project_x(lon, scale, project);
			*data++ = project_y(lat, scale, project);
			index = (MAXY(input)-j-1)*MAXX(input) + i;
			*data++ = input->data[index]*(*zscale) / (zmax-zmin);
		}
}


void get_cols(input, rgb, buffer, lower, upper)
AVSfield_float *input;
float *rgb, *buffer, lower, upper;
{
	int col, i, j, index, norm;

	for (col=0; col<3;  col++)
	    for(j=0; j<MAXY(input); j++)
		for(i=0; i<MAXX(input); i++){
			index = j*MAXX(input) + i;
			norm = (int)(255.0*(input->data[index] - lower)/
					(upper-lower));
			if (norm > 255) norm = 255;
			if (norm < 0) norm = 0;
			index = (MAXY(input)-j-1)*MAXX(input) + i;
			buffer[3*index+col] = rgb[3*norm+col];
		}
}


void get_colour_scale(input, s_data, s_col, s_boundary, rgb, lower, upper, scale, b_w, project)
AVSfield_float *input;
float *s_data, *s_col, *s_boundary, *rgb, lower, upper, scale, b_w;
int project;
{
	float *data, *boundary, *col, l_lon, l_lat, dy, lon, lat;
	int i,j;

	data = s_data;
	boundary = s_boundary;
	col = s_col;
	l_lon = input->points[1] + 2.0*b_w/DTR;
	l_lat = input->points[3];
	dy = (input->points[2] - input->points[3])/512.0;
	for (j=0; j<256; j++)
		for (i=0; i<2; i++) {
			lon = l_lon + (float)i*b_w/DTR;
			lat = l_lat + (float)j*dy;
	    		*data++ = project_x(lon, scale, project);
			*data++ = project_y(lat, scale, project);
			*data++ = 0.0;
			*col++ = rgb[j*3];
			*col++ = rgb[j*3+1];
			*col++ = rgb[j*3+2];
		}
	*boundary++ = project_x(l_lon,scale, project);
	*boundary++ = project_y(l_lat,scale, project);
	*boundary++ = 0.0;
	*boundary++ = project_x(l_lon+b_w/DTR,scale, project);
	*boundary++ = project_y(l_lat,scale, project);
	*boundary++ = 0.0;
	*boundary++ = project_x(l_lon,scale, project);
	*boundary++ = project_y(l_lat + dy*256.0,scale, project);
	*boundary++ = 0.0;
	*boundary++ = project_x(l_lon+b_w/DTR,scale, project);
	*boundary++ = project_y(l_lat + dy*256.0,scale, project);
	*boundary++ = 0.0;
}

void get_lower_boundary(input, s_buffer, scale, b_w, project)
AVSfield_float *input;
float *s_buffer, scale, b_w;
int project;
{
	float *buffer;

	buffer = s_buffer;
	*buffer++ = project_x(input->points[0], scale, project) - b_w;
	*buffer++ = project_y(input->points[3], scale, project) - b_w;
	*buffer++ = 0.0;
	*buffer++ = project_x(input->points[1], scale, project) + b_w;
	*buffer++ = project_y(input->points[3], scale, project) - b_w;
	*buffer++ = 0.0;
	*buffer++ = project_x(input->points[0], scale, project) - b_w;
	*buffer++ = project_y(input->points[3], scale, project);
	*buffer++ = 0.0;
	*buffer++ = project_x(input->points[1], scale, project) + b_w;
	*buffer++ = project_y(input->points[3], scale, project);
	*buffer++ = 0.0;
}

void get_upper_boundary(input, s_buffer, scale, b_w, project)
AVSfield_float *input;
float *s_buffer, scale, b_w;
int project;
{
	float *buffer;

	buffer = s_buffer;
	*buffer++ = project_x(input->points[0], scale, project) - b_w;
	*buffer++ = project_y(input->points[2], scale, project);
	*buffer++ = 0.0;
	*buffer++ = project_x(input->points[1], scale, project) + b_w;
	*buffer++ = project_y(input->points[2], scale, project);
	*buffer++ = 0.0;
	*buffer++ = project_x(input->points[0], scale, project) - b_w;
	*buffer++ = project_y(input->points[2], scale, project) + b_w;
	*buffer++ = 0.0;
	*buffer++ = project_x(input->points[1], scale, project) + b_w;
	*buffer++ = project_y(input->points[2], scale, project) + b_w;
	*buffer++ = 0.0;
}

void get_left_boundary(input, s_buffer, scale, b_w, project)
AVSfield_float *input;
float *s_buffer, scale, b_w;
int project;
{
	float *buffer;

	buffer = s_buffer;
	*buffer++ = project_x(input->points[0], scale, project) - b_w;
	*buffer++ = project_y(input->points[3], scale, project) - b_w;
	*buffer++ = 0.0;
	*buffer++ = project_x(input->points[0], scale, project);
	*buffer++ = project_y(input->points[3], scale, project) - b_w;
	*buffer++ = 0.0;
	*buffer++ = project_x(input->points[0], scale, project) - b_w;
	*buffer++ = project_y(input->points[2], scale, project) + b_w;
	*buffer++ = 0.0;
	*buffer++ = project_x(input->points[0], scale, project);
	*buffer++ = project_y(input->points[2], scale, project) + b_w;
	*buffer++ = 0.0;
}

void get_right_boundary(input, s_buffer, scale, b_w, project)
AVSfield_float *input;
float *s_buffer, scale, b_w;
int project;
{
	float *buffer;

	buffer = s_buffer;
	*buffer++ = project_x(input->points[1], scale, project);
	*buffer++ = project_y(input->points[3], scale, project) - b_w;
	*buffer++ = 0.0;
	*buffer++ = project_x(input->points[1], scale, project) + b_w;
	*buffer++ = project_y(input->points[3], scale, project) - b_w;
	*buffer++ = 0.0;
	*buffer++ = project_x(input->points[1], scale, project);
	*buffer++ = project_y(input->points[2], scale, project) + b_w;
	*buffer++ = 0.0;
	*buffer++ = project_x(input->points[1], scale, project) + b_w;
	*buffer++ = project_y(input->points[2], scale, project) + b_w;
	*buffer++ = 0.0;
}

float project_x(lon, scale, project)
float lon, scale;
int project;
{
	if (project == LINEAR || project == MERCATOR)	
		return(scale*lon*DTR);
}

float project_y(lat, scale, project)
float lat, scale;
int project;	
{ 
	if (project == LINEAR)
		return(scale*lat*DTR);
	else if (project == MERCATOR)
		return(scale*(float)(log(tan((lat/2.0 + 45.0)*DTR))));
}

void hsva_to_argb(cmap, rgb)
AVScolormap *cmap;
float *rgb;
{
	float h, s, v, f, p, q, t;
	int i, index, ih;

	for (i=0; i<256; i++){
		index = i*3;
		h = cmap->hue[i];
		s = cmap->saturation[i];
		v = cmap->value[i];
		if (v == 0.0){
			rgb[index] = rgb[index+1] = rgb[index+2] = 0.0;
			continue;
		}
		else if (s == 0.0){
			rgb[index] = rgb[index+1] = rgb[index+2] = v;
			continue;
		}
		else{
			h *= 6.0;
			if (h > 6.0) h = 0.0;
			ih = (int)h;
			f = h - ih;
			p = v*(1.0-s);
			q = v*(1.0-s*f);
			t = v*(1.0-s*(1.0-f));
		}
		if (ih == 0){
			rgb[index] = v;
			rgb[index+1] = t;
			rgb[index+2] = p;
		}
		else if (ih == 1){
			rgb[index] = q;
			rgb[index+1] = v;
			rgb[index+2] = p;
		}
		else if (ih == 2){
			rgb[index] = p;
			rgb[index+1] = v;
			rgb[index+2] = t;
		}
		else if (ih == 3){
			rgb[index] = p;
			rgb[index+1] = q;
			rgb[index+2] = v;
		}
		else if (ih == 4){
			rgb[index] = t;
			rgb[index+1] = p;
			rgb[index+2] = v;
		}
		else if (ih == 5){
			rgb[index] = v;
			rgb[index+1] = p;
			rgb[index+2] = q;
		}
	}
}

int AddPoint(x,y,z) 
float x, y, z;
{
	point_list[3*no_vertex + 0] = x; 
	point_list[3*no_vertex + 1] = y; 
	point_list[3*no_vertex + 2] = z; 
	no_vertex++;
	return(1);
}

