/*******************************************************************************
 *
 *  These are a set of utility file routines that can be used by 
 *  user-written AVS modules.
 *
 *	31 May 91  Phil McDonald, NOAA/ERL/FSL	Original version.
 *
 *	09 Jul 92  Phil McDonald		Changed from file_name.
 *						Add rec funcs.
 *
 *	16 Mar 92  Phil McDonald		Add file_name_check.
 *
 *	01 Feb 93  Phil McDonald		Add data_read.
 *
 ******************************************************************************/



#include	"avs_utils.h"
#include	"vax_convert.h"
#include	"data_weave.h"



/*******************************************************************************
 *
 *  Read data from a file.
 *
 */

unsigned char	*UTILS_file_data_read (file_name, file_type, offset_str, n_dat,
                                       data_type, vec_len, in_seq, out_seq)

char		*file_name, *file_type, *offset_str;
unsigned long	n_dat;
int		data_type, vec_len, in_seq, out_seq;
{
    static char			format_int[] = "%d";
    static char			format_flt[] = "%f";
    static char			format_dbl[] = "%lf";

    register unsigned long	n;
    register unsigned long	n_val, n_byt;
    register char		*p_fmt;
    register unsigned char	*p_dat, *p_tmp;
    long			byte_offset;
    FILE			*file;


    p_dat = NULL;

    file = UTILS_file_name_check (&file_name, "r", "The file");
    if (file == NULL) return (p_dat);

    if (file_type == NULL) return (p_dat);
    n = 0;
    while (file_type[n])
    {
        file_type[n] = toupper (file_type[n]);
        n++;
    }

    byte_offset = 0L;
    sscanf (offset_str, "%ld", &byte_offset);
    if (byte_offset < 0L) return (p_dat);

    if (fseek (file, byte_offset, 0)) return (p_dat);

    n_byt = (data_type ==    AVS_TYPE_BYTE) ? sizeof (char)  :
            (data_type == AVS_TYPE_INTEGER) ? sizeof (int)   :
            (data_type ==    AVS_TYPE_REAL) ? sizeof (float) :
            (data_type ==  AVS_TYPE_DOUBLE) ? sizeof (double): 0;
    if (n_byt == 0) return (p_dat);

    p_dat = (unsigned char *) malloc (n_dat * n_byt);
    if (p_dat != NULL)
    {
        n = 0;

        {
            if (!strncmp (file_type, "ASC", 3))
            {
                p_fmt = format_int;
                if (data_type == AVS_TYPE_REAL  ) p_fmt = format_flt;
                if (data_type == AVS_TYPE_DOUBLE) p_fmt = format_dbl;
                for (p_tmp = p_dat; n < n_dat; n++, p_tmp += n_byt)
                    if (fscanf (file, p_fmt, p_tmp) == EOF) break;
            }
            else
            {
                n = fread (p_dat, n_byt, (int) n_dat, file);
            }
        }

        if (n != n_dat)
        {
            free (p_dat);
            p_dat = NULL;
        }

        else
        {
/*
 *  Convert data from VMS format
 */

            if (!strncmp (file_type, "VMS", 3))
            {
                switch (data_type)
                {

                    case AVS_TYPE_BYTE :

                        break;

                    case AVS_TYPE_INTEGER :

                        vax_i4_to_int_blk (p_dat, n_dat);
                        break;

                    case AVS_TYPE_REAL :

                        vax_r4_to_float_blk (p_dat, n_dat);
                        break;

                    case AVS_TYPE_DOUBLE :

                        vax_g8_to_double_blk (p_dat, n_dat);
                        break;

                }
            }


/*
 *  Reweave vector data from sequential to interlaced or vice versa
 */

            if ((vec_len > 1) && (in_seq != out_seq))
            {
                n_val = n_dat / vec_len;

                p_dat = data_weave (p_dat, n_val, vec_len, n_byt, out_seq,
                                    TRUE);

            }
        }
    }
    fclose (file);

    return (p_dat);
}



/*******************************************************************************
 *
 *  Check a file name and open the file if possible.
 *
 */

FILE	*UTILS_file_name_check (p_file_name, mode, err_msg)

char	**p_file_name, *mode, *err_msg;
{

    register int	n, good_name;
    FILE		*file;

    file = NULL;

    if (*p_file_name != NULL)
    {
        good_name = FALSE;

        n = strlen (*p_file_name);
        while (--n >= 0) if ((*p_file_name)[n] != ' ') break;
        if (n >= 0)
        {
            if (((*p_file_name)[0] != ' ') &&
                ((*p_file_name)[n] != '/')) good_name = TRUE;
        }

        if (good_name)
        {
            if ((file = fopen (*p_file_name, mode)) == NULL)
            {
                AVSerror ("%s\n\n%s\n\n%s", err_msg, *p_file_name,
                          "could not be opened.");
                good_name = FALSE;
            }
        }

        if (!good_name)
        {
            free (*p_file_name);
            *p_file_name = NULL;
        }
    }



    return (file);
}



/*******************************************************************************
 *
 *  Concatenate a file name to a directory name.
 *
 */

int	UTILS_file_name_dir_add (file_name, file_dir)

char		*file_name, *file_dir;
{

    strcat (file_dir, file_name);

    return (strlen (file_dir));
}



/*******************************************************************************
 *
 *  Extract a file directory name from the file name.
 *
 */

int	UTILS_file_name_dir_get (file_name, file_dir)

char		*file_name, *file_dir;
{
    register int	l_dir;

    strcpy (file_dir, file_name);

    l_dir = strlen (file_dir);
    while (--l_dir >= 0)  if (file_dir[l_dir] != '/') break;
    file_dir[++l_dir] = '\0';

    return (l_dir);
}



/*******************************************************************************
 *
 *  Return through a string the remainder of the current record in a file.
 *
 */

char		*UTILS_file_rec_read (file)

FILE		*file;
{

    register int	i, j, ichr;
    register char	*rec, str[256];

    rec = NULL;

    i = 0;
    while (((ichr = fgetc(file)) != '\n') && (ichr != EOF)) str[i++] = ichr;
    str[i] = '\0';

    if (ichr == EOF) return (NULL);

    while ((--i >= 0) && (isspace (str[i]))) str[i] = '\0';
    if ((i >= 0) && (str[i] == '"')) str[i--] = '\0';
    j = 0;
    while ((j < i) && (isspace (str[j]))) j++;
    if (str[j] =='"') j++;

    if (j > i)
    {
        rec    = (char *) malloc (1);
        rec[0] = '\0';
    }
    else
    {
        rec = (char *) malloc ((i - j + 2));
        strcpy (rec, (str+j));
    }


    return (rec);
}



/*******************************************************************************
 *
 *  Return the number of records (new lines) in a file.
 *
 */

int		UTILS_file_rec_count (file)

FILE		*file;
{

    register int	nrecs, ichr;

    nrecs = 0;

    while ((ichr = fgetc (file)) != EOF)
    {
        if (ichr == '\n') nrecs++;
    }
    rewind (file);


    return (nrecs);
}
