/*******************************************************************************
 *
 *  These are a set of utility field point routines that can be used by 
 *  user-written AVS modules.
 *
 *	26 Mar 92  Phil McDonald, NOAA/ERL/FSL	Original version.
 *
 ******************************************************************************/



#include	"avs_utils.h"



/*******************************************************************************
 *
 *  Return the index of a point in a field.
 *
 */

int	UTILS_field_pt_index (ix, jy, kz)

int		ix, jy, kz;
{
    if (fldpt_ptr == NULL) return (0);

    return (ix + (fldpt_numijk[0] * (jy + (fldpt_numijk[1] * kz))));
}



/*******************************************************************************
 *
 *  Initialize the field point variables.
 *
 */

void	UTILS_field_pt_init (p_field)

AVSfield	*p_field;
{

    if (p_field == NULL)

        UTILS_field_pt_clear ();

    else
    {
        fldpt_ptr       = p_field;

        fldpt_ndim      = p_field->ndim;
        fldpt_numijk[0] = fldpt_numijk[1] = fldpt_numijk[2] = 1;
        if (p_field->ndim > 0) fldpt_numijk[0] = p_field->dimensions[0];
        if (p_field->ndim > 1) fldpt_numijk[1] = p_field->dimensions[1];
        if (p_field->ndim > 2) fldpt_numijk[2] = p_field->dimensions[2];
        fldpt_maxijk[0] = fldpt_numijk[0] - 1;
        fldpt_maxijk[1] = fldpt_numijk[1] - 1;
        fldpt_maxijk[2] = fldpt_numijk[2] - 1;

        fldpt_dattyp    = p_field->type;
        fldpt_grid      = p_field->uniform;
        fldpt_nspace    = p_field->nspace;
        fldpt_veclen    = p_field->veclen;

        fldpt_pts       = p_field->points;

        switch (fldpt_grid)
        {

            case UNIFORM :

                UTILS_field_pt_limits_uni (fldpt_minxyz, fldpt_maxxyz);
                fldpt_unifac[0] = (fldpt_numijk[0] < 1) ? 0.0 :
                                  (fldpt_maxxyz[0] - fldpt_minxyz[0]) /
                                  ((float) fldpt_maxijk[0]);
                fldpt_unifac[1] = (fldpt_numijk[1] < 1) ? 0.0 :
                                  (fldpt_maxxyz[1] - fldpt_minxyz[1]) /
                                  ((float) fldpt_maxijk[1]);
                fldpt_unifac[2] = (fldpt_numijk[2] < 1) ? 0.0 :
                                  (fldpt_maxxyz[2] - fldpt_minxyz[2]) /
                                  ((float) fldpt_maxijk[2]);
                break;

            case RECTILINEAR :

                fldpt_off1 = fldpt_numijk[0];
                fldpt_off2 = fldpt_numijk[0] + fldpt_numijk[1];
                break;

            case IRREGULAR :

                fldpt_off1 = fldpt_numijk[0] * fldpt_numijk[1] *
                             fldpt_numijk[2];
                fldpt_off2 = fldpt_off1 * 2;
                break;
        }
    }


    return;
}



/*******************************************************************************
 *
 *  Clear the field point variables.
 *
 */

void	UTILS_field_pt_clear ()

{

    fldpt_ptr = NULL;


    return;
}



/*******************************************************************************
 *
 *  Find the spatial limits of a field.
 *
 */

void	UTILS_field_pt_limits (min, max)

FLOAT3		min, max;
{
    register int	i, j, k, nx, ny, nz;
    register float	*pts;
    register FLOAT3	xyz, xyz1, xyz2;

    UTILS_float3_set (0.0, 0.0, 0.0, min);
    UTILS_float3_set (0.0, 0.0, 0.0, max);

    if (fldpt_ptr == NULL) return;

    if (fldpt_grid == UNIFORM)

        UTILS_field_pt_limits_uni (min, max);

    else
    {
        xyz1[0] = xyz1[1] = xyz1[2] =  1.0e30;
        xyz2[0] = xyz2[1] = xyz2[2] = -1.0e30;

        nx = fldpt_numijk[0], ny = fldpt_numijk[1], nz = fldpt_numijk[2];

        if (fldpt_grid == IRREGULAR) nx = ny = nz = nx * ny * nz;

        pts = fldpt_pts;

        if (fldpt_nspace > 0)
        {
            for (i = 0; i < nx; i++, pts++)
            {
                if (*pts < xyz1[0]) xyz1[0] = *pts;
                if (*pts > xyz2[0]) xyz2[0] = *pts;
            }
            fldpt_minxyz[0] = xyz1[0], fldpt_maxxyz[0] = xyz2[0];
        }

        if (fldpt_nspace > 1)
        {
            for (j = 0; j < ny; j++, pts++)
            {
                if (*pts < xyz1[1]) xyz1[1] = *pts;
                if (*pts > xyz2[1]) xyz2[1] = *pts;
            }
            fldpt_minxyz[1] = xyz1[1], fldpt_maxxyz[1] = xyz2[1];
        }

        if (fldpt_nspace > 2)
        {
            for (k = 0; k < nz; k++, pts++)
            {
                if (*pts < xyz1[2]) xyz1[2] = *pts;
                if (*pts > xyz2[2]) xyz2[2] = *pts;
            }
            fldpt_minxyz[2] = xyz1[2], fldpt_maxxyz[2] = xyz2[2];
        }

    }

    UTILS_float3_copy (fldpt_minxyz, min);
    UTILS_float3_copy (fldpt_maxxyz, max);


    return;
}



/*******************************************************************************
 *
 *  Find the spatial limits of a uniform field.
 *
 */

void	UTILS_field_pt_limits_uni (min, max)

FLOAT3		min, max;
{
    register int	i, j, n, has_exts;
    register float	*p_min, *p_max;

    UTILS_float3_set (0.0, 0.0, 0.0, min);
    UTILS_float3_set (0.0, 0.0, 0.0, max);

    if (fldpt_ptr == NULL) return;

    if (fldpt_grid != UNIFORM)

        UTILS_field_pt_limits (min, max);

    else
    {
        n = fldpt_ndim;

        UTILS_float3_set (0.0, 0.0, 0.0, fldpt_minxyz);
        UTILS_float3_set (0.0, 0.0, 0.0, fldpt_maxxyz);

        if (fldpt_pts != NULL)
        {
            p_min = fldpt_pts, p_max = fldpt_pts + 1;
            for (i = 0, j = 0; i < n; i++, j += 2)
                fldpt_minxyz[i] = p_min[j], fldpt_maxxyz[i] = p_max[j];
        }
        else
        {
            p_min    = fldpt_ptr->min_extent, p_max = fldpt_ptr->max_extent;
            has_exts = TRUE;
            for (i = 0; i < n; i++)
            {
                fldpt_minxyz[i] = p_min[i], fldpt_maxxyz[i] = p_max[i];
                if (fldpt_minxyz[i] >= fldpt_maxxyz[i]) has_exts = FALSE;
            }
            if (!has_exts)
            {
                UTILS_float3_set (0.0, 0.0, 0.0, fldpt_minxyz);
                UTILS_float3_set (0.0, 0.0, 0.0, fldpt_maxxyz);
            }
        }
    }

    UTILS_float3_copy (fldpt_minxyz, min);
    UTILS_float3_copy (fldpt_maxxyz, max);


    return;
}



/*******************************************************************************
 *
 *  Return the value at a point in a field.
 *
 */

float	UTILS_field_pt_val (ix, jy, kz)

int		ix, jy, kz;
{
    register int	i;


    if (fldpt_ptr             == NULL) return (0.0);
    if (fldpt_veclen          >     1) return (0.0);
    if (fldpt_ptr->field_data == NULL) return (0.0);

    i = UTILS_field_pt_index (ix, jy, kz);

    switch (fldpt_dattyp)
    {
        case AVS_TYPE_BYTE :

            return ((float) fldpt_ptr->field_data[i]);

        case AVS_TYPE_INTEGER :

            return ((float) fldpt_ptr->field_data_int[i]);

        case AVS_TYPE_REAL :

            return (fldpt_ptr->field_data_float[i]);

        case AVS_TYPE_DOUBLE :

            return ((float) fldpt_ptr->field_data_double[i]);

    }

    return (0.0);
}



/*******************************************************************************
 *
 *  Return the vector at a point in a field.
 *
 */

void	UTILS_field_pt_vec (ix, jy, kz, p_vec)

int		ix, jy, kz;
float		*p_vec;
{
    register int	i, iv, nv;


    if (fldpt_ptr == NULL) return;

    nv = fldpt_veclen;

    i  = UTILS_field_pt_index (ix, jy, kz) * nv;

    for (iv = 0; iv < nv; iv++)
    {
        switch (fldpt_dattyp)
        {
            case AVS_TYPE_BYTE :

                p_vec[iv] = (float) fldpt_ptr->field_data[i+iv];
                break;

            case AVS_TYPE_INTEGER :

                p_vec[iv] = (float) fldpt_ptr->field_data_int[i+iv];
                break;

            case AVS_TYPE_REAL :

                p_vec[iv] = fldpt_ptr->field_data_float[i+iv];
                break;

            case AVS_TYPE_DOUBLE :

                p_vec[iv] = (float) fldpt_ptr->field_data_double[i+iv];
                break;

            default:

                p_vec[iv] = 0.0;
                break;
        }
    }


    return;
}



/******************************************************************************/
/*
 *  Convert XY coordinates in computational space to XYZ in physical space.
 *
 */


int	UTILS_field_pt_xy_to_xyz (xyz)

FLOAT3		xyz;
{
    static float	*pts, *ptx, *pty, *ptz;

    register int	ix, iy, nx, ny, n;
    register float	x, y;
    register FLOAT3	tmp;

    if (xyz == NULL)
    {
        if (pts != NULL) free (pts), pts = NULL;
        return (0);
    }

    if (pts == NULL)
    {
        nx  = fldpt_numijk[0], ny = fldpt_numijk[1];
        n   = nx * ny;
        pts = (float *) malloc (n * sizeof (FLOAT3));
        ptx = pts, pty = ptx + n, ptz = pty + n;

        n = 0;
        for (iy = 0; iy < ny; iy++)
        {
            for (ix = 0; ix < nx; ix++, n++)
            {
                UTILS_field_pt_xyz (ix, iy, 0, tmp);
                ptx[n] = tmp[0], pty[n] = tmp[1], ptz[n] = tmp[2];
            }
        }
    }

    x = xyz[0], y = xyz[1];

    if ((x < 0.0) || (x > fldpt_maxijk[0])) return (0);
    if ((y < 0.0) || (y > fldpt_maxijk[1])) return (0);

    interp_tri (ptx, fldpt_numijk[0], fldpt_numijk[1], x, y, &xyz[0], -1);
    interp_tri (pty, fldpt_numijk[0], fldpt_numijk[1], x, y, &xyz[1], -1);
    interp_tri (ptz, fldpt_numijk[0], fldpt_numijk[1], x, y, &xyz[2], -1);


    return (1);
}



/******************************************************************************/
/*
 *  Convert XY coordinates in computational space to XYZ in physical space.
 *
 */


int	UTILS_field_pt_xy_xyz (xyz)

FLOAT3		xyz;
{
    register int	i, ix, iy, jx, jy;
    register float	x, y, dat[4];
    float		xx, yy;
    FLOAT3		pt[4];

    if (fldpt_ptr == NULL) return (0);

    x = xyz[0], y = xyz[1];

    if ((x < 0.0) || (x > fldpt_maxijk[0])) return (0);
    if ((y < 0.0) || (y > fldpt_maxijk[1])) return (0);

    jx = ix = x, jy = iy = y;
    if (jx < fldpt_maxijk[0]) jx++;
    if (jy < fldpt_maxijk[1]) jy++;

    UTILS_field_pt_xyz (ix, iy, 0, &pt[0][0]);
    UTILS_field_pt_xyz (jx, iy, 0, &pt[1][0]);
    UTILS_field_pt_xyz (ix, jy, 0, &pt[2][0]);
    UTILS_field_pt_xyz (jx, jy, 0, &pt[3][0]);

    xx = x - ix, yy = y - iy;

    for (i = 0; i < 4; i++) dat[i] = pt[i][0];
    interp_tri (dat, 2, 2, xx, yy, &xyz[0], -1);

    for (i = 0; i < 4; i++) dat[i] = pt[i][1];
    interp_tri (dat, 2, 2, xx, yy, &xyz[1], -1);

    for (i = 0; i < 4; i++) dat[i] = pt[i][2];
    interp_tri (dat, 2, 2, xx, yy, &xyz[2], -1);


    return (1);
}



/*******************************************************************************
 *
 *  Return the 3D coordinates of a point in a field.
 *
 */

int	UTILS_field_pt_xyz (ix, jy, kz, xyz)

int		ix, jy, kz;
FLOAT3		xyz;
{
    register int	i;

    if (fldpt_ptr == NULL) return (0);
    if ((fldpt_grid != UNIFORM) && (fldpt_pts == NULL)) return (0);

    if ((ix < 0) || (ix > fldpt_maxijk[0])) return (0);
    if ((jy < 0) || (jy > fldpt_maxijk[1])) return (0);
    if ((kz < 0) || (kz > fldpt_maxijk[2])) return (0);

    xyz[0] = xyz[1] = xyz[2] = UTILS_field_pt_val (ix, jy, kz); 

    switch (fldpt_grid)
    {

        case UNIFORM :

            if (fldpt_ndim > 0)
            {
                xyz[0] = fldpt_minxyz[0] + (((float) ix) * fldpt_unifac[0]);
                if (fldpt_ndim > 1)
                {
                    xyz[1] = fldpt_minxyz[1] + (((float) jy) * fldpt_unifac[1]);
                    if (fldpt_ndim > 2)
                    {
                        xyz[2] = fldpt_minxyz[2] +
                                 (((float) kz) * fldpt_unifac[2]);
                    }
                }
            }
            break;

        case RECTILINEAR :

            xyz[0] = fldpt_pts[ix];
            if (fldpt_ndim > 1)
            {
                xyz[1] = fldpt_pts[jy+fldpt_off1];
                if (fldpt_ndim > 2)
                {
                    xyz[2] = fldpt_pts[kz+fldpt_off2];
                }
            }
            break;

        case IRREGULAR :

            i = UTILS_field_pt_index (ix, jy, kz);

            xyz[0] = fldpt_pts[i];
            if (fldpt_nspace > 1)
            {
                xyz[1] = fldpt_pts[i+fldpt_off1];
                if (fldpt_nspace > 2)
                {
                    xyz[2] = fldpt_pts[i+fldpt_off2];
                }
            }
            break;
    }

    return (1);
}
