/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/******************************************************************************/
/*
 *  The Line_Probe module provides an interactive line cursor probe for
 *  sambling fields in 3-space.
 *
 */


#include "avs_utils.h"
#include "chain.h"



#define		PROBE_MODE_MANUAL	1
#define		PROBE_MODE_MOUSE	2
#define		PROBE_MODE_FILE		3
#define		PROBE_MODE_FIELD	4



static char	input_field[]		= "Input Field";
static char	input_sample_fld[]	= "Input Sample Field";
static char	input_cmap[]		= "Input Colormap";
static char	input_upgeom[]		= "upstream geom";
static char	output_field[]		= "Output Field";
static char     output_geom[]		= "Output Geom";

static char	parm_probe_mode[]	= "probe mode";
static char	parm_all_samples[]	= "all samples";
static char	parm_probe_x[]		= "probe x   ";
static char	parm_probe_y[]		= "probe y   ";
static char	parm_node_size[]	= "node size";
static char	parm_probe_vals[]	= "probe values";
static char	parm_sample_file[]	= "sample file";
static char	parm_sample_choices[]	= "sample choices";

static char	mode_choices[]	= "manual!mouse!sample file!sample field";



line_probe_desc ()
{
    int		line_probe_compute ();
    int		in_port, out_port, iparm;

    AVSset_module_name ("line probe", MODULE_MAPPER);

    AVScreate_input_port (input_field,      "field",    REQUIRED);
    AVScreate_input_port (input_sample_fld, "field",    OPTIONAL);
    AVScreate_input_port (input_cmap,       "colormap", OPTIONAL);
    in_port = AVScreate_input_port (input_upgeom, "struct upstream_geom",
                                    OPTIONAL | INVISIBLE);
    AVSset_input_class (in_port, "upstream_geom");


    out_port = AVScreate_output_port (output_field, "field float irregular");
    AVSautofree_output (out_port);
    out_port = AVScreate_output_port (output_geom, "geom");
    AVSautofree_output (out_port);


    iparm = AVSadd_parameter (parm_probe_mode, "choice",
                              "manual", mode_choices, CHOICE_DELIMITER_STR);
    AVSadd_parameter_prop (iparm, "width", "integer", 2);
    AVSadd_parameter_prop (iparm, "columns", "integer", 2);

    AVSadd_parameter (parm_all_samples, "boolean", FALSE, FALSE, TRUE);

    iparm = AVSadd_float_parameter (parm_probe_x, 0.0,
                                    FLOAT_UNBOUND, FLOAT_UNBOUND);
    AVSconnect_widget (iparm, "typein_real");
    AVSadd_parameter_prop (iparm, "width", "integer", 3);
    iparm = AVSadd_float_parameter (parm_probe_y, 0.0,
                                    FLOAT_UNBOUND, FLOAT_UNBOUND);
    AVSconnect_widget (iparm, "typein_real");
    AVSadd_parameter_prop (iparm, "width", "integer", 3);

    iparm = AVSadd_float_parameter (parm_node_size, 0.0, 0.0, FLOAT_UNBOUND);
    AVSconnect_widget (iparm, "typein_real");
    AVSadd_parameter_prop (iparm, "width", "integer", 3);

    iparm = AVSadd_parameter (parm_probe_vals, "string_block",
                              NULL_STR, NULL, NULL);
    AVSconnect_widget (iparm, "text_block_browser");
    AVSadd_parameter_prop (iparm, "height", "integer", 10);
    AVSadd_parameter_prop (iparm, "width", "integer", 4);

    iparm = AVSadd_parameter (parm_sample_file, "string",
                              NULL_STR, NULL, ".txt.dat.pos");
    AVSconnect_widget (iparm, "browser");
    iparm = AVSadd_parameter (parm_sample_choices, "choice",
                              NULL_STR, NULL_STR, CHOICE_DELIMITER_STR);
    AVSconnect_widget (iparm, "choice_browser");


    AVSset_compute_proc (line_probe_compute);

    return;
}



/******************************************************************************/
/*
 *  Inform AVS (the main procedure) of this module.
 *
 */


AVSinit_modules()
{
    AVSmodule_from_desc (line_probe_desc);
    return ;
}



line_probe_compute (in_field, in_sample_fld, in_cmap, in_geom_info,
                    out_field, out_geom,
                    mode_choice, all_samples,
                    p_probe_x, p_probe_y, p_node_size,
                    p_val_txt, sample_file, sample_choice)

AVSfield		*in_field, *in_sample_fld;
AVScolormap		*in_cmap;
upstream_geom		*in_geom_info;
AVSfield		**out_field;
GEOMedit_list		*out_geom;
char			*mode_choice;
int			all_samples;
float			*p_probe_x, *p_probe_y;
float			*p_node_size;
char			*p_val_txt;
char			*sample_file, *sample_choice;
{
    static char		probe_obj_name[]  	= "line_probe";
    static char		probe_nodes_name[]	= "line_probe_nodes";

    static FLOAT3	xyz_min, xyz_max;

    register int	i, j, k, n;
    register int	ipt, npts, nvec;
    register FLOAT3	xyz, tmp;
    int			ndim, dim[3];
    int			probe_mode, axis, i_axis, j_axis, pt_ok;
    int			ntxt, n_probe, nx_sam, ny_sam, nz_sam;
    char		*p_str;
    float		probe_x, probe_y, probe_z, *p_x, *p_y, *p_z;
    float		probe_r, *probe_rad;
    float		*probe_dat, *p_dat, *p_tmp, *p_prb_dat;
    CHAIN		*probe, probe_beg;
    FLOAT3		probe_xyz, *probe_col, *probe_pts, *p_pts, *p_prb_pts;
    FILE		*file;
    GEOMobj		*probe_obj, *nodes_obj;



    *out_field = NULL;
    *out_geom  = GEOMinit_edit_list (*out_geom);


    probe_mode = AVSchoice_number (parm_probe_mode, mode_choice);
    if (probe_mode == 0) return (1);

    probe_x    = *p_probe_x;
    probe_y    = *p_probe_y;
    probe_z    = 0.0;

    probe_r    = *p_node_size;

    axis       = Z_COORD;


    if (in_field == NULL)
    {
        AVSmodify_parameter (parm_probe_vals, AVS_VALUE, NULL_STR, NULL, NULL);
        GEOMedit_visibility (*out_geom, probe_obj_name, GEOM_EDIT_DELETE);
        return (1);
    }
    else if (in_field->nspace != 3)
    {
        AVSerror ("%s%s", "The line probe module works only with ",
                  "fields in 3-space.\n");
        return (1);
    }

    UTILS_field_pt_init   (in_field);
    UTILS_field_dims_get (in_field, &ndim, &dim[0], &dim[1], &dim[2]);

    if (AVSinput_changed (input_field, 0))
    {
        UTILS_field_pt_limits (NULL, NULL);
        UTILS_float3_copy (fldpt_minxyz, xyz_min);
        UTILS_float3_copy (fldpt_maxxyz, xyz_max);
    }
    else
    {
        UTILS_float3_copy (xyz_min, fldpt_minxyz);
        UTILS_float3_copy (xyz_max, fldpt_maxxyz);
    }

    i_axis = X_COORD, j_axis = Y_COORD;

    if (ndim < 3)
    {
        UTILS_field_pt_xyz (0, 0, 0, tmp);
        UTILS_field_pt_xyz (0, 1, 0, xyz);
        UTILS_float3_sub (xyz, tmp);
        if (xyz[Z_COORD] > xyz[Y_COORD]) j_axis = Z_COORD, axis = Y_COORD;
        UTILS_field_pt_xyz (1, 0, 0, xyz);
        UTILS_float3_sub (xyz, tmp);
        if (xyz[Y_COORD] > xyz[X_COORD]) i_axis = Y_COORD;
    }


    if (AVSparameter_changed (parm_sample_file))
    {
        if ((!AVSparameter_changed (parm_sample_choices)) &&
            (sample_choice != NULL)) sample_choice[0] = '\0';

        file = UTILS_choice_text_file (&sample_file, parm_sample_choices,
                                       sample_choice, &ntxt,
                                       "The line probe sample file", TRUE);
        AVSparameter_visible (parm_sample_choices, (file != NULL));
    }



    n_probe     = 0;
    probe       = &probe_beg;
    probe->next = NULL;
    probe->data = (char *) probe_xyz;

    UTILS_float3_set (probe_x, probe_y, probe_z, xyz);


    if (probe_mode == PROBE_MODE_MANUAL)
    {
        probe_save (xyz, i_axis, j_axis, &probe, &n_probe);
    }


    if (probe_mode == PROBE_MODE_MOUSE)
    {
        GEOMedit_selection_mode (*out_geom, "%top", "notify", BUTTON_DOWN);

        if ((AVSinput_changed (input_upgeom, 0)) && (in_geom_info != NULL))
        {
            if ((in_geom_info->picked_obj    != NULL) &&
                (in_geom_info->picked_obj[0] != '\0'))
            {
                probe_save (in_geom_info->mscoord, i_axis, j_axis,
                            &probe, &n_probe);
            }
        }
        else
        {
            probe_save (xyz, i_axis, j_axis, &probe, &n_probe);
        }
    }
    else
    {
        GEOMedit_selection_mode (*out_geom, "%top", "normal", 0);
    }


    if (probe_mode == PROBE_MODE_FILE)
    {
        if (all_samples)
        {
            file = UTILS_file_name_check (&sample_file, "r",
                                          "The line probe sample file");
            if (file != NULL)
            {
                while ((p_str = UTILS_file_rec_read (file)) != NULL)
                {
                    if (probe_read (p_str, xyz))
                        probe_save (xyz, i_axis, j_axis, &probe, &n_probe);

/* IAC CODE CHANGE :                     free (p_str); */
                     free(p_str);
                }
                fclose (file);
            }
        }

        else
        {
            if (probe_read (sample_choice, xyz))
                probe_save (xyz, i_axis, j_axis, &probe, &n_probe);
        }
    }


    if (probe_mode == PROBE_MODE_FIELD)
    {
        UTILS_field_pt_init  (in_sample_fld);
        UTILS_field_dims_get (in_sample_fld, &ntxt, &nx_sam, &ny_sam, &nz_sam);

        if (!all_samples) ny_sam = nz_sam = 1;

        for (k = 0; k < nz_sam; k++)
        {
            for (j = 0; j < ny_sam; j++)
            {
                UTILS_field_pt_xyz  (0, j, k, xyz);
                probe_save (xyz, i_axis, j_axis, &probe, &n_probe);
            }
        }

        UTILS_field_pt_init  (in_field);
    }

    if (n_probe == 0) return (1);



    npts = (probe_mode == PROBE_MODE_FIELD) ? nx_sam : dim[axis];
    nvec = (in_field->veclen < 1) ? 1 : in_field->veclen;

    probe_dat = (float *) calloc ((n_probe * npts * nvec), sizeof (float));
    probe_pts = (FLOAT3 *) calloc ((n_probe * npts), sizeof (FLOAT3));
    probe_col = ((in_cmap == NULL) || (nvec > 1)) ? GEOM_NULL :
                (FLOAT3 *) calloc (npts, sizeof (FLOAT3));
    probe_rad = (probe_r <= 0.0) ? GEOM_NULL :
                (float *) calloc (npts, sizeof (float));
    p_tmp     = (float *) calloc ((4 * nvec), sizeof (float));

    p_prb_pts = probe_pts, p_prb_dat = probe_dat;


    if (n_probe == 1)
    {
        UTILS_float3_get (probe_xyz, &probe_x, &probe_y, &probe_z);

        ntxt      = npts * (10 + (nvec * 12));
        p_val_txt = (char *) calloc (ntxt, 1);
        ntxt      = 0;
    }
    if (p_val_txt != NULL) p_val_txt[0] = '\0';

    AVSmodify_float_parameter (parm_probe_x, AVS_VALUE, probe_x, 0.0, 0.0);
    AVSmodify_float_parameter (parm_probe_y, AVS_VALUE, probe_y, 0.0, 0.0);


    probe_obj = GEOMcreate_obj (GEOM_POLYTRI, GEOM_NULL);
    if (probe_r > 0.0) nodes_obj = GEOMcreate_obj (GEOM_SPHERE, GEOM_NULL);

    if (probe_mode == PROBE_MODE_FIELD) j = k = 0;

    probe = &probe_beg;
    while (probe != NULL)
    {
        p_pts = p_prb_pts, p_dat = p_prb_dat;

        n = 0;
        for (ipt = 0; ipt < npts; ipt++)
        {
            if (probe_mode == PROBE_MODE_FIELD)
            {
                UTILS_field_pt_init (in_sample_fld);
                UTILS_field_pt_xyz  (ipt, j, k, p_pts);
                UTILS_field_pt_init (in_field);
                UTILS_float3_copy (p_pts, xyz);
                pt_ok = probe_get_dat (ipt, i_axis, j_axis, Z_COORD,
                                       xyz, p_dat, p_tmp);
            }
            else
            {
                UTILS_float3_copy (probe->data, p_pts);
                probe_get_dat (ipt, i_axis, j_axis, -1, p_pts, p_dat, p_tmp);
                pt_ok = TRUE;
            }

            if (pt_ok)
            {
                if (probe_col != GEOM_NULL)
                    UTILS_color_rgb_from_value (in_cmap, *p_dat,
                                                (probe_col+n));

                if (probe_r > 0.0) probe_rad[n] = probe_r;

                if (n_probe == 1)
                {
                    ntxt += sprintf ((p_val_txt+ntxt), "(%d) %g",
                                     (n+1), p_dat[0]);
                    for (i = 1; i < nvec; i++)
                        ntxt += sprintf ((p_val_txt+ntxt), ", %g", p_dat[i]);
                    ntxt += sprintf ((p_val_txt+ntxt), "\n");
                }

                n++, p_pts++, p_dat += nvec;
            }
        }

        if (n > 0)
        {
            if (n > 1)
                GEOMadd_polyline (probe_obj, p_prb_pts, probe_col, n,
                                  GEOM_COPY_DATA);
            if (probe_r > 0.0)
            {
                GEOMadd_vertices     (nodes_obj, p_prb_pts, n, GEOM_COPY_DATA);
                GEOMadd_radii        (nodes_obj, probe_rad, n, GEOM_COPY_DATA);
                GEOMadd_float_colors (nodes_obj, probe_col, n, GEOM_COPY_DATA);
            }
        }

        if (probe_mode == PROBE_MODE_FIELD)
        {
            j++;
            if (j >= ny_sam) j = 0, k++;
        }

        p_prb_pts += npts, p_prb_dat += nvec * npts;

        probe_beg.next = probe->next;
        if (probe != &probe_beg) probe_free (probe);
        probe = probe_beg.next;
    }

    AVSmodify_parameter (parm_probe_vals, AVS_VALUE, p_val_txt, NULL, NULL);


    GEOMedit_geometry (*out_geom, probe_obj_name, probe_obj);
    GEOMdestroy_obj (probe_obj);

    if (probe_r <= 0.0)
        GEOMedit_visibility (*out_geom, probe_nodes_name, GEOM_EDIT_DELETE);
    else
    {
        GEOMedit_geometry (*out_geom, probe_nodes_name, nodes_obj);
        GEOMdestroy_obj (nodes_obj);
        GEOMedit_parent (*out_geom, probe_nodes_name, probe_obj_name);
        GEOMedit_transform_mode (*out_geom, probe_nodes_name, "parent", 0);
    }


    ndim = 1, dim[0] = npts;
    if (n_probe > 1) ndim = 2, dim[1] = n_probe;
    *out_field = UTILS_field_create (ndim, dim, 3, nvec, AVS_TYPE_REAL,
                                     IRREGULAR);

    if (*out_field != NULL)
    {
        UTILS_field_copy_mem (*out_field, probe_dat, NULL);

        UTILS_field_copy_strings (in_field, *out_field);

        n     = npts * n_probe;
        p_pts = probe_pts;
        p_x   = (*out_field)->points, p_y = p_x + n, p_z = p_y + n;
        for (i = 0; i < n; i++)
        {
            UTILS_float3_get (p_pts, p_x, p_y, p_z);
            p_pts++, p_x++, p_y++, p_z++;
        }
    }


/* IAC CODE CHANGE :     free (p_tmp); */
     free(p_tmp);

/* IAC CODE CHANGE :     if (probe_dat != NULL) free (probe_dat); */
    if (probe_dat != NULL)  free(probe_dat);

/* IAC CODE CHANGE :     if (probe_pts != NULL) free (probe_pts); */
    if (probe_pts != NULL)  free(probe_pts);

/* IAC CODE CHANGE :     if (probe_rad != NULL) free (probe_rad); */
    if (probe_rad != NULL)  free(probe_rad);

/* IAC CODE CHANGE :     if (probe_col != NULL) free (probe_col); */
    if (probe_col != NULL)  free(probe_col);


    return (1);
}



/******************************************************************************/
/*
 *  Convert XY coordinates in computational space to XYZ in physical space.
 *
 */


int	probe_get_dat (ipt, i_axis, j_axis, k_axis, xyz, vec, tmp)

int		ipt, i_axis, j_axis, k_axis;
FLOAT3		xyz;
float		*vec, *tmp;
{
    register int	i, k, nv;
    register float	*pv, dat[4];
    int			i1, i2, j1, j2, k1, k2, ijk[3];
    float		frac_i, frac_j, frac_k, x;
    FLOAT3		pt[4];

    if (fldpt_ptr == NULL) return (0);

    frac_i = frac_j = frac_k = 0.0;
    i1 = i2 = j1 = j2 = k1 = k2 = ijk[0] = ijk[1] = ijk[2] = 0;


    frac_i = xyz[i_axis];
    if ((i2 = probe_get_ijk (i_axis, 0, &frac_i, ijk)) < 0) return (0);
    i1 = ijk[0];

    j1 = j2 = ipt;

    if (j_axis == Y_COORD)
    {
        frac_j = xyz[j_axis];
        if ((j2 = probe_get_ijk (j_axis, 1, &frac_j, ijk)) < 0) return (0);
        j1 = ijk[1];

        k1 = k2 = ipt;

        if (k_axis == Z_COORD)
        {
            frac_k = xyz[k_axis];
            if ((k2 = probe_get_ijk (k_axis, 2, &frac_k, ijk)) < 0) return (0);
            k1 = ijk[2];
        }
    }

    for (k = k1; k <= k2; k++)
    {
        UTILS_field_pt_xyz (i1, j1, k, &pt[0][0]);
        UTILS_field_pt_xyz (i2, j1, k, &pt[1][0]);
        UTILS_field_pt_xyz (i1, j2, k, &pt[2][0]);
        UTILS_field_pt_xyz (i2, j2, k, &pt[3][0]);


        x = xyz[0];
        for (i = 0; i < 4; i++) dat[i] = pt[i][0];
        interp_tri (dat, 2, 2, frac_i, frac_j, &xyz[0], -1);
        if (k != k1) xyz[0] = x + ((xyz[0] - x) * frac_k);

        x = xyz[1];
        for (i = 0; i < 4; i++) dat[i] = pt[i][1];
        interp_tri (dat, 2, 2, frac_i, frac_j, &xyz[1], -1);
        if (k != k1) xyz[1] = x + ((xyz[1] - x) * frac_k);

        x = xyz[2];
        for (i = 0; i < 4; i++) dat[i] = pt[i][2];
        interp_tri (dat, 2, 2, frac_i, frac_j, &xyz[2], -1);
        if (k != k1) xyz[2] = x + ((xyz[2] - x) * frac_k);


        if (vec != NULL)
        {
            nv = fldpt_veclen;
            pv = tmp;
            UTILS_field_pt_vec (i1, j1, k, (pv));
            UTILS_field_pt_vec (i2, j1, k, (pv+(nv)));
            UTILS_field_pt_vec (i1, j2, k, (pv+(nv*2)));
            UTILS_field_pt_vec (i2, j2, k, (pv+(nv*3)));

            for (i = 0; i < nv; i++, pv++)
            {
                x      = *(vec+i);
                dat[0] = *pv;
                dat[1] = *(pv+nv);
                dat[2] = *(pv+(nv*2));
                dat[3] = *(pv+(nv*3));
                interp_tri (dat, 2, 2, frac_i, frac_j, (vec+i), -1);
                if (k != k1) *(vec+i) = x + ((*(vec+i) - x) * frac_k);
            }
        }
    }


    return (1);
}



int	probe_get_ijk (icoord, iaxis, p_a, ijk)

int		icoord, iaxis;
float		*p_a;
int		*ijk;
{
    register int	ia, ja;
    register float	xyz[3], a, aa;

    aa = a = *p_a;
    ia = -1;
    for (ja = 0; ja <= fldpt_maxijk[iaxis]; ja++)
    {
        ijk[iaxis] = ja;
        UTILS_field_pt_xyz (ijk[0], ijk[1], ijk[2], xyz);
        if (xyz[icoord] <= a)
            aa = xyz[icoord], ia = ja;
        else
            ja = fldpt_maxijk[iaxis];
    }
    if (ia == -1) return (-1);

    ja = ia;
    if (ja < fldpt_maxijk[iaxis]) ja++;

    aa = (xyz[icoord] == aa) ? 0.0 : (a - aa) / (xyz[icoord] - aa);

    *p_a        = aa;
    ijk[icoord] = ia;


    return (ja);
}



probe_free (probe)

CHAIN		*probe;
{


/* IAC CODE CHANGE :     free (probe->data); */
     free(probe->data);

/* IAC CODE CHANGE :     free (probe); */
     free(probe);


    return;
}



probe_read (str, xyz)

char		*str;
FLOAT3		xyz;
{
    register int	i, j, n;
    int			nchr;

    UTILS_float3_set (0.0, 0.0, 0.0, xyz);

    if ((n = strlen (str)) == 0) return (FALSE);

    j = 0;
    for (i = X_COORD; i <= Y_COORD; i++)
    {
        nchr = 0;
        while (!sscanf ((str+j), "%f%n", &xyz[i], &nchr))
        {
            if (++j >= n) return (FALSE);
        }
        j += nchr;
    }

    return (TRUE);
}



probe_save (xyz, i_axis, j_axis, p_probe, p_n)

FLOAT3		xyz;
int		i_axis, j_axis;
CHAIN		**p_probe;
int		*p_n;
{

    if ((xyz[0] < fldpt_minxyz[0]) || (xyz[0] > fldpt_maxxyz[0])) return;
    if ((xyz[1] < fldpt_minxyz[1]) || (xyz[1] > fldpt_maxxyz[1])) return;

    if (*p_n > 0)
    {
        (*p_probe)->next = (CHAIN *) calloc (1, sizeof (CHAIN));
        (*p_probe)       = (*p_probe)->next;
        (*p_probe)->data = calloc (1, sizeof (FLOAT3));
    }

    (*p_probe)->next = NULL;

    probe_get_dat (0, i_axis, j_axis, -1, xyz, NULL, NULL);
    UTILS_float3_copy (xyz, (*p_probe)->data);

    (*p_n)++;


    return;
}
