/*******************************************************************************
 *
 *  These are a set of utility dump routines that can be used by 
 *  user-written AVS modules.
 *
 *	29 Mar 91  Phil McDonald, NOAA/ERL/FSL	Original version.
 *
 *	29 May 91  Phil McDonald		Add dump_field_info.
 *
 *	28 Feb 92  Phil McDonald		Complete obj and list funcs.
 *
 ******************************************************************************/



#include	"avs_utils.h"



/*******************************************************************************
 *
 *  Dump a field's description parameters.
 *
 */

void	UTILS_dump_field_info (stream, field)

FILE		*stream;
AVSfield	*field;
{


    fprintf (stream, "\n");
    dump_info_item (stream, "          ", field, 0x03);
    if (field == NULL)
    {
        fprintf (stream, "\n"), fflush (stream);
        return;
    }

    dump_info_item (stream, "ndim      ", field->ndim, 0x01);
    dump_info_list (stream, "dimensions", field->dimensions, field->ndim,
                                                             AVS_TYPE_INTEGER);
    dump_info_item (stream, "nspace    ", field->nspace, 0x01);
    dump_info_item (stream, "veclen    ", field->veclen, 0x01);
    dump_info_item (stream, "type      ", field->type, 0x01);
    dump_info_item (stream, "size      ", field->size, 0x01);
    dump_info_item (stream, "sgl_blk   ", field->single_block, 0x01);
    dump_info_item (stream, "uniform   ", field->uniform, 0x01);
    dump_info_item (stream, "flags     ", field->flags, 0x03);
    dump_info_item (stream, "data      ", field->field_data, 0x03);
    dump_info_item (stream, "points    ", field->points, 0x03);

    dump_info_list (stream, "min extent", field->min_extent, field->nspace,
                                                             AVS_TYPE_REAL);
    dump_info_list (stream, "max extent", field->max_extent, field->nspace,
                                                             AVS_TYPE_REAL);

    dump_info_list (stream, "minimum   ", field->min_data, field->veclen,
                                                           field->type);
    dump_info_list (stream, "maximum   ", field->max_data, field->veclen,
                                                           field->type);

    fprintf (stream, "Field shared mem = %d %d %8.8x\n",
             field->shm_key, field->shm_id, field->shm_base), fflush (stream);

    fprintf (stream, "Field labels     = %8.8x %s\n",
             field->labels, field->labels), fflush (stream);
    fprintf (stream, "Field units      = %8.8x %s\n",
             field->units, field->units), fflush (stream);

    fprintf (stream, "\n"), fflush (stream);

    return;
}


dump_info_item (stream, label, item, flags)

FILE		*stream;
char		*label;
int		item, flags;
{

    if ((flags & 0x02))
        fprintf (stream, "Field %10s = %8.8x", label, item);
    else
        fprintf (stream, "Field %10s = %d", label, item);

    if ((flags & 0x01)) fprintf (stream, "\n");

    fflush (stream);

    return;
}



dump_info_list (stream, label, p_list, nlist, type)

FILE		*stream;
char		*label, *p_list;
int		nlist, type;
{
    register int	i;

    dump_info_item (stream, label, (int) p_list, 0x02);
    if (p_list != NULL)
    {
        for (i = 0; i < nlist; i++)
        {
            switch (type)
            {
                case AVS_TYPE_BYTE :
                    fprintf (stream, " %d", *(p_list+i));
                    break;
                case AVS_TYPE_INTEGER :
                    fprintf (stream, " %d", *(((int *) p_list)+i));
                    break;
                case AVS_TYPE_REAL :
                    fprintf (stream, " %f", *(((float *) p_list)+i));
                    break;
                case AVS_TYPE_DOUBLE :
                    fprintf (stream, " %f", *(((double *) p_list)+i));
                    break;
            }
            fflush (stream);
        }
    }
    fprintf (stream, "\n");
    fflush (stream);

    return;
}



/*******************************************************************************
 *
 *  Dump a geometry edit list.
 *
 */

void	UTILS_dump_geom_edlist (stream, edlist, flags)

FILE		*stream;
GEOMedit_list	edlist;
int		flags;

{

    GEOMedit	*list;
    int		i;

    if (!(flags & DUMP_EDLIST_FLAG)) return;

    fprintf (stream, "EDLIST %8.8x\n", edlist), fflush (stream);
    if (edlist != GEOM_NULL)
    {
        list = edlist->l;
        while (list != GEOM_NULL)
        {
            fprintf (stream, "    LIST = %8.8x %d %d %8.8x %8.8x %8.8x", list,
                     list->type, list->size, list->next, list->data,
                     list->name);
            if (list->name != GEOM_NULL) fprintf (stream, " >%s<", list->name);
            fprintf (stream, "\n"), fflush (stream);

            if (list->type == 0)
                UTILS_dump_geom_obj (stream, list->data, flags);
            else
            {
                fprintf (stream, "\n    Edit List ");
                switch (list->type)
                {
                    case GEOM_EDLIST_VISIBILITY :
                        fprintf (stream, "Visibility   = %d\n",
                                 *(int *)list->data);
                        break;
                    case GEOM_EDLIST_PARENT :
                        fprintf (stream, "Parent       = >%s<\n", list->data);
                        break;
                    case GEOM_EDLIST_PARENT_TOP :
                        fprintf (stream, "Parent Top   = >%s<\n", list->data);
                        break;
                    case GEOM_EDLIST_WINDOW :
                        fprintf (stream, "Window       = ");
                        for (i = 0; i < 6; i += 2)
                            fprintf (stream, "%f ", *(((float *)list->data)+i));
                        fprintf (stream, "\n                             ");
                        for (i = 1; i < 6; i += 2)
                            fprintf (stream, "%f ", *(((float *)list->data)+i));
                        fprintf (stream, "\n");
                        break;
                    default :
                        fprintf (stream, "Type Unknown = %d\n", list->type);
                        UTILS_dump_hex (stream, list->data, 128);
                }
                fflush (stream);
            }

            list = list->next;
            if (list != GEOM_NULL) fprintf (stream, "\n"), fflush (stream);
        }
    }
    fprintf (stream, "\n");
    fflush (stream);

    return;
}



/*******************************************************************************
 *
 *  Dump a geometry label object.
 *
 */

void	UTILS_dump_geom_label (stream, label, flags)

FILE		*stream;
GEOMlabel	*label;
int		flags;
{

    if (!(flags & DUMP_OBJ_DATA_FLAG)) return;

    if (label == NULL) return;

    fprintf (stream, " LABEL %10x\n", label);
    fprintf (stream, "    L_FLAGS            %8.8x\n", label->lflags);
    fflush (stream);
    UTILS_dump_geom_chlist (stream, "L_LABELS ", &(label->labels), 0, flags);
    UTILS_dump_geom_vtlist (stream, "L_VERTS  ", &(label->verts), -1, flags);
    UTILS_dump_geom_vtlist (stream, "L_OFFSETS", &(label->offsets), -1, flags);
    UTILS_dump_geom_vtlist (stream, "L_COLORS ", &(label->colors), -1, flags);
    UTILS_dump_geom_sclist (stream, "L_HEIGHTS", &(label->heights), 0, flags);

    return;
}



/*******************************************************************************
 *
 *  Dump a geometry mesh object.
 *
 */

void	UTILS_dump_geom_mesh (stream, mesh, flags)

FILE		*stream;
GEOMmesh	*mesh;
int		flags;
{

    if (!(flags & DUMP_OBJ_DATA_FLAG)) return;

    if (mesh == NULL) return;

    fprintf (stream, " MESH %10x\n", mesh), fflush (stream);
    UTILS_dump_geom_vtlist (stream, "M_VERTS  ", &mesh->verts, 0, flags);
    UTILS_dump_geom_vtlist (stream, "M_NORMALS", &mesh->normals, 0, flags);
    UTILS_dump_geom_vtlist (stream, "M_COLORS ", &mesh->colors, 0, flags);
    UTILS_dump_geom_vtlist (stream, "M_UVS    ", &mesh->uvs, 0, flags);

    return;
}



/*******************************************************************************
 *
 *  Dump a geometry object.
 *
 */

void	UTILS_dump_geom_obj (stream, geom_obj, flags)

FILE		*stream;
GEOMobj		*geom_obj;
int		flags;
{
    int			i;

    if (!(flags & DUMP_GEOM_OBJ_FLAG)) return;

    if (geom_obj == NULL) return;

    fprintf (stream, "\n GEOM OBJECT %8.8x\n", geom_obj);
    fprintf (stream, "     OBJ TYPE = %d\n", geom_obj->type);
    fprintf (stream, "     OBJ DATA = %d\n", geom_obj->data);
    fprintf (stream, "     OBJ EXTS = %d", geom_obj->has_extent);
    for (i = 0; i < 6; i++) fprintf (stream, " %f", geom_obj->extent[i]);
    fprintf (stream, "\n"), fflush (stream);
    fprintf_obj_name (stream, "     OBJ NAME = ", geom_obj->name);
    fprintf_obj_name (stream, "     OBJ FNAM = ", geom_obj->filename);
    fprintf_obj_name (stream, "     OBJ GRUP = ", geom_obj->groupname);
    fprintf (stream, "     OBJ NVAL = %d\n", geom_obj->nvals);
    fprintf (stream, "     OBJ RCNT = %d\n", geom_obj->refcnt);
    fprintf (stream, "     OBJ DATA = %8.8x\n", &geom_obj->d);
    fprintf (stream, "\n"), fflush (stream);

    switch (geom_obj->type)
     {
        case GEOM_MESH :
            UTILS_dump_geom_mesh (stream, &geom_obj->d.m, flags);
            break;

        case GEOM_POLYHEDRON :
            UTILS_dump_geom_polyh (stream, &geom_obj->d.ph, flags);
            break;

        case GEOM_POLYTRI :
            UTILS_dump_geom_polytri (stream, &geom_obj->d.pt, flags);
            break;

        case GEOM_SPHERE :
            UTILS_dump_geom_sphere (stream, &geom_obj->d.sp, flags);
            break;

        case GEOM_LABEL :
            UTILS_dump_geom_label (stream, &geom_obj->d.la, flags);
            break;

        default :
            ;
    }
    fflush (stream);

    return;
}



fprintf_obj_name (stream, label, name)

FILE	*stream;
char	*label, *name;
{

    fprintf (stream, "%s", label);
    if (name != NULL) fprintf (stream, "%s", name);
    fprintf (stream, "\n"), fflush (stream);

    return;
}



/*******************************************************************************
 *
 *  Dump a geometry polyhedron object.
 *
 */

void	UTILS_dump_geom_polyh (stream, polyh, flags)

FILE		*stream;
GEOMpolyh	*polyh;
int		flags;
{

    if (!(flags & DUMP_OBJ_DATA_FLAG)) return;

    if (polyh == NULL) return;

    fprintf (stream, " POLYH %10x\n", polyh), fflush (stream);
    UTILS_dump_geom_vtlist (stream, "PH_VERTS  ", &polyh->verts, 0, flags);
    UTILS_dump_geom_vtlist (stream, "PH_NORMALS", &polyh->normals, 0, flags);
    UTILS_dump_geom_vtlist (stream, "PH_COLORS ", &polyh->colors, 0, flags);
    UTILS_dump_geom_vtlist (stream, "PH_UVS    ", &polyh->uvs, 0, flags);
    UTILS_dump_geom_ptlist (stream, "PH_PTLIST ", &polyh->ptlist, 0, flags);

    return;
}



/*******************************************************************************
 *
 *  Dump a geometry polytriangle object.
 *
 */

void	UTILS_dump_geom_polytri (stream, polytri, flags)

FILE		*stream;
GEOMpolytri	*polytri;
int		flags;
{
    int			i;

    if (!(flags & DUMP_OBJ_DATA_FLAG)) return;

    if (polytri == NULL) return;

    fprintf (stream, " POLYTRI %10x\n", polytri), fflush (stream);
    fprintf (stream, "    NPOLYTRI  %10d\n", polytri->npts), fflush (stream);
    for (i = 0; i < polytri->npts; i++)
    {
        fprintf (stream, "     Polytriangle %d\n", i+1), fflush (stream);
        UTILS_dump_geom_vtlist (stream, "PT_VERTS  ", polytri->ptverts, i,
                                flags);
        UTILS_dump_geom_vtlist (stream, "PT_NORMALS", polytri->ptnormals, i,
                                flags);
        UTILS_dump_geom_vtlist (stream, "PT_COLORS ", polytri->ptcolors, i,
                                flags);
        UTILS_dump_geom_vtlist (stream, "PT_UVS    ", polytri->ptuvs, i,
                                flags);
    }
    fprintf (stream, "    NPOLYLIN  %10d\n", polytri->npls), fflush (stream);
    for (i = 0; i < polytri->npls; i++)
    {
        fprintf (stream, "      Polyline %d\n", i+1), fflush (stream);
        UTILS_dump_geom_vtlist (stream, "PL_VERTS  ", polytri->plverts, i,
                                flags);
        UTILS_dump_geom_vtlist (stream, "PL_COLORS ", polytri->plcolors, i,
                                flags);
    }
    UTILS_dump_geom_vtlist (stream, "DL_VERTS  ", &polytri->dlverts, 0, flags);
    UTILS_dump_geom_vtlist (stream, "DL_COLORS ", &polytri->dlcolors, 0, flags);

    return;
}


/*******************************************************************************
 *
 *  Dump a geometry point index list.
 *
 */

void	UTILS_dump_geom_ptlist (stream, label, ptlist, index, flags)

FILE		*stream;
char		*label;
GEOMp_list	*ptlist;
int		index, flags;
{
    GEOMp_list		*pp;
    int			i, j, n, size = 0, np = 0, nv = 0;
    int			*pl = NULL;

    if (!(flags & DUMP_LIST_DATA_FLAG)) return;

    pp = ptlist;
    if (pp != NULL)
    {
        pp += index;
        size = pp->alloced_size;
        np   = pp->nps;
        nv   = pp->nvs;
        pl   = pp->l;
    }
    fprintf (stream, "    %s %10d %10d %10d %10.8x\n", label, size, np, nv, pl);
    fflush (stream);
    for (i = 1; i <= np; i++)
    {
        n = *(pl++);
        fprintf (stream, "\n        Polygon %d, Npts = %d\n", i, n);
        fflush (stream);
        for (j = 0; j < n; j++)
        {
            if ((j % 10) == 0) fprintf (stream, "\n            ");
            fprintf (stream, "%6d", *(pl++)), fflush (stream);
        }
        fprintf (stream, "\n"), fflush (stream);
    }
    fprintf (stream, "\n"), fflush (stream);

    return;
}



/*******************************************************************************
 *
 *  Dump a geometry sphere object.
 *
 */

void	UTILS_dump_geom_sphere (stream, sphere, flags)

FILE		*stream;
GEOMsphere	*sphere;
int		flags;
{

    if (!(flags & DUMP_OBJ_DATA_FLAG)) return;

    if (sphere == NULL) return;

    fprintf (stream, " SPHERE %10x\n", sphere), fflush (stream);
    UTILS_dump_geom_vtlist (stream, "SP_VERTS  ", &sphere->verts, 0, flags);
    UTILS_dump_geom_vtlist (stream, "SP_NORMALS", &sphere->normals, 0, flags);
    UTILS_dump_geom_vtlist (stream, "SP_COLORS ", &sphere->colors, 0, flags);
    UTILS_dump_geom_vtlist (stream, "SP_RADII  ", &sphere->radii, 0, flags);

    return;
}


/*******************************************************************************
 *
 *  Dump a geometry char list.
 *
 */

void	UTILS_dump_geom_chlist (stream, label, chlist, index, flags)

FILE		*stream;
char		*label;
GEOMchar_list	*chlist;
int		index, flags;
{
    GEOMchar_list	*pc;
    int			size = 0, n = 0;
    char		**pl = NULL;

    pc = chlist;
    if (pc != NULL)
    {
        pc += index;
        size = pc->alloced_size;
        n    = pc->n;
        pl   = pc->l;
    }
    fprintf (stream, "    %s %10d %10d %10.8x\n", label, size, n, pl);
    fflush (stream);

    if (!(flags & DUMP_LIST_DATA_FLAG)) return;

    for (; n > 0; n--, pl++)
        fprintf (stream, "         >%s<\n", *pl), fflush (stream);
    fprintf (stream, "\n"), fflush (stream);

    return;
}


/*******************************************************************************
 *
 *  Dump a geometry scalar list.
 *
 */

void	UTILS_dump_geom_sclist (stream, label, sclist, index, flags)

FILE		*stream;
char		*label;
GEOMvert_list	*sclist;
int		index, flags;
{
    GEOMvert_list	*ps;
    int			size = 0, n = 0;
    float		*pl = NULL;

    ps = sclist;
    if (ps != NULL)
    {
        if (index >= 0) ps += index;
        size = ps->alloced_size;
        n    = ps->n;
        pl   = ps->l;
    }
    fprintf (stream, "    %s %10d %10d %10.8x\n", label, size, n, pl);
    fflush (stream);

    if (!(flags & DUMP_LIST_DATA_FLAG)) return;

    for (; n > 0; n--, pl++)
        fprintf (stream, "         %10.5f\n", *pl), fflush (stream);
    fprintf (stream, "\n"), fflush (stream);

    return;
}


/*******************************************************************************
 *
 *  Dump a geometry vertex list.
 *
 */

void	UTILS_dump_geom_vtlist (stream, label, vtlist, index, flags)

FILE		*stream;
char		*label;
GEOMvert_list	*vtlist;
int		index, flags;
{
    GEOMvert_list	*pv;
    int			i, size = 0, n = 0;
    FLOAT3		min, max;
    float		*pl = NULL;

    pv = vtlist;
    if (pv != NULL)
    {
        if (index >= 0) pv += index;
        size = pv->alloced_size;
        n    = pv->n;
        pl   = pv->l;
        for (i = 0; i < 3; i++) min[i] = 1.0e+30, max[i] = -min[i];
    }
    fprintf (stream, "    %s %10d %10d %10.8x\n", label, size, n, pl);
    fflush (stream);

    if (!(flags & DUMP_LIST_DATA_FLAG)) return;

    for (; n > 0; n--, pl += 3)
    {
        fprintf (stream, "         %10.5f %10.5f %10.5f\n",
            *(pl), *(pl+1), *(pl+2));
        fflush (stream);
        for (i = 0; i < 3; i++)
        {
            if (*(pl+i) < min[i]) min[i] = *(pl+i);
            if (*(pl+i) > max[i]) max[i] = *(pl+i);
        }
    }
    if ((pl != NULL) && (index >= 0))
    {
        fprintf (stream, "\n");
        fprintf (stream, "     Min %10.5f %10.5f %10.5f\n",
                 min[0], min[1], min[2]);
        fprintf (stream, "     Max %10.5f %10.5f %10.5f\n",
                 max[0], max[1], max[2]);
        fflush (stream);
    }
    fprintf (stream, "\n"), fflush (stream);

    return;
}



/*******************************************************************************
 *
 *  Make a hex dump of NBYTES starting at PTR
 *
 */

void	UTILS_dump_hex (stream, ptr, nbytes)

FILE		*stream;
unsigned char	*ptr;
int		nbytes;

{

    unsigned char	*p;
    int			i;

    if (ptr != NULL)
    {
        p = ptr;
        i = (long)p & 0x0fL;
        if (i != 0)
        {
            fprintf (stream, "\n %8.8x   ", p);
            for (; i > 0; i-- ) fprintf (stream, "   ");
        }
        for (i = 0; i < nbytes; i++, p++)
        {
            if ((((long)p) & 0x0fL) == 0) fprintf (stream, "\n %8.8x   ", p);
            fprintf (stream, " %2.2x", *p);
        }
    }
    fprintf (stream, "\n");
    fflush (stream);

    return;
}



/*******************************************************************************
 *
 *  Open a file for debugging output.
 *
 */

void	UTILS_dbg_open (dbg_file)

char	*dbg_file;
{
    static char		file[256] = "/home/star1/b/mcdonald/avs/modules/src/";
    int			i;

    i = strlen (file);
    strcat (file, dbg_file);

    dbg = fopen (file, "w");

    file[i] = '\0';


    return;
}
