/*******************************************************************************
 *
 *  These are a set of utility field routines that can be used by 
 *  user-written AVS modules.
 *
 *	29 Mar 91  Phil McDonald, NOAA/ERL/FSL	Original version.
 *
 *	18 Apr 91  Phil McDonald		Add field_dims_get.
 *
 *	08 May 91  Phil McDonald		Corrected uniform coordinate
 *						normalization.
 *
 *	31 May 91  Phil McDonald		Add field_destroy.
 *
 *	03 Jun 91  Phil McDonald		Add field_create, from_laps.
 *
 *	07 Aug 91  Phil McDonald		Some mods for AVS 3.0.
 *
 *	08 Aug 91  Phil McDonald		Add alloc, copy, n funcs.
 *
 *	19 Aug 91  Phil McDonald		Add extents_find, data_mirror_y.
 *
 *	18 Sep 91  Phil McDonald		Add vec_to_float_get.
 *
 *	04 Oct 91  Phil McDonald		Add grid_spacing and
 *						offsets_from_surf.
 *
 *	11 Dec 91  Phil McDonald		Field_from_laps moved to laps.c
 *
 *	27 Mar 92  Phil McDonald		Move point funcs to field_pt.
 *
 ******************************************************************************/



#include	"avs_utils.h"




/*******************************************************************************
 *
 *  Allocate memory for field data.
 *
 */

unsigned char	*UTILS_field_alloc_dat (p_field)

AVSfield	*p_field;
{
    register int	nbytes;

    if (p_field == NULL) return (NULL);

    nbytes = UTILS_field_ndat (p_field) * p_field->size;

    if (nbytes <= 0) return (NULL);

    return ((unsigned char *) malloc (nbytes));
}



/*******************************************************************************
 *
 *  Allocate memory for field coordinates.
 *
 */

float	*UTILS_field_alloc_pts (p_field)

AVSfield	*p_field;
{
    register int	nbytes;

    if (p_field == NULL) return (NULL);

    nbytes = UTILS_field_npts (p_field) * sizeof (float);

    if (nbytes <= 0) return (NULL);

    return ((float *) malloc (nbytes));
}



/*******************************************************************************
 *
 *  Make a copy of a field.
 *
 */

AVSfield	*UTILS_field_copy (p_fld_src)

AVSfield	*p_fld_src;
{
    register int	n;
    AVSfield		*p_fld_dst;

    if (p_fld_src == NULL) return (NULL);

    if ((p_fld_dst = UTILS_field_create (p_fld_src->ndim,
                                         p_fld_src->dimensions,
                                         p_fld_src->nspace,
                                         p_fld_src->veclen,
                                         p_fld_src->type,
                                         p_fld_src->uniform)) == NULL)
                                             return (NULL);

    if (UTILS_field_copy_dat (p_fld_src, p_fld_dst) > 0)
    {
        if ((p_fld_src->min_data != NULL) && (p_fld_src->max_data != NULL))
        {
            n                   = p_fld_src->veclen * p_fld_src->size;
            p_fld_dst->min_data = (unsigned char *) malloc (n);
            p_fld_dst->max_data = (unsigned char *) malloc (n);
            AVSfield_set_minmax (p_fld_dst,
                                 p_fld_src->min_data, p_fld_src->max_data);
        }
        if (p_fld_src->labels == NULL)
        {
            if (p_fld_dst->labels != NULL)
            {
                free (p_fld_dst->labels);
                p_fld_dst->labels = NULL;
            }
        }
        else
        {
            if (p_fld_dst->labels == NULL)
                p_fld_dst->labels = (char *) calloc (AVS_FIELD_LABEL_LEN,
                                                    sizeof (char));
            AVSfield_set_labels (p_fld_dst, (p_fld_src->labels+1),
                                 p_fld_src->labels);
        }
        if (p_fld_src->units == NULL)
        {
            if (p_fld_dst->units != NULL)
            {
                free (p_fld_dst->units);
                p_fld_dst->units = NULL;
            }
        }
        else
        {
            if (p_fld_dst->units == NULL)
                p_fld_dst->units = (char *) calloc (AVS_FIELD_LABEL_LEN,
                                                    sizeof (char));
            AVSfield_set_units (p_fld_dst, (p_fld_src->units+1),
                                 p_fld_src->units);
        }
        if (UTILS_field_copy_pts (p_fld_src, p_fld_dst) > 0) return (p_fld_dst);
    }
    UTILS_field_destroy (p_fld_dst);
    return (NULL);
}



/*******************************************************************************
 *
 *  Copy data from one field to another.
 *
 */

int	UTILS_field_copy_dat (p_fld_src, p_fld_dst)

AVSfield	*p_fld_src, *p_fld_dst;
{
    register int	nbytes;

    if (p_fld_src == NULL) return (0);
    if (p_fld_dst == NULL) return (0);

    if (p_fld_src->field_data == NULL) return (0);
    if (p_fld_dst->field_data == NULL)
        if ((p_fld_dst->field_data = UTILS_field_alloc_dat (p_fld_dst)) == NULL)
            return (0);

    nbytes = UTILS_field_ndat (p_fld_src) * p_fld_src->size;
    if (nbytes > 0)
         memcpy (p_fld_dst->field_data, p_fld_src->field_data, nbytes);

    return (nbytes);
}



/*******************************************************************************
 *
 *  Copy coordinates from one field to another.
 *
 */

int	UTILS_field_copy_pts (p_fld_src, p_fld_dst)

AVSfield	*p_fld_src, *p_fld_dst;
{
    register int	nbytes;

    if (p_fld_src == NULL) return (0);
    if (p_fld_dst == NULL) return (0);

    if (p_fld_src->points == NULL) return (0);
    if (p_fld_dst->points == NULL)
        if ((p_fld_dst->points = UTILS_field_alloc_pts (p_fld_dst)) == NULL)
            return (0);

    nbytes = UTILS_field_npts (p_fld_src) * sizeof (float);
    if (nbytes > 0)
         memcpy (p_fld_dst->points, p_fld_src->points, nbytes);

    return (nbytes);
}



/*******************************************************************************
 *
 *  Create a field.
 *
 */

AVSfield	*UTILS_field_create (ndim, dims, nspace, veclen, type, grid)

int		ndim, *dims, nspace, veclen, type, grid;
{
    AVSfield		*field;
    register int	i;

    if (ndim   <= 0) return (NULL);
    if (nspace <= 0) return (NULL);
    if (veclen <= 0) return (NULL);
    if (type   <  0) return (NULL);
    if (grid   <  0) return (NULL);
    for (i = 0; i < ndim; i++) if (dims[i] <= 0) return (NULL);

    field = (AVSfield *) malloc (sizeof (AVSfield));

    field->ndim         = ndim;
    field->dimensions   = (int *) malloc (ndim * sizeof (int));
    for (i = 0; i < ndim; i++) field->dimensions[i] = dims[i];
    field->nspace       = nspace;
    field->veclen       = veclen;
    field->type         = type;
    field->size         = field_data_size[type];
    field->single_block = FALSE;
    field->shm_key      = 0;
    field->shm_id       = 0;
    field->shm_base     = NULL;
    field->uniform      = grid;
    field->field_data   = NULL;
    field->points       = NULL;
    field->labels       = (char *) calloc (AVS_FIELD_LABEL_LEN, sizeof (char));
    field->units        = (char *) calloc (AVS_FIELD_UNIT_LEN, sizeof (char));
    field->min_extent   = (float *) calloc (nspace, sizeof (float));
    field->max_extent   = (float *) calloc (nspace, sizeof (float));
    field->min_data     = (unsigned char *) calloc (veclen, field->size);
    field->max_data     = (unsigned char *) calloc (veclen, field->size);

    if (grid == UNIFORM)
    {
        field->points = UTILS_field_alloc_pts (field);
        UTILS_field_extents_find (field);
    }

    return (field);
}



/*******************************************************************************
 *
 *  Mirror the data for a field around the Y axis.
 *
 */

void		UTILS_field_data_mirror_y (p_field)

AVSfield	*p_field;
{
    register int	k, j1, j2, n, nx, ny, nz;
    register char	*tmp;

    nx = (p_field->ndim > 0) ? p_field->dimensions[0] : 1;
    ny = (p_field->ndim > 1) ? p_field->dimensions[1] : 1;
    nz = (p_field->ndim > 2) ? p_field->dimensions[2] : 1;

    n   = (p_field->veclen > 1) ? p_field->veclen : 1;
    n   = n * nx * p_field->size;
    tmp = (char *) malloc (n);

    ny = (ny - 1) * n;
    for (k = 0; k < nz; k++)
    {
        j1 = 0, j2 = ny;
        while (j1 < j2)
        {
            memcpy (tmp, (p_field->field_data+j1), n);
            memcpy ((p_field->field_data+j1), (p_field->field_data+j2), n);
            memcpy ((p_field->field_data+j2), tmp, n);
            j1 += n, j2 -= n;
        }
    }
    free (tmp);

    return;
}




/*******************************************************************************
 *
 *  Destroy a field.
 *
 */

AVSfield	*UTILS_field_destroy (field)

AVSfield	*field;
{
    if (field != NULL)
    {
        if (field->dimensions != NULL) free (field->dimensions);
        if (field->field_data != NULL) free (field->field_data);
        if (field->points     != NULL) free (field->points);
        if (field->min_extent != NULL) free (field->min_extent);
        if (field->max_extent != NULL) free (field->max_extent);
        if (field->min_data   != NULL) free (field->min_data);
        if (field->max_data   != NULL) free (field->max_data);
        if (field->labels     != NULL) free (field->labels);
        if (field->units      != NULL) free (field->units);
        free (field);
    }

    return (NULL);
}




/*******************************************************************************
 *
 *  Return the dimensions of a field.
 *
 */

void	UTILS_field_dims_get (p_field, ndim, dim0, dim1, dim2)

AVSfield	*p_field;
int		*ndim, *dim0, *dim1, *dim2;
{

    *ndim = *dim0 = *dim1 = *dim2 = 0;
    if (p_field != NULL)
    {
        *ndim = p_field->ndim;
        *dim0 = *dim1 = *dim2 = 1;
        if (*ndim > 0)
        {
            *dim0 = p_field->dimensions[0];
            if (*ndim > 1)
            {
                *dim1 = p_field->dimensions[1];
                if (*ndim > 2) *dim2 = p_field->dimensions[2];
            }
        }
    }

    return;
}




/*******************************************************************************
 *
 *  Find the extents of a field.
 *
 */

void	UTILS_field_extents_find (p_field)

AVSfield	*p_field;
{
    register int	i, j, k, m, n;
    register int	nx, ny, nz;
    FLOAT3		xyz;

    n = (p_field->nspace < 1) ? 1 : (p_field->nspace > 3) ? 3 : p_field->nspace;

    nx = (p_field->ndim > 0) ? p_field->dimensions[0] : 1;
    ny = (p_field->ndim > 1) ? p_field->dimensions[1] : 1;
    nz = (p_field->ndim > 2) ? p_field->dimensions[2] : 1;

    if (p_field->uniform == UNIFORM)
    {
        for (m = 0; m < n; m++)
        {
            p_field->min_extent[m] = 0.0;
            p_field->max_extent[m] = 0.0;
        }
        if (nx > 1) p_field->max_extent[0] = nx - 1;
        if (ny > 1) p_field->max_extent[1] = ny - 1;
        if (nz > 1) p_field->max_extent[2] = nz - 1;

        if (p_field->points != NULL)
        {
            for (i = 0; i < p_field->nspace; i++)
            {
                p_field->points[(i*2)+0] = p_field->min_extent[i];
                p_field->points[(i*2)+1] = p_field->max_extent[i];
            }
        }
    }

    else	/* RECTILINEAR or IRREGULAR	*/
    {
        for (m = 0; m < n; m++)
        {
            p_field->min_extent[m] = 1.0e+30;
            p_field->max_extent[m] = -1.0e+30;
        }

        UTILS_field_pt_init (p_field);

        for  (k = 0; k < nz; k++)
        {
            for (j = 0; j < ny; j++)
            {
                for (i = 0; i < nx; i++)
                {
                    UTILS_field_pt_xyz (i, j, k, xyz);
                    for (m = 0; m < n; m++)
                    {
                        if (xyz[m] < p_field->min_extent[m])
                            p_field->min_extent[m] = xyz[m];
                        if (xyz[m] > p_field->max_extent[m])
                            p_field->max_extent[m] = xyz[m];
                    }
                }
            }
        }
    }

    return;
}


/*******************************************************************************
 *
 *  Return a approximation of the grid spacings of a field.
 *
 */

float	UTILS_field_grid_spacing (p_field, xyz)

AVSfield	*p_field;
FLOAT3		xyz;
{
    register FLOAT3	xyz1, xyz2;
    int			i, j, k, n;

    xyz[0] = xyz[1] = xyz[2] = 0.0;

    if (p_field == NULL) return (0.0);

    UTILS_field_pt_init (p_field);
    UTILS_field_pt_limits (xyz1, xyz2);

    UTILS_field_dims_get (p_field, &n, &i, &j, &k);

    if (i > 1) xyz[0] = (xyz2[0] - xyz1[0]) / ((float) (i - 1));
    if (j > 1) xyz[1] = (xyz2[1] - xyz1[1]) / ((float) (j - 1));
    if (k > 1) xyz[2] = (xyz2[2] - xyz1[2]) / ((float) (k - 1));
    if (xyz[0] < 0.0) xyz[0] = -xyz[0];
    if (xyz[1] < 0.0) xyz[1] = -xyz[1];
    if (xyz[2] < 0.0) xyz[2] = -xyz[2];

    return ((float) sqrt ((double) (xyz[0] * xyz[0]) +
                          (double) (xyz[1] * xyz[1]) +
                          (double) (xyz[2] * xyz[2])));
}


/*******************************************************************************
 *
 *  Return the number of data elements in a field.
 *
 */

int	UTILS_field_ndat (p_field)

AVSfield	*p_field;
{
    register int	ndat, i;

    ndat = 0;

    if (p_field != NULL)
    {
        if (p_field->ndim > 0)
        {
            ndat = p_field->veclen;
            for (i = 0; i < p_field->ndim; i++)
                 ndat *= p_field->dimensions[i];
        }
    }

    return (ndat);
}



/*******************************************************************************
 *
 *  Return the number of points in a field.
 *
 */

int	UTILS_field_npts (p_field)

AVSfield	*p_field;
{
    register int	npts, i;

    npts = 0;

    if (p_field != NULL)
    {
        if (p_field->ndim > 0)
        {
            if (p_field->uniform == UNIFORM)
            {
                npts = p_field->nspace * 2;	/* For "extents"	*/
            }

            else if (p_field->uniform == RECTILINEAR)
            {
                for (i = 0; i < p_field->ndim; i++)
                    npts += p_field->dimensions[i];
            }

            else if (p_field->uniform == IRREGULAR)
            {
                npts = p_field->nspace;
                for (i = 0; i < p_field->ndim; i++)
                    npts *= p_field->dimensions[i];
            }
        }
    }

    return (npts);
}



/*******************************************************************************
 *
 *  Return approximate offsets from the surface of a 2D, 3-space field
 *  necessary for lines to appear slightly off of the surface.
 *
 *
 */

void	UTILS_field_offsets_from_surf (p_field, offset)

AVSfield	*p_field;
FLOAT3		offset;
{
    register int	i;
    register float	x;

    offset[0] = offset[1] = offset[2] = 0.0;

    if (p_field == NULL) return;

    x = UTILS_field_grid_spacing (p_field, offset);
    i = 0;
    if (offset[1] < offset[i]) i = 1;
    if (offset[2] < offset[i]) i = 2;
    offset[0] = offset[1] = offset[2] = 0.0;
    offset[i] = x * FUDGE_LINE_FROM_SURF;
    offset[1] = -offset[1];


    return;
}



/*******************************************************************************
 *
 *  Add to a geometry object those polygons that make up an X-plane cross
 *  section of a field.  The top or bottom of these polygons will be on 
 *  the surface of the field; the opposite end will have z values of ZBASE. 
 *
 */

void	UTILS_field_xsectx_add (p_field, p_colors, ixplane, iy1, iy2, norm, 
                                zbase, geom_obj)

AVSfield	*p_field;
float		*p_colors;
int		ixplane, iy1, iy2, norm;
float		zbase;
GEOMobj		*geom_obj;
{
    register FLOAT3	pt1, pt2, verts[4];
    register int	ix, iy, iylast, iyinc, i1, i2;

    ix     = ixplane;
    iy     = iy1;
    iylast = iy2;
    iyinc  = (iy2 < iy1) ? -1 : 1;

    if (norm > 0)
        i1 = 1, i2 = 3;
    else
        i1 = 3, i2 = 1;

    UTILS_field_pt_init (p_field);

    UTILS_field_pt_xyz (ix, iy, 0, pt1);
    while (iy != iylast)
    {
        iy += iyinc;
        UTILS_field_pt_xyz (ix, iy, 0, pt2);
        UTILS_float3_copy (pt1, (verts));
        UTILS_float3_set (pt2[0], pt2[1], zbase, (verts+2));
        if (pt2[2] <= zbase)
        {
            UTILS_float3_copy (pt2, (verts+i1));
            UTILS_float3_set (pt1[0], pt1[1], zbase, (verts+i2));
        }
        else
        {
            UTILS_float3_set (pt1[0], pt1[1], zbase, (verts+i1));
            UTILS_float3_copy (pt2, (verts+i2));
        }
        GEOMadd_disjoint_polygon (geom_obj, verts, GEOM_NULL, p_colors, 4, 
                                  GEOM_NOT_SHARED | GEOM_CONVEX,
                                  GEOM_COPY_DATA);
        UTILS_float3_copy (pt2, pt1);
    }

    return;
}




/*******************************************************************************
 *
 *  Add to a geometry object those polygons that make up a Y-plane cross
 *  section of a field.  The top or bottom of these polygons will be on 
 *  the surface of the field; the opposite end will have z values of ZBASE. 
 *
 */

void	UTILS_field_xsecty_add (p_field, p_colors, iyplane, ix1, ix2, norm, 
                                zbase, geom_obj)

AVSfield	*p_field;
float		*p_colors;
int		iyplane, ix1, ix2, norm;
float		zbase;
GEOMobj		*geom_obj;
{
    register FLOAT3	pt1, pt2, verts[4];
    register int	iy, ix, ixlast, ixinc, i1, i2;

    iy     = iyplane;
    ix     = ix1;
    ixlast = ix2;
    ixinc  = (ix2 < ix1) ? -1 : 1;

    if (norm > 0)
        i1 = 1, i2 = 3;
    else
        i1 = 3, i2 = 1;

    UTILS_field_pt_init (p_field);

    UTILS_field_pt_xyz (ix, iy, 0, pt1);
    while (ix != ixlast)
    {
        ix += ixinc;
        UTILS_field_pt_xyz (ix, iy, 0, pt2);
        UTILS_float3_copy (pt1, (verts));
        UTILS_float3_set (pt2[0], pt2[1], zbase, (verts+2));
        if (pt2[2] <= zbase)
        {
            UTILS_float3_copy (pt2, (verts+i1));
            UTILS_float3_set (pt1[0], pt1[1], zbase, (verts+i2));
        }
        else
        {
            UTILS_float3_set (pt1[0], pt1[1], zbase, (verts+i1));
            UTILS_float3_copy (pt2, (verts+i2));
        }
        GEOMadd_disjoint_polygon (geom_obj, verts, GEOM_NULL, p_colors, 4, 
                                  GEOM_NOT_SHARED | GEOM_CONVEX,
                                  GEOM_COPY_DATA);
        UTILS_float3_copy (pt2, pt1);
    }

    return;
}
