/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/*   
     This software is copyright (C) 1991,  Regents  of  the
University  of  California.   Anyone may reproduce mesh_axis.c,
the software in this distribution, in whole or in part, pro-
vided that:

(1)  Any copy  or  redistribution  of  mesh_axis.c  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;

(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
in  mesh_axis.c  under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     mesh_axis.c is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
The Regents of the University of California  shall  have  no
liability  with respect to the infringement of copyrights by
mesh_axis.c, or any part thereof.     

*/

/*
 *  filename mesh_axis.c
 *  Tim Robinson, UCB College of Chemistry. Feb 1992
 *  Wes Bethel, LBL. May 1991
 *  Wayne Hurlbert , LBL. May 1990
 *  
 *
 *  Generate 3D coordinate axis for mesh data.
 *  This module is a modified version of the LBL 3D axis module.  The module
 *  is designed to take a 2-D field of f(x,y) which is being mapped to a
 *  3-D mesh by another module.  This module outputs axes, tick marks, and 
 *  labels which will reside on-screen with the rendered mesh.  The scaling
 *  paradigm is flexible in the "z" dimension, allowing a scale factor for
 *  the z-axis.  This lets users "blow up" their mesh without worrying about
 *  the displayed units becoming irrelevant.  Also, the initial pass search   
 *  was modified to handle meshes which had their maximum or minimum x, y 
 *  or z value for RECTILINEAR or IRREGULAR data at an arbitrary point.  
 *  Base font switched to helvetica which looks better because of its narrow
 *  width.
 *  The comments from the previous authors follow:
 *
 *  AVS 2 comments:
 *  UNIFORM data sets are mapped into a 3D space of dimensions -1 to 1, so
 *  a line drawn from say, (-1,-1,-1) to (1,1,1), will go from the far lower
 *  left to the near upper right of the data set.  Here, (-1,-1,-1) is used
 *  as the origin of the 3D axis, but ticks and labels are based on the
 *  dimensions of the data set (0 - MAXX, 0 - MAXY, 0 - MAXZ).  These min and
 *  max values may be changed by the user, but such changes will only change
 *  ticks and labels, not the drawn axis.
 **
     AVS 3 revision:
     UNIFORM fields are now mapped into a space corresponding to
     0..size-1 for each of the data dimensions.  This mapping corresponds
     to the AVS3 world of uniform fields. 

     Access to the world of AVS3 is enabled by using the compile
     flag -DAVS3.

     Significant reworking of the loops for tick marks, labels and
     meshes have been redone to reflect this change, and also to
     support having the "min" value greater than the "max" value.

     22 May 1991 w.bethel
 **
 *  RECTILINEAR and IRREGULAR data sets, and the axes generated here, are
 *  drawn in the world space of the coordinate info stored in the data set.
 *  The origin of the axis system is placed at the minimum coordinate value
 *  of each dimension.  Note that this will probably not coincide with the
 *  world origin (0,0,0), but it is usually the far lower left corner of the
 *  data set.  This means that changing the axis minimum and maximum values
 *  will change the positions of the drawn axis, but will not change label
 *  values or the screen location of any particular point.
 */

#include <stdio.h>
/* IAC CODE CHANGE : #include <math.h> */
#include <avs/avs_math.h>

#include <avs/vex.h>
#include <avs/flow.h>
#include <avs/field.h>
#include <avs/geom.h>
#include <avs/geomdata.h>

#define max(a, b) ((a) > (b) ? (a) : (b))
#define min(a, b) ((a) < (b) ? (a) : (b))

int first_time = TRUE;
int avs2 = 0;
float zref[2];


/********** AVS initialization ************************************************/

AVSinit_modules()
{
    int axis_desc();
    AVSmodule_from_desc(axis_desc);
}


/******************************************************************************/

int axis_desc()
{
    int axis_compute();
    int parm;

    AVSset_module_name("mesh axis", MODULE_MAPPER);

    AVScreate_input_port("Field Input", "field", REQUIRED);
    AVScreate_input_port("Z scale","real", OPTIONAL);
    AVScreate_output_port("Geometry", "geom");

    parm = AVSadd_parameter("normalized uniform field","boolean",0,0,1);
    AVSadd_parameter_prop(parm,"width","integer",3);
    /*
     *  Set number of ticks per axis, and tick size.
     */
    parm = AVSadd_parameter("message1","string","Set ticks:", "","");
    AVSconnect_widget(parm,"text");
    AVSadd_parameter_prop(parm,"width","integer",4);
    parm = AVSadd_parameter("x axis ticks", "integer", 0, 0, 50);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("y axis ticks", "integer", 0, 0, 50);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("z axis ticks", "integer", 0, 0, 50);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_float_parameter("tick size", 0.01, 0.0, 1.0);
    AVSconnect_widget(parm,"slider");


    /*
     *  Turn planes on and off, and set the position of the plane in the
     *  3rd dimension. The plane mesh is determined by the tick spacing.
     */
    parm = AVSadd_parameter("message2","string","Set mesh planes:", "","");
    AVSconnect_widget(parm,"text");
    AVSadd_parameter_prop(parm,"width","integer",4);
    parm = AVSadd_float_parameter("xz plane position", 0.0, 0.0, 1.0);
    AVSconnect_widget(parm,"slider");
    parm = AVSadd_float_parameter("xy plane position", 0.0, 0.0, 1.0);
    AVSconnect_widget(parm,"slider");
    parm = AVSadd_float_parameter("yz plane position", 0.0, 0.0, 1.0);
    AVSconnect_widget(parm,"slider");
    AVSadd_parameter("Show xz plane", "boolean", 0, 0, 1);
    AVSadd_parameter("Show xy plane", "boolean", 0, 0, 1);
    AVSadd_parameter("Show yz plane", "boolean", 0, 0, 1);


    /*
     *  Turn tick labels on and off, set font size, set precision, set
     *  number of ticks per label, and set which axis labels the origin.
     */
    parm = AVSadd_parameter("message3","string","Set labels:", "","");
    AVSconnect_widget(parm,"text");
    AVSadd_parameter_prop(parm,"width","integer",4);
    parm = AVSadd_float_parameter("font size", 0.07, 0.01, 0.1);
    AVSconnect_widget(parm,"slider");
    parm = AVSadd_parameter("label precision", "integer", 1, 0, 9);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("x label density", "integer", 1, 1, 100);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("y label density", "integer", 1, 1, 100);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("z label density", "integer", 1, 1, 100);
    AVSconnect_widget(parm,"islider");

    parm = AVSadd_parameter("origin labels","string","","","");
    AVSconnect_widget(parm,"typein");
    AVSadd_parameter("Show labels","boolean", 0, 0, 1);


    /*
     *  Show the min and max values of the axis, as defined by the data
     *  set.  Then allow the user to change those values.  For a UNIFORM
     *  data set this simply changes the values associated with the axis
     *  ticks.  For RECTILINEAR and IRREGULAR data sets, this changes the
     *  actual positions of the axis endpoints.
     */
    parm = AVSadd_parameter("message4","string","Set axis min and max:", "","");
    AVSconnect_widget(parm,"text");
    AVSadd_parameter_prop(parm,"width","integer",4);
    parm = AVSadd_float_parameter("x axis min ", 0.0,
				  FLOAT_UNBOUND, FLOAT_UNBOUND);
    AVSconnect_widget(parm,"typein_real");
    parm = AVSadd_float_parameter("x axis max", 1.0,
				  FLOAT_UNBOUND, FLOAT_UNBOUND);
    AVSconnect_widget(parm,"typein_real");
    parm = AVSadd_float_parameter("y axis min ", 0.0,
				  FLOAT_UNBOUND, FLOAT_UNBOUND);
    AVSconnect_widget(parm,"typein_real");
    parm = AVSadd_float_parameter("y axis max", 1.0,
				  FLOAT_UNBOUND, FLOAT_UNBOUND);
    AVSconnect_widget(parm,"typein_real");
    parm = AVSadd_float_parameter("z axis min ", 0.0,
				  FLOAT_UNBOUND, FLOAT_UNBOUND);
    AVSconnect_widget(parm,"typein_real");
    parm = AVSadd_float_parameter("z axis max", 1.0,
				  FLOAT_UNBOUND, FLOAT_UNBOUND);
    AVSconnect_widget(parm,"typein_real");

    AVSset_compute_proc(axis_compute);
}


/******************************************************************************/

axis_compute(input, zscale, output, norm, mess1, x_ticks, y_ticks, z_ticks,
	     t_size, mess2, xz_pos, xy_pos, yz_pos, xz_pl, xy_pl, yz_pl,
	      mess3, font_size, precision, x_lab_den, y_lab_den,
	     z_lab_den, origin_labels, show_labels,
	     mess4, x_min, x_max, y_min, y_max, z_min, z_max)

AVSfield *input;
float *zscale;
GEOMedit_list **output;
int norm;                            /* turns on normalized field code */
char *mess1,*mess2,*mess3,*mess4;    /* Dummy messages for titles.  */
int x_ticks,y_ticks,z_ticks;         /* Number of ticks per axis.   */
float *t_size;                       /* Tick size.                  */
float *xz_pos,*xy_pos,*yz_pos;       /* Positioning of planes.      */
int xz_pl,xy_pl,yz_pl;               /* Plane on/off switchs.       */
float *font_size;                    /* Size of label font.         */
int precision;                       /* Label precision. */
int show_labels;                     /* Turn labels on and off. */
char *origin_labels;                 /* Axis whose labels are at origin, */
int x_lab_den,y_lab_den,z_lab_den;   /* Number of ticks per label.  */
float *x_min,*x_max,*y_min,*y_max,*z_min,*z_max;  /* Settable axis endpoints. */
{
    GEOMobj *axis_obj;
    GEOMobj *label_x_obj;
    GEOMobj *label_y_obj;
    GEOMobj *label_z_obj;
    int label_flg;
    float axis_pts[6];
    int i, size, num_ticks[3], which_planes[3], label_density[3];
    float where[3], zfact;

    if (input->ndim != 2 && input->nspace != 2) {
	AVSerror("Input field must have 2 dimensions in computational or coordinate space.");
	return(0);
    }
    if (!(zscale)) zfact = 1.0;
    else zfact = *zscale;
    /* If the field has changed, we've got to re-deterimine max & mins */
    if (AVSinput_changed("Field Input",0)) first_time = TRUE; 
    if (AVSparameter_changed("normalized uniform field")) {
       avs2 = norm;
       first_time = TRUE;
       }

    *output     = (GEOMedit_list *)GEOMinit_edit_list(*output);
    axis_obj    = GEOMcreate_obj(GEOM_POLYTRI, NULL);
    label_flg   = GEOMcreate_label_flags(5,0,0,0,GEOM_LABEL_CENTER,0);
    label_x_obj = GEOMcreate_label(GEOM_NULL,label_flg);
    label_z_obj = GEOMcreate_label(GEOM_NULL,label_flg);
    label_flg   = GEOMcreate_label_flags(5,0,0,0,GEOM_LABEL_RIGHT,0);
    label_y_obj = GEOMcreate_label(GEOM_NULL,label_flg);

    /* Put this stuff in vectors for ease of passing. */
    num_ticks[0] = x_ticks;
    num_ticks[1] = y_ticks;
    num_ticks[2] = z_ticks;
    where[0] = *xz_pos;
    where[1] = *xy_pos;
    where[2] = *yz_pos;
    which_planes[0] = xz_pl;
    which_planes[1] = xy_pl;
    which_planes[2] = yz_pl;
    label_density[0] = x_lab_den;
    label_density[1] = y_lab_den;
    label_density[2] = z_lab_den;

    switch(input->uniform) {
	case UNIFORM:
	    if (first_time) {  /* Show axis min and max points of data set. */
		axis_pts[0] = axis_pts[2] = axis_pts[4] =  1.0e6;
		axis_pts[1] = axis_pts[3] = axis_pts[5] = -1.0e6;
		first_time = FALSE;
		AVSmodify_float_parameter("x axis min ",AVS_VALUE,0.0,0.0,0.0);
		AVSmodify_float_parameter("x axis max",AVS_VALUE,
					  (float)(input->dimensions[0]-1), 0.0, 0.0);
		AVSmodify_float_parameter("y axis min ",AVS_VALUE,0.0,0.0,0.0);
		AVSmodify_float_parameter("y axis max",AVS_VALUE,
					  (float)(input->dimensions[1]-1), 0.0, 0.0);
		for (i = 0;i < MAXX(input) * MAXY(input);
		     i++)
		{
		switch (input->type)
		   {
		   case AVS_TYPE_BYTE:
		      axis_pts[4] = min(axis_pts[4], 
			 input->field_union.field_data_char[i]);
		      axis_pts[5] = max(axis_pts[5],
			 input->field_union.field_data_char[i]);
		   break;
		   case AVS_TYPE_INTEGER:
		      axis_pts[4] = min(axis_pts[4],
			 input->field_union.field_data_int_u[i]);
		      axis_pts[5] = max(axis_pts[5],
			 input->field_union.field_data_int_u[i]);
		   break;
		   case AVS_TYPE_REAL:
		      axis_pts[4] = min(axis_pts[4],
			 input->field_union.field_data_float_u[i]);
		      axis_pts[5] = max(axis_pts[5],
			 input->field_union.field_data_float_u[i]);
		   break;
		   case AVS_TYPE_DOUBLE:
		      axis_pts[4] = min(axis_pts[4],
			 input->field_union.field_data_double_u[i]);
		      axis_pts[5] = max(axis_pts[5],
			 input->field_union.field_data_double_u[i]);
		   break;
		   }
		}
                if (avs2) {
		zref[0] = axis_pts[4];
		zref[1] = axis_pts[5];
		}
		AVSmodify_float_parameter("z axis min ",AVS_VALUE,
					   axis_pts[4], 0.0,0.0);
		AVSmodify_float_parameter("z axis max",AVS_VALUE,
					  axis_pts[5], 0.0, 0.0);
	    }

            if (!(avs2)) {
	    axis_pts[0] = *x_min;
	    axis_pts[1] = *x_max;
	    axis_pts[2] = *y_min;
	    axis_pts[3] = *y_max;
	    axis_pts[4] = *z_min * zfact;
	    axis_pts[5] = *z_max * zfact;
	    }
            else {
/* Draw the axis in normalized space. OK for AVS 2 UNIFORM */
	    axis_pts[0] = axis_pts[2] = -1.0;
	    axis_pts[1] = axis_pts[3] = 1.0;
	    axis_pts[4] = zref[0] * zfact;
	    axis_pts[5] = zref[1] * zfact;
	    }

	    make_axis(axis_obj,axis_pts);

	    /* Everyone else needs data set size. */
	    axis_pts[0] = *x_min;
	    axis_pts[1] = *x_max;
	    axis_pts[2] = *y_min;
	    axis_pts[3] = *y_max;
	    axis_pts[4] = *z_min * zfact;
	    axis_pts[5] = *z_max * zfact;

	    make_ticks(axis_obj,axis_pts,num_ticks,t_size,input->uniform,zfact);
	    make_planes(axis_obj,axis_pts,num_ticks,which_planes,where,
			input->uniform,zfact);
	    make_labels(axis_pts,num_ticks,input->uniform, label_x_obj,
			label_y_obj,label_z_obj, show_labels,precision,
			label_density,font_size,origin_labels,zfact);
	    break;
	case RECTILINEAR:
	case IRREGULAR:
	    /*
	     *  Here we search all the stored coordinate data for the
	     *  min and max points the first time through.
	     */
	    if (first_time) {
		axis_pts[0] = axis_pts[2] = axis_pts[4] =  1.0e6;
		axis_pts[1] = axis_pts[3] = axis_pts[5] = -1.0e6;
		first_time = FALSE;
		if (input->uniform == RECTILINEAR) {   /* find extents */
		    size = MAXX(input) * MAXY(input);
		    for(i = 0; i < MAXX(input); i++) {
			axis_pts[0] = min(axis_pts[0], input->points[i]);
			axis_pts[1] = max(axis_pts[1], input->points[i]);
		    }
		    for(i = MAXX(input); i < MAXX(input) + MAXY(input); i++) {
			axis_pts[2] = min(axis_pts[2], input->points[i]);
			axis_pts[3] = max(axis_pts[3], input->points[i]);
		    }
		    for (i = 0;i < input->dimensions[0] * input->dimensions[1];
			 i++) {
		    switch (input->type)
		       {
			  case AVS_TYPE_BYTE:
			  axis_pts[4] = min(axis_pts[4], 
			    input->field_union.field_data_char[i]);
			  axis_pts[5] = max(axis_pts[5],
			    input->field_union.field_data_char[i]);
			  break;
			  case AVS_TYPE_INTEGER:
			  axis_pts[4] = min(axis_pts[4],
			    input->field_union.field_data_int_u[i]);
			  axis_pts[5] = max(axis_pts[5],
			    input->field_union.field_data_int_u[i]);
			  break;
			  case AVS_TYPE_REAL:
			  axis_pts[4] = min(axis_pts[4],
			     input->field_union.field_data_float_u[i]);
			  axis_pts[5] = max(axis_pts[5],
			     input->field_union.field_data_float_u[i]);
			  break;
			  case AVS_TYPE_DOUBLE:
			  axis_pts[4] = min(axis_pts[4],
			     input->field_union.field_data_double_u[i]);
			  axis_pts[5] = max(axis_pts[5],
			     input->field_union.field_data_double_u[i]);
			  break;
		       }
		    }
		} else {  /* IRREGULAR, find extents */
		    for (i = 0, size = 1; i < input->ndim; ++i)
			size *= input->dimensions[i];
		    fprintf(stderr,"size = %d\n",size);
		    for(i=0; i<size; i++) {
			axis_pts[0] = min(axis_pts[0], input->points[i]);
			axis_pts[1] = max(axis_pts[1], input->points[i]);
			axis_pts[2] = min(axis_pts[2], input->points[i+size]);
			axis_pts[3] = max(axis_pts[3], input->points[i+size]);
			switch (input->type)
			{
			  case AVS_TYPE_BYTE:
			  axis_pts[4] = min(axis_pts[4], 
			    input->field_union.field_data_char[i]);
			  axis_pts[5] = max(axis_pts[5],
			    input->field_union.field_data_char[i]);
			  break;
			  case AVS_TYPE_INTEGER:
			  axis_pts[4] = min(axis_pts[4],
			    input->field_union.field_data_int_u[i]);
			  axis_pts[5] = max(axis_pts[5],
			    input->field_union.field_data_int_u[i]);
			  break;
			  case AVS_TYPE_REAL:
			  axis_pts[4] = min(axis_pts[4],
			     input->field_union.field_data_float_u[i]);
			  axis_pts[5] = max(axis_pts[5],
			     input->field_union.field_data_float_u[i]);
			  break;
			  case AVS_TYPE_DOUBLE:
			  axis_pts[4] = min(axis_pts[4],
			     input->field_union.field_data_double_u[i]);
			  axis_pts[5] = max(axis_pts[5],
			     input->field_union.field_data_double_u[i]);
			  break;
			}
		    }
		}
		AVSmodify_float_parameter("x axis min ",AVS_VALUE,
					  axis_pts[0], 0.0, 0.0);
		AVSmodify_float_parameter("x axis max",AVS_VALUE,
					  axis_pts[1], 0.0, 0.0);
		AVSmodify_float_parameter("y axis min ",AVS_VALUE,
					  axis_pts[2], 0.0, 0.0);
		AVSmodify_float_parameter("y axis max",AVS_VALUE,
					  axis_pts[3], 0.0, 0.0);
		AVSmodify_float_parameter("z axis min ",AVS_VALUE,
					  axis_pts[4], 0.0, 0.0);
		AVSmodify_float_parameter("z axis max",AVS_VALUE,
					  axis_pts[5], 0.0, 0.0);
	    }
	    axis_pts[0] = *x_min;
	    axis_pts[1] = *x_max;
	    axis_pts[2] = *y_min;
	    axis_pts[3] = *y_max;
	    axis_pts[4] = *z_min * zfact;
	    axis_pts[5] = *z_max * zfact;

	    make_axis(axis_obj,axis_pts);
	    make_ticks(axis_obj,axis_pts,num_ticks,t_size,input->uniform,zfact);
	    make_planes(axis_obj,axis_pts,num_ticks,which_planes,where,
			input->uniform,zfact);
	    make_labels(axis_pts,num_ticks,input->uniform, label_x_obj,
			label_y_obj,label_z_obj, show_labels,precision,
			label_density,font_size,origin_labels,zfact);
	    break;
	default : break;
    } /* of switch */

    /* Add into the edit list */
    GEOMedit_geometry(*output, "mesh_axis", axis_obj);
    GEOMedit_geometry(*output, "mesh_axis", label_x_obj);
    GEOMedit_geometry(*output, "mesh_axis", label_y_obj);
    GEOMedit_geometry(*output, "mesh_axis", label_z_obj);
    /* Now kill it all. */
    GEOMdestroy_obj(axis_obj);
    GEOMdestroy_obj(label_x_obj);
    GEOMdestroy_obj(label_y_obj);
    GEOMdestroy_obj(label_z_obj);
    return(1);
}


/******************************************************************************/

make_axis(obj,axis_pts)
GEOMobj *obj;
float axis_pts[6];
/*
 *  This function loads 3D coordinates for 3 lines into axis_lines[].
 *  These lines will then be added to a list of ines for display
 *  by GEOMadd_disjoint_line().  axis_lines[] stores the lines as
 *  x, y, and z axis, where each axis is specified by the three
 *  coordinate values for its 2 endpoints (axis_lines[0][?] and
 *  axis_lines[1][?] make up the x axis).  Colors are stored in
 *  axis_color by rgb values in a similar way.
 */
{
    float axis_color[6][3],
	  axis_lines[6][3];

    axis_lines[0][0] = axis_pts[0];    /* x axis */
    axis_lines[0][1] = axis_pts[2];
    axis_lines[0][2] = axis_pts[4];

    axis_lines[1][0] = axis_pts[1];
    axis_lines[1][1] = axis_pts[2];
    axis_lines[1][2] = axis_pts[4];

    axis_lines[2][0] = axis_pts[0];    /* y axis */
    axis_lines[2][1] = axis_pts[2];
    axis_lines[2][2] = axis_pts[4];

    axis_lines[3][0] = axis_pts[0];
    axis_lines[3][1] = axis_pts[3];
    axis_lines[3][2] = axis_pts[4];

    axis_lines[4][0] = axis_pts[0];    /* z axis */
    axis_lines[4][1] = axis_pts[2];
    axis_lines[4][2] = axis_pts[4];

    axis_lines[5][0] = axis_pts[0];
    axis_lines[5][1] = axis_pts[2];
    axis_lines[5][2] = axis_pts[5];

    axis_color[0][0] = 1.0;
    axis_color[0][1] = 0.0;
    axis_color[0][2] = 0.0;

    axis_color[1][0] = 1.0;
    axis_color[1][1] = 0.0;
    axis_color[1][2] = 0.0;

    axis_color[2][0] = 0.0;
    axis_color[2][1] = 1.0;
    axis_color[2][2] = 0.0;

    axis_color[3][0] = 0.0;
    axis_color[3][1] = 1.0;
    axis_color[3][2] = 0.0;

    axis_color[4][0] = 0.0;
    axis_color[4][1] = 1.0;
    axis_color[4][2] = 1.0;

    axis_color[5][0] = 0.0;
    axis_color[5][1] = 1.0;
    axis_color[5][2] = 1.0;

    GEOMadd_disjoint_line(obj,axis_lines,axis_color,6,GEOM_COPY_DATA);
}

make_ticks(axis_obj,axis_pts,num_ticks,tick_size,data_type,zfact)
GEOMobj *axis_obj;
float axis_pts[6];
int num_ticks[3];
float *tick_size;
int data_type;
float zfact;
/*
 *  This function places ticks along each axis.  There are actually 2
 *  marks, one parallel to each of the other 2 axis.  We get the
 *  increment value and work our way along the axis making the 2 marks.
 *
 *  In the case of a UNIFORM data set, axis_pts[] contains the dimensions
 *  of the data set, but the actual plotting is normalized to the interval
 *  -1 to 1.  So we first save the dimensions and then for each axis get
 *  an increment for the dimensions, normalize the increment, and change
 *  axis_pts[] to normalized dimensions (-1 to 1).  (After each axis is
 *  done we restore axis_pts[] to the original dimensions so we can do
 *  the next one.)
 */
{
    float get_increment();
    float increment;
    float loc;
    float t_size = 2 * *tick_size;
    float axis_tick[4][3];
    float tick_color[4][3];
    int i;
    float temp_axis_pts[6];

    if (avs2) {
    if (data_type == UNIFORM)
	for (i = 0; i < 6; ++i)
	    temp_axis_pts[i] = axis_pts[i];
    }

    /* Ticks on x axis. */
    if (num_ticks[0]) {
	increment = get_increment(num_ticks[0],axis_pts[0],axis_pts[1]);
    if (avs2) {
	if (data_type == UNIFORM)
	    normalize(&increment,axis_pts,axis_pts[0],axis_pts[1],zfact);
    }
	set_tick_color(tick_color, 1.0, 0.0, 0.0);
	loc = axis_pts[0] + increment;
	for (i=0;i<=num_ticks[0];i++)
	{
	    axis_tick[0][0] = loc;
	    axis_tick[0][1] = axis_pts[2];
	    axis_tick[0][2] = axis_pts[4] + t_size;

	    axis_tick[1][0] = loc;
	    axis_tick[1][1] = axis_pts[2];
	    axis_tick[1][2] = axis_pts[4];

	    axis_tick[2][0] = loc;
	    axis_tick[2][1] = axis_pts[2] + t_size;
	    axis_tick[2][2] = axis_pts[4];

	    axis_tick[3][0] = loc;
	    axis_tick[3][1] = axis_pts[2];
	    axis_tick[3][2] = axis_pts[4];

	    GEOMadd_disjoint_line(axis_obj,axis_tick,tick_color,4,GEOM_COPY_DATA);
	    loc += increment;
	}
	if (avs2) {
	if (data_type == UNIFORM)
	    for (i = 0; i < 6; ++i)
		axis_pts[i] = temp_axis_pts[i];
        }
    }

    /* Ticks on y axis. */
    if (num_ticks[1]) {
	increment = get_increment(num_ticks[1],axis_pts[2],axis_pts[3]);
	if (avs2) {
	if (data_type == UNIFORM)
	    normalize(&increment,axis_pts,axis_pts[2],axis_pts[3],zfact);
	}
	set_tick_color(tick_color, 0.0, 1.0, 0.0);
	loc = axis_pts[2] + increment;
	for (i=0;i<=num_ticks[1];i++)
	{
	    axis_tick[0][0] = axis_pts[0];
	    axis_tick[0][1] = loc;
	    axis_tick[0][2] = axis_pts[4] + t_size;

	    axis_tick[1][0] = axis_pts[0];
	    axis_tick[1][1] = loc;
	    axis_tick[1][2] = axis_pts[4];

	    axis_tick[2][0] = axis_pts[0] + t_size;
	    axis_tick[2][1] = loc;
	    axis_tick[2][2] = axis_pts[4];

	    axis_tick[3][0] = axis_pts[0];
	    axis_tick[3][1] = loc;
	    axis_tick[3][2] = axis_pts[4];

	    GEOMadd_disjoint_line(axis_obj,axis_tick,tick_color,4,GEOM_COPY_DATA);
	    loc += increment;
	}
	if (avs2) {
	if (data_type == UNIFORM)
	    for (i = 0; i < 6; ++i)
		axis_pts[i] = temp_axis_pts[i];
	}
    }

    /* Ticks on z axis. */
    if (num_ticks[2]) {
	increment = get_increment(num_ticks[2],axis_pts[4],axis_pts[5]);
	if (avs2) {
	if (data_type == UNIFORM)
	    {
	    normalize(&increment,axis_pts,axis_pts[4],axis_pts[5],zfact);
            increment = get_increment(num_ticks[2],axis_pts[4],axis_pts[5]);
	    }
	}
	set_tick_color(tick_color, 0.0, 1.0, 1.0);
	loc = axis_pts[4] + increment;
	for(i=0;i<=num_ticks[2];i++)
	{
	    axis_tick[0][0] = axis_pts[0] + t_size;
	    axis_tick[0][1] = axis_pts[2];
	    axis_tick[0][2] = loc;

	    axis_tick[1][0] = axis_pts[0];
	    axis_tick[1][1] = axis_pts[2];
	    axis_tick[1][2] = loc;

	    axis_tick[2][0] = axis_pts[0];
	    axis_tick[2][1] = axis_pts[2] + t_size;
	    axis_tick[2][2] = loc;

	    axis_tick[3][0] = axis_pts[0];
	    axis_tick[3][1] = axis_pts[2];
	    axis_tick[3][2] = loc;

	    GEOMadd_disjoint_line(axis_obj,axis_tick,tick_color,4,GEOM_COPY_DATA);
	    loc += increment;
	}
	if (avs2) {
	if (data_type == UNIFORM)
	    for (i = 0; i < 6; ++i)
		axis_pts[i] = temp_axis_pts[i];
	}
    }
}

make_planes(obj,axis_pts,mesh_size,which_planes,where,data_type,zfact)
GEOMobj *obj;
float axis_pts[6];
int mesh_size[3];       /* this is the number of ticks */
int which_planes[3];
float where[3];
int data_type;
float zfact;
/*
 *  This routine creates mesh planes by extending perpendiculars to the
 *  appropriate axis at tick mark locations.  This is all very similar
 *  to the make_ticks() function.
 */
{
    float get_increment();
    float increment;
    float loc;
    float mesh_line[2][3];
    float plane_color[2][3];
    int i;
    int ct;
    float temp_axis_pts[6];

    if (avs2) {
    if (data_type == UNIFORM)
	for (i = 0; i < 6; ++i)
	    temp_axis_pts[i] = axis_pts[i];
    }

    /* xz plane */
    if (which_planes[0]) {
	increment = get_increment(mesh_size[0],axis_pts[0],axis_pts[1]);
    if (avs2) {
	if (data_type == UNIFORM)
	    normalize(&increment,axis_pts,axis_pts[0],axis_pts[1],zfact);
    }
	set_plane_color(plane_color, 0.0, 1.0, 0.0);
	loc = axis_pts[0];
	for(ct=0;ct<=mesh_size[0]+1;ct++)
	{
	    mesh_line[0][0] = loc;
	    mesh_line[0][1] = axis_pts[2]+where[0]*(axis_pts[3]-axis_pts[2]);
	    mesh_line[0][2] = axis_pts[4];
	    mesh_line[1][0] = loc;
	    mesh_line[1][1] = axis_pts[2]+where[0]*(axis_pts[3]-axis_pts[2]);
	    mesh_line[1][2] = axis_pts[5];
	    GEOMadd_disjoint_line(obj,mesh_line,plane_color,2,GEOM_COPY_DATA);
	    loc += increment;
	}
	if (avs2) {
	if (data_type == UNIFORM)
	    for (i = 0; i < 6; ++i)
		axis_pts[i] = temp_axis_pts[i];
        }
	increment = get_increment(mesh_size[2],axis_pts[4],axis_pts[5]);
	if (avs2) {
	if (data_type == UNIFORM)
	    {
	    normalize(&increment,axis_pts,axis_pts[4],axis_pts[5],zfact);
            increment = get_increment(mesh_size[2],axis_pts[4],axis_pts[5]);
	    }
	}
	set_plane_color(plane_color, 0.0, 1.0, 0.0);
	loc = axis_pts[4];
	for (ct=0;ct<=mesh_size[2]+1;ct++)
	{
	    mesh_line[0][0] = axis_pts[0];
	    mesh_line[0][1] = axis_pts[2]+where[0]*(axis_pts[3]-axis_pts[2]);
	    mesh_line[0][2] = loc;
	    mesh_line[1][0] = axis_pts[1];
	    mesh_line[1][1] = axis_pts[2]+where[0]*(axis_pts[3]-axis_pts[2]);
	    mesh_line[1][2] = loc;
	    GEOMadd_disjoint_line(obj,mesh_line,plane_color,2,GEOM_COPY_DATA);
	    loc += increment;
	}
	if (avs2) {
	if (data_type == UNIFORM)
	    for (i = 0; i < 6; ++i)
		axis_pts[i] = temp_axis_pts[i];
	}
    }

    /* xy plane */
    if (which_planes[1]) {
	increment = get_increment(mesh_size[0],axis_pts[0],axis_pts[1]);
	if (avs2) {
	if (data_type == UNIFORM)
	    normalize(&increment,axis_pts,axis_pts[0],axis_pts[1],zfact);
	}
	set_plane_color(plane_color, 0.0, 1.0, 1.0);
	loc = axis_pts[0];
	for(ct=0;ct<=mesh_size[0]+1;ct++)
	{
	    mesh_line[0][0] = loc;
	    mesh_line[0][1] = axis_pts[2];
	    mesh_line[0][2] = axis_pts[4]+where[1]*(axis_pts[5]-axis_pts[4]);
	    mesh_line[1][0] = loc;
	    mesh_line[1][1] = axis_pts[3];
	    mesh_line[1][2] = axis_pts[4]+where[1]*(axis_pts[5]-axis_pts[4]);
	    GEOMadd_disjoint_line(obj,mesh_line,plane_color,2,GEOM_COPY_DATA);
	    loc += increment;
	}
	if (avs2) {
	if (data_type == UNIFORM)
	    for (i = 0; i < 6; ++i)
		axis_pts[i] = temp_axis_pts[i];
        }

	increment = get_increment(mesh_size[1],axis_pts[2],axis_pts[3]);
	if (avs2) {
	if (data_type == UNIFORM)
	    normalize(&increment,axis_pts,axis_pts[2],axis_pts[3],zfact);
	}
	set_plane_color(plane_color, 0.0, 1.0, 1.0);
	loc = axis_pts[2];
	for (ct=0;ct<=mesh_size[1]+1;ct++)
	{
	    mesh_line[0][0] = axis_pts[0];
	    mesh_line[0][1] = loc;
	    mesh_line[0][2] = axis_pts[4]+where[1]*(axis_pts[5]-axis_pts[4]);
	    mesh_line[1][0] = axis_pts[1];
	    mesh_line[1][1] = loc;
	    mesh_line[1][2] = axis_pts[4]+where[1]*(axis_pts[5]-axis_pts[4]);
	    GEOMadd_disjoint_line(obj,mesh_line,plane_color,2,GEOM_COPY_DATA);
	    loc += increment;
	}
	if (avs2) {
	if (data_type == UNIFORM)
	    for (i = 0; i < 6; ++i)
		axis_pts[i] = temp_axis_pts[i];
	}
    }

    /* yz plane */
    if (which_planes[2]) {
	increment = get_increment(mesh_size[1],axis_pts[2],axis_pts[3]);
	if (avs2) {
	if (data_type == UNIFORM)
	    normalize(&increment,axis_pts,axis_pts[2],axis_pts[3],zfact);
	}
	set_plane_color(plane_color, 1.0, 0.0, 0.0);
	loc = axis_pts[2];
	for (ct=0;ct<=mesh_size[1]+1;ct++)
	{
	    mesh_line[0][0] = axis_pts[0]+where[2]*(axis_pts[1]-axis_pts[0]);
	    mesh_line[0][1] = loc;
	    mesh_line[0][2] = axis_pts[4];
	    mesh_line[1][0] = axis_pts[0]+where[2]*(axis_pts[1]-axis_pts[0]);
	    mesh_line[1][1] = loc;
	    mesh_line[1][2] = axis_pts[5];
	    GEOMadd_disjoint_line(obj,mesh_line,plane_color,2,GEOM_COPY_DATA);
	    loc += increment;
	}
	if (avs2) {
	if (data_type == UNIFORM)
	    for (i = 0; i < 6; ++i)
		axis_pts[i] = temp_axis_pts[i];
        }

	increment = get_increment(mesh_size[2],axis_pts[4],axis_pts[5]);
	if (avs2) {
	if (data_type == UNIFORM)
	    {
	    normalize(&increment,axis_pts,axis_pts[4],axis_pts[5],zfact);
            increment = get_increment(mesh_size[2],axis_pts[4],axis_pts[5]);
	    }
	}
	set_plane_color(plane_color, 1.0, 0.0, 0.0);
	loc = axis_pts[4];
	for (ct=0;ct<=mesh_size[2]+1;ct++)
	{
	    mesh_line[1][0] = axis_pts[0]+where[2]*(axis_pts[1]-axis_pts[0]);
	    mesh_line[0][1] = axis_pts[2];
	    mesh_line[0][2] = loc;
	    mesh_line[1][0] = axis_pts[0]+where[2]*(axis_pts[1]-axis_pts[0]);
	    mesh_line[1][1] = axis_pts[3];
	    mesh_line[1][2] = loc;
	    GEOMadd_disjoint_line(obj,mesh_line,plane_color,2,GEOM_COPY_DATA);
	    loc += increment;
	}
	if (avs2) {
	if (data_type == UNIFORM)
	    for (i = 0; i < 6; ++i)
		axis_pts[i] = temp_axis_pts[i];
	}
    }
}

make_labels(axis_pts,num_ticks,data_type,label_x_obj,label_y_obj,
	    label_z_obj,show_labels,precision,label_density,
	    font_size,origin_labels,zfact)
float axis_pts[6];
int num_ticks[3];
int data_type;
GEOMobj *label_x_obj,*label_y_obj,*label_z_obj;
int precision;
int label_density[3];  /* Number of ticks per label. */
float *font_size;
int show_labels;       /* Turn labels on and off. */
char *origin_labels;   /* Axis whose label is at the origin. */
float zfact;           /* Blow-up factor */
/*
 *  This routine puts labels on the various axis, at tick points,
 *  if requested.  Labels may not occur at every tick, as determined
 *  by the label_density parameter.  This is all very similar to the
 *  make_ticks() function.
 *
 *  Note that for a UNIFORM data set we label the axis according to
 *  the dimensions of the data set, or the user supplied dimensions,
 *  as opposed to the (normalized) screen size of the data.  This
 *  amounts to using the local variables u_inc and u_loc to track the
 *  the values we need for the labels.
 *
 *  origin_labels is a string which we scan for the characters x,y,
 *  and z.  For each one found we will label the origin according
 *  to the min value of that axis.  So the origin can get pretty
 *  busy.  If a particular label is called for we start labeling
 *  at axis_pts[min], otherwise we start at axis_points[min] +
 *  increment.
 */
{
    float get_increment();
    float increment,u_inc;
    float loc,u_loc;
    float temp_axis_pts[6];
    float ref_pt[3];
    float offset[3];
    float color[3];
    char num[30];
    char arg[5];
    int i;
    int x_label,y_label,z_label;
    int ct;

    /* Find out which axis labels the origin. */
    x_label = y_label = z_label = FALSE;
    for (i = 0; i < strlen(origin_labels); ++i) {
	if (origin_labels[i] == 'x')
	    x_label = TRUE;
	if (origin_labels[i] == 'y')
	    y_label = TRUE;
	if (origin_labels[i] == 'z')
	    z_label = TRUE;
    }

    /* But show the x origin if everyone is the same. */
    if (axis_pts[0] == axis_pts[2] && axis_pts[0] == axis_pts[4]
				   && y_label == FALSE
				   && z_label == FALSE)
	x_label = TRUE;

    /* Set the number of digits to the right of the decimal via precision. */
    arg[0] = '%';
    arg[1] = '.';
    arg[2] = precision + 0x30;
    arg[3] = 'f';
    arg[4] = '\0';

    color[0] = color[1] = color[2] = 1.0;
    if (avs2) {
    if (data_type == UNIFORM)
	for (i = 0; i < 6; ++i)
	    temp_axis_pts[i] = axis_pts[i];
    }

    /* Labels on x axis. */
    if (num_ticks[0] && show_labels) {
	increment = u_inc = get_increment(num_ticks[0],axis_pts[0],axis_pts[1]);
    if (avs2) {
	if (data_type == UNIFORM)
	    normalize(&increment,axis_pts,axis_pts[0],axis_pts[1],zfact);
    }
	loc = axis_pts[0] + ((x_label) ? 0.0 : increment);
        if (avs2) {
	if (data_type == UNIFORM) /* Get labels from data dimensions or user. */
	    u_loc = temp_axis_pts[0] + ((x_label) ? 0.0 : u_inc);
	}
	for (ct=0;ct<=(num_ticks[0]/label_density[0]+((x_label)?1:0));ct++)
	{
	    ref_pt[0] = loc;
	    ref_pt[1] = axis_pts[2];
	    ref_pt[2] = axis_pts[4];
	    offset[0] = 0.0;
	    offset[1] = -0.06;
	    offset[2] = 0.0;
	    if (!(avs2)) 
	       sprintf(num,arg,loc);
	    else 
	       sprintf(num,arg,(data_type == UNIFORM) ? u_loc : loc);
	    GEOMadd_label(label_x_obj,num,ref_pt,offset,*font_size,color,-1);
	    loc += increment * label_density[0];
	    if (avs2) {
	    if (data_type == UNIFORM)
		u_loc += u_inc * label_density[0];
	    }
	}
/* do the last label NOT! TOR 2/28/92 added garbage to displayed labels */
/*	if (x_label)
	{
#ifdef AVS3
	    loc = axis_pts[1];
	    sprintf(num,arg,loc);
#else
	    u_loc = axis_pts[1]; 
	    sprintf(num,arg,loc);
#endif
	    GEOMadd_label(label_x_obj,num,ref_pt,offset,*font_size,color,-1);
	}
*/		
	if (avs2) {
	if (data_type == UNIFORM)
	    for (i = 0; i < 6; ++i)
		axis_pts[i] = temp_axis_pts[i];
	}
    }

    /* Labels on y axis. */
    if (num_ticks[1] && show_labels) {
	increment = u_inc = get_increment(num_ticks[1],axis_pts[2],axis_pts[3]);
	if (avs2) {
	if (data_type == UNIFORM)
	    normalize(&increment,axis_pts,axis_pts[2],axis_pts[3],zfact);
	}
	loc = axis_pts[2] + ((y_label) ? 0.0 : increment);
	if (avs2) {
	if (data_type == UNIFORM)
	    u_loc = temp_axis_pts[2] + ((y_label) ? 0.0 : u_inc);
	}
	for (ct=0;ct<=(num_ticks[1]/label_density[1] + (y_label?1:0));ct++)
	{
	    ref_pt[0] = axis_pts[0];
	    ref_pt[1] = loc;
	    ref_pt[2] = axis_pts[4];
	    offset[0] = -0.02;
	    offset[1] = -0.02;
	    offset[2] = 0.0;
	    if (!(avs2))
	       sprintf(num,arg,loc);
	    else
	       sprintf(num,arg,(data_type == UNIFORM) ? u_loc : loc);
	    GEOMadd_label(label_y_obj,num,ref_pt,offset,*font_size,color,-1);
	    loc += increment * label_density[1];
	    if (avs2) {
	    if (data_type == UNIFORM)
		u_loc += u_inc * label_density[1];
	    }
	}
/* do the last label NOT! TOR 2/28/92 */
/*	if (y_label)
	{
#ifdef AVS3
	    loc = axis_pts[3];
	    sprintf(num,arg,loc);
#else
	    u_loc = axis_pts[3];  
	    sprintf(num,arg,loc);
#endif
	    GEOMadd_label(label_x_obj,num,ref_pt,offset,*font_size,color,-1);
	}
*/
	if (avs2) {
	if (data_type == UNIFORM)
	    for (i = 0; i < 6; ++i)
		axis_pts[i] = temp_axis_pts[i];
	}
    }

    /* Labels on z axis. */
    if (num_ticks[2] && show_labels) {
	increment = u_inc = get_increment(num_ticks[2],axis_pts[4],axis_pts[5]);
	if (avs2) {
	if (data_type == UNIFORM)
	    {
	    normalize(&increment,axis_pts,axis_pts[4],axis_pts[5],zfact);
            increment = get_increment(num_ticks[2],axis_pts[4],axis_pts[5]);
	    }
	}
	loc = axis_pts[4] + ((z_label) ? 0.0 : increment);
	if (avs2) {
	if (data_type == UNIFORM)
	    u_loc = temp_axis_pts[4] + ((z_label) ? 0.0 : u_inc);
	}
	for (ct=0;ct<=(num_ticks[2]/label_density[2]+(z_label?1:0));ct++)
	{
	    ref_pt[0] = axis_pts[0];
	    ref_pt[1] = axis_pts[2];
	    ref_pt[2] = loc;
	    offset[0] = 0.0;
	    offset[1] = -0.06;
	    offset[2] = 0.0;
	    if (!(avs2))
	       sprintf(num,arg,loc/zfact);
	    else
	       sprintf(num,arg,((data_type == UNIFORM) ? u_loc : loc) / zfact);
	    GEOMadd_label(label_z_obj,num,ref_pt,offset,*font_size,color,-1);
	    loc += increment * label_density[2];
	    if (avs2) {
	    if (data_type == UNIFORM)
		u_loc += u_inc * label_density[2];
	    }
	}
/* do the last label NOT! TOR 2/28/92 */
/*	if (z_label)
	{
#ifdef AVS3
	    loc = axis_pts[5];
	    sprintf(num,arg,loc/zfact);
#else
	    u_loc = axis_pts[5]; 
	    sprintf(num,arg,loc/zfact);
#endif
	    GEOMadd_label(label_x_obj,num,ref_pt,offset,*font_size,color,-1);
	}
*/
#ifndef AVS3
	if (data_type == UNIFORM)
	    for (i = 0; i < 6; ++i)
		axis_pts[i] = temp_axis_pts[i];
#endif
    }
}


/************** Support stuff. **********************************************/

set_tick_color(tc,r,g,b)
float tc[4][3];
float r,g,b;
{
    tc[0][0] = tc [1][0] = tc[2][0] = tc[3][0] = r;
    tc[0][1] = tc [1][1] = tc[2][1] = tc[3][1] = g;
    tc[0][2] = tc [1][2] = tc[2][2] = tc[3][2] = b;
}

set_plane_color(pc,r,g,b)
float pc[2][3];
float r,g,b;
{
    pc[0][0] = pc [1][0] = r;
    pc[0][1] = pc [1][1] = g;
    pc[0][2] = pc [1][2] = b;
}

normalize(increment,axis_pts,min_pt,max_pt,zfact)
float *increment;
float axis_pts[6];
float min_pt,max_pt;
float zfact;
/*
 *  This is a uniform data set so we want to scale increment
 *  and set the axis dimensions back to -1,1.
 **** undone 22 may 1991
 */
{
    *increment = 2 * *increment / (max_pt - min_pt);
    axis_pts[0] = axis_pts[2] =  -1.0;
    axis_pts[1] = axis_pts[3] =   1.0;
    axis_pts[4] = zref[0] * zfact;
    axis_pts[5] = zref[1] * zfact;
}


/************** Increment calculation. **************************************/

double mypow(),nicenum();

float get_increment(ticks,axis_min,axis_max)
int ticks;
float axis_min,axis_max;
{
    double range,d;
#if 0
    if (ticks <= (axis_max - axis_min)) {
	d = (double)((int)((axis_max - axis_min)/ticks + 0.5));
    }
    else {
	range = nicenum(axis_max - axis_min, 0);
	d = nicenum(range / (ticks - 1), 1);
    }
#endif
    d = (axis_max - axis_min)/(ticks+1);
    return (float)d;
}

double nicenum(x, round)
double x;
int round;
/*
 * nicenum: find a "nice" number approximately equal to x
 * round if round=1, ceil if round=0
 */
{
    long exp;
    double f, y;

    exp = floor(log10(x));
    f = x/mypow(10., exp);   /* fraction between 1 and 10 */
    if (round)
	if (f<1.5) y = 1.;
	else if (f<3.) y = 2.;
	else if (f<7.) y = 5.;
	else y = 10.;
    else
	if (f<=1.) y = 1.;
	else if (f<=2.) y = 2.;
	else if (f<=5.) y = 5.;
	else y = 10.;
    return y*mypow(10., exp);
}


double mypow(a, n)
double a;
register int n;
/*
 * mypow(a,n)=a^n for integer n
 * roundoff errors in pow were causing problems, so I wrote my own
 */
{
    double x;

    x = 1.;
    if (n>0)
	for (; n>0; n--)
	    x *= a;
    else
	for (; n<0; n++)
	    x /= a;
    return x;
}


