/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/

/* NOTE:  THIS MODULE AND SOURCE CODE IS FOR USE 
   WITH THE AVS SOFTWARE ENVIRONMENT ONLY */
/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/* 
 * mesh_to_ucd - Create a mesh_to_ucd plot from 2D scalar data
 *
 * Derived from the concept of field to mesh,
 * with more specific assumptions about the input field,
 * and alternative to the colour scheme.
 *
 * Author: Ian Curington, AVS Inc.
 *
 * Revision History:
 * 10 Jan 92  ianc - Original (Carpet)
 * 11 Jan 92  ianc - tidy up, sent to G. Walker, BT Labs
 *  7 Sep 93  ianc - upgrade to irregular 3space fields for 
 *                   better hole algorithm
 * 15 Sep 93  ianc - conversion to ucd out rather than geom
 *                   added edit input port for use with
 *                   ucd_cell_geom mesh editing
 * 21 Sep 93  ianc - added flip normal button
 *
 * ----------------- man page -------------------
 *
 * Module: "mesh_to_ucd"
 * 
 * Type:   mapper
 * 
 * Author: Ian Curington, Advanced Visual Systems, Inc.
 * 
 * Source: mesh_to_ucd.c
 * 
 * Description:
 * ------------
 *   A mesh_to_ucd makes a 2D UCD surface showing surface height variation
 *   from one field.
 *   Each cell is drawn as per field to mesh, (linear interpolation),
 *   or in block mode with descrete non-interpolated colors.
 *   It will optionally leave holes in the UCD output
 *   where the data goes out of range for the float dials, allowing
 *   mask values to control cell visibility, or region boundaries.
 *   It is basically a variation on field_to_mesh, allowing more
 *   advanced UCD type visualizations and data processing.
 * 
 * Inputs:
 *   Field:
 *     "field 2D uniform/irregular scalar float"
 *     This controls the height map of the surface.
 *     It also controls the scalar values, on the UCD nodes
 *     Can be 2-space or 3-space.
 *
 * Output:
 *  UCD, with coords of mesh, and scalar value as node data.
 * 
 * Parameters:
 * -----------
 * 
 * Z-Scale
 *   This controls the relative offset from the base plane of each
 *   cell, creating the wavy mesh_to_ucd effect.
 * 
 * Leave Holes
 *   This will allow drop outs in the field where the field
 *   values go out of range for the Lower-Upper dials.
 *   This means UCD cells will be missing from the structure
 *   if any corner touches an invalid node.
 *
 * Lower
 *   A float value to determin the range of valid data at nodes
 *   if "leave holes" mode is on.
 * 
 * Upper 
 *   A float value to determin the range of valid data at nodes
 *   if "leave holes" mode is on.
 * 
 * Example Network:
 * ----------------
 * 
 *                   Read Image (=avs.x)
 *                      |
 *                   Downsize    = 12
 *                      |
 *                   field to float
 *                      |
 *                   extract scalar
 *                      |
 *                   mesh_to_ucd
 *                      |
 * gen colormap         | 
 *       |              |
 *       |    |-------- |
 *    ucd_contour       |
 *        |----------|  |
 *                ucd to geom
 *                   |
 *                geometry viewer
 *  
 * 
 * 
 */

/**********************************************************************/

#include <stdio.h>

#include <avs/avs.h>
#include <avs/field.h>
#include <avs/ucd_defs.h>

/**
#define INVERT_Z
 **/

#define CHECK_RANGE(a,b,c) \
  ( (a) >= (b) && (a) <= (c) )

/**********************************************************************/
/* Module Description Section */

mesh_to_ucd_desc()
{
      int mesh_to_ucd_compute();

      AVSset_module_name("mesh_to_ucd", MODULE_MAPPER);
      AVSset_module_flags( COOPERATIVE | REENTRANT);
      AVScreate_input_port("Field Input",
             "field 2D scalar float", REQUIRED);
      AVScreate_input_port("Edit",
             "field 1D 2-vector integer 3-space irregular", OPTIONAL);

      AVScreate_output_port("ucd_mesh", "ucd");
      AVSadd_float_parameter("Z scale", 0.0, FLOAT_UNBOUND, FLOAT_UNBOUND);
      AVSadd_parameter("leave holes", "boolean", 0, 0, 1 );
      AVSadd_float_parameter("lower", 0.0, FLOAT_UNBOUND, FLOAT_UNBOUND);
      AVSadd_float_parameter("upper", 1.0, FLOAT_UNBOUND, FLOAT_UNBOUND);
      AVSadd_parameter("flip normals", "boolean", 0, 0, 1 );

      AVSset_compute_proc(mesh_to_ucd_compute);
}

/**********************************************************************/
/* Module Compute Function */

mesh_to_ucd_compute(input, Edit, ucd_output,
		    zscale, holes, p_lower, p_upper, flip  )

    AVSfield *input;
    AVSfield_int *Edit;
    UCD_structure **ucd_output;
    float *zscale;
    int   holes;
    float *p_lower, *p_upper;
    int   flip;
{
   float        *fmesh;
   float        min, max, min2, max2;
   int          size, i, j, k, indexer;
   int          axis;
   int          ii;
   float        extent[6], min_extent[3], max_extent[3];
   float        tx, ty, tz;
   int          method;
   int          i1, i2, i3, i4;
   int          active;
   int          *active_list;
   float        *values;
   float        lower, upper;
   int          stat;
   float        *xc, *yc, *zc;
   float        *node_data;
   float        val;

/*
 * B O D Y
 */

   size = MAXX(input) * MAXY(input);

   lower = *p_lower;
   upper = *p_upper;


/* Edit conditions */
    if (  Edit  != NULL &&
         *ucd_output != NULL &&
         AVSinput_changed("Edit",0) &&
         Edit->data[0] >= 0 )
    {
        UCDstructure_get_node_positions (*ucd_output, &xc, &yc, &zc);
        UCDstructure_get_node_data ( *ucd_output, &node_data);

        val = Edit->points[2];
        ii  = Edit->data[0];
        zc[ ii ] = val;
#ifdef INVERT_Z
        node_data[ ii ] = -1.0 * val;
#else
        node_data[ ii ] = val;
#endif
        return(1);
    }



   /*
    * Build the Mesh from the input field data
    */
   axis = 2;   /* default axes */

   fmesh  = (float *) ALLOC_LOCAL (3 * size * sizeof(float));
   values = (float *) ALLOC_LOCAL (    size * sizeof(float));
   active_list = (int *) ALLOC_LOCAL (    size * sizeof(int));

   if ( input->uniform == UNIFORM )
   {
     if (input->points[0] == input->points[1])
        axis = 0;
     else if (input->points[2] == input->points[3])
        axis = 1;
     else if (input->points[4] == input->points[5])
        axis = 2;
     for (i=0; i<MAXY(input); i++) {
        for (j=0; j<MAXX(input); j++) {
         if (input->nspace == 3) {       /* slices comes from a volume */
             if (axis == 0) {        /* X Axis */
               UTILget_world_coords (&tx, &ty, &tz, (int)(input->points[0]),
                               j, i, input);
               fmesh[(i*MAXX(input)+j)*3+0] = tx;               
               fmesh[(i*MAXX(input)+j)*3+1] = ty;        
               fmesh[(i*MAXX(input)+j)*3+2] = tz;
             }  /* end if */ 
             else if (axis == 1) {  /* Y Axis */
               UTILget_world_coords (&tx, &ty, &tz, j, 
                               (int) (input->points[2]), i, input);
               fmesh[(i*MAXX(input)+j)*3+0] = tx; 
               fmesh[(i*MAXX(input)+j)*3+1] = ty;
               fmesh[(i*MAXX(input)+j)*3+2] = tz;
             }  /* end else if */ 
             else  {                /* Z Axis */
               UTILget_world_coords (&tx, &ty, &tz, j, i, 
                               (int) (input->points[4]), input);
               fmesh[(i*MAXX(input)+j)*3+0] = tx;
               fmesh[(i*MAXX(input)+j)*3+1] = ty;
               fmesh[(i*MAXX(input)+j)*3+2] = tz;
             }  /* end else */ 
         } /* end if for 3-space */
         else {                              /* 2D images */
              fmesh[(i*MAXX(input)+j)*3+0] = (float) j;
              fmesh[(i*MAXX(input)+j)*3+1] = (float)(MAXY(input)-i-1.0);
            fmesh[(i*MAXX(input)+j)*3+2] = 0.0;
            axis = 2;
         }  /* end else for images */
        }  /* end for j */
     }  /* end for i */

   }
   else if ( input->uniform == IRREGULAR && input->nspace == 3 )
   {
     k = 0;
     for (i=0; i<MAXY(input); i++) {
        for (j=0; j<MAXX(input); j++) {
               fmesh[(i*MAXX(input)+j)*3+0] = input->points[0*size+k];
               fmesh[(i*MAXX(input)+j)*3+1] = input->points[1*size+k];        
               fmesh[(i*MAXX(input)+j)*3+2] = input->points[2*size+k];
               k++;
        }  /* end for j */
     }  /* end for i */
   }   /* end of uniform-irregular test */
   else
   {
      AVSerror("mesh_to_ucd: field type not compatable, type=%d, nspace=%d",
                input->uniform, input->nspace );
      return(0);
   }

   /*
    * scan for min-max data,
    */
   min = max = input->field_data_float[0];
   for (i=0; i<size; i++)
   {
         min = (input->field_data_float[i] < min ) ?
                  input->field_data_float[i] : min;
         max = (input->field_data_float[i] > max ) ?
                  input->field_data_float[i] : max;
   }

   /*
    * offset the mesh geometry with scalar warp for first input port
    */
   if (max == min)
   {
         for (i=0; i<MAXY(input); i++) 
           for (j=0; j<MAXX(input); j++) 
             fmesh[(i*MAXX(input)+j)*3+axis] += *zscale;
   } else {
         for (i=0; i<MAXY(input); i++) 
           for (j=0; j<MAXX(input); j++) 
             fmesh[(i*MAXX(input)+j)*3+axis] += *zscale *
             ((float)(input->field_data_float[i*MAXX(input)+j]) - min) /
                 (max-min);
   }

   /*
    * Get Values
    */
    if (max == min)
    {
         for (i=0; i<MAXY(input); i++) 
         for (j=0; j<MAXX(input); j++) 
               values[i*MAXX(input)+j] = min;
    } else {
         for (i=0; i<MAXY(input); i++) 
         for (j=0; j<MAXX(input); j++) 
               values[i*MAXX(input)+j] =
                     input->field_data_float[i*MAXX(input)+j];
    }

    /*
     * Check Range
     */
    for (i=0; i<MAXY(input)-1; i++)
    {
       for (j=0; j<MAXX(input)-1; j++)
       {
               i1 = (i*MAXX(input)+j);
               active_list[i1] = CHECK_RANGE (input->field_data_float[i1],
                                  lower, upper );
       }
    }

    if ( holes )
    {
          min = lower;
          max = upper;
    }

    /*
     * Create the UCD data structure
     */
    stat = populate_ucd ( ucd_output, fmesh, values, active_list,
                   MAXX(input), MAXY(input), holes, min, max, flip );


    FREE_LOCAL(fmesh);
    FREE_LOCAL(values);
    FREE_LOCAL(active_list);

    return( stat );
}


/**********************************************************************/
/*  Populate UCD Structure                                            */
/**********************************************************************/
int
populate_ucd ( Output, fmesh, values, active_list,
               nx, ny, holes, min, max, flip )

        UCD_structure **Output;
        float  *fmesh;
        float  *values;
        int    *active_list;
        int    nx, ny, holes;
        float  min, max;
	int    flip;
{
    char model_name[80];
    char data_labels[200];
    char delim[2];
    int node, element, solid;
    int i,j,k;
    int ucd_flags, cell_tsize, node_csize;
    int num_model_data, num_cell_data, num_node_data;
    int util_flag;
    float min_extent[3], max_extent[3];
    float min_node_data[10], max_node_data[10];
    float *xc, *yc, *zc, *node_data;
    int node_comp_list[10], num_node_comp;
    int conn_index;
    int num_cells, num_nodes;
    int cell, cell_id;
    int node_list[20];
    int *node_list_2d;
    float scalar_value, tx, ty;
    int size, dz;
    int slice;
    int mat_id;
    int v, index;
    int old_cell;
    float xmin, xmax;
    float ymin, ymax;
    float zmin, zmax;
    int   num_output_layers;
    int   active_count;
    int   i1, i2, i3, i4;
    int   active;

        /**************
         ***  body  ***
         **************/

    /*********************************/
    /* Count Active Cells            */
    /*********************************/
    if ( !holes )
        num_cells = (nx-1) * (ny-1);
    else
    {
         num_cells = 0;
         for (i=0; i<ny-1; i++)
         for (j=0; j<nx-1; j++)
         {
               i1 = (i*nx+j);
               i2 = ((i+1)*nx+j);
               i3 = ((i+1)*nx+j+1);
               i4 = ((i)*nx+j+1);

               num_cells += active_list[i1] &&
                            active_list[i2] &&
                            active_list[i3] &&
                            active_list[i4];
         }
    }
    if ( num_cells <= 0 ) return(0);



    /*********************************/
    /* establish the UCD structure   */
    /*********************************/
    ucd_flags = UCD_INT          |
                UCD_MATERIAL_IDS |
                UCD_NODE_NAMES   |
                UCD_CELL_NAMES   |
                UCD_CELL_TYPES;

    num_nodes = nx * ny;
    if ( num_nodes <= 0 ) return(0);

    /* no node connectivity, and only scalar nodal data */
    node_csize = num_model_data = num_cell_data = util_flag = 0;
    cell_tsize = num_cells * 4;  /* assume quad cell topo at worst */

    num_node_data = 1;  /* assume scalar */
    num_node_comp = 1;  /* assume scalar */
    for ( i=0; i< 1; i++)
        node_comp_list[i] = 1;


    strcpy (model_name, "Mesh to UCD");

    /*
     * allocate some shared memory structure to hold the ucd info.
     * This example just throws away the old one each time, in
     * case the size has changed, etc. If it is really not changing,
     * then keep the old structure and modify in-place.
     */

    if (*Output)
        UCDstructure_free (*Output);

    *Output = (UCD_structure *)UCDstructure_alloc (model_name,
              num_model_data, ucd_flags, num_cells, cell_tsize,
              num_cell_data, num_nodes, node_csize, num_node_data, util_flag);


    /*************************************************/
    /* process Geometry - Node Positions and Data    */
    /*************************************************/
    UCDstructure_get_node_positions (*Output, &xc, &yc, &zc);
    UCDstructure_get_node_data ( *Output, &node_data);

    /* fill the coordinate information with a 3D mesh */
    for (i=0; i<ny; i++)
    for (j=0; j<nx; j++)
    {
        index = i*nx+j;
        xc[ index ] = fmesh[ index * 3 + 0 ]; /* X */
        yc[ index ] = fmesh[ index * 3 + 1 ]; /* Y */
        zc[ index ] = fmesh[ index * 3 + 2 ]; /* Z */
        node_data[ index ] = values[ index ];
    }

    /****************************************/
    /* process each cell for Topology       */
    /****************************************/
    /* fill the connectivity information with a 2D mesh */

    mat_id = 1;    /* static per cell */
    cell_id = 1;   /* cell identifier tag */
    cell = 0;      /* cell list indexer */

    for (i=0; i<ny-1; i++)
    for (j=0; j<nx-1; j++)
    {
        i1 = (i*nx+j);
        i2 = ((i+1)*nx+j);
        i3 = ((i+1)*nx+j+1);
        i4 = ((i)*nx+j+1);

        active = active_list[i1] && active_list[i2] &&
                 active_list[i3] && active_list[i4];

        if ( holes == 0 || holes == 1 && active )
        {

             /* here is the logical quad topology */
	     if ( !flip )
	     {
                 node_list[0] = i1;
                 node_list[1] = i2;
                 node_list[2] = i3;
                 node_list[3] = i4;
	     }
	     else
	     {
                 node_list[0] = i1;
                 node_list[1] = i4;
                 node_list[2] = i3;
                 node_list[3] = i2;
	     }

            /* set cell information - UCD struct
             *                   cell to set
             *                   cell name
             *                   element type
             *                   material id (cell grouping)
             *                   cell type
             *                   mid edge data ? (0-no,1-yes)
             *                   list of nodes
             */
            if ( UCDcell_set_information ( *Output,
                                      cell,
                                      cell_id,
                                      "quad",
                                      mat_id,
                                      UCD_QUADRILATERAL,
                                      0,
                                      node_list ) == 0)
            {
             AVSerror (" Error: cannot set cell information\n");
             return(0);
            }

            /* bump id for next cell */
            cell++;
            cell_id++;

        } /* end of active test */
    }  /* end of cell loop */

    if ( cell <= 0 ) return(0);

    /****************************************/
    /* process Node Data                    */
    /****************************************/
    /* add some additional structure info, such as extents and labels */

    UCDstructure_set_node_components ( *Output, node_comp_list, num_node_comp);

    UCDstructure_set_node_labels ( *Output, "mesh_scalar", delim);

    min_node_data[0] = min;
    max_node_data[0] = max;
    UCDstructure_set_node_minmax ( *Output, min_node_data , max_node_data );

    /* update the extents with the new values */
    xmin = ymin = zmin = 1.0e+15;
    xmax = ymax = zmax = -1.0e+15;

    for ( node  = 0; node  < num_nodes; node++ )
    {
      if ( holes == 0 || holes == 1 && active_list[node] )
      {
        if ( xc[ node ] < xmin) xmin = xc[ node ];
        if ( xc[ node ] > xmax) xmax = xc[ node ];

        if ( yc[ node ] < ymin) ymin = yc[ node ];
        if ( yc[ node ] > ymax) ymax = yc[ node ];

        if ( zc[ node ] < zmin) zmin = zc[ node ];
        if ( zc[ node ] > zmax) zmax = zc[ node ];
      }
    }

    min_extent[0] = xmin;
    max_extent[0] = xmax;
    min_extent[1] = ymin;
    max_extent[1] = ymax;
    min_extent[2] = zmin;
    max_extent[2] = zmax;

    UCDstructure_set_extent (*Output, min_extent, max_extent);


    return(1);

}


/**********************************************************************/

#ifdef SEP_EXE
/*
 * initialization for modules contained in this file.
 */

int ((*mapper_mod_list[])()) = {
  mesh_to_ucd_desc,
};

#define NMODS sizeof(mapper_mod_list) / sizeof(char *) 

AVSinit_modules ()
{
      AVSinit_from_module_list(mapper_mod_list, NMODS);
}
#endif

