/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/*   
     This software is copyright (C) 1992,  Regents  of  the
University  of  California.   Anyone may reproduce new_bubbleviz.c,
the software in this distribution, in whole or in part, pro-
vided that:

(1)  Any copy  or  redistribution  of  new_bubbleviz.c  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;

(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
in  new_bubbleviz.c  under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     new_bubbleviz.c is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
The Regents of the University of California  shall  have  no
liability  with respect to the infringement of copyrights by
new_bubbleviz.c, or any part thereof.     

Author:
	Wes Bethel   
	Lawrence Berkeley Laboratory
	1 Cyclotron Rd.   Mail Stop 50-F
	Berkeley CA 94720
	510-486-6626
	ewbethel@lbl.gov

    27 aug 92

    written to replace the bubbleviz module shipped with avs.

    bubbleviz takes a scalar field and performs a colorizing operation,
    effectively replacing each scalar datum in the field with a 4-vector
    datum representing ARGB.  This module is designed to work in concert
    with the [new] scatter dots module to produce spheres of varying
    color and radius.
*/

#include <stdio.h>
#include <avs/avs.h>
#include <avs/field.h>
#include <avs/colormap.h>

AVSinit_modules()
{
    int new_bviz();
    AVSmodule_from_desc(new_bviz);
}

int
new_bviz()
{
    int p;
    int new_bviz_p();
    
    AVSset_module_name("new bubbleviz",MODULE_MAPPER);
    AVScreate_input_port("Input Scalar Field","field scalar",REQUIRED);
    AVScreate_input_port("Colormap","colormap",REQUIRED);
    AVScreate_output_port("Output Color Field","field 1D 3-space irregular 4-vector real");
    AVSadd_float_parameter("Opacity Scale",1.0,0.0,100.0);
    AVSset_compute_proc(new_bviz_p);
}

#define MAX_DATA_DIMS 128

int
new_bviz_p(inf,cmap,outf,scale)
AVSfield *inf;
AVScolormap *cmap;
AVSfield_float **outf;
float *scale;
{
    int i,j,k,npoints;
    unsigned char *table,*t1,*t2;
    float *x,*y,*z,t,dt;
    int outpoints;
    int map_size;
    float map_min,map_max,map_scale;
    AVSfield_float template;
    int dimensions[MAX_DATA_DIMS];
    float *R,*G,*B,*A;
    float r,g,b;

    /* first, figure out if the field is acceptable or not.  we permit
       and 3-space field, or 3d uniform fields. */
    if (inf->nspace != 3)
    {
	if ((inf->ndim != 3) && (inf->uniform == UNIFORM))
	{
	    AVSwarning("Only 3-space irregular, rectilinear or 3D uniform fields are permitted.");
	    return(0);
	}
    }
    
    /* pass 1 - figure out how many entries in output field.  what
       we do is scan through the input field and do two things: 1. we
       construct a table such that the first column is the colormap
       index, and the second column tells if the input data has non-zero
       opacity or not.  the two-column table business is done, rather than
       using a signed short type since some machines have schizophrenic
       definitions for shorts, and there is the potential for extremely
       large amounts of memory consumption by this module. */

    for (i=0,npoints=1;i<inf->ndim;i++)
	npoints *= inf->dimensions[i];

    table = (unsigned char *)malloc(sizeof(unsigned char) * 2 * npoints);
    if (table == NULL)
    {
	AVSfatal(" Can't malloc %d bytes for index table.  Bye.",npoints*2);
	return(0);
    }

    /* set up pointers for coords.  */
    
    if (inf->uniform == UNIFORM)
    {
	x = (float *)malloc(sizeof(float)*inf->dimensions[0]);
	y = (float *)malloc(sizeof(float)*inf->dimensions[1]);
	z = (float *)malloc(sizeof(float)*inf->dimensions[2]);
	
	if (inf->points != NULL) /* use defined extents */
	{
	    t = inf->points[0];  /* xmin */
	    dt = (inf->points[1] - inf->points[0]); /* xmax - xmin */
	    dt = dt / (inf->dimensions[0] - 1); /* divide by # of steps */
	    for (i=0;i<inf->dimensions[0];i++,t+=dt)
		*(x+i) = t;
	    
	    t = inf->points[2];  /* ymin */
	    dt = (inf->points[3] - inf->points[2]); /* ymax - ymin */
	    dt = dt / (inf->dimensions[1] - 1); /* divide by # of steps */
	    for (i=0;i<inf->dimensions[1];i++,t+=dt)
		*(y+i) = t;
	    
	    t = inf->points[4];  /* zmin */
	    dt = (inf->points[5] - inf->points[4]); /* zmax - zmin */
	    dt = dt / (inf->dimensions[2] - 1); /* divide by # of steps */
	    for (i=0;i<inf->dimensions[2];i++,t+=dt)
		*(z+i) = t;
	}
	else  /* use size of field as coords */
	{
	    t = 0.;
	    dt = 1.;
	    for (i=0;i<inf->dimensions[0];i++,t+=dt)
		*(x+i) = t;
	    
	    t = 0.;
	    for (i=0;i<inf->dimensions[1];i++,t+=dt)
		*(y+i) = t;
	    
	    t = 0.;
	    for (i=0;i<inf->dimensions[2];i++,t+=dt)
		*(z+i) = t;
	}
    }
    else if (inf->uniform == RECTILINEAR)
    {
	x = inf->points;
	y = x + inf->dimensions[0];
	z = y + inf->dimensions[1];
    }
    else /* its IRREGULAR */
    {
	x = inf->points;
	y = x + npoints;
	z = y + npoints;
    }

    map_min = cmap->lower;
    map_max = cmap->upper;
    map_size = cmap->size;
    map_scale = (map_size - 1)/(map_max - map_min);

    AVSmodule_status(" Scanning data...",0);

    scan_data(inf,npoints,table,&outpoints,map_size,map_min,map_scale,cmap->alpha);

    /* now, figure out size of output field. */

    memset((char *)&template,0,sizeof(AVSfield_char));
    
    template.ndim = 1;
    template.nspace = 3;
    template.veclen = 4;
    template.type = AVS_TYPE_REAL;
    template.size = sizeof(float);
    template.uniform = IRREGULAR;

    dimensions[0] = outpoints;
    
    *outf = (AVSfield_float *)AVSfield_alloc(&template,dimensions);
    if (*outf == NULL)
    {
	AVSfatal(" Can't malloc enough space for output field. ");
	return(0);
    }

    /* convert colortable to RGB */
    
    R = (float *)malloc(sizeof(float)*map_size);
    G = (float *)malloc(sizeof(float)*map_size);
    B = (float *)malloc(sizeof(float)*map_size);
    A = (float *)malloc(sizeof(float)*map_size);

    for (i=0;i<map_size;i++)
    {
	hsv_to_rgb(&r,&g,&b,cmap->hue[i],cmap->saturation[i],cmap->value[i]);
#if 0
	*(R+i) = r * 255.;
	*(G+i) = g * 255.;
	*(B+i) = b * 255.;
	*(A+i) = cmap->alpha[i] * 255.;
#endif
	*(R+i) = r;
	*(G+i) = g;
	*(B+i) = b;
	*(A+i) = cmap->alpha[i];
    }

    AVSmodule_status(" Processing...",0);
    if (inf->uniform == IRREGULAR)
	process_irreg_data(inf,*outf,x,y,z,npoints,outpoints,table,cmap,A,R,G,B,scale);
    else
	process_rect_data(inf,*outf,x,y,z,npoints,outpoints,table,cmap,A,R,G,B,scale);

    if (inf->uniform == UNIFORM)
    {

/* IAC CODE CHANGE : 	free((char *)x); */
	 free(x);

/* IAC CODE CHANGE : 	free((char *)y); */
	 free(y);

/* IAC CODE CHANGE : 	free((char *)z); */
	 free(z);
    }
    

/* IAC CODE CHANGE :     free((char *)R); */
     free(R);

/* IAC CODE CHANGE :     free((char *)G); */
     free(G);

/* IAC CODE CHANGE :     free((char *)B); */
     free(B);

/* IAC CODE CHANGE :     free((char *)A); */
     free(A);

/* IAC CODE CHANGE :     free((char *)table); */
     free(table);
    
    return(1);
}

scan_data(inf,npoints,table,outpoints,msize,mmin,mscale,opacity)
AVSfield *inf;
int npoints;
unsigned char *table;
int *outpoints;
int msize;
float mmin,mscale;
float *opacity;
{
    switch (inf->type)
    {
    case AVS_TYPE_BYTE:
	scan_byte_data(inf,npoints,table,outpoints,msize,mmin,mscale,opacity);
	break;
    case AVS_TYPE_INTEGER:
	scan_integer_data(inf,npoints,table,outpoints,msize,mmin,mscale,opacity);
	break;
	
    case AVS_TYPE_REAL:
	scan_float_data(inf,npoints,table,outpoints,msize,mmin,mscale,opacity);
	break;
	
    case AVS_TYPE_DOUBLE:
	scan_double_data(inf,npoints,table,outpoints,msize,mmin,mscale,opacity);
	break;
    }
}

scan_byte_data(inf,npoints,table,outpoints,msize,mmin,mscale,opacity)
AVSfield *inf;
int npoints;
unsigned char *table;
int *outpoints;
int msize;
float mmin,mscale;
float *opacity;
{
    register unsigned char *s,*t1,*t2;
    register int index,accumulator;
    int i,j,k,map_size;
    float t;

    s = inf->field_union.field_data_char;
    t1 = table;
    t2 = t1 + npoints;

    map_size = msize;

    accumulator = 0;

    for (i=0;i<npoints;i++,t1++,t2++)
    {
    
	t = *s++;
	index = (t - mmin)*mscale;
		
	if (index < 0)
	    index = 0;
	else if (index >= map_size)
	    index = map_size-1;
#if 0
	/* if you're paranoid, compile this code and include
	   it to check for exceeding 8 bits of colortable index. */

	if (index >> 8)
	    AVSwarning(" exceeded 8 bits of colortable index: %d ",index);

#endif		
	*t1 = index;  /* if colormap is ever > 256 in length,
			 the table will have to be changed
			 to an int table. */
		
	if (*(opacity+index) != 0.)
	{
	    accumulator++;
	    *t2 = 1;  /* signal valid data */
	}
	else
	    *t2 = 0;
    }
    *outpoints = accumulator;
}

scan_integer_data(inf,npoints,table,outpoints,msize,mmin,mscale,opacity)
AVSfield *inf;
int npoints;
unsigned char *table;
int *outpoints;
int msize;
float mmin,mscale;
float *opacity;
{
    register int *s;
    register unsigned char *t1,*t2;
    register int index,accumulator;
    int i,j,k,map_size;
    float t;

    s = inf->field_union.field_data_int_u;
    t1 = table;
    t2 = t1 + npoints;

    map_size = msize;

    accumulator = 0;

    for (i=0;i<npoints;i++,t1++,t2++)
    {
    
	t = *s++;
	index = (t - mmin)*mscale;
		
	if (index < 0)
	    index = 0;
	else if (index >= map_size)
	    index = map_size-1;
#if 0
	/* if you're paranoid, compile this code and include
	   it to check for exceeding 8 bits of colortable index. */

	if (index >> 8)
	    AVSwarning(" exceeded 8 bits of colortable index: %d ",index);

#endif		
	*t1 = index;  /* if colormap is ever > 256 in length,
			 the table will have to be changed
			 to an int table. */
		
	if (*(opacity+index) != 0.)
	{
	    accumulator++;
	    *t2 = 1;  /* signal valid data */
	}
	else
	    *t2 = 0;
    }
    *outpoints = accumulator;
}

scan_float_data(inf,npoints,table,outpoints,msize,mmin,mscale,opacity)
AVSfield *inf;
int npoints;
unsigned char *table;
int *outpoints;
int msize;
float mmin,mscale;
float *opacity;
{
    register float *s;
    register unsigned char *t1,*t2;
    register int index,accumulator;
    int i,j,k,map_size;
    float t;

    s = inf->field_union.field_data_float_u;
    t1 = table;
    t2 = t1 + npoints;

    map_size = msize;

    accumulator = 0;

    for (i=0;i<npoints;i++,t1++,t2++)
    {
    
	t = *s++;
	index = (t - mmin)*mscale;
		
	if (index < 0)
	    index = 0;
	else if (index >= map_size)
	    index = map_size-1;
#if 0
	/* if you're paranoid, compile this code and include
	   it to check for exceeding 8 bits of colortable index. */

	if (index >> 8)
	    AVSwarning(" exceeded 8 bits of colortable index: %d ",index);

#endif		
	*t1 = index;  /* if colormap is ever > 256 in length,
			 the table will have to be changed
			 to an int table. */
		
	if (*(opacity+index) != 0.)
	{
	    accumulator++;
	    *t2 = 1;  /* signal valid data */
	}
	else
	    *t2 = 0;
    }
    *outpoints = accumulator;
}

scan_double_data(inf,npoints,table,outpoints,msize,mmin,mscale,opacity)
AVSfield *inf;
int npoints;
unsigned char *table;
int *outpoints;
int msize;
float mmin,mscale;
float *opacity;
{
    register double *s;
    register unsigned char *t1,*t2;
    register int index,accumulator;
    int i,j,k,map_size;
    float t;

    s = inf->field_union.field_data_double_u;
    t1 = table;
    t2 = t1 + npoints;

    map_size = msize;

    accumulator = 0;

    for (i=0;i<npoints;i++,t1++,t2++)
    {
    
	t = *s++;
	index = (t - mmin)*mscale;
		
	if (index < 0)
	    index = 0;
	else if (index >= map_size)
	    index = map_size-1;
#if 0
	/* if you're paranoid, compile this code and include
	   it to check for exceeding 8 bits of colortable index. */

	if (index >> 8)
	    AVSwarning(" exceeded 8 bits of colortable index: %d ",index);

#endif		
	*t1 = index;  /* if colormap is ever > 256 in length,
			 the table will have to be changed
			 to an int table. */
		
	if (*(opacity+index) != 0.)
	{
	    accumulator++;
	    *t2 = 1;  /* signal valid data */
	}
	else
	    *t2 = 0;
    }
    *outpoints = accumulator;
}

process_rect_data(inf,outf,x,y,z,npoints,outpoints,table,cmap,A,R,G,B,scale)
AVSfield *inf;
AVSfield_float *outf;
float *x,*y,*z;
int npoints;
unsigned char *table;
float *A,*R,*G,*B;
float *scale;
{
    register int t;
    int i,j,k;
    float *dest;
    unsigned char *index,*valid;
    float *ox,*oy,*oz;

    ox = outf->points;
    oy = ox + outpoints;
    oz = oy + outpoints;

    dest = outf->data;
    index = table;
    valid = index + npoints;
    
    /* take indeces and copy over colors and opacities if
       opacity is non-zero.*/
    
    for (k=0;k<inf->dimensions[2];k++)
    {
	for (j=0;j<inf->dimensions[1];j++)
	{
	    for (i=0;i<inf->dimensions[0];i++,valid++,index++)
	    {
		if (*valid)
	        {
		    t = *index;
		    *dest++ = A[t] * *scale;
		    *dest++ = R[t];
		    *dest++ = G[t];
		    *dest++ = B[t];

		    *ox++ = x[i];
		    *oy++ = y[j];
		    *oz++ = z[k];
		}
	    }
	}
    }
}

process_irreg_data(inf,outf,x,y,z,npoints,outpoints,table,cmap,A,R,G,B,scale)
AVSfield *inf;
AVSfield_float *outf;
float *x,*y,*z;
int npoints;
unsigned char *table;
float *A,*R,*G,*B;
float *scale;
{
    register int t;
    int i,j,k,p;
    float *dest;
    unsigned char *index,*valid;
    float *ox,*oy,*oz;

    ox = outf->points;
    oy = ox + outpoints;
    oz = oy + outpoints;

    dest = outf->data;
    index = table;
    valid = index + npoints;
    
    /* take indeces and copy over colors and opacities if
       opacity is non-zero.*/

    p = 0;
    
    for (i=0;i<npoints;i++,valid++,index++,p++)
    {
	if (*valid)
	{
	    t = *index;
	    *dest++ = A[t] * *scale;
	    *dest++ = R[t];
	    *dest++ = G[t];
	    *dest++ = B[t];

	    *ox++ = x[p];
	    *oy++ = y[p];
	    *oz++ = z[p];
	}
    }
}

static int
hsv_to_rgb(r,g,b,h,s,v)
float h,s,v,*r,*g,*b;
{
    static hsv_to_pixel();
   
    hsv_to_pixel(r,g,b,h,s,v);
}

static
hsv_to_pixel(R,G,B,h,s,v)
float *R,*G,*B;
float h,s,v;
{
    float f, p, q, t;
    float r, g, b;
    float ht;
    int i;
 
/*  Make sure not to trash the input colormap */
    ht = h;

    if (v == 0.)
    {
	r=0.;
	g=0.;
	b=0.;
    }
    else
    {
	if (s == 0.)
	{
	    r = v;
	    g = v;
	    b = v;
	}
	else
	{
	    ht = ht * 6.0;
	    if (ht >= 6.0)
		ht = 0.0;
      
	    i = ht;
	    f = ht - i;
	    p = v*(1.0-s);
	    q = v*(1.0-s*f);
	    t = v*(1.0-s*(1.0-f));
      
 	    if (i == 0) 
	    {
		r = v;
		g = t;
		b = p;
	    }
	    else if (i == 1)
	    {
		r = q;
		g = v;
		b = p;
	    }
	    else if (i == 2)
	    {
		r = p;
		g = v;
		b = t;
	    }
	    else if (i == 3)
	    {
		r = p;
		g = q;
		b = v;
	    }
	    else if (i == 4)
	    {
		r = t;
		g = p;
		b = v;
	    }
	    else if (i == 5)
	    {
		r = v;
		g = p;
		b = q;
	    }
	}
    }
    *R = r;
    *G = g;
    *B = b;
#if 0
    ir = r*255.0;
    ig = g*255.0;
    ib = b*255.0;

    *rgb = (unsigned char)ia;
    *(rgb+1) = (unsigned char)ir;
    *(rgb+2) = (unsigned char)ig;
    *(rgb+3) = (unsigned char)ib;
#endif
}
