/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

AUTHOR : Stardent Computer, Inc.
******************************************************************************/
/*
 * probe_path.c - Interactive probe module to sample 3D volume data
 *                using an 3D cursor, accumulating history of sample
 *                values.
 *
 * This is a special version modified for local project
 * Author: Ian Curington, Stardent Computer Ltd., UK
 *         0483 505388
 * Revision   27 March 91 Prototype
 *             3 April 91 First release
 *
 *  This is a variant of "probe" extended such that:
 *  - no sample input
 *  - trilinear only
 *  - field output with sampled data value
 *  - path accumulation
 *  - position and value in text block, not geometry window
 *
 */

#include <stdio.h>

#include <avs/avs.h>
#include <avs/field.h>
#include <avs/geom.h>
#include <avs/avs_math.h>
#include <avs/colormap.h>
#include <avs/udata.h>

/**********************************************************************/
/* ================= global statics ================== */
#define MAX_PATH  1000     /* max length of path storage */

static float       Cursor[(8*MAX_PATH+6)*3];   /* cursor vertex list */
static float       Colors[(8*MAX_PATH+6)*3];   /* line colors */
static char        *block_table = NULL;
static AVScolormap *ColorData;

static struct {
    int     num;                      /* index to last path entry */
    float   value [MAX_PATH];         /* scalar value list */
    float   coord [MAX_PATH][3];      /* coordinates of path */
} path;

/******************************************************************************/
AVSinit_modules()
{
    int probe_path();
    AVSmodule_from_desc(probe_path);
}

/**********************************************************************/
 int
probe_path()
{
    int probe_compute(), probe_destroy(), probe_init(), param;
    
    AVSset_module_name("probe path", MODULE_MAPPER);
    AVSset_module_flags( COOPERATIVE );
    
    AVScreate_input_port("field", "field 3D", REQUIRED);
    AVScreate_input_port("Input Colormap", "colormap", OPTIONAL);
    /* get upstream transform info from the geom viewer */
    param = AVScreate_input_port("Transform Info",
                                 "struct upstream_transform", 
                                   OPTIONAL | INVISIBLE);
    AVSset_input_class(param,"geometry:upstream_transform");

    param = AVScreate_input_port("upstream geom",
                                 "struct upstream_geom", 
                                   OPTIONAL | INVISIBLE);
    AVSset_input_class(param,"geometry:upstream_geom");

    AVScreate_output_port("geometry", "geom");

    /* send sampled values out on field port */
    AVScreate_output_port("Field Output","field 1D uniform scalar float");

    AVSadd_parameter("Accumulate Path", "boolean", 0, 0, 0);
    AVSadd_parameter("Pick Geometry",   "boolean", 0, 0, 0);
    param = AVSadd_parameter("Value","string_block","Probe Path","","");
    AVSadd_parameter_prop(param,"height","integer",10);
    AVSadd_parameter_prop(param,"width", "integer",4);

    AVSset_compute_proc(probe_compute);
    AVSset_destroy_proc(probe_destroy);
    AVSset_init_proc   (probe_init);
}

/**********************************************************************/
static probe_destroy () {
    if (block_table)
      UTILfield_delete_block_table(block_table);
}

/**********************************************************************/
static probe_init () {
    path.num = 0;     /* reset path */
}

/**********************************************************************/
/* =========== probe path compute routine =========== */
static
probe_compute (field, colormap, transform_info, select_info,
               output, val_out,
               init_path, pick_geom, text)
    AVSfield       *field;
    AVScolormap    *colormap;
    upstream_transform *transform_info;
    upstream_geom  *select_info;
    GEOMedit_list  **output;
    AVSfield_float **val_out;
    int init_path, pick_geom;
    char *text;
{
    GEOMobj *obj;
    float sample_pos[3], pos[3];
    float value[3], ext[6], min_extent[3], max_extent[3];
    int i, size, label_flags, cell;
    int field_size[2];
    float scalar_value;
    char string[256];
    static float deltavec[3] = {0.0, 0.0, 0.0};

    /* Set extents */
    if (!(field->flags & AVS_VALID_EXTENT)) 
      AVSfield_get_extent(field, min_extent, max_extent);

    if (field->uniform == IRREGULAR && AVSinput_changed("field", 0))
    {
        if (block_table)
        UTILfield_delete_block_table(block_table);
        block_table = (char *) UTILfield_build_block_table(field);
    }

    /* set up output field */
    if ( init_path == 0 ) path.num = 0;
    field_size[0] = path.num + 1;
    if (*val_out) AVSfield_free(*val_out);
    *val_out =(AVSfield_float *)
              AVSdata_alloc("field 1D scalar uniform float",field_size);

    ColorData = colormap;

    AVSfield_get_extent (field, min_extent, max_extent);

    /* find the extent of the bounding hull */
    for (i = 0; i < 3; i++)
    {
        ext[i*2]   = min_extent[i];
        ext[i*2+1] = max_extent[i];
    }

    /* set initial position to center */
    for (i = 0; i < 3; i++)
        pos[i] = (max_extent[i] - min_extent[i]) * 0.5;

    /* use upstream transform data if present */
    if (transform_info) 
        if (!strncmp(transform_info->object_name, "probe", strlen("probe")))
            mat_vecmul(pos, transform_info->msxform);

    if (AVSinput_changed("upstream geom", 0) && select_info)
    {
            deltavec[0] = select_info->mscoord[0] - pos[0];
            deltavec[1] = select_info->mscoord[1] - pos[1];
            deltavec[2] = select_info->mscoord[2] - pos[2];
    }

    vec_add(pos,deltavec,pos);

    sample_pos[0] = pos[0];
    sample_pos[1] = pos[1];
    sample_pos[2] = pos[2];

    if (field->uniform == UNIFORM)
    {
        for(i=0; i<3; i++)
        {
            min_extent[i] = field->points[2*i];
            max_extent[i] = field->points[2*i+1];
        }
    }

    *output = (GEOMedit_list *)GEOMinit_edit_list(*output);

    if (pick_geom) 
       GEOMedit_selection_mode(*output, "%top", "notify", 
       BUTTON_DOWN | BUTTON_MOVING | BUTTON_UP);
    else 
       GEOMedit_selection_mode(*output, "%top", "normal", 0);

    sprintf(string,
        "Probe Path\n\nPosition:  %g  %g  %g\n\n",pos[0],pos[1],pos[2]);
    if ((pos[0] >= min_extent[0] && pos[0] <= max_extent[0]) &&
        (pos[1] >= min_extent[1] && pos[1] <= max_extent[1]) &&
        (pos[2] >= min_extent[2] && pos[2] <= max_extent[2]))
    {    

        cell = UTILfield_find_ijk(field, sample_pos, block_table);
        if (field->veclen == 1)
        {
            UTILscalar_trilinear(field, sample_pos, cell, &value[0]);
            sprintf(string, "%s Value:  %g\n",string, value[0]);
            scalar_value = value[0];
        } else if (field->veclen == 3) {
            UTILvector_trilinear(field, sample_pos, cell, value);
            sprintf(string,
                "%s Values:  %g  %g  %g",string, value[0],value[1],value[2]);
            scalar_value = sqrt(value[0]*value[0] +
                                value[1]*value[1] +
                                value[2]*value[2]);
        } else {/* use only nearest neighbor for veclen > 3 */
            sprintf(string, "%s Value: ",string);
            scalar_value = 0.0;
            for (i=0; i<field->veclen; i++)
            {
                switch(field->type) {
                case AVS_TYPE_BYTE:
                  value[0] = (float)(field->field_data[field->
                                     veclen*cell+i]);
                  break;
                case AVS_TYPE_REAL:
                  value[0] = (float)(field->field_data_float[field->
                                     veclen*cell+i]);
                  break;
                case AVS_TYPE_DOUBLE:
                  value[0] = (float)(field->field_data_double[field->
                                     veclen*cell+i]);
                  break;
                case AVS_TYPE_INTEGER:
                  value[0] = (float)(field->field_data_int[field->
                                     veclen*cell+i]);
                  break;
                }
                sprintf(string, "%s %g", string, value[0]);
                scalar_value += (value[0] * value[0]);
            }
            scalar_value = sqrt(scalar_value);
        }
    }

    AVSmodify_parameter("Value",AVS_VALUE,string,NULL,NULL); 
    (*val_out)->data[0] = scalar_value;

    if ( init_path  )
    {
        path.value[ path.num ]    = scalar_value;
        path.coord[ path.num ][0] = pos[0];
        path.coord[ path.num ][1] = pos[1];
        path.coord[ path.num ][2] = pos[2];
        if ( path.num < MAX_PATH-1 ) path.num++;

        for ( i=0; i<path.num; i++ )
        (*val_out)->data[i] = path.value[i];
    }

    make_cursor(pos[0],pos[1],pos[2], scalar_value);

    obj = GEOMcreate_obj(GEOM_POLYTRI, NULL);
    if ( path.num <= 1 )
        i = 6;
    else
        i = 6 + 8 * (path.num-1);
    GEOMadd_disjoint_line(obj, Cursor, Colors, i, GEOM_COPY_DATA);
    GEOMedit_geometry(*output, "probe", obj);
    GEOMedit_transform_mode(*output, "probe", "redirect",
                             BUTTON_MOVING | BUTTON_UP);
    GEOMdestroy_obj(obj);

    return(1);
}

/******************************************************************************/
#define SET_CURSOR(x,y,z)    Cursor[ind+0] = x;  \
                             Cursor[ind+1] = y;  \
                             Cursor[ind+2] = z;  \
                             Colors[ind+0] = r; \
                             Colors[ind+1] = g; \
                             Colors[ind+2] = b; \
                             ind+=3;

static
make_cursor(x,y,z,value)
float x, y, z, value;
{
    int    i, j, ind = 0;
    float  r, g, b;

    if (ColorData)
    {
        j = PIXELget_index(ColorData, value);
        PIXELhsv_to_rgb( ColorData->hue[j],
                         ColorData->saturation[j],
                         ColorData->value[j], &r, &g, &b);
    } else {
        r = g = b = 1.0;
    }

    if (ColorData == NULL) { r = 1.0; g = b = 0.0; }
    SET_CURSOR(0.0, y,   z);
    SET_CURSOR(x  , y,   z);
    if (ColorData == NULL) { g = 1.0; r = b = 0.0; }
    SET_CURSOR(x  , 0.0, z);
    SET_CURSOR(x  , y,   z);
    if (ColorData == NULL) { b = 1.0; r = g = 0.0; }
    SET_CURSOR(x  , y, 0.0);
    SET_CURSOR(x  , y,   z);

    if (ColorData == NULL) { b = 0.8; r = 0.8; g = 0.8; }
    for (i=0; i<path.num-1; i++ )
    {
      if (ColorData)
      {
        j = PIXELget_index(ColorData, path.value[i]);
        PIXELhsv_to_rgb( ColorData->hue[j],
                         ColorData->saturation[j],
                         ColorData->value[j], &r, &g, &b);
      }
      SET_CURSOR(path.coord[i][0],   path.coord[i][1],   path.coord[i][2]);
      SET_CURSOR(path.coord[i+1][0], path.coord[i+1][1], path.coord[i+1][2]);
      SET_CURSOR(0.0,                path.coord[i][1],   path.coord[i][2]);
      SET_CURSOR(0.0,                path.coord[i+1][1], path.coord[i+1][2]);
      SET_CURSOR(path.coord[i][0],   0.0,                path.coord[i][2]);
      SET_CURSOR(path.coord[i+1][0], 0.0,                path.coord[i+1][2]);
      SET_CURSOR(path.coord[i][0],   path.coord[i][1],   0.0);
      SET_CURSOR(path.coord[i+1][0], path.coord[i+1][1], 0.0);
    }
}

