/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/

/* NOTE:  THIS MODULE AND SOURCE CODE IS FOR USE 
   WITH THE AVS SOFTWARE ENVIRONMENT ONLY */
/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/*
 * scatter_ribbon - draw geometric ribbon from scatter data
 *
 * If your vector data is 1D n-vector float irregular,
 * this module will draw a ribbon going along the side of
 * the polyline path, from joining up the specified positions
 * in the scatter field coord array.
 * This module is used for Well Log data display,
 * showing various properties down a deviated well shaft.
 * The width, and color of the ribbon can be
 * controlled to create a ribbon data representation.
 * Tick marks and labels will annotate the graph.
 * Interpolation can be turned on and off.
 *
 * Input: field 1D irregular float (scatter)
 *
 * Output: geometry of ribbon
 *
 * Parameters:
 *     scale     - relative ribbon width
 *     Angle     - 3D angle away from Z-axis
 *     Width Low - lowest data value to map to ribbon width
 *     Width High- highest data value to map to ribbon width
 *     mode      - No color, checker colors, scalar colors
 *     Uniform   - fixed width ribbon
 *     Block     - solid fill each block, stair-step edge
 *     Color_Vector    - which scalar to map to ribbon color
 *     Width_Vector    - which scalar to map to ribbon width
 *     tic_interval,   - how often to draw tick marks
 *     label_interval, - how often to draw labels
 *     draw_tics,      - on/off control
 *     draw_labels,     - on/off control
 *     On/Off          - total module control
 *     
 *
 * Author: I. Curington, AVS Inc, UK
 *
 * Revision:
 * 31 March 92  Original (from ucd_cell_ribbon, cell data)
 * 14 July  92  Conversion to scatter fields
 * 15 July  92  bug fix in positioning
 * 15 Aug   92  conversion of scatter_arrows to scatter_ribbon
 * 24 Aug   93  adapted scatter_ribbon to scatter ribbons
 *              added controls
 * 25 Aug   93  ripped out tristrip, put in disj quad code
 * 26 Aug   93  added width min-max control
 *              added tics and labels
 * 13 Sept  93  added Z inversion define
 * 15 Sept  93  extra annotation lines
 *
 */


/*-----------------------------------------------------*
 *                                                     *
 *      ****  scatter_ribbon   module  ****            *
 *                                                     *
 *                                                     *
 *-----------------------------------------------------*/

#include <stdio.h>


#include <avs/avs.h>
/* IAC CODE CHANGE : #include <avs/avs_math.h> */
#include <avs/avs_math.h>
#include <avs/geom.h>
#include <avs/field.h>
#include <avs/colormap.h>

static float    orange[3] = { 0.99, 0.3, 0.1 };
static float    violet[3] = { 0.9, 0.35, 0.62 };
static float    blue[3]   = { 0.1, 0.1, 0.5 };
static float    grey[3]   = { 0.6, 0.6, 0.6 };
static float    red[3]    = { 0.99, 0.0, 0.0 };

#define NUM_STEPS  2

/**
#define INVERT_Z   1
 **/

/* =============== compute routine ============== */

int scatter_ribbon (scatter,
                    cmap,
                    output,
                    scale,
                    angle,
                    width_low,
                    width_high,
                    uniform,
                    block,
                    tic_interval,
                    label_interval,
                    draw_tics,
                    draw_labels,
                    label_dummy_1,
                    variable,
                    label_dummy_2,
                    color_vec_choice,
                    label_dummy_3,
                    width_vec_choice,
                    on_off )

  AVSfield_float *scatter;  /* scatter field */
  AVScolormap    *cmap;     /* color port */
  GEOMedit_list  *output;   /* geom out */
  float         *scale;     /* ribbon width */
  float         *angle;     /* x-y rotation control for ribbon */
  float         *width_low; /* ribbon width min-var */
  float         *width_high;/* ribbon width max-var */
  int           uniform;    /* uniform or variable width */
  int           block;      /* block or smooth colors */
  int           tic_interval;   /* spacing of tic marks */
  int           label_interval; /* spacing of labels */
  int           draw_tics;      /* draw-hide tics */
  int           draw_labels;    /* draw-hide labels */
  char          *label_dummy_1;  /* panel annotation */
  char          *variable;  /* mode control */
  char          *label_dummy_2;  /* panel annotation */
  char          *color_vec_choice;
  char          *label_dummy_3;  /* panel annotation */
  char          *width_vec_choice;
  int           on_off;      /* module on-off control */
{
    GEOMobj *obj;
    GEOMobj *obj_base;
    GEOMobj *obj_label;
    int   mode;
    int   color_vector;
    int   width_vector;
    int   num_vertex;
    int   result;
    float ang;
    float offset;
    char  data_labels[200];
    char  delim[4];
    int   veclen;

   /***********
    * Body    *
    ***********/

    /* update the color and width user-interface */
    if ( AVSfield_get_labels( scatter, data_labels, delim ) )
    {
        AVSmodify_parameter("Color_Vector", AVS_MINVAL | AVS_MAXVAL,
        color_vec_choice, data_labels, delim );
        AVSmodify_parameter("Width_Vector", AVS_MINVAL | AVS_MAXVAL,
        width_vec_choice, data_labels, delim );
    }

    /* resolve string-choices */
    /* Mode = 1: No Color
       Mode = 2: checkerboard
       Mode = 3: scalar field color
     */

    mode         = AVSchoice_number("Mode",variable);
    color_vector = AVSchoice_number("Color_Vector",color_vec_choice)-1;
    width_vector = AVSchoice_number("Width_Vector",width_vec_choice)-1;
    if ( color_vector >= scatter->veclen ) color_vector = -1;
    if ( width_vector >= scatter->veclen ) width_vector = -1;
    if ( uniform ) width_vector = -1;
    if ( mode == 3 && !cmap ) mode = 1;
    if ( mode == 3 && color_vector < 0 ) mode = 1;
    offset = *scale;
    if ( scatter->veclen <= 0 )
         color_vector = width_vector = -1;

    /* establish the geometry output object */
    *output = GEOMinit_edit_list(*output);
    obj    =  GEOMcreate_obj ( GEOM_POLYHEDRON, GEOM_NULL);
    obj_base  = GEOMcreate_obj ( GEOM_POLYTRI,    GEOM_NULL );
    obj_label = GEOMcreate_label(GEOM_NULL, NULL);


    /* flag direction vector of ribbon offset */
    /* convert degrees to radians */

    ang =  (M_PI / 180.) * (*angle);


    /* draw the geometry from the scatter list */

    if ( on_off )
    {
        ribbon_gen   (obj,
                      scatter,
                      cmap,
                      mode,
                      offset,
                      color_vector,
                      width_vector,
                      block,
                      ang, 
                      *width_low,
                      *width_high );

        GEOMgen_normals(obj, 0);
        GEOMcvt_polyh_to_polytri(obj, GEOM_SURFACE );
        /* choices: GEOM_NO_CONNECTIVITY | GEOM_SURFACE | GEOM_WIREFRAME */
    }

    /* draw ticmarks down side of scatter */
    if ( draw_tics && on_off )
    {
        draw_base_line (obj_base, scatter, ang, offset,
                        tic_interval, label_interval );
    }

    /* draw labels down side of scatter */
    if ( draw_labels && on_off )
    {
        draw_label_object (obj_label, scatter, ang, offset, label_interval );
    }


    /* complete the geometry to output port process */


    GEOMedit_geometry (*output, "scatter_ribbon", obj);
    GEOMedit_geometry (*output, "scatter_ribbon", obj_base);
    GEOMedit_geometry (*output, "scatter_ribbon", obj_label);

    GEOMdestroy_obj (obj);
    GEOMdestroy_obj (obj_base);
    GEOMdestroy_obj (obj_label);

    GEOMedit_transform_mode (*output, "scatter_ribbon", "parent");


    return(1);
}

/*-----------------------------------------------------*
 *                                                     *
 *       ****  create_ribbon  ****                     *
 *                                                     *
 * Utility to make ribbon from a scatter field         *
 *-----------------------------------------------------*/
 
ribbon_gen      (obj,
                 scatter,
                 cmap,
                 mode,
                 offset,
                 color_vector,
                 width_vector,
                 block,
                 ang,
                 width_low,
                 width_high )

    GEOMobj    *obj;           /* output geom struct */
    AVSfield_float *scatter;   /* scatter polyline */
    AVScolormap *cmap;         /* colormap struct */
    int         mode;          /* input color method parameters */
    float       offset;        /* relative width control */
    int         color_vector;  /* which data vector to map to color, -1 for none */
    int         width_vector;  /* which data vector to map to width, -1 for none */
    int         block;         /* block or smooth color mode */
    float       ang;           /* angle to direct ribbon */
    float       width_low;     /* ribbon width min-var */
    float       width_high;    /* ribbon width max-var */
{
    int         i, j, k, n;
    float       p[3], q[3];
    float       l, ct, st;
    float       scalar_sample;
    float       base_col[3];
    float       tip_col[3];
    float       base_width;
    float       tip_width;
    int         veclen;
    float       verts[ 4 * 3 ];
    float       cols [ 4 * 3 ];
    float       *p_cols;
    float       xt[3];          /* ribbon angle vector */
    int         geom_flags;

    /********************
     *  B O D Y         *
     ********************/

    n = MAXX( scatter );
    veclen = scatter->veclen;
    if ( mode == 1 )
       p_cols = NULL;
    else
       p_cols = cols;

    if ( block )
       geom_flags = GEOM_NOT_SHARED;
    else
       geom_flags = GEOM_SHARED;

    xt[0] = sin ( ang );
    xt[1] = cos ( ang );
    xt[2] = 0.0;
 
    /* loop over input polyline length */
    for( i=0; i < n-1; i++)
    {
 
      /* get first point */
      p[0] = scatter->points[0*n+i];
      p[1] = scatter->points[1*n+i];
      p[2] = scatter->points[2*n+i];

      /* get second point */
      q[0] = scatter->points[0*n+i+1];
      q[1] = scatter->points[1*n+i+1];
      q[2] = scatter->points[2*n+i+1];
 
      /* find scalar field to color */
      if ( mode == 3 && color_vector >= 0 )
      {
         scalar_sample = scatter->data[ i*veclen + color_vector ];
         map_to_colours ( cmap, scalar_sample,
                               &base_col[0], &base_col[1], &base_col[2] );

         if (!block )
         {
             scalar_sample = scatter->data[ (i+1)*veclen + color_vector ];
             map_to_colours ( cmap, scalar_sample,
                               &tip_col[0], &tip_col[1], &tip_col[2] );
         }
         else
         {
             tip_col[0] = base_col[0];
             tip_col[1] = base_col[1];
             tip_col[2] = base_col[2];
         }
      }

      /* find scalar field to control ribbon width */
      if ( width_vector >= 0 )
      {
         scalar_sample = scatter->data[ i*veclen + width_vector ];
         map_data_to_width ( width_low, width_high, scalar_sample, &base_width );

         if (!block )
         {
             scalar_sample  = scatter->data[ (i+1)*veclen + width_vector ];
             map_data_to_width ( width_low, width_high, scalar_sample, &tip_width );
         }
         else
         {
             tip_width = base_width;
         }
      }
      else
      {
         base_width = tip_width = 1.0;
      }


      /* X, Y, Z loop: */
      for( k=0; k < 3; k++ )
      {
            /* create vertex positions */
            verts[ 0 * 3 + k ]= p[k];
            verts[ 1 * 3 + k ]= p[k] + base_width*offset*xt[k];
            verts[ 2 * 3 + k ]= q[k] + tip_width *offset*xt[k];
            verts[ 3 * 3 + k ]= q[k];
 
            /* set vertex colours, checkerboard mode */
            if (mode == 2)
              if ( (i & 0x00000001 ) == 0 )
              {
                cols[ 0 * 3 + k ] = orange[k];
                cols[ 1 * 3 + k ] = orange[k];
                cols[ 2 * 3 + k ] = orange[k];
                cols[ 3 * 3 + k ] = orange[k];
              }
              else
              {
                cols[ 0 * 3 + k ] = violet[k];
                cols[ 1 * 3 + k ] = violet[k];
                cols[ 2 * 3 + k ] = violet[k];
                cols[ 3 * 3 + k ] = violet[k];
              }
 
            /* set vertex colors, based on scalar field data */
            else if (mode == 3 && color_vector >= 0 )
            {
                cols[ 0 * 3 + k ] = base_col[k];
                cols[ 1 * 3 + k ] = base_col[k];
                cols[ 2 * 3 + k ] = tip_col[k];
                cols[ 3 * 3 + k ] = tip_col[k];
            }
      }

      if ( tip_width > 0.0 && base_width > 0.0 )
          GEOMadd_disjoint_polygon ( obj, verts, NULL, p_cols, 4,
                                     geom_flags, GEOM_COPY_DATA);


    } /* end of loop over input scatter length */
 
    return;
}

/*-----------------------------------------------------*
 *                                                     *
 *             ****  map_to_colours  ****         *
 *                                                     *
 *-----------------------------------------------------*/

map_to_colours ( cmap, val, red, green, blue )

AVScolormap *cmap;
float val;    /* input value to color */
float *red;   /* output color for sample */
float *green; /* output color for sample */
float *blue;  /* output color for sample */

{

  int index;
  float scale;
  float vmin, vmax;

  vmin = cmap->lower;
  vmax = cmap->upper;

  if (vmax != vmin)
    scale =  (float)(BYTE_TABLE - 1) / (vmax - vmin);
  else
    scale = 0.0;

  index = (int)((val - vmin) * scale);

  if ( index < 0 ) index = 0;
  else if ( index >= BYTE_TABLE ) index = BYTE_TABLE -1;

  PIXELhsv_to_rgb( cmap->hue       [index],
                   cmap->saturation[index],
                   cmap->value     [index],
                   red, green, blue ); 
}


/*-----------------------------------------------------*
 *                                                     *
 *             ****  map_data_to_width  ****           *
 *             normalized 0-1 output                   *
 *                                                     *
 *-----------------------------------------------------*/

map_data_to_width ( width_low, width_high, val, new_val )

    float width_low;
    float width_high;
    float val;       /* input sample value */
    float *new_val;  /* output width value for sample */

{

  float scale;

  if (width_high != width_low)
    scale =  1.0 / (width_high - width_low);
  else
    scale = 0.0;

  *new_val = (val - width_low) * scale;

  if ( *new_val < 0.0 ) *new_val = 0.0;
  else if ( *new_val > 1.0 ) *new_val = 1.0;

}

/*-----------------------------------------------------*
 *                                                     *
 *    ****  draw tick marks along scatter  ****        *
 *                                                     *
 *-----------------------------------------------------*/

draw_base_line (obj_base, scatter, ang, offset,
                tic_interval, label_interval )
    GEOMobj *obj_base;
    AVSfield_float *scatter;
    float ang;
    float offset;
    int   tic_interval;
    int   label_interval;
{

    int         i, j, k, n;
    float       p[3];
    float       q[3];
    float       verts[ 10 * 3 ];
    float       cols [ 10 * 3 ];
    float       xt[3];          /* tic angle vector */
    int         next;

    /********************
     *  B O D Y         *
     ********************/

    n = MAXX( scatter );

    xt[0] = sin ( ang );
    xt[1] = cos ( ang );
    xt[2] = 0.0;
 
    /* Create Minor Tic marks on Oposite side of ribbon */
    cols [ 0 * 3 + 0 ] = cols [ 1 * 3 + 0 ] = red[0];
    cols [ 0 * 3 + 1 ] = cols [ 1 * 3 + 1 ] = red[1];
    cols [ 0 * 3 + 2 ] = cols [ 1 * 3 + 2 ] = red[2];

    /*  MINOR TICKS - loop over input polyline length */
    for( i=0; i < n; i+= tic_interval)
    {
      for( k=0; k < 3; k++ )     /* X, Y, Z loop: */
      {
            /* get point */
            p[k] = scatter->points[k*n+i];

            /* create vertex positions */
            verts[ 0 * 3 + k ]= p[k];
            verts[ 1 * 3 + k ]= p[k] - 0.3 * offset*xt[k];
      }
      GEOMadd_disjoint_line ( obj_base, verts, cols, 2, GEOM_COPY_DATA);
    }

    /* Create MAJOR Tic marks on Oposite side of ribbon */
    cols [ 0 * 3 + 0 ] = cols [ 1 * 3 + 0 ] = violet[0];
    cols [ 0 * 3 + 1 ] = cols [ 1 * 3 + 1 ] = violet[1];
    cols [ 0 * 3 + 2 ] = cols [ 1 * 3 + 2 ] = violet[2];

    /*  MAJOR TICKS - loop over input polyline length */
    for( i=0; i < n; i+= label_interval)
    {
      for( k=0; k < 3; k++ )     /* X, Y, Z loop: */
      {
            /* get point */
            p[k] = scatter->points[k*n+i];

            /* create vertex positions */
            verts[ 0 * 3 + k ]= p[k];
            verts[ 1 * 3 + k ]= p[k] - 0.5 * offset*xt[k];
      }
      GEOMadd_disjoint_line ( obj_base, verts, cols, 2, GEOM_COPY_DATA);
    }

    /* Create Axes ruler lines */
    for( i=0; i < n; i+= tic_interval)
    {
      next = i + tic_interval;
      if ( next >= n ) next = n-1;

      for( k=0; k < 3; k++ )     /* X, Y, Z loop: */
      {
            /* set line color */
            cols [ 0 * 3 + k ] = cols [ 1 * 3 + k ] = red[k];
            cols [ 2 * 3 + k ] = cols [ 3 * 3 + k ] = red[k];

            /* get point */
            p[k] = scatter->points[k*n+i];
            q[k] = scatter->points[k*n+next];

            /* create vertex positions */
            verts[ 0 * 3 + k ]= p[k] + 0.5 * offset*xt[k];
            verts[ 1 * 3 + k ]= q[k] + 0.5 * offset*xt[k];
            verts[ 2 * 3 + k ]= p[k] + 1.0 * offset*xt[k];
            verts[ 3 * 3 + k ]= q[k] + 1.0 * offset*xt[k];
      }
      GEOMadd_disjoint_line ( obj_base, verts, cols, 4, GEOM_COPY_DATA);
    }



}



/*-----------------------------------------------------*
 *                                                     *
 *    ****  draw labels along scatter  ****            *
 *                                                     *
 *-----------------------------------------------------*/

draw_label_object (obj, scatter, ang, offset, label_interval )
    GEOMobj *obj;
    AVSfield_float *scatter;
    float ang;
    float offset;
    int   label_interval;
{

    int         i, k, n;
    float       p[3];
    float       verts[ 2 * 3 ];
    float       xt[3];          /* tic angle vector */
    float       z;

    int label_flags;
    int font, title, background, drop, align, stroke;
    float label_center[3];
    float label_offset[3];
    float label_size;
    char  string[40];


    /********************
     *  B O D Y         *
     ********************/

    n = MAXX( scatter );

    xt[0] = sin ( ang );
    xt[1] = cos ( ang );
    xt[2] = 0.0;
 
    font = 1;                   /* font 0-21 */
    title = 0;                  /* use ref_point */
    background = 1;             /* 1=filled rectable */
    drop = 1;                   /* 0=no, 1=drop shadow */
    stroke = 0;                 /* always zero */
    align = GEOM_LABEL_RIGHT;
    label_size = 0.04;          /* height in screen space */
    label_center[2] =
    label_offset[0] = label_offset[1] = label_offset[2] = 0.0;
    label_flags = GEOMcreate_label_flags(
               font, title, background, drop,align, stroke);


    /* loop over input polyline length */
    for( i=0; i < n; i+= label_interval )
    {
 
      /* X, Y, Z loop: */
      for( k=0; k < 3; k++ )
      {
            /* get label point */
            z    = scatter->points[k*n+i];
            p[k] = scatter->points[k*n+i] - 0.75 * offset*xt[k];
      }

#ifdef INVERT_Z
      sprintf(string,"%f", -1.0 * z );   /* negative depth */
#else
      sprintf(string,"%f", z);   /*  depth */
#endif
      GEOMadd_label ( obj, string, p,
                      label_offset, label_size, GEOM_NULL, label_flags );

    }


}


/*-----------------------------------------------------*
 *                                                     *
 *        ****  scatter_ribbon_desc  ****               *
 *                                                     *
 *-----------------------------------------------------*/
int
scatter_ribbon_desc()
{
  extern int scatter_ribbon();
  int  param;

  /* name on the box */
  AVSset_module_name ("scatter_rib", MODULE_MAPPER);
  AVSset_module_flags( COOPERATIVE | REENTRANT );


  /*
   * Input Ports
   */
  /* input scatter field structure to draw */
  AVScreate_input_port ("Scatter",
     "field 1D 3-space irregular float", REQUIRED);

  AVScreate_input_port("Colormap", "colormap", OPTIONAL);

  /*
   * Output Ports
   */
  /* output geomerty to draw */
  AVScreate_output_port ("scatter_ribbon", "geom");

  /*
   * Parameters
   */
  AVSadd_float_parameter("Scale", 1.0, FLOAT_UNBOUND, FLOAT_UNBOUND);

  AVSadd_float_parameter("Angle", 0.0, 0.0, 360.);

  AVSadd_float_parameter("Width Low",  0.0, FLOAT_UNBOUND, FLOAT_UNBOUND);

  AVSadd_float_parameter("Width High", 1.0, FLOAT_UNBOUND, FLOAT_UNBOUND);

  AVSadd_parameter("Uniform",      "boolean", 0, 0, 1 );

  AVSadd_parameter("Block Colors", "boolean", 0, 0, 1 );

  param = AVSadd_parameter("tic interval", "integer", 10, 1, 1000 );
  AVSconnect_widget( param, "typein_integer");

  param = AVSadd_parameter("label interval", "integer", 50, 1, 1000 );
  AVSconnect_widget( param, "typein_integer");

  param = AVSadd_parameter("draw ticmarks", "boolean", 1, 0, 1 );

  param = AVSadd_parameter("draw labels",   "boolean", 1, 0, 1 );

  param =AVSadd_parameter("Mode_Label","string","Display Mode","Display Mode","");
  AVSconnect_widget( param, "text" );

  AVSadd_parameter("Mode","choice",
     "none", "none#checker#scalar field", "#" );

  param =AVSadd_parameter("Color_Label","string","Color Variable","Color Variable","");
  AVSconnect_widget( param, "text" );
  AVSadd_parameter("Color_Vector","choice",
     "dat0", "dat0#dat1#dat2#dat3#dat4#dat5#dat6#dat7#dat8#dat9",
     "#" );

  param =AVSadd_parameter("Width_Label","string","Width Variable","Width Variable","");
  AVSconnect_widget( param, "text" );
  AVSadd_parameter("Width_Vector","choice",
     "dat0", "dat0#dat1#dat2#dat3#dat4#dat5#dat6#dat7#dat8#dat9",
     "#" );

  AVSadd_parameter("On / Off",      "boolean", 1, 0, 1 );

  /* routine pointers */
  AVSset_compute_proc (scatter_ribbon);

}  /* end of description routine */

/* ***********************************************************************/
/* Initialization for modules contained in this file.                    */
/* ***********************************************************************/
#ifdef SEP_EXE
int ((*mod_list[])()) = {
        scatter_ribbon_desc,
};
#define NMODS (sizeof(mod_list) / sizeof(char *))

AVSinit_modules()
{
        AVSinit_from_module_list(mod_list, NMODS);
}
#endif


