/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
#ifdef SCCS
static char sccsid[]="@(#)ucd_planes.c 6.0  Jonathan Cox 29/10/93";
#endif
/*
 * FILE: ucd_planes.c
 *
 * MODULE
 *
 * ucd planes - investigation of UCD structures with planar probes
 *
 * DESCRIPTION
 * 
 * The UCD PLANES module supports the interactive sub-sampling of node data
 * held  within  a  3D  UCD  structure.  The  region  sampled  is  selected
 * interactively in the region  of  a  transparent 'probe' object.  Several
 * different types of probes are supported, all  based upon a collection of
 * planes.  For each probe type, the sampled region can be output either as
 * various  forms  of  geometry, as a  UCD  structure or  as  both.  When a
 * geometry  is output, 2D  contours of  a  scalar  data  component can  be
 * superimposed on the sample object.
 * 
 * The selection of probe type is made  from the 'Probe Type' choice widget
 * on the module's control panel.  Possible  probes include  a single plane
 * (SIMPLE), two and three  crossed orthogonal planes  (SHEAF  and CROSSED)
 * and multiple  parallel  planes (MULTIPLE).  These probe types produce  a
 * 'sample' that is  the  same  shape as the probe  object  but is coloured
 * according to a scalar component of the structure's nodal data.
 * 
 * In addition,  a further probe type (PROJECTED) is supported which uses a
 * probe  consisting of a single plane just as with the SIMPLE plane probe.
 * However, the  sample produced  is not  a simple  coloured display of the
 * slice plane.   Instead of this, another component of nodal  data is used
 * to add height  to the sample.  This forms  a 3D graph of the data on the
 * sampled  plane.   Axes,  tick  marks  and  labels  can  be automatically
 * generated for  the graph.  The number of tick marks is controllable.  It
 * is also  possible to add a  flat wire-frame  rendering of the sliced UCD
 * mesh  to the base of the  graph.  This makes the perception  of  surface
 * height clearer.
 * 
 * For all probe types the  orientation and position  of the probe  can  be
 * controlled in two ways.   The first method is to use the dial widgets on
 * the module's control  panel.   The  XRotation, YRotation  and  ZRotation
 * parameters control the orientation of the plane as three rotations about
 * the major axes.  The rotations are applied in the order ZYX.   The Scale
 * parameter specifies a uniform scaling factor.  The Distance parameter is
 * a positional offset.  Due to the varying make up of  the different types
 * of  probes, some  parameters are of limited  use with some probe  types.
 * This is described in more detail below.
 * 
 * The alternative method  of controlling the  orientation  and position of
 * the  probe is by  direct  manipulation  of  the probe  object within the
 * geometry viewer.   During direct  manipulation the  dial  parameters are
 * calculated  from  the new  transformation  matrix for the probe  object.
 * This ensures that  the dial parameters and  the probe's orientation  are
 * always mutually consistent.
 * 
 * The  correspondence  between  transformation by the dials and  by direct
 * manipulation is not 1 to 1.  This is because, by default, it is possible
 * to translate objects in the geometry viewer in 3D where as there is only
 * one dial controlling position of the probe - the Distance dial.
 * 
 * This  restriction is fine for  those probe types  (SIMPLE, MULTIPLE  and
 * PROJECTED) in which all constituent  planes  are parallel to each other.
 * As translation parallel to the planes is irrelevant to the definition of
 * the probe, translation can be  restricted to being in  the  direction of
 * the planes' normals without any loss of generality.  In  fact,  this  is
 * desirable as it ensures that the normals always pass through  the centre
 * of  the UCD structure.  With this in mind, translations in the  geometry
 * viewer  for  these probe types are  restricted to  the direction  of the
 * probe's  normal.   This  feature  can be turned  off with  a compilation
 * option as  it  does not work well  on AVS platforms which are not highly
 * interactive - such as an X terminal.
 * 
 * For  the  SHEAF  and  CROSSED  probe  types  translation  in  3D becomes
 * significant.   For  these probe  types 3D  positioning is  only possible
 * through direct manipulation of  the  probe  object within  the  geometry
 * viewer  window  or  from the  geometry viewer's  transformation  options
 * panel.
 * 
 * The module  also  supports the  generation of line  contours across  the
 * sample  surface.   This can be done with all probe types.  In most cases
 * the contours are superimposed  on  the sample object.   This can lead to
 * them being obscured.  The 'Offset' toggle allows the contour lines to be
 * offset slightly to the  front  and behind the sample  plane so that they
 * can  be seen clearly.  If the PROJECTED probe type is selected, contours
 * are superimposed on the  projected  surface unless the  'Base' toggle is
 * on.  In this case contours are added to the base object instead.
 * 
 * The  'Make  UCD'  toggle  causes a UCD_structure  to be  output  that is
 * representative  of the  sampled region.   The  structure  is composed of
 * UCD_TRIANGLE primitives.  The  nodal data vector has the same components
 * as  the input  structure.  Data  values  are  constructed  using  linear
 * interpolation.
 *
 * NOTICE
 *
 * (c) University of Manchester 1991, 1992, 1993.
 * Author:  Jonathan Cox.
 * Address: IT301, Department of Computer Science, University of Manchester,
 *          Oxford Road, Manchester.  M13 9PL.  England.
 * Email:   coxjp@cs.man.ac.uk
 *
 * No part of this software may be altered, incorporated in other programs or
 * given to third parties, except under the conditions of the license, if any,
 * with which it was distributed.  This notice may not be removed from the
 * source code with which it is associated.
 *
 * VERSION HISTORY
 *
 * 5.0 - Jonathan Cox.  First version for general release.
 * 5.1 - Jonathan Cox.  Fixed problem with dissapearing input ports.
 * 5.2 - Jonathan Cox.  Made modules cooperative and rentrant.
 * 5.3 - Jonathan Cox.  ANSI C.
 * 5.4 - Jonathan Cox.  Added code for the generation of contours, 2D UCD,
 *                      and default layout.  Code reorganisation.
 * 5.5 - Jonathan Cox.  Removed 'Scale Info' input port.  Log scale now
 *                      specified by component label and units.
 * 5.6 - Jonathan Cox.  Default layout now includes stack.
 * 5.7 - Jonathan Cox.  Fixed memory allocation and axes labelling bugs.
 * 6.0 - Jonathan Cox.  Second version for general release.
 *
 */

/*
 * Include Files.
 */

#include <stdio.h>
#include <stdlib.h>
/* IAC CODE CHANGE : #include <math.h> */
/* IAC CODE CHANGE : #include <avs/avs_math.h> */
/* IAC CODE CHANGE : #include <avs/avs_math.h> */
#include <avs/avs_math.h>

#include <avs.h>
#include <port.h>
#include <field.h>
#include <geom.h>
#include <ucd_defs.h>
#include <udata.h>

#include "cut_ucd.h"
#include "utils.h"
#include "mat.h"


/*****************************************************************************
 *
 * Global Defaults.
 */

#ifdef UCD_MAX_COMPONENTS
#define MAX_COMPONENTS UCD_MAX_COMPONENTS
#else
#define MAX_COMPONENTS 50
#endif

#ifdef UCD_LABEL_LEN
#define LABEL_LEN UCD_LABEL_LEN
#else
#define LABEL_LEN 1024
#endif

/*
 * If the compilation macro INITIALISE_LAYOUT is defined the layout of the
 * modules widgets is predefined in the module description function.  This
 * causes module's widgets to be organised into three subpanels; the module has
 * too many parameters to fit comfortably onto one panel.
 */

#define INITIALISE_LAYOUT

/*
 * The module usually modifies the upstream probe transformation matrix it
 * receives so that a) the plane normal always passes through the centre of the
 * UCD structure and b) the distance parameter is kept constant during rotates
 * and scales.  If NORMAL_TRANSFORMS is defined these features are turned off.
 *
 * If REDIRECT is defined the transformation of the probe object is controlled
 * by this module rather than in the geometry viewer.  This makes it physically
 * impossible to translate the probe other than along the plane normal (thus
 * ensuring that the normal always passes through the center of the structure)
 * and secondly the centre of rotation is always the centre of the UCD
 * structure (thus preserving distance parameter during rotations and scales).
 * Unfortunately it also makes the position of the bounding box appear pretty
 * arbitrary.  Therefore, it is not particularly suitable for use on a non
 * responsive AVS platform such as an X terminal.  REDIRECT cannot be defined
 * if NORMAL_TRANSFORMS is.
 *
 * If neither REDIRECT nor NORMAL_TRANSFORMS is defined it will be possible to
 * transform the probe as normal but there will be a second transformation of
 * the probe, once the mouse button is released.  This ensures that the plane
 * normal passes through the centre and that the distance parameter is
 * preserved.
 *
 * CONTINUAL_UPDATE will cause the dial parameters to be continually updated
 * while the probe object is being transformed.  If, in addition,
 * CONTINUAL_SAMPLE is defined the sample will be continually regenerated while
 * the probe is being transformed.
 *
 * CONTINUAL_UPDATE and CONTINUAL_SAMPLE work best with either REDIRECT or
 * NORMAL_TRANSFORMS defined as otherwise the second plane movement and time
 * lag interferes with the transformation of the probe.
 */

#define REDIRECT

#define CONTINUAL_UPDATE
#define CONTINUAL_SAMPLE

#if defined(NORMAL_TRANSFORMS) && defined(REDIRECT)
#undef REDIRECT
#endif

#if defined(CONTINUAL_SAMPLE)
#define CONTINUAL_UPDATE
#endif

/*
 * If MID_EDGE_NODES is defined then they will be taken into consideration,
 * otherwise they are ignored.  Support for mid edge nodes is very simple.
 * Linear interpolation is used rather than a higher order interpolation scheme
 * and the algorithm will not detect edges that are intersected twice by the
 * plane.
 */
/* #define MID_EDGE_NODES */

/*
 * PRECISION is a small floating point value used to ensure that comparisons
 * for equality are interpreted correctly even if the values are slightly
 * inaccurate.
 */
#define PRECISION 1.0e-4

/*
 * CONTOUR_OFFSET and CONTOUR_DIFFUSE control the action of the 'offset'
 * toggle.  CONTOUR_OFFSET is a fraction of the Domain Size.  If the toggle is
 * on, contours are offset from each side of the sample plane by this fraction
 * of the plane's normal.  CONTOUR_DIFFUSE_DIM and CONTOUR_DIFFUSE_LIT are used
 * to set the diffuse lighting coefficient of the sample object.  Values of
 * -1.0 will disable this feature.
 */

#define CONTOUR_OFFSET 0.0022
#define CONTOUR_DIFFUSE_DIM 0.25
#define CONTOUR_DIFFUSE_LIT 0.80

/*
 * Default camera viewing position and scale for a projected plane
 */
#define PROJECTED_ZROT 80.0
#define PROJECTED_XROT -80.0
#define PROJECTED_VSCALE 0.7

/*
 * Default camera scale for viewing simple slice planes
 */
#define SLICE_VSCALE 0.9

/*
 * Default transparency alpha coefficient for the probe object
 */
#define ALPHA 0.3

/*
 * Default number of planes and ticks and maximum values
 */
#define DEFAULT_PLANES 5
#define MAX_PLANES 10
#define DEFAULT_TICKS 10
#define MAX_TICKS 20

/*
 * Offsets and sizes for various parts of the 3D graph
 */
#define Z_AXIS_OFFSET 0.0
#define XY_AXIS_OFFSET 0.0
#define AXIS_OFFSET 0.05
#define AXIS_LABEL_OFFSET 0.1
#define TICK_LABEL_OFFSET 0.03
#define AXES_LABEL_HEIGHT 0.05
#define TICK_LABEL_HEIGHT 0.04
#define TITLE_LABEL_HEIGHT 0.15
#define TICK_LENGTH 0.03

/*
 * If PARSE_ZDATA_LABELS is defined the ZData component's labels and units are
 * parsed to find information about any logarithmic data mapping.  If the label
 * starts with 'log' then it is assumed that the data has undergone a mapping
 * of the form: z -> log (z - inf + 1.0) and the inverse mapping is used to
 * correctly label the axes.  The lower bound 'inf' can be specified with a
 * clause 'inf=v' in the units string.  This clause will not be included in the
 * graph title.  If it is not found inf defaults to 0.
 */
#define PARSE_ZDATA_LABELS

/*****************************************************************************
 *
 * Global definitions and static read only (or write once) data.
 */

/*
 * The different types of probes.
 */
#define SIMPLE 1
#define CROSSED 2
#define SHEAF 3
#define MULTI 4
#define PROJECTED 5

static char Options[] = "Simple;Crossed;Sheaf;Multiple;Projected";

static float K[3] = { 0.0, 0.0, 0.1 };

/*****************************************************************************
 *
 * Type definitions.
 */

typedef struct _plane {
  int axis;			/* The axis which the plane cuts */
  float centre[3];		/* The plane's centre point */
} plane;


/*
 * Structure to hold information about nodes in the structure necessary for the
 * calculation of a cutting plane.
 */
typedef struct _node_info {
  float s;			/* Distance between node and plane */
  int flag;			/* Status flag */
  int targets[12];		/* Target node of edges from this node */
  int indices[12];		/* Indices of the vertices on these edges */
} node_info;


/*
 * Structure to hold the intersecting 'sample' plane as it is constructed.
 * Information contains: 'node_info' for each node - detailing which edges are
 * intersected, and a list of polygons which make up the sample - stored as a
 * list of indices indirectly referencing shared vertices, colours and normals.
 */
typedef struct _sample_s {
  node_info *ntable;		/* Table of information for nodes */
  int num_polygons;		/* Number of polygons in indices array */
  int *polygon_cells;		/* Input UCD cells that form polygons */
  int num_triangles;		/* Number of potential triangles in sample */
  int *next_index;		/* A pointer to the next slot in 'indices' */
  int *indices;			/* A list of polygon indices for the sample */
  int num_vertices;		/* The number of vertices in the sample */
  float (*vertices)[3];		/* sample's vertices' positions */
  float *data[MAX_COMPONENTS];	/* interpolated nodal data */
  float (*colours)[3];		/* interpolated colours */
  float (*normals)[3];		/* interpolated normals */
} sample_s;


/*
 * Stack structure to hold partially constructed contour segments.  These
 * polyline segments are generated by considering where contour values
 * intersect edges of a polygon in the sample.  The segments are held in the
 * stack until no more vertices can be added to them.  The segments are then
 * poped from the stack and added to the contours object.
 */
typedef struct _contour_stack {
  int contour;			/* an integer contour id */
  int nvert;			/* number of vertices in the polyline */
  float vertices[7][3];		/* vertices of the polyline segment */
  float colours[7][3];		/* interpolated colours for the polyline */
  struct _contour_stack *next;
} *contour_stack;


/*****************************************************************************
 *
 * Type definition for global read write static data (allocated in AVSstatic).
 */

typedef struct _static_data_s {
  int UCD_Stated;		/* Flag to indicate if UCD input checks out */

  /*
   * Description of the Domain - the 3D region covered by the UCD structure.
   */
  float Domain_Size;		/* Max dimension of the domain. */
  float Domain_Extent[6];	/* Size of the domain. */
  float Domain_Centre[3];	/* Centre of the domain. */

  /*
   * Definition of the Probe - the collection of planes used to sample the UCD
   * structure.  Although the probe's orientation can be arbitrary, at present
   * the probe can only contain planes that are orthogonal to a major axis.
   * However, these could easily be altered.
   */
  int Num_Planes;		/* The number of planes in the current probe */
  plane Planes[MAX_PLANES];	/* List of planes in the probe */

  /*
   * The transformation matrix for the Probe and its inverse.
   * (Established in get_probe_trans).
   */
  float Probe_Trans[4][4];
  float Inverse_Trans[4][4];

  /*
   * Description of the Nodal Data vector.
   */
  int Num_Components;
  int *Node_Components;
  float *Node_Data[MAX_COMPONENTS];
  float Min_Node_Data[MAX_COMPONENTS];
  float Max_Node_Data[MAX_COMPONENTS];

  /*
   * Descriptions of the ZData - this is the component of nodal data to be
   * projected in a projected plane probe.
   */
  int ZData;			/* component of node data */
  float ZData_Min;		/* Bracketing values for the data. */
  float ZData_Max;
  char ZData_Label[LABEL_LEN];	/* Label for the data component. */
  char ZData_Units[LABEL_LEN];	/* Units for the data component. */
  int ZData_Log_Scale;		/* True if ZData is in a log scale. */
  float ZData_Lower_Limit;	/* Lower limit for log scale. */
  float ZData_Scale;		/* Makes range same size as domain. */

  /*
   * Description of the CData - this is the component of nodal data that is to
   * be used to generate contours.
   */
  int CData;			/* component of node data. */
  float CData_Min;		/* Bracketing values for the data. */
  float CData_Max;
  float Lo_Contour_Val;		/* Range of contour values. */
  float Hi_Contour_Val;

  /*
   * Description of the Sample - this is the geometric structure created by
   * sampling (ie. slicing) the UCD structure in the domain of the Probe.
   */
  sample_s *Sample;		/* The sample */
  float Sample_Obj_Extent[6];	/* Extent of the sample object */
  float Sample_Centre[3];	/* Centre of the sample object */
  float Sample_Extent[6];	/* Extent of the sample inc. framework */
  int Axes_Present;		/* True if the axes are currently displayed. */
  int Base_Present;		/* True if the base is currently displayed. */
  int Contours_Present;		/* True if contours are currently displayed. */

} static_data_s;

extern char *AVSstatic;


/****************************************************************************
 *
 * Subroutines to control the setting of viewing matrices.
 *
 */

set_no_trans(elist, name)
GEOMedit_list elist; char *name;
/*
 * Resets the given object's transformation matrix
 */
{
  float matrix[4][4];

  mat_identity(matrix);
  GEOMedit_set_matrix(elist, name, (float *)matrix);
}


set_slice_view_trans(elist)
GEOMedit_list elist;
/*
 * Sets the camera's transformation matrix to the sensible value for the
 * viewing of a simple slice plane.
 */
{
  float matrix[4][4];

  mat_scale(matrix, SLICE_VSCALE, SLICE_VSCALE, SLICE_VSCALE);
  GEOMedit_set_matrix(elist, "camera1", (float *)matrix);
}


set_projected_view_trans(elist)
GEOMedit_list elist;
/*
 * Sets the camera's transformation matrix to a sensible value for the viewing
 * of projected planes
 */
{
  float temp[4][4], matrix[4][4];

  mat_z_rotate(PROJECTED_ZROT, matrix);
  mat_x_rotate(PROJECTED_XROT, temp);
  mat_multiply(matrix, temp, matrix);
  mat_scale(temp, PROJECTED_VSCALE, PROJECTED_VSCALE, PROJECTED_VSCALE);
  mat_multiply(matrix, temp, matrix);
  GEOMedit_set_matrix(elist, "camera1", (float *)matrix);
}


set_default_view_trans(S, elist, probe_type,
		       toggle)
static_data_s *S; GEOMedit_list elist; int probe_type;
		       int toggle;
{
  if (probe_type == SIMPLE && !toggle)
    {
      set_no_trans(elist, "Sample");
      GEOMedit_window(elist, "Sample", S->Sample_Extent);
      set_slice_view_trans(elist);
    }
  else if (probe_type == PROJECTED)
    {
      set_no_trans(elist, "Sample");
      GEOMedit_window(elist, "Sample", S->Sample_Extent);
      set_projected_view_trans(elist);
    }
  else
    GEOMedit_window(elist, "Sample", S->Sample_Extent);
}


/**************************************************************************
 *
 * Code to generate a transparent probe, in the correct position and
 * orientation on the first geometry output port.
 *
 */

get_planes_probe_trans(S, 
		       trans_info, xrot, yrot, zrot, 
                       scale, dist, centre, redirect, 
                       matrix, position)
		       static_data_s *S; upstream_transform
		       *trans_info; float xrot; float yrot; float zrot; float
                       scale; float dist; int centre; int redirect; float
                       matrix[4][4]; float position[3];
/*
 * Calculates the Probe_Trans matrix either from the upstream transformation
 * matrix or the dial widgets.  In the case of the upstream transformation
 * matrix the widgets values are modified to reflect the new transformation
 * matrix.  Otherwise the Probe_Trans is formed according to the parameter
 * values and the transformation matrix and the probe object's transformation
 * matrix is reset according to 'matrix' and 'position' which are also
 * calculated by this function.  If 'centre' is true the translation component
 * of the upstream matrix is reset so that the plane normal always passes
 * through the centre of the ucd structure.  Depending on whether 'redirect' is
 * true this can cause a second transformation of the probe object in the
 * geometry viewer once the mouse button is released.
 */
{
  int i;
  float theta, phi, psi, d, s;
  float v[3], n[3], diff[3];
  float m1[4][4], m2[4][4];
  float phi_sgn, angle;

  if (trans_info)
    {
      /*
       * calculate and update the transformation parameters from the new
       * transformation matrix.
       */

      mat_copy(S->Probe_Trans, trans_info->msxform);
      mat_copy(matrix, trans_info->msxform);

      /*
       * In order to decode the transformation matrix into parameters we assume
       * that the transformation matrix was constructed from a series of
       * transformations: translate by -c, scale by s, rotate in z by psi,
       * rotate in y by phi, rotate in x by theta and translate by c + d.n;
       * where c is the centre of rotation; s is a scaling vector; theta, phi
       * and psi are angles in degrees; n is the normalised normal to the
       * plane; and d is an offset distance along the normal.
       *
       * Note that the order of rotations is opposite to the order of rotations
       * used in AVS's transformations options panel.  This applies the
       * separate rotations in the order X Y Z.  However, rotating about Z
       * first makes more sense in this case as, as the probe plane is z=0, the
       * xrot and yrot parameters then control the orientation of the probe and
       * the zrot parameter controls the angle of rotation of the probe about
       * the plane normal.  Hence the user can usually disregard the zrot
       * parameter as it does not effect the definition of the plane.  It's
       * only use is to control the orientation of the sample with respect to
       * the plane definition.  If the user wishes to define the plane in the
       * normal AVS way, the transformation options panel can still be used as
       * normal.  If this is done the parameters will be calculated and updated
       * to reflect an equivalent z, y, x rotation order.
       */

      /*
       * The normal is the transformation of the vector k [0, 0, 1], ie. the
       * third column of the transformation matrix.
       */
      vec_copy(n, S->Probe_Trans[2]);
      normalize_vec(n);

      if (! centre)
	{
	  /*
	   * 'd' can be found by taking the difference between the dot product
	   * of any point on the transformed plane (eg. the transformation of
	   * the centre including any translations) with the plane normal, and
	   * the dot product of the centre of rotation with the normal.  To
	   * avoid trouble with floating point inaccuracies, some rounding is
	   * done.
	   */

	  vec_copy(v, S->Domain_Centre);
	  mat_vecmul(v, S->Probe_Trans);
	  d = 0.0;
	  for (i = 0; i < 3; i++)
	    d += (v[i] - S->Domain_Centre[i]) * n[i];
	  if (fabs(d) < PRECISION * S->Domain_Size) d = 0.0;
	}
      else
	{
	  /*
	   * The plane in the geometry viewer can be transformed away from the
	   * centre of the domain.  To ensure that the normal of the
	   * transformed plane always passes through the centre of the domain,
	   * we reset the translation vector of the Probe_Trans to c-v+dn where
	   * v is the transformation of c ignoring translations.  (In redirect
	   * mode c is the origin otherwise c is Domain_Centre).
	   */

	  vec_copy(v, S->Domain_Centre);
	  mat3_vecmul(v, S->Probe_Trans);

	  /*
	   * First task is to establish d.  If the transformation was just a
	   * rotation or scale it makes more sense to keep d constant.  (This
	   * case can be spotted as 'diff' the difference between the
	   * transformation of c by Probe_Trans and c itself will still be
	   * of magnitude 'dist'.
	   */

	  for (i = 0; i < 3; i++)
	    if (redirect)
	      diff[i] = S->Probe_Trans[3][i];
	    else
	      diff[i] = v[i] + S->Probe_Trans[3][i] -
		S->Domain_Centre[i];

	  if (fabs(vec_mag(diff) - dist) > PRECISION *
	      S->Domain_Size) 
	    {
	      d = 0.0;
	      for (i = 0; i < 3; i++)
		d += diff[i] * n[i];
	    }
	  else
	    d = dist;

	  if (fabs(d) < PRECISION * S->Domain_Size) d = 0.0;

	  for (i = 0; i < 3; i++)
	    {
	      position[i] = d * n[i];
	      matrix[3][i] = S->Probe_Trans[3][i] - diff[i];
	      S->Probe_Trans[3][i] = S->Domain_Centre[i] -
		v[i] + position[i];
	    }
	}

#ifndef CONTINUAL_UPDATE
      /*
       * If the trans_info has been generated mid way through a mouse
       * transformation the dial parameters are not updated.
       */
      if (trans_info->flags != BUTTON_UP) return;
#endif

      /*
       * The scale parameter can be obtained by taking any row or column vector
       * magnitude in the upper 3x3 matrix of the transformation matrix.  To
       * avoid rounding errors the value is snapped to 1.0 if it is close.
       */
      s = (float)vec_mag(S->Probe_Trans[0]);
      if (fabs(s - 1.0) < PRECISION) s = 1.0;

      /*
       * Theta (zrot), phi (yrot) and psi (zrot) are calculated below by
       * looking at the transformation various points under various
       * transformations.  However, the method falls apart in phi = 90 degrees
       * (when the plane normal is parallel to the x-axis) so this is dealt
       * with as a special case.
       */
      vec_copy(v, S->Probe_Trans[2]);
      if (v[1] == 0.0 && v[2] == 0.0)
	{
	  /*
	   * In the case that phi is +/- 90 degrees the net rotation is
	   * equivalent to a rotation of psi-/+theta in z followed by a +/-90
	   * degree rotation in y.  This angle can be found by taking the arc
	   * target of -y/-+z of the transformation of i.  Both y and z cannot
	   * be zero in this case as x is.
	   */
	  phi_sgn = v[0] >= 0.0 ? 1.0 : -1.0;
	  theta = 0.0;
	  phi = 90.0 * phi_sgn;
	  vec_copy(v, S->Probe_Trans[0]);
	  angle = atan2(- v[1], - phi_sgn * v[2]) / 3.14159265358979323846 *
	    180.0;

	  /*
	   * In order to prevent jumps in the dial widgets while rotating the
	   * probe plane in the geometry viewer the previous values for xrot
	   * and zrot are examined in order to make dial movement as smoothly
	   * as possible.
	   */
	  if (xrot == 0.0)
	    {
	      theta = 0.0;
	      psi = angle;
	    }
	  else if (zrot == 0.0)
	    {
	      psi = 0.0;
	      theta = - phi_sgn * angle;
	    }
	  else if (! different_quadrant(zrot - phi_sgn * xrot, angle))
	    {
	      theta = xrot;
	      psi = angle + phi_sgn * theta;
	    }
	  else
	    {
	      theta = 0.0;
	      psi = angle;
	    }
	}
      else
	{
	  /*
	   * Theta can be obtained by taking the arctan of -y/z where y and z
	   * are the respective coordinates of v, the transformation (ignoring
	   * translations) of the vector k.  Theta is restricted to lie in the
	   * range (-90,90] degrees.  This is to ensure that theta, phi and psi
	   * are unambiguous (see below).
	   */
	  theta = (v[2] != 0.0) ? atan( -v[1]/v[2] ) / 3.14159265358979323846 *
	    180.0 : 90.0;

	  /*
	   * Once theta is known we can rotate v by - theta and take the arc
	   * tangent of x/z to get phi.  Arctan2 is used to obtain the correct
	   * quadrant for theta.  Note that atan2 is total except for the case
	   * where both arguments are zero.  In this case z and x cannot both
	   * be zero as y is.
	   */
	  mat_x_rotate(-theta, m1);
	  mat3_vecmul(v, m1);
	  phi = atan2(v[0], v[2]) / 3.14159265358979323846 * 180.0;

	  /*
	   * Psi can be obtained by taking the arc tangent of -y/x where x and
	   * y are the respective components of the transformation of the
	   * vector i ([1, 0, 0]) rotated by -theta in x and -phi in y.  Note
	   * that both y and x cannot be zero as z is.
	   */
	  vec_copy(v, S->Probe_Trans[0]);
	  mat3_vecmul(v, m1);
	  mat_y_rotate(-phi, m2);
	  mat3_vecmul(v, m2);
	  psi = atan2(-v[1], v[0]) / 3.14159265358979323846 * 180.0;

	  /*
	   * Theta, phi and psi are not the only specification of the plane
	   * possible.  This is because for every theta, phi and psi there is
	   * also 180-theta, 180-phi and 180-psi which perform the identical
	   * transformation.  Rather than making the specification unambiguous
	   * (by restricting the range of one of them to +-90 degrees) theta,
	   * phi and psi are chosen so that at least two of them are in the
	   * same quadrant as the previous values xrot, yrot and zrot.
	   */
	  if (different_quadrant(theta, xrot) +
	      different_quadrant(phi, yrot) +
	      different_quadrant(psi, zrot) > 1)
	    {
	      theta = rot180(theta);
	      phi = rot180(phi);
	      psi = rot180(psi);
	    }

	  /*
	   * However, so that reset works correctly, if none of theta, phi and
	   * psi are in the quadrant as 0.0 then the other set of angles are
	   * taken.
	   */
	  if (different_quadrant(theta, 0.0) &&
	      different_quadrant(phi, 0.0) &&
	      different_quadrant(psi, 0.0))
	    {
	      theta = rot180(theta);
	      phi = rot180(phi);
	      psi = rot180(psi);
	    }
	}

      /*
       * In order to avoid floating point rounding errors the dial rotation
       * values are set to the nearest degree.  However, the slice plane is
       * calculated to full accuracy.
       */
      theta = floor(theta + 0.5);
      phi = floor(phi + 0.5);
      psi = floor(psi + 0.5);

      /*
       * Update the parameters to reflect the calculated parameters.
       */
      AVSmodify_float_parameter("XRotation", AVS_VALUE, theta, 0.0, 0.0);
      AVSmodify_float_parameter("YRotation", AVS_VALUE, phi, 0.0, 0.0);
      AVSmodify_float_parameter("ZRotation", AVS_VALUE, psi, 0.0, 0.0);
      AVSmodify_float_parameter("Scale", AVS_VALUE, s, 0.0, 0.0);
      AVSmodify_float_parameter("Distance", AVS_VALUE, d, 0.0, 0.0);
    }
  else
    {
      /*
       * Calculate the probe's transformation matrix from the module
       * parameters.  Matrix is formed from a series of simple transformations.
       * First the Domain_Centre is translated to the origin.
       */
      mat_translate(S->Probe_Trans, - S->Domain_Centre[0], -
		    S->Domain_Centre[1], - S->Domain_Centre[2]);

      /*
       * Then the probe is scaled and rotated about the z, y and x axes.
       */
      mat_scale(m1, scale, scale, scale);
      mat_multiply(S->Probe_Trans, m1, S->Probe_Trans);
      mat_z_rotate(zrot, m1);
      mat_multiply(S->Probe_Trans, m1, S->Probe_Trans);
      mat_y_rotate(yrot, m1);
      mat_multiply(S->Probe_Trans, m1, S->Probe_Trans);
      mat_x_rotate(xrot, m1);
      mat_multiply(S->Probe_Trans, m1, S->Probe_Trans);

      /*
       * Then the origin is translated to the S->Domain_Centre.
       */
      for (i = 0; i < 3; i++)
	S->Probe_Trans[3][i] += S->Domain_Centre[i];

      /*
       * Copy Probe_Trans into 'matrix' which is used to set the Probes
       * transformation matrix.  If 'redirect' is true the centre of rotation
       * is the origin, so the translation component of the matrix is zeroed.
       */
      mat_copy(matrix, S->Probe_Trans);
      if (redirect)
	vec_copy(matrix[3], Origin);

      /*
       * Calculate the last translation component of the transformation.
       */
      d = dist / scale;
      for (i = 0; i < 3; i++)
	{
	  position[i] = d * S->Probe_Trans[2][i];
	  S->Probe_Trans[3][i] += position[i];
	}
    }
 
  /*
   * Calculate a matrix to map the sample plane back onto the
   * plane parallel to the z axis through the Domain_Centre.
   */
  mat_inverse(S->Inverse_Trans, S->Probe_Trans);
}


define_planes_probe(S, probe_type, num_planes)
static_data_s *S; int probe_type; int num_planes;
/*
 * Defines the position and orientation of the planes in this probe type.
 */
{
  int i;
  float c[3];

  /*
   * Fill in the global definition of the structure of the current probe
   */
  switch (probe_type)
    {
    case SIMPLE:
    case PROJECTED:
      /*
       * A single plane z=0 centred on the origin
       */
      S->Num_Planes = 1;
      S->Planes->axis = 2;
      vec_copy(S->Planes->centre, S->Domain_Centre);
      break;
    case CROSSED:
      /*
       * 3 orthogonal planes centred on the origin
       */
      S->Num_Planes = 3;
      for (i = 0; i < 3; i++)
	{
	  S->Planes[i].axis = i;
	  vec_copy(S->Planes[i].centre, S->Domain_Centre);
	}
      break;
    case SHEAF:
      /*
       * 2 orthogonal planes (x=0 and y=0) centred on the origin
       */
      S->Num_Planes = 2;
      for (i = 0; i < 2; i++)
	{
	  S->Planes[i].axis = i;
	  vec_copy(S->Planes[i].centre, S->Domain_Centre);
	}
      break;
    case MULTI:
      /*
       * Num_Planes planes parallel to z=0 centred on the origin
       */
      S->Num_Planes = num_planes;
      vec_copy(c, S->Domain_Centre);
      for (i = 0; i < S->Num_Planes; i++)
	{
	  S->Planes[i].axis = 2;
	  c[2] = S->Domain_Centre[2] + ((float)i / (float)S->Num_Planes - 0.5)
	    * S->Domain_Size;
	  vec_copy(S->Planes[i].centre, c);
	}
      break;
    }
}


create_planes_probe(S, elist, transform)
static_data_s *S; GEOMedit_list elist; int transform;
/*
 * Creates a probe object to replace the current probe object in elist.
 */
{
  int i;
  GEOMobj *obj;

  /*
   * Create the polytriangle probe object from the description above
   */
  obj = GEOMcreate_obj(GEOM_POLYTRI, NULL);

  /*
   * For each plane in the probe add a square to the object
   */
  for (i = 0; i < S->Num_Planes; i++)
    add_square(obj, S->Planes[i].centre, S->Domain_Size * 0.707,
	       S->Planes[i].axis, transform ? S->Probe_Trans : 0);

  /*
   * If there is only one probe plane add a second interior square.
   */
  if (S->Num_Planes == 1)
    add_square(obj, S->Planes->centre, S->Domain_Size * 0.5, S->Planes->axis,
	       transform ? S->Probe_Trans : 0);

  /*
   * Add object to the elist list and then free its storage.
   */
  GEOMedit_geometry(elist, "Probe", obj);
  GEOMdestroy_obj(obj);

  /*
   * Set the probe object's colour and rendering mode.
   */
  GEOMedit_properties(elist, "Probe", 1.0, 0.0, 0.0, 0.0, ALPHA, White);
  GEOMedit_render_mode(elist, "Probe", "flat");
}


update_planes_probe(S, elist, probe_type, 
                    count, trans_info, xrot, 
                    yrot, zrot, scale, dist, 
                    do_define_probe)

		    static_data_s *S; GEOMedit_list *elist; int probe_type; int
                    count; upstream_transform *trans_info; float xrot; float
                    yrot; float zrot; float scale; float dist; int
                    do_define_probe;
/*
 * Creates a geometry representing the probe in the correct orientation.
 */
{
  int i;
  int centre;
  int redirect;
  float extent[6];
  float matrix[4][4];
  float position[3];

#ifdef NORMAL_TRANSFORMS
  centre = 0;
  redirect = 0;
#else
  centre = (probe_type != SHEAF && probe_type != CROSSED);
#ifdef REDIRECT
  redirect = centre;
#endif
#endif

  get_planes_probe_trans(S, trans_info, xrot, yrot, zrot, scale, dist, centre,
			 redirect, matrix, position);

  if (!do_define_probe && trans_info && !centre)
    {
      AVSmark_output_unchanged("probe");
      return;
    }

  /*
   * If necessary redefine the probe object.
   */
  if (do_define_probe)
    define_planes_probe(S, probe_type, count);

  /*
   * Initialise the edit list to be passed to the geometry viewer.
   */
  *elist = GEOMinit_edit_list(*elist);

  /*
   * If necessary create the probe object.
   */
  if (do_define_probe || redirect)
    create_planes_probe(S, *elist, redirect);

  /*
   * Set the probe's matrix and position vector.
   */
  if (redirect)
    GEOMedit_transform_mode(*elist, "Probe", "redirect", BUTTON_DOWN );
  else
    GEOMedit_transform_mode(*elist, "Probe", "normal", 0);
  GEOMedit_set_matrix(*elist, "Probe", (float *)matrix);
  GEOMedit_position(*elist, "Probe", position);

  /*
   * Set the probe object's transformation mode.
   */
  if (redirect)
    GEOMedit_transform_mode(*elist, "Probe", "redirect", BUTTON_MOVING |
			    BUTTON_UP);
  else
    {
#ifdef CONTINUAL_UPDATE
      GEOMedit_transform_mode(*elist, "Probe", "notify", BUTTON_UP |
			      BUTTON_MOVING);
#else
      GEOMedit_transform_mode(*elist, "Probe", "notify", BUTTON_UP);
#endif
    }

  /*
   * Set the probe's centre of rotation.
   */
  GEOMedit_center(*elist, "Probe", redirect ? Origin : S->Domain_Centre);

  /*
   * Set the window to a root3/2 times the maximum extent of the domain.
   */
  for (i=0; i<3; i++)
    {
      extent[2*i] = S->Domain_Centre[i] - S->Domain_Size * 0.866;
      extent[2*i+1] = S->Domain_Centre[i] + S->Domain_Size * 0.866;
    }
  GEOMedit_window(*elist, "Probe", extent);
}


/****************************************************************************
 *
 * Code to generate a sample of the ucd structure, such as a sliced plane, in
 * the correct position and orientation on the second geometry output port.
 *
 */

node_info *preprocess(ucd_struct, nnodes,
                      eqn, dist)
UCD_structure *ucd_struct; int nnodes; float eqn[3]; float dist;
{
  int i;
  float *x, *y, *z;		/* Structure's node positions */
  node_info *ntable;		/* The created table of node information */
  node_info *table_ptr;		/* Pointer into the node table */

  /*
   * Query ucd structure.
   */
  UCDstructure_get_node_positions(ucd_struct, &x, &y, &z);

  /*
   * Allocate storage space for information about each node.
   */
  ntable = (node_info *)calloc(nnodes, sizeof(node_info));

  /*
   * Compute distance between plane and nodes and set some flags to speed
   * computation.  Flag is true if the node is in front of the plane.  List of
   * edges is initialised to empty.
   */
  table_ptr = ntable;
  for (i = 0; i < nnodes; i++)
    {
      table_ptr->s = (eqn[0]*x[i] + eqn[1]*y[i] + eqn[2]*z[i] - dist);
      table_ptr->flag = (table_ptr->s >= 0 ? 1 : 0);
      table_ptr->targets[0] = 0;
      table_ptr++;
    }

  return ntable;
}


add_vertex(S, n1, n2, n3, mid_edge_node,
	   ucd_struct, colour_field, 
	   project, map_to_z0, normal, sample)
           static_data_s *S; int n1; int n2; int n3; int mid_edge_node;
	   UCD_structure *ucd_struct; AVSfield_float *colour_field; int
	   project; int map_to_z0; float *normal; sample_s *sample;
/*
 * Add the single intersection point on the edge n1 n2 to the sample - if it
 * has not already been calculated.
 */
{
  int i, j, veclen;
  float w1, w2;			/* Interpolation weights */
  float *x, *y, *z;		/* Node positions */
  float *n1_data, *n2_data;	/* data values to be interpolated */
  float *data;			/* interpolated data */
  float *col1, *col2;		/* Two colours to interpolate between */
  int *targets, *indices;	/* Pointers into the node_info structure */
  float *v;

  targets = sample->ntable[n1].targets;
  indices = sample->ntable[n1].indices;

  /*
   * Check that the particular edge intersection has not already been
   * calculated.
   */
  while (*targets)
    {
      if (*targets == n2)
	{
	  *(sample->next_index)++ = *indices;
	  return;
	}
      targets++;
      indices++;
    };

  /*
   * The target node of the newly intersected edge is n2.
   */
  *targets++ = n2;
  *targets = 0;

#ifdef MID_EDGE_NODES
  if (int mid_edge_node)
    {
      /*
       * Very simple approximation for mid edge nodes.  Uses linear
       * interpolation with the mid edge node rather than the target node.
       * A better approximation for mid edge nodes would be to use a quadratic
       * and calculate the intersection of the quadratic and the plane.
       * However, even this will not work if the edges are not approximately
       * linear as the plane could intersect the edge twice.  To deal with this
       * properly would require significant alteration to the code.
       */
      if (sample->ntable[n3].flag == sample->ntable[n1].flag) n1 = n3;
      if (sample->ntable[n3].flag == sample->ntable[n2].flag) n2 = n3;
    }
#endif

  /*
   * Calculate the interpolation weights for calculating the point of
   * intersection.  (Uses straightforward linear interpolation).
   */
  w2 = sample->ntable[n1].s / (sample->ntable[n1].s - sample->ntable[n2].s);
  w1 = 1.0 - w2;

  /*
   * Calculate the position of the intersection with the edge.
   */
  UCDstructure_get_node_positions(ucd_struct, &x, &y, &z);
  v = sample->vertices[sample->num_vertices];
  v[0] = x[n1]*w1 + x[n2]*w2;
  v[1] = y[n1]*w1 + y[n2]*w2;
  v[2] = z[n1]*w1 + z[n2]*w2;

  /*
   * If necessary transform the point to a point on the plane parallel to z=0
   * passing through the centre of the domain.
   */
  if (map_to_z0)
    mat2_vecmul(v, S->Inverse_Trans);

  /*
   * Interpolate any nodal data to the samples vertices.
   */
  for (i = 0; i < S->Num_Components; i++)
    {
      veclen = S->Node_Components[i];
      n1_data = S->Node_Data[i] + veclen * n1;
      n2_data = S->Node_Data[i] + veclen * n2;
      data = sample->data[i] + veclen * sample->num_vertices;
      for (j = 0; j < veclen; j++)
	*data++ = *n1_data++ * w1 + *n2_data++ * w2;
    }

  /*
   * If there is data to be projected, set the z coordinate according to the
   * interpolated data values of the ZData.
   */
  if (project)
    v[2] = (sample->data[S->ZData][sample->num_vertices] - S->ZData_Min) *
      S->ZData_Scale;

  /*
   * Set the colour of the intersection vertex by interpolating from the
   * colours of the nodes at each end of the edge.
   */
  if (colour_field)
    {
      v = sample->colours[sample->num_vertices];
      col1 = I1DV(colour_field, n1);
      col2 = I1DV(colour_field, n2);
      for (i=0; i<3; i++)
	v[i] = col1[i]*w1 + col2[i]*w2;
    }

  /*
   * If the normal for the point of intersection is known, copy it into the
   * normals array.
   */
  if (normal)
    vec_copy(sample->normals[sample->num_vertices], normal);

  /*
   * Set the index for the newly intersected edge (from n1) and next index in
   * the sample's polygons array to point to this vertex.
   * NB. Vertices are numbered from 0 but indices are numbered from 1.
   */
  *indices = ++(sample->num_vertices);
  *(sample->next_index)++ = *indices;
}


add_cutting_polygon(S, cell, ucd_struct,
		    colour_field, project, map_to_z0,
		    normal, sample)
		    static_data_s *S; int cell; UCD_structure *ucd_struct;
		    AVSfield_float *colour_field; int project; int map_to_z0;
		    float *normal; sample_s *sample;
{
  register int i, bit, temp_index;
  int *node;			/* Temporary pointers */
  float *v0, *v1, *v2;
  node_info *nt;
  int nvert;			/* Number of vertices of the polygon */
  int *polygon_indices;		/* Pointer to the polygon's indices list */
  char elem_type[9];		/* Variables describing UCD cell info */
  int name, mat_type, cell_type;
  int mid_edge_flags, *node_list;
  register int s;		/* Cell's vertices bit map. One bit for each */
				/* node: 1->above, 0->below plane */
  int *polygon_nodes;		/* The current edge which is being cut */
  int n1, n2;			/* The two nodes on the edge being cut */
  cutting_info *stable;		/* A pointer to the current cutting case */

  /*
   * Find out what type of cell we are dealing with.
   */
  UCDcell_get_information(ucd_struct, cell, &name, elem_type, &mat_type,
			  &cell_type, &mid_edge_flags, &node_list);

  /*
   * Determine which cutting case we're in.  A cutting case is numbered
   * according to a bit map where each bit represents a node.  A 1 implies the
   * node is above the plane a zero that the node is below.
   */
  s = 0;
  bit = Ucd_Num_Nodes[cell_type];
  if (!bit) return;
  node = node_list + bit;
  nt = sample->ntable;
  while (bit--)
      s |= (nt[*--node].flag & 1) << bit;

  /*
   * Use the predefined tables to find the number of vertices the cutting
   * polygon has.  If cell is not cut (nvert=0) return.
   */
  stable = Ucd_Cutting_Info[cell_type] + s;
  nvert = stable->nvert;
  if (!nvert) return;

  /*
   * If the cell is cut store the number of vertices in the sample's indices
   * array.  The actual indices values are then listed immediately afterwards.
   * These indices are derived from the node triples in the cutting table.
   */
  *sample->next_index++ = nvert;
  polygon_indices = sample->next_index;
  polygon_nodes = stable->node_triples;

  /*
   * Record the original cell id of the cell.  Also update the number of
   * polygons and potential triangles contained in the sample.  This
   * information is used when generating UCD structures and contours.
   */
  sample->polygon_cells[sample->num_polygons++] = cell;
  sample->num_triangles += nvert - 2;

  /*
   * Each pair of nodes, as indicated in the 'cutting_info' table, are
   * interpolated between to produce one polygon vertex.
   */
  i = nvert;
  while (i--)
    {
      /*
       * Add the point of intersection of the edge and the plane to the sample
       */
      n1 = node_list[*polygon_nodes++];
      n2 = node_list[*polygon_nodes++];

#ifdef MID_EDGE_NODES
      n3 = node_list[*polygon_nodes++];
      if (n1 < n2)
	add_vertex(S, n1, n2, n3, (mid_edge_flags | 1), ucd_struct,
		   colour_field, project, map_to_z0, normal, sample);
      else
	add_vertex(S, n2, n1, n3, (mid_edge_flags | 1), ucd_struct,
		   colour_field, project, map_to_z0, normal, sample);
      mid_edge_flags >> 1;
#else
      polygon_nodes++;
      if (n1 < n2)
	add_vertex(S, n1, n2, 0, 0, ucd_struct, colour_field, project,
		   map_to_z0, normal, sample);
      else
	add_vertex(S, n2, n1, 0, 0, ucd_struct, colour_field, project,
		   map_to_z0, normal, sample);
#endif
    }

  /*
   * In order for the surface to be shaded correctly all polygons must be
   * ordered in the same direction.  This is most easily ensured by reversing
   * the order of the indices in those polygons in which the z component of the
   * normal is positive and so pointing away from the camera.
   */
  v0 = sample->vertices[polygon_indices[0]-1];
  v1 = sample->vertices[polygon_indices[1]-1];
  v2 = sample->vertices[polygon_indices[2]-1];
  if (((v0[0]-v1[0])*(v2[1]-v1[1])-(v0[1]-v1[1])*(v2[0]-v1[0])) > 0.0)
    for (i=0; i < (nvert >> 1); i++)
      {
	temp_index = polygon_indices[i];
	polygon_indices[i] = polygon_indices[nvert-1-i];
	polygon_indices[nvert-1-i] = temp_index;
      }
}


add_plane(S, nnodes, ncells, eqn, dist,
	  ucd_struct, colour_field, project,
	  map_to_z0, normal, sample)
	  static_data_s *S; int nnodes; int ncells; float eqn[3]; float dist;
	  UCD_structure *ucd_struct; AVSfield_float *colour_field; int project;
	  int map_to_z0; float *normal; sample_s *sample;
{
  int i;

  /*
   * The simple implementation adopted here is to simply test every cell in the
   * UCD structure.  A more efficient approach for larger structures would to
   * be to compute connectivity information and use it to follow the plane
   * through the structure.
   */
  sample->ntable = preprocess(ucd_struct, nnodes, eqn, dist);
  for (i=0; i<ncells; i++)
    add_cutting_polygon(S, i, ucd_struct, colour_field, project, map_to_z0,
			normal,	sample);

/* IAC CODE CHANGE :   free(sample->ntable); */

/* IAC CODE CHANGE :    free(sample->ntable); */

/* IAC CODE CHANGE :     free(sample->ntable); */
     free(sample->ntable);
}


sample_s *alloc_sample(S, nnodes, ncells, 
		       colour_field, project)
static_data_s *S; int nnodes; int ncells; AVSfield_float
		       *colour_field; int project;
/*
 * Allocate memory for the relevant parts of sample.
 */
{
  int i;
  sample_s *sample;

  sample = (sample_s *)malloc(sizeof(struct _sample_s));

  sample->num_polygons = 0;
  sample->polygon_cells = (int *)calloc(ncells*S->Num_Planes, sizeof(int));
  sample->num_triangles = 0;
  sample->num_vertices = 0;
  sample->indices = (int *)calloc(ncells*8*S->Num_Planes, sizeof(int));
  sample->next_index = sample->indices;
  sample->vertices = (float (*)[3])calloc(nnodes*S->Num_Planes,
					  3*sizeof(float));
  for (i = 0; i < S->Num_Components; i++)
    sample->data[i] = (float *)calloc(nnodes * S->Num_Planes *
				  S->Node_Components[i], sizeof(float));
  sample->colours = colour_field ? (float (*)[3])calloc(nnodes*S->Num_Planes,
						    3*sizeof(float)) : 0;
  sample->normals = project ? 0 : (float (*)[3])calloc(nnodes*S->Num_Planes,
						   3*sizeof(float));

  return sample;
}


free_sample(S, sample)
static_data_s *S; sample_s *sample;
/*
 * Deallocate memory for the sample from the static data.
 */
{
  int i;

  sample = S->Sample;

  if (sample)
    {

/* IAC CODE CHANGE :       free(sample->polygon_cells); */

/* IAC CODE CHANGE :        free(sample->polygon_cells); */

/* IAC CODE CHANGE :         free(sample->polygon_cells); */
         free(sample->polygon_cells);

/* IAC CODE CHANGE :       free(sample->indices); */

/* IAC CODE CHANGE :        free(sample->indices); */

/* IAC CODE CHANGE :         free(sample->indices); */
         free(sample->indices);

/* IAC CODE CHANGE :       free(sample->vertices); */

/* IAC CODE CHANGE :        free(sample->vertices); */

/* IAC CODE CHANGE :         free(sample->vertices); */
         free(sample->vertices);
      for (i = 0; i < S->Num_Components; i++)

/* IAC CODE CHANGE : 	free(sample->data[i]); */

/* IAC CODE CHANGE : 	 free(sample->data[i]); */

/* IAC CODE CHANGE : 	  free(sample->data[i]); */
	   free(sample->data[i]);

/* IAC CODE CHANGE :       if (sample->colours) free(sample->colours); */

/* IAC CODE CHANGE :       if (sample->colours)  free(sample->colours); */

/* IAC CODE CHANGE :       if (sample->colours)   free(sample->colours); */
      if (sample->colours)    free(sample->colours);

/* IAC CODE CHANGE :       if (sample->normals) free(sample->normals); */

/* IAC CODE CHANGE :       if (sample->normals)  free(sample->normals); */

/* IAC CODE CHANGE :       if (sample->normals)   free(sample->normals); */
      if (sample->normals)    free(sample->normals);
      

/* IAC CODE CHANGE :       free(sample); */

/* IAC CODE CHANGE :        free(sample); */

/* IAC CODE CHANGE :         free(sample); */
         free(sample);
    }

  S->Sample = 0;
}


sample_s *generate_sample(S, ucd_struct,
			  colour_field, project, 
			  map_to_z0)
static_data_s *S; UCD_structure *ucd_struct;
			  AVSfield_float *colour_field; int project; int
			  map_to_z0;
{
  sample_s *sample;
  register int i;
  float c[3];
  float eqn[3];			/* Equation of the plane */
  float dist;			/* and its distance from the Domain Centre*/
  float matrix[4][4];
  float normal[3];		/* Normal to the plane */
  char name[20];		/* Variables for reading the UCD structure */
  int data_veclen, name_flag;
  int ncells, cell_veclen, nnodes, node_veclen;
  int util_flag;

  if (project)
    map_to_z0 = 1;

  /*
   * Get the number of cells and nodes in the structure
   */
  UCDstructure_get_header(ucd_struct, name, &data_veclen, &name_flag, &ncells,
			  &cell_veclen, &nnodes, &node_veclen, &util_flag);

  /*
   * Initialise the sample_s of the sample
   */
  sample = alloc_sample(S, nnodes, ncells, colour_field, project);

  /*
   * For each plane in the sample
   */
  for (i = 0; i < S->Num_Planes; i++)
    {
      /*
       * Calculate the plane's equation.
       */
      vec_copy(eqn, S->Probe_Trans[S->Planes[i].axis]);
      vec_copy(c, S->Planes[i].centre);
      mat_vecmul(c, S->Probe_Trans);
      dist = dot_product(c, eqn);

      /*
       * If the sample if flat and so has a normal calculate it.
       */
      if (!project)
	if (map_to_z0)
	  {
	    vec_copy(normal, Origin);
	    normal[S->Planes[i].axis] = 1.0;
	  }
	else
	  vec_copy(normal, eqn);

      /*
       * Add the current plane to the sample.
       */
      add_plane(S, nnodes, ncells, eqn, dist, ucd_struct, colour_field,
		project, map_to_z0, project ? 0 : normal, sample);
    }

  /*
   * The list of indices should be null terminated.
   */
  *(sample->next_index) = 0;

  if (!sample->num_vertices)
    {
      free_sample(S, sample);
      sample = 0;
    }

  return sample;
}


get_sample_extent(S, vertices, num_vertices, 
		  probe_type)
static_data_s *S; float (*vertices)[3]; int num_vertices; int
		  probe_type;
/*
 * Calculate extent and centre for the sample object.
 */
{
  register i;
  register float *v, *e;

  v = (float *)vertices;
  e = S->Sample_Obj_Extent;
  i = 3;
  while (i--)
    {
      *e++ = *v;
      *e++ = *v++;
    }
  i = num_vertices;
  while (--i)
    {
      e = S->Sample_Obj_Extent;
      if (*v < *e) *e = *v;
      e++;
      if (*v > *e) *e = *v;
      v++;
      e++;
      if (*v < *e) *e = *v;
      e++;
      if (*v > *e) *e = *v;
      v++;
      e++;
      if (*v < *e) *e = *v;
      e++;
      if (*v > *e) *e = *v;
      e++;
      v++;
    }
  for (i = 0; i < 3; i++)
    S->Sample_Centre[i] =
      (S->Sample_Obj_Extent[(i<<1)+1] + S->Sample_Obj_Extent[i<<1]) / 2.0;

  if (probe_type == PROJECTED)
    {
      /*
       * Make the sample size in the vertical equal to the size of the domain.
       */
      S->Sample_Obj_Extent[4] = 0.0;
      S->Sample_Obj_Extent[5] = S->Domain_Size;
      S->Sample_Centre[2] = S->Domain_Size / 2.0;
    }

  for (i = 0; i < 6; i++)
    S->Sample_Extent[i] = S->Sample_Obj_Extent[i];
}


add_sample_object(S, elist, sample, 
		  probe_type)
static_data_s *S; GEOMedit_list elist; sample_s *sample; int
		  probe_type;
{
  GEOMobj *obj;

  /*
   * The sample is a polyhedron object.
   */
  get_sample_extent(S, sample->vertices, sample->num_vertices, probe_type);
  obj = GEOMcreate_obj(GEOM_POLYHEDRON, S->Sample_Obj_Extent);

  /*
   * Add the polygons in the sample to the sample object.
   */
  GEOMadd_polygons(obj, sample->indices, GEOM_COMPLEX, GEOM_COPY_DATA);
  GEOMadd_vertices(obj, (float *)sample->vertices, sample->num_vertices,
		   GEOM_COPY_DATA);

  /*
   * If there are colours add these as well.
   */
  if (sample->colours)
    GEOMadd_float_colors(obj, sample->colours, sample->num_vertices,
			 GEOM_COPY_DATA);

  /*
   * If the surface is not flat calculate the surface normals.  Otherwise
   * set the vertex normals to the normal for the plane.
   */
  if (sample->normals)
    GEOMadd_normals(obj, (float *)sample->normals, sample->num_vertices,
		    GEOM_COPY_DATA);
  else
    GEOMgen_normals(obj, 0);

  /*
   * Convert the polyhedron object into a polytriangle with both a surface
   * and a wireframe description. (So that the 'Outline Gourand' rendering
   * mode works.
   */
  GEOMcvt_polyh_to_polytri(obj, GEOM_SURFACE | GEOM_WIREFRAME);

  /*
   * Replace the sample object in the viewer with the new object.
   */
  GEOMedit_geometry(elist, "Sample", obj);
  GEOMdestroy_obj(obj);
  GEOMedit_center(elist, "Sample", S->Sample_Centre);
  GEOMedit_transform_mode(elist, "Sample", "normal", BUTTON_UP);
}


add_base_object(S, elist, sample)
static_data_s *S; GEOMedit_list elist; sample_s *sample;
{
  register int i;
  float base_pos;
  float extent[6], centre[3], *vertices, *v;
  GEOMobj *obj;

  /*
   * Flatten the sample in z.
   */
  base_pos = - Z_AXIS_OFFSET * S->Domain_Size;
  vertices = (float *)malloc(sizeof(float) * sample->num_vertices * 3);
  v = vertices;
  for (i = 0; i < sample->num_vertices; i++)
    {
      *v++ = sample->vertices[i][0];
      *v++ = sample->vertices[i][1];
      *v++ = base_pos;
    }
  for (i = 0; i < 4; i++)
    extent[i] = S->Sample_Obj_Extent[i];
  extent[4] = extent[5] = base_pos;
  for (i = 0; i < 2; i++)
    centre[i>>1] = S->Sample_Centre[i>>1];
  centre[2] = base_pos;

  /*
   * The base is a polyhedron object.
   */
  obj = GEOMcreate_obj(GEOM_POLYHEDRON, extent);
  GEOMadd_polygons(obj, sample->indices, GEOM_COMPLEX, GEOM_COPY_DATA);
  GEOMadd_vertices(obj, vertices, sample->num_vertices, GEOM_COPY_DATA);
  GEOMcvt_polyh_to_polytri(obj, GEOM_WIREFRAME);

/* IAC CODE CHANGE :   free(vertices); */

/* IAC CODE CHANGE :    free(vertices); */

/* IAC CODE CHANGE :     free(vertices); */
     free(vertices);

  /*
   * Replace the sample object in the viewer with the new object.
   */
  GEOMedit_geometry(elist, "Base", obj);
  GEOMdestroy_obj(obj);
  GEOMedit_center(elist, "Base", centre);
  GEOMedit_parent(elist, "Base", "Sample");
  GEOMedit_transform_mode(elist, "Base", "parent", BUTTON_UP);

  S->Base_Present = 1;
}


remove_base_object(S, elist)
static_data_s *S; GEOMedit_list elist;
{
  if (S->Base_Present)
    {
      GEOMedit_visibility(elist, "Base", GEOM_EDIT_DELETE);
      S->Base_Present = 0;
    }
}


int add_contour_segments(obj, stack_ptr, contour,
			 sample, offset, normal)
GEOMobj *obj; contour_stack *stack_ptr; int contour;
			 sample_s *sample; float offset; float *normal;
/*
 * Adds contour segments from the contour stack to the contours object.  All
 * contours down to a specific contour are added.
 */
{
  contour_stack stack;
  float (*vertices)[3];
  float (*colours)[3];
  int i, j, nvert;

  while((stack = *stack_ptr) && stack->contour != contour)
    {
      nvert = stack->nvert;
      vertices = stack->vertices;
      colours = stack->colours;

      if (nvert > 2)
	{
	  /*
	   * If the number of vertices in the contour segment > 2 close the
	   * polyline.
	   */
	  vec_copy(vertices[nvert], vertices[0]);
	  vec_copy(colours[nvert], colours[0]);
	  nvert++;
	}

      if (nvert > 1)
	{
	  /*
	   * Add the polyline to the contours object.  If the line is to be
	   * offset the polyline is duplicated and offset both infront and
	   * behind the sample plane along the plane's normal.
	   */
	  if (offset == 0.0)
	    GEOMadd_polyline(obj, (float *)vertices, sample->colours ?
			     (float *)colours : 0, nvert, GEOM_COPY_DATA);
	  else
	    {
	      for (j = 0; j < nvert; j++)
		for (i = 0; i < 3; i++)
		  vertices[j][i] += offset * normal[i];
	      GEOMadd_polyline(obj, (float *)vertices, sample->colours ?
			       (float *)colours : 0, nvert, GEOM_COPY_DATA);
	      offset *= - 2.0;
	      for (j = 0; j < nvert; j++)
		for (i = 0; i < 3; i++)
		  vertices[j][i] += offset * normal[i];
	      GEOMadd_polyline(obj, (float *)vertices, sample->colours ?
			       (float *)colours : 0, nvert, GEOM_COPY_DATA);
	    }
	}

      *stack_ptr = stack->next;

/* IAC CODE CHANGE :       free(stack); */

/* IAC CODE CHANGE :        free(stack); */

/* IAC CODE CHANGE :         free(stack); */
         free(stack);
    }
}


int add_contour_vertex(obj, stack_ptr, 
		       add_segments, contour, sample, 
		       vertex1, vertex2, weight, map_to_z0, 
		       offset, normal)
		       GEOMobj *obj; contour_stack *stack_ptr; int
		       *add_segments; int contour; sample_s *sample; int
		       vertex1; int vertex2; float weight; int map_to_z0; float
		       offset; float *normal;
/*
 * Adds a vertex of a particular contour to the stack of partically completed
 * contour segments.  This may result in completed segments being added to the
 * contours object.
 */
 {
  contour_stack stack;
  int i;
  float *vertices, *colours, *pvertices, *pcolours;

  /*
   * Pop those contour segments which are closed off by the current contour
   * value.  If this empties the stack then reinitialise add_segments in order
   * to start following edge direction changes again.
   */
  if (*add_segments)
    {
      add_contour_segments(obj, stack_ptr, contour, sample, offset, normal);
      if (*stack_ptr == 0)
	*add_segments = 0;
    }

  /*
   * If there is not a contour segment held on the stack for the current
   * contour create one.
   */
  stack = *stack_ptr;
  if (!stack || stack->contour != contour)
    {
      *stack_ptr = (contour_stack)malloc(sizeof(struct _contour_stack));
      (*stack_ptr)->next = stack;
      stack = *stack_ptr;
      stack->contour = contour;
      stack->nvert = 0;
    }

  /*
   * Find intersection point for the contour by interpolating from the edges'
   * vertices.
   */
  vertices = stack->vertices[stack->nvert];
  pvertices = sample->vertices[vertex1];
  for (i = 0; i < 3; i++)
    vertices[i] =  pvertices[i] * weight;

  weight = 1.0 - weight;
  pvertices = sample->vertices[vertex2];
  for (i = 0; i < 3; i++)
    vertices[i] +=  pvertices[i] * weight;
  /*
   * If required project plane to z=0.
   */
  if (map_to_z0)
    vertices[2] = 0.0;

  /*
   * If the intersection point is a duplicate return without incrementing the
   * vertex count.
   */
  if (stack->nvert &&
      vertices[-3] == vertices[0] &&
      vertices[-2] == vertices[1] &&
      vertices[-1] == vertices[2])
    return;

  /*
   * Interpolate color values from the edge ends.  Colouring information comes
   * from the colour_field input.
   */
  if (sample->colours)
    {
      weight = 1.0 - weight;
      colours = stack->colours[stack->nvert];
      pcolours = sample->colours[vertex1];
      for (i = 0; i < 3; i++)
	colours[i] = pcolours[i] * weight;

      weight = 1.0 - weight;
      pcolours = sample->colours[vertex2];
      for (i = 0; i < 3; i++)
	colours[i] += pcolours[i] * weight;
    }

  /*
   * Increment the vertex count for the polygon and return.
   */
  stack->nvert++;
  return;
}


add_contours_object(S, elist, sample,
                    contours, map_to_z0, do_offset, do_diffuse)
		    static_data_s *S; GEOMedit_list elist; sample_s *sample;
                    int contours; int map_to_z0; int do_offset; int do_diffuse;
{
  GEOMobj *obj;			/* the contours object */
  contour_stack stack;		/* stack of partial contour segments */
  int *indices, nvertices;	/* indices and vertices of current polygon */
  int vertex1, vertex2;
  float *contour_data;
  float value1, value2;
  float *normal;		/* polygon's normal */
  int direction, add_segments;	/* flags controlling generation of contours */
  float cstep, lo_val, hi_val;	/* the contour range */
  int first, last, nsteps;	/* contours crossed by the current edge */
  float weight, wstep;		/* interpolation weights */
  int i, j;
  float offset, diffuse;

  /*
   * Locate the data values for the component of nodal data.
   */

  if (S->CData < 0) return;
  contour_data = sample->data[S->CData];

  stack = 0;
  obj = GEOMcreate_obj(GEOM_POLYTRI, 0);

  /*
   * Initialise variables describing the contour range.  There are contours-1
   * intervals of cstep between lo_val and hi_val.
   */
  hi_val = S->Hi_Contour_Val;
  lo_val = S->Lo_Contour_Val;
  cstep = (hi_val - lo_val) / (contours - 1);

  /*
   * 'offset' is the fraction of the sample plane's normal added the contour
   * lines in order to separate them from the sample object.  'diffuse' is the 
   * diffuse lighting coefficient of the sample object when contours are being
   * superimposed on it.  These features aid contour visibility.
   */
  if (map_to_z0)
    do_offset = 0;
  offset = do_offset ? CONTOUR_OFFSET * S->Domain_Size : 0.0;
  diffuse = do_offset ? CONTOUR_DIFFUSE_DIM : CONTOUR_DIFFUSE_LIT;

  /*
   * The indices array is a null terminated list the polygon's indices.  The
   * entry for each polygon begins with the number of vertices and then lists
   * their indices.  The indices are the position of the vertices in the
   * vertices array numbered from 1 to num_vertices.
   */
  indices = sample->indices;
  while (*indices)
    {
      /*
       * For each polygon in the sample ... 
       *
       * Initialise variables.  'direction' is 1 or -1 depending on whether the
       * data values increase or decrease along the current edge.  If the
       * 'add_segments' flag is true contour segments held on the stack are
       * poped and added to the contours object before the next contour is
       * considered.  'normal' points to a normal vector for the sample
       * polygon.  This is used to offset the contours from the sample object.
       * If the sample object is projected, and the normals are not known, the
       * vector defaults to K the unit vector in Z.
       */
      nvertices = *indices++;
      direction = 0;
      add_segments = 0;
      normal = sample->normals ? sample->normals[indices[1] - 1] : K;
      
      /*
       * For each edge in the sample polygon ...
       */
      for (i = 0; i < nvertices; i++)
	{
	  /*
	   * Find the edges' two vertices and data values at these points.
	   */
	  vertex1 = indices[i] - 1;
	  vertex2 = indices[i == nvertices - 1 ? 0 : i + 1] - 1;

	  value1 = contour_data[vertex1];
	  value2 = contour_data[vertex2];

	  if (value1 < value2)
	    {
	      /*
	       * If data values are increasing along the edge ...
	       */
	      if (direction != 1)
		{
		  direction = 1;
		  /*
		   * You can only be sure that segments held on the stack are
		   * complete, and so should be added to the contours object,
		   * once they have been closed off by another contour value.
		   * This happens following alternate changes in the direction
		   * of data values between consecutive edges.
		   */
		  add_segments = stack  ? ! add_segments : 0;
		}

	      /*
	       * ignore edges that do not span contour values.
	       */
	      if (value2 < lo_val || hi_val < value1)
		continue;

	      /*
	       * establish the first contour crossing the edge.
	       */
	      first = 0;
	      if (value1 > lo_val)
		{
		  first = (int)((value1 - lo_val) / cstep) - 1;
		  while (first * cstep + lo_val < value1) first++;
		}

	      /*
	       * establish the last contour crossing the edge.
	       */
	      last = contours - 1;
	      if (value2 < hi_val)
		{
		  last = (int)((value2 - lo_val) / cstep) + 1;
		  while (last * cstep + lo_val > value2) last--;
		}

	      nsteps = last - first + 1;
	    }
	  else if (value1 > value2)
	    {
	      /*
	       * If data values are decreasing along the edge ...
	       */
	      if (direction != -1)
		{
		  direction = -1;
		  add_segments = stack ? ! add_segments : 0;
		}

	      /*
	       * ignore edges that do not span contour values.
	       */
	      if (value1 < lo_val || value2 > hi_val)
		continue;

	      /*
	       * establish the first contour crossing the edge.
	       */
	      first = contours - 1;
	      if (value1 < hi_val)
		{
		  first = (int)((value1 - lo_val) / cstep) + 1;
		  while (first * cstep + lo_val > value1) first--;
		}

	      /*
	       * establish the last contour crossing the edge.
	       */
	      last = 0;
	      if (value2 > lo_val)
		{
		  last = (int)((value2 - lo_val) / cstep) - 1;
		  while (last * cstep + lo_val < value2) last++;
		}

	      nsteps = first - last + 1;
	    }
	  else
	    {
	      /*
	       * If the data is stationary along the edge add the end points of
	       * the edge as vertices of the contour iff the data value of the
	       * edge is a contour value.
	       */
	      first = (int)((value1 - lo_val) / cstep);
	      if (first == (value1 - lo_val) / cstep)
		{
		  add_contour_vertex(obj, &stack, &add_segments, first, sample,
				     vertex1, vertex2, 0.0, map_to_z0, offset,
				     normal);
		  add_contour_vertex(obj, &stack, &add_segments, first, sample,
				     vertex1, vertex2, 1.0, map_to_z0, offset,
				     normal);
		}
	      continue;
	    }

	  /*
	   * Calculate initial and incremental weights for interpolating along
	   * the edge.
	   */
	  weight = (value2 - first * cstep - lo_val) / (value2 - value1);
	  wstep = cstep / (value2 - value1);

	  /*
	   * For each of the contour values crossed by the edge add a vertex to
	   * the partially completed contour segment (if any) held on the
	   * stack.
	   */
	  j = first;
	  while (nsteps--)
	    {
	      add_contour_vertex(obj, &stack, &add_segments, j, sample,
				 vertex1, vertex2, weight, map_to_z0, offset,
				 normal);
	      weight -= direction * wstep;
	      j += direction;
	    }
	}

      add_contour_segments(obj, &stack, -1, sample, offset, normal);

      indices += nvertices;
    }

  GEOMedit_geometry(elist, "Contours", obj);
  GEOMdestroy_obj(obj);
  GEOMedit_parent(elist, "Contours", "Sample");
  GEOMedit_transform_mode(elist, "Contours", "parent", BUTTON_UP);

  if (do_diffuse)
    GEOMedit_properties(elist, "Sample", -1.0, diffuse, -1.0, -1.0, -1.0, 0);
}


remove_contours_object(S, elist)
static_data_s *S; GEOMedit_list elist;
{
  if (S->Contours_Present)
    {
      GEOMedit_visibility(elist, "Contours", GEOM_EDIT_DELETE);
      S->Contours_Present = 0;
    }
}


add_framework_objects(S, elist, nticks, do_base)
		      static_data_s *S; GEOMedit_list elist; int nticks;
		      int do_base;
/*
 * Adds a title and axes with ticks and labels to the sample geometry.
 */
{
  int i;
  int zpow;			/* A multiplier for the vertical axes labels */
  float vertices[2][3];		/* Variables for adding lines and labels */
  float colours[2][3];
  float pos[3], offset[3];
  char tlabel[16];
  float z, zstep, zscale;	/* Variables for generating tick marks */
  float zstart, zfinish;
  float z_offset, xy_offset;	/* Various graph dimensions */
  float a_offset, la_offset;
  float lt_offset, t_length;
  float t_offset;
  float ae[6];			/* The extent of the axes */
  float le[6];			/* The extent of the labels */
  int font, label_flags;	/* Font numbers and label flags */
  GEOMobj *axes;		/* The objects to be created */
  GEOMobj *axes_labels;
  GEOMobj *tick_labels;
  GEOMobj *title;

  /*
   * Scale the offset relative to the Domain Size
   */
  z_offset = Z_AXIS_OFFSET * S->Domain_Size;
  xy_offset = XY_AXIS_OFFSET * S->Domain_Size;
  a_offset = AXIS_OFFSET * S->Domain_Size;
  la_offset = AXIS_LABEL_OFFSET * S->Domain_Size;
  t_length = TICK_LENGTH * S->Domain_Size;
  lt_offset = t_length + TICK_LABEL_OFFSET * S->Domain_Size;

  /*
   * Axes are offset from the sample below in xy by XY_AXIS_OFFSET and in z by
   * Z_AXIS_OFFSET.  All axes are offset above the sample by AXIS_OFFSET.
   */
  ae[0] = S->Sample_Obj_Extent[0] - xy_offset;
  ae[1] = S->Sample_Obj_Extent[1] + a_offset;
  ae[2] = S->Sample_Obj_Extent[2] - xy_offset;
  ae[3] = S->Sample_Obj_Extent[3] + a_offset;
  ae[4] = S->Sample_Obj_Extent[4] - z_offset;
  ae[5] = S->Sample_Obj_Extent[5] + a_offset;

  /*
   * Labels are offset from the axes below and above by TICK_LABEL_OFFSET and
   * AXIS_LABEL_OFFSET respectively.
   */
  le[0] = ae[0] - lt_offset;
  le[1] = ae[1] + la_offset;
  le[2] = ae[2] - lt_offset;
  le[3] = ae[3] + la_offset;
  le[4] = ae[4] - lt_offset;
  le[5] = ae[5] + la_offset;

  /*
   * All lines are white.
   */
  vec_copy(colours[0], White);
  vec_copy(colours[1], White);

  /*
   * Create the objects.
   */
  axes = GEOMcreate_obj(GEOM_POLYTRI, GEOM_NULL);
  axes_labels = GEOMcreate_obj(GEOM_LABEL, GEOM_NULL);
  tick_labels = GEOMcreate_obj(GEOM_LABEL, GEOM_NULL);
  title = GEOMcreate_obj(GEOM_LABEL, GEOM_NULL);

  /*
   * Create Major axes.
   */
  vertices[0][0] = ae[0];
  vertices[0][1] = ae[2];
  vertices[0][2] = ae[4];
  vertices[1][0] = ae[1];
  vertices[1][1] = ae[2];
  vertices[1][2] = ae[4];
  GEOMadd_disjoint_line(axes, (float *)vertices, (float *)colours, 2,
			GEOM_COPY_DATA);
  vertices[1][0] = ae[0];
  vertices[1][1] = ae[3];
  GEOMadd_disjoint_line(axes, (float *)vertices, (float *)colours, 2,
			GEOM_COPY_DATA);
  vertices[1][0] = ae[0];
  vertices[1][1] = ae[2];
  vertices[1][2] = ae[5];
  GEOMadd_disjoint_line(axes, (float *)vertices, (float *)colours, 2,
			GEOM_COPY_DATA);

  /*
   * Label X, Y and Z axes.
   */
  font = GEOMget_font_number("Helvetica", 0, 0);
  label_flags = GEOMcreate_label_flags(font, 0, 0, 0, GEOM_LABEL_CENTER, 0);

  vec_copy(offset, Origin);
  offset[1] = - AXES_LABEL_HEIGHT / 2.0;
  pos[0] = le[1];
  pos[1] = ae[2];
  pos[2] = ae[4];
  GEOMadd_label(axes_labels, "X", pos, offset, AXES_LABEL_HEIGHT, White,
		label_flags);
  pos[0] = ae[0];
  pos[1] = le[3];
  pos[2] = ae[4];
  GEOMadd_label(axes_labels, "Y", pos, offset, AXES_LABEL_HEIGHT, White,
		label_flags);
  pos[0] = ae[0];
  pos[1] = ae[2];
  pos[2] = le[5];
  GEOMadd_label(axes_labels, S->ZData_Label, pos, offset, AXES_LABEL_HEIGHT,
		White, label_flags);

  /*
   * Add ticks and labels on the vertical axis.
   */
  zpow = 0;
  if (nticks)
    {
      z = 0.0;
      zstep = S->Domain_Size / (float)nticks;
      if (S->ZData_Log_Scale == 0)
	{
	  zpow = (int) log10((S->ZData_Max - S->ZData_Min) / (float)nticks);
	  zscale = pow(0.1, (double)zpow);
	}
      else
	zscale = 1.0;
      zstart = S->ZData_Min * zscale;
      zfinish = S->ZData_Max * zscale;
      zscale /= S->ZData_Scale;
      vertices[0][0] = ae[0];
      vertices[0][1] = ae[2];
      vertices[1][0] = ae[0] - t_length;
      vertices[1][1] = ae[2] - t_length;
      t_offset = t_length + S->Domain_Size * TICK_LABEL_HEIGHT *
	(S->ZData_Log_Scale ? 2.0 : 1.5);
      pos[0] = ae[0] - t_offset;
      pos[1] = ae[2] - t_offset;
      vec_copy(offset, Origin);
      offset[1] = - TICK_LABEL_HEIGHT / 2.0;
      while (nticks--)
	{
	  vertices[0][2] = z;
	  vertices[1][2] = z;
	  pos[2] = z;
	  GEOMadd_disjoint_line(axes, (float *)vertices, (float *)colours, 2,
				GEOM_COPY_DATA);
	  if (S->ZData_Log_Scale)
	    sprintf(tlabel, "%.1e", pow(10.0, zstart + z * zscale) +
		    S->ZData_Lower_Limit - 1.0);
	  else
	    sprintf(tlabel, "%.2f", zstart + z * zscale);
	  GEOMadd_label(tick_labels, tlabel, pos, offset, TICK_LABEL_HEIGHT,
			White, label_flags);
	  z += zstep;
	}
      vertices[0][2] = S->Domain_Size;
      vertices[1][2] = S->Domain_Size;
      pos[2] = S->Domain_Size;
      GEOMadd_disjoint_line(axes, (float *)vertices, (float *)colours, 2,
			    GEOM_COPY_DATA);
      if (S->ZData_Log_Scale)
	sprintf(tlabel, "%.1e", pow(10.0, (double)zfinish) +
		S->ZData_Lower_Limit - 1.0);
      else
	sprintf(tlabel, "%.2f", zfinish);
      GEOMadd_label(tick_labels, tlabel, pos, offset, TICK_LABEL_HEIGHT, White,
		    label_flags);
    }

  /*
   * Add ticks and labels on the horizontal axes.
   */
  if (nticks && do_base)
    {
      vertices[0][1] = ae[2];
      vertices[0][2] = ae[4];
      vertices[1][1] = ae[2] - t_length;
      vertices[1][2] = ae[4] - t_length;
      vertices[0][0] = S->Sample_Obj_Extent[0];
      vertices[1][0] = S->Sample_Obj_Extent[0];
      GEOMadd_disjoint_line(axes, (float *)vertices, (float *)colours, 2,
			    GEOM_COPY_DATA);
      vertices[0][0] = S->Sample_Obj_Extent[1];
      vertices[1][0] = S->Sample_Obj_Extent[1];
      GEOMadd_disjoint_line(axes, (float *)vertices, (float *)colours, 2,
			    GEOM_COPY_DATA);
      vertices[0][0] = ae[0];
      vertices[1][0] = ae[0] - t_length;
      vertices[0][1] = S->Sample_Obj_Extent[2];
      vertices[1][1] = S->Sample_Obj_Extent[2];
      GEOMadd_disjoint_line(axes, (float *)vertices, (float *)colours, 2,
			    GEOM_COPY_DATA);
      vertices[0][1] = S->Sample_Obj_Extent[3];
      vertices[1][1] = S->Sample_Obj_Extent[3];
      GEOMadd_disjoint_line(axes, (float *)vertices, (float *)colours, 2,
			    GEOM_COPY_DATA);

      vec_copy(offset, Origin);
      offset[1] = - TICK_LABEL_HEIGHT / 2.0;
      pos[2] = le[4];
      pos[1] = le[2];
      pos[0] = S->Sample_Extent[0];
      sprintf(tlabel, "%.2g", S->Sample_Obj_Extent[0]);
      GEOMadd_label(tick_labels, tlabel, pos, offset, TICK_LABEL_HEIGHT, White,
		    label_flags);
      pos[0] = S->Sample_Extent[1];
      sprintf(tlabel, "%.2g", S->Sample_Obj_Extent[1]);
      GEOMadd_label(tick_labels, tlabel, pos, offset, TICK_LABEL_HEIGHT, White,
		    label_flags);
      pos[0] = le[0];
      pos[1] = S->Sample_Extent[2];
      sprintf(tlabel, "%.2g", S->Sample_Obj_Extent[2]);
      GEOMadd_label(tick_labels, tlabel, pos, offset, TICK_LABEL_HEIGHT, White,
		    label_flags);
      pos[1] = S->Sample_Extent[3];
      sprintf(tlabel, "%.2g", S->Sample_Obj_Extent[3]);
      GEOMadd_label(tick_labels, tlabel, pos, offset, TICK_LABEL_HEIGHT, White,
		    label_flags);
    }

  /*
   * Add a plot title.
   */
  font = GEOMget_font_number("Times", 0, 0);
  label_flags = GEOMcreate_label_flags(font, 1, 1, 0, GEOM_LABEL_CENTER, 0);
  if (zpow && *S->ZData_Units)
    sprintf(tlabel, "%s (in %sx1%se%d)", S->ZData_Label, S->ZData_Units,
	    zpow > 0 ? "+" : "-", abs(zpow));
  else if (zpow)
    sprintf(tlabel, "%s (in x1%se%d)", S->ZData_Label, zpow > 0 ? "+" : "-",
	    abs(zpow));
  else if (*S->ZData_Units)
    sprintf(tlabel, "%s (in %s)", S->ZData_Label, S->ZData_Units);
  else
    sprintf(tlabel, "%s", S->ZData_Label);
  pos[0]=0.0;
  pos[1]=1.0 - TITLE_LABEL_HEIGHT * 1.0;
  pos[2]=0.0;
  GEOMadd_label(title, tlabel, pos, Origin, TITLE_LABEL_HEIGHT, White,
		label_flags);

  /*
   * Add the newly defined objects into the geometry.
   */
  GEOMset_computed_extent(axes, ae);
  GEOMedit_geometry(elist, "Axes", axes);
  GEOMdestroy_obj(axes);
  GEOMedit_parent(elist, "Axes", "Sample");
  GEOMedit_transform_mode(elist, "Axes", "parent", BUTTON_UP);

  GEOMset_computed_extent(axes_labels, le);
  GEOMedit_geometry(elist, "Axes Labels", axes_labels);
  GEOMdestroy_obj(axes_labels);
  GEOMedit_parent(elist, "Axes Labels", "Sample");
  GEOMedit_transform_mode(elist, "Axes Labels", "parent", BUTTON_UP);

  GEOMset_computed_extent(tick_labels, le);
  GEOMedit_geometry(elist, "Tick Labels", tick_labels);
  GEOMdestroy_obj(tick_labels);
  GEOMedit_parent(elist, "Tick Labels", "Sample");
  GEOMedit_transform_mode(elist, "Tick Labels", "parent", BUTTON_UP);

  GEOMedit_geometry(elist, "Title", title);
  GEOMdestroy_obj(title);
  GEOMedit_parent(elist, "Title", "Sample");
  GEOMedit_transform_mode(elist, "Title", "parent", BUTTON_UP);

  /*
   * Extent for the sample are now the label extent.
   */
  for (i = 0; i < 6; i++)
    S->Sample_Extent[i] = le[i];

  S->Axes_Present = 1;
}


remove_framework_objects(S, elist)
static_data_s *S; GEOMedit_list elist;
{
  if (S->Axes_Present)
    {
      GEOMedit_visibility(elist, "Axes", GEOM_EDIT_DELETE);
      GEOMedit_visibility(elist, "Axes Labels", GEOM_EDIT_DELETE);
      GEOMedit_visibility(elist, "Tick Labels", GEOM_EDIT_DELETE);
      GEOMedit_visibility(elist, "Title", GEOM_EDIT_DELETE);
      S->Axes_Present = 0;
    }
}


UCD_structure *generate_ucd(S, ucd_out,
			    ucd_in, sample, map_to_z0)
			    static_data_s *S; UCD_structure *ucd_out;
			    UCD_structure *ucd_in; sample_s *sample; int
			    map_to_z0;
/*
 * Converts polygons in the sample to a 2D UCD structure containing triangles.
 */
{
  char str[100];		/* Variables to read names, labels etc. */
  char delimiter[2];
  int i, j;
  int data_veclen, name_flag;	/* Values to hold info about ucd_struct */
  int ncells, cell_veclen;
  int nnodes, node_veclen;
  int util_flag;
  float min_extent[3], max_extent[3];
  int cell_name, material_id;
  int cell_type, mid_edge_flags;
  float *from, *to;		/* vars for creating nodal data */
  int *active_list;
  float *nodal_data;
  int cellid;			/* vars for converting polygons to cells */
  int *node_list;
  int nvert, *pindices;
  int ntriangles, alternate;
  int ttopol[3], tindices[3];
  float *x, *y, *z;		/* nodes' coordinates */
  int comp_list[1];

  if (ucd_out)
    UCDstructure_free(ucd_out);

  /*
   * Query input ucd structure
   */
  UCDstructure_get_header(ucd_in, str, &data_veclen, &name_flag,
			  &ncells, &cell_veclen, &nnodes, &node_veclen,
			  &util_flag);

  ncells = sample->num_triangles;
  nnodes = sample->num_vertices;

  UCDstructure_get_extent(ucd_in, min_extent, max_extent);

  /*
   * Allocate new output structure.
   */
  name_flag = (name_flag & UCD_MATERIAL_IDS) | UCD_NODE_NAMES | UCD_CELL_NAMES
    | UCD_INT;
  ucd_out = (UCD_structure *)UCDstructure_alloc(str, 0, name_flag, ncells, 3 *
						ncells, 0, nnodes, 0,
						node_veclen, 0);

  UCDstructure_set_extent(ucd_out, min_extent, max_extent);


  /*
   * Store information about the nodes (the vertices fo the sample).
   */

  for (i = 0; i < nnodes; i++)
    UCDnode_set_information(ucd_out, i, (int *)(i + 1), 0, 0);

  x = (float *)calloc(nnodes, sizeof(float));
  y = (float *)calloc(nnodes, sizeof(float));
  z = (float *)calloc(nnodes, sizeof(float));
  for (i = 0; i < nnodes; i++)
    {
      x[i] = sample->vertices[i][0];
      y[i] = sample->vertices[i][1];
    }
  if (!map_to_z0)
    for (i = 0; i < nnodes; i++)
      z[i] = sample->vertices[i][2];
  UCDstructure_set_node_positions(ucd_out, x, y, z);

/* IAC CODE CHANGE :   free(x); */

/* IAC CODE CHANGE :    free(x); */

/* IAC CODE CHANGE :     free(x); */
     free(x);

/* IAC CODE CHANGE :   free(y); */

/* IAC CODE CHANGE :    free(y); */

/* IAC CODE CHANGE :     free(y); */
     free(y);

/* IAC CODE CHANGE :   free(z); */

/* IAC CODE CHANGE :    free(z); */

/* IAC CODE CHANGE :     free(z); */
     free(z);
  
  if (node_veclen)
    {
      nodal_data = (float *)malloc(node_veclen * nnodes * sizeof(float));
      to = nodal_data;
      for (i = 0; i < S->Num_Components; i++)
	{
	  from = sample->data[i];
	  j = S->Node_Components[i] * nnodes;
	  while (j--)
	    *to++ = *from++;
	}
      UCDstructure_set_node_components(ucd_out, S->Node_Components, i);
      UCDstructure_set_node_data(ucd_out, nodal_data);
      UCDstructure_set_node_minmax(ucd_out, S->Min_Node_Data,
				   S->Max_Node_Data);

/* IAC CODE CHANGE :       free(nodal_data); */

/* IAC CODE CHANGE :        free(nodal_data); */

/* IAC CODE CHANGE :         free(nodal_data); */
         free(nodal_data);

      UCDstructure_get_node_active(ucd_in, &active_list);
      UCDstructure_set_node_active(ucd_out, active_list);
      UCDstructure_get_node_labels(ucd_in, str, delimiter);
      UCDstructure_set_node_labels(ucd_out, str, delimiter);
      UCDstructure_get_node_units(ucd_in, str, delimiter);
      UCDstructure_set_node_units(ucd_out, str, delimiter);
     }

  /*
   * Store information about the cells
   */

  cellid = 0;
  pindices = sample->indices;
  for (i = 0; i < sample->num_polygons; i++)
    {
      UCDcell_get_information(ucd_in, sample->polygon_cells[i], &cell_name,
			      str, &material_id, &cell_type, &mid_edge_flags,
			      &node_list);

      nvert = *pindices++;
      ttopol[0] = 0;
      ttopol[1] = 1;
      ttopol[2] = nvert - 1;

      for (j = 0; j < 3; j++)
	tindices[j] = pindices[ttopol[j]] - 1;

      UCDcell_set_information(ucd_out, cellid++, (int *)cellid, 0,
			      material_id, UCD_TRIANGLE, 0, tindices);

      ntriangles = nvert - 3;
      alternate = 1;
      while (ntriangles--)
	{
	  if (alternate)
	    {
	      ttopol[0] = ttopol[1];
	      ttopol[1]++;
	    }
	  else
	    {
	      ttopol[0] = ttopol[2];
	      ttopol[2]--;
	    }
	  alternate = ! alternate;

	  for (j = 0; j < 3; j++)
	    tindices[j] = pindices[ttopol[j]] - 1;
	  
	  UCDcell_set_information(ucd_out, cellid++, (int *)cellid, 0,
				  material_id, UCD_TRIANGLE, 0, tindices);
	}

      pindices += nvert;
    }

  return ucd_out;
}


  
/******************************************************************************
 *
 * Functions to decode inputs and parameter changes.
 *
 */

int modify_probe_type_parameters(probe_type, toggle, count)
int probe_type; int *toggle; int *count;
/*
 * Makes sure that the modules parameters which are dependent on probe type are
 * in a consistent state with the current probe_type.
 */
{
  AVSparameter_visible("ZData Lab", probe_type == PROJECTED);
  AVSparameter_visible("ZData Choice", probe_type == PROJECTED);

  if (probe_type == PROJECTED)
    {
      AVSmodify_parameter_prop("Count", "title", "string", "#Ticks");
      AVSmodify_parameter("Count", AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
			  DEFAULT_TICKS, 0, MAX_TICKS);
      *count = DEFAULT_TICKS;
      AVSparameter_visible("Count", 1);
      AVSmodify_parameter_prop("Toggle", "title", "string", "Base");
      AVSmodify_parameter("Toggle", AVS_VALUE , 0, 0, 0);
      *toggle = 0;
      AVSparameter_visible("Toggle", 1);
    }
  else if (probe_type == MULTI)
    {
      AVSmodify_parameter_prop("Count", "title", "string", "#Planes");
      AVSmodify_parameter("Count", AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
			  DEFAULT_PLANES, 2, MAX_PLANES);
      *count = DEFAULT_PLANES;
      AVSparameter_visible("Count", 1);
      AVSparameter_visible("Toggle", 0);
    }
  else if (probe_type == SIMPLE)
    {
      AVSmodify_parameter_prop("Toggle", "title", "string",
			       "Transform Sample");
      AVSmodify_parameter("Toggle", AVS_VALUE , 0, 0, 0);
      *toggle = 0;
      AVSparameter_visible("Toggle", 1);
      AVSparameter_visible("Count", 0);
    }
  else
    {
      AVSparameter_visible("Count", 0);
      AVSparameter_visible("Toggle", 0);
    }
}


int stat_ucd(S, ucd_struct)
static_data_s *S; UCD_structure *ucd_struct;
/*
 * Check that the ucd structure exists and if it has changed find its extent.
 */
{
  int ok, i;
  float dom_min[3], dom_max[3];	/* Variables to read domain extent */
  char str[LABEL_LEN];	/* Variables to read names, labels etc. */
  char label[LABEL_LEN];
  char delimiter[2];
  int data_veclen, name_flag;	/* Values to hold info about ucd_struct */
  int ncells, cell_veclen;
  int nnodes, node_veclen;
  int util_flag;
  float *nodal_data;

  S->UCD_Stated = 0;
  free_sample(S, S->Sample);
  S->Sample = 0;

  /*
   * Query ucd structure
   */
  ok = UCDstructure_get_header(ucd_struct, str, &data_veclen, &name_flag,
			       &ncells, &cell_veclen, &nnodes, &node_veclen,
			       &util_flag);
  if (!ok)
    {
      AVSerror("Error: Couldn't get structure information");
      return 0;
    }

  /*
   * Find the centre, extent and size of the domain.
   */
  ok = UCDstructure_get_extent(ucd_struct, dom_min, dom_max);
  if (!ok)
    {
      AVSerror("Error: Couldn't find domain's extent");
      return 0;
    }

  for (i = 0; i < 3; i++)
    S->Domain_Centre[i] = (dom_max[i] + dom_min[i])/2;
  S->Domain_Size = 0.0;
  for (i = 0; i < 3; i++)
    {
      S->Domain_Extent[2*i] = dom_min[i];
      S->Domain_Extent[2*i+1] = dom_max[i];
      if (S->Domain_Size < dom_max[i] - dom_min[i])
	S->Domain_Size = dom_max[i] - dom_min[i];
    }

  /*
   * Set Distance parameter limits to reflect domain size.
   */
  AVSmodify_float_parameter("Distance", AVS_MINVAL | AVS_MAXVAL, 0.0, -
			    S->Domain_Size, S->Domain_Size);
  
  /*
   * Make the component parameters reflect the choice of node data within
   * the UCD structure.
   */
  if (node_veclen)
    {
      UCDstructure_get_node_components(ucd_struct, &S->Node_Components);
      UCDstructure_get_node_minmax(ucd_struct, S->Min_Node_Data,
				   S->Max_Node_Data);
      UCDstructure_get_node_data(ucd_struct, &nodal_data);
      UCDstructure_get_node_labels(ucd_struct, str, delimiter);
      UCDstructure_get_node_label(ucd_struct, 0, label);
      AVSmodify_parameter("ZData Choice", AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
			  label, str, delimiter);
      AVSmodify_parameter("CData Choice", AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
			  label, str, delimiter);

      i = 0;
      while (S->Node_Components[i] && node_veclen > 0)
	{
	  S->Node_Data[i] = nodal_data;
	  nodal_data += S->Node_Components[i] * nnodes;
	  node_veclen -= S->Node_Components[i];
	  i++;
	}
      S->Num_Components = i;
    }
  else
    {
      AVSmodify_parameter("ZData Choice", AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
			  "<none>", "<none>", ";");
      AVSmodify_parameter("CData Choice", AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
			  "<none>", "<none>", ";");
      S->Num_Components = 0;
    }

  S->UCD_Stated = 1;

  return 1;
}


int stat_zdata(S, ucd_struct, component)
static_data_s *S; UCD_structure *ucd_struct; int component;
/*
 * Stat the component of nodal data to be used on the Z axis of a 3D graph.
 */
{
  int ok, i;
  int data_pos;
  char label[UCD_LABEL_LEN];

  /*
   * Wipe any previous data.
   */
  S->ZData = -1;

  /*
   * If the data component is scalar retrieve and store info about it.
   */
  if (S->Node_Components[component] == 1)
    {
      /*
       * Locate the actual array of data and its bounding values.
       */
      i = 0;
      data_pos = 0;
      while (i < component)
	{
	  data_pos += S->Node_Components[i];
	  i++;
	}
      S->ZData = component;
      S->ZData_Min = S->Min_Node_Data[data_pos];
      S->ZData_Max = S->Max_Node_Data[data_pos];
      
      /*
       * Work out a scaling value to make projected graphs look
       * approximately cubic.
       */
      S->ZData_Scale = S->Domain_Size / (S->ZData_Max - S->ZData_Min);
      
      /*
       * Find the label and units for the data component.
       */
      *S->ZData_Label = 0;
      UCDstructure_get_node_label(ucd_struct, component, S->ZData_Label);

#ifdef PARSE_ZDATA_LABELS
      S->ZData_Log_Scale = !strncmp(S->ZData_Label, "log ", 4) ? 1 : 0;
#else
      S->ZData_Log_Scale = 0;
#endif

      *label = 0;
      if (UCDstructure_get_node_unit(ucd_struct, component, label) && *label)
	{
	  if (!S->ZData_Log_Scale ||
	      sscanf(label, "%s inf=%g", S->ZData_Units,
		     &S->ZData_Lower_Limit) != 2)
	    {
	      strcpy(S->ZData_Units, label);
	      S->ZData_Lower_Limit = 0.0;
	    }
	}
      else
	{
	  *S->ZData_Units = 0;
	  S->ZData_Lower_Limit = 0.0;
	}
    }
  else
    AVSerror("Error: Selected ZAxis component is not a scalar.");

  return 1;
}


int stat_cdata(S, ucd_struct, 
	       contour_range, component, lo_val, hi_val)
	       static_data_s *S; UCD_structure *ucd_struct; ucd_legend_output
	       *contour_range; int component; float lo_val; float hi_val;
/*
 * Stat the component of Nodal Data to be used to generate contours.
 */
{
  int ok, i;
  int data_pos;

  /*
   * Wipe any previous CData in the Static Store.
   */
  S->CData = -1;

  /*
   * Find the data component and range of contours to be used.
   */

  if (contour_range && AVSinput_changed("contour range", 0))
    {
      component = AVSchoice_number("CData Choice", contour_range->val_name)-1;
      AVSmodify_parameter("CData Choice", AVS_VALUE, contour_range->val_name,
			  "", "");
      S->Lo_Contour_Val = contour_range->lo_val;
      S->Hi_Contour_Val = contour_range->hi_val;
    }
  else if (AVSparameter_changed("Hi Val") || AVSparameter_changed("Lo Val"))
    {
      if (lo_val < hi_val)
	{
	  S->Lo_Contour_Val = lo_val;
	  S->Hi_Contour_Val = hi_val;
	}
      else
	{
	  S->Lo_Contour_Val = hi_val;
	  S->Hi_Contour_Val = lo_val;
	}
    }


  /*
   * If the selected component is scalar then retrieve and store info about it.
   */
  if (S->Node_Components[component] == 1)
    {
      /*
       * Locate the actual array of data and its bounding values.
       */
      i = 0;
      data_pos = 0;
      while (i < component)
	{
	  data_pos += S->Node_Components[i];
	  i++;
	}
      S->CData = component;
      S->CData_Min = S->Min_Node_Data[data_pos];
      S->CData_Max = S->Max_Node_Data[data_pos];
      
      if (S->Lo_Contour_Val < S->CData_Min ||
	  S->Lo_Contour_Val > S->CData_Max ||
	  S->Hi_Contour_Val < S->CData_Min ||
	  S->Hi_Contour_Val > S->CData_Max)
	{
	  S->Lo_Contour_Val = S->CData_Min;
	  S->Hi_Contour_Val = S->CData_Max;
	}
      
      AVSmodify_float_parameter("Lo Val", AVS_VALUE | AVS_MINVAL |
				AVS_MAXVAL, S->Lo_Contour_Val,
				S->CData_Min, S->CData_Max);
      AVSmodify_float_parameter("Hi Val", AVS_VALUE | AVS_MINVAL |
				AVS_MAXVAL, S->Hi_Contour_Val,
				S->CData_Min, S->CData_Max);
    }
  else
    AVSerror("Error: Selected contour component is not a scalar.");

  return 1;
}



/*****************************************************************************
 *
 * The computation function of the module.
 */

int ucd_planes_compute(ucd_struct, colour_field,
                       contour_range, 
                       trans_info, probe_out, 
                       sample_out, ucd_out, make_sample,
                       make_ucd, xrot, yrot, zrot,
                       scale, dist, pctitle, 
                       probe_choice, zctitle, zdata_choice, 
                       toggle, count, contours, offset, 
                       lo_val, hi_val, cctitle, 
                       cdata_choice)
		       UCD_structure *ucd_struct; AVSfield_float *colour_field;
                       ucd_legend_output *contour_range; upstream_transform
                       *trans_info; GEOMedit_list *probe_out; GEOMedit_list
                       *sample_out; UCD_structure **ucd_out; int make_sample;
                       int make_ucd; float *xrot; float *yrot; float *zrot;
                       float *scale; float *dist; char *pctitle; char
                       *probe_choice; char *zctitle; char *zdata_choice; int
                       toggle; int count; int contours; int offset; float
                       *lo_val; float *hi_val; char *cctitle; char
                       *cdata_choice;
{
  static_data_s *S;
  int probe_type, zdata, cdata;
  int trans_info_flag;
  int new_ucd, new_zdata, new_cdata;
  int new_probe, new_probe_def;
  int new_sample, new_sample_region, reset_sample;
  int do_base;

  if (!(probe_type = AVSchoice_number("Probe Type Choice", probe_choice)))
    return 0;
  if (!(zdata = AVSchoice_number("ZData Choice", zdata_choice)))
    return 0;
  if (!(cdata = AVSchoice_number("CData Choice", cdata_choice)))
    return 0;

  if (!AVSstatic || AVSparameter_changed("Probe Type Choice"))
    modify_probe_type_parameters(probe_type, &toggle, &count);

  if (!AVSstatic)
    {
      AVSstatic = (char *)malloc(sizeof(struct _static_data_s));
      S = (static_data_s *)AVSstatic;
      memset(S, 0, sizeof(struct _static_data_s));
    }
  else
    S = (static_data_s *)AVSstatic;

  /*
   * Check the inputs and parameters are in a well defined state.
   */

  if (AVSinput_changed("ucd in", 0))
    {
      stat_ucd(S, ucd_struct);
      new_ucd = 1;
    }
  else
    new_ucd = 0;

  if (!S->UCD_Stated) return 0;

  if (new_ucd ||
/*       AVSinput_changed("scale info", 0) || */
      AVSparameter_changed("ZData Choice"))
    {
      stat_zdata(S, ucd_struct, /* ucd_scales, */ zdata - 1);
      new_zdata = 1;
    }
  else
    new_zdata = 0;

  if (new_ucd ||
      AVSinput_changed("contour range", 0) ||
      AVSparameter_changed("CData Choice") ||
      AVSparameter_changed("Lo Val") ||
      AVSparameter_changed("Hi Val"))
    {
      stat_cdata(S, ucd_struct, contour_range, cdata - 1, *lo_val, *hi_val);
      new_cdata = 1;
    }
  else
    new_cdata = 0;

  new_probe = new_ucd || AVSparameter_changed("Probe Type Choice");
  new_probe_def =
    (new_probe || AVSparameter_changed("Count") && probe_type == MULTI);
  trans_info_flag = get_trans_info_flag(trans_info);
  
  /*
   * If necessary update the probe geometry.
   */
  if (new_probe_def ||
      trans_info_flag ||
      AVSparameter_changed("XRotation") ||
      AVSparameter_changed("YRotation") ||
      AVSparameter_changed("ZRotation") ||
      AVSparameter_changed("Scale") ||
      AVSparameter_changed("Distance"))
    {
      update_planes_probe(S, probe_out, probe_type, count, trans_info_flag ?
			  trans_info : 0, *xrot, *yrot, *zrot, *scale, *dist,
			  new_probe_def);

#if defined(CONTINUAL_SAMPLE) && defined(CONTINUAL_UPDATE)
      new_sample_region = 1;
#else
      new_sample_region = !(trans_info_flag & BUTTON_MOVING);
#endif
    }
  else
    {
      AVSmark_output_unchanged("probe");
      new_sample_region = 0;
    }


  /*
   * If necessary recalculate the sample.
   */
  reset_sample =
    (new_probe ||
     make_sample && AVSparameter_changed("Make Sample") ||
     make_ucd && AVSparameter_changed("Make UCD"));
    
  if ((make_sample || make_ucd) &&
      (reset_sample || new_sample_region ||
       colour_field && AVSinput_changed("colour field", 0) ||
       probe_type == SIMPLE && AVSparameter_changed("Toggle")))
    {
      if (S->Sample)
	free_sample(S, S->Sample);
      else
	reset_sample = 1;
      S->Sample = generate_sample(S, ucd_struct, colour_field, probe_type ==
				  PROJECTED, probe_type == SIMPLE && !toggle);
      new_sample = 1;
    }
  else
    new_sample = 0;


  /*
   * If necessary update the sample geometry edit list.
   */
  if (S->Sample && make_sample)
    {
      *sample_out = GEOMinit_edit_list(*sample_out);

      if (reset_sample || new_sample)
	add_sample_object(S, *sample_out, S->Sample, probe_type);

      do_base = probe_type == PROJECTED && toggle;
      if (reset_sample ||
	  AVSparameter_changed("Toggle") ||
	  do_base && new_sample)
	{
	  if (do_base)
	    add_base_object(S, *sample_out, S->Sample);
	  else
	    remove_base_object(S, *sample_out);
	}
      
      if (S->CData < 0)	contours = 0;
      if (reset_sample || 
	  AVSparameter_changed("Contours") ||
	  contours &&
	  (new_cdata || new_sample ||
	   AVSparameter_changed("Offset") ||
	   probe_type == PROJECTED && AVSparameter_changed("Toggle")))
	{
	  if (contours)
	    add_contours_object(S, *sample_out, S->Sample, contours, do_base,
				offset, reset_sample ||
				AVSparameter_changed("Offset"));
	  else
	    remove_contours_object(S, *sample_out);
	}
      
      if (reset_sample ||
	  probe_type == PROJECTED &&
	  (new_zdata || new_sample ||
	   AVSparameter_changed("Toggle") ||
	   AVSparameter_changed("Count")))
	{
	  if (probe_type == PROJECTED)
	    add_framework_objects(S, *sample_out, count, toggle);
	  else
	    remove_framework_objects(S, *sample_out);
	}

      if (reset_sample)
	set_default_view_trans(S, *sample_out, probe_type, toggle);
    }
  else
    AVSmark_output_unchanged("sample");

  /*
   * If necessary regenerate the ucd structure.
   */
  if (S->Sample && make_ucd && new_sample)
    *ucd_out = generate_ucd(S, *ucd_out, ucd_struct, S->Sample, 0);
  else
    AVSmark_output_unchanged("ucd out");

  /*
   * Return success.
   */
  return 1;
}


/*****************************************************************************
 *
 * Module initialisation and destruction functions.
 *
 */

ucd_planes_init()
{
  AVSstatic = 0;
}

ucd_planes_finis()
{
  static_data_s *S;

  /*
   * Free static storage
   */
  S = (static_data_s *)AVSstatic;
  free_sample(S, S->Sample);

/* IAC CODE CHANGE :   free(S); */

/* IAC CODE CHANGE :    free(S); */

/* IAC CODE CHANGE :     free(S); */
     free(S);
  AVSstatic = 0;
}

/*****************************************************************************
 *
 * Module description functions.
 */

add_layout(param, layout)
int param; char *layout;
{
#ifdef INITIALISE_LAYOUT
  AVSadd_parameter_prop(param, "layout", "string_block", layout);
#endif
}


ucd_planes_desc()
{
  int param, port;

  AVSset_module_name("ucd planes", MODULE_MAPPER);
  AVSset_module_flags(COOPERATIVE | REENTRANT);

  AVScreate_input_port("ucd in", "ucd", OPTIONAL);
  AVScreate_input_port("colour field", "field 1D 3-vector real", OPTIONAL);
  AVScreate_input_port("contour range", "struct ucd_legend_output", OPTIONAL);
  port = AVScreate_input_port("transformation info",
			      "struct upstream_transform",
			      OPTIONAL | INVISIBLE);
  AVSset_input_class(port, "probe:upstream_transform");

  AVScreate_output_port("probe", "geom");
  AVScreate_output_port("sample", "geom");
  AVScreate_output_port("ucd out", "ucd");

  param = AVSadd_parameter("Make Sample", "boolean", 0, 0, 0);
  add_layout(param, "panel \"$Module!stack.0\" -w stack -xy 0,64 -wh 258,667 -P title string \"ucd planes\"\npanel \"$Module\" -w panel -p \"$Module!stack.0\" -xy 0,107 -wh 257,386 -P title string \"ucd planes control\"\n manipulator \"$Module:Make Sample\" -w toggle -p \"$Module\" -xy 10,10 -wh 118,21");

  param = AVSadd_parameter("Make UCD", "boolean", 0, 0, 0);
  add_layout(param, "manipulator \"$Module:Make UCD\" -w toggle -p \"$Module\" -xy 128,10 -wh 118,21");

  param = AVSadd_float_parameter("XRotation", 0.0, -180.0, 180.0);
  add_layout(param, "panel \"$Module!page.1\" -w panel -p \"$Module!stack.0\" -xy 0,194 -wh 236,426 -P title string \"ucd planes orientation\"\nmanipulator \"$Module:XRotation\" -w dial -p \"$Module!page.1\" -xy 24,10 -wh 90,130");

  param = AVSadd_float_parameter("YRotation", 0.0, -180.0, 180.0);
  add_layout(param, "manipulator \"$Module:YRotation\" -w dial -p \"$Module!page.1\" -xy 142,11 -wh 90,130");

  param = AVSadd_float_parameter("ZRotation", 0.0, -180.0, 180.0);
  add_layout(param, "manipulator \"$Module:ZRotation\" -w dial -p \"$Module!page.1\" -xy 24,147 -wh 90,130");

  param = AVSadd_float_parameter("Scale", 1.0, 0.0, 1.0);
  add_layout(param, "manipulator \"$Module:Scale\" -w dial -p \"$Module!page.1\" -xy 142,147 -wh 90,130");

  param = AVSadd_float_parameter("Distance", 0.0, -5.0, 5.0);
  add_layout(param, "manipulator \"$Module:Distance\" -w dial -p \"$Module!page.1\" -xy 83,284 -wh 90,130");

  param = AVSadd_parameter("Probe Type Lab", "string", "Probe Type", NULL,
			   NULL);
  AVSconnect_widget(param, "text");
  AVSadd_parameter_prop(param, "width", "integer", 2);
  add_layout(param, "manipulator \"$Module:Probe Type Lab\" -w text -p \"$Module\" -xy 69,34 -wh 118,21");

  param = AVSadd_parameter("Probe Type Choice", "choice", "Simple", Options,
			   ";");
  AVSadd_parameter_prop(param, "width", "integer", 2);
  add_layout(param, "manipulator \"$Module:Probe Type Choice\" -w radio_buttons -p \"$Module\" -xy 69,57 -wh 118,108");

  param = AVSadd_parameter("ZData Lab", "string", "ZAxis Data", "", "");
  AVSconnect_widget(param, "text");
  AVSadd_parameter_prop(param, "width", "integer", 2);
  add_layout(param, "manipulator \"$Module:ZData Lab\" -w text -p \"$Module\" -xy 69,331 -wh 118,21");

  param = AVSadd_parameter("ZData Choice", "choice", "<none>", "<none>",
			   ";");
  AVSadd_parameter_prop(param, "width", "integer", 2);
  add_layout(param, "manipulator \"$Module:ZData Choice\" -w radio_buttons -p \"$Module\" -xy 69,354 -wh 118,21");

  param = AVSadd_parameter("Toggle", "boolean", 0, 0, 0);
  add_layout(param, "manipulator \"$Module:Toggle\" -w toggle -p \"$Module\" -xy 69,171 -wh 118,21");

  param = AVSadd_parameter("Count", "integer", 1, 1, 10);
  add_layout(param, "manipulator \"$Module:Count\" -w idial -p \"$Module\" -xy 83,194 -wh 90,130");

  param = AVSadd_parameter("Contours", "integer", 0, 0, 50);
  add_layout(param, "panel \"$Module!page.2\" -w panel -p \"$Module!stack.0\" -xy 0,194 -wh 237,560 -P title string \"ucd planes contours\"\n manipulator \"$Module:Contours\" -w idial -p \"$Module!page.2\" -xy 83,11 -wh 90,130");

  param = AVSadd_parameter("Offset", "boolean", 0, 0, 0);
  add_layout(param, "manipulator \"$Module:Offset\" -w toggle -p \"$Module!page.2\" -xy 69,148 -wh 118,21");

  param = AVSadd_float_parameter("Lo Val", 0.0, -100.0, 100.0);
  add_layout(param, "manipulator \"$Module:Lo Val\" -w dial -p \"$Module!page.2\" -xy 24,182 -wh 90,130");

  param = AVSadd_float_parameter("Hi Val", 0.0, -100.0, 100.0);
  add_layout(param, "manipulator \"$Module:Hi Val\" -w dial -p \"$Module!page.2\" -xy 142,182 -wh 90,130");

  param = AVSadd_parameter("CData Lab", "string", "Contour Data", "", "");
  AVSconnect_widget(param, "text");
  AVSadd_parameter_prop(param, "width", "integer", 2);
  add_layout(param, "manipulator \"$Module:CData Lab\" -w text -p \"$Module!page.2\" -xy 69,320 -wh 118,21");

  param = AVSadd_parameter("CData Choice", "choice", "<none>", "<none>",
			   ";");
  AVSadd_parameter_prop(param, "width", "integer", 2);
  add_layout(param, "manipulator \"$Module:CData Choice\" -w radio_buttons -p \"$Module!page.2\" -xy 69,342 -wh 118,21");

  AVSset_init_proc(ucd_planes_init);
  AVSset_destroy_proc(ucd_planes_finis);
  AVSset_compute_proc(ucd_planes_compute);
}


#ifdef sep_exe

AVSinit_modules()
{
  AVSmodule_from_desc(ucd_planes_desc);
}

#endif /* sep_exe */
