/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/******************************************************************************/
/*
 *  Wire_Bender fits the lines of a polytriangle geometry object to the
 *  mesh of a field.  If no field is specified, the output geometry
 *  is the same as the input geometry.  If the name of an ASCII file is
 *  given, the line vectors are read from this file and any input geometry
 *  is ignored.
 *
 */



/* IAC CODE CHANGE : #include	"avs_utils.h" */
#include "avs_utils.h"
/* IAC CODE CHANGE : #include	"clip.h" */
#include "clip.h"



static char	input_geom[]		= "Input Geom";
static char	input_field[]		= "Input Field";
static char	output_geom[]		= "Output Geom";

static char	parm_clip[]		= "clip";
static char	parm_regrid[]		= "regrid";
static char	parm_file_name[]	= "vector file name";
static char	parm_z_scale[]		= "z scale factor";



wire_bender_desc ()

{
    int		wire_bender_compute ();
    int		out_port, iparm;

    AVSset_module_name ("wire bender", MODULE_MAPPER);

    AVScreate_input_port (input_geom, "geom", OPTIONAL);
    AVScreate_input_port (input_field, "field 2D scalar", OPTIONAL);

    out_port = AVScreate_output_port (output_geom, "geom");
    AVSautofree_output (out_port);

    AVSadd_parameter (parm_clip, "boolean", TRUE, FALSE, TRUE);

    AVSadd_parameter (parm_regrid, "boolean", TRUE, FALSE, TRUE);

    iparm = AVSadd_parameter (parm_file_name, "string", NULL_STR, NULL, NULL);
    AVSconnect_widget (iparm, "browser");

    AVSadd_float_parameter (parm_z_scale, 1.0, FLOAT_UNBOUND, FLOAT_UNBOUND);

    AVSset_compute_proc (wire_bender_compute);

    return;
}



/******************************************************************************/
/*
 *  Inform AVS of the name of the description procedure for this module.
 *
 */


AVSinit_modules()
{
    AVSmodule_from_desc (wire_bender_desc);
    return ;
}



wire_bender_compute (in_geom, in_field, out_geom, clip, regrid,
                     file_name, p_z_scale)

GEOMedit_list	in_geom;
AVSfield	*in_field;
GEOMedit_list	*out_geom;
int		clip, regrid;
char		*file_name;
float		*p_z_scale;
{
    static char			vec_edlist_name[] = "wires";

    static GEOMedit_list	inp_edlist = GEOM_NULL;
    static GEOMedit_list	vec_edlist = GEOM_NULL;

    register int		i, n;
    register float		z_fac, *p_z;
    register FLOAT3		*p_v, xyz, offs;
    GEOMedit_list		geom_to_use;
    GEOMedit			*list;
    GEOMobj			*obj;
    GEOMpolytri			*pt;
    GEOMlabel			*pl;
    GEOMvert_list		*p_vlist;




    *out_geom = GEOMinit_edit_list (*out_geom);


    z_fac = *p_z_scale;



    if (AVSparameter_changed (parm_file_name))
    {
        obj = UTILS_geom_polyline_from_text_file (file_name, NULL);
        if (obj == NULL)
        {
            AVSmodify_parameter (parm_file_name, AVS_VALUE,
                                 NULL_STR, NULL, NULL);

            wires_destroy_edlist (*out_geom, &vec_edlist);

            if (in_geom != GEOM_NULL)
                inp_edlist = UTILS_geom_copy_edlist (in_geom);
        }

        else
        {
            vec_edlist = GEOMinit_edit_list (vec_edlist);
            GEOMedit_geometry (vec_edlist, vec_edlist_name, obj);
            GEOMdestroy_obj (obj);
        }
    }

    if ((AVSinput_changed (input_geom, 0)) || (in_geom == GEOM_NULL))
    {
        wires_destroy_edlist (*out_geom, &inp_edlist);

        if (in_geom != GEOM_NULL)
            inp_edlist = UTILS_geom_copy_edlist (in_geom);
    }



    geom_to_use = GEOM_NULL;

    if (vec_edlist != GEOM_NULL)
    {
        wires_destroy_edlist (*out_geom, &inp_edlist);
        geom_to_use = UTILS_geom_copy_edlist (vec_edlist);
    }
    else if (inp_edlist != GEOM_NULL)
    {
        wires_destroy_edlist (*out_geom, &vec_edlist);
        geom_to_use = UTILS_geom_copy_edlist (inp_edlist);
    }

    if (geom_to_use != GEOM_NULL)
    {

        if (in_field != NULL)
        {
            UTILS_field_pt_init (in_field);

            UTILS_field_offsets_from_surf (in_field, offs);
            if (z_fac != 0.0)
                offs[0] /= z_fac, offs[1] /= z_fac, offs[2] /= z_fac;
        }

        list = geom_to_use->l;
        while (list != GEOM_NULL)
        {
            if (list->type != 0)
            {
                if ((*out_geom)->l == NULL)
                    (*out_geom)->l = UTILS_geom_copy_list (list);
                else
                    (*out_geom)->l->next = UTILS_geom_copy_list (list);
            }

            else
            {
                obj = (GEOMobj *) list->data;

                switch (obj->type)
                {

                    case GEOM_POLYTRI :

                        pt = &(obj->d.pt);

                        wires_shift_xy (&pt->plverts, pt->npls, in_field);
                        wires_shift_xy (&pt->dlverts,       -1, in_field);

                        if (clip)
                        {
                            wires_clip (&pt->plverts, pt->npls, in_field, obj);
                            wires_clip (&pt->dlverts,       -1, in_field, obj);
                        }

                        if (regrid)
                        {
                            wires_regrid (&pt->plverts, pt->npls, in_field);
                            wires_regrid (&pt->dlverts,       -1, in_field);
                        }

                        wires_remap_z (&pt->plverts, pt->npls, in_field, offs);
                        wires_remap_z (&pt->dlverts,       -1, in_field, offs);

                        wires_scale_z (&pt->plverts, pt->npls, z_fac);
                        wires_scale_z (&pt->dlverts,       -1, z_fac);

                        break;

                    case GEOM_LABEL :

                        pl = &(obj->d.la);

                        p_vlist = &(pl->verts);

                        if (in_field != NULL)
                        {
                            wires_shift_xy (&p_vlist, 1, in_field);

                            wires_get_field_limits (NULL, xyz);

                            n   = pl->verts.n;
                            p_v = (FLOAT3 *) pl->verts.l;
                            p_z = (float *) malloc (n * sizeof (float));
                            for (i = 0; i < n; i++)
                            {
                                p_z[i] = p_v[i][2];

                                if ((clip) && (pl->labels.l[i] != GEOM_NULL))
                                {
                                    if ((p_v[i][0] <    0.0) ||
                                        (p_v[i][0] > xyz[0]) ||
                                        (p_v[i][1] <    0.0) ||
                                        (p_v[i][1] > xyz[1]))
                                    {

/* IAC CODE CHANGE :                                     free (pl->labels.l[i]); */
                                     FREE_LOCAL(pl->labels.l[i]) ;
                                    pl->labels.l[i] = GEOM_NULL;
                                    }
                                }
                            }

                            wires_remap_z (&p_vlist, 1, in_field, NULL);

                            p_v = (FLOAT3 *) pl->verts.l;
                            for (i = 0; i < n; i++) p_v[i][2] += p_z[i];

/* IAC CODE CHANGE :                             free (p_z); */
                             FREE_LOCAL(p_z) ;
                        }

                        wires_scale_z (&p_vlist, 1, z_fac);

                        break;
                }

                GEOMedit_geometry (*out_geom, list->name, obj);
            }

            list = list->next;
        }

        UTILS_geom_destroy_edlist (geom_to_use);
    }


    return (1);
}




wires_destroy_edlist (out_geom, p_edlist)

GEOMedit_list	out_geom, *p_edlist;
{

    GEOMedit		*list;

    if (*p_edlist != GEOM_NULL)
    {
        list = (*p_edlist)->l;
        while (list != GEOM_NULL)
        {
            GEOMedit_visibility (out_geom, list->name, GEOM_EDIT_DELETE);
            list = list->next;
        }
        GEOMdestroy_edit_list (*p_edlist);
        *p_edlist = GEOM_NULL;
    }


    return;
}



GEOMvert_list	*wires_check_vlist (p_p_vlist, p_n_vlists, p_field, p_disjoint)

GEOMvert_list	**p_p_vlist;
int		*p_n_vlists;
AVSfield	*p_field;
int		*p_disjoint;
{

    GEOMvert_list	*p_list;

    if (p_p_vlist == NULL) return (NULL);
    if (p_field   == NULL) return (NULL);

    if (*p_n_vlists > 0)
    {
        *p_disjoint = FALSE;
        p_list      = *p_p_vlist;
    }
    else
    {
        *p_disjoint = TRUE;
        *p_n_vlists = -*p_n_vlists;
        p_list      = (GEOMvert_list *) p_p_vlist;
    }


    return (p_list);

}




wires_get_field_limits (p_min, p_max)

FLOAT3		p_min, p_max;
{
    register FLOAT3	min, max;

    UTILS_field_pt_limits (min, max);
    max[0] -= min[0], max[1] -= min[1];

    if (p_min != NULL) UTILS_float3_copy (min, p_min);
    if (p_max != NULL) UTILS_float3_copy (max, p_max);


    return;
}



wires_clip (p_p_vlist, n_vlists, p_field, obj)

GEOMvert_list	**p_p_vlist;
int		n_vlists;
AVSfield	*p_field;
GEOMobj		*obj;
{

    register int		i, j, n, i_list, n_old, line_stat;
    register int		quit = 128;
    register FLOAT3		*v_old;
    register FLOAT3		xyz1, xyz2, abc1, abc2;
    register GEOMvert_list	*p_list;
    int				disjoint;

    if ((p_list = wires_check_vlist (p_p_vlist, &n_vlists, p_field, &disjoint))
        == NULL) return;

    wires_get_field_limits (xyz1, xyz2);
    clip_init (0.0, 0.0, xyz2[0], xyz2[1]);

    for (i_list = 0; i_list < n_vlists; i_list++, p_list++)
    {
        n_old = p_list->n;
        v_old = (FLOAT3 *) p_list->l;

        if (n_old > 0)
        {
            UTILS_vert_chain_init ();

            if (disjoint)
            {
                for (i = 0, j = 1; i < n_old; i += 2, j += 2)
                {
                    line_stat = clip_line ((v_old+i), (v_old+j));
                    if (line_stat != CLIP_LINE_OUT)
                    {
                        UTILS_vert_chain_add ((v_old+i));
                        UTILS_vert_chain_add ((v_old+j));
                    }
                }
            }

            else
            {
                UTILS_float3_copy (v_old, xyz2);

                i         = 0;
                line_stat = CLIP_LINE_OUT;
                while (line_stat == CLIP_LINE_OUT)
                {
                    if (++i >= n_old)
                        line_stat = quit;
                    else
                    {
                        UTILS_float3_copy (xyz2, xyz1);
                        UTILS_float3_copy ((v_old+i), xyz2);
                        line_stat = clip_line (xyz1, xyz2);
                    }
                }

                if (line_stat != quit)
                {
                    UTILS_vert_chain_add (xyz1);

                    while (line_stat & CLIP_LINE_PT2_IN)
                    {
                        if (++i >= n_old)
                            line_stat = quit;
                        else
                        {
                            UTILS_vert_chain_add (xyz2);
                            UTILS_float3_copy (xyz2, xyz1);
                            UTILS_float3_copy ((v_old+i), xyz2);
                            line_stat = clip_line (xyz1, xyz2);
                        }
                    }

                    n = n_old - i;
                    if (n > 1)
                    {
                        GEOMadd_polyline (obj, (v_old+i), GEOM_NULL, n,
                                          GEOM_COPY_DATA);
                        p_list = *p_p_vlist + i_list;
                        n_vlists++;
                    }

                    UTILS_vert_chain_add (xyz2);
                }
            }		/* if (disjoint) */

            UTILS_vert_chain_to_vlist (p_list);

        }		/* if (n_old > 0) */

    }		/* for (i_list...) */

    return;
}



wires_regrid (p_p_vlist, n_vlists, p_field)

GEOMvert_list	**p_p_vlist;
int		n_vlists;
AVSfield	*p_field;
{

    register int		i, j, i_list, n_old;
    register FLOAT3		*v_old;
    register float		x, y;
    register FLOAT3		xyz1, xyz2, abc1, abc2;
    register GEOMvert_list	*p_list;
    int				disjoint;

    if ((p_list = wires_check_vlist (p_p_vlist, &n_vlists, p_field, &disjoint))
        == NULL) return;

    for (i_list = 0; i_list < n_vlists; i_list++, p_list++)
    {
        n_old = p_list->n;
        v_old = (FLOAT3 *) p_list->l;

        if (n_old > 0)
        {
            UTILS_vert_chain_init ();

            UTILS_float3_copy (v_old, xyz2);

            for (i = 1; i < n_old; i++)
            {
                UTILS_vert_chain_add (xyz2);
                UTILS_vert_chain_mark ();

                UTILS_float3_copy (xyz2, xyz1);
                UTILS_float3_copy ((v_old+i), xyz2);

                if (line_2d_eqn (xyz1, xyz2, abc1))
                {
                    if (abc1[0] != 1.0)		/* Find X grid ints */
                    {
                        abc2[0] = -1.0, abc2[1] = 0.0;
                        wires_regrid_pts_find (abc1, abc2, xyz1[0], xyz2[0],
                                               xyz1[2], xyz2[2]);
                    }

                    if (abc1[1] != 1.0)		/* Find Y grid ints */
                    {
                        abc2[0] = 0.0, abc2[1] = -1.0;
                        wires_regrid_pts_find (abc1, abc2, xyz1[1], xyz2[1],
                                               xyz1[2], xyz2[2]);
                    }

                    if (abc1[0] != abc1[1])	/* Find diagonal grid ints */
                    {
                        abc2[0] = abc2[1] = -1.0;
                        x = (xyz1[0] + xyz1[1]);
                        y = (xyz2[0] + xyz2[1]);
                        wires_regrid_pts_find (abc1, abc2, x, y,
                                               xyz1[2], xyz2[2]);
                    }


                }	/* if (line_2d_eqn) */

                j = (abc1[0] != 1.0) ? X_COORD : Y_COORD;
                UTILS_vert_chain_sort (xyz1, xyz2, j);

                if (disjoint)
                {
                    UTILS_vert_chain_add (xyz2);
                    UTILS_vert_chain_disjoint ();
                    if (++i < n_old) UTILS_float3_copy ((v_old+i), xyz2);
                }

            }		/* for (i = 0; i < n_old; i++) */

            if (!disjoint)
            {
                UTILS_vert_chain_add (xyz2);
                UTILS_vert_chain_add (xyz2); /* Needed for Stardent GS-3000 */
            }

            UTILS_vert_chain_to_vlist (p_list);

        }		/* if (n_old > 0) */

    }		/* for (i_list...) */

    return;
}



wires_regrid_pts_find (abc1, abc2, xy1, xy2, zz1, zz2)

FLOAT3		abc1, abc2;
float		xy1, xy2, zz1, zz2;
{
    register float	x, x1, x2, z1, z2, dz;
    register FLOAT3	xyz;

    if (xy1 == xy2) return ;

    if (xy1 < xy2)
        x1 = xy1, x2 = xy2, z1 = zz1, z2 = zz2;
    else
        x1 = xy2, x2 = xy1, z1 = zz2, z2 = zz1;

    x = (float) ((int) x1);
    if (x >= 0.0) x++;

    dz     = (z2 - z1) / (x2 - x1);
    xyz[2] = z1 + ((x - x1) * dz);

    while (x < x2)
    {
        abc2[2] = x;
        if (line_2d_int (abc1, abc2, xyz)) UTILS_vert_chain_add (xyz);
        x++, xyz[2] += dz;
    }


    return;
}



wires_remap_z (p_p_vlist, n_vlists, p_field, offsets)

GEOMvert_list	**p_p_vlist;
int		n_vlists;
AVSfield	*p_field;
FLOAT3		offsets;
{

    register int		n, m, i_list;
    register float		z1, z2;
    register FLOAT3		*verts, xymin, xymax;
    GEOMvert_list		*p_list;
    int				disjoint;

    if ((p_list = wires_check_vlist (p_p_vlist, &n_vlists, p_field, &disjoint))
        == NULL) return;

    wires_get_field_limits (xymin, xymax);

    if (disjoint)
    {
        for (i_list = 0; i_list < n_vlists; i_list++, p_list++)
        {
            n     = p_list->n;
            verts = (FLOAT3 *) p_list->l;
            while ((--n) >= 0)
            {
                m = n, n--;
                z1 = verts[m][2];
                z2 = verts[n][2];
                wires_remap_z_get ((verts+m), xymin, xymax, offsets);
                wires_remap_z_get ((verts+n), xymin, xymax, offsets);

                if ((verts[n][0] == verts[m][0]) &&
                    (verts[n][1] == verts[m][1]) && (z1 != z2))
                    verts[n][2] += (z1 - z2);
            }
        }
    }

    else
    {
        for (i_list = 0; i_list < n_vlists; i_list++, p_list++)
        {
            n     = p_list->n;
            verts = (FLOAT3 *) p_list->l;
            while ((--n) >= 0)
                wires_remap_z_get ((verts+n), xymin, xymax, offsets);
        }
    }

    return;
}



wires_remap_z_get (verts, xymin, xymax, offsets)

FLOAT3		verts;
FLOAT3		xymin, xymax, offsets;
{
    register FLOAT3		xyz;

    if (!UTILS_field_pt_xy_to_xyz (verts))
    {
        UTILS_float3_copy (verts, xyz);
        xyz[0] = (xyz[0] < 0.0) ? 0.0 : (xyz[0] > xymax[0]) ? xymax[0] : xyz[0];
        xyz[1] = (xyz[1] < 0.0) ? 0.0 : (xyz[1] > xymax[1]) ? xymax[1] : xyz[1];
        UTILS_field_pt_xy_to_xyz (xyz);
        verts[0] += xymin[0], verts[1] += xymin[1];
        verts[2] = xyz[2];
    }

    if (offsets != NULL)
    {
        verts[0] += offsets[0];
        verts[1] += offsets[1];
        verts[2] += offsets[2];
    }


    return;
}



wires_scale_z (p_p_vlist, n_vlists, z_scale)

GEOMvert_list	**p_p_vlist;
int		n_vlists;
float		z_scale;
{

    register int		n, i_list;
    register float		z_fac;
    register FLOAT3		*verts;
    GEOMvert_list		*p_list;
    int				disjoint;
    AVSfield			*p_dummy = (AVSfield *) 1L;	/* != NULL */

    if ((p_list = wires_check_vlist (p_p_vlist, &n_vlists, p_dummy, &disjoint))
        == NULL) return;

    if (z_scale == 1.0) return;

    z_fac = z_scale;

    for (i_list = 0; i_list < n_vlists; i_list++, p_list++)
    {
        n     = p_list->n;
        verts = (FLOAT3 *) p_list->l;
        while ((--n) >= 0) verts[n][2] *= z_fac;
    }

    return;
}



wires_shift_xy (p_p_vlist, n_vlists, p_field)

GEOMvert_list	**p_p_vlist;
int		n_vlists;
AVSfield	*p_field;
{

    register int		n, i_list;
    register float		x_adj, y_adj;
    register FLOAT3		*verts, xymin;
    GEOMvert_list		*p_list;
    int				disjoint;

    if ((p_list = wires_check_vlist (p_p_vlist, &n_vlists, p_field, &disjoint))
        == NULL) return;

    wires_get_field_limits (xymin, NULL);
    x_adj = xymin[0], y_adj = xymin[1];

    if ((x_adj == 0.0) && (y_adj == 0.0)) return;

    for (i_list = 0; i_list < n_vlists; i_list++, p_list++)
    {
        n     = p_list->n;
        verts = (FLOAT3 *) p_list->l;
        while ((--n) >= 0) verts[n][0] -= x_adj, verts[n][1] -= y_adj;
    }

    return;
}
